#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import RcfAst
from RcfAgentHeartbeat import RcfAgentHeartbeat

class IVisitor:
   """ Visitor interface to override.
   """
   def visit( self, node, **kwargs ):
      raise NotImplementedError

   def __call__( self, node, **kwargs ):
      return self.visit( node, **kwargs )( node, **kwargs )

class Visitor( IVisitor ):
   """ Rcf Visitor interface to visit the Abstract Syntax Tree (AST).
   """
   def __init__( self ):
      self.dispatchMap = {
         RcfAst.Root: self.visitRoot,
         RcfAst.Function: self._visitFunction,
         RcfAst.FunctionParam: self.visitFuncParam,
         RcfAst.Block: self.visitBlock,
         RcfAst.IfStmt: self.visitIfStmt,
         RcfAst.Call: self.visitCall,
         RcfAst.ExternalRefOp: self.visitExternalRefOp,
         RcfAst.ExternalRef: self.visitExternalRef,
         RcfAst.Assign: self.visitAssign,
         RcfAst.Return: self.visitReturn,
         RcfAst.Exit: self.visitExit,
         RcfAst.CommunityValue: self.visitCommunityValue,
         RcfAst.Constant: self.visitConstant,
         RcfAst.Range: self.visitRange,
         RcfAst.Collection: self.visitCollection,
         RcfAst.Attribute: self.visitAttribute,
         RcfAst.BinOp: self.visitBinOp,
         RcfAst.LogicalOp: self.visitLogicalOp,
         RcfAst.Not: self.visitNot,
         RcfAst.Variable: self.visitVariable,
         RcfAst.Directive: self.visitDirective,
      }

   def visit( self, node, **kwargs ):
      """
      Determines the method an Ast node should be handled with.
      The function pointer is returned to reduce the call stack depth required
      when traversing the tree. The caller is responsible for calling the returned
      method.
      This halves our call stack depth by removing this visit call from the stack.
      As we can recurse deeply when dealing with very large expressions with many
      operators it is important to minimize our call stack depth.

      Args:
         Node: AstNode to be handled

      Returns:
         A pointer to the method on this visitor the caller should invoke with the
         given node.
      """
      return self.dispatchMap[ type( node ) ]

   def visitRoot( self, root, **kwargs ):
      pass

   def _visitFunction( self, function, **kwargs ):
      RcfAgentHeartbeat.punch()
      self.visitFunction( function, **kwargs )

   def visitFunction( self, function, **kwargs ):
      pass

   def visitFuncParam( self, funcParam, **kwargs ):
      pass

   def visitBlock( self, block, **kwargs ):
      pass

   def visitIfStmt( self, ifStmt, **kwargs ):
      pass

   def visitSequentialExpr( self, sequentialExpr, **kwargs ):
      pass

   def visitCall( self, call, **kwargs ):
      pass

   def visitExternalRefOp( self, externalRefOp, **kwargs ):
      pass

   def visitExternalRef( self, extRef, **kwargs ):
      pass

   def visitAssign( self, assign, **kwargs ):
      pass

   def visitReturn( self, returnOperation, **kwargs ):
      pass

   def visitExit( self, exitOperation, **kwargs ):
      pass

   def visitDirective( self, directive, **kwargs ):
      pass

   def visitCommunityValue( self, commVal, **kwargs ):
      pass

   def visitConstant( self, constant, **kwargs ):
      pass

   def visitRange( self, rangeNode, **kwargs ):
      pass

   def visitCollection( self, collection, **kwargs ):
      pass

   def visitAttribute( self, attribute, **kwargs ):
      pass

   def visitBinOp( self, binOp, **kwargs ):
      pass

   def visitLogicalOp( self, logicalOp, **kwargs ):
      pass

   def visitNot( self, notExpr, **kwargs ):
      pass

   def visitVariable( self, variable, **kwargs ):
      pass
