#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import RcfAstVisitor
import RcfAstListener
from enum import Enum

class Action( Enum ):
   ENTRY = 0
   EXIT = 1

class Walker( RcfAstVisitor.Visitor ):
   """ Rcf Ast Walker goes through each node in the Abstract Syntax Tree (AST)
       and calls the listen function from each listeners.
   """
   def __init__( self ):
      super().__init__()
      self.registeredListeners = set()

   def registerListener( self, listener ):
      assert isinstance( listener, RcfAstListener.Listener )
      self.registeredListeners.add( listener )

   def handle( self, node, action ):
      for listener in self.registeredListeners:
         listener.listen( node, action )

   # ---------------------------------------------------------------------------
   #                       visit methods override.
   # ---------------------------------------------------------------------------
   def visitRoot( self, root, **kwargs ):
      self.handle( root, Action.ENTRY )
      for function in root.validFunctions():
         self.visit( function )( function )
      self.handle( root, Action.EXIT )

   def visitFunction( self, function, **kwargs ):
      self.handle( function, Action.ENTRY )
      self.visit( function.block )( function.block )
      self.handle( function, Action.EXIT )

   def visitBlock( self, block, **kwargs ):
      self.handle( block, Action.ENTRY )
      for stmt in block.stmts:
         self.visit( stmt )( stmt )
      self.handle( block, Action.EXIT )

   def visitIfStmt( self, ifStmt, **kwargs ):
      self.handle( ifStmt, Action.ENTRY )
      self.visit( ifStmt.condition )( ifStmt.condition )
      self.visit( ifStmt.thenBlock )( ifStmt.thenBlock )
      if ifStmt.elseBlock:
         self.visit( ifStmt.elseBlock )( ifStmt.elseBlock )
      self.handle( ifStmt, Action.EXIT )

   def visitCall( self, call, **kwargs ):
      self.handle( call, Action.ENTRY )
      if call.functionSelf:
         self.visit( call.functionSelf )( call.functionSelf )
      for funcArg in call.funcArgs:
         self.visit( funcArg )( funcArg )
      self.handle( call, Action.EXIT )

   def visitExternalRefOp( self, externalRefOp, **kwargs ):
      self.handle( externalRefOp, Action.ENTRY )
      self.visit( externalRefOp.attribute )( externalRefOp.attribute )
      self.visit( externalRefOp.rhs )( externalRefOp.rhs )
      for additionalAttribute in externalRefOp.additionalAttributes:
         self.visit( additionalAttribute )( additionalAttribute )
      self.handle( externalRefOp, Action.EXIT )

   def visitExternalRef( self, extRef, **kawargs ):
      self.handle( extRef, Action.ENTRY )
      self.handle( extRef, Action.EXIT )

   def visitAssign( self, assign, **kwargs ):
      self.handle( assign, Action.ENTRY )
      self.visit( assign.attribute )( assign.attribute )
      self.visit( assign.value )( assign.value )
      self.handle( assign, Action.EXIT )

   def visitReturn( self, returnOperation, **kwargs ):
      self.handle( returnOperation, Action.ENTRY )
      self.visit( returnOperation.expr )( returnOperation.expr )
      self.handle( returnOperation, Action.EXIT )

   def visitExit( self, exitOperation, **kwargs ):
      self.handle( exitOperation, Action.ENTRY )
      self.visit( exitOperation.expr )( exitOperation.expr )
      self.handle( exitOperation, Action.EXIT )

   def visitDirective( self, directive, **kwargs ):
      self.handle( directive, Action.ENTRY )
      self.handle( directive, Action.EXIT )

   def visitCommunityValue( self, commVal, **kwargs ):
      self.handle( commVal, Action.ENTRY )
      self.handle( commVal, Action.EXIT )

   def visitConstant( self, constant, **kwargs ):
      self.handle( constant, Action.ENTRY )
      self.handle( constant, Action.EXIT )

   def visitRange( self, rangeNode, **kwargs ):
      self.handle( rangeNode, Action.ENTRY )
      self.visit( rangeNode.lowerBound )( rangeNode.lowerBound )
      self.visit( rangeNode.upperBound )( rangeNode.upperBound )
      self.handle( rangeNode, Action.EXIT )

   def visitCollection( self, collection, **kwargs ):
      self.handle( collection, Action.ENTRY )
      for value in collection.values:
         self.visit( value )( value )
      for value in collection.ranges:
         self.visit( value )( value )
      self.handle( collection, Action.EXIT )

   def visitAttribute( self, attribute, **kwargs ):
      self.handle( attribute, Action.ENTRY )
      self.handle( attribute, Action.EXIT )

   def visitBinOp( self, binOp, **kwargs ):
      self.handle( binOp, Action.ENTRY )
      self.visit( binOp.lhs )( binOp.lhs )
      self.visit( binOp.rhs )( binOp.rhs )
      self.handle( binOp, Action.EXIT )

   def visitLogicalOp( self, logicalOp, **kwargs ):
      self.handle( logicalOp, Action.ENTRY )
      for expr in logicalOp.expressionList:
         self.visit( expr )( expr )
      self.handle( logicalOp, Action.EXIT )

   def visitNot( self, notExpr, **kwargs ):
      self.handle( notExpr, Action.ENTRY )
      self.visit( notExpr.expr )( notExpr.expr )
      self.handle( notExpr, Action.EXIT )
