# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from RcfLibCallgraph import(
   CallGraphNode,
)

class BottomUpOrder:
   def __init__( self, functionNameToAst, callgraph ):
      self.functionNameToAst = functionNameToAst
      self.functionValidList = {}
      self.callgraph = callgraph
      self.visitedFunctions = set()
      self.bottomUpOrder = []

   def sortFunctionCallgraphBottomUpImpl( self, funcName ):
      '''
      Recursively do a post order traversal to find all callee functions.
      This creates an ordering where every callee is before the respective caller.
      '''
      function = self.functionNameToAst.get( funcName )

      if funcName not in self.visitedFunctions:
         self.visitedFunctions.add( funcName )
         callGraphNode = CallGraphNode( funcName )
         for callee in self.callgraph.callees( callGraphNode ):
            function.valid &= self.sortFunctionCallgraphBottomUpImpl( callee.name )

         # After having visited all callee nodes we can safely append our function
         # to the orderedFunction list
         self.bottomUpOrder.append( function )
      return function.valid

   def sortFunctionCallgraphBottomUp( self ):
      '''
      Function that generates a bottom up order where every callee is defined before
      its respective caller.

      Args:
         callGraph: callgraph containing all calls between functions
         functionIter: An iterator that provides all functions.

      returns:
         bottomUpOrder (list): A list containing all functions in the bottom up order

      '''
      for functionName in self.functionNameToAst:
         self.sortFunctionCallgraphBottomUpImpl( functionName )
      return self.bottomUpOrder

def setFunctionsInBottomUpOrder( rcfAst, callgraph ):
   '''
   The bottomUpOrder collection is the order in which the RcfAetGen should generate
   the AETs. This ordering is important, because every callee appears before the
   respective caller.
   '''
   # Usually rcfAst.validFunctions() would be used here, but we want to prefer
   # builtin functions to be compiled last. This will cause them to have the highest
   # AetNodeKey ids. Otherwise, everytime a builtin function is added we would see a
   # lot of noise in the RcfAetGenTest test changes.
   functionOrder = rcfAst.validAndInvalidFunctions( builtInFirst=False )
   functionNameToAst = { func.name: func for func in functionOrder }

   bottomUpOrder = BottomUpOrder( functionNameToAst, callgraph )

   rcfAst.setBottomUpOrder( tuple( bottomUpOrder.sortFunctionCallgraphBottomUp() ) )
