# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.


import Tac
import Tracing
import QuickTrace
from RcfAetGen import genAbstractEvalTree
from RcfCompilerStageCommon import RcfCompilerStageResultBase
from RcfLinter import lintCodeUnits
from RcfDiag import RcfDiag

t0 = Tracing.t0
qv = QuickTrace.Var
qt8 = QuickTrace.trace8

class RcfCompileRequest:
   """Input for the RCF compiler

   codeUnitMapping : CodeUnitMapping
      Mapping of code unit keys to code unit texts.
   rcfCodeVersion : int
      The version associated with the RCF code to be compiled.
   strictMode : bool
      Compiler to follow strict rules. This will fail compilation if the
      referred external constructs such as prefix-lists, community-lists,
      are not configured.
      An instance where strictMode is False is when compiling RCF code
      as part of the startup-config.
   """
   def __init__( self, codeUnitMapping, rcfCodeVersion,
                 strictMode=False ):
      self.codeUnitMapping = codeUnitMapping
      self.rcfCodeVersion = rcfCodeVersion
      self.strictMode = strictMode

class RcfCompileResult( RcfCompilerStageResultBase ):
   """Output from the RCF compiler

   Arguments
   ---------
   rcfCompileRequest : RcfCompileRequest
      The compile request to which this is a response
   success : bool
      Whether the compilation was successful or not
   allFunctionStatus : Rcf::AllFunctionStatus
      The result of compilation. This may have pointers to previously published
      AETs if nothing in them changes.
   debugSymbols : dict (Rcf::Debug::FunctionSymbol keyed by function name)
      The Debug Symbols generated for this generation of AETs.
   errorList : (list of str) or (None)
      None if compilation was successful. List of error strings otherwise.
   warningList : (list of str)
      Any warnings produced during the compilation.
   """
   def __init__( self, rcfCompileRequest, success, allFunctionStatus,
                 debugSymbols, diag ):
      super().__init__( success=success, diag=diag )
      self.rcfCompileRequest = rcfCompileRequest
      self.allFunctionStatus = allFunctionStatus
      self.debugSymbols = debugSymbols

   def publish( self, publishedAfs, rcfStatus, debugSymbols,
                instantiatingCollectionCleaner ):
      """Compare new results in self.allFunctionStatus with previously published
      state in publishedAfs and update publishedAfs. This involves
      1. deleting functions that are no longer in code
      2. updating the select functions whose aet has changed
      3. updating newly added functions
      """
      prevPublishedFnNames = set( publishedAfs.aet )
      newPublishedFnNames = set( self.allFunctionStatus.aet )
      stalePublishedFns = prevPublishedFnNames - newPublishedFnNames

      prevPublishedAets = set( publishedAfs.aet.values() )
      newPublishedAets = set( self.allFunctionStatus.aet.values() )

      # Cleanup unused AET nodes from the instantiating collections
      instantiatingCollectionCleaner.cleanup( prevPublishedAets - newPublishedAets )

      # For now at least DebugSymbols for a function are always updated as they are
      # not compared and deduped. Since we have to create a new one anyway and there
      # is no disadvantage to the client in always updating the ptr (it does not
      # cause policy reevaluation) the cost of always replacing them is not as bad
      # as it first seems since the hash and compare required to not update them are
      # not free and there must be 1 per function as names are important here.
      # Cleanup the stale DebugSymbol nodes.
      symbols = list( debugSymbols.functionSymbol.values() )
      instantiatingCollectionCleaner.cleanup( symbols )

      dedupedFunctionsPresent = len( newPublishedAets ) < len( newPublishedFnNames )
      if dedupedFunctionsPresent:
         t0( "Deduped functions are present" )
         qt8( "Deduped functions are present" )
         publishedAfs.dedupedFunctionsPresent = True
         # Increment the start marker to indicate that deduped function updates are
         # in progress. The end marker equalling the start will indicate it they are
         # over. This allows the client to know when they have received all updates.
         publishedAfs.startOfDedupedFunctionUpdatesMarker += 1
      else:
         publishedAfs.dedupedFunctionsPresent = False

      # Delete all stale functions
      for fnName in stalePublishedFns:
         t0( "Deleting RCF function|", fnName )
         qt8( "Deleting RCF function|", qv( fnName ) )
         del publishedAfs.aet[ fnName ]
         rcfStatus.functionNames.remove( fnName )
         del debugSymbols.functionSymbol[ fnName ]

      # Update all modified/new functions
      for fnName in newPublishedFnNames:
         oldAet = publishedAfs.aet.get( fnName )
         newAet = self.allFunctionStatus.aet[ fnName ]
         if oldAet != newAet:
            t0( "Updating RCF function|", fnName )
            qt8( "Updating RCF function|", qv( fnName ) )
            publishedAfs.aet[ fnName ] = newAet

         if fnName not in rcfStatus.functionNames:
            rcfStatus.functionNames.add( fnName )

         debugSymbols.functionSymbol.addMember( self.debugSymbols[ fnName ] )

      # Indicate all function AET update are complete by equalling the start and
      # end markers.
      publishedAfs.endOfDedupedFunctionUpdatesMarker \
         = publishedAfs.startOfDedupedFunctionUpdatesMarker

class RcfCompiler:
   """RCF compiler instance.

   publishedAfs : Rcf::AllFunctionStatus
      The currently published version of functions
   instantiatingCollectionCleaner (InstantiatingCollectionCleaner):
      Cleaner responsible for reming instantiated AET nodes from the instantiating
      collections.
   rcfExternalConfig : RcfHelperTypes.RcfExternalConfig
      This python object holds:
         aclConfig (Acl::AclListConfig)
         roaTableStatusDir (Rpki::RoaTableStatusDir)
         dynPfxListConfigDir (DynamicPrefixList::Config)
      These are used to validate external references during symbol table generation
      during strict compilation mode. By default, the Rcf agent does NOT compile in
      strict mode, but lets RcfLinter handle external reference resolution.
   """
   def __init__( self, publishedAfs, instantiatingCollectionCleaner,
                 rcfExternalConfig=None ):
      self.rcfExternalConfig = rcfExternalConfig
      self.publishedAfs = publishedAfs
      self.instantiatingCollectionCleaner = instantiatingCollectionCleaner

   def compile( self, request ):
      """
      request : RcfCompileRequest
         Code to compile, strictness, etc
      Returns RcfCompilerResult instance.
      """
      diag = RcfDiag( strict=request.strictMode )
      lintResult = lintCodeUnits( request.codeUnitMapping, diag,
                                  self.rcfExternalConfig )

      afs, debugSymbols = genAbstractEvalTree( lintResult.root, self.publishedAfs,
                                               request.rcfCodeVersion,
                                               self.instantiatingCollectionCleaner )
      return RcfCompileResult( request, success=True, allFunctionStatus=afs,
                               debugSymbols=debugSymbols, diag=diag )
