#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import defaultdict

import RcfAst
from antlr4.Token import Token
from antlr4.error.DiagnosticErrorListener import DiagnosticErrorListener
from RcfCodeRender import renderCodeSection
from RcfDebugLocationLib import (
   getLeftmostPointToken,
   getRightmostPointToken,
)
from RcfDiagCommon import GenericDiagMessageBase, RcfDiagGeneric, Severity
from RcfParserToggle import UnsupportedFeatureToggle
import RcfMetadata

BT = RcfMetadata.RcfBuiltinTypes

class DiagLocation:
   """
   Location a compiler diagnostic messages originated from

   Attributes:
      line (Int): line number the message originated from
      column (Int): column number the message originated from
   """
   def __init__( self, line, column ):
      self.line = int( line )
      assert self.line, "line number must be non zero"
      self.column = int( column )

def sourceLoc( context ):
   return DiagLocation( context.start.line, context.start.column )

def getStartEndForFunctionSignature( function ):
   """
   Helper method to extract the start and end tokens of a function signature

   Args:
      func (AstNode): the function we are interested in.
   """
   # Render from the beginning of the function definition up until the closing
   # parenthesis prior to the body/block.
   startToken = function.context.start
   endToken = function.context.closingParens().stop
   return startToken, endToken

class CompilerDiagMessageBase( GenericDiagMessageBase ):
   """
   Base type for compiler diagnostic messages that provides enough attributes
   to correctly format the message to the user

   Attributes:
      location (DiagLocation): the location the message originated from
      functionName (string): the RCF function that contained the code causing the
                             message (None if not applicable)
      codeUnitKey (CodeUnitKey): The code unit the error is in
      msg (string): the message issued by the compiler
   """

   def __init__( self, location, codeUnitKey, functionName, msg ):
      super().__init__( msg )
      self.location = location
      self.functionName = functionName
      self.codeUnitKey = codeUnitKey

   def render( self, codeUnitMapping ):
      """
      Render a compiler message to a string, including the code unit name and
      adjusting the line number to be relative to the code unit where applicable.
      Args:
         codeUnitMapping (CodeUnitMapping):
            Mapping of line numbers in rcfText to their originating code units and
            relative line numbers within those code units.
      """
      assert isinstance( self.severity, Severity )
      lineParts = [ str( self.severity ) ]

      def getLinePartsForPoaWrapper():
         # POA wrapper are invisible to the user, so we do not render things like
         # line numbers or function names
         return [ self.codeUnitKey.toStrepForDiag() ]

      def getLinePartsForOpenConfigFunction():
         # OC functions are in "function" config, not "code unit" config, so we do
         # not render function names
         linePartsForOpenConfigFunction = [ self.codeUnitKey.toStrepForDiag() ]
         if self.location is not None:
            linePartsForOpenConfigFunction.extend(
               [ "line", f"{self.location.line}:{self.location.column}" ] )
         return linePartsForOpenConfigFunction

      def getLinePartsForCodeUnit():
         linePartsForCodeUnit = []
         # CodeUnitKey can be None for Cycle errors, because code unit names are not
         # relevant for this error message
         if self.codeUnitKey is not None:
            unitStr = self.codeUnitKey.toStrepForDiag()
            if unitStr != "":
               lineParts.append( unitStr )
         if self.location is not None:
            linePartsForCodeUnit.extend(
               [ "line", f"{self.location.line}:{self.location.column}" ] )
         if self.functionName:
            linePartsForCodeUnit.append( f"(function '{self.functionName}')" )
         return linePartsForCodeUnit

      if self.codeUnitKey is not None:
         if self.codeUnitKey.isPoaWrapper():
            lineParts.extend( getLinePartsForPoaWrapper() )
         elif self.codeUnitKey.isOpenConfig():
            lineParts.extend( getLinePartsForOpenConfigFunction() )
         else:
            lineParts.extend( getLinePartsForCodeUnit() )
      else:
         lineParts.extend( getLinePartsForCodeUnit() )

      msg = "{}: {}".format( " ".join( lineParts ), self.msg )
      for codeMessage, codeSection in self.getCode( codeUnitMapping ):
         if codeMessage:
            msg += '\n' + codeMessage
         if codeSection:
            msg += '\n' + codeSection
      return msg

   def getCode( self, codeUnitMapping ):
      '''
      Generator rendering the code section for a given error implemented in the
      derived class. Multiple code sections may be relevant to single error
      so this is a generator.

      Return values:
      - codeMessage (str) message to prefix the code section
      - codeSection (str) the code section where the error occured, with this
                          erroneous part annotated.
      '''
      yield '', ''

class RcfUnknownError( CompilerDiagMessageBase ):
   """Holds an error of an unknown type. This is used when unexpected
   errors are generated. These do not specify a specific location or
   function name when the error happens which makes it overly broad
   and less useful for customers. Such errors should be used the least.
   """
   severity = Severity.error

   def __init__( self, msg ):
      super().__init__( None, None, None, msg )

   def render( self, codeUnitMapping ):
      return 'Error: ' + self.msg

class RcfSyntaxError( CompilerDiagMessageBase ):
   """ Holds a syntax error details as provided by Antlr's ErrorListener

   Constructor Args:
      recoginzer (Parser): Antlr Parser Instance.
      token: the unexpected token.
      line (int): line at which the error was found.
      col (int): column at which the error was found.
      msg (str): Antlr generated message.
      e (Antlr Exception): Exception details.
   """
   severity = Severity.error

   def __init__( self, codeUnitKey, recognizer, token, line, column, msg, e ):
      # Hold onto args for debugging
      self.recognizer = recognizer
      self.originalToken = token
      self.originalLine = line
      self.originalColumn = column
      self.originalMsg = msg
      self.e = e

      super().__init__(
         DiagLocation( line, column ), codeUnitKey, None, msg )

      self.previousToken = None
      self.nextToken = None

      if self.e and isinstance( self.e, UnsupportedFeatureToggle ):
         self.token = self.e.unsupportedCtx
         location = getLeftmostPointToken( self.e.unsupportedCtx )
         self.location = DiagLocation( location.line, location.column )
         return
      # Hum, not great
      tokenStream = recognizer._input # pylint: disable=protected-access
      self.previousToken = tokenStream.LT( -1 )
      self.token = token

      # If the token is for an EOF then it is often on the last line
      # which is not very useful code context to render.
      # Similarly when a semi colon is missing the blame will normally land
      # on a token on the next line.
      # In these cases shift the offending token to be at a point just after
      # the previous token so the rendered code context indicates a more useful
      # location

      if msg.startswith( "missing ';' at" ):
         self.msg = "missing ';'"
         self.shiftToken()
      elif token and token.type == Token.EOF:
         if self.previousToken is None:
            self.msg = "no text found"
            self.token = None
         else:
            self.shiftToken()
            # Rendering a blank line with an EOF marker is just noise.
            self.nextToken = None

   def shiftToken( self ):
      self.nextToken = self.token
      self.token = getRightmostPointToken( self.previousToken )
      self.location = DiagLocation( self.token.line, self.token.column )

   def getCode( self, codeUnitMapping ):
      if self.token is not None:
         # Render from the previous token to provide context of the last line as this
         # error may be at the start of a line. Render up to the next token
         # as this was the original token the error was for in the case where
         # the shiftToken was used.
         yield (
            '',
            renderCodeSection( codeUnitMapping, self.codeUnitKey,
                               partsToUnderline=self.token,
                               codeSectionStart=self.previousToken,
                               codeSectionEnd=self.nextToken )
         )
      else:
         yield '', ''

class RcfSymbolErrorBase( CompilerDiagMessageBase ):
   pass

class RcfResolutionError( RcfSymbolErrorBase ):
   """ Holds resolution error details as provided by Rcf symbolic phases.

   Constructor Arguments:
      function (RcfAst.Function): function where we found the error.
      offendingRef (RcfAst.Node): offending node in the AST (call or attribute).
      enclosingExpression (AstNode): The expression the reference
                                     was used in.
   """
   severity = Severity.error

   def __init__( self, function, offendingRef, enclosingExpression ):
      location = sourceLoc( offendingRef.context )
      if isinstance( offendingRef, RcfAst.Call ):
         offendingType = 'function'
         offendingRefName = offendingRef.funcName
      elif isinstance( offendingRef, RcfAst.Attribute ):
         offendingType = 'attribute'
         offendingRefName = offendingRef.name
      elif isinstance( offendingRef, RcfAst.Variable ):
         offendingType = 'variable'
         offendingRefName = offendingRef.name
      else:
         assert False, f"Unexpected AST node {offendingRef} for resolution error"
      msg = f"undefined reference to {offendingType} '{offendingRefName}'"
      super().__init__( location, function.codeUnitKey, function.name, msg )
      # Hold onto args for debugging
      self.function = function
      self.offendingRef = offendingRef
      self.enclosingExpression = enclosingExpression

   def getCode( self, codeUnitMapping ):
      enclosingExpressionStart, enclosingExpressionEnd = None, None
      if self.enclosingExpression:
         enclosingExpressionStart = getLeftmostPointToken(
            self.enclosingExpression.context )
         enclosingExpressionEnd = getRightmostPointToken(
            self.enclosingExpression.context )
      yield (
         '',
         renderCodeSection( codeUnitMapping, self.codeUnitKey,
                            partsToUnderline=self.offendingRef.context,
                            codeSectionStart=enclosingExpressionStart,
                            codeSectionEnd=enclosingExpressionEnd )
      )

class RcfFunctionNameDefinitionError( RcfSymbolErrorBase ):
   """ Holds details regarding definition phase errors for functions.

   It is assumed that RcfFunctionNameDefinitionError only comes from function
   definition. e.g a user defines a function called 'med', or a user re-defines the
   same function 'foo' twice.

   Constructor Arguments:
      function (Ast.Function): Ast node of the function where the error was found.
   """
   severity = Severity.error

   def __init__( self, function, existingSymbolNode=None ):
      functionDefinition = function.context.funcDef()
      existingFunctionDefinition = None
      if existingSymbolNode:
         if existingSymbolNode.context is None:
            msg = f"redefinition of builtin function '{function.name}'"
         else:
            msg = f"redefinition of function '{function.name}'"
            existingFunctionDefinition = existingSymbolNode.context.funcDef()
      else:
         msg = f"function name '{function.name}' conflicts with a language keyword"
      super().__init__(
         sourceLoc( functionDefinition ), function.codeUnitKey, None, msg )
      # Hold onto args for debugging
      self.function = function
      self.functionDefinition = functionDefinition
      self.existingSymbolNode = existingSymbolNode
      self.existingFunctionDefinition = existingFunctionDefinition

   def getCode( self, codeUnitMapping ):
      yield '', renderCodeSection(
         codeUnitMapping, self.codeUnitKey,
         partsToUnderline=self.functionDefinition )
      if self.existingFunctionDefinition:
         codeUnitKey = self.existingSymbolNode.codeUnitKey
         msg = 'Previously defined'
         if not codeUnitKey.isUnnamedCodeUnitKey():
            msg += f' in code unit {codeUnitKey.codeUnitName}'
         yield msg + ':', renderCodeSection(
            codeUnitMapping, codeUnitKey,
            partsToUnderline=self.existingFunctionDefinition )

class RcfFunctionMaxParametersDefinitionError( RcfSymbolErrorBase ):
   """ Holds details regarding definition phase errors for functions which specify
   too many function parameters.

   Constructor Arguments:
      function (Ast.Function): Ast node of the function where the error was found.
      numParamsSupported (int): maximum number of function parameters supported
   """
   severity = Severity.error

   def __init__( self, function, numParamsSupported ):
      numParamsDefined = len( function.funcParams )
      msg = ( f"function '{function.name}' has too many arguments "
              f"({numParamsDefined}), {numParamsSupported} supported" )
      funcParamsContext = function.context.funcParams()
      assert funcParamsContext
      super().__init__(
         sourceLoc( funcParamsContext ), function.codeUnitKey, None, msg )
      # Hold onto args for debugging
      self.function = function
      self.funcParamsContext = funcParamsContext
      self.numParamsSupported = numParamsSupported

   def getCode( self, codeUnitMapping ):
      startToken, endToken = getStartEndForFunctionSignature( self.function )
      yield '', renderCodeSection(
         codeUnitMapping, self.codeUnitKey,
         partsToUnderline=self.funcParamsContext,
         codeSectionStart=startToken,
         codeSectionEnd=endToken )

class RcfMismatchedArgumentCountError( RcfSymbolErrorBase ):
   """ Holds details regarding definition phase errors for functions which specify
   too many function parameters.

   Constructor Arguments:
      function (Ast.Function): Ast node of the function where the error was found.
      numParamsSupported (int): maximum number of function parameters supported
   """
   severity = Severity.error

   def __init__( self, function, call, parameterCount, argumentCount ):
      location = sourceLoc( call.context )
      msg = ( f"expected {parameterCount}, received {argumentCount} arguments when "
              f"calling function '{call.funcName}'" )
      super().__init__( location, function.codeUnitKey, function.name, msg )
      # Hold onto args for debugging
      self.function = function
      self.call = call

   def getCode( self, codeUnitMapping ):
      yield (
         '',
         renderCodeSection( codeUnitMapping, self.codeUnitKey,
                            partsToUnderline=self.call.context ) )

class RcfLabelDefinitionError( RcfSymbolErrorBase ):
   """ Holds details regarding definition phase errors for labels

   It is assumed that RcfLabelDefinitionError only comes from label
   definition errors.
   e.g a user defines a label "@foo" twice in a single function

   Constructor Arguments:
      function (Ast.Function): Ast node of the function where the error was found.
      redefiningLabel (Ast.Label): Ast object for the the redefining label
      originalLabel (Ast.Label): Ast object the first label defined label
   """
   severity = Severity.error

   def __init__( self, function, redefiningLabel, originalLabel ):
      token = redefiningLabel.token
      lname = redefiningLabel.name
      llnum = originalLabel.token.line
      msg = f"label '{lname}' already defined at line {llnum}"
      super().__init__(
         DiagLocation( token.line, token.column ), function.codeUnitKey,
         function.name, msg )
      # Hold onto args for debugging
      self.function = function
      self.redefiningLabel = redefiningLabel
      self.originalLabel = originalLabel

   def getCode( self, codeUnitMapping ):
      yield '', renderCodeSection(
         codeUnitMapping, self.codeUnitKey,
         partsToUnderline=self.redefiningLabel.token )
      yield 'Previously defined:', renderCodeSection(
         codeUnitMapping, self.codeUnitKey,
         partsToUnderline=self.originalLabel.token )

class RcfVariableNameDefinitionError( RcfSymbolErrorBase ):
   """ Holds details regarding definition phase errors for variable names

   It is assumed that RcfVariableNameDefinitionError only comes from variable
   definition errors.
   e.g a user defines a variable "$foo" twice in a single function

   Note that until we support general variable definition, all variables are
   defined from function parameters.

   Constructor Arguments:
      function (Ast.Function): Ast node of the function where the error was found.
      redefiningVariable (Ast.FunctionParam): Ast object for the the redefining
                                              variable
      originalVariable (Ast.FunctionParam): Ast object the first defined variable
   """
   severity = Severity.error

   def __init__( self, function, redefiningVariable, originalVariable ):
      msg = f"redefinition of variable '{redefiningVariable.name}'"
      super().__init__(
         sourceLoc( redefiningVariable.context ), function.codeUnitKey,
         function.name, msg )
      # Hold onto args for debugging
      self.function = function
      self.redefiningVariable = redefiningVariable
      self.originalVariable = originalVariable

   def getCode( self, codeUnitMapping ):
      startToken, endToken = getStartEndForFunctionSignature( self.function )
      yield '', renderCodeSection(
         codeUnitMapping, self.codeUnitKey,
         partsToUnderline=self.redefiningVariable.context,
         codeSectionStart=startToken,
         codeSectionEnd=endToken )
      yield 'Previously defined:', renderCodeSection(
         codeUnitMapping, self.codeUnitKey,
         partsToUnderline=self.originalVariable.context,
         codeSectionStart=startToken,
         codeSectionEnd=endToken )

class RcfVariableTypeDefinitionError( RcfSymbolErrorBase ):
   """ Holds details regarding definition phase errors for variable types

   It is assumed that RcfVariableTypeDefinitionError only comes from variable
   definition errors.
   e.g a user defines a variable with unknown type "foo_type"

   Note that until we support general variable definition, all variables are
   defined from function parameters.

   Constructor Arguments:
      function (Ast.Function): Ast node of the function where the error was found.
      variable (Ast.FunctionParam): Ast object for the variable
   """
   severity = Severity.error

   def __init__( self, function, variable ):
      vname = variable.name
      vtype = variable.typeStr
      msg = f"invalid type '{vtype}' when defining variable {vname}"
      super().__init__(
         sourceLoc( variable.context ), function.codeUnitKey, function.name, msg )
      # Hold onto args for debugging
      self.function = function
      self.variable = variable

   def getCode( self, codeUnitMapping ):
      startToken, endToken = getStartEndForFunctionSignature( self.function )
      yield '', renderCodeSection(
         codeUnitMapping, self.codeUnitKey,
         partsToUnderline=self.variable.context,
         codeSectionStart=startToken,
         codeSectionEnd=endToken )

class RcfExtResolutionError( RcfSymbolErrorBase ):
   """ Holds external resolution error details as provided by Rcf symbolic phases.

   Constructor Arguments:
      function (Ast.Function): Ast node of the function where the error was found.
      offendingSymbol (Ast.ExternalRef): Ast node of the external ref.
      enclosingExpression (AstNode): The expression the external reference
                                     was used in.
   """
   severity = Severity.error

   def __init__( self, function, offendingRef, enclosingExpression ):
      location = sourceLoc( offendingRef.context )
      msg = f"undefined reference to {offendingRef.etype} '{offendingRef.name}'"
      super().__init__( location, function.codeUnitKey, function.name, msg )
      # Hold onto args for debugging
      self.function = function
      self.offendingRef = offendingRef
      self.enclosingExpression = enclosingExpression

   def getCode( self, codeUnitMapping ):
      enclosingExpressionStart, enclosingExpressionEnd = None, None
      if self.enclosingExpression:
         enclosingExpressionStart = getLeftmostPointToken(
            self.enclosingExpression.context )
         enclosingExpressionEnd = getRightmostPointToken(
            self.enclosingExpression.context )
      yield (
         '',
         renderCodeSection( codeUnitMapping, self.codeUnitKey,
                            partsToUnderline=self.offendingRef.context,
                            codeSectionStart=enclosingExpressionStart,
                            codeSectionEnd=enclosingExpressionEnd )
      )

class ImmediateCollectionErrorBase( CompilerDiagMessageBase ):
   """ Base class for errors where the enclosing expression surrounding the
   immediate collection is printed, with the "offending" values within the
   collection underline.
   """
   def __init__( self, location, codeUnitKey, funcName, msg, enclosingExpression,
                 offendingContexts ):
      super().__init__( location, codeUnitKey, funcName, msg )
      self.enclosingExpression = enclosingExpression
      self.offendingContexts = offendingContexts

   def getCode( self, codeUnitMapping ):
      enclosingExpressionStart, enclosingExpressionEnd = None, None
      if self.enclosingExpression:
         enclosingExpressionStart = getLeftmostPointToken(
            self.enclosingExpression.context )
         enclosingExpressionEnd = getRightmostPointToken(
            self.enclosingExpression.context )
      yield (
         '',
         renderCodeSection(
            codeUnitMapping, self.codeUnitKey,
            partsToUnderline=list( self.offendingContexts ),
            codeSectionStart=enclosingExpressionStart,
            codeSectionEnd=enclosingExpressionEnd,
         )
      )

class RcfImmediateCollectionError( ImmediateCollectionErrorBase ):
   """ Holds immediate collection error details as provided by Rcf.

   Constructor Arguments:
      function (Ast.Function): Ast node of the function where the error was found.
      offendingValues (List of Ast.Constant): Constants in the collection that were
                                              invalid.
      collectionType (Ast.Constant.Type): type of the offending immediate collection
      enclosingExpression (AstNode): The expression the immediate collection was
                                     used in.
   """

   errMsgType = defaultdict( lambda: "set", {
      BT.ImmediateAsNumberSet: 'as number set',
      BT.ImmediateIntSet: 'integer set',
      BT.ImmediateCommunitySet: 'community set',
      BT.ImmediateExtCommunitySet: 'ext_community set',
      BT.ImmediateLargeCommunitySet: 'large_community set',
      BT.ImmediateIsisLevelSet: 'isis level set',
      BT.ImmediateResolutionRibList: 'resolution_ribs list',
      BT.AsPathImmediate: 'as path',
   } )

   severity = Severity.error

   def __init__( self, function, offendingValues, collectionType,
                 enclosingExpression ):
      contexts = [ v.context for v in offendingValues ]
      location = sourceLoc( contexts[ 0 ] )
      cType = self.errMsgType[ collectionType ]
      msg = f"invalid values in an immediate {cType}"
      super().__init__(
         location, function.codeUnitKey, function.name, msg, enclosingExpression,
         contexts )
      # Hold onto args for debugging
      self.function = function
      self.offendingValues = offendingValues
      self.collectionType = collectionType

class RcfRibTypeRedefinitionError( ImmediateCollectionErrorBase ):
   """ Holds resolution rib type redefinition error details as provided by Rcf
   symbolic phase.

   Constructor Arguments:
      function (Ast.Function): Ast node of the function where the error was found.
      offendingRibs (Ast.ExternalRef): Multiple ribs of the same rib_type
      ribType (str): Rib type
      enclosingExpression (AstNode): The expression the immediate collection was
                                     used in.
   """
   severity = Severity.error

   def __init__( self, function, offendingRibs, ribType, enclosingExpression ):
      assert len( offendingRibs ) > 0
      for rib in offendingRibs:
         assert rib.etype == ribType
      location = sourceLoc( offendingRibs[ 0 ].context )
      msg = f"redefinition of { ribType }"
      offendingContexts = [ rib.context for rib in offendingRibs ]
      super().__init__(
         location, function.codeUnitKey, function.name, msg, enclosingExpression,
         offendingContexts )
      # Hold onto args for debugging
      self.function = function
      self.offendingRibs = offendingRibs
      self.enclosingExpression = enclosingExpression

class RcfCycleError( CompilerDiagMessageBase ):
   """ Holds cycle error details from cycles detected after generating
   the symbol table

   Constructor Arguments:
      cycle (list of unicode string): list of function names
      representing a detected cycle
   """
   severity = Severity.error

   def __init__( self, cycle ):
      # Convert a list of functions representing a cycle into a human readable format
      cycleStr = " -> ".join( [ str( func.callerNode.name )
                                for func in cycle + [ cycle[ 0 ] ] ] )
      msg = f"cycle found in function callgraph '{cycleStr}'"
      super().__init__( None, None, None, msg )
      # Hold onto args for debugging
      self.cycle = cycle

   def getCode( self, codeUnitMapping ):
      firstDefinition = None
      for function in self.cycle:
         functionDefinition = function.callerNode.context.funcDef()
         codeUnitKey = function.callerNode.codeUnitKey
         msg = ''
         if not codeUnitKey.isUnnamedCodeUnitKey():
            msg += 'Code unit ' + codeUnitKey.codeUnitName
         definitionOutput = ( msg,
                              renderCodeSection(
                                 codeUnitMapping, codeUnitKey,
                                 partsToUnderline=functionDefinition )
         )
         if firstDefinition is None:
            firstDefinition = definitionOutput
         yield definitionOutput
         yield ( '',
                 renderCodeSection(
                    codeUnitMapping, codeUnitKey,
                    partsToUnderline=function.callSiteNode.context )
         )
      # render the first definition again to complete the loop.
      yield firstDefinition

class RcfGenericError( CompilerDiagMessageBase ):
   """ Holds generic errors where you give a message, and items need to be underlined

   Constructor Arguments:
      function (AstNode.Function): the function where we found the error.
      offendingRefs (AstNode): the AST node(s) of an invalid type.
      msg (str): the description of what went wrong.
      enclosingExpression (AstNode): (optional) The expression the typing error was
                                     found in.
   """
   severity = Severity.error

   def __init__( self, function, offendingRefs, msg, enclosingExpression=None ):
      offendingRef = offendingRefs
      if isinstance( offendingRefs, list ):
         offendingRef = offendingRefs[ 0 ]
      location = sourceLoc( offendingRef.context )
      super().__init__( location, function.codeUnitKey, function.name, msg )
      # Hold onto args for debugging
      self.function = function
      self.offendingRefs = offendingRefs
      self.enclosingExpression = enclosingExpression

   def getCode( self, codeUnitMapping ):
      if isinstance( self.offendingRefs, list ):
         context = [ node.context for node in self.offendingRefs ]
      else:
         context = self.offendingRefs.context
      enclosingExpressionStart, enclosingExpressionEnd = None, None
      if self.enclosingExpression:
         enclosingExpressionStart = self.enclosingExpression.context.start
         enclosingExpressionEnd = self.enclosingExpression.context.stop
      yield (
         '',
         renderCodeSection( codeUnitMapping, self.codeUnitKey,
                            partsToUnderline=context,
                            codeSectionStart=enclosingExpressionStart,
                            codeSectionEnd=enclosingExpressionEnd )
      )

class RcfTypingError( RcfGenericError ):
   """
   Holds typing errors found during the type binding phase.
   """

class RcfImmediateValueError( RcfGenericError ):
   """
   Holds immediate value validation errors found during the value
   validation phase.
   """

class DirectiveError( CompilerDiagMessageBase ):
   """ Holds directive errors found during the code analysis phase.

   Constructor Arguments:
      function (AstNode.Function): the function where we found the error.
      offendingRefs (AstNode): the AST node(s) of an invalid type.
      msg (str): the description of what went wrong.
   """
   severity = Severity.error

   def __init__( self, function, offendingRef, msg ):
      location = sourceLoc( offendingRef.context )
      super().__init__( location, function.codeUnitKey, function.name, msg )
      # Hold onto args for debugging
      self.function = function
      self.offendingRef = offendingRef

   def getCode( self, codeUnitMapping ):
      yield (
         '',
         renderCodeSection( codeUnitMapping, self.codeUnitKey,
                            partsToUnderline=self.offendingRef.context )
      )

class AdminAsSpecifiedWarning( CompilerDiagMessageBase ):
   """ Holds link bandwidth values with admin AS specified in ext_community set on
       RHS of assignment warnings found during the code analysis phase.

   Constructor Arguments:
      function (AstNode.Function): the function where we found the error.
      offendingRef (AstNode): the AST node of the offending immediate collection
      offendingValues (List of Ast.Constant): The specified admin AS values.
   """
   severity = Severity.warning

   def __init__( self, function, offendingRef, offendingValues ):
      msg = "Remove specified link bandwidth admin AS in ext_commmunity as it "\
            "will be discarded during assignment, admin AS determined at evaluation"
      contexts = [ v.context for v in offendingValues ]
      location = sourceLoc( contexts[ 0 ] )
      super().__init__( location, function.codeUnitKey, function.name, msg )
      # Hold onto args for debugging
      self.function = function
      self.offendingContexts = contexts
      self.offendingRef = offendingRef
      self.offendingValues = offendingValues

   def getCode( self, codeUnitMapping ):
      yield(
         '',
         renderCodeSection( codeUnitMapping, self.codeUnitKey,
                            partsToUnderline=list( self.offendingContexts ) )
      )

class MultipleLinkBandwidthWarning( CompilerDiagMessageBase ):
   """ Holds multiple link bandwidth in ext_community set on RHS of assignment
       warnings found during the code analysis phase.

   Constructor Arguments:
      function (AstNode.Function): the function where we found the error.
      offendingRef (AstNode): the AST node of the offending immediate collection
      offendingValues (List of Ast.CommunityValue): LINK-BANDWIDTH-AS entries in
                                                    the immediate collection.
   """
   severity = Severity.warning

   def __init__( self, function, offendingRef, offendingValues ):
      msg = f"multiple link bandwidth values in an immediate ext_community set for "\
            f"'{offendingRef.parentOperator}', all but one will be ignored"
      contexts = [ v.context for v in offendingValues ]
      location = sourceLoc( contexts[ 0 ] )
      super().__init__( location, function.codeUnitKey, function.name, msg )
      # Hold onto args for debugging
      self.function = function
      self.offendingContexts = contexts
      self.offendingRef = offendingRef
      self.offendingValues = offendingValues

   def getCode( self, codeUnitMapping ):
      yield(
         '',
         renderCodeSection( codeUnitMapping, self.codeUnitKey,
                            partsToUnderline=list( self.offendingContexts ) )
      )

class RcfNoEffectWarning( CompilerDiagMessageBase ):
   """ Holds a warning about statement having no effect.

   Constructor Arguments:
      function (AstNode.Function): the function where we found the warning.
      offendingRef (ParserContext): the parse context of the statement
      msg (str): the description of what went wrong.
   """
   severity = Severity.warning

   def __init__( self, function, offendingRef, msg ):
      location = sourceLoc( offendingRef )
      super().__init__( location, function.codeUnitKey, function.name, msg )
      # Hold onto args for debugging
      self.function = function
      self.offendingRef = offendingRef

   def getCode( self, codeUnitMapping ):
      yield (
         '',
         renderCodeSection( codeUnitMapping, self.codeUnitKey,
                            partsToUnderline=self.offendingRef )
      )

class RcfUnreachableCodeWarning( CompilerDiagMessageBase ):
   """ Holds a warning when there's unreachable code

   Constructor Arguments:
      function (AstNode.Function): the function where we found the warning.
      offendingRef (AstNode): the AST node(s) of an invalid type.
      msg (str): the description of what went wrong.
   """
   severity = Severity.warning

   def __init__( self, function, offendingRef, msg ):
      if isinstance( offendingRef, Token ):
         location = DiagLocation( offendingRef.line, offendingRef.column )
      else:
         location = sourceLoc( offendingRef )
      super().__init__( location, function.codeUnitKey, function.name, msg )
      # Hold onto args for debugging
      self.function = function
      self.offendingRef = offendingRef

   def getCode( self, codeUnitMapping ):
      yield (
         '',
         renderCodeSection( codeUnitMapping, self.codeUnitKey, self.offendingRef )
      )

class RouterIdAndIPv6ComparisonWarning( CompilerDiagMessageBase ):
   """ Holds the warning when there is a comparison between the router_id attribute
   and an immediate IPv6 value

   Constructor Arguments:
      function (AstNode.Function): the function where we found the warning.
      offendingRef (AstNode): the AST node(s) of the binary operation which
      compares router_id and immediate IPv6 value.
      msg (str): the description of what went wrong
   """
   severity = Severity.warning

   def __init__( self, function, offendingRef, msg ):
      location = sourceLoc( offendingRef )
      super().__init__( location, function.codeUnitKey, function.name, msg )
      # Hold onto args for debugging
      self.function = function
      self.offendingRef = offendingRef

   def getCode( self, codeUnitMapping ):
      yield (
         '',
         renderCodeSection( codeUnitMapping, self.codeUnitKey, self.offendingRef )
      )

class RcfSyntaxDiag( RcfDiagGeneric, DiagnosticErrorListener ):
   def __init__( self, codeUnitKey ):
      RcfDiagGeneric.__init__( self )
      self.codeUnitKey = codeUnitKey

   def syntaxError( self, recognizer, offendingSymbol, line, column, msg, e ):
      """ Callback when Antlr finds a syntax error. Record such error.

         Args:
            recoginzer (Parser): parser
            offendingSymbol token: the unexpected token
            line (int): line at which the error was found
            col (int): column at which the error was found
            msg (str): Antlr generated message
            e (Antlr Exception): Exception details (can be None, see RcfAstGen)
      """
      error = RcfSyntaxError( self.codeUnitKey, recognizer, offendingSymbol, line,
                              column, msg, e )
      self.add( error )

   def reportAmbiguity( self, recognizer, dfa, startIndex, stopIndex, exact,
                       ambigAlts, configs ):
      assert False, "Antlr Ambiguity reported"

   def reportAttemptingFullContext( self, recognizer, dfa, startIndex, stopIndex,
                                    conflictingAlts, configs ):
      assert False, "Antlr FullContext attempt reported"

   def reportContextSensitivity( self, recognizer, dfa, startIndex, stopIndex,
                                 prediction, configs ):
      assert False, "Antlr Context Sensitivity reported"

class RcfDiag( RcfDiagGeneric ):
   """ Collects all error and warning messages that occurs during the
   compilation.

   Attributes (from RcfDiagGeneric):
     - allErrors (list): List of messages with severity "Error"
     - allWarnings (list): List of messages with severity "Warning"
   """
   def __init__( self, strict ):
      RcfDiagGeneric.__init__( self )
      self.strict = strict

   def emptyFunctionDiag( self ):
      return RcfFunctionDiag( self.strict )

   def unknownError( self, msg ):
      error = RcfUnknownError( msg )
      self.add( error )

   def cycleError( self, cycle ):
      """ Callback when the compiler has discovered a function call while traverseing
      the callgraph generated from the symbol table.

         Args:
            cycle (list): cycle containing series of function calls
      """
      error = RcfCycleError( cycle )
      self.add( error )

   def functionNameDefinitionError( self, func, existingSymbolNode=None ):
      """ Callback when a user code uses a language keyword as a function name or
      redefines the same function twice.

      Args:
         func (Ast.Function): the function that redefines an existing symbol.
         existingSymbolNode (Ast.Function): the previous definition of the symbol
      """
      error = RcfFunctionNameDefinitionError( func,
                                              existingSymbolNode=existingSymbolNode )
      self.add( error )

class RcfFunctionDiag( RcfDiagGeneric ):
   def __init__( self, strict ):
      super().__init__()
      self.strict = strict

   def immediateCollectionError( self, func, offendingValues,
                                           collectionType, enclosingExpression ):
      """ Callback when a user code references an invalid immediate collection.

      Args:
         func (AstNode): the function where we found the error.
         offendingValues (list of Ast.Constant): nodes of the offending values
         collectionType (Ast.Constant.Type): type of the collection
         enclosingExpression (AstNode): The expression the immediate collection
                                        was used in.
      """
      error = RcfImmediateCollectionError( func, offendingValues,
                                                     collectionType,
                                                     enclosingExpression )
      self.add( error )

   def ribTypeRedefinitionError( self, function, offendingRibs, ribType,
                                 enclosingExpression ):
      """ Callback when a user code redefined a resolution rib type in an immediate
          resolution rib list

      Args:
         function (Ast.Function): Ast node of the function where the error was found.
         offendingRibs (Ast.ExternalRef): Ribs with the same type
         ribType (str): Rib type
         enclosingExpression (AstNode): The expression the immediate collection was
                                        used in.

      """
      error = RcfRibTypeRedefinitionError( function, offendingRibs, ribType,
                                           enclosingExpression )
      self.add( error )

   def extRefResolutionError( self, func, offendingExtRef, enclosingExpression ):
      """ Callback when a user code references a external construct that is not
      defined.

      Args:
         func (AstNode): the function where we found the error.
         offendingExtRef (AstNode.ExtRef): Ast node referencing an external
                                           construct.
         enclosingExpression (AstNode): The expression the external reference
                                        was used in.
      """
      error = RcfExtResolutionError( func, offendingExtRef, enclosingExpression )
      self.add( error )

   def resolutionError( self, func, offendingRef, enclosingExpression ):
      """ Callback when a user code references a construct that is not
      defined (path attribute, variable, or rcf function).

      Args:
         func (AstNode): the function where we found the error.
         offendingExtRef (AstNode): Ast node referencing something.
         enclosingExpression (AstNode): The expression the reference
                                        was used in.
      """
      error = RcfResolutionError( func, offendingRef, enclosingExpression )
      self.add( error )

   def functionMaxParametersDefinitionError( self, func, numParamsSupported ):
      """ Callback when a function defines too many function parameters.

      Args:
         func (Ast.Function): the function that defines too many function parameters
         numParamsSupported (int): maximum number of function parameters supported
      """
      error = RcfFunctionMaxParametersDefinitionError( func, numParamsSupported )
      self.add( error )

   def labelDefinitionError( self, func, redefiningLabel, originalLabel ):
      """ Callback when a user code redefines the same label twice in the same
      function.

      Args:
         func (Ast.Function): the function that redefines an existing symbol.
         redefiningLabel (Ast.Label): the label redefining an existing label
         originalLabel (Ast.Label): the orignal label
      """
      error = RcfLabelDefinitionError( func, redefiningLabel, originalLabel )
      self.add( error )

   def variableNameDefinitionError( self, func, redefiningVariable,
                                    originalVariable ):
      """ Callback when a user code redefines the variable by name twice.

      Args:
         func (Ast.Function): the function that redefines an existing symbol.
         redefiningVariable (Ast.Variable): the variable redefining an existing
                                            variable
         originalVariable (Ast.Variable): the original variable
      """
      error = RcfVariableNameDefinitionError( func, redefiningVariable,
                                              originalVariable )
      self.add( error )

   def variableTypeDefinitionError( self, func, variable ):
      """ Callback when a user code specifies a variable type which is unknown.

      Args:
         func (Ast.Function): the function that redefines an existing symbol.
         variable (Ast.Variable): the variable
      """
      error = RcfVariableTypeDefinitionError( func, variable )
      self.add( error )

   def mismatchedArgumentCountError( self, func, call, parameterCount,
                                     argumentCount ):
      """ Callback when a user causes a mismatch in the number of arguments to the
      function, and the function call

         Args:
            func (AstNode): the function where we found the error.
            call (AstNode): construction where we found the error.
            parameterCount(int): The number of expected Params
            argumentCount(int): The number of arguments in the function call
      """
      error = RcfMismatchedArgumentCountError( func, call, parameterCount,
                                               argumentCount )
      self.add( error )

   def typingError( self, func, offendingSymbol, what, enclosingExpression=None ):
      """ Callback when the compiler found a typing error during the type binding
      phase.

         Args:
            func (AstNode): the function where we found the error.
            offendingSymbol (AstNode): construction where we found the error.
            what (string): the custom message.
            enclosingExpression (AstNode): (optional) The expression the typing error
                                           was found in.
      """
      error = RcfTypingError( func, offendingSymbol, what,
                              enclosingExpression=enclosingExpression )
      self.add( error )

   def immediateValueError( self, func, offendingSymbol, what ):
      """ Callback when the compiler found a value error during the type binding
      phase.

         Args:
            func (AstNode): the function where we found the error.
            offendingSymbol (AstNode): construction where we found the error.
            what (string): the custom message.
      """
      error = RcfImmediateValueError( func, offendingSymbol, what )
      self.add( error )

   def directiveError( self, func, offendingSymbol, what ):
      """ Callback when the compiler found a DIRECTIVE statement.

         Args:
            func (AstNode): the function where we found the warning.
            offendingSymbol (AstNode): construct where we found the warning.
            what (string): the custom message.
      """
      error = DirectiveError( func, offendingSymbol, what )
      self.add( error )

   def adminAsSpecifiedWarning( self, func, offendingSymbol, offendingValues ):
      """ Callback when the compiler found link bandwidth values with admin AS
      specified in an ext_community set on the RHS of an assignment operator

         Args:
            func (AstNode): the function where we found the error.
            offendingSymbol (AstNode): construction where we found the error.
            offendingValues (List of Ast.Constant): The specified admin AS values.
      """
      warning = AdminAsSpecifiedWarning( func, offendingSymbol, offendingValues )
      self.add( warning )

   def multipleLinkBandwidthWarning( self, func, offendingSymbol, offendingValues ):
      """ Callback when the compiler found multiple link bandwidth values in an
      ext_community set on the RHS of an assignment operator

         Args:
            func (AstNode): the function where we found the error.
            offendingSymbol (AstNode): construction where we found the error.
            offendingValues (List of Ast.CommunityValue): the LINK-BANDWIDTH-AS
                                                          entries in the set
      """
      warning = MultipleLinkBandwidthWarning( func, offendingSymbol,
                                              offendingValues )
      self.add( warning )

   def noEffectWarning( self, func, offendingSymbol, what ):
      """ Callback when the compiler found a warning about a statement that has
      no effect.

         Args:
            func (AstNode): the function where we found the error.
            offendingSymbol (AstNode): construct where we found the error.
            what (string): the custom message.
      """
      warning = RcfNoEffectWarning( func, offendingSymbol, what )
      self.add( warning )

   def unreachableCodeWarning( self, func, offendingSymbol, what ):
      """ Callback when the compiler found a warning about unreachable code.

         Args:
            func (AstNode): the function where we found the error.
            offendingSymbol (AstNode): construct where we found the error.
            what (string): the custom message.
      """
      warning = RcfUnreachableCodeWarning( func, offendingSymbol, what )
      self.add( warning )

   def routerIdAndIPv6ComparisonWarning( self, func, offendingSymbol, what ):
      """ Callback when the compiler found a comparison between a router_id attribute
      and an immediate Ipv6 address.

         Args:
            func (AstNode): the function where we found the warning.
            offendingSymbol (AstNode): construct where we found the warning.
            what (string): the custom message.
      """
      warning = RouterIdAndIPv6ComparisonWarning( func, offendingSymbol, what )
      self.add( warning )
