# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from abc import abstractmethod, ABC as AbstractBaseClass
from enum import Enum, auto

class Severity( Enum ):
   warning = auto()
   error = auto()

   def __str__( self ):
      if self is Severity.warning:
         return "Warning"
      return "Error"

class GenericDiagMessageBase( AbstractBaseClass ):
   """
   Abstract type for diagnostic messages.
   """

   # Warning or Error
   severity = None

   def __init__( self, msg ):
      self.msg = msg
      assert self.msg
      assert isinstance( self.msg, str )

   def __str__( self ):
      return self.msg

   @abstractmethod
   def render( self, codeUnitMapping ):
      """
      Render a diagnostic message to a string.

      Args:
         codeUnitMapping (CodeUnitMapping):
            Mapping of line numbers in rcfText to their originating code units and
            relative line numbers within those code units.
      """

class RcfDiagGeneric:
   """ Collects all error and warning messages that occurs during the
   compilation.

   Attribute:
     - allErrors (list): List of messages with severity "Error"
     - allWarnings (list): List of messages with severity "Warning"
   """
   def __init__( self ):
      self.allErrors = []
      self.allWarnings = []

   def update( self, other ):
      self.allWarnings += other.allWarnings
      self.allErrors += other.allErrors

   def add( self, diagMessage ):
      if diagMessage.severity is Severity.error:
         self.allErrors.append( diagMessage )
      elif diagMessage.severity is Severity.warning:
         self.allWarnings.append( diagMessage )
      else:
         assert False, f"unexpected severity {diagMessage.severity}"

   def hasErrors( self ):
      """ Whether or not errors exist.
      """
      return len( self.allErrors ) > 0

   def hasWarnings( self ):
      """ Whether or not warnings exist.
      """
      return len( self.allWarnings ) > 0
