# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import defaultdict, namedtuple
import ctypes
from decimal import Decimal
from itertools import chain

import RcfAst
from RcfHelperTypes import BuiltinRibs, ResolutionRibTypes, tunnelRib
import RcfEvalType
from RcfMetadata import (
      RcfBuiltinTypes as BT,
      getRcfType,
      RcfTypeSystem,
   )
import RcfSymbol
import RcfTypeFuture as Rcf
import Tac

U64_MAX_VALUE = 0xFFFFFFFFFFFFFFFF
U32_MAX_VALUE = 0xFFFFFFFF
U16_MAX_VALUE = 0xFFFF
U6_MAX_VALUE = 0b111111
FLOAT_MAX = 3.402823466E+38
EXT_COMMUNITY_LBW_ASNUMBER_MASK = Rcf.Acl.CommunityValue( 0, 0x0000FFFF, 0 )
EXT_COMMUNITY_LBW_VALUE_MASK = Rcf.Acl.CommunityValue( 0, 0, 0xFFFFFFFF )

ExtCommWithAcl = namedtuple( "ExtCommWithAcl", ( "aclCommVal", "astNode" ),
                             defaults=( None, None ) )

class RcfImmediateValueValidationHelper:
   """ Validate that a given constant (value) is in the expected range of
   acceptable values.
   """
   def __init__( self, currentFunction, diags ):
      self.currentFunction = currentFunction
      self.diags = diags

   def validate( self, constant ):
      dispatchMap = {
         BT.Int: self._validateInt,
         BT.Bandwidth: self._validateBandwidth,
         BT.Prefix: self._validatePrefix,
         BT.IpAddress: self._validateIpAddress,
         BT.MacAddress: self._validateMacAddress,
         BT.AsNumber: self._validateAsNumber,
         BT.NonZeroAsNumber: self._validateNonZeroAsNumber,
         BT.ImmediateExtCommunityValue: self.validateExtCommunityValue,
      }

      # If there is no validation function defined for this type, validation is
      # not required.
      validator = dispatchMap.get( constant.resolveType() )
      if validator:
         validator( constant )

   def _validateInt( self, value, *, ctype="integer", minValue=-U64_MAX_VALUE,
                     maxValue=U64_MAX_VALUE ):
      if value.value < minValue:
         msg = f"invalid {ctype}: value is less than minimum value {minValue}"
         self.diags.immediateValueError( self.currentFunction, value, msg )
      elif value.value > maxValue:
         msg = f"invalid {ctype}: value is greater than maximum value {maxValue}"
         self.diags.immediateValueError( self.currentFunction, value, msg )

   def _validateBandwidth( self, value ):
      # We are not supporting negative floating point numbers
      # Bandwidth values are treated as bps in RCF
      bandwidthInBitsPerSecond = float( RcfImmediateValueHelper.bandwidthStrToNum(
         value.value, outputUnit='bps' ) )
      if bandwidthInBitsPerSecond < 0.0:
         msg = f"invalid float: value is less than minimum value {0.0}"
         self.diags.immediateValueError( self.currentFunction, value, msg )
      elif bandwidthInBitsPerSecond > FLOAT_MAX:
         msg = f"invalid float: value is greater than maximum value {FLOAT_MAX}"
         self.diags.immediateValueError( self.currentFunction, value, msg )

   def _validatePrefix( self, prefixValue ):
      try:
         Rcf.Arnet.IpGenPrefix( prefixValue.value )
      except ( ValueError, IndexError ):
         msg = "invalid IP prefix"
         self.diags.immediateValueError( self.currentFunction, prefixValue, msg )

   def _validateIpAddress( self, value ):
      try:
         ipaddr = Rcf.Arnet.IpGenAddr( value.value )
         if ipaddr.isAddrZero:
            raise ValueError
      except ( ValueError, IndexError ):
         msg = "invalid IP address"
         self.diags.immediateValueError( self.currentFunction, value, msg )

   def _validateMacAddress( self, value ):
      try:
         macaddr = Rcf.Arnet.EthAddr()
         macaddr.stringValue = value.value

         if macaddr.isZero:
            # The zero mac address is currenly used for optionality,
            # Before we allow matching with the this mac address
            # we should figure out how we want optionality to look
            # like in RCF. See bug745561
            raise ValueError
      except ( ValueError, IndexError ):
         msg = "invalid mac address"
         self.diags.immediateValueError( self.currentFunction, value, msg )

   @staticmethod
   def isValidAsNumber( asn, minValue=1, maxValue=U32_MAX_VALUE ):
      if isinstance( asn, ( RcfAst.Attribute, RcfAst.Variable ) ):
         return True
      # asn.value will be none if an invalid asDot value is used, e.g. 0.1
      if asn.value is None or asn.value < minValue or asn.value > maxValue:
         return False
      return True

   def _validateAsNumber( self, asn, *, minValue=0, maxValue=U32_MAX_VALUE ):
      # This validation is currently not used on operations that modify
      # the as_path.
      # As_path prepend <AsNumber> uses the validateAsPath validator
      ctype = "as number"
      if isinstance( asn, ( RcfAst.Attribute, RcfAst.Variable ) ):
         return
      elif asn.value is None:
         msg = f"invalid {ctype}"
         self.diags.immediateValueError( self.currentFunction, asn, msg )
      else:
         self._validateInt( asn, ctype=ctype, minValue=minValue, maxValue=maxValue )

   def _validateNonZeroAsNumber( self, asn ):
      # This validation is used for operations that modify the as_path.
      # as_path.replace( ..., <NonZeroAsNumber>)
      self._validateAsNumber( asn, minValue=1 )

   def validateExtCommunityValue( self, extComm ):
      """
      Returns an error message if the extended community is invalid.
      A return value of None means no errors were encoun
      """
      def validateExtCommunityAttributes( extComm ):
         section1, section2, section3 = extComm.sections
         if isinstance( section2, RcfAst.Attribute ):
            msg = ( f"attribute not supported for '{section1.value}' "
                     "extended community" )
            self.diags.immediateValueError( self.currentFunction, section2, msg )

         if ( section1.value != "LINK-BANDWIDTH-AS" and
              isinstance( section3, RcfAst.Attribute ) ):
            msg = ( f"attribute not supported for '{section1.value}' "
                     "extended community" )
            self.diags.immediateValueError( self.currentFunction, section3, msg )

      validateExtCommunityAttributes( extComm )

      sectionsInfo = RcfTypeSystem.extCommunitySections[ extComm.extCommType ]
      for section, sectionInfo in zip( extComm.sections, sectionsInfo ):
         sectionType = section.resolveType()
         if sectionType != sectionInfo.rcfType:
            # This type of error will be caught during the type binding
            # No point in doing further error checking here
            continue
         if not isinstance( section, RcfAst.Constant ):
            # This test only vallidates constant values. Type checking
            # for attributes will be done in the validateExtCommunityAttributes
            # function and the typebinding phase
            continue

         maxValue = 2 ** sectionInfo.bitLength - 1
         if sectionType == BT.Int:
            self._validateInt( section, minValue=0, maxValue=maxValue )
         elif sectionType == BT.AsNumber:
            self._validateAsNumber( section, minValue=0, maxValue=maxValue )
         elif sectionType == BT.IpAddress:
            self._validateIpAddress( section )
         elif sectionType == BT.Bandwidth:
            # Verify that the bandwidth val can fit in a F32 regardless of whether
            # it is converted to a bps or Bps value
            bandwidthValInBytes = RcfImmediateValueHelper.bandwidthStrToNum(
                  section.value, outputUnit='Bps' )
            bandwidthValInBits = RcfImmediateValueHelper.bandwidthStrToNum(
                  section.value, outputUnit='bps' )
            if ( bandwidthValInBytes < 0 or bandwidthValInBytes > FLOAT_MAX or
                 bandwidthValInBits < 0 or bandwidthValInBits > FLOAT_MAX ):
               msg = "Ext Comm bandwidth value out of range"
               self.diags.immediateValueError( self.currentFunction, section, msg )
         elif not isinstance( sectionType, RcfEvalType.Enum ):
            assert False, f"unknown type {sectionInfo.rcfType.displayName}"

class RcfCollectionValidationHelper:
   """ Validate that a given collection contents is in the expected range of
   acceptable values.
   """
   def __init__( self, currentFunction, currentExpression, diags ):
      self.currentFunction = currentFunction
      self.currentExpression = currentExpression
      self.diags = diags
      self.immediateHelper = RcfImmediateValueValidationHelper( currentFunction,
                                                                diags )
   def validate( self, collection ):
      dispatchMap = {
         BT.AsPathImmediate: self._validateAsPath,
         BT.ImmediateAsNumberSet: self._validateImmediateIntSet,
         BT.ImmediateIntSet: self._validateImmediateIntSet,
         BT.ImmediateCommunitySet: self._validateImmediateIntSet,
         BT.ImmediateIsisLevelSet: self._validateImmediateIntSet,
         BT.ImmediateExtCommunitySet: self._validateImmediateExtCommSet,
         BT.ImmediateLargeCommunitySet: self._validateImmediateLargeCommSet,
         BT.ImmediateResolutionRibList: self._validateImmediateResolutionRibList,
      }

      # If there is no validation function defined for this type, validation is
      # not required.
      validator = dispatchMap.get( collection.resolveType() )
      if validator:
         validator( collection )

   def _validateCollectionContentType( self, content, collectionType ):
      # The promoteToType is not set in the RcfTypeBinding phase,
      # because we would have to traverse the set multiple times.
      # We use this traversal to set the promoteToType
      for contentType in collectionType.contentTypes:
         if RcfTypeSystem.canPromote( content.evalType, contentType ):
            content.promoteToType = contentType
            break
      return content.resolveType() in collectionType.contentTypes

   def _validateAsPath( self, asPathValue ):
      validType = BT.AsPathImmediate
      offendingItems = []
      for asn in asPathValue.values:
         if not self._validateCollectionContentType( asn, validType ):
            offendingItems.append( asn )
         elif not RcfImmediateValueValidationHelper.isValidAsNumber( asn ):
            offendingItems.append( asn )

      if offendingItems:
         self.diags.immediateCollectionError(
            self.currentFunction, offendingItems,
            BT.AsPathImmediate, self.currentExpression )

   def _validateImmediateIntSet( self, immediateSetValue ):
      validType = immediateSetValue.resolveType()
      offendingItems = []
      for item in chain( immediateSetValue.values, immediateSetValue.ranges ):
         if not self._validateCollectionContentType( item, validType ):
            offendingItems.append( item )
         elif isinstance( item, RcfAst.Constant ):
            if not self._isImmediateSetIntValid( item ):
               offendingItems.append( item )
         elif isinstance( item, RcfAst.Range ):
            # Invalid ranges are handled as seperate errors
            self._validateImmediateSetRange( item )
         elif isinstance( item, RcfAst.Attribute ):
            continue
         else:
            assert False, "This item type is not allowed in sets: {}"\
                          .format( type( item ) )

      if offendingItems:
         self.diags.immediateCollectionError(
            self.currentFunction, offendingItems,
            validType, self.currentExpression )

   def _isImmediateSetIntValid( self, item ):
      return ( item.value is not None and item.value >= 0
         and item.value <= U32_MAX_VALUE )

   def _validateImmediateSetRange( self, rangeValue ):
      errors = False
      for boundValue in [ rangeValue.lowerBound, rangeValue.upperBound ]:
         if not self._isImmediateSetIntValid( boundValue ):
            errors = True
            msgIntType = ( 'integer' if boundValue.resolveType() == BT.Int
                           else 'as number' )
            msg = "invalid {} '{}'".format( msgIntType,
                                            boundValue.context.getText() )
            self.diags.immediateValueError( self.currentFunction, [ boundValue ],
                                            msg )

      if errors:
         # Don't validate the range if the values in the range are wrong
         return

      if rangeValue.lowerBound.value == rangeValue.upperBound.value:
         self.diags.immediateValueError( self.currentFunction, [ rangeValue ],
                                         "invalid range: values are equal" )
      elif rangeValue.lowerBound.value > rangeValue.upperBound.value:
         self.diags.immediateValueError(
            self.currentFunction, [ rangeValue ],
            "decrementing range: lower value is greater than upper value" )

   def _validateImmediateExtCommSet( self, immediateExtCommSetValue ):
      # For other community types multiple invalid values in a set are treated
      # as one compilation error where all of the invalid values are highlighted
      # at once together.
      # This approach is not taken for ext_community sets as there is
      # specific invalid parts to each immediate ext_community value in the set.
      # For example one may have an invalid ip address while another may have
      # an invalid color value. Since there is more information to expose per
      # invalid ext_community value in a set these are not collapsed into one
      # error but an error is produced for each.

      offendingItems = []
      # extComm.value itself must have 2 colon (":") separators
      for extComm in immediateExtCommSetValue.values:
         if not isinstance( extComm, RcfAst.CommunityValue ):
            offendingItems.append( extComm )
            continue
         if extComm.resolveType() not in BT.ImmediateExtCommunitySet.contentTypes:
            offendingItems.append( extComm )
            continue

         self.immediateHelper.validateExtCommunityValue( extComm )

      if offendingItems:
         self.diags.immediateCollectionError(
            self.currentFunction, offendingItems,
            BT.ImmediateExtCommunitySet, self.currentExpression )

   def _validateImmediateLargeCommSet( self, immediateLargeCommSetValue ):
      offendingComms = []
      for comm in immediateLargeCommSetValue.values:
         # largeComm.value is a tuple of three integers
         # each value must be between 0 and MAX_U32, inclusive
         if not isinstance( comm, RcfAst.Constant ):
            offendingComms.append( comm )
         elif comm.resolveType() not in BT.ImmediateLargeCommunitySet.contentTypes:
            offendingComms.append( comm )
         elif comm.value is None or len( comm.value ) != 3:
            offendingComms.append( comm )
         elif any( x < 0 or x > U32_MAX_VALUE for x in comm.value ):
            offendingComms.append( comm )
      if offendingComms:
         self.diags.immediateCollectionError(
            self.currentFunction, offendingComms,
            BT.ImmediateLargeCommunitySet, self.currentExpression )

   def _validateImmediateResolutionRibList( self, immediateResolutionRibList ):
      offendingValues = []
      ribTypes = defaultdict( list )
      validType = immediateResolutionRibList.resolveType()
      for rib in immediateResolutionRibList.values:
         # Verify all entries are resolution ribs
         if not self._validateCollectionContentType( rib, validType ):
            offendingValues.append( rib )
            continue
         ribTypes[ rib.etype ].append( rib )
      # Generate errors
      if offendingValues:
         self.diags.immediateCollectionError(
            self.currentFunction, offendingValues,
            BT.ImmediateResolutionRibList, self.currentExpression )
      for ribType, ribs in ribTypes.items():
         if len( ribs ) > 1:
            self.diags.ribTypeRedefinitionError( self.currentFunction, ribs, ribType,
                                                 self.currentExpression )


class RcfImmediateValueHelper:
   @staticmethod
   def floatToBin( num ):
      # this converts a float value to a 32 bit binary value
      strRep = bin(
       ctypes.c_uint.from_buffer(
           ctypes.c_float( num ) ).value ).replace( '0b', '' )
      return int( strRep, 2 )

   @staticmethod
   def bandwidthStrToBin( bandwidthStr, unit ):
      # this function converts an bandwidth string into its binary representation
      bandwidthNum = RcfImmediateValueHelper.bandwidthStrToNum( bandwidthStr, unit )
      return RcfImmediateValueHelper.floatToBin( bandwidthNum )

   @staticmethod
   def ip4StrToBin( ipAddr ):
      # this function converts an ip addr string into its binary representation
      ipAddrValue = 0
      for word in ipAddr.split( '.' ):
         ipAddrValue = ipAddrValue << 8
         ipAddrValue = ipAddrValue | int( word )
      return ipAddrValue

   @staticmethod
   def getEnumType( enumString ):
      typename = BT.enumStringToTypename.get( enumString )
      return getattr( RcfAst.Constant.Type, typename ) if typename is not None \
                                                       else None

   @staticmethod
   def getInterfaceType( value ):
      """Returns the interface constant type if the input conforms to an
         Arnet::IntfId, otherwise returns none.
      Arguments:
         value (str): the string representation of interface, short or long
      """
      value = str( value )
      try:
         Rcf.Arnet.IntfId( value )
         return RcfAst.Constant.Type.interface
      except ( ValueError, IndexError ):
         pass
      try:
         Rcf.Arnet.IntfId.fromShortName( value )
         return RcfAst.Constant.Type.interface
      except ( ValueError, IndexError ):
         pass
      return None

   @staticmethod
   def communityValueTypeToEvalType( typ ):
      mapping = {
         RcfAst.CommunityValue.Type.immediateExtCommunityValue:
            BT.ImmediateExtCommunityValue
      }
      evalType = mapping.get( typ )
      assert evalType, "Unsupported type for CommunityValue AST node : %s" % typ
      return evalType

   @staticmethod
   def constantTypeToEvalType( typ ):
      mapping = {
         RcfAst.Constant.Type.integer: BT.Int,
         RcfAst.Constant.Type.bandwidth: BT.Bandwidth,
         RcfAst.Constant.Type.boolean: BT.Boolean,
         RcfAst.Constant.Type.trilean: BT.Trilean,
         RcfAst.Constant.Type.prefix: BT.Prefix,
         RcfAst.Constant.Type.asDot: BT.AsDot,
         RcfAst.Constant.Type.asNumber: BT.AsNumber,
         RcfAst.Constant.Type.interface: BT.Interface,
         RcfAst.Constant.Type.ipAddress: BT.IpAddress,
         RcfAst.Constant.Type.macAddress: BT.MacAddress,
         RcfAst.Constant.Type.esi: BT.Esi,
         RcfAst.Constant.Type.immediateCommunityValue: BT.ImmediateCommunityValue,
         RcfAst.Constant.Type.immediateLargeCommunityValue:
            BT.ImmediateLargeCommunityValue,
         RcfAst.Constant.Type.isisLevelValue: BT.ImmediateIsisLevel,
         RcfAst.Constant.Type.none: BT.NoneType,
      }
      evalType = mapping.get( typ )
      if evalType is None:
         # This can happen for enum types
         evalType = getRcfType( typ )
      return evalType

   @staticmethod
   def collectionTypeToEvalType( typ ):
      mapping = {
         RcfAst.Collection.Type.AsPathImmediate: BT.AsPathImmediate,
         RcfAst.Collection.Type.ImmediateSet: BT.ImmediateSet,
         RcfAst.Collection.Type.ImmediateList: BT.ImmediateList,
         RcfAst.Collection.Type.empty: BT.Empty,
      }
      return mapping[ typ ]

   @staticmethod
   def bandwidthStrToNum( bandwidthStrep, outputUnit ):
      """Converts a bandwidth string to the numerical equivalent in the requested
      unit of 'bps' or 'Bps'. Note that in the context of extended communities
      bandwidth values are BYTES per second not BITS per second.
      """
      assert outputUnit in [ 'bps', 'Bps' ]
      # multiplierDict holds the conversion for unit prefixes (E.g 'Kilo' )
      multiplierDict = { 'K': 1000.0, 'M': 1.0e+6, 'G': 1.0e+9 }
      multiplier = Decimal( 1.0 )
      bandwidthUnits = bandwidthStrep.lstrip( '0123456789.' )
      bandwidthStrep = bandwidthStrep.rstrip( 'KMGBbps' )
      if bandwidthUnits.endswith( 'bps' ) and outputUnit == 'Bps':
         # BITS per second needs to be converted to BYTES per second
         multiplier /= Decimal( 8.0 )
      elif bandwidthUnits.endswith( 'Bps' ) and outputUnit == 'bps':
         # BYTES per second needs to be converted to BITS per second
         multiplier *= Decimal( 8.0 )
      if bandwidthUnits[ 0 ] in multiplierDict:
         # Here we have a K | M | G prefix
         multiplier *= Decimal( multiplierDict[ bandwidthUnits[ 0 ] ] )
      return Decimal( bandwidthStrep ) * multiplier

   @staticmethod
   def bandwidthNormalize( bandwidthStrep, outputUnit ):
      """Converts a bandwidth string to the equivalent bandwidth string in the
      requested unit of 'bps' or 'Bps'. This will also maximize the unit prefix.
      """
      bandwidthNum = RcfImmediateValueHelper.bandwidthStrToNum( bandwidthStrep,
                                                                outputUnit )
      # multiplierDict holds the conversion for unit prefixes (E.g 'Kilo' )
      multiplierDict = { 'G': 1.0e+9, 'M': 1.0e+6, 'K': 1.0e+3 }
      for prefix, value in multiplierDict.items():
         if bandwidthNum > value:
            bandwidthNum /= Decimal( value )
            return str( bandwidthNum ) + prefix + outputUnit
      return str( bandwidthNum ) + outputUnit

class AetValueGen:
   aetTriStateNameToValueMap = {
      'unknown': Rcf.Eval.TriStateBoolValueType.RcfUnknown,
      'true': Rcf.Eval.TriStateBoolValueType.RcfTrue,
      'false': Rcf.Eval.TriStateBoolValueType.RcfFalse,
   }
   aetTriStateValueToNameMap = { v: k for k, v in aetTriStateNameToValueMap.items() }

   def __init__( self, aetGenVisitor ):
      self.aetGenVisitor = aetGenVisitor
      self.valueTypeToAetBuild = {
         BT.AsDot: self.buildIntValue,
         BT.Boolean: self.buildTriStateBoolValue,
         BT.Int: self.buildIntValue,
         BT.Bandwidth: self.buildBandwidthValue,
         BT.Prefix: self.buildPrefixValue,
         BT.Trilean: self.buildTriStateBoolValue,
         BT.AsNumber: self.buildIntValue,
         BT.NonZeroAsNumber: self.buildIntValue,
         BT.Interface: self.buildInterfaceValue,
         BT.IpAddress: self.buildIpAddressValue,
         BT.MacAddress: self.buildMacAddressValue,
         BT.Esi: self.buildEsiValue,
         BT.RoaTable: self.buildRoaTableImmediate,
         BT.ImmediateLargeCommunityValue: self.buildLargeCommunityValue,
         BT.PrefixList: self.buildPrefixListTypes,
         BT.ImmediateExtCommunityValue: self.buildExtCommunityValue,
         BT.ExtCommunityNoneType: self.buildExtCommunityValueNoneType,
         BT.IpAddrNoneType: self.buildIpAddrNone,
      }

      for enumTypename in BT.enumTypes:
         enumType = getRcfType( enumTypename )
         self.valueTypeToAetBuild[ enumType ] = self.buildEnumValue

   def build( self, value ):
      return self.valueTypeToAetBuild[ value.resolveType() ]( value )

   def buildIpAddressValue( self, value ):
      if isinstance( value, RcfAst.Attribute ):
         aetIpAddress = self.aetGenVisitor.visit( value )( value )
      else:
         constant = value
         val = constant.value
         ipAddr = Rcf.Arnet.IpGenAddr( str( val ) )
         errMsg = "IP address immediate value cannot be zero: %s" % str( val )
         assert not ipAddr.isAddrZero, errMsg
         aetIpAddress = Rcf.Eval.IpAddressImmediate( ipAddr )
      return aetIpAddress

   def buildMacAddressValue( self, value ):
      if isinstance( value, RcfAst.Attribute ):
         aetMacAddress = self.aetGenVisitor.visit( value )( value )
      else:
         constant = value
         val = constant.value
         macAddr = Rcf.Arnet.EthAddr()
         macAddr.stringValue = str( val )
         # The zero mac address is currenly used for optionality,
         # Before we allow matching with the this mac address
         # we should figure out how we want optionality to look
         # like in RCF. See bug745561
         errMsg = "Mac address immediate value cannot be zero: %s" % str( val )
         assert not macAddr.isZero, errMsg
         aetMacAddress = Rcf.Eval.MacAddressImmediate( macAddr )
      return aetMacAddress

   def buildEsiValue( self, value ):
      if isinstance( value, RcfAst.Attribute ):
         aetEsi = self.aetGenVisitor.visit( value )( value )
      else:
         constant = value
         val = constant.value
         esi = Rcf.Evpn.Esi()
         esi.stringValue = str( val )
         aetEsi = Rcf.Eval.EsiImmediate( esi )
      return aetEsi

   def buildInterfaceValue( self, value ):
      """Returns the immediate Interface representation of a value.
         The value must conform to an Arnet::IntfId. We assume it will
         as the AST phase should already have verified this.
      Arguments:
         value (str): the string reperesentation of the itnerface, short or long
      """
      interfaceString = str( value.value )
      intfId = None
      try:
         intfId = Rcf.Arnet.IntfId( interfaceString )
      except ( ValueError, IndexError ):
         pass
      try:
         intfId = Rcf.Arnet.IntfId.fromShortName( interfaceString )
      except ( ValueError, IndexError ):
         pass
      aetInterface = Rcf.Eval.InterfaceImmediate( intfId )
      return aetInterface

   def buildIntValue( self, value ):
      if isinstance( value, RcfAst.Attribute ):
         aetInt = self.aetGenVisitor.visit( value )( value )
      else:
         constant = value
         aetInt = Rcf.Eval.IntImmediate( abs( constant.value ), constant.value < 0 )
      return aetInt

   def buildPrefixValue( self, value ):
      if isinstance( value, RcfAst.Attribute ):
         aetPrefix = self.aetGenVisitor.visit( value )( value )
      else:
         constant = value
         val = constant.value
         pfx = Rcf.Arnet.IpGenPrefix( str( val ) )
         aetPrefix = Rcf.Eval.IpPrefixImmediate( pfx )
      return aetPrefix

   def buildBandwidthValue( self, value ):
      if isinstance( value, RcfAst.Attribute ):
         aetBandwidth = self.aetGenVisitor.visit( value )( value )
      else:
         constant = value
         val = constant.value
         bandwidthNum = float( RcfImmediateValueHelper.bandwidthStrToNum(
            val, outputUnit='bps' ) )
         aetBandwidth = Rcf.Eval.BandwidthImmediate( bandwidthNum )
      return aetBandwidth

   def buildEnumValue( self, value ):
      if isinstance( value, RcfAst.Attribute ):
         aetNode = self.aetGenVisitor.visit( value )( value )
      else:
         typ = value.type
         val = value.value
         aetNode = Rcf.Eval.IntImmediate( getRcfType( typ ).valueDict[ val ], False )
      return aetNode

   def buildTriStateBoolValue( self, constant ):
      aetTriStateValue = AetValueGen.aetTriStateNameToValueMap[ constant.value ]
      return Rcf.Eval.TriStateBoolImmediate( aetTriStateValue )

   def buildExtCommunityValue( self, extComm ):
      """ This function is responsible for building an extended community value.
      These communitie values are not part of immediate extended community sets, but
      are on the rhs of an operation.
      For example `rt_membership.route_target is ROUTE_TARGET:100:100`
      """
      validTypes = [ RcfAst.CommunityValue.Type.immediateExtCommunityValue ]
      assert extComm.type in validTypes, ( "Unexpected type {} when building "
                                           "immediate extended community AET"
                                           .format( extComm.type ) )

      aclCommVal = AetExtCommHelper.buildExtCommAclCommValue( extComm )
      immediateNode, _ = AetExtCommHelper.buildExtCommunityValueImpl(
         extComm, ExtCommWithAcl( aclCommVal, extComm ) )
      return immediateNode

   def buildExtCommunityValueNoneType( self, extComm ):
      """ This function is used for comparison agains none type extended communities
      For example: `rt_membership.route_target is none`
      """
      return Rcf.Eval.CommunityValueImmediate(
         Rcf.Eval.CommunityValues.invalidExtendedCommunity() )

   def buildLargeCommunityValue( self, astNode ):
      largeCommParts = astNode.value
      return Rcf.Acl.CommunityValue( largeCommParts[ 0 ],
                                       largeCommParts[ 1 ],
                                       largeCommParts[ 2 ] )

   def buildRoaTableImmediate( self, value ):
      index = self.aetGenVisitor.extRefIndex( value )
      return Rcf.Eval.RoaTableImmediate( index, str( value.name ) )

   def buildPrefixListTypes( self, value ):
      index = self.aetGenVisitor.extRefIndex( value )
      if "dynamic_prefix_list" == value.etype:
         aetType = Rcf.Eval.DynamicPrefixListImmediate
         args = ()
      else:
         assert "prefix_list_" in value.etype
         aetType = Rcf.Eval.PrefixListImmediate
         args = ( value.isIpV4, )
      return aetType( str( value.name ), index, *args )

   def buildIpAddrNone( self, value ):
      # We do not support building 'none' values.
      return None

class AetExtCommHelper:
   @staticmethod
   def buildExtCommAclCommValue( commVal ):
      """
      Turn an astNode for an extended community value into an Acl::CommunityValue
      """
      CType = RcfAst.Constant.Type
      sectionsInfo = RcfTypeSystem.extCommunitySections[ commVal.extCommType ]
      commInt = 0
      for section, sectionInfo in zip( commVal.sections, sectionsInfo ):
         if not isinstance( section, RcfAst.Constant ):
            # In this function we should only build constant values
            # Attributes and Variables should be added as dynamic components
            continue
         value = None
         if section.type == CType.integer:
            value = section.value
         elif section.type == CType.asDot:
            assert sectionInfo.bitLength == 32
            value = section.value
         elif section.type == CType.asNumber:
            assert sectionInfo.bitLength == 32
            value = section.value
         elif section.type in BT.enumTypes:
            value = sectionInfo.rcfType.valueDict[ section.value ]
         elif section.type == CType.ipAddress:
            assert sectionInfo.bitLength == 32
            value = RcfImmediateValueHelper.ip4StrToBin( section.value )
         elif section.type == CType.bandwidth:
            assert sectionInfo.bitLength == 32
            value = RcfImmediateValueHelper.bandwidthStrToBin(
                  section.value, 'Bps' )

         assert value is not None, f"unknown type {sectionInfo.rcfType.name}"

         maxVal = ( 1 << sectionInfo.bitLength ) - 1
         assert value <= maxVal, f"Unexpected value {value}"
         commInt |= value << sectionInfo.offset

      part1 = ( commInt >> 32 ) & 0xFFFFFFFF
      part2 = commInt & 0xFFFFFFFF

      return Rcf.Acl.CommunityValue( 0, part1, part2 )

   @staticmethod
   def extCommunityIsLBW( aclCommVal ):
      """
      Check that looks at an Acl::CommunityValue object and determines
      if it's a LINK-BANDWITH-AS extended community
      """
      assert isinstance( aclCommVal, Tac.Type( 'Acl::CommunityValue' ) )
      mapping = BT.enumMappings[ 'ImmediateExtCommunity' ].stringToValue
      extCommType = ( aclCommVal.part1 & 0xFFFF0000 ) >> 16
      return extCommType == mapping[ 'LINK-BANDWIDTH-AS' ]

   @staticmethod
   def opIntroducesNewElement( rhsNode ):
      """
      This function is used to determine if an rhsNode is on the RHS of an
      operation where a new element is introduced to the LHS
      """
      return rhsNode.parentOperator in [ 'add', '=' ]

   @staticmethod
   def handleDynamicCommVal( aetNode, dynamicAttribute, mask ):
      """
      This function returns an object of type CommunityValueDynamic wrapped around an
      object of type `CommunityValue` with a pointer to the dynamicAttribute, whose
      getVal returns a value of type Acl:CommunityValue
      """
      additionalArgs = ( dynamicAttribute, mask )
      return Rcf.Eval.CommunityValueDynamic( aetNode, *additionalArgs )

   @staticmethod
   def handleDynamicBandwidth( aetNode, dynamicAttribute, aclCommVal, offset=0 ):
      """
      This function returns an object of type CommunityValueDynamic wrapped around an
      object of type CommunityValue with a pointer to the dynamicAttribute. This
      bandwidth type dynamicAttribute should further be wrapped in a
      DynamicBandwidthWrapper for compatibility.
      offset == 64 -> dynamic bandwidth for part0
      offset == 32 -> dynamic bandwidth for part1
      offset == 0 -> dynamic bandwidth for part2
      """
      # This assert ensures that we only create a BandwidthToCommunityValue AET
      # node when we are in the context of a link-bandwidth extended community.
      # This is because the BandwidthToCommunityValue AET node converts from bps
      # to Bps and we cannot guarantee that future *community types which somehow
      # encode a bandwidth like field will do so in Bps as opposed to bps.
      assert AetExtCommHelper.extCommunityIsLBW( aclCommVal )
      dynamicBandwidthWrapper = Rcf.Eval.BandwidthToCommunityValue(
                                       dynamicAttribute.aetType(), offset )
      return AetExtCommHelper.handleDynamicCommVal( aetNode, dynamicBandwidthWrapper,
                                                    EXT_COMMUNITY_LBW_VALUE_MASK )

   @staticmethod
   def lbwExtCommunityHasAsnWildcard( aclCommVal ):
      """
      Simple helper function to test if the ASN bits of the LBW extended community
      are set to zero
      """
      assert AetExtCommHelper.extCommunityIsLBW( aclCommVal )
      return ( aclCommVal.part1 & 0x0000FFFF ) == 0

   @staticmethod
   def getDynamicAsnAttr( rhsNode, aclCommVal ):
      """
      Function that handles all of the edge cases to retrieve the dynamic asn
      attribute for the LBW extended community
      """
      assert hasattr( rhsNode, "parentOperator" )
      assert hasattr( rhsNode, "parentLhsSymbol" )
      assert isinstance( rhsNode.parentLhsSymbol, RcfSymbol.Symbol )

      # When introducing new elements we must always use the adminAs for the asnumber
      # bits in the link-bandwidth extended community.
      if AetExtCommHelper.opIntroducesNewElement( rhsNode ):
         return Rcf.Eval.Bgp.AdminAs()

      # We should return `None` when the aclCommVal does not contain a dynamic as.
      # This will tell the caller that no dynamic community should be generator.
      if not AetExtCommHelper.lbwExtCommunityHasAsnWildcard( aclCommVal ):
         return None

      # Wildcard lbw asn values need to be populated at runtime from
      # the lbw asn value on the LHS
      aetType = rhsNode.parentLhsSymbol.aetType()
      rcfType = rhsNode.parentLhsSymbol.rcfType
      if rcfType == BT.ExtCommunity:
         # Add a wrapper that gets the link-bandwidth extended community
         # from the community set
         return Rcf.Eval.ExtCommSetToLinkBandwidthExtCommValue( aetType )
      elif rcfType == BT.ExtCommunityValue:
         # The attribute itself can be used for getting the asn bits.
         return aetType
      else:
         assert False, f"unexpected type: {rcfType.name}"

   @staticmethod
   def buildExtCommunityValueImpl( rhsNode, extComm, *, cachable=True ):
      """
      Function that takes an extended community value, and turns it into the
      corresponding AET node. This includes dynamic parts such as attributes and
      wildcarded.
      args:
         rhsNode: value that is on the rhs of an operation.
               For example:
                  - ImmediateExtCommunityValue
                  - ImmediateExtCommunitySet
         extComm: A named tuple containing the Acl::ComunityaValue, and the ast node
         cachable: whether the extended community can be cached
      """
      immediateNode = Rcf.Eval.CommunityValueImmediate( extComm.aclCommVal )

      # Go thorugh each of the sections, if they are an attribute, we create a
      # dynamic value, and prepend it to the immediateNode
      extCommType = extComm.astNode.sections[ 0 ].value
      sectionsInfo = RcfTypeSystem.extCommunitySections[ extCommType ]
      for section, sectionInfo in zip( extComm.astNode.sections, sectionsInfo ):
         if not isinstance( section, RcfAst.Attribute ):
            continue

         cachable = False
         sectionType = section.resolveType()
         assert sectionType == sectionInfo.rcfType
         # Other attributes will be added here in the future.
         if sectionType == BT.Bandwidth:
            assert sectionInfo.bitLength == 32
            immediateNode = AetExtCommHelper.handleDynamicBandwidth(
               immediateNode, section.symbol, extComm.aclCommVal,
               offset=sectionInfo.offset )
         elif not isinstance( sectionType, RcfEvalType.Enum ):
            assert False, f"unknown type {sectionInfo.rcfType.displayName}"

      # link-bandwidth extended communities need to be treated in a special way
      # The asn section of this community can have two special meanings:
      # - The first is a wildcard, where the asn part is left 0. In this case
      #   we take the asn part from the lhs and subsitute it in
      # - In the second case the extended community is part of an operation
      #   That modifies the existing lbw. In this case we must take the adminAs
      #   and subsitute it.
      if AetExtCommHelper.extCommunityIsLBW( extComm.aclCommVal ):
         asnAttr = AetExtCommHelper.getDynamicAsnAttr( rhsNode, extComm.aclCommVal )
         if asnAttr:
            cachable = False
            immediateNode = AetExtCommHelper.handleDynamicCommVal( immediateNode,
               asnAttr, EXT_COMMUNITY_LBW_ASNUMBER_MASK )

      return immediateNode, cachable

class AetCollectionGen:
   def __init__( self, aetGenVisitor ):
      self.aetGenVisitor = aetGenVisitor
      self.valueTypeToAetBuild = {
         BT.AsPathImmediate: self.buildAsPathValue,
         BT.ImmediateAsNumberSet: self.buildImmediateIntSet,
         BT.ImmediateIntSet: self.buildImmediateIntSet,
         BT.ImmediateCommunitySet: self.buildImmediateIntSet,
         BT.ImmediateIsisLevelSet: self.buildImmediateIntSet,
         BT.ImmediateExtCommunitySet: self.buildImmediateExtCommSet,
         BT.ImmediateLargeCommunitySet: self.buildImmediateLargeCommSet,
         BT.ImmediateResolutionRibList: self.buildImmediateResolutionRibList,
         BT.Empty: self.buildEmpty,
      }

   def build( self, value ):
      buildFn = self.valueTypeToAetBuild.get( value.resolveType() )
      if buildFn is None:
         return self.aetGenVisitor.visit( value )( value )
      return buildFn( value )

   def buildAsPathValue( self, astNode ):
      # Construct a IntListImmediate represented with a linked list, note we build
      # the list backwards.
      head = None
      cachable = True
      for asn in reversed( astNode.values ):
         if isinstance( asn, ( RcfAst.Attribute, RcfAst.Variable ) ):
            # The AsPath representation cannot be cached across evaluations if it
            # contains an attribute or variable value from a given evaluation.
            cachable = False
         asnVal = self.build( asn )
         head = Rcf.Eval.IntLinkedList( asnVal, head )
      return Rcf.Eval.IntListImmediate( head, cachable )

   # pylint: disable=inconsistent-return-statements
   @staticmethod
   def immediateIntSetKeyFunction( key ):
      # This function is used to sort lists with a mix of attributes and constants.
      # this function generates a tuple that can be used to sort a set with mixed
      # attributes, constants and ranges. The first number in the tuple is chosen
      # arbitrarily, and the only purpose is to prevent comparison between
      # incompatible types
      if isinstance( key, RcfAst.Attribute ):
         return 1, key.name
      if isinstance( key, RcfAst.Constant ):
         return 2, key.value
      if isinstance( key, RcfAst.Range ):
         return 2, key.lowerBound, key.upperBound
      # this case is only used in the test library
      if isinstance( key, RcfAst.CommunityValue ):
         commType, section1, section2 = key.getSectionValues()
         mapping = BT.enumMappings[ 'ImmediateExtCommunity' ].stringToValue
         commType = mapping[ commType ]

         return 3, *( commType, section1, section2 )
      assert False, "unexpected RcfAsType"

   def buildImmediateIntSet( self, astNode ):
      head = None
      cachable = True
      aetValueGen = AetValueGen( aetGenVisitor=self.aetGenVisitor )
      # We sort the list to get the constant values sorted.
      # We build the linked list of objects representing the int set backwards
      nodes = { self.immediateIntSetKeyFunction( x ): x for x in astNode.values }
      nodes = [ nodes[ i ] for i in sorted( nodes, reverse=True ) ]
      for node in nodes:
         assert isinstance( node, ( RcfAst.Constant, RcfAst.Attribute ) )
         assert node.resolveType() in astNode.resolveType().contentTypes, (
            "Unexpected type {} when building {} AET".format(
               node.resolveType().name, astNode.resolveType().name ) )
         if isinstance( node, RcfAst.Attribute ):
            cachable = False

         intVal = aetValueGen.buildIntValue( node )
         head = Rcf.Eval.IntLinkedList( intVal, head )
      return Rcf.Eval.IntSetImmediate( head, cachable )

   def buildRangeLists( self, astNode ):
      lowerBoundHead = None
      upperBoundHead = None
      nodes = [ ( node.lowerBound, node.upperBound ) for node in astNode.ranges ]

      # We build the linked lists of objects representing the ranges backwards
      for lowerBoundNode, upperBoundNode in reversed( nodes ):
         assert isinstance( lowerBoundNode, RcfAst.Constant )
         lowerVal = Rcf.Eval.IntImmediate( abs( lowerBoundNode.value ),
                                           bool( lowerBoundNode.value < 0 ) )
         lowerBoundHead = Rcf.Eval.IntLinkedList( lowerVal, lowerBoundHead )
         assert isinstance( upperBoundNode, RcfAst.Constant )
         upperVal = Rcf.Eval.IntImmediate( abs( upperBoundNode.value ),
                                           bool( upperBoundNode.value < 0 ) )
         upperBoundHead = Rcf.Eval.IntLinkedList( upperVal, upperBoundHead )

      cachable = True # we assert above that the contents are constant ints
      return ( Rcf.Eval.IntListImmediate( lowerBoundHead, cachable ),
               Rcf.Eval.IntListImmediate( upperBoundHead, cachable ) )

   def buildExtCommSetCollection( self, extCommSet ):
      """
      Function that takes an extended community set astNode, and produces an
      iterable list that is used for processing the individual extended community
      values. This list has all duplicate values removed, and is sorted.

      example:
      input:
         { COLOR::100,
           LINK-BANDWIDTH-AS:100:source_session.interface.bandwidth,
           COLOR::100,
           LINK-BANDWIDTH-AS:100:10Mbps,
           LINK-BANDWIDTH-AS:100:source_session.interface.bandwidth };
      output:
         { COLOR::100,
           LINK-BANDWIDTH-AS:100:10Mbps,
           LINK-BANDWIDTH-AS:100:source_session.interface.bandwidth };
      """
      # Collection that keeps track if extended communities are duplicate
      extCommDuplicateFilter = {}

      for extComm in extCommSet.values:
         aclCommVal = AetExtCommHelper.buildExtCommAclCommValue( extComm )
         attrs = tuple( section.name if isinstance( section, RcfAst.Attribute ) else
                        None for section in extComm.sections )

         # Key that determines if an extended community was seen before.
         key = ( aclCommVal, attrs )
         # check if the item was already processed
         if key not in extCommDuplicateFilter:
            extCommDuplicateFilter[ key ] = ExtCommWithAcl( aclCommVal, extComm )

      return sorted( extCommDuplicateFilter.values() )

   def buildImmediateExtCommSet( self, extCommSet ):
      validTypes = [ RcfAst.CommunityValue.Type.immediateExtCommunityValue ]
      # The set must be sorted as sorted vectors are required for the set operations
      # at runtime.
      cachable = True
      head = None

      # We build the linked list of objects representing the int set backwards
      for extComm in reversed( self.buildExtCommSetCollection( extCommSet ) ):
         assert extComm.astNode.type in validTypes, (
            f"Unexpected type {extComm.astNode.type} when building "
            "immediate extended community AET" )

         immediateNode, cachable = AetExtCommHelper.buildExtCommunityValueImpl(
            extCommSet, extComm, cachable=cachable )
         assert immediateNode, "immediateNode should not be None"
         head = Rcf.Eval.CommunityLinkedList( immediateNode, head )

      return Rcf.Eval.CommunitySetImmediate( head, cachable )

   def buildImmediateLargeCommSet( self, astNode ):
      validTypes = [ RcfAst.Constant.Type.immediateLargeCommunityValue ]
      nodes = astNode.values
      for node in nodes:
         assert node.type in validTypes, ( "Unexpected type {} when building "
                                           "immediate large community AET"
                                           .format( node.type ) )
      # The set must be sorted as sorted vectors are required for the set operations
      # at runtime.
      head = None
      # We build the linked list of objects representing the int set backwards
      values = sorted( { self.build( node ) for node in nodes } )
      for val in reversed( values ):
         commVal = Rcf.Eval.CommunityValueImmediate( val )
         head = Rcf.Eval.CommunityLinkedList( commVal, head )
      cachable = True # we assert above that the contents are community values
      return Rcf.Eval.CommunitySetImmediate( head, cachable )

   def buildImmediateResolutionRibList( self, astNode ):

      def astToRibConfig( astNode ):
         assert astNode.etype in ResolutionRibTypes, \
                'unexpected etype encountered in immediate resolution rib AET gen'
         rib = BuiltinRibs[ astNode.etype ].get( astNode.name, None )
         if not rib:
            # user-defined tunnel ribs handled here
            assert astNode.etype == 'rib_tunnel', \
                   f'invalid { astNode.etype } with name { astNode.name } found in' \
                   ' immediate resolution rib AET generation'
            rib = tunnelRib( astNode.name )
            # Register user-defined tunnel ribs to metadata
            self.aetGenVisitor.currentFunction.metadata.tunnelRibsUsed.add(
               astNode.name )
         return rib

      assert len( astNode.values ) <= 3, \
             "immediate resolution rib must be of size 3 or less, this should've " \
             'been validated earlier during value validation phase'
      resolutionRibProfileConfig = Rcf.Routing.Rib.ResolutionRibProfileConfig()
      for i, val in enumerate( astNode.values ):
         resolutionRibProfileConfig.resolutionMethod[ i ] = \
            astToRibConfig( val )
      return Rcf.Eval.ResolutionRibsImmediate( resolutionRibProfileConfig )

   def buildEmpty( self, value ):
      # We do not support building empty values in a generic sense. Existing uses
      # of 'empty' have specific rhsAetType and rhsAetCtorArgs in RcfMetadata.yaml.
      return None
