# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
from TypeFuture import TacLazyType

FunctionDomain = TacLazyType( "Rcf::Metadata::FunctionDomain" )

def lowercaseTypeName( typename ):
   return typename[ 0 ].lower() + typename[ 1 : ]

def stripTypeNamespace( typename ):
   return typename.split( '::' )[ -1 ]

def collectionNameFromTypename( typename ):
   """Given an RCF typename, this function returns a name that can be used for a
   type member.  This is useful when figuring out the name of the member in
   AllAetEntities for a given AET Eval type.
   """
   typenameWithoutNamespace = stripTypeNamespace( typename )
   collName = '%sColl' % lowercaseTypeName( typenameWithoutNamespace )

   return collName

def collectRcfDebugSymbolLocationsInList( head ):
   '''
   Rcf.Debug.SymbolLocation objects are stored in a Rcf.Debug.SymbolContainer
   linked list. Locations are added to the head to the list, so the list is in
   reverse order. Collect the entries in a list for easier consumption in python.
   '''
   locations = []
   while head:
      entry = head.entry
      location = {
         'line': entry.lineNumber,
         'column': entry.columnOffset,
      }
      if entry.length:
         location[ 'length' ] = entry.length
      locations.insert( 0, location )
      head = head.next
   return locations

def collectRcfDebugNodeSpecificDataInDict( nodeSpecificData ):
   '''
   Some fragmentTypes require additional information to be stored in
   Rcf.Debug.AetNodeSymbol node. This is done using a sub object. This object
   is generically collected into a dictionary. All user defined attribute on the
   object are added to the dictionary.
   '''
   attrs = {}

   for attrName in nodeSpecificData.tacType.constructorArgNameQ:
      if attrName == 'key':
         continue
      attr = getattr( nodeSpecificData, attrName )
      if attr is None:
         continue
      if isinstance( attr, Tac.Type( 'Rcf::Debug::SymbolContainer' ) ):
         attrs[ attrName ] = collectRcfDebugSymbolLocationsInList( attr )
      else:
         attrs[ attrName ] = attr
   return attrs

def collectRcfDebugAetNodeSymbolsInDict( head ):
   '''
   Rcf.Debug.AetNodeSymbol objects are stored in a Rcf.Debug.SymbolContainer
   linked list. The order of the list is irrelevant. Collect the entries in a dict
   for efficient lookup by the AetNode::key in python.
   '''
   aetSymbolsDict = {}
   while head:
      entry = head.entry
      key = entry.aetNodeKey
      if isinstance( entry, Tac.Type( 'Rcf::Debug::InternalAetNodeSymbol' ) ):
         # The only purpose of DebugSymbols for internal AET nodes is to explicitly
         # mark them as irrelevant so that if a DebugSymbol is missing when a logged
         # AET node is processed an error can be raised.
         symbolDict = {
            'internal': True
         }
      else:
         symbolDict = {
            'fragmentType': entry.fragmentType,
         }
         if entry.location:
            symbolDict[ 'location' ] = collectRcfDebugSymbolLocationsInList(
               entry.location )
         nodeSpecificData = entry.nodeSpecificData
         while nodeSpecificData:
            entry = nodeSpecificData.entry
            entryName = lowercaseTypeName( stripTypeNamespace(
               entry.tacType.fullTypeName ) )
            symbolDict[ entryName ] = collectRcfDebugNodeSpecificDataInDict( entry )
            nodeSpecificData = nodeSpecificData.next
      aetSymbolsDict[ key ] = symbolDict
      head = head.next
   return aetSymbolsDict

def convertDebugSymbolsToPythonRepr( debugSymbols ):
   '''
   Rcf.Debug.FunctionSymbols contents are structured using linked lists instead of
   tacc collections. Rebuild using python collections for easier consumption.
   '''
   debugSymbolsDict = {}
   for fn, fnSymbols in debugSymbols.items():
      functionSymbols = {}
      functionSymbols[ 'functionDomain' ] = fnSymbols.functionDomain
      if functionSymbols[ 'functionDomain' ] == FunctionDomain.USER_DEFINED:
         functionSymbols[ 'aetNodeSymbols' ] = collectRcfDebugAetNodeSymbolsInDict(
            fnSymbols.aetSymbolHead )
         functionDefinition = {}
         fnDefinition = fnSymbols.functionDefinition
         functionDefinition[ 'codeUnitName' ] = fnDefinition.codeUnitName
         functionDefinition[ 'definitionStartLine' ] = (
                     fnDefinition.definitionStartLine )
         functionDefinition[ 'definitionStartCol' ] = fnDefinition.definitionStartCol
         functionDefinition[ 'definitionEndLine' ] = fnDefinition.definitionEndLine
         functionDefinition[ 'definitionEndCol' ] = fnDefinition.definitionEndCol
         functionSymbols[ 'functionDefinition' ] = functionDefinition
         functionSymbols[ 'aetVersion' ] = fnSymbols.aetVersion
         functionSymbols[ 'rcfCodeVersion' ] = fnSymbols.rcfCodeVersion

      debugSymbolsDict[ fn ] = functionSymbols
   return debugSymbolsDict

def createRoutingAttrAetTypes( rcfAttributes, namespace ):
   attrTypePrefix = namespace + '::Attribute::'

   routingAttrAetTypes = []
   for rcfAttrCharacteristics in rcfAttributes.values():
      aetTypeName = attrTypePrefix + rcfAttrCharacteristics[ 'aetType' ]
      routingAttrAetTypes.append( aetTypeName )
      # If an attribute is modifiable (directly or indirectly)
      # we add an additional type
      if ( rcfAttrCharacteristics[ 'modifiable' ] in
            [ "directly", "indirectly" ] ):
         routingAttrAetTypes.append( aetTypeName + 'Input' )

   # This list should be sorted even if the --verify check didn't fail
   routingAttrAetTypes.sort()
   return routingAttrAetTypes
