# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import namedtuple
from dataclasses import dataclass
from Toggles import RcfLibToggleLib
import Tracing
import QuickTrace

from CodeUnitMapping import CodeUnitKey
import RcfAst
from RcfAstGen import genAst, getBuiltinFunctions
from RcfAstInterfaceGen import generateAstInterfaces
from RcfCompilerStageCommon import RcfCompilerStageResultBase
from RcfDiag import RcfDiag, RcfSyntaxDiag
from RcfSemanticAnalysis import (
   runFunctionScopeSemanticAnalysis,
   runLinker,
)
from RcfSyntaxParser import runRcfLexer, runRcfParser
import RcfTypeFuture as Rcf

t0 = Tracing.t0
qv = QuickTrace.Var
qt8 = QuickTrace.trace8

class RcfLintRequest:
   """ A lint request.
   """
   def __init__( self, codeUnitMapping, rcfCodeVersion, strictMode ):
      self.codeUnitMapping = codeUnitMapping
      # rcfCodeVersion is the codeVersion in sysdb rcfConfig when code text units
      # were extracted from sysdb. It is 0 when the code wasn't extracted from sysdb
      self.rcfCodeVersion = rcfCodeVersion
      self.strictMode = strictMode

class RcfLintResult( RcfCompilerStageResultBase ):
   """Output from the RCF compiler

   Arguments
   ---------
   rcfLintRequest : RcfLintRequest
      The lint request to which this is a response
   success : bool
      Whether the linting was successful or not
   errorList : (list of str) or (None)
   warningList : (list of str) or (None)
      None if compilation was successful. List of error strings otherwise.
   """
   def __init__( self, rcfLintRequest, success, diag ):
      super().__init__( success=success, diag=diag )
      self.rcfLintRequest = rcfLintRequest

class RcfLinter:
   """ Rcf Linter instance """
   def lint( self, request, rcfExternalConfig=None ):
      """ Lint the given request.

      Args:
         request: RcfLintRequest: the request to lint.
         rcfExternalConfig : RcfHelperTypes.RcfExternalConfig
            This python object holds:
               aclConfig (Acl::AclListConfig)
               roaTableStatusDir (Rpki::RoaTableStatusDir)
               dynPfxListConfigDir (DynamicPrefixList::Config)
            These are used to validate
            external references during symbol table generation.
      Returns:
         RcfLintResult
      """
      diag = RcfDiag( strict=request.strictMode )

      try:
         lintCodeUnits( request.codeUnitMapping, diag, rcfExternalConfig )
      except Exception as exc: # pylint: disable=broad-except
         # str( exc ) ->
         #  "'ascii' codec can't decode byte 0xc2 in position 26: ordinal
         #   not in range(128)"
         #
         # repr( exc ) ->
         #  "UnicodeDecodeError('ascii', 'function NEXT_HOP_SELF() {\\xc2\\n
         #   @seq10 {set_next_hop_self();\\n return true; }\\n}', 26, 27,
         #   'ordinal not in range(128)')"
         diag.unknownError( repr( exc ) )

      return RcfLintResult( request, success=not diag.hasErrors(), diag=diag )

@dataclass( frozen=True )
class CacheKey:
   codeUnitKey: CodeUnitKey
   codeUnitText: str

   @staticmethod
   def builtinCacheKey():
      codeUnitKey = CodeUnitKey( Rcf.Metadata.FunctionDomain.BUILTIN,
                                 Rcf.Config.unnamedCodeUnitName )
      return CacheKey( codeUnitKey, "" )

RcfLintCodeUnitResult = namedtuple( "RcfLintCodeUnitResult",
                                    ( "functions", "astInterfaces" ) )

def lintCodeUnit( cacheKey, diag ):
   # In the steps before runFunctionScopeSemanticAnalysis
   # we don't resolve anything that requires going outside
   # the scope of the function itself.

   syntaxDiag = RcfSyntaxDiag( codeUnitKey=cacheKey.codeUnitKey )

   # Run the syntax analysis phase of all code units, and add them together
   # to a single RcfAst.Root object
   _, tokenStream = runRcfLexer( syntaxDiag, cacheKey.codeUnitText )
   parseTree = runRcfParser( syntaxDiag, tokenStream )
   if not syntaxDiag.hasErrors():
      functions = genAst( parseTree, cacheKey.codeUnitKey )
   else:
      functions = []

   diag.update( syntaxDiag )

   runFunctionScopeSemanticAnalysis( functions, diag )

   if not RcfLibToggleLib.toggleRcfCompilerCachingEnabled():
      return RcfLintCodeUnitResult( functions, tuple() )

   return RcfLintCodeUnitResult( functions,
                                 tuple( generateAstInterfaces( functions ) ) )

def getBuiltinCodeUnit( cacheKey, diag, builtinFunctionMock ):
   functions = getBuiltinFunctions( cacheKey.codeUnitKey, builtinFunctionMock )
   runFunctionScopeSemanticAnalysis( functions, diag )
   if not RcfLibToggleLib.toggleRcfCompilerCachingEnabled():
      return RcfLintCodeUnitResult( functions, tuple() )

   return RcfLintCodeUnitResult( functions,
                                 tuple( generateAstInterfaces( functions ) ) )

RcfLintCodeUnitsResult = namedtuple( "RcfLintCodeUnitsResult",
                                     ( "root", "astInterfaceMapping" ) )

def lintCodeUnits( codeUnitMapping, diag, rcfExternalConfig,
                   builtinFunctionMock=None, *, linkFunctions=True ):
   '''
   Function that lints all code units in the code unit mapping.

   Args:
      codeUnitMapping(CodeUnitMapping): a mapping from code unit key to text
      diag(RcfDiag): an object that tracks all of the errors during compilation
      rcfExternalConfig(RcfExternalConfig): object that can be used for external
                                            reference resolution
      builtinFunctionMock(dict): dict of builtinFunctions to mock. This is only used
                                 for testing

   returns:
      root (list): The root containing all Rcf functions.

   '''
   builtinCacheKey = CacheKey.builtinCacheKey()
   builtins = getBuiltinCodeUnit( builtinCacheKey, diag, builtinFunctionMock )
   root = RcfAst.Root( builtins.functions )
   astInterfaceMapping = { builtinCacheKey: builtins.astInterfaces }

   for codeUnitKey, codeUnitText in codeUnitMapping.codeUnitRcfTexts.items():
      cacheKey = CacheKey( codeUnitKey, codeUnitText )
      lintResult = lintCodeUnit( cacheKey, diag )
      root.addFunctions( lintResult.functions )
      astInterfaceMapping[ cacheKey ] = lintResult.astInterfaces

   if linkFunctions:
      runLinker( root, diag, rcfExternalConfig )

   return RcfLintCodeUnitsResult( root, astInterfaceMapping )
