#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
The RCF compiler is data driven and the RcfMetadata.yaml file provides all the
metadata that the compiler uses to figure out the following:
* The set of builtin types in the language
* The set of BGP and other protocol symbols that are available for use in RCF
* The RCF language type system rules

When adding a new BGP or other protocol attribute (common) or when adding a new
RCF builtin type (less common), the information needs to be added to that yaml
file and this file rarely needs to be modified.

Please contact rcf-dev@ if there are any questions.
"""

from FrozenDict import frozendict
from collections import defaultdict, namedtuple
import yaml
import RcfAst
from RcfEvalType import (
      CollectionType,
      Enum,
      Type,
)
from RcfSymbol import (
      ConstSymbol,
      Symbol,
)
import RcfTypeFuture
import Toggles.RcfLibToggleLib  # pylint: disable=unused-import

ExtCommSection = namedtuple( "ExtCommSection", ( "rcfType", "bitLength", "offset" ) )

# Some common helpers
def getRcfType( name ):
   return getattr( RcfBuiltinTypes, name, None )

def iterRcfTypes():
   '''
   This function goes through the RcfBuiltinTypes and filters
   all possible Rcf Types.
   '''
   for builtinType in dir( RcfBuiltinTypes ):
      rcfType = getattr( RcfBuiltinTypes, builtinType )
      if isinstance( rcfType, Type ):
         yield rcfType

def getConditionOperators():
   conditionOperators = set()
   for _, op, _ in RcfTypeSystem.allowedConditionOps:
      conditionOperators.add( op )
   return conditionOperators

def getAttrsPerType():
   '''
   Create a mapping from rcfType to all the possible attributes.
   For example:
   {
      BT.Int: [weight, local_preference, ...],
      BT.AsNumber: [as_path.last_as, ...],
      ...
   }
   '''
   attrsPerType = defaultdict( list )
   for attr in RcfBuiltinSymbols.builtInAttributes.values():
      attrsPerType[ attr.rcfType ].append( attr )
   return attrsPerType

def registerAetType( name ):
   # name could be <typename> or Bgp.<typename>
   if name.startswith( 'Bgp' ):
      cls = RcfTypeFuture.Eval.Bgp
      attrName = name.split( '.' )[ 1 ]
      aetTypename = f'Rcf::Eval::Attribute::{attrName}'
   else:
      cls = RcfTypeFuture.Eval
      attrName = name
      aetTypename = f'Rcf::Eval::{name}'

   lazyType = RcfTypeFuture.TacLazyType( aetTypename )
   setattr( cls, f'{attrName}Type', lazyType )
   setattr( cls, attrName, RcfTypeFuture.aetNodeInstantiator( aetTypename ) )
   return lazyType.tacType

def getAetType( name ):
   if name is None:
      return None
   if name.startswith( 'Bgp' ):
      cls = RcfTypeFuture.Eval.Bgp
      attrName = name.split( '.' )[ 1 ]
   else:
      cls = RcfTypeFuture.Eval
      attrName = name

   aetType = getattr( cls, attrName, None )
   if not aetType:
      registerAetType( name )

   return getattr( cls, attrName )

def toggleEnabled( toggleName ):
   if toggleName is None:
      return True
   toggleFn = getattr( Toggles.RcfLibToggleLib, f'toggle{toggleName}Enabled' )
   return toggleFn()

class RcfKeywords:
   keywords = set()

   @staticmethod
   def addKeyword( keyword ):
      RcfKeywords.keywords.add( keyword )

   @staticmethod
   def isKeyword( keyword ):
      return keyword in RcfKeywords.keywords

class RcfEnumMapping:
   def __init__( self ):
      self.valueToString = defaultdict( lambda: 'unknown' )
      self.stringToValue = {}

class RcfBuiltinTypes:
   """This class holds all the builtin types (like Int, Boolean etc) in RCF.
   """
   # Fields here will be populated by the metadata processor, based on the
   # contents of RcfMetadata.yaml.

   enumTypes = []
   enumStringToTypename = {}
   enumMappings = defaultdict( RcfEnumMapping )
   contentTypeToSetType = defaultdict( lambda: None )
   contentTypeToListType = defaultdict( lambda: None )
   variableTypenames = set()

   # This is to get rid of pylint errors about using non-existant attributes.
   def __getattr__( self, key ):
      # This should never get called, since we always only access attributes that
      # have been explicitly created using setattr.
      assert False, '__getattr__ should never get invoked'

   @staticmethod
   def getDisplayName( name, suffix ):
      """The naming convention for user-facing type names is to convert the RCF type
      name to snake_case and add a "_type" at the end. For enums, we add a "_enum" at
      the end instead of "_type.
      """
      displayName = ''.join( '_' + c.lower() if c.isupper() else c for c in name )
      # Some RCF type names end with "Type"
      if not displayName.endswith( suffix ):
         displayName += suffix
      return displayName.lstrip( '_' )

   @staticmethod
   def addBuiltinType( name, variableAetTypename=None, aetOpTypes=None ):
      displayName = RcfBuiltinTypes.getDisplayName( name, suffix="_type" )
      value = Type( f'<{name}>', displayName,
                    variableAetTypename=variableAetTypename,
                    aetOpTypes=frozendict( aetOpTypes ) )
      setattr( RcfBuiltinTypes, name, value )

   @staticmethod
   def addBuiltinCollectionType( name, contentTypes, collectionType ):
      displayName = RcfBuiltinTypes.getDisplayName( name, suffix="_type" )
      value = CollectionType( f'<{name}>', displayName,
                              contentTypes=frozenset( contentTypes ) )
      setattr( RcfBuiltinTypes, name, value )
      setattr( RcfAst.Collection.Type, name, name )
      if collectionType == 'Set':
         for contentType in contentTypes:
            RcfBuiltinTypes.contentTypeToSetType[ contentType ] = value
      elif collectionType == 'List':
         for contentType in contentTypes:
            RcfBuiltinTypes.contentTypeToListType[ contentType ] = value

   @staticmethod
   def addBuiltinEnumType( name, valueDict, aetOpTypes ):
      displayName = RcfBuiltinTypes.getDisplayName( name, suffix="_enum" )
      value = Enum( f'<{name}>', displayName, valueDict=frozendict( valueDict ),
                    aetOpTypes=frozendict( aetOpTypes ) )
      setattr( RcfBuiltinTypes, name, value )
      setattr( RcfAst.Constant.Type, name, name )
      RcfBuiltinTypes.enumTypes.append( name )

      mapping = RcfBuiltinTypes.enumMappings[ name ]
      for enumKeyword in valueDict:
         RcfKeywords.addKeyword( enumKeyword )
         RcfBuiltinTypes.enumStringToTypename[ enumKeyword ] = name
         value = valueDict[ enumKeyword ]
         mapping.valueToString[ value ] = enumKeyword
         mapping.stringToValue[ enumKeyword ] = value

class RcfBuiltinSymbols:
   """This class holds all the builtin symbols (like med, prefix etc) in RCF.
   The elements are of type RcfSymbol.Symbol.
   """
   builtInAttributes = {}

   @staticmethod
   def registerSymbol( name, const, rcfType, aetType, routeMapFeatures=None,
                       inputVariant=False, rootAttribute=None ):
      SymbolType = ConstSymbol if const else Symbol
      symbol = SymbolType( name, rcfType=getRcfType( rcfType ),
                           aetType=getAetType( aetType ),
                           routeMapFeatures=routeMapFeatures,
                           inputVariant=inputVariant, rootAttribute=rootAttribute )
      RcfBuiltinSymbols.builtInAttributes[ name ] = symbol
      res = symbol

      for symbol in name.split( '.' ):
         RcfKeywords.addKeyword( symbol )

      return res

class RcfTypeSystem:
   """This class implements the RCF type system.
   """
   # ImplicitConversions[ fromType ] contains a list with all types fromType can be
   # implicitely converted to.
   allowableTypePromotions = defaultdict( set )
   allowableTypeDemotions = defaultdict( set )

   # extCommunitySections[ attribute ] contains a list of all extended communities
   # and the supported types of the sections
   extCommunitySections = {}

   # allowedModifyOps[ ( type, op ) ] will be set to True, if 'op' is allowed
   # for 'type'
   allowedModifyOps = defaultdict( lambda: False )

   # allowedConditionOps[ ( lhsType, op, rhsType ) ] will be set to the appropriate
   # resultType, if '<lhsType> <op> <rhsType>' is allowed.
   allowedConditionOps = defaultdict( lambda: None )

   # aetTypes[ ( lhsType, op, rhsType ) ] will be set to the appropriate tuple of aet
   # node types if any of the aet types are specified for a particular tuple of stmt
   # types. Additionally, a list of ctor args (or None) can be set for the rhsAetType
   # ctor.
   aetTypes = defaultdict( lambda: None )

   # For types which can be used as user-defined function parameter types, this maps
   # the user-provided "displayName" (e.g. "int_type") to the internal RCF type name.
   funcParamTypeMapping = {}

   @staticmethod
   def getAstNodeType( astNode ):
      return astNode.promoteToType if astNode.promoteToType else astNode.evalType

   @staticmethod
   def canPromote( fromType, toType ):
      return toType in RcfTypeSystem.allowableTypePromotions[ fromType ]

   @staticmethod
   def addImplicitConversion( fromTypeName, toTypeName ):
      fromType = getRcfType( fromTypeName )
      toType = getRcfType( toTypeName )
      typesUsed = { type( fromType ), type( toType ) }
      assert ( typesUsed.issubset( { Type, Enum } ) or
               typesUsed.issubset( { CollectionType } ) ), (
                  f"Cannot implicitly convert {fromTypeName}({type( fromType) })"
                  f" to {toTypeName}({type( toType )})" )
      RcfTypeSystem.allowableTypePromotions[ fromType ].add( toType )
      RcfTypeSystem.allowableTypeDemotions[ toType ].add( fromType )

   @staticmethod
   def resolveExtCommunitySections( extComms ):
      """
      Recursively resolve type for the attribute.
      """
      for extCommType, sections in extComms.items():
         sectionTypes = []
         offset = 64
         for section in sections[ 'sections' ]:
            bitLength = section[ 'bitLength' ]
            value = section.get( 'value' )
            rcfType = getRcfType( section[ 'type' ] )
            offset -= bitLength
            if value is not None:
               # BUG946273: Add support for non zero Immediate values
               assert value == 0
            else:
               extCommSection = ExtCommSection( rcfType=rcfType,
                                                bitLength=bitLength, offset=offset )
               sectionTypes.append( extCommSection )
         assert offset == 0

         RcfTypeSystem.extCommunitySections[ extCommType ] = tuple( sectionTypes )

   @staticmethod
   def addAllowedModifyOp( lhsTypename, op, rhsTypename ):
      lhsType = getRcfType( lhsTypename )
      rhsType = getRcfType( rhsTypename )
      RcfTypeSystem.allowedModifyOps[ ( lhsType, op, rhsType ) ] = True

   @staticmethod
   def modifyOpAllowed( lhsType, op, rhsType ):
      """Tells whether the given assign operation is allowed on the given type.
      """
      return RcfTypeSystem.allowedModifyOps[ ( lhsType, op, rhsType ) ]

   @staticmethod
   def addAllowedConditionOp( lhsTypename, op, rhsTypename ):
      lhsType = getRcfType( lhsTypename )
      rhsType = getRcfType( rhsTypename )
      RcfTypeSystem.allowedConditionOps[
         ( lhsType, op, rhsType ) ] = RcfBuiltinTypes.Boolean

   @staticmethod
   def conditionOpAllowed( lhsType, op, rhsType ):
      """ Get the result type of a binary operation.
      Returns the result type of this operation over these operands.
      Returns None if this operation is not allowed on these operands.
      """
      return RcfTypeSystem.allowedConditionOps[ ( lhsType, op, rhsType ) ]

   @staticmethod
   def addAetTypesForStmt( lhsTypename, op, rhsTypename, opAetTypename,
                           rhsAetTypename, rhsAetCtorArgs ):
      lhsType = getRcfType( lhsTypename )
      rhsType = getRcfType( rhsTypename )
      opAetType = getAetType( opAetTypename )
      tacType = registerAetType( opAetTypename )
      RcfTypeFuture.OperatorTypeToSyntax[ tacType ] = op
      rhsAetType = getAetType( rhsAetTypename )
      RcfTypeSystem.aetTypes[ ( lhsType, op, rhsType ) ] = (
            opAetType, rhsAetType, rhsAetCtorArgs )
      # We will go through all types that can promote to the rhs type and add the to
      # the aetTypes dictionary as well. The logic is that if a type can promote to
      # another Type, it should be allowed to do its operations as well
      for promotedType in RcfTypeSystem.allowableTypeDemotions[ rhsType ]:
         # If there are multiple occurences of the tuple, we can get ambiguous
         # behaviour. This assertion makes sure we cannot add duplicate entries.
         assert ( lhsType, op, promotedType ) not in RcfTypeSystem.aetTypes
         RcfTypeSystem.aetTypes[ ( lhsType, op, promotedType ) ] = (
               opAetType, rhsAetType, rhsAetCtorArgs )

   @staticmethod
   def aetTypesOf( lhsType, op, rhsType ):
      """ Get the aet types for a modification or condition (lhs, op, rhs).
      Return: (opAetType, rhsAetType) if aet types are specified for stmt
      """
      return RcfTypeSystem.aetTypes[ ( lhsType, op, rhsType ) ]

   @staticmethod
   def dumpState():
      """ Renders all of the allowed ops in a human readable way
      """
      print( "Allowable Type Promotions" )
      for fromType in sorted( RcfTypeSystem.allowableTypePromotions ):
         print( fromType.name + ':' )
         for toType in sorted( RcfTypeSystem.allowableTypePromotions[ fromType ] ):
            print( '   ' + toType.name )

      print( "\nAllowable Type Demotions" )
      for fromType in sorted( RcfTypeSystem.allowableTypeDemotions ):
         print( fromType.name + ':' )
         for toType in sorted( RcfTypeSystem.allowableTypeDemotions[ fromType ] ):
            print( '   ' + toType.name )

      def dumpOpColl( coll ):
         ops = []
         for ( lhsType, op, rhsType ), result in coll.items():
            ops.append( ( lhsType.name, op, rhsType.name, result ) )
         # sort by operator, then lhsType
         ops = sorted( ops, key=lambda x: ( x[ 1 ], x[ 0 ], x[ 2 ] ) )
         for lhsType, op, rhsType, result in ops:
            if result is False or result is None:
               continue
            print( f"{lhsType} {op} {rhsType}" )

      print( "\nAllowed Modify Ops" )
      dumpOpColl( RcfTypeSystem.allowedModifyOps )

      print( "\nAllowed Condition Ops" )
      dumpOpColl( RcfTypeSystem.allowedConditionOps )

      print( "\nAET Types" )
      dumpOpColl( RcfTypeSystem.aetTypes )

class RcfMetadataProcessor:
   def __init__( self ):
      with open( '/usr/share/Rcf/RcfMetadata.yaml' ) as f:
         self.metadata = yaml.safe_load( f.read() )
      self.allBuiltinTypes = self.metadata[ 'RcfTypes' ]
      self.allBuiltinAttributes = self.metadata[ 'Attributes' ]
      self.allBuiltinFunctions = self.metadata[ 'Functions' ]

      self.allTypingRules = self.metadata[ 'TypingRules' ]
      self.otherAetTypes = self.metadata[ 'OtherAetTypes' ]
      self.otherAttributeTypes = self.metadata[ 'OtherAttributeTypes' ]
      self.otherKeywords = self.metadata[ 'OtherKeywords' ]

      self.processAllBuiltinTypes()
      self.processAllBuiltinAttributes()
      self.processAllTypingRules()
      self.processOtherTypes()
      self.processAllKeywords()
      self.processAllBuiltinFunctions()
      self.processBuiltinCollectionDemotionTypes()

   def processOperatorsDict( self, operatorsMetadata ):
      aetOpTypes = {}
      for opName, aetTypeName in operatorsMetadata.items():
         tacType = registerAetType( aetTypeName )
         RcfTypeFuture.OperatorTypeToSyntax[ tacType ] = opName
         aetOpTypes[ opName ] = getAetType( aetTypeName )

         if opName[ 0 ].isalpha():
            RcfKeywords.addKeyword( opName )

      return aetOpTypes

   def processOperatorsList( self, operatorsMetadata ):
      aetOpTypes = {}
      for typeAndOp in operatorsMetadata:
         typ = typeAndOp.split( '.' )[ 0 ]
         opName = typeAndOp.split( '.' )[ 1 ]
         aetOpTypes[ opName ] = getRcfType( typ ).aetOpTypes[ opName ]
      return aetOpTypes

   def processOperatorToAetType( self, operatorsMetadata ):
      if isinstance( operatorsMetadata, dict ):
         return self.processOperatorsDict( operatorsMetadata )
      else:
         assert isinstance( operatorsMetadata, list )
         return self.processOperatorsList( operatorsMetadata )

   def processBuiltinType( self, typename, metadata ):
      aetOpTypes = None
      operatorsMetadata = metadata.get( 'OperatorToAetType' )
      if operatorsMetadata:
         aetOpTypes = self.processOperatorToAetType( operatorsMetadata )

      variableAetTypename = metadata.get( 'VariableAetType', None )
      if variableAetTypename is not None:
         registerAetType( variableAetTypename )
         RcfBuiltinTypes.variableTypenames.add( variableAetTypename )

      RcfBuiltinTypes.addBuiltinType( name=typename,
                                      variableAetTypename=variableAetTypename,
                                      aetOpTypes=aetOpTypes )

   def processAllEnumTypes( self, metadata ):
      aetOpTypes = self.processOperatorToAetType( metadata[ 'OperatorToAetType' ] )
      for typename, valueDict in metadata[ 'EnumTypes' ].items():
         RcfBuiltinTypes.addBuiltinEnumType(
               name=typename, valueDict=valueDict, aetOpTypes=aetOpTypes )

   def processAllCollectionTypes( self, metadata, collectionType ):
      for typename, valueDict in metadata.items():
         contentTypes = [ getRcfType( contentType ) for contentType in
                          valueDict[ 'contentTypes' ] ]
         RcfBuiltinTypes.addBuiltinCollectionType( typename,
                                                   contentTypes,
                                                   collectionType )

   def processAllBuiltinTypes( self ):
      for entry in self.allBuiltinTypes:
         if isinstance( entry, dict ):
            assert len( entry ) == 1
            typename = next( iter( entry ) )
            metadata = entry[ typename ]

            if typename == 'Enum':
               self.processAllEnumTypes( metadata )
            elif typename == 'Collection':
               self.processAllCollectionTypes( metadata[ 'List' ],
                                               collectionType='List' )
               self.processAllCollectionTypes( metadata[ 'Set' ],
                                               collectionType='Set' )
            else:
               self.processBuiltinType( typename, metadata )
         else:
            RcfBuiltinTypes.addBuiltinType( name=entry )

   def fullname( self, namespace, attrname ):
      if namespace:
         return f'{namespace}.{attrname}'
      else:
         return attrname

   def processAttribute( self, attrname, metadata, namespace=None,
                         inheritedRouteMapFeatures=None,
                         hasNonConstParent=False, rootAttribute=None,
                         inputRootAttribute=None ):
      if inputRootAttribute is not None:
         assert rootAttribute is not None

      fullname = self.fullname( namespace, attrname )

      routeMapFeatures = set()
      if inheritedRouteMapFeatures:
         routeMapFeatures.update( inheritedRouteMapFeatures )
      if 'routeMapFeatures' in metadata:
         routeMapFeatures.update( metadata[ 'routeMapFeatures' ] )

      aetType = metadata.get( 'aetType' )
      rcfType = metadata.get( 'rcfType' )
      symbol = None
      inputSymbol = None
      if aetType is None:
         # aetType would be None for namespaces like 'igp' which exist only to
         # define attrbutes like igp.tag.
         const = True
      else:
         const = metadata.get( 'const' )
         tacType = registerAetType( aetType )
         RcfTypeFuture.AttributeTypeToSyntax[ tacType ] = fullname

         symbol = RcfBuiltinSymbols.registerSymbol(
                     fullname, const=const,
                     rcfType=rcfType, aetType=aetType,
                     routeMapFeatures=routeMapFeatures,
                     rootAttribute=rootAttribute )

         # "input." scope
         if hasNonConstParent or not const:
            if rootAttribute is not None:
               assert inputRootAttribute is not None
            inputAttrFullname = f'input.{fullname}'
            inputAttrAetType = f'{aetType}Input'
            tacType = registerAetType( inputAttrAetType )
            RcfTypeFuture.AttributeTypeToSyntax[ tacType ] = inputAttrFullname

            inputSymbol = RcfBuiltinSymbols.registerSymbol(
                        inputAttrFullname, const=True,
                        rcfType=metadata[ 'rcfType' ], aetType=inputAttrAetType,
                        routeMapFeatures=routeMapFeatures,
                        inputVariant=True, rootAttribute=inputRootAttribute )

      subattrs = metadata.get( 'Subattributes' )
      if subattrs is not None:
         if rootAttribute is None:
            assert symbol is not None
            rootAttribute = symbol
            inputRootAttribute = inputSymbol
         self.processAttributes(
                     subattrs, namespace=fullname,
                     inheritedRouteMapFeatures=routeMapFeatures,
                     hasNonConstParent=( hasNonConstParent or not const ),
                     rootAttribute=rootAttribute,
                     inputRootAttribute=inputRootAttribute )

      scopedattrs = metadata.get( 'ScopedAttributes' )
      if scopedattrs is not None:
         assert rootAttribute is None, \
                  f"ScopedAttribute in Subattribute {rootAttribute}"
         self.processAttributes(
                     scopedattrs, namespace=fullname,
                     inheritedRouteMapFeatures=routeMapFeatures )

      methods = metadata.get( 'Methods' )
      if methods is not None:
         for methodName, methodData in methods.items():
            const = methodData.get( 'const' )
            # Const methods should be added for the input namespace attributes too
            assert not const, 'Support for const methods not yet added'
            methodData[ 'functionSelf' ] = fullname
            self.allBuiltinFunctions[ fullname + '.' + methodName ] = methodData

   def processAttributes( self, metadata, namespace=None,
                          inheritedRouteMapFeatures=None,
                          hasNonConstParent=False,
                          rootAttribute=None,
                          inputRootAttribute=None ):
      for name, value in metadata.items():
         if toggleEnabled( value.get( 'toggle' ) ):
            self.processAttribute(
                name, value, namespace=namespace,
                inheritedRouteMapFeatures=inheritedRouteMapFeatures,
                hasNonConstParent=hasNonConstParent,
                rootAttribute=rootAttribute,
                inputRootAttribute=inputRootAttribute )

   def processAllBuiltinAttributes( self ):
      self.processAttributes( self.allBuiltinAttributes )

   def processImplicitConversions( self ):
      for key, values in self.allTypingRules[ 'ImplicitConversions' ].items():
         if not isinstance( values, list ):
            values = [ values ]
         for value in values:
            RcfTypeSystem.addImplicitConversion( fromTypeName=key,
                                                 toTypeName=value )

   def processAllowedContentTypes( self ):
      for key, values in self.allTypingRules[ 'AllowedContentTypes' ].items():
         keyType = getRcfType( key )
         assert isinstance( keyType, Type ), (
                  f"Invalid type { key }({ type( keyType ) } )" )

         if keyType == RcfBuiltinTypes.ImmediateExtCommunityValue:
            RcfTypeSystem.resolveExtCommunitySections( values )
         else:
            assert False, f"unexpected type: {keyType.displayName}"

   def processAllowedUserFunctionParameterTypes( self ):
      """Register types that are allowed to be used as user-defined function
      parameter types by adding them to a mapping from the user-provided type
      name (displayName) to the internal RCF type.
      """
      for entry in self.allTypingRules[ 'AllowedUserFunctionParameterTypes' ]:
         if isinstance( entry, dict ):
            name = next( iter( entry ) )
            toggleName = entry[ name ].get( 'toggle' )
         else:
            name = entry
            toggleName = None

         if not toggleEnabled( toggleName ):
            continue

         builtinType = getattr( RcfBuiltinTypes, name )
         assert builtinType, "Function parameter type is not a known RCF type"
         RcfTypeSystem.funcParamTypeMapping[ builtinType.displayName ] = builtinType

   def processAllowedOperations( self, sectionName, updateFn ):
      for entry in self.allTypingRules[ sectionName ]:
         if isinstance( entry, dict ):
            key = next( iter( entry ) )
            toggleName = entry[ key ].get( 'toggle' )
            opAetTypename = entry[ key ].get( 'opAetType' )
            rhsAetTypename = entry[ key ].get( 'rhsAetType' )
            rhsAetCtorArgs = entry[ key ].get( 'rhsAetCtorArgs' )
         else:
            key = entry
            toggleName = None
            opAetTypename = None
            rhsAetTypename = None
            rhsAetCtorArgs = None

         if not toggleEnabled( toggleName ):
            continue

         lhsTypename, op, rhsTypename = key.split()
         if lhsTypename == 'Enum':
            for enumTypename in RcfBuiltinTypes.enumTypes:
               updateFn( enumTypename, op, enumTypename )
         else:
            updateFn( lhsTypename, op, rhsTypename )

         if opAetTypename:
            RcfTypeSystem.addAetTypesForStmt( lhsTypename,
                                              op,
                                              rhsTypename,
                                              opAetTypename,
                                              rhsAetTypename,
                                              rhsAetCtorArgs )

         if op[ 0 ].isalpha():
            RcfKeywords.addKeyword( op )

   def processAllowedConditionOperations( self ):
      self.processAllowedOperations( 'AllowedConditionOperations',
                                     RcfTypeSystem.addAllowedConditionOp )

   def processAllowedModifyOperations( self ):
      self.processAllowedOperations( 'AllowedModifyOperations',
                                     RcfTypeSystem.addAllowedModifyOp )

   def processAllTypingRules( self ):
      self.processImplicitConversions()
      self.processAllowedContentTypes()
      self.processAllowedUserFunctionParameterTypes()
      self.processAllowedConditionOperations()
      self.processAllowedModifyOperations()

   def processOtherTypes( self ):
      for name in self.otherAetTypes:
         registerAetType( name )
      for name in self.otherAttributeTypes:
         tacType = registerAetType( 'Bgp.' + name )
         fullname = self.fullname( 'Bgp', name )
         RcfTypeFuture.AttributeTypeToSyntax[ tacType ] = fullname

   def processAllKeywords( self ):
      for otherKeywordCategory in self.otherKeywords.values():
         for keyword in otherKeywordCategory:
            RcfKeywords.addKeyword( keyword )

   def processAllBuiltinFunctions( self ):
      self.allBuiltinFunctions = {
         name: functionData
         for name, functionData in self.allBuiltinFunctions.items()
         if toggleEnabled( functionData.get( 'toggle' ) )
      }
      for functionData in self.allBuiltinFunctions.values():
         tacType = registerAetType( functionData[ 'aetType' ] )
         RcfTypeFuture.BuiltinFnExprToData[ tacType ] = functionData

   def processBuiltinCollectionDemotionTypes( self ):
      # The list of contentTypeToSetType does not contain
      # keys that can promote to a 'contentType'
      # This function will add all of those statem
      def helper( contentTypeToCollectionType ):
         newContentTypeToCollectionTypes = {}
         for contentType, collectionType in contentTypeToCollectionType.items():
            for demotionType in RcfTypeSystem.allowableTypeDemotions[ contentType ]:
               # Only look through demotions types that cannot already
               # lead to a collection type
               if demotionType not in contentTypeToCollectionType:
                  # Make sure a single type cannot lead to multiple types of the
                  # same collection
                  assert demotionType not in newContentTypeToCollectionTypes
                  newContentTypeToCollectionTypes[ demotionType ] = collectionType
         return newContentTypeToCollectionTypes
      newContentTypeToSetTypes = helper( RcfBuiltinTypes.contentTypeToSetType )
      newContentTypeToListTypes = helper( RcfBuiltinTypes.contentTypeToListType )
      RcfBuiltinTypes.contentTypeToSetType.update( newContentTypeToSetTypes )
      RcfBuiltinTypes.contentTypeToListType.update( newContentTypeToListTypes )

metadataProcessor = RcfMetadataProcessor()

# TODO:
# * Need a way to define toggles for a specific combination of (attribute, operation)
#    * eg. block 'next_hop = ...' based on the RcfNextHopSet toggle, without blocking
#      it for other IpAddress based types.
# * Rename RcfBuitinType.Prefix to RcfBuitinType.IpPrefix.
