#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=line-too-long

from RcfDiagCommon import GenericDiagMessageBase, RcfDiagGeneric, Severity

class OpenConfigFunctionValidationMessageBase( GenericDiagMessageBase ):
   """
   Error in OpenConfig function text validation. This prints out the path that
   emitted the error plus a message.

   Args:
      functionName (str): function the message originated from
      openConfigPath (str): OC path the message originated from
      msg (str): the description of what went wrong
   """

   def __init__( self, functionName, openConfigPath, msg ):
      super().__init__( msg )
      self.functionName = functionName
      self.openConfigPath = openConfigPath

   def render( self, codeUnitMapping ):
      functionNameStr = f" function openconfig {self.functionName}"
      openConfigPathStr = ""
      if self.openConfigPath:
         # The path construction leaves a slash at the beginning, but this isn't
         # needed and it's ugly.
         displayedPath = self.openConfigPath.removeprefix( "/" )
         openConfigPathStr = f" path {displayedPath}"
      out = f"{self.severity}{functionNameStr}{openConfigPathStr}: {self.msg}"
      return out

class OpenConfigFunctionValidationUnexpectedTypeError(
      OpenConfigFunctionValidationMessageBase ):
   """
   Error in OpenConfig function text validation regarding a mismatched type.

   Example:
   Error function openconfig TEST_FUNCTION path statements/statement[?]/name: expected string, found integer '1234' instead

   Args:
      functionName (str): function the message originated from
      openConfigPath (str): OC path the message originated from
      expTypes (list): list of expected types
      actualType (str): the type that originated the error
      actualValue (any): the value that originated the error
   """

   severity = Severity.error

   def __init__( self, functionName, openConfigPath, expTypes, actualType,
                 actualValue ):
      actualValueStr = ""
      if actualType not in [ "list", "container", "null" ]:
         # Render value in output
         if actualType == "boolean":
            # JSON and OpenConfig use lowercase
            actualValue = "true" if actualValue else "false"
         actualValueStr = f" '{actualValue}'"
      if len( expTypes ) == 1:
         expTypeStr = expTypes[ 0 ]
      elif len( expTypes ) == 2:
         expTypeStr = " or ".join( expTypes )
      else:
         # 3 or more
         expTypeStr = ", ".join( expTypes[ : -1 ] ) + ", or " + expTypes[ -1 ]
      msg = f"expected {expTypeStr}, found {actualType}{actualValueStr} instead"
      super().__init__( functionName, openConfigPath, msg )

class OpenConfigFunctionValidationUnexpectedPathError(
      OpenConfigFunctionValidationMessageBase ):
   """
   Error in OpenConfig function text validation regarding an unexpected path.

   Example:
   Error function openconfig TEST_FUNCTION: unexpected attribute 'oops'

   Args:
      functionName (str): function the message originated from
      openConfigPath (str): OC path the message originated from
      key (str): Unexpected trailing container/list/leaf
   """

   severity = Severity.error

   def __init__( self, functionName, openConfigPath, key ):
      msg = f"unexpected attribute '{key}'"
      super().__init__( functionName, openConfigPath, msg )

class OpenConfigFunctionValidationMissingPathError(
      OpenConfigFunctionValidationMessageBase ):
   """
   Error in OpenConfig function text validation regarding a missing path.

   Example:
   Error function openconfig TEST_FUNCTION: missing attribute 'statement'

   Args:
      functionName (str): function the message originated from
      openConfigPath (str): OC path the message originated from
      key (str): Missing trailing container/list/leaf
   """

   severity = Severity.error

   def __init__( self, functionName, openConfigPath, key ):
      msg = f"missing attribute '{key}'"
      super().__init__( functionName, openConfigPath, msg )

class OpenConfigFunctionDecodeError( OpenConfigFunctionValidationMessageBase ):
   """
   Error in OpenConfig function text validation when the JSON cannot be decoded.

   Example:
   Error function openconfig TEST_FUNCTION: invalid JSON syntax

   Args:
      functionName (str): function the message originated from
      openConfigPath (str): OC path the message originated from
   """

   severity = Severity.error

   def __init__( self, functionName, openConfigPath ):
      msg = "invalid JSON syntax"
      super().__init__( functionName, openConfigPath, msg )

class OpenConfigFunctionValidationDuplicateStatementError(
      OpenConfigFunctionValidationMessageBase ):
   """
   Error in OpenConfig function text validation when two statements have the same
   name.

   Example:
   Error function openconfig TEST_FUNCTION path statements/statement: redefinition of statement 'FOO'

   Args:
      functionName (str): function the message originated from
      openConfigPath (str): OC path the message originated from
      name (str): the statement name
   """

   severity = Severity.error

   def __init__( self, functionName, openConfigPath, name ):
      msg = f"redefinition of statement '{name}'"
      super().__init__( functionName, openConfigPath, msg )

class OpenConfigFunctionValidationDuplicateContainerKey(
      OpenConfigFunctionValidationMessageBase ):
   """
   Error in OpenConfig function text validation when json text defines the same
   object key twice.

   Example:
   Error function openconfig TEST_FUNCTION: attribute 'conditions' appears multiple times in a container.

   Args:
      functionName (str): function the message originated from
      openConfigPath (str): OC path the message originated from
      key (str): the key name
   """

   severity = Severity.error

   def __init__( self, functionName, openConfigPath, key ):
      msg = f"attribute '{key}' appears multiple times in a container"
      super().__init__( functionName, openConfigPath, msg )

class OpenConfigFunctionValidationFunctionNameMismatchError(
      OpenConfigFunctionValidationMessageBase ):
   """
   Error in OpenConfig function text validation when policy-definition/name and
   the CLI token name don't match.

   Example:
   Error function openconfig TEST_FUNCTION path name: CLI name 'TEST_FUNCTION' does not match JSON name 'TEST_FUNCTION_2'

   Args:
      functionName (str): function the message originated from
      openConfigPath (str): OC path the message originated from
      jsonFunctionName (str): the name as configured in the json.
   """

   severity = Severity.error

   def __init__( self, functionName, openConfigPath, jsonFunctionName ):
      msg = ( f"CLI name '{functionName}' does not match JSON name "
              + f"'{jsonFunctionName}'" )
      super().__init__( functionName, openConfigPath, msg )

class OpenConfigFunctionValidationInvalidFunctionNameError(
      OpenConfigFunctionValidationMessageBase ):
   """
   Error in OpenConfig function text validation when the function name is not a valid
   RCF function name.

   Example:
   Error function openconfig TEST_FUNCTION path name: function name "INVALID-NAME" is not a valid RCF function name

   Args:
      functionName (str): function the message originated from
      openConfigPath (str): OC path the message originated from
      jsonFunctionName (str): the name as configured in the json.
   """

   severity = Severity.error

   def __init__( self, functionName, openConfigPath, jsonFunctionName ):
      msg = f"function name '{jsonFunctionName}' is not a valid RCF function name"
      super().__init__( functionName, openConfigPath, msg )

class OpenConfigFunctionValidationInvalidLeafValueError(
      OpenConfigFunctionValidationMessageBase ):
   """
   Error in OpenConfig function text validation when a leaf value is not valid.

   Example:
   Error function openconfig TEST_FUNCTION path statements/statement[ FOO ]/actions/bgp-actions/config/set-med: invalid string value '+XYZ'

   Args:
      functionName (str): function the message originated from
      openConfigPath (str): OC path the message originated from
      dataType (str): the type that originated the error
      value (any): the invalid leaf value
   """

   severity = Severity.error

   def __init__( self, functionName, openConfigPath, dataType, value ):
      msg = f"invalid {dataType} value '{value}'"
      super().__init__( functionName, openConfigPath, msg )

class RcfOpenConfigDiag( RcfDiagGeneric ):
   """ Collects all error and warning messages that occurs during the
   compilation.

   Attributes (from RcfDiagGeneric):
     - allErrors (list): List of messages with severity "Error"
     - allWarnings (list): List of messages with severity "Warning"
   """

   def openConfigFunctionValidationUnexpectedTypeError( self, functionName,
                                                        openConfigPath, expTypes,
                                                        actualType, actualValue ):
      """ Callback when the compiler found an error about a mismatched type in the
      OC json.

      Args:
         functionName (str): function the message originated from
         openConfigPath (str): OC path the message originated from
         expTypes (list): list of expected types
         actualType (str): the type that originated the error
         actualValue (any): the value that originated the error
      """
      error = OpenConfigFunctionValidationUnexpectedTypeError(
            functionName, openConfigPath, expTypes, actualType, actualValue )
      self.add( error )

   def openConfigFunctionValidationUnexpectedPathError( self, functionName,
                                                        openConfigPath, key ):
      """ Callback when the compiler found an error about an extra path in the OC
      json.

      Args:
         functionName (str): function the message originated from
         openConfigPath (str): OC path the message originated from
         key (str): Unexpected trailing container/list/leaf
      """
      error = OpenConfigFunctionValidationUnexpectedPathError(
            functionName, openConfigPath, key )
      self.add( error )

   def openConfigFunctionValidationMissingPathError( self, functionName,
                                                     openConfigPath, key ):
      """ Callback when the compiler found an error about a missing path in the OC
      json.

      Args:
         functionName (str): function the message originated from
         openConfigPath (str): OC path the message originated from
         key (str): Missing trailing container/list/leaf
      """
      error = OpenConfigFunctionValidationMissingPathError(
            functionName, openConfigPath, key )
      self.add( error )

   def openConfigFunctionDecodeError( self, functionName, openConfigPath ):
      """ Callback when the compiler found an error about invalid json.

      Args:
         functionName (str): function the message originated from
         openConfigPath (str): OC path the message originated from
      """
      error = OpenConfigFunctionDecodeError( functionName, openConfigPath )
      self.add( error )

   def openConfigFunctionValidationDuplicateStatementError( self, functionName,
                                                            openConfigPath, name ):
      """ Callback when the compiler found an error with two elements sharing the
      same internal key.

      Args:
         functionName (str): function the message originated from
         openConfigPath (str): OC path the message originated from
         name (str): the statement name
      """
      error = OpenConfigFunctionValidationDuplicateStatementError(
            functionName, openConfigPath, name )
      self.add( error )

   def openConfigFunctionValidationDuplicateContainerKey( self, functionName,
                                                          openConfigPath, key ):
      """ Callback when the compiler found an error with json text containing the
      same object key twice.

      Args:
         functionName (str): function the message originated from
         openConfigPath (str): OC path the message originated from
         key (str): the key name
      """
      error = OpenConfigFunctionValidationDuplicateContainerKey(
            functionName, openConfigPath, key )
      self.add( error )

   def openConfigFunctionValidationFunctionNameMismatchError( self, functionName,
                                                              openConfigPath,
                                                              jsonFunctionName ):
      """ Callback when the compiler found an error with the json function name
      and CLI function name mismatching.

      Args
         functionName (str): function the message originated from
         openConfigPath (str): OC path the message originated from
         jsonFunctionName (str): the name as configured in the json.
      """
      error = OpenConfigFunctionValidationFunctionNameMismatchError(
            functionName, openConfigPath, jsonFunctionName )
      self.add( error )

   def openConfigFunctionValidationInvalidFunctionNameError( self, functionName,
                                                             openConfigPath,
                                                             jsonFunctionName ):
      """ Callback when the compiler found an error with the function name
      as configured on the CLI not being a valid RCF function name.

      Args
         functionName (str): function the message originated from
         openConfigPath (str): OC path the message originated from
      """
      error = OpenConfigFunctionValidationInvalidFunctionNameError(
            functionName, openConfigPath, jsonFunctionName )
      self.add( error )

   def openConfigFunctionValidationInvalidLeafValueError( self, functionName,
                                                          openConfigPath,
                                                          dataType, value ):
      """ Callback when the compiler found an error with the value of a leaf.

      Args
         functionName (str): function the message originated from
         openConfigPath (str): OC path the message originated from
         dataType (str): the type that originated the error
         value (any): the invalid leaf value
      """
      error = OpenConfigFunctionValidationInvalidLeafValueError(
            functionName, openConfigPath, dataType, value )
      self.add( error )
