# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CallbackRegistry import CallbackRegistry
from RcfFunctionTextGen import (
   fragmentsToFunctionText,
   RcfFragmentsMixin,
   TextBlock,
)
import RcfOpenConfigCommon

def generateOpenConfigFunctionTexts( openConfigFunctions, ipv6PfxList ):
   """Generate RCF function text from a dictionary of validated OpenConfig functions.

   Args:
      openConfigFunctions (dict): validated OpenConfig functions, keyed by function
         name.
   Returns:
      A dictionary containing RCF function text, keyed by function name.
   """
   openConfigFunctionTexts = {}
   statementTextGenerator = RcfOpenConfigFunctionStatementTextGenerator(
         ipv6PfxList )
   for functionName, openConfigFunction in openConfigFunctions.items():
      textBlocks = statementTextGenerator.generate( openConfigFunction )
      openConfigFunctionText = fragmentsToFunctionText( functionName,
                                                        textBlocks )
      openConfigFunctionTexts[ functionName ] = openConfigFunctionText
   return openConfigFunctionTexts

class RcfOpenConfigFunctionStatementTextGenerator( CallbackRegistry,
                                                   RcfFragmentsMixin ):
   """Generates RCF text segments (TextBlock) for an OpenConfig function.
   These can be assembled into an RCF function.

   This class follows the same callback pattern as RcfOpenConfigFunctionValidator
   and uses the same format for its callback key. Some examples:
   - "" handles text generation for the root node
   - ".statements" handles text generation for the contents of statements
   - ".statements.statement" handles text generation contents of
      statements/statement, which is guaranteed to be a list.
   - ".statements.statement[]" handles text generation for each element of statement

   Args:
      ipv6PfxList (Acl::Ipv6PrefixList): Configured ipv6 prefix lists. This
         is needed by the match-prefix-set condition to determine whether to
         emit prefix_list_v4 or prefix_list_v6.
   """
   def __init__( self, ipv6PfxList ):
      self.ipv6PfxList = ipv6PfxList

   def generate( self, openConfigFunction ):
      """Generate RCF text segments.

      Args:
         functionName (str): the function name
         openConfigFunction (dict): a validated OpenConfig function
      Returns:
         A list of TextBlock objects to be consumed by
         statementsToFunctionText.
      """
      statementRcfTexts = []
      for statement in openConfigFunction[ "statements" ][ "statement" ]:
         commentLines = [ f"statement {statement[ 'name' ]}" ]
         conditionBlocks = self.getConditions( statement.get( "conditions", {} ) )
         actionBlocks = self.getActions( statement.get( "actions", {} ) )
         textBlock = TextBlock.initEmpty()
         textBlock.commentLines = commentLines
         for conditionBlock in conditionBlocks:
            textBlock.updateWith( conditionBlock )
         for actionBlock in actionBlocks:
            textBlock.updateWith( actionBlock )
         statementRcfTexts.append( textBlock )
      return statementRcfTexts

   def getConditions( self, conditions ):
      return self._getStatementRecursive( conditions,
                                          ".statements.statement[].conditions" )

   def getActions( self, actions ):
      return self._getStatementRecursive( actions,
                                          ".statements.statement[].actions" )

   def _getStatementRecursive( self, data, callbackKey ):
      textBlocks = []
      for callback in self.getCallbacks( callbackKey ):
         textBlocks.append( callback( self, data ) )
      if isinstance( data, dict ):
         for key, value in sorted( data.items() ):
            textBlocks += self._getStatementRecursive( value,
                                                       callbackKey + f".{key}" )
      return textBlocks

# Begin text generation functions

# Condition generation

@RcfOpenConfigFunctionStatementTextGenerator.registerCallback(
      ".statements.statement[].conditions.match-prefix-set.config" )
def generateMatchPrefixSet( self, data ):
   prefixSetName = data[ 'prefix-set' ]
   proto = 4
   if prefixSetName in self.ipv6PfxList:
      proto = 6
   textBlock = TextBlock.initEmpty()
   textBlock.conditionFragments.append( self.matchPrefixSet( prefixSetName, proto ) )
   return textBlock

@RcfOpenConfigFunctionStatementTextGenerator.registerCallback(
      ".statements.statement[].conditions.bgp-conditions.as-path-length.config" )
def generateAsPathLength( self, data ):
   operator = data[ 'operator' ]
   value = data[ 'value' ]
   textBlock = TextBlock.initEmpty()
   textBlock.conditionFragments += self.asPathLength( operator, value )
   return textBlock

@RcfOpenConfigFunctionStatementTextGenerator.registerCallback(
      ".statements.statement[].conditions.bgp-conditions.config.local-pref-eq" )
def generateLocalPrefEq( self, data ):
   textBlock = TextBlock.initEmpty()
   textBlock.conditionFragments.append( self.matchLocalPref( data ) )
   return textBlock

@RcfOpenConfigFunctionStatementTextGenerator.registerCallback(
      ".statements.statement[].conditions.bgp-conditions.config.route-type" )
def generateRouteType( self, data ):
   assert( data in [ 'INTERNAL', 'EXTERNAL' ] )
   routeType = {
      'INTERNAL': 'IBGP',
      'EXTERNAL': 'EBGP',
   }[ data ]
   textBlock = TextBlock.initEmpty()
   textBlock.conditionFragments.append( self.matchRouteType( routeType ) )
   return textBlock

# Action generation

@RcfOpenConfigFunctionStatementTextGenerator.registerCallback(
      ".statements.statement[].actions.bgp-actions.config.set-med" )
def generateSetMed( self, data ):
   # OpenConfig IGP comes form the bgp-set-med-type OC type. It is defined as the IGP
   # cost towards the next hop.
   # METRIC comes from the arista-bgp-set-med-type augmentation. It is defined as the
   # IGP metric.
   # arista-bgp-set-med-type integer
   value = None
   metricType = None
   if isinstance( data, int ):
      value = data
      metricType = "medNormal"
   else:
      assert isinstance( data, str )
      # arista-bgp-set-med-type string
      metricType, value = RcfOpenConfigCommon.parseSetMed( data )
      # arista-bgp-set-med-type enum
      if not metricType:
         assert data == "IGP"
         metricType = "medIgpNexthopCost"
   textBlock = TextBlock.initEmpty()
   textBlock.modificationFragments += self.setMed( metricType, value )
   return textBlock

@RcfOpenConfigFunctionStatementTextGenerator.registerCallback(
      ".statements.statement[].actions.bgp-actions.config.set-local-pref" )
def generateLocalPref( self, data ):
   textBlock = TextBlock.initEmpty()
   textBlock.modificationFragments += self.setLocalPref( data )
   return textBlock

@RcfOpenConfigFunctionStatementTextGenerator.registerCallback(
   ".statements.statement[].actions.bgp-actions.set-as-path-prepend.config" )
def generateSetAsPathPrepend( self, data ):
   repeat = data[ 'repeat-n' ]
   asn = str( data.get( 'asn', 'instance.as' ) )
   value = " ".join( asn for _ in range( repeat ) )
   textBlock = TextBlock.initEmpty()
   textBlock.modificationFragments += self.setAsPathPrepend( value )
   return textBlock

@RcfOpenConfigFunctionStatementTextGenerator.registerCallback(
      ".statements.statement[].actions.bgp-actions.config.set-next-hop" )
def generateSetNextHop( self, data ):
   textBlock = TextBlock.initEmpty()
   textBlock.modificationFragments += self.setNextHop( data )
   return textBlock

@RcfOpenConfigFunctionStatementTextGenerator.registerCallback(
   ".statements.statement[].actions.bgp-actions.config.set-route-origin" )
def generateSetRouteOrigin( self, data ):
   textBlock = TextBlock.initEmpty()
   textBlock.modificationFragments += self.setOrigin( data )
   return textBlock

@RcfOpenConfigFunctionStatementTextGenerator.registerCallback(
      ".statements.statement[].actions.config.policy-result" )
def generatePolicyResult( self, data ):
   returnValue = None
   if data != "NEXT_STATEMENT":
      returnValue = self.returnValue( data == "ACCEPT_ROUTE" )
   textBlock = TextBlock.initEmpty()
   textBlock.returnValue = returnValue
   return textBlock

@RcfOpenConfigFunctionStatementTextGenerator.registerCallback(
   ".statements.statement[].actions.isis-actions.config.set-level" )
def generateSetLevel( self, data ):
   getLevel = {
      "LEVEL_1": "{ LEVEL1 }",
      "LEVEL_2": "{ LEVEL2 }",
      "LEVEL_1_2": "{ LEVEL1, LEVEL2 }"
   }
   level = f"{{ LEVEL{data} }}" if isinstance( data, int ) else getLevel.get( data,
                                                                              None )
   textBlock = TextBlock.initEmpty()
   textBlock.modificationFragments += self.setLevel( level )
   return textBlock

@RcfOpenConfigFunctionStatementTextGenerator.registerCallback(
   ".statements.statement[].actions.isis-actions.config.set-metric" )
def generateSetMetric( self, data ):
   textBlock = TextBlock.initEmpty()
   textBlock.modificationFragments += self.setMetric( data )
   return textBlock
