# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import antlr4
from Toggles import RcfLibToggleLib

class UnsupportedFeatureToggle( antlr4.error.Errors.FailedPredicateException ):
   def __init__( self, unsupportedCtx, *args, **kwargs ):
      self.unsupportedCtx = unsupportedCtx
      super().__init__( *args, **kwargs )

class FeatureUsageParserChecker:
   # the toggle name this is checking
   toggleName = ""

   def isBeingUsed( self, ctx ):
      """ Check whether a given feature is used while parsing the code
      Args:
         - ctx: the context a given "toggled" was registered as a semantic action

      returns list of unsupported ctxs assuming feature is used or None if all ctx
         are supported.
      """
      raise NotImplementedError

class RcfLabelingIfElseChainUsed( FeatureUsageParserChecker ):
   toggleName = "RcfLabelingIfElseChain"

   def isBeingUsed( self, ctx ):
      # avoid cyclic dep: this file is imported by RcfParser itself
      from RcfParser import RcfParser # pylint: disable=import-outside-toplevel
      if isinstance( ctx, RcfParser.IfStmtContext ):
         unsupportedCtx = []
         if ctx.ifLbl:
            unsupportedCtx += [ ctx.ifLbl, ctx.ifPart ]
         if ctx.elseLbl:
            unsupportedCtx += [ ctx.elsePart, ctx.elseLbl ]
      return unsupportedCtx or None

class RcfSourceSessionInterfaceBandwidthUsed( FeatureUsageParserChecker ):
   toggleName = "RcfSourceSessionInterfaceBandwidth"

   def isBeingUsed( self, ctx ):
      return [ ctx ]

class RcfFlowspecActionsUsed( FeatureUsageParserChecker ):
   toggleName = "RcfFlowspecActions"

   def isBeingUsed( self, ctx ):
      return [ ctx ]

class RcfFunctionArgsPhase1Used( FeatureUsageParserChecker ):
   toggleName = "RcfFunctionArgsPhase1"

   def isBeingUsed( self, ctx ):
      return [ ctx ]

class RcfRouteMapConverterUsed( FeatureUsageParserChecker ):
   toggleName = "RcfRouteMapConverter"

   def isBeingUsed( self, ctx ):
      return [ ctx ]

class RcfRtMembershipRouteTargetUsed( FeatureUsageParserChecker ):
   toggleName = "RcfRtMembershipRouteTarget"

   def isBeingUsed( self, ctx ):
      return [ ctx ]

class RcfResolutionRibsUsed( FeatureUsageParserChecker ):
   toggleName = "RcfResolutionRibs"

   def isBeingUsed( self, ctx ):
      return [ ctx ]

class RcfAggregateLinkBandwidthUsed( FeatureUsageParserChecker ):
   toggleName = "RcfAggregateLinkBandwidth"

   def isBeingUsed( self, ctx ):
      return [ ctx ]

class RcfMatchDplUsed( FeatureUsageParserChecker ):
   toggleName = "RcfMatchDpl"

   def isBeingUsed( self, ctx ):
      # avoid cyclic dep: this file is imported by RcfParser itself
      from RcfParser import RcfParser # pylint: disable=import-outside-toplevel
      if isinstance( ctx, RcfParser.ExtRefContext ):
         if ctx.EXTERNAL_REF().getText() == "dynamic_prefix_list":
            return [ ctx ]
      return None

def toggled( parser, toggle ):
   featureUsedFnPerToggleName = {
      RcfLabelingIfElseChainUsed.toggleName: RcfLabelingIfElseChainUsed(),
      RcfFlowspecActionsUsed.toggleName: RcfFlowspecActionsUsed(),
      RcfFunctionArgsPhase1Used.toggleName: RcfFunctionArgsPhase1Used(),
      RcfRouteMapConverterUsed.toggleName: RcfRouteMapConverterUsed(),
      RcfRtMembershipRouteTargetUsed.toggleName: RcfRtMembershipRouteTargetUsed(),
      RcfResolutionRibsUsed.toggleName: RcfResolutionRibsUsed(),
      RcfSourceSessionInterfaceBandwidthUsed.toggleName:
         RcfSourceSessionInterfaceBandwidthUsed(),
      RcfMatchDplUsed.toggleName: RcfMatchDplUsed(),
      RcfAggregateLinkBandwidthUsed.toggleName: RcfAggregateLinkBandwidthUsed(),
   }
   ctx = parser._ctx # pylint: disable=protected-access
   contextUsingFeature = featureUsedFnPerToggleName[ toggle ].isBeingUsed( ctx )
   featureEnabled = getattr( RcfLibToggleLib, 'toggle' + toggle + 'Enabled' )()
   if contextUsingFeature and not featureEnabled:
      message = f"uses unsupported feature: {toggle}"
      raise UnsupportedFeatureToggle( contextUsingFeature, parser, message=message )
