# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import os
import re
import time

def isPreviousReloadASU():
   '''Check if the previous reload was an ASU reload, verifying it by
   checking kexec params (i.e. /proc/cmdline) and see if 'asu_hitless'
   is in there
   '''
   kexecCmdLinePath = '/proc/cmdline'
   ssuPattern = 'arista.asu_hitless'
   if os.path.isfile( kexecCmdLinePath ):
      with open( kexecCmdLinePath ) as f:
         for line in f:
            m = re.search( ssuPattern, line )
            if m:
               return True
   return False



def checkAsuReloadTime( ctx ):
   '''Check the content of /proc/cmdline to see if the previous reload
   was an ASU reload. If it was, check the timestamp to make sure we have
   waited a reasonable amount of time before performing the next ASU reload.
   A warning message will be printed out if the next ASU reload is issued
   within 20 minutes after the previous ASU reload time
   '''
   if isPreviousReloadASU():
      try:
         import LazyMount
         import Cell
         reloadHistory = LazyMount.mount( ctx.mode.entityManager,
               Cell.path( "sys/reset/history" ),
               "System::ResetCauseHistory", "ri" )
         if reloadHistory:
            lastReload = reloadHistory.rebootHistory.get( 0 )
            if lastReload:
               reloadCause = lastReload.causeQ.front()
               reloadTime = reloadCause.timestamp
               currentTime = time.time()
               duration = int( currentTime - reloadTime )
               if duration < 1200:
                  waitTime = int( ( 1200 - duration ) / 60 ) + 1
                  ctx.addWarning(
                        "Insufficient uptime since the previous reload: it is "\
                        "recommended to wait another %d minutes for the system "\
                        "to be ready for the next SSU" % ( waitTime ) )
      except AttributeError:
         return True
   return True

def Plugin( ctx ):
   category = [ 'ASU' ]
   ctx.addPolicy( checkAsuReloadTime, category )
