# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import os
import math

BLOCKS_NEEDED_SLOP_SCALE = 1.05
KERNEL_PARAM = "bootimgdevice="
KERNEL_PARAMS_PATH = "/mnt/flash/kernel-params"

# Copied from Cli/FileCliUtil.py to avoid dependency
def sizeFmt( num ):
   for x in [ 'bytes', 'KB', 'MB', 'GB', 'TB', 'PB' ]:
      if num < 1024.0:
         return "%3.1f %s" % ( num, x )
      num /= 1024.0
   return None

def getBootDevicePath():
   kernelParamsPath = os.environ.get( "STORAGE_DEVICES_RELOAD_POLICY_KERNEL_PARAMS",
                                      KERNEL_PARAMS_PATH )
   try:
      with open( kernelParamsPath, "r" ) as kparams:
         for line in kparams:
            line = line.strip()
            if line.startswith( KERNEL_PARAM ):
               return line[ len( KERNEL_PARAM ): ]
   except IOError:
      pass
   return os.path.join( os.environ.get( "FILESYSTEM_ROOT", "/mnt" ), "flash" )

def fileSize( path, blockSize ):
   return math.ceil( os.stat( path ).st_size / blockSize )

def checkStorageSpaceAvailable( ctx ):
   """Check whether there is enough free space to safely reboot. At reboot the
   image to which the boot config points is compared against a hidden copy of
   the previously booted image, and if they do not match, the new boot image is
   copied to the hidden location. If there is not sufficient free space Aboot
   will halt the boot process. We print an error and abort the reload if we
   think this might happen."""

   bootDevice = getBootDevicePath()

   # Get the amount of free space left in the boot device
   info = os.statvfs( bootDevice )
   freeBlocks = info.f_bavail
   blockSize = info.f_frsize

   # Find the difference in size between the new SWI and the original SWI
   try:
      originalBlocks = fileSize( os.path.join( bootDevice, ".boot-image.swi" ),
                                 blockSize )
   except OSError:
      # .boot-image.swi does not exist yet, therefore it does not take up any space
      originalBlocks = 0

   newBlocks = fileSize( ctx.nextImage, blockSize )
   blocksNeeded = newBlocks - originalBlocks
   blocksNeeded *= BLOCKS_NEEDED_SLOP_SCALE # Scale up by a small amount to be sure

   if blocksNeeded > freeBlocks:
      requiredSpace = sizeFmt( blocksNeeded * blockSize )
      freeSpace = sizeFmt( freeBlocks * blockSize )
      ctx.addError( "Insufficient free space on the internal flash disk to copy "
                    "the boot image.\nMake sure at least {} are free and "
                    "try again. Remaining free space: {}"
                    .format( requiredSpace, freeSpace ) )
      return False

   return True

def Plugin( ctx ):
   # Define a set of categories for this plugin.
   category = [ "StorageDevices", "General" ]
   # Register the check with the context.
   ctx.addPolicy( checkStorageSpaceAvailable, category )
