# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import re

def getVerNums( ver ):
   '''Given a version string (i.e. '4.20.1F'), parse the string and return
   a list of version numbers in the string.

   Arguments:
      version: (str) EOS version string

   Returns:
      (list of int) EOS version numbers
   '''
   return [ int( n ) for n in re.split( r'(\d+)', ver ) if n.isdigit() ]

def getProtocolAgentModel( mode ):
   '''Check the default and currently configured routing protocols model
   in Sysdb.

   Arguments:
      mode (CliMode) the CLI mode
   Returns:
      (str, str) a tuple of (default model, configured model)
   '''
   try:
      l3Conf = mode.entityManager.root()[ 'l3' ][ 'config' ]
      return l3Conf.protocolAgentModelDefault, l3Conf.configuredProtocolAgentModel
   except ( AssertionError, AttributeError, KeyError, TypeError ):
      return '', ''

def getStartupConfigProtocolsModel():
   '''Check the startup config to see if a protocol model is configured.

   Returns:
      startupConfigModel: (str) the currently configured protocols model
   '''
   startupConfigPath = '/mnt/flash/startup-config'
   if 'SIMULATION_VMID' in os.environ:
      startupConfigPath = '/tmp/routingprotocolsmodel-startup-config'

   startupConfigModel = ''
   try:
      modelPattern = r'service routing protocols model (ribd|multi-agent)'
      with open( startupConfigPath ) as f:
         for line in f:
            modelMatch = re.search( modelPattern, line )
            if modelMatch:
               startupConfigModel = modelMatch.group( 1 )
               break
   # Catch IOError for python2.x compatibility.
   except IOError:
      pass

   return startupConfigModel

def checkProtocolAgentModel( ctx ):
   '''Determine if the next image will change the routing protocols model
   and warn the user if the current protocols model won't be preserved.

   Starting in version 4.24.1F, EOS always saves the currently configured
   routing protocols model in the startup-config, overriding the default.
   If EOS is older than that, then we need to warn the user that the
   behavior will change after reloading the next image.

   Arguments:
      ctx: (_PluginContext): Any errors, warnings are added here. Used to
         retrieve the current/next image versions.

   Returns:
      True because this plugin is a warning so it will never block reload.
   '''
   defaultModel, configuredModel = getProtocolAgentModel( ctx.mode )
   startupConfigModel = getStartupConfigProtocolsModel()
   if defaultModel == 'ribd':
      if ( getVerNums( ctx.currentVersion.version() ) < getVerNums( '4.24.1F' )
           and configuredModel == 'ribd' and not startupConfigModel ):
         warningMsg = ( 'Routing protocol model will change from ribd to '
                        'multi-agent in this image unless the preference '
                        'is saved in startup-config.' )
         ctx.addWarning( warningMsg )
   if ( getVerNums( ctx.currentVersion.version() ) >= getVerNums( '4.32.0F' )
         and startupConfigModel == 'ribd' ):
      warningMsg = ( "Routing protocol model ribd is unavailable in 4.32.0F and "
                     "newer releases. The switch will be forced into multi-agent "
                     "mode after reload." )
      ctx.addWarning( warningMsg )

   return True

def Plugin( ctx ):
   category = [ 'ASU', 'ASU+', 'General' ]
   ctx.addPolicy( checkProtocolAgentModel, category )
