# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliMatcher
from CliPlugin.TeCli import ( parseMixedAdminGroupList,
                              populateAdminGroupValuesAndNames )
import Tac
from TypeFuture import TacLazyType

RsvpLerAdminGroupConstraintsWithNames = \
   TacLazyType( 'Rsvp::RsvpLerAdminGroupConstraintsWithNames' )

def dictToTaccForCli( entityDict, tacEntity ):
   '''Simultaneously checks whether dictionary and entity attributes match
      and if they don't, sets tacEntity to values in entityDict. Collection
      attributes can be specified as a dict, list, or set.

      entityDict {
         'attributeName' : attributeValue, # Value attribute
         'collectionName' : { key1: val1, key2: val2 }, # Dict
         'collectionName' : [ ( key1, val1 ), ( key2, val2 ) ], # List
         'collectionName' : { key1, key2 }, # Set
      }
   '''
   changed = False
   # pylint: disable-msg=R1702, too-many-nested-blocks
   if entityDict:
      for attr, val in entityDict.items():
         if isinstance( val, ( dict, list, set ) ):
            valDict = val
            if isinstance( val, list ):
               valDict = dict( val )
            if isinstance( val, set ):
               valDict = dict().fromkeys( val, True )

            collection = getattr( tacEntity, attr )
            if len( valDict ) != len( collection ):
               changed = True
            if not valDict:
               collection.clear()
               continue

            if collection.tacAttr.isVector:
               # Indexing for vectors needs to start from 0 and it needs to be
               # consecutive
               valDict = dict( sorted( valDict.items() ) )
               assert list( valDict.keys() ) == list( range( len( valDict ) ) )
               for k, v in valDict.items():
                  if k < len( collection ):
                     if collection[ k ] != v:
                        changed = True
                        collection[ k ] = v
                  else:
                     changed = True
                     collection.push( v )
               while len( valDict ) < len( collection ):
                  changed = True
                  collection.pop()
               continue

            stateKeys = []
            for k, v in valDict.items():
               if k not in collection or collection[ k ] != v:
                  changed = True
                  collection[ k ] = v
               stateKeys.append( k )

            for k in collection:
               if k not in stateKeys:
                  changed = True
                  del collection[ k ]
         elif getattr( tacEntity, attr ) != val:
            changed = True
            setattr( tacEntity, attr, val )
   return changed

def dictToTaccNominalForCli( entityDict, tacNominal ):
   '''Simultaneously checks whether dictionary and nominal attributes match
      and if they don't, creates a copy of the nominal using the values in
      entityDict. Collection attributes can be specified as a dict, list, or set.

      entityDict {
         'attributeName' : attributeValue, # Value attribute
         'collectionName' : { key1: val1, key2: val2 }, # Dict
         'collectionName' : [ ( key1, val1 ), ( key2, val2 ) ], # List
         'collectionName' : { key1, key2 }, # Set
      }
   '''
   newNominal = Tac.nonConst( tacNominal )
   dictToTaccForCli( entityDict, newNominal )
   return newNominal

def bandwidthBitsToBytes( bits ):
   if bits > 1000:
      # Our CLI precision is %.2f, so round up the bits to the nearest
      # multiple of 40 (because LCM( 8, 10 ) is 40)
      # then divide by 8 to convert to bytes
      return ( ( bits + 39 ) // 40 ) * 5
   else:
      # simply round up to the next multiple of 8 and divide by 8
      return ( bits + 7 ) // 8

matcherAdminGroupInc = CliMatcher.KeywordMatcher( 'include',
            helpdesc='Must include the following admin groups' )
matcherAdminGroupExc = CliMatcher.KeywordMatcher( 'exclude',
            helpdesc='Must exclude the following admin groups' )
agRange = 127
matcherAdminGrpLists = CliMatcher.PatternMatcher( pattern='.+',
      helpdesc='Administrative Group value in hexadecimal, range or name format',
      helpname=f'<0x0-0xFFFFFFFF>,<0-{agRange}>,<0-{agRange}>-<0-{agRange}>,WORD' )

def parseAdminGroupConstraints( mode, args ):
   incAllAdminGroupMixedLists = args.get( 'INCL_ALL_MIXED_LISTS', [] )
   incAllResult = parseMixedAdminGroupList( mode, incAllAdminGroupMixedLists,
         extended=True )
   incAnyAdminGroupMixedLists = args.get( 'INCL_ANY_MIXED_LISTS', [] )
   incAnyResult = parseMixedAdminGroupList( mode, incAnyAdminGroupMixedLists,
         extended=True )
   exclAdminGroupMixedLists = args.get( 'EXCL_MIXED_LISTS', [] )
   exclResult = parseMixedAdminGroupList( mode, exclAdminGroupMixedLists,
         extended=True )

   adminGroupConstraints = RsvpLerAdminGroupConstraintsWithNames()

   # Include all, Include any, and Exclude admin group
   adminGroupConstraintsMap = { 'includeAll': incAllResult,
                                'includeAny': incAnyResult,
                                'exclude': exclResult }
   for collName, parsedAgResult in adminGroupConstraintsMap.items():
      populateAdminGroupValuesAndNames( parsedAgResult, adminGroupConstraints,
            collName, f'{collName}ByName', indexType='Rsvp::RsvpLerAdminGroupIndex' )

   return adminGroupConstraints
