#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
import sys

def inputSection( lines ):
   '''This generator returns a block of lines starting with the first/current
      line and the lines thereafter as long as they start with a space or tab.
      That is, except maybe for the first block, the first line of the returned
      block does not start with a space/tab and all subsequent lines do (this
      is a top-level-section).'''

   section = []
   for line in lines:
      if section and not line.startswith( ( ' ', '\t' ) ):
         yield section
         section = []
      section.append( line )
   if section:
      yield section

def searchSection( inLines, lineNum, currentIndent, regexPattern ):
   ''' Return a list of matching sections. Calling args:
      inLines is a list of lines
      lineNum is the index into inLines of the current line
      currentIndent is the level of indentation already obtained, simply a
             count of leading spaces.
      regexPattern is the Pattern we're searching for. It may be a
            wildcard if the previous line matched.
   returns:
      A tuple in the form (lineNum, [matchedLines])
   '''
   matchedLines = []
   numLines = len( inLines )
   while lineNum < numLines:
      line = inLines[ lineNum ]
      indent = len( line ) - len( line.lstrip() )
      if indent < currentIndent:
         break # back to a shorter indent level
      if regexPattern.match( line ):
         # We matched a line. Include it and check to see if it has any
         # submodes or subsection that are automatically incuded. Append those
         # as well. A submode or subsection is identified by an indent level
         # greater than the matched line.
         matchedLines.append( line )
         lineNum += 1
         while lineNum < numLines:
            nextLine = inLines[ lineNum ]
            nextIndent = len( nextLine ) - len( nextLine.lstrip() )
            if nextIndent <= indent:
               break
            matchedLines.append( nextLine )
            lineNum += 1
      else:
         # We didn't match. If this wasn't a command that put us in a submode
         # or enterered a new subsection, then ignore this line.
         # If this is entering a submode or subsection, then recurse to see if
         # there is a match within the submode
         lineNum += 1
         if lineNum >= numLines:
            break
         nextLine = inLines[ lineNum ]
         nextIndent = len( nextLine ) - len( nextLine.lstrip() )
         if nextIndent <= indent:
            continue    # No submode, ignore line
         # We do have a submode, see if there's a match inside it
         lineNum, matchedSubMode = searchSection( inLines, lineNum,
                                                  nextIndent, regexPattern )
         if matchedSubMode:
            # Something inside the submode matched. Output the command to get
            # into the submode plus all the matching commands from inside the
            # submode
            matchedLines.append( line )
            matchedLines += matchedSubMode
   return ( lineNum, matchedLines )

def inputSectionAll( lines ):
   ''' This is a helper generator for sectionFilter function to yield a
   "section" for show startup-config and show running-config.  A section
   is defined as any main mode config command and its subsequent sub-mode
   commands.  Currently, this only matches a single top level mode all
   sub-modes under it.  The section yielded includes newlines, so any
   subsequent actions on a section needs to bear this in mind.'''

   section = ""
   for line in lines:
      if section and not line.startswith( ( ' ', '\t' ) ):
         yield section
         section = ""
      section += line
   if section:
      yield section

def compileRes_( regexList, prefix='' ):
   # pylint: disable-next=consider-using-f-string
   return re.compile( prefix + ''.join( '(?=.*%s)' % r for r in regexList ),
                      re.DOTALL + re.IGNORECASE )

def sectionFilterGen( inputText, regexList ):
   '''
   inputText - accepts multiline input
   regexList - list of regular expressions.

   Returns: a generator of filtered section text based on the regex list provided.
   This text will be limited to parent text, matching text, and child text.
   This function may raise an re.error.
   '''

   # Using regex lookaheads to allow order independent searches.
   # First regex matches all bang seperators, so we include them
   # in the output.
   regexPattern = compileRes_( regexList )

   # return the sparse match but feed it through inputSection for scale
   for section in inputSection( inputText ):
      _, matched = searchSection( section, 0, 0, regexPattern )
      for line in matched:
         if line not in ( '\n', '' ):
            yield line

def sectionFilterAllGen( inputText, regexList ):
   '''
   inputText - accepts multiline input
   regexList - list of regular expressions.

   Returns: a generator offiltered section text based on the regex list provided.
   This text will be limited to parent text, but include all its children.
   This function may raise an re.error.
   '''
   regexPattern = compileRes_( regexList, prefix=r'(?=^\!$)|' )

   matchedSections = ( matched
                       for matched in inputSectionAll( inputText )
                       if regexPattern.match( matched ) )

   # The following state machine makes sure that bang seperators in a
   # show run/start are only outputed once for any series of bangs.
   # Since the regex above always matches bang only lines, this
   # provides a clean output.
   formerSectionBang = False
   firstBang = True
   for section in matchedSections:
      if section.strip() == '!':
         formerSectionBang = True
      else:
         if formerSectionBang and not firstBang:
            yield '!\n'
         yield section
         firstBang = False
         formerSectionBang = False

def sectionFilter( inputText, regexList ):
   # Similar to sectionFilterGen but output to stdout
   for line in sectionFilterGen( inputText, regexList ):
      sys.stdout.write( line )

def sectionFilterAll( inputText, regexList ):
   # Similar to sectionFilterAllGen but output to stdout
   for line in sectionFilterAllGen( inputText, regexList ):
      sys.stdout.write( line )
