#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import Tracing
import Tac
import CliSession
import ConfigMount
import LazyMount

__defaultTraceHandle__ = Tracing.Handle( "CliSessionAgent" )
t0 = Tracing.trace0
t1 = Tracing.trace1
t2 = Tracing.trace2

def sessionConfig( entityManager, disableAaa, cliSession=None, showAll=False,
                   showDetail=False, showSanitized=False, sessionName="",
                   showJson=False, showNoSeqNum=False ):
   # Modified from Cli/Url.py, localRunningConfigAll
   # These urls are for internal use and are not displayed externally to user.
   # Currently, there is no other better way to pass arguments to Url.get() method.
   # These urls are used to pass appropriate flags to saver functions.
   import Url # pylint: disable=import-outside-toplevel
   context = Url.Context( entityManager, disableAaa, cliSession )
   # pylint: disable-next=consider-using-f-string
   urlString = "session:/%s-session-config" % sessionName
   if showAll or showDetail:
      urlString += ":all"
   if showDetail:
      urlString += ":detail"
   if showJson:
      urlString += ":json"
   if showSanitized:
      urlString += ":sanitized"
   if showNoSeqNum:
      urlString += ":noseqnum"
   return Url.parseUrl( urlString, context )

def getSessionStatus( entityManager ):
   CliSession.waitForSessionStatusInitialized( entityManager )
   return CliSession.sessionStatus

def _processHandler( handler, handlerDir, relevantPrefixes ):
   path = handler.pathPrefix.rstrip( '/' )

   # We allow a "" path. The saver presumably does not need anything from Sysdb
   if not path:
      return

   assert handlerDir.configRoot is not None
   assert handlerDir.configRoot.rootsComplete
   assert handlerDir.configRoot.rootTrie.hasPrefix( path ), \
          f"SaveRoot {path} not covered by session roots!"
   for path in handler.requireMounts:
      if path and not handlerDir.configRoot.rootTrie.hasPrefix( path ):
         relevantPrefixes.add( path )

__mountsForSaveSessionMounted__ = False

def saveSessionMountsDone():
   return bool( __mountsForSaveSessionMounted__ )

def doMountsForSaveSession( entityManager, sessionStatus, callback ):
   if __mountsForSaveSessionMounted__:
      if callback:
         callback()
      return

   def finish():
      global __mountsForSaveSessionMounted__
      __mountsForSaveSessionMounted__ = True
      t0( 'Config root and requireMounts mounted.' )
      if callback:
         callback()

   # Load the CliSave plugins the first time this function is called.
   mg = entityManager.mountGroup()
   t0( 'Loading CliSavePlugins' )
   CliSave.maybeLoadCliSavePlugins( entityManager )

   relevantPrefixes = set()
   handlerDir = CliSession.handlerDir( entityManager )

   for typeHandlerInfo in CliSave.saveHandlers_:
      _processHandler( typeHandlerInfo, handlerDir, relevantPrefixes )

   for path in handlerDir.configRoot.root:
      t = handlerDir.configRoot.rootTrie.prefixTypename( path )
      flags = handlerDir.configRoot.rootTrie.prefixFlags( path )
      t0( 'Mounting root', path, ' type', t, ' flags', flags )
      ConfigMount.mount( entityManager, path, t, flags )

   for path in relevantPrefixes:
      flags = LazyMount.mountFlag( entityManager, path ) or 'r'
      t0( 'Mounting', path, ', flags', flags )
      mg.mount( path, '', flags )


   # Force all config mounts too. This may help with BUG222721.
   ConfigMount.doDeferredMounts( block=False )

   mg.close( blocking=False, callback=finish )
