# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=chained-comparison
# pylint: disable=consider-using-f-string

import os
import re
import json
import subprocess
import textwrap
import traceback
import weakref

import Fru
import Tac
import Tracing
from SysConstants.if_arp_h import ARPHRD_ETHER

__defaultTraceHandle__ = Tracing.Handle( "Fru.SfFruHelper" )
Tracing.traceSettingIs( ",".join( [ os.environ.get( "TRACE", "" ) ] +
   [ "Fru.SfFruHelper/*" ] ) )
t0 = Tracing.trace0

# Keys used in device cache dictionary
FIRST_MAC = "firstMacAddr"
DEVICE = "device"
MAC = "macAddr"
PCI = "pciAddr"
DRIVER = "pciDriver"
PORT = "port"
ROLE = "role"
TYPE = "type"
VENDOR_ID = "vendor_id"
DEVICE_ID = "device_id"
VF2PF = "vf2pf"

# global constants
SYS_CLASS_NET = "/sys/class/net"
DEVICE_CACHE_FILE = "/var/run/sfaFruPluginDevices.json"
SUPPORTED_INTF_REGEX = r"(ma|hv_ma|vmnicet|et|hv_et|eth|eno|ens|enp\d+s\d+f"\
    r"|enp\d+s)(\d+)"

DRIVER_DEFAULT = "BESS"

CLI_CONFIG_MODE = textwrap.dedent( """
   enable
   configure terminal
   """ )

# Interface types
INTFTYPE_MGMT = "ma"
INTFTYPE_ETH = "et"
INTFTYPE_VIRT = "vm"
INTFTYPE_CONTAINER = "co"

# Reserved ports
PORT_INVALID = 0
PORT_MGMT = 99          # Should we change it to 100 now that it is not used for Dps
PORT_PHY_START = 1
PORT_PHY_END = 99       # Should we change it to 100 now that it is not used for Dps
PORT_VIRT_START = 101
PORT_VIRT_END = PORT_VIRT_START + 63
PORT_MAX = PORT_VIRT_END + 1

# TODO FixBUG402968: change the device cache to below format:
# {
#    "firstMacAddr": "aa:aa:aa:aa:aa:aa",
#    "devices": {
#       "et1" : {
#          "pci": "00:00:11",
#          "driver": "virtio",
#          "mac": "aa:aa:aa:aa:aa:aa",
#          "portId": 1",
#          "role": "Management",
#          "description": ""
#       }
#    }
# }

class SfFruHelper:
   def __init__( self, driver ):
      self._portId = PORT_INVALID
      self._portIdPool = []
      self._driver = weakref.proxy( driver )
      self._veosConfig = self._driver.veosConfig
      self.kernelDeviceNames = {} # lookup: portId -> devName
      self._frontPanelPrefix = "vmnicet" if self.veosConfig( "MODE" ) == "test" \
         else "et"

      """
      A cache is used when in sfe mode to handle the case where
      ProcMgr is restarted and Fru is restarted as well. When Fru runs
      for the second+ time, there are no kernel interfaces because they have
      been allocated to DPDK drivers. Since we can't scan the kernel network
      interfaces we cache them so next time we run we just read the cache
      and construct the interface inventories and MACs from the cache.
      """
      self.deviceCache = None

      intfList = os.environ.get( "FORCE_MAP_INTERFACE_REGEX_LIST",
            self.veosConfig( "FORCE_MAP_INTERFACE_REGEX_LIST", "" ) ).split( "," )
      self._forceMappedIntfRegxList = [ d.strip() for d in intfList if d.strip() ]

      intfList = os.environ.get( "FORCE_IGNORE_INTERFACE_REGX_LIST",
            self.veosConfig( "FORCE_IGNORE_INTERFACE_REGX_LIST", "" ) ).split( "," )
      self._forceIgnoredIntfRegxList = [ d.strip() for d in intfList if d.strip() ]

   def getEventFifoPath( self ):
      return "/var/run/UdevToSfaHotPlugPipe"

   def veosConfig( self, key, defaultValue=None ):
      return self._veosConfig.get( key, defaultValue )

   def hasManagementIntf( self ):
      # On cloud platforms( AWS, Azure, GCP ), number of interfaces for a vEOS
      # instances are limited, cannot sacrifice an interface for management.
      # This function need to be overridden by cloud platforms to return False.
      #
      # We will skip renaming first physical interface to Management1 interface,
      # if this function returns False.
      return True

   def useKernelDevNames( self ):
      return False

   # Will be overridden by cEOSR helper
   def isContainerInterface( self, devName ):
      return False

   def interfaceExists( self, intfName ):
      m = re.match( self._frontPanelPrefix + r"\d+", intfName )
      return m or intfName.startswith( "ma" ) or \
             intfName in self.deviceCache[ DEVICE ]

   # Hotpugin handler for adding a new interface
   def handleInterfaceEventAdd( self, intfName, **extraArgs ):
      if self.interfaceExists( intfName ):
         t0( "Ignoring already present interface port %s" % intfName )
         return
      elif self.isIgnoredInterface( intfName ):
         return

      # Allocate portId
      portId = self.getNextPortId( intfName )
      if not portId:
         # We run out of portIds
         return

      # Rename interface
      devName = self.renameInterface( intfName, portId )

      # Add device in the cache
      self.addDeviceInCache( devName, portId, **extraArgs )
      # if the interface was hotplugged, write it to the device cache file so that
      # the child drivers can read the updated cache
      self.writeDeviceCache()

      # Add interface into Sysdb
      self.addInterface( devName, **extraArgs )
      self._driver.reinstantiateDrivers()
      self.updateIntfConfig( devName )
      t0( "handleInterfaceAdd: done for %s %s\n" % ( intfName, devName ) )

   # Hotpugin handler for deleting a interface
   def handleInterfaceEventDel( self, intfName ):
      self.delInterface( intfName )
      self._driver.reinstantiateDrivers()
      self.writeDeviceCache()
      t0( "handleInterfaceEventDel: done for %s\n" % intfName )

   # Hotplugin handler
   def handleInterfaceEvent( self, intfName ):
      self.handleInterfaceEventAdd( intfName )

   def resetPortId( self ):
      self._portIdPool = [ True for _ in range( PORT_MAX ) ]

      # Mark reserved portId as not available
      self._portIdPool[ 0 ] = False # portId 0 is unused
      self._portIdPool[ PORT_MGMT ] = False

      self._portId = PORT_INVALID
      for portId in self.deviceCache[ PORT ].values():
         self._portIdPool[ portId ] = False

         # Reset the current portId pointer
         if portId > self._portId and portId >= PORT_PHY_START and \
            portId <= PORT_PHY_END:
            self._portId = portId

   def releasePortId( self, portId ):
      t0( "releasePortId: %d" % portId )
      self._portIdPool[ portId ] = True # Mark as available

   def getNextPortIdFromPool( self, portId, startId, endId ):
      p = portId
      pInit = p
      while True:
         if p == PORT_INVALID:
            # If portId has been reset, use first port in the pool
            p = startId
            pInit = startId
         else:
            p = p + 1
            if p > endId:
               p = startId
            # We ran out of portIds in the pool
            if p == pInit:
               break

         if self._portIdPool[ p ]:
            return p
      return 0

   def getNextPortId( self, devName ):
      t0( "getNextPortId: %s" % devName )
      newPortId = self.getNextPortIdFromPool(
            self._portId, PORT_PHY_START, PORT_PHY_END )
      if newPortId:
         self._portId = newPortId
         self._portIdPool[ newPortId ] = False # Mark as not available
         t0( "getNextPortId: portId %d allocated for %s" % ( newPortId, devName ) )
         return newPortId

      t0( "Ran out of portIds( %d - %d ) for %s" %
          ( PORT_PHY_START, PORT_PHY_END, devName ) )
      return 0

   def _renameInterface( self, oldDevName, newDevName ):
      # Rename interface
      Tac.run( [ "ip", "link", "set", "dev", oldDevName, "down" ] )
      Tac.run( [ "ip", "link", "set", "dev", oldDevName,
                    "name", oldDevName + "tmp" ] )

      # Now rename interface to their final name and bring them
      # back up
      Tac.run( [ "ip", "link", "set", "dev", oldDevName + "tmp",
                 "name", newDevName ] )
      Tac.run( [ "ip", "link", "set", "dev", newDevName, "up" ] )
      t0( "renameInterface: Old = %r, New = %r" % ( oldDevName, newDevName ) )
      return newDevName

   def _initDeviceCache( self ):
      self.deviceCache = {}
      self.deviceCache[ DEVICE ] = []
      self.deviceCache[ FIRST_MAC ] = ""
      self.deviceCache[ MAC ] = {}
      self.deviceCache[ PCI ] = {}
      self.deviceCache[ DRIVER ] = {}
      self.deviceCache[ PORT ] = {}
      self.deviceCache[ ROLE ] = {}
      self.deviceCache[ TYPE ] = {}
      self.deviceCache[ VF2PF ] = {}
      self.kernelDeviceNames = {}

   # Function loadDeviceCache populates deviceCache from file or
   # from kernel netdevices.
   def loadDeviceCache( self ):
      # On sfe mode read the device information from cache file if it exists.
      # sfa/sfe_fail_safe mode will not read cache from file.

      self.deviceCache = None
      if self.veosConfig( "MODE" ) == "sfe":
         self.readDeviceCache()
      if self.deviceCache:
         # Update portId and return
         self.resetPortId()
         return

      # Reset data
      self._initDeviceCache()
      self.resetPortId()

      # Get kernel netdevices and add the same to cache
      devNames = self.getKernelDevices()
      if not devNames:
         return

      # FIRST_MAC is used for generating system mac-address
      self.deviceCache[ FIRST_MAC ] = self.getMac( devNames[ 0 ] )
      t0( "FirstMac address is %s" % self.deviceCache[ FIRST_MAC ] )

      # Allocate portIds and rename the interface
      for i, devName in enumerate( devNames ):
         if i == 0 and ( self.hasManagementIntf() or
                         self.veosConfig( "PRIMARYPORT" ) == "management" ):
            # First device will be used as Management interface
            portId = PORT_MGMT
         else:
            portId = self.getNextPortId( devName )
         if not portId:
            # We run out of portIds, ignore rest of the interfaces
            break

         # Rename interface
         newDevName = self.renameInterface( devName, portId )

         # Add device in the cache
         self.addDeviceInCache( newDevName, portId )
      self.writeDeviceCache()

   def writeDeviceCache( self ):
      with open( DEVICE_CACHE_FILE, "w" ) as f:
         json.dump( self.deviceCache, f, indent=3 )

   def readDeviceCache( self, force=False ):
      t0( "Reading devices from cache file" )
      assert not force and self.deviceCache is None, \
             "Overridding deviceCache not allowed"

      if not os.path.isfile( DEVICE_CACHE_FILE ):
         return

      with open( DEVICE_CACHE_FILE ) as f:
         self.deviceCache = json.load( f )

   def addDeviceInCacheExt( self, devName, **extraProps ):
      # Implemented by platform specific helper
      # to store additional data in map
      pass

   def deleteFromDeviceCacheExt( self, devName ):
      # Implemented by platform specific helper
      # to delete additional data from map
      pass

   def getDevType( self, driver ):
      devType = "nicUnknown"

      if driver in [ 'ena', 'gve', 'hv_netvsc', 'netvsc', 'virtio_net', 'vmxnet' ]:
         devType = "virtualNic"
      # gve is driver for gVNIC in GCP
      # ena is driver for virtual NIC ENA in AWS
      # hv_netvsc / netvsc is driver in Azure NIC

      return devType

   def addDeviceInCache( self, devName, port, **extraProps ):
      mac = self.getMac( devName )
      pci = self.getPci( devName ) # Store address of devName in cache
      driver = self.getDriver( devName ) # Store address of devName in cache
      devType = self.getDevType( driver )
      t0( "DevType for %s is %s" % ( devName, devType ) )

      if devName not in self.deviceCache[ DEVICE ]:
         self.deviceCache[ DEVICE ].append( devName )
      self.deviceCache[ MAC ][ devName ] = mac
      self.deviceCache[ PCI ][ devName ] = pci
      self.deviceCache[ DRIVER ][ devName ] = driver
      self.deviceCache[ PORT ][ devName ] = port
      self.deviceCache[ TYPE ][ devName ] = devType
      self.kernelDeviceNames[ port ] = devName
      self.addDeviceInCacheExt( devName, **extraProps )

   def deleteFromDeviceCache( self, devName ):
      self.deleteFromDeviceCacheExt( devName )
      port = None
      if devName in self.deviceCache[ DEVICE ]:
         self.deviceCache[ DEVICE ].remove( devName )
      if devName in self.deviceCache[ MAC ]:
         del self.deviceCache[ MAC ][ devName ]
      if devName in self.deviceCache[ PCI ]:
         del self.deviceCache[ PCI ][ devName ]
      if devName in self.deviceCache[ DRIVER ]:
         del self.deviceCache[ DRIVER ][ devName ]
      if devName in self.deviceCache[ PORT ]:
         port = self.deviceCache[ PORT ][ devName ]
         del self.deviceCache[ PORT ][ devName ]
      if port and port in self.kernelDeviceNames:
         del self.kernelDeviceNames[ port ]
      if devName in self.deviceCache[ TYPE ]:
         del self.deviceCache[ TYPE ][ devName ]

   def getDpdkBoundDevices( self ):
      op = Tac.run( [ "/usr/share/dpdk/tools/dpdk-devbind.py", "--status-dev",
                      "net" ], stdout=Tac.CAPTURE, asRoot=True )
      t0( "Output of dpdk-devbind.py --status-dev net: %s" % op )

      # Pick up if any DPDK bound interfaces
      op1 = Tac.run( [ "sed", "-n",
                       "/Network devices using DPDK-compatible driver/,/^$/p" ],
                       input=op, stdout=Tac.CAPTURE )
      devs = Tac.run( [ "grep", "0000:" ], input=op1,
                       stdout=Tac.CAPTURE ).splitlines()
      devs = [ d.split()[ 0 ] for d in devs ]
      return devs

   # Will be overridden by platform-specific helper
   def getEthernetDriver( self ):
      lspciOp = Tac.run( [ "/sbin/lspci" ], stdout=Tac.CAPTURE )
      lspciOp = [ x for x in lspciOp.splitlines() if 'Ethernet' in x ]
      driver = None

      if any( x for x in lspciOp if 'Virtio' in x ):
         driver = "virtio"
      elif any( x for x in lspciOp if 'Intel' in x ):
         driver = "ixgbevf"
      return driver

   def bindInterfaceToKernel( self, pciAddr, driver ):
      t0( "bindInterfaceToKernel: pciAddr = %s, driver = %s" % ( pciAddr, driver ) )
      Tac.run( [ "/usr/share/dpdk/tools/dpdk-devbind.py", "--force", "-b",
                 driver, pciAddr ], asRoot=True, stdout=Tac.CAPTURE )

   def __bindInterfacesToKernel( self ):
      driver = self.getEthernetDriver()
      devs = self.getDpdkBoundDevices()
      for pciAddr in devs:
         # Rebind to kernel driver
         self.bindInterfaceToKernel( pciAddr, driver )

   def bindInterfacesToKernel( self ):
      t0( "bindInterfacesToKernel" )
      try:
         self.__bindInterfacesToKernel()
      except Tac.SystemCommandError as e:
         t0( "Ignoring Caught exception %s" % e )

   def isDeviceUnderKernel( self, devName ):
      # Assuming that device is not under kernel control if MODE is sfe
      # and devName is not in cache.
      # Once we add devName to cache, that means we are about to move
      # device to DPDK control in the init process.
      # If there is a better way of checking this, then will change this
      # logic.
      return self.veosConfig( "MODE" ) != "sfe" or \
             devName not in self.deviceCache[ DEVICE ]

   # Read mac address from kernel device
   # Will be overridden by platform-specific helper
   # pylint: disable=bare-except
   def doGetMac( self, devName ):
      assert self.isDeviceUnderKernel( devName ), \
             "Device %s not under kernel control" % devName

      mac = None
      try:
         mac = open( # pylint: disable=consider-using-with
               os.path.join( SYS_CLASS_NET, devName, "address" ) ).read().strip()
      except:
         t0( "Unexpected exception: %s" % str( traceback.print_exc() ) )
      return mac

   # Read PCI address from kernel device
   # Will be overridden by platform-specific helper
   def doGetPci( self, devName ):
      assert self.isDeviceUnderKernel( devName ), \
             "Device %s not under kernel control" % devName
      pci = None
      try:
         output = Tac.run( [ "ethtool", "-i", devName ], stdout=Tac.CAPTURE )
         pci = re.search( "bus-info: (.*)", output ).group( 1 )
      except:
         t0( "Unexpected exception: %s" % str( traceback.format_exc() ) )
      return pci

   # Read Driver from kernel device
   # Will be overridden by platform-specific helper
   def doGetDriver( self, devName ):
      assert self.isDeviceUnderKernel( devName ), \
             "Device %s not under kernel control" % devName

      driver = None
      try:
         output = Tac.run( [ "ethtool", "-i", devName ], stdout=Tac.CAPTURE )
         driver = re.search( "driver: (.*)", output ).group( 1 )
      except:
         t0( "Unexpected exception: %s" % str( traceback.format_exc() ) )
      return driver

   # Get mac address form deviceCache
   def getMac( self, devName ):
      if devName in self.deviceCache[ MAC ]:
         mac = self.deviceCache[ MAC ][ devName ]
      else:
         mac = self.doGetMac( devName )

      t0( "Mac for %s is %s" % ( devName, mac ) )
      return mac

   # Get devName form deviceCache for given mac address
   def getDevnameFromMac( self, mac ):
      devName = None
      for d, m in self.deviceCache[ MAC ].items():
         if mac.lower() == m.lower():
            devName = d
            break
      t0( "devName for mac %s is %s" % ( mac, devName ) )
      return devName

   # Get portId from deviceCache
   def getPortId( self, devName ):
      portId = None
      if devName in self.deviceCache[ PORT ]:
         portId = self.deviceCache[ PORT ][ devName ]

      t0( "PortId for %s is %s" % ( devName, portId ) )
      return portId

   # Get pci address form deviceCache
   def getPci( self, devName ):
      if devName in self.deviceCache[ PCI ]:
         pci = self.deviceCache[ PCI ][ devName ]
      else:
         pci = self.doGetPci( devName )

      t0( "PCI for %s is %s" % ( devName, pci ) )
      return pci

   # Get driver form deviceCache
   def getDriver( self, devName ):
      if devName in self.deviceCache[ DRIVER ]:
         driver = self.deviceCache[ DRIVER ][ devName ]
      else:
         driver = self.doGetDriver( devName )

      t0( "Driver for %s is %s" % ( devName, driver ) )
      return driver

   def isPhysicalDevice( self, devName ):
      return os.path.exists( os.path.join( SYS_CLASS_NET, devName, "device" ) )

   # Will be overridden by platform-specific helper
   def isIgnoredInterface( self, devName ):
      if self.isDeviceUserForceMapped( devName ):
         t0( "Include interface as it matches FORCE_MAP_INTERFACE_REGEX_LIST: %s" %
             devName )
         return False
      if self.isDeviceUserForceIgnored( devName ):
         t0( "Skip interface as it matches FORCE_IGNORE_INTERFACE_REGX_LIST: %s" %
             devName )
         return True

      skipIntf = False
      if self.isContainerInterface( devName ):
         skipIntf = False
      elif not self.isPhysicalDevice( devName ):
         # Skip interfaces like tunnelTap0, dummy0, etc.
         t0( "Skip virtual interface: %s" % devName )
         skipIntf = True
      elif not re.match( SUPPORTED_INTF_REGEX, devName ):
         # Skip unsupported interfaces
         t0( "Skip unsupported interface: %s" % devName )
         skipIntf = True

      return skipIntf

   def sortDevNamesByPci( self, devNames ):
      # Sort device names by pci address.
      address = {}

      for devName in devNames:
         path = os.path.realpath( os.path.join( SYS_CLASS_NET, devName, "device" ) )
         address[ devName ] = re.match( "/sys/devices/(.*)", path ).group( 1 )

      devNames.sort( key=lambda x: address[ x ] )

      t0( "Sorted interfaces: %s" % devNames )
      return devNames

   # Will be overridden by platform-specific helper
   def sortDevNames( self, devNames ):
      # First order the interfaces specified by the user in the
      # veos-config with mac-addresses, then sort the remaining
      # interface in the pci address order. If nothing is
      # specified in veos-config, sort by pci address.
      if not self.veosConfig( "DEV_MACADDR_LIST" ):
         t0( "Sort by PCI address" )
         return self.sortDevNamesByPci( devNames )

      devMap = {}
      # Get the Mac address of all network interfaces.
      for devName in devNames:
         devMap[ self.getMac( devName ) ] = devName

      # Sort device names by DEV_MACADDR_LIST from veos-config.
      macAddrStr = self.veosConfig( "DEV_MACADDR_LIST" ).strip()
      macAddrList = macAddrStr[ 1 : -1 ].split( ',' )
      macDevNames = []
      for macAddr in macAddrList:
         macAddr = macAddr.strip()
         if macAddr not in devMap:
            t0( "interfaces not found for Mac address: %s" % macAddr )
         else:
            macDevNames.append( devMap[ macAddr ] )
      addDevNames = []
      # Sort the remaining devices by pci Address
      if len( macDevNames ) != len( devNames ):
         addDevNames = list( set( macDevNames ) ^ set( devNames ) )
         addDevNames = self.sortDevNamesByPci( addDevNames )
         t0( "Sorted remaning devices by PCI address %s" % addDevNames )
      devNames = macDevNames + addDevNames

      t0( "Sorted interfaces by order specified in veos-config: %s" % devNames )
      return devNames

   def getKernelDevicesFromSysClassPath( self ):
      # Get all ethernet interface device names
      devNames = []
      for devName in os.listdir( SYS_CLASS_NET ):
         if not os.path.isdir( os.path.join( SYS_CLASS_NET, devName ) ):
            # Possibly a file, like "bonding_masters".
            continue

         devType = int(
               # pylint: disable-next=consider-using-with
               open( os.path.join( SYS_CLASS_NET, devName, "type" ) ).read() )
         t0( "device = %s, type = %s" % ( devName, devType ) )
         # This type apparently corresponds to include/linux/if_arp.h
         # ARPHRD_* values.
         if devType != ARPHRD_ETHER:
            continue
         devNames.append( devName )
      return devNames

   def getKernelDevices( self ):
      # Filter and sort the deviceNames
      devNames = []
      for devName in self.getKernelDevicesFromSysClassPath():
         if self.isIgnoredInterface( devName ):
            continue
         # Aboot randomly renames 1G port as ma*, while other 10G ports exists.
         # This conflicts with Sfe naming convention of network devices
         # which is based on pci address order. Rename this interface to
         # temporary name, so that it does not conflict Fru rename other
         # device with ma*.
         if devName.startswith( 'ma' ):
            devName = self._renameInterface( devName, 'eth' + devName )
         devNames.append( devName )

      # Sort device names by pci address.
      return self.sortDevNames( devNames )

   # Will be overridden by Azure helper. On other platforms
   # realDevName is same as devName
   def getRealDevName( self, devName ):
      return devName

   # Will be overridden by platform-specific helper, provide a new name
   # for kernel netdevice
   def getNewDevName( self, oldDevName, portId ):
      if self.useKernelDevNames():
         newDevName = oldDevName
      elif portId == PORT_MGMT:
         newDevName = "ma1"
      else:
         newDevName = "%s%d" % ( self._frontPanelPrefix, portId )
      return newDevName

   def renameInterface( self, oldDevName, portId ):
      if ( os.environ.get( "P4USER" ) or
           os.environ.get( "ABUILD" ) or
           os.environ.get( "A4_CHROOT" ) ):
         # Yikes - we seem to be on a build server, don't do anything
         t0( "Interface rename aborted since we seem to be on a build server" )
         return oldDevName

      if self.veosConfig( "MODE" ) not in [ "test", "linux", "sfe", "sfe_failsafe" ]:
         t0( "Unknown vEOS mode. Skipping interface rename" )
         return oldDevName

      if self.useKernelDevNames():
         return oldDevName

      newDevName = self.getNewDevName( oldDevName, portId )
      if newDevName == oldDevName:
         return newDevName

      return self._renameInterface( oldDevName, newDevName )

   def getNewPort( self, devName, **extraArgs ):
      m = re.match( SUPPORTED_INTF_REGEX, devName )
      if not m:
         return None, None

      ( prefix, _ ) = m.groups()
      label = self.deviceCache[ PORT ][ devName ]
      portId = self.deviceCache[ PORT ][ devName ]
      mac = self.getMac( devName )
      if prefix == "ma" or portId == PORT_MGMT:
         label = 1
         port = self._driver.pciPortDir.newPort( portId )
         port.description = "Management:%d" % label
         port.role = "Management"
         port.macAddr = mac
         intftype = INTFTYPE_MGMT
      elif ( prefix.startswith( "e" ) or prefix == "hv_et" ):
         port = self._driver.ethPortDir.newPort( portId )
         port.description = "FrontPanel:%d" % label
         port.role = "Switched"
         port.macAddr = mac
         intftype = INTFTYPE_ETH
      elif prefix == "vmnicet":
         port = self._driver.ethPortDir.newPort( portId )
         port.description = "Virtual:%d" % label
         port.role = "Switched"
         port.macAddr = mac
         intftype = INTFTYPE_VIRT
      else:
         return None, None

      port.label = label
      return port, intftype

   def getPhyName( self, devName, portId ):
      if self.useKernelDevNames() and portId:
         if portId == PORT_MGMT:
            phyName = "ma1"
         else:
            phyName = "et%d" % portId
      else:
         phyName = devName
      return phyName

   def addInterface( self, devName, **extraArgs ):
      port, intftype = self.getNewPort( devName, **extraArgs )
      t0( "addInterface devName: %s, intftype: %s" % ( devName, intftype ) )
      if port is None:
         t0( "Ignoring unrecognized interface %s" % devName )
         return
      invEntities = [ port ]

      mode = self.veosConfig( "MODE" )
      phyName = self.getPhyName( devName, port.id )
      # Create an Inventory::EthPort
      if intftype == INTFTYPE_MGMT:
         # Don't set a pci address, the EthPciPort fru driver will assign
         # things correctly based on the device name
         # Create an Inventory::PhyEthtoolPhy
         phy = self._driver.phyEthtoolPhyDir.newPhy( phyName )
         invEntities.append( phy )
         phy.port = port
         phy.ethtoolConfigEnabled = self._driver.mgmtConfigEnabled_
         t0( "ma%d added to phy/ethtool/config" % port.label )
      elif intftype == INTFTYPE_ETH:
         # Create an Inventory::PhyEthtoolPhy or SfePhyDir
         if mode == "sfe":
            phy = self._driver.sfePhyDir.newPhy( phyName )
            invEntities.append( phy )
            t0( "et%d added to phy/sfe/config" % port.label )
            phy.port = port
         # In sfe_failsafe mode, add et1 interface to PhyEthtool's inventory
         # to allow management access to the instance, skip addition of front
         # panel ports as they will be removed from kernel's control.
         elif mode == "linux" or ( mode == "sfe_failsafe" and port.label == 1 ):
            phy = self._driver.phyEthtoolPhyDir.newPhy( phyName )
            invEntities.append( phy )
            t0( "et%d added to phy/ethtool/config" % port.label )
            phy.port = port
         else:
            t0( "Skipping addition of et%d to phy/ethtool/config in failsafe mode"
                % port.label )
      elif intftype == INTFTYPE_VIRT:
         # Force the MTU of these ports to 10000. This is how
         # real hardware ports behave.
         try:
            Tac.run( [ "ifconfig", devName, "mtu", "10000" ] )
         except Tac.SystemCommandError:
            # Shoot, we're seem to be using a virtual network
            # adapter that does not support jumbo frames
            # (ie hyper-v).
            # Fall back to 1500 bytes
            Tac.run( [ "ifconfig", devName, "mtu", "1500" ] )
            self._driver.driverCtx.entity(
                  "bridging/hwcapabilities" ).maxFrameSize = 1500
         # Make sure our vmnicet interfaces don't respond to arp
         # requests. This is done by the etN interface created on
         # top of it
         # pylint: disable-next=consider-using-with
         f = open( "/proc/sys/net/ipv4/conf/%s/arp_ignore" % devName, "w" )
         f.write( "1" )
         f.close()
         # Make sure the device is actually up. This is required
         # before the Etba agent attempts to bind to it
         Tac.run( [ "ifconfig", devName, "promisc", "up" ] )
      else:
         assert False, "Unknown port %s" % intftype

      if mode != "sfe":
         phy = self._driver.sfaPhyDir.newPhy( phyName )
         invEntities.append( phy )
         phy.port = port

      # Create a DependentSet for each of the entity
      # under inventory/genericPc/component/...
      # All dependents can be cleaned up when the entity is deleted.
      for e in invEntities:
         Fru.newDependentSet( e )

      t0( "Added device %s, mac %s, pci %s" %
          ( devName, self.getMac( devName ), self.getPci( devName ) ) )

   def cleanupInventory( self, devName, portId=None ):
      t0( "Cleanup interface %s" % devName )
      if not portId and devName in self.deviceCache[ PORT ]:
         portId = self.deviceCache[ PORT ][ devName ]
      phyName = self.getPhyName( devName, portId )

      if phyName in self._driver.sfePhyDir.phy:
         Fru.deleteDependents( self._driver.sfePhyDir.phy[ phyName ] )
         del self._driver.sfePhyDir.phy[ phyName ]
         t0( "Deleted interface from  sfePhyDir %s" % phyName )

      if phyName in self._driver.sfaPhyDir.phy:
         Fru.deleteDependents( self._driver.sfaPhyDir.phy[ phyName ] )
         del self._driver.sfaPhyDir.phy[ phyName ]
         t0( "Deleted interface from  sfaPhyDir %s" % phyName )

      if phyName in self._driver.phyEthtoolPhyDir.phy:
         Fru.deleteDependents( self._driver.phyEthtoolPhyDir.phy[ phyName ] )
         del self._driver.phyEthtoolPhyDir.phy[ phyName ]
         t0( "Deleted interface from  phyEthtoolPhyDir %s" % phyName )

      if portId and portId in self._driver.ethPortDir.port:
         Fru.deleteDependents( self._driver.ethPortDir.port[ portId ] )
         del self._driver.ethPortDir.port[ portId ]
         t0( "Deleted port from ethPortDir %d" % portId )

   def delInterface( self, devName ):
      t0( "Deleting interface %s" % devName )
      self.cleanupInventory( devName )

      if devName in self.deviceCache[ PORT ]:
         self.releasePortId( self.deviceCache[ PORT ][ devName ] )
      self.deleteFromDeviceCache( devName )
      t0( "Done deletion of interface %s" % devName )

   def updateRunningConfig( self, config ):
      t0( "updateRunningConfig %s" % config )
      if not config:
         return

      cliConfig = CLI_CONFIG_MODE + config
      # pylint: disable-next=consider-using-with
      subprocess.Popen( [ "echo -e \"" + cliConfig + "\" | Cli -A" ], shell=True,
                        stdin=None, stdout=None, stderr=None, close_fds=True )

   # Will be overridden by platform-specific helper
   def updateIntfConfig( self, devName ):
      pass

   # Will be overridden by platform-specific helper
   def updateIntfConfigAll( self ):
      pass

   def isDeviceUserForceMapped( self, devName ):
      for d in self._forceMappedIntfRegxList:
         if re.match( d, devName ):
            return True
      return False

   def isDeviceUserForceIgnored( self, devName ):
      for d in self._forceIgnoredIntfRegxList:
         if re.match( d, devName ):
            return True
      return False

   def getSerialNumber( self ):
      return ""
