# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
Provides utilities for Pythonic access to SflowAccel FPGAs, most notably:
- a list of valid and supported register and table names;
- the Fpga class, used to read from/write to the FPGAs.
"""

import Cell
import EntityManager
import Tac


# Load some TACC types
FpgaType = Tac.Type( 'Inventory::SflowAccelFpga::SflowAccelFpgaType' )
RegName = Tac.Type( 'Hardware::SflowAccelFpga::RegName' )
TblName = Tac.Type( 'Hardware::SflowAccelFpga::TblName' )


# Export lists of valid register and table names as obtained from the RegName and
# TblName TACC types.
registerNames = list( RegName.attributes )
tableNames = list( TblName.attributes )
regAndTableNames = registerNames + tableNames

def validRegisterName( name ):
   """Return True if the specified name is a valid FPGA register name."""
   return name in registerNames

def validTableName( name ):
   """Return True if the specified name is a valid FPGA table name."""
   return name in tableNames

def validRegOrTableName( name ):
   """Return True if the specified name is a valid FPGA register or table name."""
   return name in regAndTableNames


# Hardware::SflowAccelFpga::ConfigDir at /hardware/sflowAccelFpga/config/slice
_fpgaConfigDir = None

def _mountFpgaConfigDir():
   """
   Mount (if necessary) and return the SflowAccel FPGA config dir from Sysdb.
   """
   global _fpgaConfigDir

   if not _fpgaConfigDir:
      em = EntityManager.Sysdb( 'ar' )
      mg = em.mountGroup()

      _fpgaConfigDir = mg.mount( 'hardware/sflowAccelFpga/config/slice',
                                 'Tac::Dir', 'ri' )
      # Need to mount pciDeviceStatusDir since we'll be accessing hamImpls.
      mg.mount( Cell.path( 'hardware/pciDeviceStatusDir' ),
               'Hardware::PciDeviceStatusDir', 'r' )

      mg.close( blocking=True )

   return _fpgaConfigDir

def _getFpgaConfig( fpgaName ):
   """
   Return the Hardware::SflowAccelFpga::Config entity corresponding to the specified
   FPGA name.
   """
   fpgaConfigDir = _mountFpgaConfigDir()

   for fpgaSliceConfig in fpgaConfigDir.values():
      for fpgaConfig in fpgaSliceConfig.config.values():
         if fpgaConfig.name == fpgaName:
            return fpgaConfig

   return None


def getFpgaNames():
   """
   Successively yield the name of every FPGA in the system.
   """
   fpgaConfigDir = _mountFpgaConfigDir()

   for fpgaSliceConfig in fpgaConfigDir.values():
      for fpgaConfig in fpgaSliceConfig.config.values():
         yield fpgaConfig.name

def getFpgas():
   """
   Return a dict mapping FPGA name to Fpga object for all FPGAs in the system.
   """
   return { fpgaName: Fpga( fpgaName ) for fpgaName in getFpgaNames() }


def _tacHwAccessorType( hwAccessorName ):
   """
   Load and return the requested unqualified HwAccessor TACC type.
   """
   return Tac.Type( f'Hardware::SflowAccelFpga::{hwAccessorName}' )

# A map of FPGA type to HwAccessor type
_tacHwAccessorTypes = {
   FpgaType.halo : _tacHwAccessorType( 'SflowAccelFpgaNiagaraHwAccessor' ),
   FpgaType.scd : _tacHwAccessorType( 'SflowAccelFpgaNiagaraHwAccessor' ),
   FpgaType.oyster : _tacHwAccessorType( 'SflowAccelFpgaNiagaraHwAccessor' ),
}

def _tacHwAccessor( fpgaName ):
   """
   Given a Hardware::SflowAccelFpga::Config object, return a HwAccessor instance for
   the appropriate FPGA type.
   """
   fpgaConfig = _getFpgaConfig( fpgaName )
   fpgaType = fpgaConfig.sflowAccelFpgaType

   hwAccessorType = _tacHwAccessorTypes.get( fpgaType, None )

   if not hwAccessorType:
      message = f'No hardware accessor for FPGA type: {fpgaType}'
      raise Exception( message )

   # Reading the revision from the FPGA to determine the FPGA mode.
   rev = hwAccessorType.readFpgaRev( fpgaType, fpgaConfig.ham.hamImpl )
   fpgaMode = hwAccessorType.fpgaModeFromRev( rev )
   return hwAccessorType( fpgaConfig.name,
                          fpgaType,
                          fpgaMode,
                          fpgaConfig.ham.hamImpl )


# Values with special formatting functions.
#
# When reading from the registers and tables in this list, values are passed to the
# corresponding callable to produce a human-readable string representation of said
# values. This is useful for things like MAC and IP addresses. The default format is
# hexadecimal.
_valueFormatFuncs = {
}

class Fpga:
   """
   Python class to facilitate reading from and writing to an FPGA. Instances can be
   used as dict-like objects to access individual tables and registers, e.g.

      fpga = Fpga( 'SflowAccelFpga0' )
      print fpga[ 'fpgaRevision' ]
      print fpga[ 'snmpIfIndexTbl' ][ 0 ]

   The available registers and tables are directly loaded from the
   Hardware::SflowAccelFpga::RegName and Hardware::SflowAccelFpga::TblName TACC
   types.
   """
   def __init__( self, fpgaName ):
      if fpgaName not in getFpgaNames():
         raise Exception( f'No such FPGA: {fpgaName}' )

      self.name = fpgaName
      self.hwAccessor = _tacHwAccessor( fpgaName )

   def __repr__( self ):
      return f'<Fpga {self.name}>'

   def __getitem__( self, regOrTblId ):
      regOrTblName, instanceId = regOrTblId
      if validRegisterName( regOrTblName ):
         return Fpga.Register( regOrTblName, instanceId, self.hwAccessor )
      elif validTableName( regOrTblName ):
         return Fpga.Table( regOrTblName, instanceId, self.hwAccessor )

      raise KeyError( regOrTblName )

   def __setitem__( self, regOrTblId, value ):
      regOrTblName, _ = regOrTblId
      if validRegisterName( regOrTblName ):
         self[ regOrTblId ].value = value
      elif validTableName( regOrTblName ):
         raise TypeError( regOrTblName )

      raise KeyError( regOrTblName )

   def validRegisterInstance( self, name, instanceId ):
      """Return True if the specified name and instanceId is a valid FPGA
      register."""
      numInstances = self.hwAccessor.numRegInstances( name )
      return 0 <= instanceId < numInstances

   def validTableInstance( self, name, instanceId ):
      """Return True if the specified name and instanceId is a valid FPGA
      table."""
      numInstances = self.hwAccessor.numTblInstances( name )
      return 0 <= instanceId < numInstances

   class HardwareValue:
      """
      Represents a value read from one of the FPGA's registers or tables. The class
      overrides the special __str__ method to implement special formatting for
      certain values.
      """
      def __init__( self, name, value ):
         self.name = name
         self.value = value

         # Lookup the formatting function for this value, if any; otherwise, just
         # fall back to hex
         self.formatFunc = _valueFormatFuncs.get( name, None ) or hex

      def __str__( self ):
         return str( self.formatFunc( self.value ) )

      def __int__( self ):
         return self.value

   class Register:
      """
      Represents a register on the FPGA. Can be read from and written to using the
      "value" attribute, e.g.

         register = fpga[ ( 'sflowVersion', 0, ) ]
         print register.value
         register.value = 0x1
      """
      def __init__( self, name, instanceId, hwAccessor ):
         self.name = name
         self.instanceId = instanceId
         self.hwAccessor = hwAccessor

      @property
      def value( self ):
         value = self.hwAccessor.readReg32( self.name, self.instanceId )
         return Fpga.HardwareValue( self.name, value )

      @value.setter
      def value( self, value ):
         self.hwAccessor.writeReg32( self.name, self.instanceId, value )

      def __int__( self ):
         return int( self.value )

   class Table:
      """
      Represents a table on the FPGA. Can be indexed to access individual entries,
      e.g.

         table = fpga[ ( 'snmpIfIndexTbl', 0 ) ]
         print table[ 0 ]
         table[ 0 ] = ...
      """
      def __init__( self, name, instanceId, hwAccessor ):
         self.name = name
         self.instanceId = instanceId
         self.hwAccessor = hwAccessor

      def __getitem__( self, index ):
         value = self.hwAccessor.readTable( self.name, self.instanceId, index, True )
         return Fpga.HardwareValue( self.name, value )

      def __setitem__( self, index, value ):
         self.hwAccessor.writeTable( self.name, self.instanceId, index, value )
