# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
Enhanced shell infrastructure, built on top of the standard "cmd" module, for use by
the SflowAccel FPGA bash utility.

Notable enhancements include:

- The ability to specify argparse-like argument formats for each shell command using
  the @cmd decorator. The description of the command is taken from the function's
  docstring. When the command is invoked, the parsed arguments are passed to the
  function as its only argument.

  For example:

     @cmd(
        arg( '-f', '--flag', action='store_true', help='My flag description' ),
        arg( 'posArg', type=int, help='This is a positional argument.' )
     )
     def do_stuff( self, args ):
        '''
        This text will show up when running "help stuff"!
        '''
        if args.flag:
           return False
        print args.posArg

- Support for non-interactive mode. If the "batchFile" constructor argument of the
  ShellBase class is provided, the shell will read the contents of this readable file
  object and interpret each line as a command. "sys.stdin" can be used as value to
  implement Unix shell redirection. If "batchFile" is not provided, the shell will
  use the readline module to interact with the terminal/TTY device interactively.

- Support for rendering commands in JSON format. "do_*" functions should return a
  JSON-compatible object. Then, if "json=True" was passed to the ShellBase instance,
  or if running in non-interactive, the object is dumped as JSON. Otherwise, it is
  passed to the corresponding "render_*" function if it exists or simply printed as
  is.
"""

import argparse
import functools
import json
import re
from cmd import Cmd
from json import JSONEncoder
from SflowAccelFpgaLib import Fpga


class ShellError( Exception ):
   """
   Custom Exception for shell-related errors.
   """
   def __init__( self, messageFormat, *args ):
      Exception.__init__( self, messageFormat.format( *args ) )


class ShellEncoder( JSONEncoder ):
   """
   Custom JSON encoder which recognizes SflowAccelFpgaLib and ShellError objects.
   """
   # pylint: disable-msg=method-hidden
   # (false positive)
   def default( self, obj ): # pylint: disable=arguments-renamed
      if isinstance( obj, Fpga.HardwareValue ):
         return obj.value
      elif isinstance( obj, ShellError ):
         return { 'error': obj.message }

      return JSONEncoder.default( self, obj )


def safeEval( expression ):
   """
   Evaluate the given expression in an environment with no globals, locals, or
   builtins. This should be enough for basic arithmetic expressions.
   """
   try:
      # Only allow the following characters:
      # - Digits
      # - Radix prefixes: 0b, 0o, 0x, etc.
      # - Binary arithmetic operators and spaces
      assert re.match( r'^[\dbBoOxX<>\|&^~ ]+$', expression )

      # pylint: disable-next=eval-used
      return eval( expression, { '__builtins__': None }, {} )
   except (AssertionError, SyntaxError):
      # pylint: disable-next=raise-missing-from
      raise ShellError( f'Invalid expression: {expression!r}' )


class ShellArgumentParser( argparse.ArgumentParser ):
   """
   Custom ArgumentParser which raises a ShellError when encountering an error rather
   than exit the program.
   """
   def error( self, message ):
      raise ShellError( message.capitalize() )

def arg( *args, **kwargs ):
   """
   Capture the positional and keyword arguments to forward them to
   parser.add_argument(). Meant to be used with the @cmd decorator below.
   """
   return args, kwargs

def cmd( *argsList ):
   """
   Specify a list of arguments for a shell command.

   This must be used to decorate "do_*" functions from a ShellBase-derived object.

   Each argument should be specified using the arg() function above, which accepts
   the same positional and keyword arguments as the add_argument() method of
   argparse.ArgumentParser objects (see the example in the docstring at the top of
   this module).

   Before the decorated "do_*" function is called, arguments are parsed from the full
   line entered by the user, then passed to the function as its sole argument.
   """

   def wrapper( func ):
      # Remove the 'do_' prefix from the function name and extract the docstring for
      # use as the command's description in the help message.
      cmdName = func.__name__[ 3: ]
      cmdDescription = func.__doc__ or ''

      # Instantiate the argument parser then initialize it with the argument list
      # provided to cmd().
      formatter = argparse.RawDescriptionHelpFormatter
      parser = ShellArgumentParser( prog=cmdName,
                                    description=cmdDescription.strip(),
                                    formatter_class=formatter,
                                    add_help=False )

      for args, kwargs in argsList:
         parser.add_argument( *args, **kwargs )

      # Generate a new function to be used as drop-in replacement for the decorated
      # method.
      @functools.wraps( func )
      def newFunc( obj, line ):
         args = parser.parse_args( line.split() )
         return func( obj, args )

      # Set the new function's docstring to the help message generated by the parser.
      # This is picked up by the cmd.Cmd object when help needs to be displayed for
      # the command, e.g. when "help <cmd>" is run.
      newFunc.__doc__ = parser.format_help().strip()
      return newFunc

   return wrapper


class ShellBase( Cmd ):
   """
   Extended Cmd class which implements batch files and JSON rendering, meant to be
   used as a base class.
   """

   # pylint: disable-next=redefined-outer-name
   def __init__( self, json=False, batchFile=None ):
      """
      Initialize this shell.

      Arguments:
      - json: bool
         If True, render the results returned by all commands in JSON format.
         Otherwise, try to invoke the corresponding "render_*" method, or just print
         the results in text format.
      - batchFile: file object or None
         If provided, read commands from this file rather than from the interactive
         TTY device.
      """
      Cmd.__init__( self, stdin=batchFile )

      self.json = json

      if batchFile:
         # In non-interactive mode:
         # - don't display a prompt;
         # - use stdin.readline() rather than raw_input() to read commands.
         self.prompt = ''
         self.use_rawinput = False

   def promptIs( self, prompt ):
      """
      In interactive mode, set the shell's prompt to the provided value. In
      non-interactive mode, do nothing since we don't want to display a prompt.
      """
      if self.use_rawinput:
         self.prompt = prompt

   def onecmd( self, line ):
      """
      Execute the given line of text as a command, then render the results as JSON or
      pure text.
      """
      # Use the parent class to invoke the "do_*" method, and capture the result. If
      # the method throws a ShellError, capture that instead.
      try:
         result = Cmd.onecmd( self, line )
      except ShellError as e:
         result = e

      # If the "do_*" method returns nothing, or returns a boolean to indicate
      # whether the shell should continue interpreting commands, skip the rendering
      # step.
      if result is None or isinstance( result, bool ):
         return result

      # Render the results as usual.
      cmdName, args, line = self.parseline( line ) # pylint: disable=unused-variable
      self.renderResult( cmdName, result )

      # Indicate that the shell should not exit.
      return False

   def renderResult( self, cmdName, result ):
      """
      Render the provided result, as returned by the specified command.
      """
      if self.json:
         print( json.dumps( result, cls=ShellEncoder ) )
      elif isinstance( result, ShellError ):
         print( f'*** {result.message}' )
      else:
         renderFunc = getattr( self, f'render_{cmdName}', None )

         if renderFunc:
            renderFunc( result )
         else:
            print( result )

   def completions( self, text, collection ):
      """
      Utility function which returns all entries in the specified collection that
      match with the given text.
      """
      return [ s for s in collection if s.startswith( text ) ]

   def do_EOF( self, line ):
      """Exit the shell."""

      # When the user hits Ctrl+D, print "exit" if in interactive mode and invoke the
      # corresponding do_exit() command to handle the shell quitting.

      if self.use_rawinput:
         print( 'exit' )

      return self.onecmd( 'exit' )
