#!/usr/bin/env python3
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Arnet
import Tac
import re
from SflowConst import ( defaultPort, defaultIp, defaultIp6,
                         sflowDestDefaultMsg, sflowSourceDefaultMsg,
                         sflowUdpSrcPortMsg, sflowAgentDefaultMsg )
from IpLibConsts import DEFAULT_VRF
from Toggles.SflowLibToggleLib import (
      toggleSflowMaxDatagramSizeEnabled,
      toggleSflowIndependentAddressConfigEnabled,
      toggleSflowL2SubIntfVlanEnabled, )

IpGenAddr = Tac.Type( 'Arnet::IpGenAddr' )


ipAddrPattern = r"[12]?\d{1,2}\.[12]?\d{1,2}\.[12]?\d{1,2}\.[12]?\d{1,2}"

def cleanupSflowVrfConfig( config, vrfName, resetSwitchIp=False ):
   vrfConfig = config.sflowVrfConfig.get( vrfName )
   if not vrfConfig:
      return
   if resetSwitchIp:
      vrfConfig.switchIpAddr = IpGenAddr()
      vrfConfig.switchIp6Addr = IpGenAddr()
      vrfConfig.agentIpAddr = IpGenAddr()
      vrfConfig.agentIp6Addr = IpGenAddr()
      vrfConfig.addrConfigType = 'none'
   if ( not vrfConfig.switchIpAddr and not vrfConfig.switchIp6Addr and
        not vrfConfig.agentIpAddr and not vrfConfig.agentIp6Addr and
        not vrfConfig.srcIntfName and not vrfConfig.collectorHostAndPort ):
      del config.sflowVrfConfig[ vrfName ]

def addConfigCollectorInfo( config, hostname, port=defaultPort,
                            vrfName=DEFAULT_VRF ):
   if vrfName not in config.sflowVrfConfig:
      config.sflowVrfConfig.newMember( vrfName )
   config.sflowVrfConfig[ vrfName ].collectorHostAndPort.addMember(
         Tac.Value( 'Arnet::HostAndPort', hostname=hostname, port=port ) )

def removeConfigCollectorInfo( config, hostname, port=None,
                               vrfName=DEFAULT_VRF ):
   vrfConfig = config.sflowVrfConfig.get( vrfName )
   if vrfConfig:
      hostAndPort = vrfConfig.collectorHostAndPort.get( hostname )
      if hostAndPort:
         # Keyed by hostname, so only check that port matches if it is provided
         if port and port != hostAndPort.port:
            return
         del vrfConfig.collectorHostAndPort[ hostname ]
         cleanupSflowVrfConfig( config, vrfName )

def removeConfigCollectorInfoAll( config, vrfName=DEFAULT_VRF ):
   vrfConfig = config.sflowVrfConfig.get( vrfName )
   if vrfConfig:
      vrfConfig.collectorHostAndPort.clear()
      cleanupSflowVrfConfig( config, vrfName )

def addConfigCollectorInfoDict( config, hostAndPorts,
                                vrfName=DEFAULT_VRF ):
   for hostname, port in hostAndPorts.items():
      addConfigCollectorInfo( config, hostname, port, vrfName )

def verifyCollectorInfoHasElement( collectorInfo, hostname,
   port=defaultPort, only=False ):
   return hostname in collectorInfo \
         and collectorInfo[ hostname ].port == port \
         and ( not only or len( collectorInfo ) == 1 )

def verifyConfigCollectorInfo( collectorHostAndPort, hostAndPortDict ):
   if len( collectorHostAndPort ) != len( hostAndPortDict ):
      return False
   for hostname, port in hostAndPortDict.items():
      if not hostname in collectorHostAndPort:
         return False
      if port != collectorHostAndPort[ hostname ].port:
         return False
   return True

def verifyStatusCollectorInfo( collectorIpAndPort, hostAndIpDict ):
   if len( hostAndIpDict ) != len( collectorIpAndPort ):
      return False
   for hostname, ip in hostAndIpDict.items():
      if not hostname in collectorIpAndPort:
         return False
      if ip != collectorIpAndPort[ hostname ].ip:
         return False
   return True

def addConfigSourceIp( config, sourceIp, vrfName=DEFAULT_VRF,
                       addrConfigType='dependent' ):
   vrfConfig = config.sflowVrfConfig.get( vrfName )

   if ( vrfConfig and addrConfigType == 'independent'
         and vrfConfig.addrConfigType == 'dependent' ):
      vrfConfig.agentIpAddr = IpGenAddr()
      vrfConfig.agentIp6Addr = IpGenAddr()
      vrfConfig.addrConfigType = addrConfigType

   if sourceIp == defaultIp and addrConfigType == 'dependent':
      if vrfConfig:
         removeConfigSourceIp( config, vrfConfig.switchIpAddr, vrfName )
   elif sourceIp == defaultIp6 and addrConfigType == 'dependent':
      if vrfConfig:
         removeConfigSourceIp( config, vrfConfig.switchIp6Addr, vrfName )
   else:
      if not vrfConfig:
         config.sflowVrfConfig.newMember( vrfName )
         vrfConfig = config.sflowVrfConfig.get( vrfName )
      sourceIpGenAddr = IpGenAddr( sourceIp )
      vrfConfig.addrConfigType = addrConfigType
      if sourceIpGenAddr.af == 'ipv6':
         vrfConfig.switchIp6Addr = sourceIpGenAddr
         if addrConfigType == 'dependent':
            vrfConfig.agentIp6Addr = sourceIpGenAddr
      else:
         vrfConfig.switchIpAddr = sourceIpGenAddr
         if addrConfigType == 'dependent':
            vrfConfig.agentIpAddr = sourceIpGenAddr

def removeConfigSourceIp( config, ipAddr, vrfName=DEFAULT_VRF ):
   vrfConfig = config.sflowVrfConfig.get( vrfName )
   if vrfConfig:
      if ipAddr == vrfConfig.switchIpAddr:
         vrfConfig.switchIpAddr = IpGenAddr()
         if vrfConfig.addrConfigType == 'dependent':
            vrfConfig.agentIpAddr = IpGenAddr()
      elif ipAddr == vrfConfig.switchIp6Addr:
         vrfConfig.switchIp6Addr = IpGenAddr()
         if vrfConfig.addrConfigType == 'dependent':
            vrfConfig.agentIp6Addr = IpGenAddr()
      if ( vrfConfig.agentIpAddr.af != 'ipv4' and
            vrfConfig.agentIp6Addr.af != 'ipv6' ):
         cleanupSflowVrfConfig( config, vrfName, resetSwitchIp=True )

def addConfigAgentIp( config, agentIp, vrfName=DEFAULT_VRF ):
   vrfConfig = config.sflowVrfConfig.get( vrfName )

   if ( vrfConfig and vrfConfig.addrConfigType == 'dependent' ):
      vrfConfig.switchIpAddr = IpGenAddr()
      vrfConfig.switchIp6Addr = IpGenAddr()
      vrfConfig.addrConfigType = 'independent'

   if not vrfConfig:
      config.sflowVrfConfig.newMember( vrfName )
      vrfConfig = config.sflowVrfConfig.get( vrfName )
   vrfConfig.addrConfigType = 'independent'
   agentIpGenAddr = IpGenAddr( agentIp )

   if agentIpGenAddr.af == 'ipv6':
      vrfConfig.agentIp6Addr = agentIpGenAddr
   else:
      vrfConfig.agentIpAddr = agentIpGenAddr

def removeConfigAgentIp( config, ipAddr, vrfName=DEFAULT_VRF ):
   vrfConfig = config.sflowVrfConfig.get( vrfName )
   if vrfConfig:
      if not ipAddr or ipAddr == vrfConfig.agentIpAddr:
         vrfConfig.agentIpAddr = IpGenAddr()
      if not ipAddr or ipAddr == vrfConfig.switchIp6Addr:
         vrfConfig.agentIp6Addr = IpGenAddr()
      if ( vrfConfig.switchIpAddr.af != 'ipv4' and
            vrfConfig.switchIp6Addr.af != 'ipv6' ):
         cleanupSflowVrfConfig( config, vrfName, resetSwitchIp=True )

def removeConfigSourceIpAll( config, vrfName=DEFAULT_VRF ):
   cleanupSflowVrfConfig( config, vrfName, resetSwitchIp=True )

def addConfigSourceIntf( config, sourceIntf, vrfName=DEFAULT_VRF ):
   if vrfName not in config.sflowVrfConfig:
      config.sflowVrfConfig.newMember( vrfName )
   config.sflowVrfConfig[ vrfName ].srcIntfName = sourceIntf

def removeConfigSourceIntf( config, vrfName=DEFAULT_VRF ):
   vrfConfig = config.sflowVrfConfig.get( vrfName )
   if vrfConfig:
      vrfConfig.srcIntfName = ''
      cleanupSflowVrfConfig( config, vrfName )

def parseDestinations( lines ):
   destLines = []
   dest6Lines = []
   curLine = lines.pop( 0 )
   while 'Source' not in curLine:
      dstLine = curLine
      # This is for IPv4 source addresses.
      # The output format is either "(<ip>):<port> <hostname> (VRF: <vrfName>)" or
      # "<ip>:<port> (VRF: <vrfName>)"
      p = (
         r"(?P<ip>%s):(?P<port>\d+)(?P<host>( \(\S+\) )|(\s))\(VRF: (?P<vrf>\S+)\)"
         % ipAddrPattern )
      m = re.search( p, dstLine )
      if m:
         host = m.groupdict()[ 'host' ]
         # get rid of () and spaces
         hostName = None if host == ' ' else host[ 2 : -2 ]
         ipAddr = m.groupdict()[ 'ip' ]
         port = int( m.groupdict()[ 'port' ] )
         vrf = m.groupdict()[ 'vrf' ]
         destLines.append( ( hostName, ipAddr, port, vrf ) )

      # This is for IPv6 source addresses.
      # The output format is either "(<ip>):<port> <hostname> (VRF: <vrfName>)" or
      # "[<ip>]:<port> (VRF: <vrfName>)"
      p = r"\[(?P<ip>%s)\]:(?P<port>\d+)(?P<host>( \(\S+\) )|(\s))" \
            r"\(VRF: (?P<vrf>\S+)\)" % Arnet.Ip6AddrRe
      m = re.search( p, dstLine )
      if m:
         host = m.groupdict()[ 'host' ]
         # get rid of () and spaces
         hostName = None if host == ' ' else host[ 2 : -2 ]
         ip6Addr = m.groupdict()[ 'ip' ]
         port = int( m.groupdict()[ 'port' ] )
         vrf = m.groupdict()[ 'vrf' ]
         dest6Lines.append( ( hostName, ip6Addr, port, vrf ) )
      curLine = lines.pop( 0 )
   return ( lines, curLine, destLines, dest6Lines )

def parseSources( lines ):
   srcIpLines = []
   srcIp6Lines = []
   curLine = lines.pop( 0 )
   # For sFlow samples, we parse sources until we reach the line
   # that contains "Agent" (ie "Agent ID Addresses"),
   # 'UDP source port for HW sFlow' (or sampleRateMsg when
   # SflowIndependentAddressConfig toggle is false) Whereas, for
   # drop samples, we parse the sources until we reach a new line.
   endStr = ( 'Agent' if toggleSflowIndependentAddressConfigEnabled() else
              'Hardware sample rate for' )
   while curLine and \
         not ( endStr in curLine or sflowUdpSrcPortMsg in curLine ):
      m = re.search( r'\s*(%s).*\(VRF: (\S+)\)' % ipAddrPattern, curLine )
      if m:
         srcIpLines.append( m.groups() )
      else:
         m = re.search( r'\s*(%s).*\(VRF: (\S+)\)' % Arnet.Ip6AddrRe, curLine )
         if m:
            srcIp6Lines.append( m.groups() )
      curLine = lines.pop( 0 )
   return lines, curLine, srcIpLines, srcIp6Lines

def parseAgentAddrs( lines ):
   agentIpLines = []
   agentIp6Lines = []
   curLine = lines.pop( 0 )
   # For sFlow samples, we parse agents until we reach the line
   # 'Hardware sample rate for' or 'UDP source port for HW sFlow'
   # Whereas, for drop samples, we parse the sources until we reach
   # a new line
   sampleRateMsg = 'Hardware sample rate for'
   while curLine and \
         not ( sampleRateMsg in curLine or sflowUdpSrcPortMsg in curLine ):
      m = re.search( r'\s*(%s).*\(VRF: (\S+)\)' % ipAddrPattern, curLine )
      if m:
         agentIpLines.append( m.groups() )
      else:
         m = re.search( r'\s*(%s).*\(VRF: (\S+)\)' % Arnet.Ip6AddrRe, curLine )
         if m:
            agentIp6Lines.append( m.groups() )
      curLine = lines.pop( 0 )
   return lines, curLine, agentIpLines, agentIp6Lines

def parseSendDatagrams( lines ):
   curLine = lines.pop( 0 )
   sendLines = []
   while 'BGP export' not in curLine and curLine:
      # Two possible formats:
      #   (<Yes/No>) (VRF: <vrf>)
      #   (<Yes/No>) (<Reason>) (VRF: <vrf>)
      m = re.search( r'(\S+)(\s*\(.+\))*\s*\(VRF: (\S+)\)', curLine )
      if m:
         sendLines.append( ( m.group( 1 ),
                             m.group( 2 ) if m.group( 2 ) else '',
                             m.group( 3 ) ) )
      curLine = lines.pop( 0 )
   return lines, curLine, sendLines

def parseCollectors( lines, dstHostAndPort, dstHost6AndPort ):
   numberOfCollectors = len( dstHostAndPort + dstHost6AndPort )

   colLines = []
   col6Lines = []
   datagramsCount = {}
   colVrf = {}
   for _ in range( numberOfCollectors ):
      curLine = lines.pop( 0 )
      pat = (
         r"(?P<vrf>[A-Za-z0-9]+)\s+(?P<ipAddr>"
         r"%s)\s*\(*(?P<hostname>"
         r"[A-Za-z0-9\.]*)\)*\s+(?P<datagramsCount>[0-9]+)" % ipAddrPattern )
      m = re.search( pat, curLine )
      if m:
         if m.groupdict()[ 'hostname' ]:
            colLines.append( m.groupdict()[ 'hostname' ] )
         else:
            colLines.append( m.groupdict()[ 'ipAddr' ] )
         datagramsCount[ m.groupdict()[ 'ipAddr' ] ] = (
            int( m.groupdict()[ 'datagramsCount' ] )
         )
         colVrf[ m.groupdict()[ 'ipAddr' ] ] = m.groupdict()[ 'vrf' ]
      else:
         pat = (
            r"(?P<vrf>[A-Za-z0-9]+)\s+(?P<ipAddr>"
            r"%s)\s*\(*(?P<hostname>[A-Za-z0-9\.]*)\)*\s+"
            r"(?P<datagramsCount>[0-9]+)" % Arnet.Ip6AddrRe )

         m = re.search( pat, curLine )
         if m:
            if m.groupdict()[ 'hostname' ]:
               col6Lines.append( m.groupdict()[ 'hostname' ] )
            else:
               col6Lines.append( m.groupdict()[ 'ipAddr' ] )
            datagramsCount[ m.groupdict()[ 'ipAddr' ] ] = (
               int( m.groupdict()[ 'datagramsCount' ] )
            )
            colVrf[ m.groupdict()[ 'ipAddr' ] ] = m.groupdict()[ 'vrf' ]

   return lines, colLines, col6Lines, datagramsCount, colVrf

def parseSflowCliOutput( lines, detail=False ):
   lines = lines[ : ]
   blankLines = []
   parsedLines = {}
   curLine = lines.pop( 0 )
   ignoreLines = [ 'waiting for counter update',
                   'Tac.waitFor: waiting for',
                   'Displaying stale counters',
                   'Displaying counters that may be stale' ]
   for ignoreLine in ignoreLines:
      if ignoreLine in curLine:
         curLine = lines.pop( 0 )

   parsedLines[ 'headerLine' ] = curLine
   parsedLines[ 'dashLine' ] = lines.pop( 0 )

   # parsedLines[ 'dstHostAndPort' ] is a list of ( destHostname, destIp, destPort,
   # vrfName ) tuples
   curLine = lines.pop( 0 )
   # enabled: line
   assert 'Enabled:' in curLine
   parsedLines[ 'enabledLine' ] = curLine
   curLine = lines.pop( 0 )
   destLines = []
   dest6Lines = []
   if sflowDestDefaultMsg in curLine:
      curLine = lines.pop( 0 )
   elif "Destination" in curLine:
      lines, curLine, destLines, dest6Lines = parseDestinations( lines )
   parsedLines[ 'dstHostAndPort' ] = destLines
   parsedLines[ 'dstHost6AndPort' ] = dest6Lines

   # parsedLines[ 'srcIp/srcIp6' ] is a list of ( srcIp, vrfName ) tuples
   srcIpLines = []
   srcIp6Lines = []
   if sflowSourceDefaultMsg in curLine:
      curLine = lines.pop( 0 )
   elif "Source" in curLine:
      lines, curLine, srcIpLines, srcIp6Lines = parseSources( lines )

   parsedLines[ 'srcIp' ] = srcIpLines
   parsedLines[ 'srcIp6' ] = srcIp6Lines

   # parsedLines[ 'agentIp/agentIp6' ] is a list of (agentIp, vrfName) tuples
   agentIpLines = []
   agentIp6Lines = []
   if sflowAgentDefaultMsg in curLine:
      curLine = lines.pop( 0 )
   elif "Agent" in curLine and toggleSflowIndependentAddressConfigEnabled():
      lines, curLine, agentIpLines, agentIp6Lines = parseAgentAddrs( lines )
      parsedLines[ 'agentIp' ] = agentIpLines
      parsedLines[ 'agentIp6' ] = agentIp6Lines

   # Add Udp source port info if available
   if sflowUdpSrcPortMsg in curLine:
      parsedLines[ 'dynamicUdpSourcePortLine' ] = curLine
      curLine = lines.pop( 0 )
   else:
      parsedLines[ 'dynamicUdpSourcePortLine' ] = []

   parsedLines[ 'sampleRateLine' ] = curLine
   curLine = lines.pop( 0 )
   if curLine.startswith( 'Hardware sample rate for HW sFlow' ):
      parsedLines[ 'accelSampleRateLine' ] = curLine
      curLine = lines.pop( 0 )
   parsedLines[ 'interval' ] = curLine
   parsedLines[ 'rewriteDscp' ] = lines.pop( 0 )
   parsedLines[ 'dscpValue' ] = lines.pop( 0 )
   if lines[ 0 ].startswith( 'Datagram transmission threshold' ):
      parsedLines[ 'hwDatagramTransmissionThreshold' ] = lines.pop( 0 )

   # blankline
   curLine = lines.pop( 0 )
   assert not curLine
   blankLines.append( curLine )

   # Parse the attributes under the title "Configuration For Drop Packet Samples"
   curLine = lines.pop( 0 )
   if 'Configuration For Drop Packet Samples' in curLine:
      curLine = lines.pop( 0 )
      # There must be a dash line after the heading
      assert '----' in curLine
      curLine = lines.pop( 0 )
      assert 'Enabled:' in curLine
      parsedLines[ 'enabledForDropSamplesLine' ] = curLine
      destLines = []
      dest6Lines = []
      curLine = lines.pop( 0 )
      if sflowDestDefaultMsg in curLine:
         curLine = lines.pop( 0 )
      elif "Destination" in curLine:
         lines, curLine, destLines, dest6Lines = parseDestinations( lines )
         parsedLines[ 'dstHostAndPortForDropSamples' ] = destLines
         parsedLines[ 'dstHost6AndPortForDropSamples' ] = dest6Lines

      srcIpLines = []
      srcIp6Lines = []
      if sflowSourceDefaultMsg in curLine:
         curLine = lines.pop( 0 )
      elif "Source" in curLine:
         lines, curLine, srcIpLines, srcIp6Lines = parseSources( lines )
         parsedLines[ 'srcIpForDropSamples' ] = srcIpLines
         parsedLines[ 'srcIp6ForDropSamples' ] = srcIp6Lines
      # This section must end with a new line
      assert not curLine
      # Line corresponding to the heading "Status"
      curLine = lines.pop( 0 )

   assert curLine == 'Status'
   parsedLines[ 'statusLine' ] = curLine
   curLine = lines.pop( 0 )
   assert curLine == '------'
   parsedLines[ 'statusDashLine' ] = curLine
   parsedLines[ 'runningLine' ] = lines.pop( 0 )
   curLine = lines.pop( 0 )

   if curLine.startswith( 'Hardware acceleration running' ):
      parsedLines[ 'accelEnabledLine' ] = curLine
      curLine = lines.pop( 0 )
   parsedLines[ 'polling' ] = curLine
   parsedLines[ 'sample' ] = lines.pop( 0 )
   parsedLines[ 'truncate' ] = lines.pop( 0 )
   curLine = lines.pop( 0 )
   if curLine.startswith( 'Sample truncation size for HW sFlow' ):
      parsedLines[ 'hwTruncate' ] = curLine
      curLine = lines.pop( 0 )
   if toggleSflowMaxDatagramSizeEnabled():
      if curLine.startswith( 'Datagram maximum size' ):
         parsedLines[ 'maxDatagramSize' ] = curLine
         curLine = lines.pop( 0 )

   sendLines = []
   if "Send datagrams" in curLine:
      lines, curLine, sendLines = parseSendDatagrams( lines )
   parsedLines[ 'send' ] = sendLines

   sendLines = []
   if "BGP export" in curLine:
      curLine = lines.pop( 0 )
      while 'Hardware sample' not in curLine:
         #   (<Yes/No>) (VRF: <vrf>)
         m = re.search( r'(\S+)\s*\(VRF: (\S+)\)', curLine )
         if m:
            sendLines.append( ( m.group( 1 ), m.group( 2 ) ) )
         curLine = lines.pop( 0 )
   parsedLines[ 'export' ] = sendLines

   parsedLines[ 'hwRate' ] = curLine
   curLine = lines.pop( 0 )
   if curLine.startswith( 'Hardware sample rate for HW sFlow' ):
      parsedLines[ 'accelHwRate' ] = curLine
      curLine = lines.pop( 0 )
   if detail:
      parsedLines[ 'hwEnabled' ] = curLine
      parsedLines[ 'outputInterface' ] = lines.pop( 0 )
      parsedLines[ 'mplsExtension' ] = lines.pop( 0 )
      parsedLines[ 'evpnMplsExtension' ] = lines.pop( 0 )
      parsedLines[ 'vplsExtension' ] = lines.pop( 0 )
      parsedLines[ 'vxlanExtension' ] = lines.pop( 0 )
      parsedLines[ 'switchExtension' ] = lines.pop( 0 )
      parsedLines[ 'routerExtension' ] = lines.pop( 0 )
      parsedLines[ 'tunnelIpv4EgrExtension' ] = lines.pop( 0 )
      parsedLines[ 'ingressSubintf' ] = lines.pop( 0 )
      parsedLines[ 'egressSubintf' ] = lines.pop( 0 )
      if toggleSflowL2SubIntfVlanEnabled():
         parsedLines[ 'updateSubIntfVlan' ] = lines.pop( 0 )
      parsedLines[ 'vxlanHeaderStrip' ] = lines.pop( 0 )
      parsedLines[ 'portChannelIfindex' ] = lines.pop( 0 )
      parsedLines[ 'sviInputIfindex' ] = lines.pop( 0 )
      parsedLines[ 'sviOutputIfindex' ] = lines.pop( 0 )
      curLine = lines.pop( 0 )
      if curLine.startswith( 'Hardware acceleration unsupported features' ):
         # We just drop all the unsupported features lines.
         curLine = lines.pop( 0 )
         while re.match( r'^\s.*', curLine ):
            curLine = lines.pop( 0 )
      # Drop the packet format line as it is tied to subinterface feature status
      curLine = lines.pop( 0 )

   # blankline
   assert not curLine
   blankLines.append( curLine )

   curLine = lines.pop( 0 )
   if 'Status For Drop Packet Samples' in curLine:
      curLine = lines.pop( 0 )
      assert '----' in curLine
      curLine = lines.pop( 0 )
      assert curLine.startswith( 'Running' )
      parsedLines[ 'runningForDropSamplesLine' ] = curLine
      curLine = lines.pop( 0 )
      sendLines = []
      if "Send datagrams" in curLine:
         lines, curLine, sendLines = parseSendDatagrams( lines )
      parsedLines[ 'sendForDropSamples' ] = sendLines
      assert not curLine
      curLine = lines.pop( 0 )

   # parsedLines[ 'export' ] is a list of ( export, vrfName ) tuples
   assert curLine == 'Statistics'
   parsedLines[ 'headerStatsLine' ] = curLine
   curLine = lines.pop( 0 )
   assert curLine == '----------'
   parsedLines[ 'statsDashLine' ] = curLine

   parsedLines[ 'totalPkts' ] = lines.pop( 0 )
   parsedLines[ 'swSamples' ] = lines.pop( 0 )
   parsedLines[ 'samplePoolLine' ] = lines.pop( 0 )
   parsedLines[ 'hwSamples' ] = lines.pop( 0 )
   parsedLines[ 'numDatagramsLine' ] = lines.pop( 0 )

   if detail:
      curLine = lines.pop( 0 )
      parsedLines[ 'samplesDiscarded' ] = curLine
   if lines and not lines[ 0 ]:
      curLine = lines.pop( 0 )
      blankLines.append( curLine )

   def getPerCollectorCounters( lines, dropSamples=False ):
      #      Getting Per Collector Counters from the table format
      #
      # VRF     Collector             Datagrams
      # ------- --------------------- ---------
      # default 127.0.0.1                   500
      # default 1.1.1.1 (hostnamev4)        500
      # default 1:2:3::4                    500
      # default 1:2:3::5 (hostnamev6)       500
      #
      curLine = lines.pop( 0 )
      assert all( tableHeader in curLine
                  for tableHeader in ( 'VRF', 'Collector', 'Datagrams' ) )
      if dropSamples:
         parsedLines[ 'dropSamplesTableHeaders' ] = curLine
      else:
         parsedLines[ 'sflowTableHeaders' ] = curLine
      curLine = lines.pop( 0 )
      assert '----' in curLine
      if dropSamples:
         lines, collectors, collectorsV6, datagramsCount, colVrf = (
            parseCollectors( lines, parsedLines[ 'dstHostAndPortForDropSamples' ],
                             parsedLines[ 'dstHost6AndPortForDropSamples' ] )
         )
         parsedLines[ 'perCollectorCountersForDropSamples' ] = collectors
         parsedLines[ 'perCollectorCountersV6ForDropSamples' ] = collectorsV6
         parsedLines[ 'perCollectorDatagramsCountForDropSamples' ] = datagramsCount
         parsedLines[ 'perCollectorVrfForDropSamples' ] = colVrf
      else:
         lines, collectors, collectorsV6, datagramsCount, colVrf = (
            parseCollectors( lines, parsedLines[ 'dstHostAndPort' ],
                             parsedLines[ 'dstHost6AndPort' ] )
         )
         parsedLines[ 'perCollectorCounters' ] = collectors
         parsedLines[ 'perCollectorCountersV6' ] = collectorsV6
         parsedLines[ 'perCollectorDatagramsCount' ] = datagramsCount
         parsedLines[ 'perCollectorVrf' ] = colVrf

      # blankline
      curLine = lines.pop( 0 )
      assert not curLine
      blankLines.append( curLine )

   if lines and 'Drop Packet Sample Counters:' in lines[ 0 ]:
      parsedLines[ 'headerDropSamples' ] = lines.pop( 0 )
      parsedLines[ 'numDatagramsForDropSamplesLine' ] = lines.pop( 0 )
      parsedLines[ 'numDropSamplesReceived' ] = lines.pop( 0 )
      parsedLines[ 'numDropSamplesSent' ] = lines.pop( 0 )
      if lines:
         curLine = lines.pop( 0 )
         assert not curLine
         blankLines.append( curLine )

   if lines and 'Collector Counters' in lines[ 0 ]:
      parsedLines[ 'headerCollectorCountersLine' ] = lines.pop( 0 )
      parsedLines[ 'collectorCountersDashedLine' ] = lines.pop( 0 )
      curLine = lines.pop( 0 )
      assert not curLine
      blankLines.append( curLine )

      if lines and 'Flow and Counter Samples' in lines[ 0 ]:
         parsedLines[ 'headerSflowCollector' ] = lines.pop( 0 )
         curLine = lines.pop( 0 )
         assert not curLine
         blankLines.append( curLine )
         getPerCollectorCounters( lines )

      if lines and 'Drop Samples' in lines[ 0 ]:
         parsedLines[ 'headerDropSamplesCollector' ] = lines.pop( 0 )
         curLine = lines.pop( 0 )
         assert not curLine
         blankLines.append( curLine )
         getPerCollectorCounters( lines, dropSamples=True )

   parsedLines[ 'blankLines' ] = blankLines

   return parsedLines

def sflowHwStatus( sflowHwStatusDir ):
   if len( sflowHwStatusDir ):
      return next( iter( sflowHwStatusDir.values() ) )
   else:
      return Tac.newInstance( 'Sflow::HwStatus', 'empty' )

def sflowHwSampleRate( sflowHwStatusDir ):
   for status in sflowHwStatusDir.values():
      if status.sampleRate:
         return status.sampleRate
   return 0
