# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import ReversibleSecretCli
from datetime import datetime
import time

# Useful time references
oneMin = 60
oneHour = 60 * oneMin
oneDay = 24 * oneHour

# All tests run soon after `rightNow` which is after `now` and before
# `nearlyEndOfTime`.
# 'now' is after `beginningOfTime`.
rightNow = int( Tac.utcNow() )
now = rightNow - oneDay
soon = rightNow + oneDay
beginningOfTime = 0    # Thursday, January 1,   1970 12:00:00 AM
nearlyEndOfTime = 2 ** 64 - 2 # largest legal value which is not infinity

assert now > beginningOfTime
assert now < nearlyEndOfTime

_LifetimeType = Tac.Type( "Mgmt::Security::SharedSecretProfile::Lifetime" )
isInfiniteEnd = _LifetimeType.isInfiniteEnd
infiniteLifetimeEnd = _LifetimeType.infiniteLifetimeEnd

# Some of the tests need to be time-invariant but use future keys, and so use these
# time-dependent variables. Other tests can be firmly in the past, and so are
# time-invariant.
current = _LifetimeType( now, soon )
currentEndless = _LifetimeType.infiniteLifetimeEnd( now )
endsSoon = _LifetimeType( beginningOfTime, soon )
infiniteLifetime = _LifetimeType.infiniteLifetime()
future = _LifetimeType( soon + oneDay, nearlyEndOfTime )
# Overlap by 10 seconds with "current"
futureOverlapping = _LifetimeType( soon - 10, nearlyEndOfTime )
almostInfinite = _LifetimeType( beginningOfTime, nearlyEndOfTime )
fromNowOn = _LifetimeType( now, nearlyEndOfTime )
untilNow = _LifetimeType( beginningOfTime, now )
justStarted = _LifetimeType( rightNow, nearlyEndOfTime )
hasNotStarted = _LifetimeType( nearlyEndOfTime - 1, nearlyEndOfTime )
startsSoon = _LifetimeType( soon + oneDay, nearlyEndOfTime )

class LifetimeHelper:
   """
   Python interface for string (de)serialization, among other helpers, for use by
   Cli and CliSave plugins, as well as tests.
   """
   tacType = "Mgmt::Security::SharedSecretProfile::_LifetimeType"

   @staticmethod
   def _safeDateStr( timestamp, fmt, timezone ):
      # This API is only correct when 'timezone' is either UTC
      # or the timezone of the system clock.
      assert timezone in ( 'UTC', time.strftime( '%Z' ) )
      try:
         dt = datetime.utcfromtimestamp( timestamp ) if timezone == 'UTC' \
              else datetime.fromtimestamp( timestamp )
      except ( ValueError, OverflowError, OSError ):
         dt = datetime.max
      return dt.strftime( fmt )

   @staticmethod
   def toString( lifetime, timezone='UTC' ):
      if lifetime.isInfinite():
         return "infinite"
      fmt = "%Y-%m-%d %H:%M:%S"
      startStr = LifetimeHelper._safeDateStr( lifetime.start, fmt, timezone )
      endStr = "infinite"
      if not lifetime.hasInfiniteEnd():
         endStr = LifetimeHelper._safeDateStr( lifetime.end, fmt, timezone )
      return f"{startStr} {endStr}"

   @staticmethod
   def _displayDate( date, timezone='UTC' ):
      fmt = f"%B %d %Y, %H:%M {timezone}"
      return LifetimeHelper._safeDateStr( date, fmt, timezone )

   @staticmethod
   def displayExpiration( lifetimeEnd ):
      if isInfiniteEnd( lifetimeEnd ):
         return "Does not expire"
      return LifetimeHelper._displayDate( lifetimeEnd )

   @staticmethod
   def displayDuration( start, end, timezone='UTC' ):
      """
      The start / end times are in UTC seconds. This method renders them with the
      full month, e.g. 'January 5th, 1970' or 'infinite'. For large time values, the
      date is truncated to datetime.max.
      """
      if _LifetimeType( start, end ).isInfinite():
         return 'infinite'
      startString = LifetimeHelper._displayDate( start, timezone )
      if _LifetimeType( start, end ).hasInfiniteEnd():
         endString = 'infinite'
      else:
         endString = LifetimeHelper._displayDate( end, timezone )
      return "%s to %s" % ( startString, endString )

   @staticmethod
   def displayLifetime( lifetime, timezone='UTC' ):
      return LifetimeHelper.displayDuration( lifetime.start, lifetime.end, timezone )

def Secret( sid=None, secret=None, receiveLifetime=infiniteLifetime,
            transmitLifetime=infiniteLifetime ):
   if sid is None:
      return Tac.newInstance( "Mgmt::Security::SharedSecretProfile::SecretConfig" )
   else:
      return Tac.newInstance( "Mgmt::Security::SharedSecretProfile::SecretConfig",
                              sid, secret, receiveLifetime, transmitLifetime )

def createSharedProfileSecret( secretId, password,
                               receiveLifetime=infiniteLifetime,
                               transmitLifetime=infiniteLifetime,
                               recvLifetimeStart=None, recvLifetimeEnd=None,
                               transLifetimeStart=None, transLifetimeEnd=None,
                               salt=None, initialVector=None ):
   if recvLifetimeStart is not None:
      assert recvLifetimeEnd is not None
      receiveLifetime = _LifetimeType( recvLifetimeStart, recvLifetimeEnd )
   if transLifetimeStart is not None:
      assert transLifetimeEnd is not None
      transmitLifetime = _LifetimeType( transLifetimeStart, transLifetimeEnd )
   return Secret( str( secretId ),
                  ReversibleSecretCli.generateSecretEntity(
                     password,
                     salt=salt,
                     initialVector=initialVector ),
                  receiveLifetime, transmitLifetime )

def createBasicSspProfileValidator( config, distinctRxTxLifetimeSupported,
      maxIdSizeBytes, maxKeyLength ):
   return Tac.newInstance(
      "Mgmt::Security::SharedSecretProfile::BasicValidator", config,
      distinctRxTxLifetimeSupported, maxIdSizeBytes, maxKeyLength )
