#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import SharedMem
import threading

# To allow users to do SharkLazyMount.mountInfo()
# pylint: disable=unused-import
from Shark import mountInfo

t3 = Tracing.trace3

# Dictionary of MountProxies, keys are Shark entity paths
sharkProxies = {}
proxyLock = threading.Lock()

def mount( entityManager, entityPath, entityType, entityInfo, autoUnmount,
           unmountTimeout=600 ):
   ''' Returns a SharedMem proxy for the given entity path, type and info.
   Any attempt to access the proxy will trigger the mount of the entity. After
   the specified timeout, the entity will be unmounted if possible (ie if there
   are no other references to the entity)'''

   # Why provide the flag at all? First off, future-proofing (there
   # may be cases where it makes sense to stay mounted). But perhaps
   # more importantly, I want it to be _extremely_ clear to any users
   # of this that SharkLazyMount performs autoUnmounting under the
   # hood. So, we _force_ the user to provide this flag.
   assert autoUnmount, 'LazyMount currently only supports autoUnmount'

   shmemEm = SharedMem.entityManager( sysdbEm=entityManager )

   # Check if a proxy was already created for this entity.
   # NOTE: We cannot invoke the __nonzero__ operator on the potentially existing
   # proxy, hence the two lookups.
   fullPath = "%s/%s" % ( shmemEm.sysname, entityPath )
   with proxyLock:
      if fullPath in sharkProxies:
         proxy = sharkProxies.get( fullPath )
         t3( 'SharkLazyMount.mount(): found _Proxy at ', fullPath )
         # Make sure that the existing proxy has the requested entity type
         assert proxy.entityType_ == entityType, \
            'Trying to lazy-mount %s with type %s but proxy has type %s' % \
            ( entityPath, entityType, proxy.entityType_ )
         # In non-cohab mode, make sure that the existing proxy has
         # the requested mode
         if not shmemEm.isLocalEm:
            assert proxy.entityInfo_ == entityInfo, \
               'Trying to lazy-mount %s with info %s but proxy has info %s' % \
               ( entityPath, entityInfo, proxy.entityInfo_ )
         return proxy

      # We don't have a proxy for this entity yet
      t3( 'SharkLazyMount.mount(): creating SharedMem.AutoUnmountEntityProxy for '
          'type:{} path:{} info:{} isLocalMem:{}'.format(
             entityType, entityPath, entityInfo, shmemEm.isLocalEm ) )

      # AutoUnmount should only be triggered if the EntityManager sees
      # 3 references (the EntityManager reference itself, plus the
      # circular reference present because of the Manager, plus the
      # entry in the ShadowManager offsetToSharko collection). For
      # cohab, we get a WriterManager, and there is no offsetToSharko
      # reference.
      minRefs = 2 if shmemEm.isLocalEm else 3
      proxy = SharedMem.AutoUnmountEntityProxy(
         shmemEm, entityPath, entityType, entityInfo, minRefs, unmountTimeout )
      sharkProxies[ fullPath ] = proxy
      return proxy

def force( proxy ):
   '''Ensures that the underlying entity wrapped by the proxy is mounted, and
      returns it. This is particularly useful when passing the entity to c++,
      where it requires the actual entity instance and not a proxy'''
   return proxy.force()
