# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''
Contains functions for adding a signature file to a zip file
such as a SWI or a SWIX, particularly preparing the SWI/SWIX
with a null signature and adding the final signature file to
the SWI/SWIX. The signature is generated from functions used
in SignatureRequest.py.
'''

from __future__ import absolute_import, division, print_function
import base64
import os
import shutil
from binascii import crc32
import hashlib
import zipfile
import codecs

import CRC32Collision
import SignatureRequest
import SwiSignLib
from ArPyUtils.ArTrace import Tracing

traceHandle = Tracing.Handle( 'SwiSign' )
t0 = traceHandle.trace0

# Constants for signing
SIGN_VERSION = 1
SIGN_HASH = 'SHA-256'
SIGNATURE_MAX_SIZE = 8192

class Signature( object ):
   ''' The contents of the signature file that will go into the zip file
   being signed. '''
   def __init__( self ):
      self.version = SIGN_VERSION
      self.hash = ""
      self.cert = ""
      self.signature = ""
      self.crcpadding = [ 0, 0, 0, 0 ]
      self.offset = 0
      self.nullcrc32 = 0

   def __repr__( self ):
      data = r''
      data += "HashAlgorithm:" + self.hash + "\n"
      data += "IssuerCert:" + self.cert + "\n"
      data += "Signature:" + self.signature + "\n"
      data += "Version:" + str( self.version ) + "\n"
      crcPadding = "CRCPadding:"
      for byte in self.crcpadding:
         crcPadding += "%c" % ( byte & 0xff )
      data += "Padding:"
      # We need to add padding to make the null signature the same length as
      # the actual signature so they will generate the same hash.
      # The padding amount factors in the length of the current signature data,
      # the CRC padding, and the newline character for the padding field.
      paddingAmt = SIGNATURE_MAX_SIZE - len( data ) - len( crcPadding ) - 1
      data += "*" * paddingAmt + "\n"
      data += "CRCPadding:" # we add actual crc padding later since it is in bytes
      assert len( data ) == SIGNATURE_MAX_SIZE - len( self.crcpadding )
      return data

   def getBytes( self ):
      data = b''
      data += self.__repr__().encode()
      # Add CRC padding
      for byte in self.crcpadding:
         data += b'%c' % ( byte & 0xff )
      return data

def addNullSig( swiFile, size ):
   ''' Add a null signature to the zip file, noting the location of the
   signature and the CRC32 of the resulting file, which will be used to replace
   the signature with the real one later. '''
   data = '\000' * size
   nullcrc32 = crc32( data.encode() ) & 0xffffffff
   sigFileName = SwiSignLib.signatureFileName( swiFile )

   # Use run-of-the-mill /usr/bin/zip, so signatures can be extracted with unzip and
   # re-inserted with zip. The problem with zipfile is that it sets file meta info
   # "version required to extract" as 2.0 and /usr/bin/zip uses 1.0, thus messing
   # the signature. And even if we used zipfile to re-insert, zipfile has no option
   # to preserve the date, again messing the signature. With meta images (swim), we
   # need to extract and insert signatures, so zipfile.py is a non-starter.
   tmpDir = "/tmp/insert-sig-%d" % os.getpid()
   os.mkdir( tmpDir )
   f = open( "%s/%s" % ( tmpDir, sigFileName ), "w" )
   f.write( data )
   f.close()
   # We rename the image temporarily because zip will create a new file "file.zip"
   # if the archive target has no exension (some tests create images without .ext)
   ret = os.system( 'set -e; image=$(readlink -f "%s"); tmpDir=%s; file=%s; '
                    'cd $tmpDir; cp "$image" temp.swi; '
                    'zip -q -0 -X temp.swi $file; '
                    'mv temp.swi "$image"' % ( swiFile, tmpDir, sigFileName ) )
   shutil.rmtree( tmpDir )
   assert ret == 0, "Cannot add Null signature to %s" % swiFile

   with zipfile.ZipFile( swiFile, 'r' ) as swi:
      with swi.open( swi.getinfo( sigFileName ) ) as sigFile:
         # pylint: disable-msg=protected-access
         offset = sigFile._fileobj.tell()
   return offset, nullcrc32

def generateHash( swi, hashAlgo, blockSize=65536 ):
   # For now, we always use SHA-256. This is not a user input (we hardcode it in)
   assert hashAlgo == 'SHA-256'
   # pylint: disable-msg=no-member
   sha256sum = hashlib.sha256()
   with open( swi, 'rb' ) as swiFile:
      for block in iter( lambda: swiFile.read( blockSize ), b'' ):
         sha256sum.update( block )
   return sha256sum.hexdigest()

def prepareDataForServer( swiFile, version, swiSignature, product='EOS' ):
   ''' Adds a null signature to the zip file and returns the data to send to a
   signing server to get the real signature. '''
   body = {}
   hashAlgo = SIGN_HASH
   # Add null signature to swiFile (swiSignature is a null signature at this point)
   offset, nullcrc32 = addNullSig( swiFile, len( swiSignature.getBytes() ) )

   # Update swiSignature with the offset and nullcrc32
   swiSignature.offset = offset
   swiSignature.nullcrc32 = nullcrc32

   # Generate hash of the swi file with the null signature
   hashStr = generateHash( swiFile, hashAlgo )
   hashStr = base64.standard_b64encode( codecs.decode( hashStr, 'hex' ) ).decode()
   t0( "hashStr is: ", hashStr )
   body = { "version": version,
            "product": product,
            "hash_algorithm": hashAlgo,
            "input_hash": hashStr
          }
   return body

def generateSigFileFromServer( signatureData, swiFile, swiSignature ):
   ''' Replaces the null signature of the zip file with a signature obtained from
   an external source such as a signing server. '''
   data = SignatureRequest.extractServerData( signatureData )

   # Update the certificate and signature, and hash fields of swiSignature.
   # Fields from the server are returned in unicode and must be converted to
   # byte string for crc padding
   swiSignature.cert = base64.standard_b64encode(
      data.certificate.encode() ).decode()
   swiSignature.hash = str( data.hashAlgorithm )
   swiSignature.signature = data.signature

   # Update crc padding for swiSignature to match the null signature.
   # The last 4 bytes of swiSignature is the crc padding of swiSignature.
   nullcrc32 = swiSignature.nullcrc32
   swiSigCrc32 = crc32( str( swiSignature ).encode() ) & 0xffffffff
   swiSignature.crcpadding = CRC32Collision.matchingBytes( nullcrc32, swiSigCrc32 )

   # Rewrite the swi-signature in the right place, replacing the null signature
   offset = swiSignature.offset
   with open( swiFile, 'r+b' ) as outfile:
      outfile.seek( offset )
      outfile.write( swiSignature.getBytes() )
