#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac
import Tracing
import SharedMem
import threading

# To allow users to do SmashLazyMount.mountInfo()
# pylint: disable-msg=W0611
from Smash import mountInfo

t3 = Tracing.trace3

# Dictionary of MountProxies, keys are Smash entity paths
smashProxies = {}
proxyLock = threading.Lock()

def mount( entityManager, entityPath, entityType, entityInfo, autoUnmount=False,
           unmountTimeout=600 ):
   ''' Returns a SharedMem proxy for the given entity path, type and info.
   Any attempt to access the proxy will trigger the mount of the entity.
   If the entity was already mounted, it is returned directly,
   instead of returning a proxy.'''

   # Extraction of {sysname,isLocalEm} attributes.
   if isinstance( entityManager, Tac.Type( 'TacSharedMem::EntityManager' ) ):
      # If the entityManager is SharedMem type, then we obtain the
      # {sysname,isLocalEm} attributes directly.
      emSysname = entityManager.sysname
      emIsLocalEm = entityManager.isLocalEm

      # We do not need to re-create or retrieve a shmemEm if one is passed in the 
      # arguments of this function.
      isShmemEm = True
   else:
      try:
         # If the entityManager is derived from the EntityManager._NotSimple class,
         # then its {sysname,isLocalEm} attributes can be extracted through member
         # methods. It is expected that the passed in entity manager is either an
         # EntityManager.Sysdb or an EntityManager.Local if it didn't hit the
         # TacSharedMem::EntityManager case above.
         emSysname = entityManager.sysname()
         emIsLocalEm = entityManager.isLocalEm()
         isShmemEm = False
      except ( NameError, TypeError ) as ex:
         # NameError is raised if the attribute does not exist on the passed in
         # value. Meanwhile, TypeError is raised if the attribute is not callable.
         # (Ex. if the sysname attribute is a str instead of a bound method)
         raise Exception( "Error: Unrecognized entityManager type." ) from ex

   # Check if a proxy was already created for this entity.
   # NOTE: We cannot invoke the __nonzero__ operator on the potentially existing
   # proxy, hence the two lookups.
   fullPath = "%s/%s" % ( emSysname, entityPath )
   with proxyLock:
      if fullPath in smashProxies:
         proxy = smashProxies.get( fullPath )
         t3( 'SmashLazyMount.mount(): found _Proxy at ', fullPath )
         # Make sure that the existing proxy has the requested entity type
         assert proxy.entityType_ == entityType, \
            'Trying to lazy-mount %s with type %s but proxy has type %s' % \
            ( entityPath, entityType, proxy.entityType_ )
         # In non-cohab mode, make sure that the existing proxy has
         # the requested mode
         if not emIsLocalEm:
            assert proxy.entityInfo_.mode == entityInfo.mode, \
               'Trying to lazy-mount %s with mode %s but proxy has mode %s' % \
               ( entityPath, entityInfo.mode, proxy.entityInfo_.mode )
         # If both the existing proxy's mode and the requested mode are 'writer',
         # make sure that the collection infos match
         if proxy.entityInfo_.mode == 'writer' and entityInfo.mode == 'writer':
            assert proxy.entityInfo_.collectionInfo == entityInfo.collectionInfo, \
               'Trying to lazy-mount %s with collection info %s and mode writer' \
               ' but proxy has collection info %s and mode writer' % \
               ( entityPath, entityInfo.collectionInfo,
                 proxy.entityInfo_.collectionInfo )
         if autoUnmount:
            assert isinstance( proxy, SharedMem.AutoUnmountEntityProxy ), \
               'Trying to lazy-mount %s with autoUnmount=True but found existing' \
               ' proxy with autoUnmount=False' % ( entityPath )
         else:
            assert isinstance( proxy, SharedMem.EntityProxy ), \
               'Trying to lazy-mount %s with autoUnmount=False but found existing' \
               ' proxy with autoUnmount=True' % ( entityPath )
         return proxy

      # Check if the entity is already mounted
      if not isShmemEm:
         # Create or retrieve a SharedMem entityManager
         shmemEm = SharedMem.entityManager( sysname=emSysname, local=emIsLocalEm )
      else:
         shmemEm = entityManager
      entity = shmemEm.getTacEntity( entityPath )
      if entity: # pylint: disable=too-many-nested-blocks
         t3( 'SmashLazyMount.mount(): entity already mounted' )
         # Make sure that the mounted entity has the requested type
         # and its collections are initialized with the requested info
         assert entity.tacType.fullTypeName == entityType, \
            'Trying to lazy-mount %s with type %s but mounted entity has type %s' % \
            ( entityPath, entityType, entity.tacType.fullTypeName )
         if SharedMem.cohabMultithreadMounts():
            # Don't check initialization parameters in multithreading mode
            # See doMount() in TacSharedMem/EntityManager.tin for details.
            return entity
         for attr in entity.attributes:
            if attr.endswith( 'Control' ):
               try:
                  control = getattr( entity, attr )
               except AttributeError:
                  continue
               if control.tacType.fullTypeName == 'TacSmash::CollectionControl':
                  collectionName = attr[:-( len( 'Control' ) )]
                  assert collectionName in entity.attributes
                  if emIsLocalEm:
                     # In cohab mode, the actual mode should always be 'writer'
                     assert control.mode() == 'writer', 'Trying to lazy-mount %s' \
                        ' in cohab mode but %s in mounted entity has mode %s' % \
                        ( entityPath, collectionName, control.mode() )
                  else:
                     # In non-cohab mode, make sure the modes match
                     assert control.mode() == entityInfo.mode, 'Trying to lazy-' \
                        'mount %s with mode %s but %s in mounted entity has mode ' \
                        '%s' % ( entityPath, entityInfo.mode, collectionName,
                                 control.mode() )
                  if entityInfo.mode == 'writer':
                     assert entityInfo.collectionInfo, 'Trying to lazy-mount %s as' \
                        ' writer but collection info is missing'
                     if control.maxSize() != 0:
                        reqSize = entityInfo.collectionInfo[ collectionName ].size
                        assert control.maxSize() == reqSize, 'Trying to' \
                           ' lazy-mount %s with size %s for %s but mounted entity' \
                           ' has size %s' % ( entityPath, reqSize, collectionName,
                                              control.maxSize() )

      # We don't have a proxy for this entity yet
      proxy = None
      if autoUnmount:
         t3( 'SmashLazyMount.mount(): creating SharedMem.AutoUnmountEntityProxy ' \
             'for type:{} path:{} info:{} isLocalMem:{} unmountTimeout:{}'.format(
                entityType, entityPath, entityInfo, emIsLocalEm,
                unmountTimeout ) )
         proxy = SharedMem.AutoUnmountEntityProxy(
            shmemEm, entityPath, entityType, entityInfo,
            timeout=unmountTimeout )
         smashProxies[ fullPath ] = proxy
      else:
         t3( 'SmashLazyMount.mount(): creating SharedMem.EntityProxy for ' \
             'type:{} path:{} info:{} isLocalMem:{}'.format(
                entityType, entityPath, entityInfo, emIsLocalEm ) )
         proxy = SharedMem.EntityProxy( shmemEm, entityPath, entityType, entityInfo )
         smashProxies[ fullPath ] = proxy

      return proxy

def force( proxy ):
   '''Ensures that the underlying entity wrapped by the proxy is mounted, and
      returns it. This is particularly useful when passing the entity to c++,
      where it requires the actual entity instance and not a proxy'''
   if isinstance( proxy, SharedMem.EntityProxy ):
      # This proxy type currently leverages __tac_object__ to perform
      # the mount, so we can just operate directly on it
      return proxy

   return proxy.force()
