#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#
# This file provides a library for developers to use when writing scripts and other
# tools which need transceiver SMBus device information.
#
# There have been many instances where developers have needed to quickly prototype
# some code which issues SMBus requests to transeivers.  This library exists to
# assist the speed and quality in which we can develop SMBus scripts either for
# customers, for transceiver bringup, or even for transceiver qualification.
#
# This file is installed in EOS and therefore present on any switch running EOS.

import json
import re

import Tac

class TransceiverSmbusDevice:
   """Data object which represents a transceiver SMBus device"""

   def __init__( self, accelId, busId, deviceId,
                 deviceName, linecardNumber=None ):
      """accelId : int
         busId : int
         deviceId : int
         deviceName : string of the form 'EthernetX'
         linecardNumber : int or None
      """
      self.accelId = accelId
      self.busId = busId
      self.deviceId = deviceId
      self.linecardNumber = linecardNumber
      self.deviceName = deviceName

class TransceiverSmbusDeviceMap:
   """Represents a mapping from interface name to SMBus device data

   The class variable 'mapping' is a map of { str : TransceiverSmbusDevice }.
   A key of 'mapping' will be of the form 'EthernetX' on a fixed system, or
   'EthernetX/X' on a modular system.
   """

   def __init__( self ):
      output = self.__runSmbusShowCmd()
      self.mapping = TransceiverSmbusDeviceMap.createMapping( output )

   def __runSmbusShowCmd( self ):
      """Runs 'show platform smbus counters' by invoking the CLI"""
      cmd = [ 'FastCli', '-p', '15' ]
      output = Tac.run( cmd, input='show platform smbus counters | json',
                        shell=True, stdout=Tac.CAPTURE )
      return json.loads( output )

   @staticmethod
   def parseDevice( smbusDevice, name ):
      """
      Parses single line of 'show platfrom smbus counters' output.

      Parameters
      ----------
      smbusDevice : str
         Name of the device, for example "01/05/0x50"

      name : str
         Name of the thing using SMBus, for example "Ethernet9/32 or Fabric1"

      Returns
      -------
      None if the line does not contain a transceiver SMBus path,
      returns a TransceiverSmbusDevice otherwise.
      """
      # Pre-process the line to make it easier to work with
      name = name.replace( 'Xcvr', 'Ethernet' )

      # First search for 'Ethernet'
      ethSearch = re.search( r'(Ethernet|Fabric)?\d+(/\d+)?', name )
      if ethSearch is None:
         return None

      ethName = ethSearch.group( 0 )

      linecardNum = None
      if smbusDevice.startswith( 'Linecard' ):
         linecardSearch = re.search( r'Linecard(\d+)/', smbusDevice )
         linecardNum = int( linecardSearch.group( 1 ) )

      # Search for Smbus address path
      pathSearch = re.search( r'(\d{2})/(\d{2})/(0x50)', smbusDevice )
      if pathSearch is None:
         # We should always match, but we gate with a 'None' check anyways
         # because there might be some cases I didn't consider where the
         # Smbus address path does not exist and we would rather not crash.
         return None

      pathTuple = pathSearch.groups()
      if pathTuple is not None:
         accelId, busId, deviceId = pathTuple
         accelId = int( accelId )
         busId = int( busId )
         deviceId = int( deviceId, 16 )  # hex
         return TransceiverSmbusDevice( accelId, busId, deviceId,
                                        ethName, linecardNum )
      else:
         return None

   @staticmethod
   def createMapping( output ):
      """Returns a mapping from interface name to SMBus device address

      The keys of the returned dictionary will be strings of the format
      'Ethernet1' / 'Fabric1' (or 'Ethernet3/1' / 'Fabric3/1' if a modular system).
      The values of the dictionary will be objects of type TransceiverSmbusDevice.
      Return type -> { str : TransceiverSmbusDevice }
      """
      smbusDeviceMap = {}

      for dev, info in output[ "devices" ].items():
         xcvr = TransceiverSmbusDeviceMap.parseDevice( dev, info[ "name" ] )
         if xcvr:
            smbusDeviceMap[ xcvr.deviceName ] = xcvr

      return smbusDeviceMap
