# Copyright (c) 2007-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# Other Snmp plugins should not mount entities under interface/config as
# read-only. To keep those entities mounted as writable, they should use
# IntfSnmp's mounts instead.  In order to use IntfSnmp's mounts, they should
# require snmp/intf and call Snmp.addStage2MountFunction. This means after all
# initial mounts are completed, the next stage of mounts and callbacks occur.

import Cell
import Plugins
import Tac, Tracing
import SharedMem
import Smash

traceHandle = Tracing.Handle( 'IntfSnmp' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1

class IntfSnmpPlugin:
   def __init__( self, ctx ):

      entityManager = ctx.entityManager()
      self.entityManager_ = entityManager
      sysdb = entityManager.root()
      snmpRoot = sysdb.parent[ 'snmp' ]
      intfDir = snmpRoot.mkdir( 'intf' )
      ifmib = intfDir.newEntity( 'IntfSnmp::IfMib', 'ifmib' )
      ifqueuemib = intfDir.newEntity( 'IntfSnmp::IfQueueMib', 'ifqueuemib' )
      self.root_ = None

      mg = entityManager.mountGroup()

      # This plugin mounts IntfConfigs writable, so they have multiple writers, a
      # situation we usually try to avoid.  Here it's unavoidable--we can't do
      # configuration through Snmp unless we can change values.  At some point we
      # may need to rethink the implementation, perhaps multiplexing all attribute
      # modifications for a given IntfConfig through a single entity. For now, we
      # can just pretend that the IfTableRowUpdater is another Cli session.

      mg.mount( 'interface/config', 'Tac::Dir', 'wi' )
      intfStatusAll = mg.mount( 'interface/status/all',
                                'Interface::AllIntfStatusDir', 'r' )
      snmpIntfConfig = mg.mount( 'snmp/interface/config',
                                 'Snmp::Interface::Config', 'r' )
      snmpIntfStatus = mg.mount( 'snmp/interface/status',
                                 'Snmp::Interface::Status', 'w' )
      globalIntfConfig = mg.mount( 'interface/config/global',
                                   'Interface::GlobalIntfConfig', 'r' )
      mg.mount( 'interface/counter', 'Tac::Dir', 'ri' )
      aclDropConfigDir = mg.mount( 'interface/aclDropCounter/aclDropConfigDir',
                                   'Tac::Dir', 'ri' )

      # We are the writer to ifIndex values.
      dynamicIfIndexStatus = mg.mount( 'interface/ifIndex/dynamic',
                                       'Interface::IfIndexStatus', 'w' )
      allIfIndexStatus = mg.mount( 'interface/ifIndex/all',
                                   'Interface::IfIndexStatus', 'w' )

      # read mount the interface error disable status
      # useful to send additional information on link-down traps
      intfErrdisableStatus = mg.mount( 'interface/errdisable/status',
                                       'Errdisable::Status' )

      # mount the interface queue counters through the writerConfigDirSm
      writerConfigDir = mg.mount(
         Cell.path( 'interface/queueCounter/writerConfigDir' ), 'Tac::Dir', 'ri' )
      ctrAccessor = Tac.newInstance( 'Interface::QueueCounter::CounterAccessor' )
      mountHelper = Tac.newInstance( 'Interface::QueueCounter::SmashMountHelper',
                                     entityManager.cEntityManager() )

      # mount the smash counters for subintf, svi and tunnelIntf (which may reside in
      # Smash, depending on the platform)
      shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
      smashCounterInfo = Smash.mountInfo( 'reader' )
      shmemEm.doMount( 'interface/counter/subintf/current', 
                   'Smash::Interface::AllIntfCounterDir', smashCounterInfo )
      shmemEm.doMount( 'interface/counter/svi/current', 
                   'Smash::Interface::AllIntfCounterDir', smashCounterInfo )
      shmemEm.doMount( 'interface/counter/tunnel/fastpath/current',
                       'Smash::Interface::AllIntfCounterDir', smashCounterInfo )
      Dot1xConsts = Tac.Type( "Interface::IntfDroppedCounterConstants" )
      dot1xDroppedPktCntrs = \
         shmemEm.doMount( Dot1xConsts.smashPath,
                          "Interface::AllIntfDot1xDroppedCounter",
                          smashCounterInfo )
      # Mount QoS in order to update ifSpeed/ifHighSpeed as needed
      qosStatus = mg.mount( 'qos/status', 'Qos::Status', 'r' )

      snmpAgentRoot = sysdb.parent[ 'Snmp' ]

      def _finishMounts():
         writerConfigDirSm = \
               Tac.newInstance( 'Interface::QueueCounter::WriterConfigDirSm',
                                 writerConfigDir, ctrAccessor, mountHelper )

         intfConfigAll = self.entityManager_.lookup( "interface/config/all" )
         self.root_ = snmpAgentRoot.newEntity( 'IntfSnmp::Agent::Root',
                                'IntfSnmp' )

         # TODO - We should convert Interface::AclDropDirSm to store a
         # TacSharedMem::EntityManager::Ptr instead of the deprecated TacSmash::Mount
         smashMount = Tac.newInstance(
            'TacSmash::Mount',
            self.entityManager_.cEntityManager().sysname,
            self.entityManager_.cEntityManager().isLocalEm )
         aclDropDirSm = Tac.newInstance( "Interface::AclDropDirSm",
                                         aclDropConfigDir, smashMount )

         aclDropDirSm.initialized = True
         self.root_.agent = ( globalIntfConfig, intfConfigAll, intfStatusAll,
                              snmpIntfConfig, snmpIntfStatus, ifmib, ifqueuemib,
                              aclDropDirSm, writerConfigDirSm,
                              dynamicIfIndexStatus, allIfIndexStatus,
                              dot1xDroppedPktCntrs, intfErrdisableStatus, qosStatus )

      mg.close( lambda: None )
      ctx.callbackIs( _finishMounts )

@Plugins.plugin( provides=( 'snmp/intf', ) )
def Plugin( ctx ):
   t0( 'IntfSnmp Plugin Loading...' )
   IntfSnmpPlugin( ctx )
   Tac.dlopen( 'libIfMib.so' )
   Tac.dlopen( 'libIfQueueMib.so' )

