# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import weakref

t0 = Tracing.trace0
t4 = Tracing.trace4

# pkgdeps: library LldpSnmp
# pkgdeps: library EbraSnmp
class EthIntfStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Interface::EthIntfStatus'

   def __init__( self, notifier, plugin ):
      Tac.Notifiee.__init__( self, notifier )
      self.plugin_ = plugin
      self.portId_ = None
      self._addNewPortIdMapping()

   def close( self ):
      self._removeOldPortIdMapping()
      Tac.Notifiee.close( self )

   def _addNewPortIdMapping( self ):
      t0( "Interface", self.notifier_.intfId, "has portId", self.notifier_.portId )
      if self.notifier_.portId != Tac.Type( "Interface::EthPortId" ).invalid:
         self.plugin_.addPortIdMapping( self.notifier_.portId,
               self.notifier_.intfId )
      self.portId_ = self.notifier_.portId

   def _removeOldPortIdMapping( self ):
      if self.portId_:
         self.plugin_.removePortIdMapping( self.portId_, self.notifier_.intfId )
      self.portId_ = None

   @Tac.handler( 'portId' )
   def handlePortId( self ):
      self._removeOldPortIdMapping()
      self._addNewPortIdMapping()
      # Handle a case where portId allocation from Ebra agent
      # comes with a delay. Make sure PortConfig, PortStatus and LocalPort
      # are created now if the interface id is present in corresponding notifier's
      # collection - check done already inside every handler.
      self.plugin_.configReactor_.handlePortConfig( self.notifier_.intfId )
      self.plugin_.statusReactor_.handlePortStatus( self.notifier_.intfId )
      self.plugin_.localSystemReactor_.handleLocalPort( self.notifier_.intfId )

class LldpConfigReactor( Tac.Notifiee ):
   notifierTypeName = 'Lldp::Config'

   def __init__( self, notifier, plugin ):
      Tac.Notifiee.__init__( self, notifier )
      self.plugin_ = plugin
      # No action necessary on creation -- if both an Lldp::PortConfig and an
      # EthPhyIntfStatus exist, then the EthIntfStatusReactor will take care of
      # everything.

   @Tac.handler( 'portConfig' )
   def handlePortConfig( self, intfName ):
      if intfName in self.plugin_.intfNameToPortId:
         portId = self.plugin_.intfNameToPortId[ intfName ]
         if intfName in self.notifier_.portConfig:
            self.plugin_.createSnmpPortConfig( intfName, portId )
         else:
            self.plugin_.destroySnmpPortConfig( intfName, portId )

class LldpStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Lldp::AllStatus'

   def __init__( self, notifier, plugin ):
      Tac.Notifiee.__init__( self, notifier )
      self.plugin_ = plugin
      # No action necessary on creation -- if both an Lldp::PortStatus and an
      # EthPhyIntfStatus exist, then the EthIntfStatusReactor will take care of
      # everything.

   @Tac.handler( 'portStatus' )
   def handlePortStatus( self, intfName ):
      if intfName in self.plugin_.intfNameToPortId:
         portId = self.plugin_.intfNameToPortId[ intfName ]
         if intfName in self.notifier_.portStatus:
            self.plugin_.createSnmpPortStatus( intfName, portId )
         else:
            self.plugin_.destroySnmpPortStatus( intfName, portId )

class LldpLocalSystemReactor( Tac.Notifiee ):
   notifierTypeName = 'Lldp::LocalSystem'

   def __init__( self, notifier, plugin ):
      Tac.Notifiee.__init__( self, notifier )
      self.plugin_ = plugin
      # No action necessary on creation -- if both an Lldp::LocalPort and an
      # EthPhyIntfStatus exist, then the EthIntfStatusReactor will take care of
      # everything.

   @Tac.handler( 'localPort' )
   def handleLocalPort( self, intfName ):
      if intfName in self.plugin_.intfNameToPortId:
         portId = self.plugin_.intfNameToPortId[ intfName ]
         if intfName in self.notifier_.localPort:
            self.plugin_.createSnmpLocalPort( intfName, portId )
         else:
            self.plugin_.destroySnmpLocalPort( intfName, portId )
         self.plugin_.root.lldpSm.handleLocalPort( intfName, portId )

# pkgdeps: library Launcher
class AgentStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Launcher::AgentStatus'

   def __init__( self, notifier, plugin ):
      Tac.Notifiee.__init__( self, notifier )
      self.plugin_ = plugin

   @Tac.handler( 'runnable' )
   def handleRunnable( self ):
      if not self.plugin_.snmpStatus.initialized:
         return

      if self.notifier_.runnable:
         t0( 'Lldp runnable, initMibs' )
         self.plugin_.root.initMibs()
      else:
         t0( 'Lldp not runnable, unregisterMibs' )
         self.plugin_.root.unregisterMibs()

class SnmpStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Snmp::Status'

   def __init__( self, notifier, plugin ):
      Tac.Notifiee.__init__( self, notifier )
      self.plugin_ = plugin

   @Tac.handler( 'initialized' )
   def handleSnmpInit( self ):
      lldpAgentStatusReactor = \
         self.plugin_.agentStatusDirReactor_.reactors_.get( 'Lldp' )

      if lldpAgentStatusReactor and lldpAgentStatusReactor.notifier().runnable:
         lldpAgentStatusReactor.handleRunnable()

class LldpSnmpPlugin:
   def __init__( self, ctx ):
      entityManager = ctx.entityManager()
      self.entityManager_ = entityManager
      sysdb = entityManager.root()
      snmpRoot = sysdb.parent[ 'snmp' ]
      lldpRoot = snmpRoot.mkdir( 'lldp' )
      self.root = lldpRoot.newEntity( 'Lldp::Snmp::Root', 'root' )
      self.snmpStatus = sysdb[ 'snmp' ][ 'status' ]

      self.portIdToIntfNames = {} # Maps portId -> set of intfNames.
      self.intfNameToPortId = {} # Maps chosen intfName -> portId.

      mg = entityManager.mountGroup()
      self.config = mg.mount( 'l2discovery/lldp/config', 'Lldp::Config', 'r' )
      # Explicitly mount the following mount referenced by
      # mount l2discovery/lldp/status/all as it is used in the agent
      #
      # l2discovery/lldp/status/all.portStatus/<foo::ptr>
      # -> l2discovery/lldp/status/local.status/1.portStatus/<foo>
      mg.mount( 'l2discovery/lldp/status/local', 'Tac::Dir', 'ri' )
      self.status = mg.mount( 'l2discovery/lldp/status/all', 'Lldp::AllStatus', 'r' )
      self.localSystem = mg.mount(
         'l2discovery/lldp/localSystem', 'Lldp::LocalSystem', 'r' )
      mg.mount( 'interface/status/eth/phy', 'Tac::Dir', 'ri' )
      self.ethPhyIntfStatusDir = mg.mount(
         'interface/status/eth/phy/all', 'Interface::AllEthPhyIntfStatusDir', 'r' )
      self.agentStatusDir = mg.mount( 'sys/status/agentStatusDir',
                                       'Launcher::AgentStatusDir', 'r' )

      self.ethIntfStatusCollReactor_ = None
      self.configReactor_ = None
      self.statusReactor_ = None
      self.localSystemReactor_ = None
      self.agentStatusDirReactor_ = None
      self.snmpStatusReactor_ = None

      def _finishMounts():
         plugin = weakref.proxy( self )
         self.ethIntfStatusCollReactor_ = Tac.collectionChangeReactor(
            self.ethPhyIntfStatusDir.intfStatus,
            EthIntfStatusReactor,
            reactorArgs=( plugin, ) )
         self.configReactor_ = LldpConfigReactor( self.config, plugin )
         self.statusReactor_ = LldpStatusReactor( self.status, plugin )
         self.localSystemReactor_ = LldpLocalSystemReactor(
            self.localSystem, plugin )
         # React to the agentStatusDir::agent collection containing AgentStatus, and
         # only instantiate an AgentStatusReactor for the AgentStatus representing
         # Lldp
         self.agentStatusDirReactor_ = Tac.collectionChangeReactor(
            self.agentStatusDir.agent,
            AgentStatusReactor,
            reactorArgs=( plugin, ),
            reactorFilter=lambda agentName: agentName == 'Lldp' )
         self.snmpStatusReactor_ = SnmpStatusReactor( self.snmpStatus, plugin )

         self.root.lldpSm = ( self.status, self.localSystem, self.config, self.root )

      mg.close( lambda: None )
      ctx.callbackIs( _finishMounts )

   def createSnmpPortConfig( self, intfName, portId ):
      portConfig = self.config.portConfig[ intfName ]
      self.root.newPortConfig( portId, portConfig )

   def createSnmpPortStatus( self, intfName, portId ):
      portStatus = self.status.portStatus[ intfName ]
      self.root.newPortStatus( portId, portStatus )
      self.root.newPortStatusSm( portId, portStatus, self.root )

   def createSnmpLocalPort( self, intfName, portId ):
      localPort = self.localSystem.localPort[ intfName ]
      self.root.newLocalPort( portId, localPort )

   def destroySnmpPortConfig( self, intfName, portId ):
      if portId in self.root.portConfig:
         del self.root.portConfig[ portId ]

   def destroySnmpPortStatus( self, intfName, portId ):
      if portId in self.root.portStatus:
         self.root.portStatusSm[ intfName ].doClose()
         del self.root.portStatusSm[ intfName ]
         del self.root.portStatus[ portId ]

   def destroySnmpLocalPort( self, intfName, portId ):
      if portId in self.root.localPort:
         del self.root.localPort[ portId ]

   def _addIntfNameMapping( self, intfName, portId ):
      """Adds a mapping from portId to the single chosen intfName for the portId."""
      t4( "Interface", intfName, "has been chosen for portId", portId )
      self.intfNameToPortId[ intfName ] = portId
      if intfName in self.config.portConfig:
         self.createSnmpPortConfig( intfName, portId )
      if intfName in self.status.portStatus:
         self.createSnmpPortStatus( intfName, portId )
      if intfName in self.localSystem.localPort:
         self.createSnmpLocalPort( intfName, portId )

   def _removeIntfNameMapping( self, intfName ):
      """Removes a mapping from portId to intfName."""
      t4( "Interface", intfName, "has been un-chosen" )
      portId = self.intfNameToPortId[ intfName ]
      del self.intfNameToPortId[ intfName ]
      if intfName in self.config.portConfig:
         self.destroySnmpPortConfig( intfName, portId )
      if intfName in self.status.portStatus:
         self.destroySnmpPortStatus( intfName, portId )
      if intfName in self.localSystem.localPort:
         self.destroySnmpLocalPort( intfName, portId )

   def addPortIdMapping( self, portId, intfName ):
      """Adds a mapping from an intfName to the portId for that interface.  There may
      be mappings from several intfNames to a particular portId; of these intfNames,
      only one will be chosen to correspond to the portId."""
      t4( "Interface", intfName, "now has portId", portId )
      intfNames = self.portIdToIntfNames.setdefault( portId, set() )
      if not intfNames:
         self._addIntfNameMapping( intfName, portId )
      intfNames.add( intfName )

   def removePortIdMapping( self, portId, intfName ):
      """Removes a mapping from intfName to portId."""
      t4( "Interface", intfName, "no longer has portId", portId )
      if intfName in self.intfNameToPortId:
         self._removeIntfNameMapping( intfName )
      intfNames = self.portIdToIntfNames[ portId ]
      intfNames.remove( intfName )
      if not intfNames:
         del self.portIdToIntfNames[ portId ]
      else:
         # Select one of the remaining intfNames.
         otherIntfName = None
         for otherIntfName in intfNames:
            break
         assert otherIntfName is not None
         self._addIntfNameMapping( otherIntfName, portId )

thePlugin = None

def Plugin( ctx ):
   t0( 'Loading LLDP SnmpPlugin' )
   global thePlugin
   thePlugin = LldpSnmpPlugin( ctx )
