# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Plugins
import Tac

# Types instantiated locally
PimAllStatusColl = Tac.Type( 'Routing::Pim::AllStatusColl' )
PimStatusColl = Tac.Type( 'Routing::Pim::StatusColl' )
PimGlobalStatusModeFilterSm = Tac.Type( 'Routing::Pim::GlobalStatusModeFilterSm' )
# Types mounted
PimGlobalStatus = Tac.Type( 'Routing::Pim::GlobalStatus' )
PimConfigColl = Tac.Type( 'Routing::Pim::ConfigColl' )
PimBsrStatusColl = Tac.Type( 'Routing::Pim::Bsr::StatusColl' )
PimBsrConfigColl = Tac.Type( 'Routing::Pim::Bsr::ConfigColl' )
PimRpCandidateStatusColl = Tac.Type( 'Routing::Pim::RpCandidateStatusColl' )
McastVrfConfig = Tac.Type( 'Routing::Multicast::Fib::VrfConfig' )

# Sets the helper inside the singleton entity
def registerMrouteCounterHelper( af, helper ):
   pimSnmpHelpers = Tac.singleton( 'Routing::Pim::Snmp::PimSnmpHelpers' )
   pimSnmpHelpers.mrouteCounterHelper[ af ] = helper

class PimSnmpPlugin:
   def __init__( self, ctx ):
      entityManager = ctx.entityManager()
      AddressFamily = Tac.Type( 'Arnet::AddressFamily' )
      mg = entityManager.mountGroup()

      snmpAgentStatus = mg.mount( 'snmp/status', 'Snmp::Status', 'r' )
      allIntfStatusDir = mg.mount( 'interface/status/all',
                                   'Interface::AllIntfStatusDir', 'r' )

      # The following dicts are to store the mounts that all have AF variants
      pimGlobalStatusByAf = {}
      pimConfigCollByAf = {}
      bsrStatusCollByAf = {}
      bsrConfigCollByAf = {}
      crpStatusCollByAf = {}
      staticRpStatusCollByAf = {}
      mcastFibVrfConfigByAf = {}
      addressFamilies = [ AddressFamily.ipv4, AddressFamily.ipv6 ]
      for af in addressFamilies:
         pimGlobalStatusByAf[ af ] = mg.mount( PimGlobalStatus.mountPath( af ),
                                               'Routing::Pim::GlobalStatus', 'r' )
         pimConfigCollByAf[ af ] = mg.mount( PimConfigColl.mountPath( af ),
                                             'Routing::Pim::ConfigColl', 'r' )
         bsrStatusCollByAf[ af ] = mg.mount( PimBsrStatusColl.mountPath( af ),
                                             'Routing::Pim::Bsr::StatusColl', 'r' )
         bsrConfigCollByAf[ af ] = mg.mount( PimBsrConfigColl.mountPath( af ),
                                             'Routing::Pim::Bsr::ConfigColl', 'r' )
         crpStatusCollByAf[ af ] = mg.mount(
               PimRpCandidateStatusColl.bsrMountPath( af ),
               'Routing::Pim::RpCandidateStatusColl', 'r' )
         staticRpStatusCollByAf[ af ] = mg.mount(
               PimRpCandidateStatusColl.staticMountPath( af, 'sparsemode' ),
               'Routing::Pim::RpCandidateStatusColl', 'r' )
         mcastFibVrfConfigByAf[ af ] = mg.mount( McastVrfConfig.mountPath( af ),
               'Routing::Multicast::Fib::VrfConfig', 'r' )

      sysdb = entityManager.root()
      snmpRoot = sysdb.parent[ 'snmp' ]
      pimRoot = snmpRoot.mkdir( 'pim' )
      pimMibRoot = pimRoot.mkdir( 'snmpPimMIB' )
      ifMib = snmpRoot[ 'intf' ][ 'ifmib' ]

      snmpAgentRoot = sysdb.parent[ 'Snmp' ]
      self.root_ = None

      # Initialize the singleton which will be used to store any helpers we need.
      # It's extremely important for us to setup our registration function as part of
      # our plugin initialization. This will be used to register the mroute counter
      # helper by another plugin.
      pimSnmpHelpers = Tac.singleton( 'Routing::Pim::Snmp::PimSnmpHelpers' )
      pimSnmpHelpers.mrouteCounterHelper.clear()
      ctx.helperRegistrarIs( 'MrouteCounterHelper', registerMrouteCounterHelper )

      def _finishMounts():
         self.root_ = snmpAgentRoot.newEntity( "Routing::Pim::SnmpAgent::Root",
                      "pimSnmpAgentRoot" )
         self.root_.pimMibRoot = pimMibRoot
         self.root_.em = entityManager.cEntityManager()
         self.root_.mibRegistrationHelper = ()
         self.root_.ifMib = ifMib
         self.root_.snmpAgentStatus = snmpAgentStatus
         self.root_.allIntfStatusDir = allIntfStatusDir

         for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
            afRoot = self.root_.afRoot.newMember( af, self.root_.pimMibRoot,
                  self.root_.mibRegistrationHelper, self.root_.em, self.root_.ifMib,
                  self.root_.snmpAgentStatus, self.root_.allIntfStatusDir,
                  pimGlobalStatusByAf[ af ], pimConfigCollByAf[ af ],
                  bsrStatusCollByAf[ af ], bsrConfigCollByAf[ af ],
                  crpStatusCollByAf[ af ], staticRpStatusCollByAf[ af ],
                  mcastFibVrfConfigByAf[ af ] )

            # Instantiate singletons and SMs, per AfRoot
            afRoot.pimAllStatusColl = ()
            afRoot.pimStatusColl = ()

            # TODO - We should convert PimGlobalStatusReaderSm to store a
            # TacSharedMem::EntityManager::Ptr instead of the deprecated
            # TacSmash::Mount
            smashMount = Tac.newInstance( 'TacSmash::Mount',
                                          entityManager.cEntityManager().sysname,
                                          entityManager.cEntityManager().isLocalEm )
            afRoot.pimGlobalStatusReaderSm = ( Tac.activityManager.clock,
                                               entityManager.cEntityManager(),
                                               smashMount )
            afRoot.pimGlobalStatusModeFilterSm = ()
            afRoot.rootSm = ()

      mg.close( lambda: None )
      ctx.callbackIs( _finishMounts )

@Plugins.plugin( requires=( 'snmp/intf', ) )
def Plugin( ctx ):
   PimSnmpPlugin( ctx )
   Tac.dlopen( "libPimSnmp.so" )
