# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import re

from SpaceMgmtLib import Utils
import Tac

class QuotaCmdException( Exception ):
   """
   Exception raised when there is an error executing a linux quota related command.

   The exception message respects the following format:
      Error <action>: '<cmd>':\n
      <output>

   <action>: Message describing the intended action.
   <cmd>: The actual command that was run.
   <output>: Output of the command which should explain the error, usually stderr.

   Notes
   -----
   This class is just a renaming of the base class Exception.
   """

def quotaCheck( mntPt, createFiles=False, remountRO=True ):
   """
   Run quotacheck on a filesystem.

   Notes
   -----
   Quota needs to be turned off before running quotacheck. See man quotacheck for
   details.

   Parameters
   ----------
   mntPt: str
      Absolute path of the filesystem mountpoint.
   createFiles: bool, optional
      If True, run quotacheck with --create-files. See man quotacheck for details.

   Raises
   -------
   QuotaCmdException
      If quotacheck had a non-zero exit code.
   """

   # Make sure the temporary quota file doesn't exists
   Utils.removeFile( os.path.join( mntPt, Utils.QUOTA_USER_NEW_FILE ) )

   try:
      opts = '-u'
      if createFiles:
         opts += 'c'
      if not remountRO:
         opts += 'm'
      cmd = [ 'quotacheck', opts, mntPt ]
      Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.CAPTURE )
   except Tac.SystemCommandError as e:
      raise QuotaCmdException( 'Error checking quota: %s:\n%s'
                               % ( e.args[ 0 ], e.output ) ) from e

def _checkQuotaState( mntPt, state ):
   """Helper function for parsing quotaon --print-state output."""

   # Ignore return code since the command exists with a non-zero status if the
   # filesystem is over quota.
   try:
      cmd = [ 'quotaon', '--print-state', mntPt ]
      out = Tac.run( cmd,
                     ignoreReturnCode=True,
                     stdout=Tac.CAPTURE,
                     stderr=Tac.CAPTURE )
   except Tac.SystemCommandError as e:
      raise QuotaCmdException( 'Error checking quota state: %s:\n%s'
                               % ( e.args[ 0 ], e.output ) ) from e

   pattern = r'^user quota on .* (.*) is (?P<status>on|off)$'
   match = re.search( pattern, out, flags=re.MULTILINE )
   if match is None:
      raise QuotaCmdException( "Error checking quota state: '%s':\n%s"
                               % ( ' '.join( cmd ), out ) )

   realState = match.group( 'status' ) == 'on'
   return state == realState

def checkQuotaOn( mntPt ):
   """
   Check if quota is enabled for a filesystem.

   Parameters
   ----------
   mntPt: str
      Absolute path of the filesystem mountpoint.

   Returns
   -------
   True if quota is on, False otherwise.

   Raises
   ------
   QuotaCmdException
      In case of error checking quota state.
   """

   return _checkQuotaState( mntPt, True )

def checkQuotaOff( mntPt ):
   """
   Check if quota is disabled for a filesystem.

   Parameters
   ----------
   mntPt: str
      Absolute path of the filesystem mountpoint.

   Returns
   -------
   True if quota is off, False otherwise.

   Raises
   ------
   QuotaCmdException
      In case of error checking quota state.
   """

   return _checkQuotaState( mntPt, False )

def _setQuotaState( mntPt, state ):
   """Helper function to enable/disable linux quotas using quotaon/quotaoff."""

   stateStr = 'on' if state else 'off'

   try:
      cmd = [ 'quota%s' % stateStr, '-u', mntPt ]
      out = Tac.run( cmd, asRoot=True, stdout=Tac.CAPTURE,
              stderr=Tac.CAPTURE ).split( '\n' )
   except Tac.SystemCommandError as e:
      raise QuotaCmdException( 'Error turning quota %s: %s:\n%s'
                               % ( stateStr, e.args[ 0 ], e.output ) ) from e

   # For the following error, quotaon doesn't have a non-zero exit status.
   # Check the output to make sure it succedeed.
   invalidFsMsgRegex = ( r'quota%s: Mountpoint \(or device\) %s not found '
                         'or has no quota enabled.' % ( stateStr, mntPt ) )

   matches = [ line for line in out if re.match( invalidFsMsgRegex, line ) ]
   if matches:
      raise QuotaCmdException( "Error turning quota %s: '%s'\n%s"
                               % ( stateStr,
                                   'sudo -E ' + ' '.join( cmd ), matches[ 0 ] ) )

def enableQuota( mntPt ):
   """
   Turn quota on for a filesystem.

   Parameters
   ----------
   mntPt: str
      Absolute path of the filesystem mountpoint.

   Raises
   ------
   QuotaCmdException
      In case of error running quotaon.
   """

   _setQuotaState( mntPt, True )

def disableQuota( mntPt ):
   """
   Turn quota off for a filesystem.

   Parameters
   ----------
   mntPt: str
      Absolute path of the filesystem mountpoint.

   Raises
   ------
   QuotaCmdException
      In case of error running quotaoff.
   """

   _setQuotaState( mntPt, False )

def _repquotaOutput( mntPt ):
   """Run repquota on the specified mountpoint and return its output."""

   try:
      cmd = [ 'repquota', '--verbose', '--raw-grace', '-u', mntPt ]
      out = Tac.run( cmd, asRoot=True, stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
   except Tac.SystemCommandError as e:
      raise QuotaCmdException( 'Error reporting quota: %s:\n%s'
                               % ( e.args[ 0 ], e.output ) ) from e
   return out

def _extractRepquotaInfoLines( output ):
   """Extract meaningful lines from repquota output."""

   lines = output.splitlines()

   # Line describing grace times
   graceTimesLine = next( line
                          for line in lines
                          if line.startswith( 'Block grace time:' ) )

   # Index of the separation line composed only of '-'
   linesIdxGen = ( idx
                   for idx, line in enumerate( lines )
                   if re.match( '-+$', line ) is not None )
   firstUserLine = 1 + next( linesIdxGen, len( lines ) )
   # Last line of user quotas is followed by a blank line
   lastUserLine = 1 + firstUserLine + next( idx for ( idx, line ) in
           enumerate( lines[ firstUserLine + 1 : ] ) if line == "" )

   # Keep only users quota report lines. Discard all the preambule as well as
   # the statistics lines at the end.
   return graceTimesLine, lines[ firstUserLine : lastUserLine ]

def _parseRepquotaGraceTimesLine( line ):
   """Parse the line from repquota output containing the grace times."""

   # Try to match the format hours:minutes
   grp = '(?P<%s>[0-9][0-9])'
   groups = ( grp % 'blkHr', grp % 'blkMin', grp % 'inodeHr', grp % 'inodeMin' )
   pattern = 'Block grace time: %s:%s; Inode grace time: %s:%s' % groups
   match = re.match( pattern, line )

   if match is not None:
      convertTime = lambda h, m: int( h ) * 3600 + int( m ) * 60
      return ( convertTime( match.group( 'blkHr' ), match.group( 'blkMin' ) ),
               convertTime( match.group( 'inodeHr' ), match.group( 'inodeMin' ) ) )

   # Try to match the format <num>days
   grp = '(?P<%s>[0-9]+)'
   groups = ( grp % 'blkDays', grp % 'inodeDays' )
   pattern = 'Block grace time: %sdays; Inode grace time: %sdays' % groups
   match = re.match( pattern, line )

   if match is not None:
      convertTime = lambda days: int( days ) * 86400
      return ( convertTime( match.group( 'blkDays' ) ),
               convertTime( match.group( 'inodeDays' ) ) )

   # Unable to parse the line
   raise QuotaCmdException( "Invalid grace times line format: '%s'" % line )

def _parseRepquotaInfoLine( line ):
   """Parse a line from repquota output containing quota stats for a user."""

   try:
      name, _, bUsed, bSoft, bHard, bGrace, fUsed, fSoft, fHard, fGrace = (
         line.split() )

      info = {
         'blkUsed' : int( bUsed ),
         'blkSoftLimit' : int( bSoft ),
         'blkHardLimit' : int( bHard ),
         'blkGraceLimit' : int( bGrace ),
         'fileUsed' : int( fUsed ),
         'fileSoftLimit' : int( fSoft ),
         'fileHardLimit' : int( fHard ),
         'fileGraceLimit' : int( fGrace )
      }

      return name, info
   except ValueError as e:
      raise QuotaCmdException( f"Invalid quota stats line format: '{line}'" ) from e

class QuotaStats:
   """
   Wraps all the data parsed from repquota output.

   Attributes
   ----------
   blkGraceTime: int
      Grace time in seconds for block quota.
   fileGraceTime:
      Grace time in seconds for inode quota.
   statsPerUser: dict of username: quotaStats
      username: str
         User name.
      quotaStats: dict of str: int
         'blkUsed': Number of blocks used, each block is 1KiB.
         'blkSoftLimit': Block quota soft limit in KiB.
         'blkHardLimit': Block hard limit in KiB.
         'blkGraceLimit': If the user  is in grace period, time in seconds
                          since epoch when his block grace time runs out (or
                          has run out). Is 0 when no grace time is in effect.
         'fileUsed': Number of inode used.
         'fileSoftLimit': Inode quota soft limit (Number of inodes.)
         'fileHardLimit': Inode quota hard limit (Number of inodes.)
         'fileGraceLimit': If the user  is in grace period, time in seconds
                           since epoch when his file grace time runs out (or
                           has run out). Is 0 when no grace time is in effect.
   """

   def __init__( self, blkGraceTime, fileGraceTime, statsPerUser ):
      self.blkGraceTime = blkGraceTime
      self.fileGraceTime = fileGraceTime
      self.statsPerUser = statsPerUser

   def __members( self ):
      return ( self.blkGraceTime, self.fileGraceTime, self.statsPerUser )

   def __hash__( self ):
      return hash( self.__members() )

   def __eq__( self, other ):
      if not isinstance( other, self.__class__ ):
         return NotImplemented

      return ( self.blkGraceTime == other.blkGraceTime
               and self.fileGraceTime == other.fileGraceTime
               and self.statsPerUser == other.statsPerUser )

def repquotaInfo( mntPt ):
   """
   Quota statistics inluding global grace times and per user statistics.

   Notes
   -----
   Information if obtained by parsing the output of repquota.

   Parameters
   ----------
   mntPt: str
      Absolute path of the filesystem mountpoint.

   Returns
   -------
   QuotaStats
      Information parsed from repquota output (See QuotaStats class).

   Raises
   ------
   QuotaCmdException
      If case of error running repquota or parsing its output.
   """

   output = _repquotaOutput( mntPt )
   graceLine, limitLines = _extractRepquotaInfoLines( output )

   blkGraceTime, fileGraceTime = _parseRepquotaGraceTimesLine( graceLine )

   statsPerUser = dict( map( _parseRepquotaInfoLine, limitLines ) )

   return QuotaStats( blkGraceTime, fileGraceTime, statsPerUser )

def setQuotaLimits( mntPt,
                    userName,
                    blockSoftLimit,
                    blockHardLimit,
                    inodeSoftLimit=0,
                    inodeHardLimit=0 ):
   """
   Set quotas for a filesystem.

   Notes
   -----
   Quotas are set using sequota.

   Parameters
   ----------
   mntPt: str
      Absolute path of the filesystem mountpoint.
   userName: str
      name of the user for which quotas should be set.
   blockSoftLimit: int
      Block quota soft limit in KiB.
   blockHardLimit: int
      Block quota hard limit in KiB.
   inodeSoftLimit: int, optional
      Inode quota soft limit (number of inodes).
   inodeHardLimit: int, optional
      Inode quota hard limit (number of inodes).

   Raises
   ------
   QuotaCmdException
      In case of error running setquota.
   """

   try:
      cmd = [
         'setquota',
         '-u',
         userName,
         str( blockSoftLimit ),
         str( blockHardLimit ),
         str( inodeSoftLimit ),
         str( inodeHardLimit ),
         mntPt
      ]
      Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.CAPTURE )
   except Tac.SystemCommandError as e:
      raise QuotaCmdException( 'Error setting quota limits: %s:\n%s'
                               % ( e.args[ 0 ], e.output ) ) from e

def setQuotaGraceTimes( mntPt, blockGrace, inodeGrace ):
   """
   Set quotas grace times of a filesystem.

   Notes
   -----
   Grace times are set using setquota, they are the same for all users.

   Parameters
   ----------
   mntPt: str
      Absolute path of the filesystem mountpoint.
   blockGrace: int
      Block grace time in seconds.
   inodeGraceTime: int
      Inode grace time in seconds.

   Raises
   ------
   QuotaCmdException
      In case of error running setquota.
   """

   try:
      cmd = [
         'setquota',
         '-t',
         '-u',
         str( blockGrace ),
         str( inodeGrace ),
         mntPt
      ]
      Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.CAPTURE )
   except Tac.SystemCommandError as e:
      raise QuotaCmdException( 'Error setting quota grace times: %s:\n%s'
                               % ( e.args[ 0 ], e.output ) ) from e

def setQuotaPct( quotaPct, userName, mntPt, totalSizeKiB=None ):
   """
   Set quota block limits for filesystem to a percentage of its total size.

   Notes
   -----
   This is a wrapper over setQuotaLimits to compute limits from a percentage.
   The limit will be applied to the soft and hard limit.

   Parameters
   ----------
   quotaPct: int
      constaints: value in the range [0-100]
      Percentage of the filesystem size the block quota limits should be set to.
   userName: str
      Name of the user block quota limits should be set for.
   mntPt: str
      Absolute path of the filesystem mountpoint.
   totalSizeKiB: int, optional
      Total size of the filesystem in KiB. The quota percentage is relative to this
      value. If not provided, it will be retrieved using fileSystemInfo function.

   Raises
   ------
   QuotaCmdException
      In case of error running setquota.
   AssertionError
      If fileSystemInfo failed to retrived the filesystem size.
   """

   assert Utils.validPct( quotaPct ), 'Invalid percentage value: %s' % quotaPct

   if totalSizeKiB is None:
      fsInfo = Utils.fileSystemInfo( mntPt )
      assert fsInfo is not None, 'Unable to get filesystem info for %s' % mntPt
      totalSizeKiB = fsInfo[ 'size' ]

   blkCount = Utils.pctToRatio( quotaPct, totalSizeKiB )
   setQuotaLimits( mntPt, userName, blkCount, blkCount )
