# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import errno
import numbers
import os
import shutil

import Tac

#####################################################################################
# Global constants
#####################################################################################

# Quota files created by linux at the root of the filesystem
QUOTA_USER_FILE = 'aquota.user'
QUOTA_USER_NEW_FILE = 'aquota.user.new'

# Size units in bytes
ONE_KIB = 1024
ONE_MIB = ONE_KIB * 1024
ONE_GIB = ONE_MIB * 1024
ONE_TIB = ONE_GIB * 1024

# Eos log directories paths
EOS_LOG_DIR_PATH = '/var/log'
EOS_AGENT_LOG_DIR_PATH = '/var/log/agent'
EOS_CORE_DIR_PATH = '/var/core'

#####################################################################################
# Utils
#####################################################################################

def validPct( pct ):
   """Return True if `pct` is a valid percentage, False otherwise."""

   return isinstance( pct, numbers.Integral ) and 0 <= pct <= 100

def ratioToPct( value, total ):
   """
   Convert a ratio to a percentage.

   Parameters
   ----------
   value: int
      Numerator of the ratio.
      constraints: `value` >= 0 and `value` <= `total`
   total: int
      Denominator of the ratio.
      constraints: `total` > 0

   Returns
   -------
   int
      Value between 0 and 100 representing the ration `value`/`total`
   """

   assert total > 0 and 0 <= value <= total
   return int( round( float( value ) / float( total ) * 100 ) )

def pctToRatio( pct, total ):
   """
   Convert a percentage to a ratio.

   Parameters
   ---------
   pct: int
      Percentage to convert, .i.e integer int the [0-100] range.
      constraints: integer value in the range [0-100]
   total: int
      Value to which the percentage is relative to, .i.e the 100% value.
      constrains: `total` > 0

   Returns
   -------
   int
      Numerator of the ratio corresponding to the percentage.
   """

   assert validPct( pct ) and total >= 0
   return int( round( float( total ) / 100 * float( pct ) ) )

def toKiB( size, units ):
   """
   Convert a size to kibibytes.

   Parameters
   ----------
   size: int
      The size value to convert.
   units: int
      The units of `size`, the value is the number of bytes in the unit:
      `size` in KiB: units == 1024
      `size` in GiB: units == 1024 * 1024
      etc...
      There is variables for common units (See Global constants section):
         * ONE_KIB
         * ONE_MIB
         * ONE_GIB
         * ONE_TIB

   Returns
   -------
   int
      Given size in kibibytes.
   """

   return size * ( units // ONE_KIB )

def scaleSizeFromKiB( sizeKiB ):
   """
   Convert a size in kibibytes to a human friendly string using the higher relevent
   unit possible.

   Parameters
   ----------
   sizeKiB: int
      Size to convert in units of kibibytes.

   Returns
   -------
   string
      Human readable string of the size.
      String format: two decimal digits float followed by a space and a unit.
   """

   units = [ 'KiB', 'MiB', 'GiB', 'TiB', 'PiB', 'EiB', 'ZiB', 'YiB' ]
   unitTransition = 1024 # 1 MiB = 1024 KiB, 1 GiB = 1024 MiB, etc ...

   q, r, divider, unit = sizeKiB, 0, 1, units[ 0 ]
   for idx, unit in enumerate( units ):
      divider = unitTransition ** idx
      q, r = divmod( sizeKiB, divider )
      if q < unitTransition:
         break

   value = q + round( r / float( divider ), 2 )
   return f'{value:.2f} {unit}'

#####################################################################################
# Files Utils
#####################################################################################

def createFile( path ):
   """
   Create a file at path.

   Notes
   -----
   The file is created by opening it with the O_CREATE option.

   Parameters
   ----------
   path: string
      Absolute path of the file to create.
   sync: bool, optional
      If True, perform an fsync on the file before returning.
   """

   with open( path, 'w+' ):
      pass

def removeFile( path, retSize=False ):
   """
   Delete a file. Eventually return the freed size.

   Notes
   -----
   Silence the OSError if the file does not exists.

   Parameters
   ----------
   path: string
      Absolute path of the file to delete.
   retSize: bool, optional
      If True, compute the size of the file before deleting it.

   Returns
   -------
   int
      Size of the file in kibibytes if `retSize` is True, 0 otherwise
   """

   size = spaceUsage( path ) if retSize else 0

   try:
      os.remove( path )
   except OSError as e:
      if e.errno != errno.ENOENT:
         raise

   return size

def removeDir( path, retSize=False ):
   """
   Delete a directory and all its content. Eventually return the freed size.

   Notes
   -----
   Silence the OSError if directory does not exists.

   Parameters
   ----------
   path: string
      Absolute path of the directory to delete.
   retSize: bool, optional
      If True, compute the size of the directory before deleting it.

   Returns
   -------
   int
      Size of the directory in kibibytes if `retSize` is True, 0 otherwise
   """

   size = spaceUsage( path ) if retSize else 0

   try:
      shutil.rmtree( path )
   except OSError as e:
      if e.errno != errno.ENOENT:
         raise

   return size

def removeFileOrDir( path, retSize=False ):
   """
   Delete a file or a directory. Eventually return the freed size.

   Notes
   -----
   Silence the OSError if the given path does not exist.

   Parameters
   ----------
   path:
      Absolute path of the file or directory to delete.
   retSize: bool, optional
      If True, compute the size of the file or directory before deleting it.

   Returns
   -------
   int
      Size of the file directory in kibibytes if `retSize` is True, 0 otherwise
   """

   if os.path.isfile( path ):
      return removeFile( path, retSize=retSize )
   else:
      return removeDir( path, retSize=retSize )

def sortDirByMtime( dirPath, reverse=False ):
   """
   List directory files sorted by time of last modification.

   Parameters
   ----------
   dirPath: string
      Absolute path of the directory.
   reverse: bool, optional
      If True sort the list with the newest modification time first. Sort by oldest
      modification time otherwise.

   Returns
   -------
   list of str
      Name of the directory's files sorted by modification time.
   """

   absPath = lambda filename: os.path.join( dirPath, filename )
   files = [
      ( os.stat( absPath( filename ) ).st_mtime, filename )
      for filename in os.listdir( dirPath )
      if os.path.isfile( absPath( filename ) )
   ]
   return [ filename for _, filename in sorted( files, reverse=reverse ) ]

def spaceUsage( path ):
   """
   Size of a file or directory tree in KiB.

   Notes
   -----
   The size is computed using du.

   Parameters
   ----------
   path: str
      Absolute path of the file or directory.

   Returns
   -------
   int
      Size of the file or directory tree in KiB.
      0 in case of error running du or parsing its output.

   """

   try:
      cmd = [ 'du', '--block-size=%s' % ONE_KIB, '-s', path ]
      out = Tac.run( cmd, asRoot=True, stdout=Tac.CAPTURE, stderr=Tac.DISCARD )
      return int( out.split()[ 0 ] )
   except ( Tac.SystemCommandError, ValueError ):
      return 0

#####################################################################################
# Filesystem and mountpoints
#####################################################################################

def fileSystemInfo( path ):
   """
   Information about a filesystem.

   Notes
   -----
   Information is gathered using df.

   Parameters
   ----------
   path: string
      An existing absolute path, the filesystem consulted is the one containing it.

   Returns
   -------
   dict of str: str or int
      'dev': Device name.
      'size': Filesystem size in KiB.
      'used': Used space in KiB.
      'available': Available space in KiB.
      'usedPct': Used percentage of total size.
      'mntPt': Mountpoint of the filesystem.

   None
      In case of error running df or parsing the output.
   """

   try:
      cmd = [ 'df', '-P', '--block-size=%d' % ONE_KIB, path ]
      out = Tac.run( cmd, asRoot=True, stdout=Tac.CAPTURE, stderr=Tac.DISCARD )
   except Tac.SystemCommandError:
      return None

   try:
      dev, size, used, available, usedPct, mntPt = out.splitlines()[ 1 ].split()

      # Allow creating small filesystems in a test, but for them to appear to
      # be large.
      sizeForTest = os.environ.get( 'SIZE_' + mntPt.replace( '/', '_' ) )
      if sizeForTest is not None:
         size = sizeForTest

      return {
         'dev' : dev,
         'size' : int( size ),
         'used' : int( used ),
         'available' : int( available ),
         'usedPct' : int( usedPct[ : -1 ] ),
         'mntPt' : mntPt
      }
   except ( IndexError, ValueError ):
      return None

def mountInfo( devName=None, mntPt=None ):
   """
   Information about a mount.

   Notes
   -----
   Information is gathered parsing /proc/mounts
   One of the two optional parameters must not be None. If both are provided,
   they have to match the same entry in /proc/mounts.

   Parameters
   ----------
   devName: str, optional
      Name of the mounted device.
   mntPt: str, optional
      Absolute path of the mountpoint.

   Returns
   -------
   dict of str: str or set
      'dev': Mounted device name.
      'mntPt': Mountpoint absolute path.
      'fsType': Mount type as reported in /proc/mounts.
      'mntOpts': set of str
                    Mount options used.

   None
      In case of error parsing /proc/mounts or if the given device and/or mountpoint
      was not found in /proc/mounts.
   """

   assert bool( devName ) or bool( mntPt ), 'No device or mounpoint given.'

   try:
      with open( '/proc/mounts' ) as f:
         lines = f.readlines()
   except OSError:
      return None

   devMatch = lambda _devName: devName is None or _devName == devName
   mntPtMatch = lambda _mntPt: mntPt is None or _mntPt == mntPt

   try:
      for line in lines:
         _devName, _mntPt, fsType, mntOpts, _, _ = line.split()
         if devMatch( _devName ) and mntPtMatch( _mntPt ):
            return {
               'dev' : _devName,
               'mntPt' : _mntPt,
               'fsType' : fsType,
               'mntOpts' : set( mntOpts.split( ',' ) )
            }
   except ValueError:
      return None

   return None
