# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import StageSysdbHelper
from Toggles.StageMgrToggleLib import toggleMacsecSsuEnabled

def registerAsuHitlessBootSand( entityManager ):
   sh = StageSysdbHelper.StageHelper( entityManager, 'boot' )
   sh.applicationIs( "Asu Hitless boot" )
   sh.stageTimeoutIsFatalIs( True )

   sh.registerStageDependency( "CriticalAgent", [] )
   sh.registerStage( "Aaa", "CriticalAgent", complete=True )
   sh.registerStage( "SuperServer", "CriticalAgent", complete=True )
   sh.registerStage( "FastClidHelper-Cli", "CriticalAgent", complete=True )
   sh.registerStage( "PhyEthtool", "CriticalAgent", complete=True )
   # MaintenanceMode is started to set qualPath for Rib/ArBgp
   sh.registerStage( "MaintenanceMode", "CriticalAgent", complete=True )
   if toggleMacsecSsuEnabled():
      # From Macsec shutdown to boot up we can only wait for 120 secs. Therefore
      # staring the agent early
      sh.registerStageDependency( "MacsecBootup", [] )
      sh.registerStage( "Macsec", "MacsecBootup", timeout=120 )

   # Adding 'InfrastructureAgent' stage before NetworkAgent stage. As of now
   # Ira is the only participating agent. This is required as Ira agent needs to
   # set the qualpath for the multi-routing agents are to run
   sh.registerStageDependency( "InfrastructureAgent", [] )
   sh.registerStage( "Ira", "InfrastructureAgent", timeout=60 )

   sh.registerStageDependency( "Fru-Plugins", [] )
   sh.registerStage( "Fru", "Fru-Plugins", timeout=60 )

   # Qos is not dependent on any stage however forwarding agents should start
   # only after Qos has completed publishing info into qos/status
   sh.registerStageDependency( "QosStatusUpdate", [] )
   sh.registerStage( "Qos", "QosStatusUpdate", timeout=60 )

   sh.registerStageDependency( "PrimaryAgent", [ "Fru-Plugins" ] )
   sh.registerStage( "Scd", "PrimaryAgent" )
   sh.registerStage( "ReloadCauseAgent", "PrimaryAgent", complete=True )
   sh.registerStage( "PLSystem", "PrimaryAgent" )
   sh.registerStage( "Smbus", "PrimaryAgent" )
   sh.registerStage( "PciBus", "PrimaryAgent", timeout=60 )
   sh.registerStage( "Si5327", "PrimaryAgent", complete=True )
   sh.registerStage( "KernelNetworkInfo", "PrimaryAgent", complete=True )

   sh.registerStageDependency( "SecondaryHardwareAgent",
                               [ "PrimaryAgent" ] )
   sh.registerStage( "Xcvr", "SecondaryHardwareAgent", complete=True )

   sh.registerStageDependency( "XcvrWarm", [ "SecondaryHardwareAgent" ] )
   sh.registerStage( "Xcvr", "XcvrWarm", timeout=60 )

   sh.registerStageDependency( "AvagoStatusUpdate", [ "XcvrWarm" ] )
   sh.registerStage( "Avago", "AvagoStatusUpdate", timeout=30,
                     completeNotRunnable=True )

   sh.registerStageDependency( "SandStage", [ "PrimaryAgent" ] )
   # We need Fru to finish before we can start Sand.  SandAgent will signal stage
   # done after it has created SandFap's runability objects.
   sh.registerStage( "Sand", "SandStage", timeout=30 )
   sh.registerStage( "Picasso", "SandStage", complete=True )

   # LagIntfStatusReady will need Sand agent to set extendedLagIdSupported
   # capabilities to decide whether to create EthLagIntfStatus or not for a
   # given EthLagIntfConfig.
   sh.registerStageDependency( "LagIntfStatusReady",  [ "SandStage" ] )
   sh.registerStage( "Lag", "LagIntfStatusReady", timeout=30 )

   # Ebra needs to start only after XcvrWarm, else due to race conditions
   # in Ebra writing to EthPhyIntfConfig and Xcvr agent reading that, it's
   # possible that some links may get disabled, leading to traffic loss.
   sh.registerStageDependency( "ErrdisableStatusUpdate",
                               [ "XcvrWarm", "LagIntfStatusReady" ] )
   sh.registerStage( "Ebra", "ErrdisableStatusUpdate", timeout=60 )

   sh.registerStageDependency( "MirroringHwConfigUpdate",
                               [ "ErrdisableStatusUpdate" ] )
   sh.registerStage( "Mirroring", "MirroringHwConfigUpdate", timeout=60 )


   # SandFapAgent needs SandAgent to finish its stage.
   # Need SandFap with complete so that agent gets started.
   # SandFap-FixedSystem name needed by Launcher.
   sh.registerStageDependency( "PostSandStage", [ "SandStage",
                                                  "QosStatusUpdate",
                                                  "ErrdisableStatusUpdate",
                                                  "AvagoStatusUpdate",
                                                  "XcvrWarm",
                                                  "MirroringHwConfigUpdate" ] )
   sh.registerStage( "SandFapV1", "PostSandStage", complete=True )
   sh.registerStage( "SandFapV2", "PostSandStage", complete=True )
   sh.registerStage( "SandFap-FixedSystem", "PostSandStage", timeout=180 )

   sh.registerStageDependency( "AvagoReconcile", [ "PostSandStage" ] )
   sh.registerStage( "Avago", "AvagoReconcile", timeout=60,
                     completeNotRunnable=True )

   sh.registerStageDependency( "CellStateUpdate", [ "PostSandStage" ] )
   sh.registerStage( "Sand", "CellStateUpdate", timeout=30 )

   sh.registerStageDependency( "LinkStatusUpdate", [ "PostSandStage" ] )
   sh.registerStage( "SandFap-FixedSystem", "LinkStatusUpdate", timeout=30 )

   # The XcvrAgent reacts to link changes and interface state changes.  When link
   # state and interface state is updated during hitless reload, the XcvrAgent must
   # not react because it may affect the link.  This 'XcvrConfigAllowed' stage will
   # be started (and auto completed) once the LinkStatusUpdate stage is complete.
   # Once 'XcvrConfigAllowed' is complete, the XcvrAgent is allowed to react to
   # interface state changes.
   sh.registerStageDependency( "XcvrConfigAllowed", [ "LinkStatusUpdate" ] )

   sh.registerStageDependency( "NetworkAgent", [ 'LinkStatusUpdate' ] )
   sh.registerStage( "Stp", "NetworkAgent", complete=True )
   sh.registerStage( "StpTopology", "NetworkAgent", complete=True )
   sh.registerStage( "TopoAgent", "NetworkAgent", complete=True )
   sh.registerStage( "Arp", "NetworkAgent", complete=True )
   sh.registerStage( "Acl", "NetworkAgent", complete=True )
   sh.registerStage( "PortSec", "NetworkAgent", complete=True )
   sh.registerStage( "KernelMfib", "NetworkAgent", complete=True )
   sh.registerStage( "BessMgr", "NetworkAgent", complete=True )
   sh.registerStage( "KernelFib", "NetworkAgent", complete=True )
   sh.registerStage( "Lag", "NetworkAgent" )
   sh.registerStage( "Rib", "NetworkAgent", timeout=60 )
   sh.registerStage( "Rib-vrf", "NetworkAgent", timeout=60 )
   sh.registerStage( "Ospf", "NetworkAgent", timeout=60 )
   sh.registerStage( "Ospf-vrf", "NetworkAgent", timeout=120 )
   sh.registerStage( "Ospf3", "NetworkAgent", timeout=60 )
   sh.registerStage( "Ospf3-vrf", "NetworkAgent", timeout=120 )
   sh.registerStage( "Isis", "NetworkAgent", timeout=60 )
   sh.registerStage( "Isis-vrf", "NetworkAgent", timeout=120 )
   sh.registerStage( "IpRib", "NetworkAgent", timeout=60 )
   sh.registerStage( "ConnectedRoute", "NetworkAgent", timeout=60 )
   sh.registerStage( "RouteInput", "NetworkAgent", timeout=60 )
   sh.registerStage( "Bgp", "NetworkAgent", timeout=60 )
   sh.registerStage( "Ipv6RouterAdvt", "NetworkAgent", complete=True )
   sh.registerStage( "Pim", "NetworkAgent", complete=True )
   sh.registerStage( "Pimsm", "NetworkAgent", complete=True )
   sh.registerStage( "PimReg", "NetworkAgent", complete=True )
   sh.registerStage( "PimBsr", "NetworkAgent", complete=True )
   sh.registerStage( "PimBidir", "NetworkAgent", complete=True )
   sh.registerStage( "PimBidirDf", "NetworkAgent", complete=True )
   sh.registerStage( "Igmp", "NetworkAgent", complete=True )
   sh.registerStage( "Msdp", "NetworkAgent", complete=True )
   sh.registerStage( "McastCommon", "NetworkAgent", complete=True )
   sh.registerStage( "IgmpSnooping", "NetworkAgent", complete=True )
   sh.registerStage( "DhcpRelay", "NetworkAgent", complete=True )
   sh.registerStage( "Fhrp", "NetworkAgent", complete=True )

   # Start Bfd agent before Lag enters LagStatusUpdate stage. Wait till
   # linkStatus is known before starting Bfd.
   sh.registerStageDependency( "BfdStatusUpdate", [ "NetworkAgent" ] )
   sh.registerStage( "Bfd", "BfdStatusUpdate", timeout=60 )

   sh.registerStageDependency( "PlatformBfdStatusUpdate", [ "BfdStatusUpdate" ] )
   sh.registerStage( "SandBfd", "PlatformBfdStatusUpdate", complete=True )

   # Lag status reconcile can only happen after Lag agent is started and
   # platform agent publish HW Lag groups under hardware/lag/input/config
   sh.registerStageDependency( "LagStatusReconcile", [ "NetworkAgent",
                                                       "LinkStatusUpdate",
                                                       "LagIntfStatusReady" ] )
   sh.registerStage( "Lag", "LagStatusReconcile", timeout=30 )

   sh.registerStageDependency( "SandLagStatusUpdate",
                               [ "LagStatusReconcile", ] )
   sh.registerStage( "SandLag", "SandLagStatusUpdate", timeout=30 )

   sh.registerStageDependency( "LagStatusUpdate", [
                               "SandLagStatusUpdate", "BfdStatusUpdate" ] )
   sh.registerStage( "Lag", "LagStatusUpdate", timeout=30 )

   # Start LacpTxAgent after Lag agent finishes its initialization to prevent
   # sending inconsistent LACP state to the partner
   sh.registerStageDependency( "LacpTxStatusUpdate",
                               [ "LagStatusUpdate" ] )
   sh.registerStage( "LacpTxAgent", "LacpTxStatusUpdate", complete=True,
                     timeout=60 )

   sh.registerStageDependency( "EbraStatusUpdate", [ "ErrdisableStatusUpdate" ] )
   sh.registerStage( "Ebra", "EbraStatusUpdate", timeout=120 )

   # Vxlan agent update
   # Vxlan agent needs to complete it's processing before SandTunnel, SandMact
   # and SandL3Unicast are started.
   sh.registerStageDependency( "VxlanConfigUpdate", [ "NetworkAgent" ] )
   sh.registerStage( "Vxlan", "VxlanConfigUpdate", timeout=60 )

   sh.registerStageDependency( "DynVlanReconcile", [ "VxlanConfigUpdate" ] )
   sh.registerStage( "Vxlan", "DynVlanReconcile", enableAutoComplete=True,
                        autoCompleteTimeout=60 )
   sh.registerStageDependency( "VxlanAsuSync", [ "DynVlanReconcile" ] )
   sh.registerStage( "Vxlan", "VxlanAsuSync", timeout=60 )
   # Start VxanSwFwd agent
   sh.registerStageDependency( "VxlanSwFwdStatusUpdate", [ "VxlanConfigUpdate" ] )
   sh.registerStage( "VxlanSwFwd", "VxlanSwFwdStatusUpdate", complete=True )

   # Start L2Rib agent before SandMact. This is so that L2Rib publishes the
   # BUM floodlist before SandMact processes it.
   sh.registerStageDependency( "L2RibReconcile", [ "VxlanSwFwdStatusUpdate" ] )
   sh.registerStage( "L2Rib", "L2RibReconcile", timeout=60 )

   # Placeholder for Sand Feature agents.  Boring names for now...
   #
   # Mark them complete until the agents are modified.
   # When modified, change "completed=True" to a "timeout=[value]".
   # Sand ASU2 uses a 30 second timer while waiting for the agent to be ready.
   # Use a timeout greater than 30 seconds. 60 second seems be a widely used
   # convention in this file, so hey, why not?

   sh.registerStageDependency( "Sand_1", [ "PostSandStage", "LagStatusUpdate",
                                           "L2RibReconcile" ] )
   sh.registerStage( "SandTunnel", "Sand_1", timeout=30 )

   sh.registerStageDependency( "Sand_2", [ "Sand_1" ] )
   sh.registerStage( "SandTopo", "Sand_2", timeout=30 )
   sh.registerStage( "SandMact", "Sand_2", timeout=30 )
   sh.registerStage( "SandDanz", "Sand_2", complete=True )
   sh.registerStage( "SandIpsec", "Sand_2", complete=True )

   sh.registerStageDependency( "Sand_3", [ "Sand_2" ] )
   sh.registerStage( "AleL3Agent-primary-SandL3Unicast", "Sand_3", complete=True )
   sh.registerStage( "SandL3Unicast", "Sand_3", timeout=60 )
   sh.registerStage( "SandMcast", "Sand_3", timeout=30 )

   # SandAcl has a dependency on SandL3Unicast for programming egress ipv4/ipv6 acls
   sh.registerStageDependency( "SandAclReconcile", [ "Sand_3" ] )
   sh.registerStage( "SandAcl", "SandAclReconcile", timeout=30 )

   sh.registerStageDependency( "Sand_5", [ "SandAclReconcile" ] )
   sh.registerStage( "SandTcam", "Sand_5", timeout=30 )

   # VlanTopo depends on TopoConfig creation for BridgingTopoStatusUpdate
   sh.registerStageDependency( "BridgingTopoConfigCreate",
                               [ "NetworkAgent", "Sand_2" ] )
   sh.registerStage( "StpTopology", "BridgingTopoConfigCreate", timeout=60 )

   sh.registerStageDependency( "BridgingTopoInstUpdate",
                               [ "BridgingTopoConfigCreate" ] )
   sh.registerStage( "SandTopo", "BridgingTopoInstUpdate", timeout=60 )

   sh.registerStageDependency( "BridgingTopoStatusUpdate",
                               [ "EbraStatusUpdate",
                                 "BridgingTopoInstUpdate" ] )
   sh.registerStage( "TopoAgent", "BridgingTopoStatusUpdate", timeout=60 )

   sh.registerStageDependency( "BridgingTopoConfigUpdate",
                               [ "BridgingTopoStatusUpdate" ] )
   sh.registerStage( "StpTopology", "BridgingTopoConfigUpdate", timeout=60 )

   sh.registerStageDependency( "L2MacAddrSmReconcile",
                               [ "EbraStatusUpdate",
                                 "LinkStatusUpdate",
                                 "LagStatusUpdate", "BridgingTopoConfigUpdate" ] )

   sh.registerStageDependency( "VlanIntfOperStatusUpdate",
                               [ "BridgingTopoConfigUpdate", "DynVlanReconcile" ] )
   sh.registerStage( "Ebra", "VlanIntfOperStatusUpdate", timeout=30 )

   sh.registerStageDependency( "EbraSubIntfUp",
                               [ "VlanIntfOperStatusUpdate" ] )
   sh.registerStage( "Ebra", "EbraSubIntfUp", timeout=60 )

   sh.registerStageDependency( "VrfStatusUpdate",
                               [ "EbraStatusUpdate", "EbraSubIntfUp", "Sand_3" ] )
   sh.registerStage( "Ira", "VrfStatusUpdate", timeout=60 )

   sh.registerStageDependency( "VrfRoutingInfoUpdate", [ "VrfStatusUpdate" ] )
   sh.registerStage( "IpRib", "VrfRoutingInfoUpdate", timeout=60 )

   sh.registerStageDependency( "IpIntfStatusUpdate", [ "DynVlanReconcile",
                                                       "VrfRoutingInfoUpdate" ] )
   sh.registerStage( "Ira", "IpIntfStatusUpdate", timeout=60 )
   sh.registerStage( "Bgp", "IpIntfStatusUpdate", timeout=60 )

   sh.registerStageDependency( "VrfIdAllocationUpdate",
                               [ "EbraStatusUpdate", "IpIntfStatusUpdate" ] )
   sh.registerStage( "SandL3Unicast", "VrfIdAllocationUpdate", complete=True )

   sh.registerStageDependency( "ArpStatusUpdate", [ "EbraStatusUpdate",
                                                    "IpIntfStatusUpdate" ] )
   sh.registerStage( "Arp", "ArpStatusUpdate", timeout=60 )

   sh.registerStageDependency( "FibRouteUpdate", [ "ArpStatusUpdate" ] )
   sh.registerStage( "Rib", "FibRouteUpdate", timeout=60 )
   sh.registerStage( "Rib-vrf", "FibRouteUpdate", timeout=60 )
   sh.registerStage( "Bgp", "FibRouteUpdate", timeout=60 )
   sh.registerStage( "ConnectedRoute", "FibRouteUpdate", timeout=60 )
   sh.registerStage( "RouteInput", "FibRouteUpdate", timeout=60 )
   sh.registerStage( "Isis", "FibRouteUpdate", timeout=60 )
   sh.registerStage( "Isis-vrf", "FibRouteUpdate", timeout=60 )
   sh.registerStage( "Ospf", "FibRouteUpdate", timeout=60 )
   sh.registerStage( "Ospf-vrf", "FibRouteUpdate", timeout=60 )
   sh.registerStage( "Ospf3", "FibRouteUpdate", timeout=60 )
   sh.registerStage( "Ospf3-vrf", "FibRouteUpdate", timeout=60 )

   sh.registerStageDependency( "L3FibReady", [ "FibRouteUpdate" ] )
   sh.registerStage( "SandL3Unicast", "L3FibReady", timeout=600 )

   # Register only Bgp agent for L2RibUpdate and L2RibReady stage for now.
   # We do not support Sand Evpn ASU, and L2RibReconcile is started
   # way early in the progression. These stages are required for Bgp agent to
   # complete their internal processing and do not serve any other purpose for now.
   sh.registerStageDependency( "L2RibUpdate", [ "L3FibReady" ] )
   sh.registerStage( "Bgp", "L2RibUpdate", timeout=120 )
   sh.registerStageDependency( "L2RibReady", [ "L2RibReconcile", "L2RibUpdate" ] )
   sh.registerStage( "Bgp", "L2RibReady", timeout=120 )

   sh.registerStageDependency( "L3UcastSmsReconcile",
                               [ "ArpStatusUpdate", "L3FibReady",
                                 "L2MacAddrSmReconcile", "VrfIdAllocationUpdate" ] )
   sh.registerStage( "SandL3Unicast", "L3UcastSmsReconcile", complete=True )

   sh.registerStageDependency( "DownLinksUpdate",
                               [ "BridgingTopoConfigUpdate", "LinkStatusUpdate",
                                 "L3UcastSmsReconcile" ] )
   sh.registerStage( "Ebra", "DownLinksUpdate", timeout=30 )

   sh.registerStageDependency( "ShadowTableReconcile",
                               [ "DownLinksUpdate", "Sand_3", "Sand_2", "Sand_5",
                                 "L3FibReady" ] )
   # ShadowTableReconcile is used by agents to explicitly verify that their
   # shadow tables are ready for download into hardware.
   sh.registerStage( "SandFap", "ShadowTableReconcile", complete=True )
   sh.registerStage( "SandFap-FixedSystem", "ShadowTableReconcile", complete=True )
   # SandL3Unicast can take a long time to process routes after AleSms are started
   # during L3FibReady
   sh.registerStage( "SandL3Unicast", "ShadowTableReconcile", timeout=300 )
   sh.registerStage( "SandMact", "ShadowTableReconcile", timeout=30 )
   sh.registerStage( "SandMcast", "ShadowTableReconcile", timeout=30 )
   sh.registerStage( "SandTopo", "ShadowTableReconcile", timeout=30 )
   sh.registerStage( "SandAcl", "ShadowTableReconcile", timeout=30 )
   sh.registerStage( "SandTunnel", "ShadowTableReconcile", timeout=30 )

   sh.registerStageDependency( "QuiesceTraffic", [ "ShadowTableReconcile" ] )
   sh.registerStage( "SandFap", "QuiesceTraffic", complete=True )
   sh.registerStage( "SandFap-FixedSystem", "QuiesceTraffic", timeout=30 )

   sh.registerStageDependency( "ResetTcams", [ "QuiesceTraffic" ] )
   sh.registerStage( "SandTcam", "ResetTcams", timeout=30 )

   sh.registerStageDependency( "BulkDownload", [ "ResetTcams" ] )
   sh.registerStage( "SandFap", "BulkDownload", complete=True )
   sh.registerStage( "SandFap-FixedSystem", "BulkDownload", timeout=30 )
   sh.registerStage( "SandL3Unicast", "BulkDownload", timeout=30 )
   sh.registerStage( "SandMact", "BulkDownload", timeout=30 )
   sh.registerStage( "SandMcast", "BulkDownload", timeout=30 )
   sh.registerStage( "SandTopo", "BulkDownload", timeout=30 )
   sh.registerStage( "SandLag", "BulkDownload", timeout=30 )
   sh.registerStage( "SandAcl", "BulkDownload", timeout=30 )
   sh.registerStage( "SandTunnel", "BulkDownload", timeout=30 )

   sh.registerStageDependency( "ResumeTraffic", [ "BulkDownload" ] )
   sh.registerStage( "SandFap", "ResumeTraffic", complete=True )
   sh.registerStage( "SandFap-FixedSystem", "ResumeTraffic", timeout=30 )

   bootSanityCheckDeps = []
   bootSanityCheckDeps += [ "DownLinksUpdate" ]
   bootSanityCheckDeps += [ "AvagoReconcile" ]
   bootSanityCheckDeps += [ "SandAclReconcile" ]
   bootSanityCheckDeps += [ "ResumeTraffic" ]
   bootSanityCheckDeps += [ "LacpTxStatusUpdate" ]
   bootSanityCheckDeps += [ "PlatformBfdStatusUpdate" ]
   bootSanityCheckDeps += [ "CriticalAgent" ]
   bootSanityCheckDeps += [ "L2RibUpdate" ]
   bootSanityCheckDeps += [ "CellStateUpdate" ]
   bootSanityCheckDeps += [ "L2RibReady" ]
   bootSanityCheckDeps += [ "VxlanAsuSync" ]
   bootSanityCheckDeps += [ "XcvrConfigAllowed" ]
   bootSanityCheckDeps += [ "InfrastructureAgent" ]
   sh.registerStageDependency( "BootSanityCheck", bootSanityCheckDeps )
   sh.registerStage( "Launcher", "BootSanityCheck", timeout=120 )
