#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import QuickTrace
import Cell
from EntityManager import sysdbPyServerInetPort
from PyClient import PyClient
from PyClientBase import RpcError

traceNormal = Tracing.trace7
traceFunc = Tracing.trace8

qtError = QuickTrace.trace1
qtFunc = QuickTrace.trace8

def peerRedSupPyClient( sysname='ar' ):
   traceNormal( 'Creating a new pyclient connection' )
   # pylint: disable-next=consider-using-f-string
   peerIp = '127.1.0.%d' % Cell.peerCellId()
   try:
      port = sysdbPyServerInetPort()
      pc = PyClient( sysname, 'ElectionMgr', host=peerIp, connectTimeout=10,
                     reconnect=False, port=port )
      traceNormal( 'Successfully created new pyClient connection' )
      return pc
   except Exception as e:  # pylint: disable-msg=W0703
      traceNormal( 'Failed to establish pyclient due to exception: ',
                   e )
      return None


class RprStandbyPoller:

   def __init__( self, rprStandbyStatusTracker ):
      self.rprStandbyStatusTracker = rprStandbyStatusTracker
      self.standbySyncStats = Tac.Value( "PhyEthtool::StandbySyncStats" )
      self.legacyEpisSync = None
      self.archerEpisSync = None

      # Establish frequency at which we want to poll the standby status
      self.pollInterval = self.rprStandbyStatusTracker.pollInterval
      self.standbySyncStats.successfulSyncCount = 0
      self.pc = None
      self.clockNotifiee = Tac.ClockNotifiee(
            handler=self.doScanMgmtStatus, timeMin=Tac.now() )

   def getPyClient( self ):
      if self.pc is not None:
         try:
            # Verify we have a valid pyclient connection
            if self.pc.eval( 'True' ):
               return self.pc
         except RpcError as e:
            traceNormal( 'Remote server threw an error: ', str( e ) )
            self.pc = None
      if self.pc is None:
         self.pc = peerRedSupPyClient()
      return self.pc

   def getSwVersion( self, standbySysdbRoot ):
      traceNormal( 'Grabbing the SwVersion' )
      # pylint: disable-next=consider-using-f-string
      path = 'cell/%s/Sysdb/localStatus' % Cell.peerCellId()
      try:
         swVersion = standbySysdbRoot.entity[ path ].swVersion
         return swVersion
      except Exception as e:  # pylint: disable-msg=W0703
         # If unable to grab swVersion during restart or shutdown,
         # return swVersion as None
         traceNormal( 'Failed to access swVersion due to: ',
                      e )
         return None

   def doScanMgmtStatus( self ):
      # Handle any unforseen exceptions gracefully with try/except,
      # but most specific places where we sync have specific exceptions instead
      # of a catch-all
      try:
         self.standbySyncStats.lastPollEvent = Tac.now()
         # If pyclient conn is none, establish new connection
         # If connection is not none, use the etablished connection instead of
         # re-creating a new one
         # If we are unable to establish a connection, re-arm the poller and exit
         self.pc = self.getPyClient()
         if self.pc is None:
            self.clockNotifiee.timeMin = Tac.now() + self.pollInterval
            return

         # If both self.standbyEpisDir  and self.archerStandbyEpisDir are none,
         # immediately return after re-arming the poller trigger
         traceNormal( 'legacyEpisSync is: ', self.legacyEpisSync,
                      'archerEpisSync is', self.archerEpisSync )
         if self.legacyEpisSync or self.archerEpisSync:
            root = self.pc.root()
            standbySysdbRoot = root[ 'ar' ][ 'Sysdb' ]

            swVersion = self.getSwVersion( standbySysdbRoot )

            if self.legacyEpisSync:
               traceNormal( 'Syncing legacy EpisDir' )
               self.legacyEpisSync.sync( standbySysdbRoot, swVersion )
            if self.archerEpisSync:
               traceNormal( 'Syncing archer EpisDir ' )
               self.archerEpisSync.sync( standbySysdbRoot, swVersion )
            self.standbySyncStats.lastSuccessfulSync = Tac.now()
            self.standbySyncStats.successfulSyncCount += 1
      except Exception as e:  # pylint: disable-msg=W0703
         traceNormal( 'Failed to sync due to unexpected ', type( e ), ' : ', e )

      self.rprStandbyStatusTracker.stats = self.standbySyncStats
      self.clockNotifiee.timeMin = Tac.now() + self.pollInterval

   def standbyEpisDirIs( self, standbyEpisDir ):
      peerCell = Cell.peerCellId()
      traceNormal( 'standbyEpisDir is ', standbyEpisDir )
      if standbyEpisDir:
         # pylint: disable-next=consider-using-f-string
         path = 'interface/status/eth/phy/slice/%s' % peerCell
         self.legacyEpisSync = EpisSync( standbyEpisDir, path )
      else:
         self.legacyEpisSync = None

   def archerStandbyEpisDirIs( self, archerStandbyEpisDir ):
      peerCell = Cell.peerCellId()
      traceNormal( 'archerStandbyEpisDir is ', archerStandbyEpisDir )
      if archerStandbyEpisDir:
         # pylint: disable-next=consider-using-f-string
         path = 'interface/archer/status/eth/phy/slice/%s/PhyEthtool' % peerCell
         self.archerEpisSync = EpisSync( archerStandbyEpisDir, path )
      else:
         self.archerEpisSync = None

MgmtIntfId = Tac.Type( "Arnet::MgmtIntfId" )
class EpisSync:
   ExcludedAttributesList = [ "name", "fullName", "parent", "parentAttrName",
                              "entity", "intfId", "intfConfig",
                              "defaultConfig", "genId", "writerStatusDir",
                              "ethCounter", "counterHolder" ]

   def __init__( self, episDir, episDirPath ):
      self.episDir = episDir
      self.episDirPath = episDirPath
      self.compatibleAttrList = []
      self.cachedStandbySwVersion = None
      self.compatibleAttrListInitialized = False

   def sync( self, standbySysdbRoot, standbySwVersion ):
      if standbySwVersion is None:
         return
      peerEpisDir = standbySysdbRoot.entity[ self.episDirPath ]
      standbyMgmtIntfs = [ intfId for intfId in peerEpisDir.intfStatus
                           if MgmtIntfId.isMgmtIntfId( intfId ) ]

      for mgmtIntfId in standbyMgmtIntfs:
         peerEpis = peerEpisDir.intfStatus.get( mgmtIntfId )
         localEpis = self.episDir.intfStatus.get( mgmtIntfId )

         if peerEpis and localEpis:
            traceNormal( 'Starting to sync EPIS for ', mgmtIntfId )
            # Only update attribute list once for this for loop when it's
            # creating the list for the first time or SWI version are
            # different
            if self.cachedStandbySwVersion != standbySwVersion:
               self.updateCompatibleAttributes( peerEpis, localEpis )
               self.cachedStandbySwVersion = standbySwVersion
            for attrName in self.compatibleAttrList:
               try:
                  standbyAttr = getattr( peerEpis, attrName )
                  setattr( localEpis, attrName, standbyAttr )
                  traceNormal( 'Successfuly synced ', attrName, 'with ',
                               standbyAttr )
               except ( TypeError, NotImplementedError ) as e:
                  traceNormal( 'Failed to sync attr:',
                               attrName, 'exception: ', e )

   def updateCompatibleAttributes( self, standbyMgmtIntfEntity,
                                   activeMgmtIntfEntity ):
      traceNormal( 'Updating compatible attributes' )
      compatibleAttrList = []
      for attr in standbyMgmtIntfEntity.attributes:
         if ( attr not in EpisSync.ExcludedAttributesList and
              attr in activeMgmtIntfEntity.attributes ):
            traceNormal( 'Adding ', attr,
                         ' to compatible list ' )
            compatibleAttrList.append( attr )
      self.compatibleAttrList = compatibleAttrList
      self.compatibleAttrListInitialized = True

class AllEpisDirReactor( Tac.Notifiee ):
   """ Reactor that reacts to interface slices being added to
   /interface/status/eth/phy/slice. If standby slice is modified,
   create the standbyEpisReactor which reacts to intfIds added to that slice"""
   notifierTypeName = "Tac::Dir"

   def __init__( self, allEpisDir, parent, peerSliceName, archer ):
      traceNormal( "Calling __init__ for AllEpisDirReactor " )
      self.allEpisDir_ = allEpisDir
      self.parent_ = parent
      self.peerSliceName_ = peerSliceName
      self.archerAgentReactor_ = None
      self.standbyEpisReactor_ = None
      self.archer_ = archer
      Tac.Notifiee.__init__( self, allEpisDir )

      traceNormal( "Check if peerSlice is an existing entry in AllEpisDir " )
      self.handleEntity( self.peerSliceName_ )

   @Tac.handler( "entityPtr" )
   def handleEntity( self, sliceName ):
      traceNormal( "Calling handle entity on slice ", sliceName )
      if sliceName == self.peerSliceName_:
         sliceEpisDir = self.allEpisDir_.get( sliceName )
         if sliceEpisDir:
            if self.archer_:
               self.archerAgentReactor_ = \
                        ArcherAgentDirReactor( sliceEpisDir, self.parent_ )
            else:
               traceNormal( "peerSlice is in the allEpisDir, "
                            "creating standbyEpisReactor" )
               self.standbyEpisReactor_ = \
                              StandbyEpisDirReactor( sliceEpisDir,
                                                     self.parent_ )
         else:
            # Peer slice was removed, delete standbyEpisDir reactor
            traceNormal( 'Deleting archer StandbyEpisDirReactor' )
            self.archerAgentReactor_ = None
            self.standbyEpisReactor_ = None

class ArcherAgentDirReactor( Tac.Notifiee ):
   """ Reactor that reacts to agent name "PhyEthtool" being added to
   /interface/status/eth/phy/slice/<sliceName>.Creates StandbyEpisDirReactor
   for the archer path"""

   notifierTypeName = "Tac::Dir"

   def __init__( self, sliceEpisDir, parent ):
      traceNormal( "Calling __init__ for ArcherAgentDirReactor " )
      self.sliceEpisDir_ = sliceEpisDir
      self.parent_ = parent
      self.standbyEpisReactor_ = None
      Tac.Notifiee.__init__( self, sliceEpisDir )
      self.agentName_ = "PhyEthtool"

      self.handleEntity( self.agentName_ )

   @Tac.handler( "entityPtr" )
   def handleEntity( self, agentName ):
      if agentName == self.agentName_:
         episDir = self.sliceEpisDir_.get( agentName )
         if episDir:
            traceNormal( 'Creating archer standbyEpisDirReactor' )
            self.standbyEpisReactor_ = \
               StandbyEpisDirReactor( episDir, self.parent_ )
         else:
            traceNormal( 'Peer slice was removed, ',
                         'deleting legacy StandbyEpisDirReactor ',
                         'and ArcherAgentReactor.' )
            self.standbyEpisReactor_ = None


class StandbyEpisDirReactor( Tac.Notifiee ):
   """ Reactor that tracks the creation of EthPhyIntfStatus instances in an
   EthPhyIntfStatusDir on the active supersivor for a standby supervisor"""

   notifierTypeName = "Interface::EthPhyIntfStatusDir"

   def __init__( self, standbyEpisDir, parent ):
      traceNormal( "Calling __init__ for StandbyEpisDirReactor " )
      self.parent_ = parent
      Tac.Notifiee.__init__( self, standbyEpisDir )
      self.handleStandbyEpisDir( None )

   @Tac.handler( 'intfStatus' )
   def handleStandbyEpisDir( self, intfId ):
      traceNormal( " intfStatus modified, calling handleStandbyEpisDir" )
      self.parent_.maybePollStandby()

class RedSupStatusReactor( Tac.Notifiee ):
   """ Reactor that tracks the redundancyStatus mode and protocol,
   used in PhyEthtool and differs from handleRedMode because is not using
   MultiSliceEthPhyIntfCreator class at all"""
   notifierTypeName = "Redundancy::RedundancyStatus"

   def __init__( self, redundancyStatus, parent ):
      traceNormal( "Calling __init__ for RedSupStatusReactor" )
      self.parent_ = parent
      Tac.Notifiee.__init__( self, redundancyStatus )
      self.handleRedundancyStatusMode( )
      self.handleRedundancyStatusProtocol( )

   @Tac.handler( 'mode' )
   def handleRedundancyStatusMode( self ):
      traceNormal( "redundancyStatus mode changed, "
                   "calling handleRedundancyStatusMode" )
      self.parent_.maybePollStandby()

   @Tac.handler( 'protocol' )
   def handleRedundancyStatusProtocol( self ):
      traceNormal( "Poll: redundancyStatus protocol changed, calling "
                   "handleRedundancyStatusProtocol" )
      self.parent_.maybePollStandby()

class RprStandbyMgmtStatusSync:
   def __init__( self, peerSliceName, redundancyStatus, allEpisDir, allArcherEpisDir,
                 rprStandbyStatusTracker ):
      traceNormal( "Calling __init__ for RprStandbyMgmtStatusSync" )
      self.peerSliceName_ = peerSliceName
      self.standbyPoller_ = None
      self.allEpisDir_ = allEpisDir
      self.allArcherEpisDir_ = allArcherEpisDir
      self.redundancyStatus_ = redundancyStatus
      self.rprStandbyStatusTracker_ = rprStandbyStatusTracker
      self.allEpisDirReactor_ = AllEpisDirReactor(
         self.allEpisDir_, self, self.peerSliceName_, False )
      self.allArcherEpisDirReactor_ = AllEpisDirReactor(
         self.allArcherEpisDir_, self, self.peerSliceName_, True )
      self.redSupStatusReactor_ = RedSupStatusReactor(
         self.redundancyStatus_, self )

   def maybePollStandby( self ):
      traceNormal( "Calling maybeCreateStandbyPoller" )
      episDir = self.allEpisDir_.get( self.peerSliceName_ )
      archerEpisSliceDir = self.allArcherEpisDir_.get(
         self.peerSliceName_ )
      archerEpisDir = archerEpisSliceDir.get( 'PhyEthtool', None ) \
                      if archerEpisSliceDir else None
      traceNormal( "StandbyEpisDir: ", episDir,
                   " archerStandbyEpisDir: ", archerEpisDir )
      traceNormal( "redundancyStatus mode: ",
                   self.redundancyStatus_.mode, "redundancyStatus protocol: ",
                   self.redundancyStatus_.protocol )
      if ( ( self.redundancyStatus_.mode == 'active' and
           self.redundancyStatus_.protocol == 'rpr' ) and
           ( ( episDir and episDir.intfStatus ) or
             ( archerEpisDir and archerEpisDir.intfStatus ) ) ) :
         # check length of episDir intfStatus or length of archer intfStatus
         if self.standbyPoller_ is None:
            traceNormal( "Creating standbyPoller" )
            self.standbyPoller_ = RprStandbyPoller( self.rprStandbyStatusTracker_ )
         traceNormal( "Updating standbyPoller's EthPhyIntfStatus directories" )
         self.standbyPoller_.standbyEpisDirIs( episDir )
         self.standbyPoller_.archerStandbyEpisDirIs( archerEpisDir )
      else:
         if self.standbyPoller_:
            traceNormal( "Deleting standbyPoller" )
            self.standbyPoller_ = None
