# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import shutil

snapshotFsLocation = ".user_recovery"
tmpSnapshotFsLocation = f"{snapshotFsLocation}_tmp"
tmpSnapshotFsLocation2 = f"{snapshotFsLocation}_tmp2"
snapshotFsName = "snapshot:"

def filesystemType( path ):
   path = os.path.realpath( os.path.abspath( path ) )
   depth = 0
   fsType = None

   with open( "/proc/mounts" ) as procMounts:
      for line in procMounts:
         _, mount, fsType_, *_ = line.split()
         mount = os.path.realpath( mount )

         if ( not os.path.isdir( mount ) or
              os.path.commonpath( [ mount, path ] ) != mount ):
            continue

         depth_ = 0 if mount == "/" else len( mount.split( os.path.sep ) ) - 1
         if depth_ >= depth:
            depth = depth_
            fsType = fsType_
   return fsType

def copyTree( src, dst, dirsExistOk=False ):
   """
   A simple version of shutil.copytree that doesn't preserve permissions. Makes deep
   copies of symlinks.

   It's possible not to preserve permissions for files by passing a custom
   copy_function to shutil.copytree, but permissions for directories will still be
   preserved.
   """

   with os.scandir( src ) as it:
      entries = list( it )

   os.makedirs( dst, exist_ok=dirsExistOk )
   errors = []

   for srcEntry in entries:
      srcName = os.path.join( src, srcEntry.name )
      dstName = os.path.join( dst, srcEntry.name )
      try:
         if srcEntry.is_dir():
            copyTree( srcEntry, dstName, dirsExistOk=dirsExistOk )
         else:
            shutil.copyfile( srcEntry, dstName )
      # catch the Error from the recursive copyTree so that we can continue with
      # other files
      except shutil.Error as err:
         errors.extend( err.args[ 0 ] )
      except OSError as why:
         errors.append( ( srcName, dstName, str( why ) ) )
   if errors:
      raise shutil.Error( errors )

def copyTreeMaybePreservePerms( src, dst, dirsExistOk=False ):
   """
   Copy files and directories recursively, preserving permissions if the destination
   filesystem supports them (not vfat).

   The vfat filesystem does not support unix-style permissions. Instead, permissions
   for all files under the mount point are determined at the time of mount. If you
   attempt to change permissions in a vfat filesystem (os.chmod or shutil.copytree)
   an OSError is thrown.

   In princlple, one could do something like

   try:
      shutil.copytree( src, dst )
   except OSError as e:
      pass

   However, due to the shutil.copytree implemenation, it is difficult to distinguish
   legit OSErrors from exceptions due to vfat.

   If dst is not a vfat mount, it must not have any vfat mount subdirectories.
   """

   if filesystemType( dst ) == "vfat":
      copyTree( src, dst, dirsExistOk=dirsExistOk )
   else:
      shutil.copytree( src, dst, dirs_exist_ok=dirsExistOk )

def copyMaybePreservePerms( src, dst, followSymlinks=True ):
   """
   Copy files (like copy2), preserving permissions if the destination
   filesystem supports them (not vfat).

   See copyTreeMaybePreservePerms for more details
   """

   if filesystemType( dst ) == "vfat":
      if os.path.isdir( dst ):
         dst = os.path.join( dst, os.path.basename( src ) )
      shutil.copyfile( src, dst, follow_symlinks=followSymlinks )
   else:
      shutil.copy2( src, dst, follow_symlinks=followSymlinks )
