#!/usr/bin/env arista-python
# Copyright (c) 2008, 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os, sys
 
def tag( t ):
   """A little helper function to put a tag in the strace output by
   trying to chdir to '@@@<string>' so you can search for it.  tag
   will fail if the directory '@@@'+t exists."""
   try: 
      os.chdir( "@@@" + t ) 
   except: # pylint: disable=bare-except
      pass
   else:
      raise Exception("Uhoh -- this was a real directory")
 
from contextlib import contextmanager  # pylint: disable=wrong-import-position
@contextmanager 
def straced(pid=None,trace=None,tracefile=None, quiet=False):
   """straced is a contextmanager (for use with 'with') that straces
   the current process for the duration of the body of the with
   statement.  It invokes strace on the current process, sending the
   output to tracefile, if specified.  If not the output is sent to
   'straced.<pid>' in the current directory.

   Use it like this:

      with straced( tracefile="/tmp/trace.out" ):
         result = someFunctionToTrace()
         someOtherFunctionToTrace(result)

   If the traced pid is the current process, then there is
   synchronization to make sure that the entire body of the with
   statement is straced.  This synchronization relies on system calls
   and as a result, one or two addtional syscalls show up in the
   strace output before the execution (read and sometimes select), and
   one additional syscall shows up at the end (kill).

   There is currently no way to send the output to anything
   other than a file."""
   mypid = os.getpid()
   if pid is None: 
      pid = mypid
   log = tracefile or ("straced." + str( pid ))
   fp = open( log, "wb+" )
   fp.seek(0,os.SEEK_END)
   if not quiet:
      # pylint: disable-next=consider-using-f-string
      sys.stderr.write( "[Tracing %s to %s]\n" % (pid,log) )
   cpid = os.fork()
   if not cpid: 
      argv = ["strace", "-q", "-o", log, "-f", "-t", "-p", str( pid )] 
      if trace: 
         argv += ['-e','trace=' + trace]
      os.setsid()
      try:
         os.execvp( "strace", argv )
      except Exception:
         # Touch the file that the parent is waiting on.
         with open( log, "wb+" ) as f:
            f.write( "Failed to start strace\n" )
         sys.stderr.write("Failed to start strace\n")
         raise
   else:
      if pid == mypid:
         import time # pylint: disable=import-outside-toplevel
         t = .005
         w = .005
         while True:
            bytes = fp.read(1) # pylint: disable=redefined-builtin
            if bytes: break # pylint: disable=multiple-statements
            time.sleep( t )
            if t < 1:
               t += w
               w *= 2
      yield
      import signal # pylint: disable=import-outside-toplevel
      try:
         # The kernel sends a SIGTRAP to a ptraced process when it
         # enters and exits every system call.  When we send a SIGKILL
         # to the child, as we are returning back to user space, the
         # kernel will send a SIGTRAP back to the child.  When the
         # just-killed child runs, the kernel will disconnect it as a
         # ptracer, but there is a race between the dying child and
         # the killer.  If the killer runs fast enough, it will exit
         # the syscall before the dying ptracer has had a chance to
         # disconnect, and this will cause the SIGTRAP to get queued
         # up for the child.  If the child runs and disconnects before
         # the parent exits the kill() syscall, then the SIGTRAP is
         # not delivered and there is no problem.  We handle this race
         # by ignoring SIGTRAP until we are sure that the child has died.
         old = signal.signal( signal.SIGTRAP, signal.SIG_IGN )
         os.kill( cpid, signal.SIGKILL )
         os.waitpid( cpid, 0 )
      finally:
         # pylint: disable-next=used-before-assignment
         signal.signal( signal.SIGTRAP, old )
         
      fp.close()

@contextmanager 
def signalHandler(signum, handler): 
   import signal  # pylint: disable=import-outside-toplevel
   oldHandler = signal.signal( signum, handler ) 
   yield 
   signal.signal( signum, oldHandler ) 
