#!/usr/bin/env python3
# Copyright (c) 2011, 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''
Cli scheduler is responsible for executing scheduled Cli jobs which are used to
periodically execute CLI commands (e.g. show tech-support).
'''

import CliSchedulerLib, Tac, Url, subprocess
import os, time, Logging, QuickTrace, Tracing, SuperServer
import errno
import Cell
import signal
from socket import gethostname
import sys

t0 = Tracing.trace0
t1 = Tracing.trace1

qv = QuickTrace.Var
qt0 = QuickTrace.trace0
qt1 = QuickTrace.trace1

# Timeout of starting the scheduler even if system.initialized is not True
TIMEOUT_SYSTEMNOTINITIALIZED = 60 * 60

def removeFromJobsInProgressList( jobName, agent ):
   for job in agent.cliJobsInProgress:
      if job[0] == jobName:
         agent.cliJobsInProgress.remove( job )
         qt1( 'Removed from JobsInProgress list: ', qv( jobName ))

def _rescheduleJob( agent, jobName, subproc ):
   removeFromJobsInProgressList( jobName, agent )
   if jobName in agent.config.scheduledCli:
      jobStatus = agent.status.scheduledCliJobStatus[ jobName ]
      updateStatus( agent, jobName, time.time(), subproc.returncode,
                    jobStatus.lastExecutionStartTime )
   cliSchedDispatcher( agent )

def _isSubprocDone( agent, jobName ):
   # check if cli job is still running
   result = agent.cliJobSubprocess[ jobName ].poll() is not None

   if jobName in agent.compressJobSubprocess:
      result = ( result and
                 agent.compressJobSubprocess[ jobName ].poll() is not None )

   if jobName in agent.ddJobSubprocess:
      result = ( result and agent.ddJobSubprocess[ jobName ].poll() is not None )

   return result

def cliJobCompletionHandler( jobArgs, timedOut=False ):
   agent, jobName, logPath, _maxLogFiles, _maxTotalSize, cliTimeout, \
      verbose = jobArgs

   def killSubprocs( agent ):
      subprocs = [ agent.cliJobSubprocess, agent.compressJobSubprocess,
                   agent.ddJobSubprocess ]
      subprocList = [ subproc for subproc in subprocs if subproc is not None ]

      for proc in subprocList:
         try:
            proc[ jobName ].kill()
         except OSError:
            pass

   subproc = agent.cliJobSubprocess[ jobName ]
   # get ddProc if it exists
   ddProc = agent.ddJobSubprocess.get( jobName )
   if logPath:
      logDir = os.path.dirname( logPath ) + '/'
   qt1( 'cliJobCompletionHandler job:', qv( jobName ), 'ml:', qv( _maxLogFiles ),
        'timedOut:', qv( timedOut ), 'rc:', qv( subproc.returncode if not timedOut
           else 0 ) )
   if timedOut:
      Logging.log( CliSchedulerLib.SYS_CLI_SCHEDULER_ABORT, jobName,
            f"Timed out after {cliTimeout} seconds" )
      
      try:
         os.killpg( subproc.pid, signal.SIGKILL )
      except OSError:
         pass

      # give some time for the processes to exit
      try:
         Tac.waitFor( lambda: _isSubprocDone( agent, jobName ),
                      timeout=10,
                      description=f'cli job process({jobName}) to die',
                      sleep=True )
      except ( Tac.SystemCommandError, Tac.Timeout ):
         killSubprocs( agent )

      subproc.returncode = -1 #using returncode = -1 for timeout
      _rescheduleJob( agent, jobName, subproc )

   elif ddProc and ddProc.returncode != 0:
      # Error trying to save output to file
      # If logfile destination is almost full and Cli encounters ENOSPC
      # error. We use statvfs to detect this and generate filesystem full syslog
      # message instead of abort syslog message
      def _getMount( path ):
         path = os.path.realpath( os.path.abspath( path ) )
         while path != os.path.sep:
            if os.path.ismount( path ):
               return path
            path = os.path.abspath( os.path.join( path, os.pardir ) )
         return path
      if os.statvfs( _getMount( logPath ) ).f_bfree == 0:
         Logging.log( CliSchedulerLib.SYS_CLI_SCHEDULER_FILESYSTEM_FULL,
                      jobName )
      else:
         Logging.log( CliSchedulerLib.SYS_CLI_SCHEDULER_ABORT, jobName,
                      f"dd rc={ddProc.returncode}" )
      _rescheduleJob( agent, jobName, ddProc )      

   elif subproc.returncode != 0:
      # Error running the cli command
      Logging.log( CliSchedulerLib.SYS_CLI_SCHEDULER_ABORT, jobName,
                   f"rc={subproc.returncode}" )

      _rescheduleJob( agent, jobName, subproc )

   # ignoring compressProc status because it should not fail unless ddProc fails
   else:
      if _maxLogFiles:
         t1( 'Initiating log rotation' )
         displayLog = Url.filenameToUrl( logPath )
         if verbose:
            Logging.log( CliSchedulerLib.SYS_CLI_SCHEDULER_JOB_COMPLETED,
                         jobName, f"Logfile is stored in {displayLog}" )
      else:
         Logging.log( CliSchedulerLib.SYS_CLI_SCHEDULER_JOB_COMPLETED,
                      jobName, None )
      _rescheduleJob( agent, jobName, subproc )

   if logPath:
      logMgr = CliSchedulerLib.CliSchedLogMgr( logDir, _maxLogFiles - 1,
                                               _maxTotalSize )
      if logMgr.purgeFiles() :
         qt1( 'cliJobCompletionHandler files purged' )
      qt1( 'cliJobCompletionHandler total size:', ( qv( logMgr.diskSpaceUsed() ) ) )

def updateStatus( agent, jobName, lastExecutionTime=0, lastExecutionStatus=0,
                  startTime=0, jobInProgress=False ):
   qt1( "updateStatus job:", qv( jobName ), 'st:', qv( startTime ), "let:",
        qv( lastExecutionTime ), "les:", qv( lastExecutionStatus ), 'jip:',
        qv( jobInProgress ) )

   schedStatusType = Tac.Type( "System::CliScheduler::ScheduledCliJobStatus" ) 
   schedStatus = schedStatusType( jobName, jobInProgress, startTime,
                                  lastExecutionTime, lastExecutionStatus )
   agent.status.scheduledCliJobStatus.addMember( schedStatus )

def cliSchedDispatcher( agent ):

   while ( len( agent.cliJobsInProgress ) < agent.config.jobsInProgress ) and \
           len( agent.cliJobsPending ) != 0:
      job = agent.cliJobsPending.pop()
      ( jobName, logDir, maxLogFiles, maxTotalSize, cliCommand, cliTimeout,
         verbose, compressAlgo ) = job
      compressAlgo = compressAlgo if compressAlgo else \
         CliSchedulerLib.compressAlgoDefault
      if not agent.status.enabled:
         Logging.log( CliSchedulerLib.SYS_CLI_SCHEDULER_DISABLED_SKIP, jobName )
         qt1( 'CliScheduler is disabled, skip dispatching job:', ( qv( jobName ) ) )
         # In the normal case where agent.config is enabled, once the job is
         # completed, we'll run CliJobCompletionHandler, which calls _rescheduleJob.
         # In there, the completed job is removed from jobsInProgress list and the
         # job status is updated before it calls cliSchedDispatcher again.  There is
         # no need for that if agent.config is disalbed.  Here, we only update the
         # jobsInProgress list and return.
         removeFromJobsInProgressList( jobName, agent )
         continue
      qt1( 'cliSchedDispatcher dispatching job:', ( qv( jobName ) ) )

      origMask = os.umask( 0 )
      # Frequency of scheduled CLI command execution shouldn't be more than one
      # every minute. Hence we can store logfiles under /mnt/flash/schedule with
      # following file name <jobName>_YYYY-MM-DD.HH-MM.log # this file is compressed
      # Date and time used above will always be in local time
      if not logDir or logDir == CliSchedulerLib.logPrefixDefault:
         logDir = agent.rootFs
         if not logDir.endswith( '/' ):
            logDir += '/'
         logDir = f"{logDir}schedule/"

      logSuffix = f"{jobName}/"
      logDir += logSuffix
      t1( f'logDir: {logDir}' )
      logPath = None

      if maxLogFiles:
         timeExtension = time.strftime( "_%Y-%m-%d.%H%M", time.localtime() )
         logFile = f"{jobName}{timeExtension}.log.gz"
         if compressAlgo == "bzip2":
            logFile = f"{jobName}{timeExtension}.log.bz2"
         elif compressAlgo == "xz":
            logFile = f"{jobName}{timeExtension}.log.xz"

         if agent.config.prependHostname:
            logFile = f"{gethostname()}_{logFile}"
         logPath = f"{logDir}/{logFile}"

         try:
            os.makedirs( logDir, 0o777 )
         except OSError as e:
            if e.errno == errno.EEXIST:
               pass
            elif e.errno == errno.ENOSPC:
               Logging.log( CliSchedulerLib.SYS_CLI_SCHEDULER_FILESYSTEM_FULL,
                            jobName )
               os.umask( origMask )
               return
            elif e.errno == errno.EROFS:
               Logging.log( CliSchedulerLib.SYS_CLI_SCHEDULER_ABORT, jobName,
                            "read-only filesystem" )
               os.umask( origMask )
               return
            else:
               os.umask( origMask )
               raise

         try:
            logMgr = CliSchedulerLib.CliSchedLogMgr( logDir, maxLogFiles - 1,
                                                     maxTotalSize )
            qt1( 'CliScheduler total size:', ( qv( logMgr.diskSpaceUsed() ) ) )
            logMgr.rotateSnapshots()
         except OSError as e:
            if e.errno == errno.EROFS:
               Logging.log( CliSchedulerLib.SYS_CLI_SCHEDULER_ABORT, jobName,
                            "read-only filesystem" )
               return
            raise

      newCliCommand = CliSchedulerLib.replaceToken( cliCommand )
      cmdArgs = [ sys.executable, CliSchedulerLib.eosCliShell, "-s",
                  agent.cliSchedSysname, "--disable-aaa", "--disable-automore", "-p",
                  "15", "-c", newCliCommand ]
   
      agent.cliJobsInProgress.append( job )
      jobStatus = agent.status.scheduledCliJobStatus.get( jobName )
      lastExecutionStartTime = jobStatus.lastExecutionStartTime if jobStatus else 0
      updateStatus( agent, jobName, startTime=time.time(), jobInProgress=True,
                    lastExecutionTime=lastExecutionStartTime )
      qt1( 'cliSchedDispatcher inprogress:', qv( agent.cliJobsInProgress is not None
           ), 'njobs:', qv( len( agent.cliJobsInProgress ) ) )
      # pylint: disable-next=subprocess-popen-preexec-fn,consider-using-with
      agent.cliJobSubprocess[ jobName ] = subprocess.Popen( 
         cmdArgs, stdout=subprocess.PIPE, stderr=subprocess.STDOUT,
         preexec_fn=os.setsid )

      if maxLogFiles:
         compressArgs = [ f"/bin/{compressAlgo}", "-f", "-9" ]
         # pylint: disable-next=consider-using-with
         agent.compressJobSubprocess[ jobName ] = subprocess.Popen(
            compressArgs, stdin=agent.cliJobSubprocess[ jobName ].stdout,
            stdout=subprocess.PIPE, stderr=subprocess.DEVNULL )

         agent.cliJobSubprocess[ jobName ].stdout.close()

         # pipe output of gzip/bzip2/xz to file and sync afterward
         ddArgs = [ "/bin/dd", "of=" + logPath, "conv=fdatasync" ]
         # pylint: disable-next=consider-using-with
         agent.ddJobSubprocess[ jobName ] = subprocess.Popen(
            ddArgs, stdin=agent.compressJobSubprocess[ jobName ].stdout,
            stdout=subprocess.DEVNULL,
            stderr=subprocess.DEVNULL )

         agent.compressJobSubprocess[ jobName ].stdout.close()

      os.umask( origMask )
      jobArgs = ( agent, jobName, logPath, maxLogFiles, maxTotalSize,
                  cliTimeout, verbose )

      agent.cliJobPoller = Tac.Poller(
            lambda: _isSubprocDone( agent, jobName ),
            handler=lambda ignored: cliJobCompletionHandler( jobArgs ),
            timeoutHandler=lambda: cliJobCompletionHandler( jobArgs, timedOut=True ),
            warnAfter=cliTimeout * 2,
            timeout=cliTimeout,
            description="scheduled Cli Job {jobName} to complete" )

class CliSchedExec:
   activeExecsWithAt_ = set()
   timePoller_ = None
   lastTimeDiff_ = None
   chkDelay_ = 10

   @staticmethod
   def adjustSchedule():
      time.sleep ( 1 )
      curDateTime = Tac.utcNow()
      curMonoDateTime = Tac.now()
      timeDiff = int( curDateTime - curMonoDateTime )
      if CliSchedExec.lastTimeDiff_ != timeDiff:
         qt0( "Clock change detected diff=", qv( timeDiff ) )
         # Clock was updated, time to change timeMin of all activeExecsWithAt_
         for execs in CliSchedExec.activeExecsWithAt_:
            if curDateTime >= execs.at:
               execs.act.timeMin = Tac.now()
            else:
               execs.act.timeMin = Tac.now() + (execs.at - curDateTime)
            qt1( "CliSchedExec job:", qv( execs.jobName ),
                 "is rescheduled to run in", qv( execs.at - curDateTime ),
                 "seconds" )
      CliSchedExec.lastTimeDiff_ = timeDiff
      CliSchedExec.timePoller_.timeMin = Tac.now() + CliSchedExec.chkDelay_

   def __init__( self, agent, jobName, logDir, at, interval, maxLogFiles,
          maxTotalSize, cliCommand, timeout, verbose, compressAlgo ):
      self.agent = agent
      self.jobName = jobName
      self.logDir = logDir
      self.interval = interval
      self.at = at
      self.maxLogFiles = maxLogFiles
      self.maxTotalSize = maxTotalSize
      self.cliCommand = cliCommand
      if 'CLI_SCHED_TIMEOUT' in os.environ:
         self.timeout = int( os.environ[ 'CLI_SCHED_TIMEOUT' ] )
      else:
         self.timeout = timeout

      self.verbose = verbose
      self.compressAlgo = compressAlgo

      self.act = Tac.ClockNotifiee( handler=self.run )

      if at == CliSchedulerLib.scheduleNow:
         self.act.timeMin = Tac.now()
         qt1( "CliSchedExec job:", qv( jobName ), "is scheduled to run now" )
      else:
         curDateTime = Tac.utcNow()
         if curDateTime >= at:
            self.act.timeMin = Tac.now()
            qt1( "CliSchedExec job:", qv( jobName ), "is scheduled to run now" )
         else:
            CliSchedExec.activeExecsWithAt_.add( self )
            if CliSchedExec.timePoller_ is None:
               qt0( "Starting time poller to detect clock changes by job:",
                    qv ( jobName ) )
               CliSchedExec.lastTimeDiff_ = int( curDateTime - Tac.now() )
               CliSchedExec.timePoller_ = Tac.ClockNotifiee(
                  handler=CliSchedExec.adjustSchedule )
               CliSchedExec.timePoller_.timeMin = Tac.now() + CliSchedExec.chkDelay_
            self.act.timeMin = Tac.now() + (at - curDateTime)
            qt1( "CliSchedExec job:", qv( jobName ), "is scheduled to run in",
                  qv( at - curDateTime ), "seconds" )

   def run( self ):
      self.removeFromactiveExecs()
      cliJob = ( self.jobName, self.logDir, self.maxLogFiles,
                 self.maxTotalSize, self.cliCommand, self.timeout,
                 self.verbose, self.compressAlgo )
      jobNamesProgressList = [ job[0] for job in self.agent.cliJobsInProgress ]
      if jobNamesProgressList.count( self.jobName )  or \
            self.agent.cliJobsPending.count( cliJob ):
         qt1( "Jobname skipped ", qv( self.jobName ) )
         Logging.log( CliSchedulerLib.SYS_CLI_SCHEDULER_SKIP, self.jobName )
      else:
         self.agent.cliJobsPending.append( ( self.jobName, self.logDir,
                                             self.maxLogFiles,
                                             self.maxTotalSize,
                                             self.cliCommand, self.timeout,
                                             self.verbose, self.compressAlgo ) )
         cliSchedDispatcher( self.agent )
      self.act.timeMin = Tac.endOfTime \
          if self.interval == CliSchedulerLib.scheduleOnce \
          else Tac.now() + self.interval * 60

   def removeFromactiveExecs( self ):
      if self in CliSchedExec.activeExecsWithAt_:
         CliSchedExec.activeExecsWithAt_.remove( self )
         if len(CliSchedExec.activeExecsWithAt_) == 0:
            qt0( "Removing time poller" )
            CliSchedExec.timePoller_.timeMin = Tac.endOfTime
            CliSchedExec.timePoller_ = None

   def close( self ):
      if not self.act:
         return
      qt1( "CliSchedExec job:", qv( self.jobName ), "is being removed" )
      self.removeFromactiveExecs()
      self.act.timeMin = Tac.endOfTime
      cliJob = ( self.jobName, self.logDir, self.maxLogFiles,
                 self.maxTotalSize, self.cliCommand, self.timeout,
                 self.verbose, self.compressAlgo )
      try:
         self.agent.cliJobsPending.remove( cliJob )
         qt1( "CliSchedExec job:", qv( self.jobName ), "is removed from list of "\
              "pending jobs" )
      except ValueError:
         qt1( "CliSchedExec failed to find pending job:", qv( self.jobName ) )
      self.act = None

   def __del__( self ):
      self.close()

class ScheduledCliReactor:
   """A reactor for each scheduled CLI command execution job."""
   notifierTypeName = 'System::CliScheduler::ScheduledCli'
   def __init__( self, cliConfig, schedConfig, agent ):
      startAt = cliConfig.startAt
      now = time.time()
      # In case SuperServer restarts, we have to check if a job has already run
      # so we don't immediately reschedule them to restart.
      # This will influence 2 cases:
      #   1) The job has "schedule now"
      #   2) The job has a timed schedule but has already run.
      if cliConfig.name in agent.status.scheduledCliJobStatus and (
            startAt == CliSchedulerLib.scheduleNow or now >= startAt ):
         schJobStatus = agent.status.scheduledCliJobStatus[ cliConfig.name ]
         startAt = self.getStartAt( cliConfig, schJobStatus, now )
      self.schedConfig = schedConfig
      self.cliConfig = cliConfig
      self.agent = agent
      qt0( 'ScheduledCliReactor: scheduledCli collection has been updated key:',
           qv( cliConfig.name ) )

      if cliConfig.interval == CliSchedulerLib.scheduleOnce:
         if agent.status.scheduledCliJobStatus[ cliConfig.name ].lastExecutionTime:
            qt0( f'job {qv( cliConfig.name )} has already executed once' )
            return

      self.schedExec = CliSchedExec( self.agent, cliConfig.name,
                        cliConfig.logDir, startAt, cliConfig.interval,
                        cliConfig.maxLogFiles, cliConfig.maxTotalSize,
                        cliConfig.cliCommand, int( cliConfig.timeout ),
                        cliConfig.verbose, cliConfig.compressAlgo )
      qt0( 'ScheduledCliReactor job:', qv( cliConfig.name ), 'has been scheduled' )

   def getStartAt( self, cliConfig, schJobStatus, now ):
      startAt = cliConfig.startAt
      lastExecutionStart = schJobStatus.lastExecutionStartTime
      if startAt == CliSchedulerLib.scheduleNow:
         if lastExecutionStart == 0:
            # this is a "schedule now" job which has yet to run
            # leave the startAt as is
            return CliSchedulerLib.scheduleNow
         startAt = lastExecutionStart
      interval = cliConfig.interval * 60 
      # Add one or more multiples of the scheduled interval to the
      # last execution time. We check for multiple intervals in 
      # the unlikely case that SuperServer has been gone away for an
      # extended time.
      startAt += interval * ( ( ( now - lastExecutionStart ) // interval ) + 1 )
      return startAt

   def close( self ):
      qt0( 'ScheduledCliReactor job:', qv( self.cliConfig.name ), 'close' )
      self.schedExec.close()

class ConfigReactor( Tac.Notifiee ):
   notifierTypeName = 'System::CliScheduler::Config'
   def __init__( self, agent, notifier, status ):
      self.notifier = notifier
      self.status = status
      self.reactors = {}
      self.agent = agent
      Tac.Notifiee.__init__( self, notifier )
      for key in self.notifier.scheduledCli:
         if key not in self.status.scheduledCliJobStatus:
            self.handleConfig( key )
         else:
            self.reactors[ key ] = ScheduledCliReactor( self.notifier.\
                                                    scheduledCli[ key ],\
                                                    self.notifier, self.agent )

      # Delete stale status entries.
      for key in self.status.scheduledCliJobStatus:
         if key not in self.notifier.scheduledCli:
            self.handleConfig( key )
   
   @Tac.handler( 'scheduledCli' )
   def handleConfig( self, name ):
      if name in self.notifier.scheduledCli:
         schedStatusType = Tac.Type( "System::CliScheduler::"\
                                     "ScheduledCliJobStatus" )
         schedStatus = schedStatusType( name, False, 0, 0, 0 )
         self.status.scheduledCliJobStatus.addMember( schedStatus )
         self.reactors[ name ] = ScheduledCliReactor( self.notifier.\
                                                    scheduledCli[ name ],\
                                                    self.notifier, self.agent )
      else:
         if name in self.reactors:
            qt1( 'handleConfig removing reactor for job:', qv( name ) )
            self.reactors[ name ].close()
            del self.reactors[ name ]
         if name in self.status.scheduledCliJobStatus:
            qt1( 'handleConfig removing status for job:', qv( name ) )
            del self.status.scheduledCliJobStatus[ name ]

   def close( self ):
      qt0( 'ConfigReactor close' )
      Tac.Notifiee.close( self )

class LowMemReactor( Tac.Notifiee ):
   notifierTypeName = 'ProcMgr::LowMemoryModeStatus'
   def __init__( self, notifier, status ):
      self.notifier = notifier
      self.status = status
      Tac.Notifiee.__init__( self, notifier )
      self.handleStatus()
   
   @Tac.handler( 'status' )
   def handleStatus( self ):
      # If the lowMemMode status is True, disable CliScheduler. Otherwise, enable it.
      if self.notifier.status:
         Logging.log( CliSchedulerLib.SYS_MEMORY_EXHAUSTION_CLI_SCHEDULER_DISABLED )
         self.status.enabled = False
      else:
         Logging.log( CliSchedulerLib.SYS_CLI_SCHEDULER_ENABLED )
         self.status.enabled = True

class SystemInitializedReactor( Tac.Notifiee ):
   notifierTypeName = 'System::Status'

   def __init__( self, agent, notifier ):
      self.notifier = notifier
      self.agent = agent
      self.timeoutClock = Tac.ClockNotifiee( handler=self.timeout )
      self.timeoutClock.timeMin = Tac.now() + TIMEOUT_SYSTEMNOTINITIALIZED
      Tac.Notifiee.__init__( self, notifier )
      self.handleInitialized()

   @Tac.handler( 'initialized' )
   def handleInitialized( self ):
      initialized = self.notifier.initialized
      qt0( 'SystemInitializedReactor handleInitialized=' + str( initialized ) )
      if initialized:
         self.agent.run()
         self.close()

   def timeout( self ):
      qt0( 'SystemInitializedReactor timeout' )
      self.agent.run()
      self.close()

   def close( self ):
      qt0( 'SystemInitializedReactor close' )
      self.timeoutClock.timeMin = Tac.endOfTime
      Tac.Notifiee.close( self )

class CliSchedulerMgr( SuperServer.SuperServerAgent ):

   def __init__( self, entityManager, rootFs ):
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      self.rootFs = rootFs
      self.cliJobsInProgress = []
      self.cliJobSubprocess = {}
      self.ddJobSubprocess = {}
      self.cliJobsPending = []
      self.compressJobSubprocess = {}
      self.cliSchedAct = None
      self.cliSchedSysname = None
      self.act = None
      self.reactor = None
      self.lowMemReactor = None
      self.systemInitializedDetector = None
      self.asuCompletionDetector = None
      self.config = mg.mount( 'sys/clischeduler/config',
                              'System::CliScheduler::Config', 'r' )
      self.status = mg.mount(  'sys/clischeduler/status',
                              'System::CliScheduler::Status', 'w' )
      self.lowMemStatus = mg.mount( 'sys/status/lowMemStatus',
                              'ProcMgr::LowMemoryModeStatus', 'r' )
      self.asuHwStatus = mg.mount( 'asu/hardware/status', 'Asu::AsuStatus', 'r' )
      self.systemInitializedStatus = mg.mount( 'sys/status/system',
                                             'System::Status', 'r' )
      self.asuBootStageCompletionStatus = mg.mount(
         Cell.path( 'stage/boot/completionstatus' ),
         'Stage::CompletionStatusDir', 'r' )

      def _finish():
         self.cliSchedSysname = entityManager.sysname()
         self.asuCompletionDetector = Tac.newInstance(
               "Stage::AsuCompletionDetector", self.asuHwStatus,
               self.asuBootStageCompletionStatus, self.redundancyStatus() )
         # don't run until active (rpr or sso)
         if not self.active():
            return
         if self.asuCompletionDetector.isHitlessReloadInProgress():
            qt0( 'Defer Cli scheduler for hitless reload' )
            self.defer( 5 * 60 )
         else:
            qt0( 'Waiting for system initialization ...' )
            self.systemInitializedDetector = SystemInitializedReactor(
                                                   self,
                                                   self.systemInitializedStatus )
      mg.close( _finish )

   def runWhenReady( self ):
      # Delay at least 5 minutes if hitlessly reloading
      if self.asuCompletionDetector.isHitlessReloadInProgress():
         qt0( 'Defer Cli scheduler start for hitless reload' )
         self.act.timeMin = Tac.now() + ( 5 * 60 )
         return
      # Otherwise start once the system is initialized
      if self.systemInitializedDetector is None:
         qt0( 'Hitless reload complete, waiting for system initialization ...' )
         self.systemInitializedDetector = SystemInitializedReactor(
                                                   self,
                                                   self.systemInitializedStatus )

   def defer( self, delay ):
      self.act = Tac.ClockNotifiee( handler=self.runWhenReady )
      self.act.timeMin = Tac.now() + delay

   def run( self ):
      if self.lowMemReactor is None:
         self.lowMemReactor = LowMemReactor( self.lowMemStatus, self.status )
      self.reactor = ConfigReactor( self, self.config, self.status )
      if self.act is not None:
         self.act.timeMin = Tac.endOfTime

   def onSwitchover( self, protocol ):
      for key in self.status.scheduledCliJobStatus:
         schedStatusType = Tac.Type( "System::CliScheduler::"\
                                     "ScheduledCliJobStatus" )
         schedStatus = schedStatusType( key, False, 0, 0, 0 )
         self.status.scheduledCliJobStatus.addMember( schedStatus )
      # start Cli scheduler in 5 minutes
      qt0( 'Starting Cli scheduler in 5 minutes ...' )
      self.defer( 5 * 60 )

def Plugin( ctx ):
   rootFs = Url.getFilesystem( 'flash:' ).location_
   ctx.registerService( CliSchedulerMgr( ctx.entityManager, rootFs ) )
