# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import SuperServer
import Tac
import Tracing
import re

t0 = Tracing.t0

class ConsoleConfigReactor( SuperServer.GenericService ):
   notifierTypeName = 'Mgmt::Console::Config'

   def __init__( self, serviceName, config ):
      self.config_ = config
      SuperServer.GenericService.__init__( self, serviceName, config,
                                           sync=False )
      self.handleLongHostname()

      t0( "Write login defs file" )
      self.configLoginFile()

   def warm( self ):
      return True

   def sync( self ):
      fileText = ""
      t0( "Modify Agetty file" )

      with open( "/usr/sbin/Agetty" ) as f:
         fileText = ""
         for line in f:
            if "exec /sbin/mingetty" in line:
               if self.config_.longHostname:
                  if "--long-hostname" not in line:
                     line = line.replace( "exec /sbin/mingetty",
                           "exec /sbin/mingetty --long-hostname" )
               else:
                  line = line.replace( "--long-hostname", "" )
            fileText += line

      t0( "Write modified string back to the file" )
      with open( "/usr/sbin/Agetty", "w") as f:
         f.write( fileText )

   @Tac.handler( "longHostname" )
   def handleLongHostname( self ):
      self.sync()

   def configLoginFile( self ):
      # Configure login command related settings in /etc/login.defs
      loginFilePath = "/etc/login.defs"
      fileCtx = ""
      with open( loginFilePath ) as loginFile:
         fileCtx = loginFile.read()
         # Set default timeout to 120 seconds
         timeoutRegex = r"^LOGIN_TIMEOUT.*\d+.*\n"
         timeoutComment = " # ARISTA: This timeout is changed by EOS\n"
         timeoutConfig = "LOGIN_TIMEOUT 120" + timeoutComment
         if re.search( timeoutRegex, fileCtx, flags=re.MULTILINE ):
            fileCtx = re.sub( timeoutRegex, timeoutConfig, fileCtx,
                              flags=re.MULTILINE )
         else:
            fileCtx += ( "\n" + timeoutConfig )

      # Write to the file
      self.writeConfigFile( loginFilePath, fileCtx, saveBackupConfig=False,
                            writeHeader=False )

class ConsoleManager( SuperServer.SuperServerAgent ):
   def __init__( self, entityManager ):
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      self.config_ = mg.mount( 'mgmt/console/config',
                               'Mgmt::Console::Config', 'r' )
      self.consoleConfigReactor_ = None
      def _finished():
         self.consoleConfigReactor_ = \
               ConsoleConfigReactor( "ConsoleConfig",
                                     self.config_ )
      mg.close( _finished )

def Plugin( ctx ):
   ctx.registerService( ConsoleManager( ctx.entityManager ) )
