# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import re, os, errno

from HttpServiceConstants import ServerConstants
from HttpServiceConstants import CapiCipherSuites
import HttpServiceSsl
import Cell
import Arnet.NsLib
import SysMgrLib
from IpLibConsts import DEFAULT_VRF
import Logging
import MgmtSecuritySslStatusSm
import SslCertKey
import SuperServer
import Tac
import Tracing
import Plugins
import PyWrappers.Nginx as nginx
from GenericReactor import GenericReactor

MGMT_SECURITY_SSL_CONSTANTS = Tac.Type( "Mgmt::Security::Ssl::Constants" )
LOG_LEVELS = Tac.Type( "HttpService::LogLevel" )
ErrorType = Tac.Type( "Mgmt::Security::Ssl::ErrorType" )
SslFeature = Tac.Type( "Mgmt::Security::Ssl::SslFeature" )

traceHandle = Tracing.Handle( 'HttpServiceSuperServerPlugin' )
warn = traceHandle.trace1
info = traceHandle.trace2
trace = traceHandle.trace3
debug = traceHandle.trace4

import QuickTrace # pylint: disable=wrong-import-position
qt0 = QuickTrace.trace0

CAPI_CERTIFICATE_CONFIGURATION_CONFLICT = Logging.LogHandle(
               "CAPI_CERTIFICATE_CONFIGURATION_CONFLICT",
               severity=Logging.logWarning,
               fmt="Certificate/key pair in the SSL Profile will not be used",
               explanation="A server certificate is configured in HTTPS config."
               "Only one server certificate can be configured at a time.",
               recommendedAction="Remove the server certificate in HTTPS config"
               "with 'no protocol https certificate' in 'management api"
               "http-commands'" )

CAPI_CIPHERSUITE_AND_HTTPS_CONFIG_INCOMPATIBLE = Logging.LogHandle(
              "CAPI_CIPHERSUITE_AND_HTTPS_CONFIG_INCOMPATIBLE",
              severity=Logging.logWarning,
              fmt="SSL profile ciphersuite and protocol HTTPS cipher "
              "configuration are incompatible, so HTTP server will not start.",
              explanation="Configure either SSL profile ciphersuite or HTTPS "
              "ciphers, mac algorithms and key-exchange. They cannot both be "
              "configured.",
              recommendedAction="Remove the cipher, mac, and key-exchange "
              "configurations in HTTPS config with 'no protocol https cipher', "
              "'no protocol https mac', and 'no protocol https key-exchange' in "
              "'management api http-commands'." )

def confFile( vrf=None ):
   if not vrf:
      return '/etc/nginx/nginx.conf'
   else:
      return '/etc/nginx/nginx-%s.conf' % vrf

def pidFile( vrf=None ):
   if not vrf:
      return '/var/run/nginx/nginx.pid'
   else:
      return '/var/run/nginx/nginx-%s.pid' % vrf

def getNsFromVrf( vrf ):
   if vrf == DEFAULT_VRF:
      return Arnet.NsLib.DEFAULT_NS
   else:
      return 'ns-%s' % vrf

class HttpServiceSslReactor( MgmtSecuritySslStatusSm.SslStatusSm ):
   __supportedFeatures__ = [ SslFeature.sslFeatureCertKey,
                             SslFeature.sslFeatureChainedCert,
                             SslFeature.sslFeatureTrustedCert,
                             SslFeature.sslFeatureCrl,
                             SslFeature.sslFeatureTls,
                             SslFeature.sslFeatureFips,
                             SslFeature.sslFeatureCipher,
                             SslFeature.sslFeatureCipherV1_3,
                             SslFeature.sslFeatureDhparam,
                           ]

   def __init__( self, config, status, profileName, capiConfig, containingAgent ):
      self.constants_ = MGMT_SECURITY_SSL_CONSTANTS
      self.capiConfig_ = capiConfig
      self.containingAgent_ = containingAgent
      super().__init__( status, profileName,
                        'HttpService' )

   def handleProfileState( self ):
      if ( self.profileStatus_ and self.profileStatus_.state == "valid" ):
         if ( self.capiConfig_.httpsConfig.sslCertificate != "" and
              self.capiConfig_.httpsConfig.sslKey != "" ):
            Logging.log( CAPI_CERTIFICATE_CONFIGURATION_CONFLICT )

      # Update cipherFilter after any change in profile state
      self.containingAgent_.serverStatus.cipherFilter = getCipherFilter(
         self.capiConfig_.httpsConfig, self )
      # let restart if we the https server enabled
      if self.capiConfig_.httpsConfig.enabled:
         self.containingAgent_.restartServer()

   def handleProfileDelete( self ):
      # Update cipherFilter after deleting SSL profile for consistency
      self.containingAgent_.serverStatus.cipherFilter = getCipherFilter(
            self.capiConfig_.httpsConfig, self )
      if self.capiConfig_.httpsConfig.enabled:
         self.containingAgent_.restartServer()

# run a bash command (usually 'service blah') inside a vrf (or directly if no vrf)
def vrfTacRun( vrfName, cmd, ignoreReturnCode=False, ignoreTimeout=False ):
   serviceCmd = [ "service", "nginx", cmd ]
   if vrfName != DEFAULT_VRF:
      trace( 'VrfManager:', cmd, 'vrf entry', vrfName )
      serviceCmd += [ vrfName ]
   
   try:
      result = Tac.run( serviceCmd, timeout=SuperServer.defaultTimeout,
                        stdout=Tac.CAPTURE, stderr=Tac.CAPTURE, asRoot=True, 
                        ignoreReturnCode=ignoreReturnCode )
   except Tac.Timeout:
      qt0( "Timed out while trying to run", " ".join( serviceCmd ) )
      if not ignoreTimeout:
         raise
      result = f"Timeout when running command '{ ' '.join( serviceCmd ) }'"
   trace( 'vrfTacRun: vrf entry', vrfName, cmd, result )
   return result

def httpServiceEnabled( capiConfig ):
   for serviceConfig in capiConfig.service.values():
      if serviceConfig.enabled:
         return True
   return False

def removeFile( name ):
   try:
      os.unlink( name )
   except OSError as e:
      if e.errno != errno.ENOENT:
         raise
 
def cleanupNginx( vrfName ):
   assert vrfName
   trace( 'cleanupNginx for VRF', vrfName )
   # this function is used both when we tear down an nginx service
   # normally and also in restart cases when we need to find leftover 
   # nginx service config files that need to be pruned.
   vrfTacRun( vrfName, 'stop', ignoreReturnCode=True, ignoreTimeout=True )

   removeFile( confFile( vrfName ) )
   removeFile( pidFile( vrfName ) )

class HttpService( SuperServer.LinuxService ): 
   notifierTypeName = "HttpService::Config"

   def __init__( self, containingAgent, vrf=None ):
      self.containingAgent_ = containingAgent
      self.capiConfig_ = containingAgent.capiConfig
      self.serverStatus_ = containingAgent.serverStatus
      self.allVrfStatus_ = containingAgent.allVrfStatus
      self.sslServiceStatus_ = containingAgent.sslServiceStatus
      self.mgmtSecurityConfig_ = containingAgent.mgmtSecurityConfig
      serviceName = nginx.name()
      self.vrf_ = vrf or DEFAULT_VRF
      # Record the namespace with a uniqueId where nginx is running to
      # distinguish namespace recreation and exit the old one. 
      self.namespace_ = SysMgrLib.netnsNameWithUniqueId( getNsFromVrf( self.vrf_ ) )
      if vrf:
         serviceName += '-' + vrf
      else:
         # create VRF status for default VRF
         containingAgent.serverStatus.newVrfStatus( DEFAULT_VRF )
         
      # Remove the previous nginx service running in the network namespace
      serviceConfFile = confFile( vrf )
      self._maybeCleanupOldService( serviceConfFile )

      SuperServer.LinuxService.__init__( self, serviceName, nginx.name(),
                                         self.capiConfig_, serviceConfFile ) 
   
   def _maybeCleanupOldService( self, configFileName ):
      # In Sysdb restart, Ira creates a network namespace 
      # by deleting and re-creating it unless it's a hitless restart.
      # It could be an issue for nginx service running in the previous 
      # network namespace that fail to realize something has changed.
      # The nginx service will only reload its configuration and hold
      # the old network namespace preventing it from being re-created.
      # Therefore we need to clean up the previous nginx service 
      # when the network namespace name changes.
      if self.namespace_ == Arnet.NsLib.DEFAULT_NS:
         return

      if os.path.exists( configFileName ):
         with open( configFileName ) as configFile:
            oldConfigFileContents = configFile.read()
            if self.namespace_ not in oldConfigFileContents:
               cleanupNginx( self.vrf_ )

   def serviceProcessWarm( self ):
      """ Returns whether or not the process associated with the service has
      fully restarted and is running based on the new configuration.
      NOTE - this can be very difficult to get right. Generally, one
      must override the startService, stopService, and restartService
      commands in order to save enough information that one can
      reliably determine whether or not the service has been completely
      restarted and is running based on its new configuration. """
      
      serviceEnabled = self.serviceEnabled()
      if ( serviceEnabled and
           self.vrf_ in self.serverStatus_.vrfStatus and
           'nginx' in self.serverStatus_.vrfStatus[ self.vrf_ ].vrfError and
           self.serverStatus_.vrfStatus[ self.vrf_ ].vrfError[ 'nginx' ] ):
         # If there is an error with nginx(such as nginx conf check fails or
         # nginx fails to reload), then we shouldn't try to restart nginx
         return True

      try:
         result = vrfTacRun( self.vrf_, "status", ignoreReturnCode=True )
      except Tac.Timeout:
         # nginx status can not be fetched, considered as not warm
         trace( "Fail to get service nginx status due to timeout" ) 
         return False

      if serviceEnabled: 
         return self.checkServiceCmdOutput( 'status', [ result ] )
      else:
         return "inactive (dead)" in result or result == "nginx is stopped\n"

   def _externalInternalServerEnabled( self ):
      # whether protocols are enabled in VRF
      serviceConfigEnabled = ( self.capiConfig_.httpsConfig.enabled or 
                               self.capiConfig_.httpConfig.enabled or
                               self.capiConfig_.localHttpConfig.enabled )
      
      # whether VRF is active
      vrfActive = ( self.vrf_ == DEFAULT_VRF )
      if self.vrf_ in self.allVrfStatus_.vrf:
         vrfActive = ( self.allVrfStatus_.vrf[ self.vrf_ ].state == "active" )
      
      # whether there are enabled services configured in VRF
      vrfEnabled = bool( self._getVrfEnabledService() )
      # check overridden VRF configuration in http-server mode 
      if self.vrf_ in self.capiConfig_.vrfConfig:
         vrfConfig = self.capiConfig_.vrfConfig[ self.vrf_ ]
         if vrfConfig.serverState == 'enabled':
            vrfEnabled = True
         elif vrfConfig.serverState == 'disabled':
            vrfEnabled = False

      trace( "_externalInternalServerEnabled nginx", self.vrf_, 
             "is service on:", serviceConfigEnabled )
      trace( "_externalInternalServerEnabled nginx", self.vrf_, 
             "is vrf enabled:", vrfEnabled )
      trace( "_externalInternalServerEnabled nginx", self.vrf_, 
             "is vrf active:", vrfActive )
      if self.vrf_ in self.serverStatus_.vrfStatus:
         vrfStatus = self.serverStatus_.vrfStatus[ self.vrf_ ]
         if not vrfActive:
            vrfStatus.vrfError[ 'vrf' ] = ServerConstants.CAPI_VRF_ERROR % self.vrf_
         else:
            vrfStatus.vrfError[ 'vrf' ] = ''
      return ( serviceConfigEnabled and vrfEnabled and vrfActive )

   def _unixServerEnabled( self, unixOnly=False ):
      if self.vrf_ != DEFAULT_VRF:
         # uds only runs in default VRF
         return False
      
      # whether protocol is enabled
      unixConfigEnabled = self.capiConfig_.unixConfig.enabled
      if unixOnly:
         # nginx is running in default VRF with only Uds enabled
         # when there is no service enabled in default VRF which means
         # external and internal servers are disabled in VRF
         unixConfigEnabled &= not self._externalInternalServerEnabled()
      
      # whether there are enabled services using uds
      serviceUnixEnabled = bool( self._getVrfEnabledService( serverType="uds" ) ) 

      trace( "_unixServerEnabled is unix enabled:", unixConfigEnabled )
      trace( "_unixServerEnabled is service enabled:", serviceUnixEnabled ) 
      return ( unixConfigEnabled and serviceUnixEnabled )

   def serviceEnabled( self ):
      """ Returns whether the service is enabled / whether there is
      enough configuration to properly start the service """
      # if https is enabled, we need to make sure SSL configuration is valid
      if self.capiConfig_.httpsConfig.enabled:
         # move Capi SSL profile configuration validity check 
         # from startService() to serviceEnable() to 
         # avoid periodically SuperServer healthCheck to restart nginx
         sslConfValid, sslConfError = self.sslConfigurationValid()
         if not sslConfValid and self.vrf_ in self.serverStatus_.vrfStatus:
            self.stopService()
            self.serverStatus_.vrfStatus[ self.vrf_ 
                                        ].vrfError[ 'nginx' ] = sslConfError
            return False
      
      return self._externalInternalServerEnabled() or self._unixServerEnabled()

   def restartService( self ):
      trace( "restartService enter" )
      self.startService()
      trace( "restartService exit" )

   def sslConfigurationValid( self ):
      if self._isInvalidSslCertKeyPair():
         trace( "sslConfigurationValid exit no cert" )
         return False, "No server certificate configured"
      
      certStatus = self._checkCert()
      if certStatus:
         trace( "sslConfigurationValid exit invalid cert" )
         return False, certStatus

      cipherConfigStatus = self._checkCipherConfig()
      if cipherConfigStatus:
         trace( "sslConfigurationValid exit ciphersuite conflict" )
         return False, cipherConfigStatus
           
      return True, ""

   def nginxConfValid( self ):
      try:
         vrfTacRun( self.vrf_, "configtest" )
      except Tac.Timeout:
         trace( "nginxConfValid exit timeout on test" )
         return False, "Timeout when testing new configuration"
      except Tac.SystemCommandError as e:
         trace( "nginxConfValid exit system command error on test", e.output )
         return False, "Error testing new configuration. " + e.output
      
      return True, ""

   def cleanupNginxUds( self ):
      # clean up possible stale nginx_status unix domain socket files
      removeFile( ServerConstants.NGINX_STATUS_UDS % self.vrf_ )

      # for default VRF, also check possible service UDS configured
      if self.vrf_ == DEFAULT_VRF:
         for serviceClass in self.containingAgent_.httpServices.values():
            removeFile( serviceClass.getUnixDomainSocket() )
 
   def reloadOrRestartService( self, restart=True ):
      # clean up possible stale UDS files when we start nginx 
      if restart:
         self.cleanupNginxUds()
      nginxCmd = 'start' if restart else 'reload'
 
      # Try to start the service, return False if there is an error
      try:
         result = vrfTacRun( self.vrf_, nginxCmd )
      except Tac.SystemCommandError as e:
         trace( f"startService exit due to system command error on {nginxCmd}",
                e.output )
         result = e.output.splitlines()
         seen = set()
         errors = [ x for x in result if x not in seen and not seen.add( x ) ]
         return False, "\n".join( errors )
      except Tac.Timeout:
         trace( f"startService exit due to command timeout on {nginxCmd}" )
         return False, f"Timed out waiting for nginx to {nginxCmd}"

      return True, ""
   
   def startService( self ):
      trace( "startService enter" )
      # If nginx is already running, we reload nginx. Otherwise we start it.
      try:
         result = vrfTacRun( self.vrf_, "status", ignoreReturnCode=True )
      except Tac.Timeout:
         # nginx status can not be fetched at this time, 
         # might due to system slowness during OOM. Therefore retry it later
         trace( "Fail to get service nginx status due to timeout" )
         self.sync()
         return
      restart = not self.checkServiceCmdOutput( 'status', [ result ] )
      self._updateVrfStatus()

      # if nginx conf isn't valid, stop nginx and return
      nginxConfValid, nginxError = self.nginxConfValid()
      if not nginxConfValid:
         self.stopService()
         self.serverStatus_.vrfStatus[ self.vrf_ ].vrfError[ 'nginx' ] = nginxError
         qt0( 'Invalid nginx config for VRF %s with error %s'
              % ( self.vrf_, nginxError ) )
         return
      
      trace( "starting server http", self.capiConfig_.httpConfig.enabled, "on port,",
             self.capiConfig_.httpConfig.port )
      trace( "starting server https", self.capiConfig_.httpsConfig.enabled, 
             "on port,", self.capiConfig_.httpsConfig.port )

      # Attempt to start or reload nginx
      serviceStarted, serviceError = self.reloadOrRestartService( restart )
      if not serviceStarted:
         self.stopService()
         self.serverStatus_.vrfStatus[ self.vrf_ ].vrfError[ 'nginx' ] = serviceError
         qt0( 'Fail to start nginx service in VRF %s due to %s'
               % ( self.vrf_, serviceError ) )
         return

      self._updateSslServiceStatus( serviceEnabled=True )
      self._updateVrfStatus( self.capiConfig_ )
      trace( "startService exit" )
   
   def stopService( self ):
      trace( "stopService enter" )
      self._updateSslServiceStatus( serviceEnabled=False )
      self._updateVrfStatus()

      try:
         vrfTacRun( self.vrf_, "stop" )
      except Tac.SystemCommandError as e:
         qt0( "error stopping nginx", e )
      except Tac.Timeout:
         # nginx service times out to stop,
         # might due to system slowness during OOM. Therefore retry it later
         trace( "Fail to get service nginx status due to timeout" )
         self.sync()

      trace( "stopService exit" )
   
   def _getVrfEnabledService( self, serverType="" ):
      def _enabledServiceFunc( serviceClass ):
         if serverType == "uds":
            return serviceClass.unixDomainSocketServerEnabled()
         return serviceClass.serviceEnabled( self.vrf_ )

      enabledVrfServices = []
      for s, serviceClass in self.containingAgent_.httpServices.items():
         if _enabledServiceFunc( serviceClass ):
            enabledVrfServices.append( s )
      return enabledVrfServices 

   def _updateVrfStatus( self, config=None ):
      #  Set the status of HttpServer per VRF.
      if self.vrf_ in self.serverStatus_.vrfStatus:
         # if nginx runs in default VRF with only Uds server,
         # then set external and internal server status to False
         enabled = bool( config ) and not self._unixServerEnabled( unixOnly=True )
         unixServerEnabled = bool( config ) and self._unixServerEnabled()
         # get all enabled services in VRF
         enabledServices = self._getVrfEnabledService() if enabled else []
         if unixServerEnabled:
            # if nginx is running with uds enabled, also get services enabled in uds
            enabledServices += self._getVrfEnabledService( serverType="uds" )

         vrfStatus = self.serverStatus_.vrfStatus[ self.vrf_ ]
         vrfStatus.enabled = bool( config )
         vrfStatus.httpsEnabled = config.httpsConfig.enabled if enabled else False
         vrfStatus.httpEnabled = config.httpConfig.enabled if enabled else False
         vrfStatus.httpLocalEnabled = ( config.localHttpConfig.enabled 
                                                            if enabled else False )
         vrfStatus.unixEnabled = ( config.unixConfig.enabled 
                                                   if unixServerEnabled else False )
         vrfStatus.vrfError[ 'nginx' ] = ''
         if self.vrf_ == DEFAULT_VRF:
            # default VRF status depends on enabled services that configures it
            # specifically, has no VRF configuration or has uds enabled
            vrfStatus.vrfService.clear()
            for s in enabledServices:
               # skip implicitly enabled services 
               if ( s in self.capiConfig_.service and
                    self.capiConfig_.service[ s ].enabled ):
                  vrfStatus.newVrfService( s ).enabled = True
         else:
            for ( s, sStatus ) in vrfStatus.vrfService.items():
               sStatus.enabled = s in enabledServices
  
   def _updateSslServiceStatus( self, serviceEnabled ):
      if self.capiConfig_.httpsConfig.enabled and serviceEnabled:
         # update ssl service status when https is enabled and 
         # nginx is successfully running
         self.sslServiceStatus_.profileName = self.capiConfig_.httpsConfig.sslProfile
         self.sslServiceStatus_.certificate = self._getSslServiceCert()
      else:
         self.sslServiceStatus_.profileName = ''
         self.sslServiceStatus_.certificate = ''

   def serviceCmd( self, cmd ):
      cmdline = [ "service", "nginx", cmd ]
      if self.vrf_ != DEFAULT_VRF:
         cmdline.append( self.vrf_ )
      return cmdline

   def checkServiceCmdOutput( self, cmd, output ):
      if cmd == 'status':
         return ( not output or
                  any(  "active (running)" in line or
                         re.match( r'^nginx \(pid  .*\) is running...', line )
                         for line in output  ) )
      return SuperServer.LinuxService.checkServiceCmdOutput( self, cmd, output )

   def _checkCipherConfig( self ):
      # If we're in the "old mode", capi ciphers/mac/key-exchange can still be 
      # configured. If someone configures ciphersuites in the SSL profile as well,
      # we don't know what they really want, so give up and return an error.
      # This is different from being in the "new mode", where we can't actually
      # configure capi ciphers 
      status = self._getProfileStatus()
      defaultCipherSuite = MGMT_SECURITY_SSL_CONSTANTS.defaultCipherSuite() 
      if ( self.capiConfig_.httpsConfig.hasCipherConfig or
           self.capiConfig_.httpsConfig.hasKeyExchangeConfig or
           self.capiConfig_.httpsConfig.hasMacConfig ):
         if ( status and status.state == "valid" and
              status.cipherSuite != defaultCipherSuite ):
            Logging.log( CAPI_CIPHERSUITE_AND_HTTPS_CONFIG_INCOMPATIBLE )
            return 'SSL profile ciphersuite and capi ciphers are both configured.'
      return ''

   def _checkCert( self ):
      """ Validates the certificate """
      trace( "_checkCert enter" )
      if self.serverStatus_.sslCertLastError:
         return 'SSL certificate error: %s' % self.serverStatus_.sslCertLastError
      if self.serverStatus_.sslKeyLastError:
         return 'SSL key error: %s' % self.serverStatus_.sslKeyLastError
     
      # find out the cert/key that capi is using
      status = self._getProfileStatus()
      certFilepath, keyFilepath = self._getServerCertKeyFilepaths()
      
      if( self.capiConfig_.httpsConfig.sslProfile and 
          status and certFilepath == status.certKeyPath ):
         # we have a valid ssl profile configured
         return ''

      # valid the content of cert/key
      errorString = HttpServiceSsl.validateCertificateAndKeyFiles( certFilepath,
                                                                   keyFilepath )
      if ( errorString and 
           not self.capiConfig_.httpsConfig.sslCertificate and
           not self.capiConfig_.httpsConfig.sslKey ):
         # if there is no customer certificate configured,
         # and the build-in cert/key are invalid, then regenerate a new one
         # for nginx to keep running properly
         self.containingAgent_.generateCert()
         return ''
      else:
         return errorString

   def _getHttpConfig( self ):
      if not self.capiConfig_.httpConfig.enabled:
         return ""
      
      return "listen [::]:%s ipv6only=off%s;" % ( self.capiConfig_.httpConfig.port,
                                                  self._getIpTosConfig() )
   
   def _getHttpsConfig( self ):
      if not self.capiConfig_.httpsConfig.enabled:
         return ""
      
      return ( "listen [::]:%s ssl ipv6only=off%s;" % 
               ( self.capiConfig_.httpsConfig.port, self._getIpTosConfig() ) )

   def _getFipsMode( self ):
      status = self._getProfileStatus()
      return status.fipsMode if status else False

   def _getFipsModeStr( self ):
      fipsOn = self._getFipsMode()
      return "on" if fipsOn else "off"

   def _getFipsLoggingStr( self ):
      fipsLoggingOn = self.mgmtSecurityConfig_.fipsLogging
      return "on" if fipsLoggingOn else "off"

   def _getDhparams( self ):
      status = self._getProfileStatus()
      if not status:
         return MGMT_SECURITY_SSL_CONSTANTS.dhParamPath()
      return MGMT_SECURITY_SSL_CONSTANTS.namedDhparamPath( status.dhparam )

   def _getLocalHttpConfig( self ):
      return """listen [::1]:%s;
       listen 127.0.0.1:%s;""" % ( self.capiConfig_.localHttpConfig.port,
                                   self.capiConfig_.localHttpConfig.port )

   def _getAntiClickJackingHeaders( self ):
      if not self.capiConfig_.contentFrameAncestor:
         return """add_header 'X-Frame-Options' 'DENY';
          add_header 'Content-Security-Policy' 'frame-ancestors \\'none\\'';"""
      else:
         return """if ($unsupportedIframing = 1) {{
             add_header 'X-Frame-Options' 'DENY';
          }}

          if ($unsupportedIframing = 0) {{
             add_header 'X-Frame-Options' 'ALLOW-FROM {uri}';
             add_header 'Content-Security-Policy' 'frame-ancestors {uri}';
          }}""".format( uri=self.capiConfig_.contentFrameAncestor )

   def _getIpTosConfig( self ):
      if self.capiConfig_.qosDscp == 0:
         return ""
      else:
         tosVal = self.capiConfig_.qosDscp << 2
         return " ip_tos=0x%s" % format( tosVal, '02x' )

   def _getProfileStatus( self ):
      if ( self.containingAgent_.sslReactor and 
           self.containingAgent_.sslReactor.profileStatus_ ):
         return self.containingAgent_.sslReactor.profileStatus_
      return None
   
   def _getSslProtocols( self ):
      enabledVersions = []
      status = self._getProfileStatus()
      tlsVersionMap = [ ( MGMT_SECURITY_SSL_CONSTANTS.tlsv1, 'TLSv1' ),
                        ( MGMT_SECURITY_SSL_CONSTANTS.tlsv1_1, 'TLSv1.1' ),
                        ( MGMT_SECURITY_SSL_CONSTANTS.tlsv1_2, 'TLSv1.2' ),
                        ( MGMT_SECURITY_SSL_CONSTANTS.tlsv1_3, 'TLSv1.3' ) ]
      for mask, version in tlsVersionMap:
         if not status or status.tlsVersion & mask:
            enabledVersions.append( version )
      return ' '.join( enabledVersions )

   def _getSslCiphersConfig( self ):
      if self.serverStatus_.cipherFilter:
         ciphers = self.serverStatus_.cipherFilter
      else:
         ciphers = MGMT_SECURITY_SSL_CONSTANTS.defaultCipherSuite()
      cipherConfig = f'ssl_ciphers {ciphers};'
      
      status = self._getProfileStatus()
      if status and status.state == 'valid':
         cipherConfig += f"""
       ssl_conf_command Ciphersuites {status.cipherSuiteV1_3};"""

      return cipherConfig

   def _getHttpServiceConfig( self, serverType ):
      block = ""
      if serverType not in [ 'external', 'internal', 'uds', 'virtual' ]:
         return block
      for service in self._getVrfEnabledService( serverType ):
         serviceClass = self.containingAgent_.httpServices[ service ]
         serviceConfig = ""
         if serverType == 'external':
            serviceConfig = serviceClass.externalServerConfig( self.vrf_ )
         elif serverType == 'internal':
            serviceConfig = serviceClass.internalServerConfig( self.vrf_ )
         elif serverType == 'uds':
            serviceConfig = serviceClass.unixDomainSocketServerConfig()
         elif serverType == 'virtual':
            serviceConfig = serviceClass.virtualServerConfig()
            # service-specific server block
            serviceConfig += self.capiConfig_.service[ service ].configServerBlock
         block += serviceConfig
      return block

   def _getRootLocation( self ):
      if self.capiConfig_.defaultServicesEnabled:
         return """
       location / {{
          include uwsgi_params;
          uwsgi_param  SSL_CLIENT_VERIFY  $ssl_client_verify;
          uwsgi_param  SSL_CLIENT_S_DN    $ssl_client_s_dn_legacy;
          uwsgi_param  SSL_CLIENT_SPIFFE  $ssl_client_spiffe;
          uwsgi_param  SERVER_ADDR        $server_addr;
          uwsgi_param  VRF_NAME           {vrfName};
          uwsgi_read_timeout 60m;
          uwsgi_send_timeout 60m;
          uwsgi_pass unix://{uwsgiSocket};
          uwsgi_intercept_errors on;
          error_page 444 @drop;         
          # Define headers to prevent click-jacking attempts
          {antiClickJackingHeaders}
          {securityHeaders}
       }}""".format( vrfName=self.vrf_, uwsgiSocket="/var/run/capi.uwsgi.sock",
                     antiClickJackingHeaders=self._getAntiClickJackingHeaders(),
                     securityHeaders=ServerConstants.nginxSecurityHeaders(10) )

      else:
         return """
       location / {
          return 404;
       }"""
 
   def _externalServerBlock( self ):
      if self._unixServerEnabled( unixOnly=True ):
         # start default VRF only for uds
         return ""
      if ( self.capiConfig_.httpsConfig.enabled or 
           self.capiConfig_.httpConfig.enabled ):
         return """
    server {{
       {httpServer}
       {httpsServer}
       {securityHeaders}
 
       {rootLocationBlock}
              
       {locationEndpoints}       

       location @drop {{
          return 444;
       }}

       # SSL Configuration
       {serverCertConfiguration}
       {sslCiphersConfiguration}
       {clientCertConfiguration}
       ssl_verify_client {clientVerify};
       ssl_prefer_server_ciphers on;
       ssl_dhparam {dhParams};
       ssl_session_cache shared:SSL:1m;
       ssl_session_timeout 60m;
       ssl_protocols {sslProtocols};
       # prevent CVE-2014-3567
       ssl_session_tickets off;
       ssl_fips_mode {fipsMode};
       ssl_fips_logging {fipsLogging};
       ssl_enforce_cert_extended_attributes {fipsMode};
       ssl_enforce_client_cert_username {fipsMode};
       ssl_handle_client_cert_failures_tls {fipsMode};

       # Set keepalive for long-lived connections
       keepalive_requests 72000; 
       keepalive_timeout 3600s; # 1 hour
       
       # Turn on gzip for larger command outputs
       gzip on;
       gzip_min_length 8192;
       gzip_types application/json;

    }}""".format( httpServer=self._getHttpConfig(), 
            httpsServer=self._getHttpsConfig(),
            securityHeaders=ServerConstants.nginxSecurityHeaders(7),
            rootLocationBlock=self._getRootLocation(),
            locationEndpoints=self._getHttpServiceConfig( 'external' ),
            serverCertConfiguration=self._getServerCertConfiguration(),
            sslCiphersConfiguration=self._getSslCiphersConfig(),
            fipsMode=self._getFipsModeStr(),
            fipsLogging=self._getFipsLoggingStr(),
            clientCertConfiguration=self._getClientCertConfiguration(),
            clientVerify=self._getClientCertVerify(),
            dhParams=self._getDhparams(),
            sslProtocols=self._getSslProtocols() )

      else:
         return ""
      
   def _internalServerBlock( self ):
      if self._unixServerEnabled( unixOnly=True ):
         # start default VRF only for uds
         return ""

      # pylint: disable-next=superfluous-parens
      if ( self.capiConfig_.localHttpConfig.enabled ):
         return """
    server {{
       {localHttpServer}

       {localHttpEndpoints}

       server_name localhost;
    }}""".format( localHttpServer=self._getLocalHttpConfig(),
                  localHttpEndpoints=self._getHttpServiceConfig( 'internal' ) )
      else:
         return ""

   def _unixDomainSocketServerBlock( self ):
      serviceUdsConfig = ''
      if self._unixServerEnabled():
         serviceUdsConfig = self._getHttpServiceConfig( 'uds' )
      
      return """
    server {{
       listen unix:{statsSocket};
       server_name localhost;

       location /nginx_status {{
           stub_status on;
           access_log off;
       }}

    }}
    {serviceUdsConf}""".format( 
          statsSocket=ServerConstants.NGINX_STATUS_UDS % self.vrf_,
          serviceUdsConf=serviceUdsConfig )

   def _virtualServerBlock( self ):
      return self._getHttpServiceConfig( 'virtual' )

   def _syslogErrorBlock( self ):
      if self.capiConfig_.syslogLevel != LOG_LEVELS.none:
         return """
   error_log /var/log/nginx-error.log %s;
   error_log syslog:server=unix:/dev/log,nohostname %s;
""" % ( self.capiConfig_.syslogLevel, self.capiConfig_.syslogLevel )
      else:
         return "   error_log /var/log/nginx-error.log;"

   def conf( self ):
      config = r"""
worker_processes  1;
events {
   worker_connections 1024;
}
http {
   include mime.types;
   default_type  application/octet-stream;
   server_tokens off;
   sendfile on;
   log_format  main  '$remote_addr - $remote_user [$time_local] "$request" '
                     '$ssl_protocol/$ssl_cipher '
                     '$status $body_bytes_sent "$http_referer" '
                     '"$http_user_agent" "$http_x_forwarded_for" '
                     '"$http_origin"';
   access_log /var/log/nginx-access.log main;
%s

   map $http_user_agent $unsupportedIframing {
      default                                  0;
      "~AppleWebKit.*Version/[0-9]*\..*Safari" 1;
   }

   map "" $nginx_namespace {
      default                                   %s;
   }

%s
%s
%s
%s
}

""" % ( self._syslogErrorBlock(), self.namespace_, 
        self._externalServerBlock(), self._internalServerBlock(), 
        self._unixDomainSocketServerBlock(), self._virtualServerBlock() )
      debug( "nginx config", config )
      return config
   
   def _isInvalidSslCertKeyPair( self ):
      """ If we have https configured, but no certificate configured then 
      that is an error """
      debug( "_isInvalidSslCertKeyPair https is ", 
             self.capiConfig_.httpsConfig.enabled, "and our server config is:",
             self._getServerCertConfiguration() )
      return ( self.capiConfig_.httpsConfig.enabled and 
               self._getServerCertConfiguration() == "" )

   def _getClientCertConfiguration( self ):
      status = self._getProfileStatus()
      conf = ''
      if status and status.state == "valid" and status.trustedCertsPath:
         conf += """ssl_verify_depth 100;
       ssl_client_certificate %s;""" % status.trustedCertsPath
         if status.crlsPath:
            conf += '\n       ssl_crl %s;' % status.crlsPath
      return conf

   def _getClientCertVerify( self ):
      status = self._getProfileStatus()
      if status and status.state == "valid" and status.trustedCertsPath:
         return 'optional'
      return 'off'

   def _getServerCertConfiguration( self ):
      certTemplate = """
       ssl_certificate %s;
       ssl_certificate_key %s;"""
       
      filePaths = self._getServerCertKeyFilepaths()
      if not filePaths:
         return ""
      ( certFilepath, keyFilepath ) = filePaths
      return certTemplate % ( certFilepath, keyFilepath )

   def _getServerCertKeyFilepaths( self ):
      status = self._getProfileStatus()
      # we don't have a valid capi cert
      if not self.capiConfig_.httpsConfig.sslCertificate:
         # if we have an ssl profile configured
         if self.capiConfig_.httpsConfig.sslProfile: 
            # we have valid ssl profile
            if status and status.state == "valid" and status.certKeyPath and \
                  ( ( not status.verifyExtendedParameters ) or \
                     status.isServerCert ):
               return ( status.certKeyPath, status.certKeyPath )
            else: # we have an invalid ssl profile
               return ""
         else: # we don't have an ssl profile configured
            return ( HttpServiceSsl.getCertFilepath(), 
                     HttpServiceSsl.getKeyFilepath() )
          
      else: # we have a valid capi cert, always use it!
         return ( HttpServiceSsl.getCertFilepath(), 
                  HttpServiceSsl.getKeyFilepath() )

   def _getSslServiceCert( self ):
      if not self.sslServiceStatus_.profileName:
         return ""
      filePaths = self._getServerCertKeyFilepaths()
      if not filePaths:
         return ""
      ( certFilepath, _ ) = filePaths
      return SslCertKey.getCertPem( certFilepath )

class FipsLoggingReactor( Tac.Notifiee ):
   """ Adjust nginx config in response to MgmtSecurity config changes. """

   notifierTypeName = 'Mgmt::Security::Config'

   def __init__( self, mgmtSecurityConfig, containingAgent ):
      self.containingAgent = containingAgent
      super().__init__( mgmtSecurityConfig )
   
   @Tac.handler( 'fipsLogging' )
   def handleFipsLogging( self ):
      if self.containingAgent.capiConfig.httpsConfig.enabled:
         self.containingAgent.restartServer()

class HttpConfigReactor( Tac.Notifiee ):
   """ Adjusts daemon status in response to HTTP config changes. """

   notifierTypeName = 'HttpService::HttpConfig'

   def __init__( self, httpConfig, containingAgent ):
      self.containingAgent = containingAgent
      super().__init__( httpConfig )

   @Tac.handler( 'enabled' )
   def handleEnabled( self ):
      self.containingAgent.restartServer()

   @Tac.handler( 'port' )
   def handlePort( self ):
      if self.notifier_.enabled:
         self.containingAgent.restartServer()
         
class HttpLocalConfigReactor( Tac.Notifiee ):
   """ Adjusts daemon status in response to HTTP config changes. """

   notifierTypeName = 'HttpService::HttpLocalConfig'

   def __init__( self, httpLocalConfig, containingAgent ):
      self.containingAgent = containingAgent
      super().__init__( httpLocalConfig )

   @Tac.handler( 'enabled' )
   def handleEnabled( self ):
      self.containingAgent.restartServer()

   @Tac.handler( 'port' )
   def handlePort( self ):
      if self.notifier_.enabled:
         self.containingAgent.restartServer()
         
class UnixConfigReactor( Tac.Notifiee ):
   """ Adjusts daemon status in response to unix socket config changes. """

   notifierTypeName = 'HttpService::UnixConfig'

   def __init__( self, UnixConfig, containingAgent ):
      self.containingAgent = containingAgent
      super().__init__( UnixConfig )

   @Tac.handler( 'enabled' )
   def handleEnabled( self ):
      self.containingAgent.restartServer()

def getFilterList( cliKeywords, filterMappings ):
   """ Converts the configuration to the list of filters understood by openssl.
   Configuration order is preserved. """

   filterStrings = []
   for keyword in cliKeywords:
      filterStrings.append( filterMappings[ keyword ][ "filter" ] )

   if not filterStrings:
      filterStrings = [ filterMap[ "filter" ]
                        for filterMap in filterMappings.values() ]

   trace( "getFilterList ", filterStrings )
   return filterStrings

def filterWithSeperator( filterStr ):
   return filterStr + "-" if filterStr else ""

def getCipherFilter( httpsConfig, sslReactor ):
   """ If none of the filters configured, use the SSL profile ciphersuites if valid.
   If the profile is invalid, returns None so that the default capi
   cipherSuites are enabled. Otherwise capi configured cipherSuites are enabled """

   if not httpsConfig.ciphers and not httpsConfig.keyExchange \
         and not httpsConfig.mac:
      if not httpsConfig.hasCipherConfig and not httpsConfig.hasKeyExchangeConfig \
            and not httpsConfig.hasMacConfig:
         if ( sslReactor and sslReactor.profileStatus_ and 
              sslReactor.profileStatus_.state == 'valid' ):
            cipherSuite = sslReactor.profileStatus_.cipherSuite
            trace( "Using cipherSuites : ", cipherSuite )
            return Tac.run( [ 'openssl', 'ciphers', cipherSuite ], 
                            stdout=Tac.CAPTURE ).strip()
      return ""

   cipherList = getFilterList( list( httpsConfig.ciphers.values() ),
                               CapiCipherSuites.CIPHERS )
   keyExchangeList = getFilterList( list( httpsConfig.keyExchange.values() ), 
                                    CapiCipherSuites.KEYEXCHANGE )
   macList = getFilterList( list( httpsConfig.mac.values() ), CapiCipherSuites.MAC )

   cipherSuites = \
      [ filterWithSeperator( keyExchange ) + filterWithSeperator( cipher ) + mac
        for keyExchange in keyExchangeList 
        for cipher in cipherList
        for mac in macList ]
   trace( "Using cipherSuites : ", cipherSuites )
   cipherSuites = ":".join( cipherSuites )
   trace( "Using cipherSuites : ", cipherSuites )
   return cipherSuites

class HttpsConfigReactor( Tac.Notifiee ):
   """ Adjusts daemon status in response to HTTPS config changes. """

   notifierTypeName = 'HttpService::HttpsConfig'

   def __init__( self, httpsConfig, containingAgent ):
      self.containingAgent = containingAgent
      super().__init__( httpsConfig )

   @Tac.handler( 'enabled' )
   def handleEnabled( self ):
      trace( 'HttpsConfigReactor.handleEnabled' )
      self.containingAgent.restartServer()

   @Tac.handler( 'port' )
   def handlePort( self ):
      self._doMaybeRestartServer()

   @Tac.handler( 'sslCertificate' )
   def handleSslCert( self ):
      trace( 'HttpsConfigReactor.handleSslCert' )
      if self.notifier_.sslCertificate:
         self.containingAgent.updateSslCert( self.notifier_.sslCertificate )
      else:
         self.containingAgent.generateCert()
      self._doMaybeRestartServer()

   @Tac.handler( 'sslKey' )
   def handleSslKey( self ):
      trace( 'HttpsConfigReactor.handleSslKey' )
      if self.notifier_.sslKey:
         self.containingAgent.updateSslKey( self.notifier_.sslKey )
      self._doMaybeRestartServer()

   @Tac.handler( 'sslProfile' )
   def handleSslProfile( self ):
      trace( 'HttpsConfigReactor.handleSslProfile' )
      self.containingAgent.initSslReactor()
      self._updateCipherFilter()

   @Tac.handler( 'ciphers' )
   def handleCiphers( self, _ ):
      trace( 'HttpsConfigReactor.handleCiphers' )
      self._updateCipherFilter()

   @Tac.handler( 'keyExchange' )
   def handleKeyExchange( self, _ ):
      self._updateCipherFilter()

   @Tac.handler( 'mac' )
   def handleMac( self, _ ):
      self._updateCipherFilter()

   @Tac.handler( 'hasCipherConfig' )
   def handleHasCipherConfig( self ):
      self._updateCipherFilter()

   @Tac.handler( 'hasKeyExchangeConfig' )
   def handleHasKeyExchangeConfig( self ):
      self._updateCipherFilter()

   @Tac.handler( 'hasMacConfig' )
   def handleHasMacConfig( self ):
      self._updateCipherFilter()

   def _updateCipherFilter( self ):
      self.containingAgent.serverStatus.cipherFilter = getCipherFilter(
            self.containingAgent.capiConfig.httpsConfig,
            self.containingAgent.sslReactor )
      self._doMaybeRestartServer()

   def _doMaybeRestartServer( self ):
      if self.notifier_.enabled:
         # Only restart if we, the HTTPS server, is enabled.
         self.containingAgent.restartServer()

class ServiceConfigReactor( Tac.Notifiee ):
   """Adjusts daemon status in response to service config changes."""

   notifierTypeName = 'HttpService::ServiceConfig'

   def __init__( self, serviceConfig, containingAgent ):
      self.containingAgent = containingAgent
      Tac.Notifiee.__init__( self, serviceConfig )

   @Tac.handler( 'enabled' )
   def handleEnabled( self ):
      self.containingAgent.restartServer()

   @Tac.handler( 'configServerBlock' )
   def handleConfigServerBlock( self ):
      self.containingAgent.restartServer()

class ServerConfigReactor( Tac.Notifiee ):

   """Adjusts daemon status
   in response to config changes."""

   notifierTypeName = 'HttpService::Config'

   def __init__( self, serverConfig, containingAgent ):
      self.containingAgent = containingAgent
      Tac.Notifiee.__init__( self, serverConfig )

   @Tac.handler( 'enabled' )
   def handleEnabled( self ):
      self.containingAgent.restartServer()

   @Tac.handler( 'qosDscp' )
   def handleQosDscp( self ):
      self.containingAgent.restartServer()

   @Tac.handler( 'syslogLevel' )
   def handleSyslogLevel( self ):
      self.containingAgent.restartServer()

   @Tac.handler( 'defaultServicesEnabled' )
   def handleDefaultServicesEnabled( self ):
      self.containingAgent.restartServer()

   @Tac.handler( 'service' )
   def handleService( self, key ):
      self.containingAgent.restartServer()

# pylint: disable-next=inconsistent-return-statements
def createNginxService( vrf, agent ):
   if vrf not in agent.nginxServices_: 
      trace( 'create nginx for VRF', vrf )
      service = HttpService( agent, vrf )
      agent.nginxServices_[ vrf ] = service
      return service

def cleanUpNginxService( vrf, agent ):
   if vrf in agent.nginxServices_: 
      trace( 'clean up nginx for VRF', vrf )
      # pylint: disable-msg=W0212
      agent.nginxServices_[ vrf ]._maybeRestartService()
      agent.nginxServices_[ vrf ].cleanupService()
      del agent.nginxServices_[ vrf ]
      cleanupNginx( vrf )

class VrfStateReactor( Tac.Notifiee ):
   """ Adjusts daemon status
       in response to VRF local status. """

   notifierTypeName = 'Ip::VrfStatusLocal'

   def __init__( self, vrfStatus, containingAgent ):
      self.containingAgent = containingAgent
      Tac.Notifiee.__init__( self, vrfStatus )
      self.handleState()

   @Tac.handler( 'state' )
   def handleState( self ):
      vrf =  self.notifier_.vrfName
      state = self.notifier_.state
      agent = self.containingAgent
      if vrf in agent.capiConfig.vrfConfig:
         if state == 'active':
            trace( 'VRF moved to active state. Starting Capi in VRF', vrf )
         elif state == 'deleting':
            trace( 'VRF deleting. Stop Capi in VRF', vrf )
         agent.restartServer()

class VrfConfigReactor( Tac.Notifiee ):
   """ Adjusts daemon status 
       in response to http-server or http service VRF configuration. """

   notifierTypeName = "HttpService::VrfConfig"

   def __init__( self, vrfConfig, containingAgent ):
      self.containingAgent = containingAgent
      Tac.Notifiee.__init__( self, vrfConfig )
      self.reactorSetUp()

   def close( self ):
      trace( "VrfConfigReactor.close", self.notifier_.name )
      if self.notifier_.name != DEFAULT_VRF:
         cleanUpNginxService( self.notifier_.name, self.containingAgent )
         del self.containingAgent.serverStatus.vrfStatus[ self.notifier_.name ]
      self.containingAgent.restartServer()
      Tac.Notifiee.close( self )
   
   def reactorSetUp( self ):
      vrf = self.notifier_.name
      createNginxService( vrf, self.containingAgent )
      self.containingAgent.serverStatus.newVrfStatus( vrf )
      for s in self.notifier_.vrfService:
         trace( 'VrfConfigReactor.reactorSetUp', s, 'create notifiee' )
         self.containingAgent.serverStatus.vrfStatus[ vrf ].newVrfService( s )
      self.containingAgent.restartServer()

   @Tac.handler( 'serverState' )
   def handleServerState( self ):
      self.containingAgent.restartServer()
   
   @Tac.handler( 'vrfService' )
   def handleVrfService( self, key ):
      vrf = self.notifier_.name
      agent = self.containingAgent
      if key in self.notifier_.vrfService:
         trace( 'VrfConfigReactor.handleVrfService', key, 'create notifiee' )
         agent.serverStatus.vrfStatus[ vrf ].newVrfService( key )
      elif key in agent.serverStatus.vrfStatus[ vrf ].vrfService:
         trace( 'VrfConfigReactor.handleVrfService', key, 'remove notifiee' )
         del agent.serverStatus.vrfStatus[ vrf ].vrfService[ key ]
      else:
         return
      agent.restartServer()

class Capi( SuperServer.SuperServerAgent ):
   def __init__( self, entityManager ):
      trace( 'Capi.doInit entry sysname',  entityManager.sysname() )
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      self.entityManager = entityManager
      self.httpConfigReactor_ = None
      self.httpsConfigReactor_ = None
      self.httpLocalConfigReactor_ = None
      self.unixConfigReactor_ = None
      self.serverConfigReactor_ = None
      self.serviceConfigReactor_ = None
      self.warm_ = False
      self.onSwitchoverDone_ = False
      self.nginxServices_ = {}
      self.allVrfStateReactor_ = None
      self.vrfConfigReactor_ = None
      self.fipsLoggingReactor_ = None
      self.httpServices = {}
      self.sslReactor = None
      self.fipsLoggingReactor = None
      self.sslDhparamsReactor = None
      
      trace( 'Capi.doInit preparing mount group' )
      mountGroup = entityManager.mountGroup()
      self.capiConfig = mountGroup.mount( 'mgmt/capi/config', 
                                          'HttpService::Config', 'r' )
      self.serverStatus = mountGroup.mount( 'mgmt/httpserver/status', 
                                            'HttpService::Status', 'w' )
      self.allVrfStatus = mountGroup.mount( Cell.path( "ip/vrf/status/local" ), 
                                            'Ip::AllVrfStatusLocal', 'r' )
      self.sslConfig = mountGroup.mount( 'mgmt/security/ssl/config',
                                         'Mgmt::Security::Ssl::Config', 'r' )
      self.sslStatus = mountGroup.mount( 'mgmt/security/ssl/status',
                                         'Mgmt::Security::Ssl::Status', 'r' )
      self.sslServiceStatus = mountGroup.mount( 
                                    'mgmt/security/ssl/serviceStatus/httpServer',
                                    'Mgmt::Security::Ssl::ServiceStatus', 'w' )
      self.mgmtSecurityConfig = mountGroup.mount(
                                    'mgmt/security/config',
                                    'Mgmt::Security::Config', 'r' )
      
      def _mountDone():
         # run only if active
         if self.active():
            self.onSwitchover( None )
      
      trace( 'closing mountgroup' )
      mountGroup.close( _mountDone )
      trace( 'Capi.__init__ exit' )

   def registerHttpService( self, serviceClass ):
      self.httpServices[ serviceClass.service ] = serviceClass

   def onSwitchover( self, protocol ):
      """The _mountDone method is called when the mount group close is done."""
      trace( 'Capi._mountDone entry' )

      # Load HttpServicePlugins
      class Context:
         def __init__( self, entMan, serverConfig, serverStatus, registerServiceFn ):
            self.entityManager = entMan
            self.serverConfig = serverConfig
            self.serverStatus = serverStatus
            setattr( self, "registerService", registerServiceFn )

      ctx = Context( self.entityManager, self.capiConfig, 
                     self.serverStatus, self.registerHttpService )
      Plugins.loadPlugins( 'HttpServicePlugin', context=ctx )
   
      trace( 'updating certificate' )
      self.initSslReactor()
      if ( self.capiConfig.httpsConfig.sslCertificate and
           self.capiConfig.httpsConfig.sslKey ):
         self.updateSslCert( self.capiConfig.httpsConfig.sslCertificate )
         self.updateSslKey( self.capiConfig.httpsConfig.sslKey )
      elif HttpServiceSsl.validateCertificateAndKeyFiles( 
                                                   HttpServiceSsl.getCertFilepath(), 
                                                   HttpServiceSsl.getKeyFilepath() ):
         self.generateCert()

      self.serverStatus.cipherFilter = getCipherFilter( self.capiConfig.httpsConfig,
                                                        self.sslReactor )
   
      trace( 'instantating server configuration reactors' )
      self.httpConfigReactor_ = HttpConfigReactor( self.capiConfig.httpConfig,
                                                   self )
      self.httpsConfigReactor_ = HttpsConfigReactor( self.capiConfig.httpsConfig,
                                                     self )
      self.httpLocalConfigReactor_ = HttpLocalConfigReactor( 
                                         self.capiConfig.localHttpConfig, self )
      self.unixConfigReactor_ = UnixConfigReactor( self.capiConfig.unixConfig, 
                                                   self  )
      self.serverConfigReactor_ = ServerConfigReactor( self.capiConfig, self )
      self.serviceConfigReactor_ = \
         Tac.collectionChangeReactor( self.capiConfig.service, ServiceConfigReactor,
                                      reactorArgs=( self, ) )
   
      self.cleanupOldNginxs()
      self.nginxServices_[ DEFAULT_VRF ] = HttpService( self )
      self.allVrfStateReactor_ = Tac.collectionChangeReactor( 
                                       self.allVrfStatus.vrf, VrfStateReactor,
                                       reactorArgs=( self, ) )
      self.vrfConfigReactor_ = Tac.collectionChangeReactor( 
                                       self.capiConfig.vrfConfig, VrfConfigReactor,
                                       reactorArgs=( self, ) )
      self.fipsLoggingReactor_ = FipsLoggingReactor( self.mgmtSecurityConfig, self )
      self.sslDhparamsReactor = GenericReactor( self.sslStatus,
                                                [ 'dhparamsResetProcessed' ],
                                                self.handleDhparamsReset )

      self.onSwitchoverDone_ = True
      self.restartServer()
      self.warm_ = True
      trace( 'Capi._mountDone exit' )

   def handleDhparamsReset( self, n=None ):
      trace( 'Capi.handleDhparamsReset' )
      if self.capiConfig.httpsConfig.enabled:
         self.restartServer()
      
   def cleanupOldNginxs( self ):
      # Find out which files/processes refer to VRFs that 
      # have been removed or are inactive and clean them up.
      trace( 'Capi.cleanupOldNginxs' )
      def addVrfIfInactive( vrfSet, vrf, allVrfStatus ):
         # If VRF has been removed or VRF's state is not active, 
         # clean up the nginx before initializing Capi
         if ( vrf not in allVrfStatus.vrf or 
               allVrfStatus.vrf[ vrf ].state != 'active' ):
            vrfSet.add( vrf )
      
      removedVrfs = set()
      # nginx.conf files
      for fileName in os.listdir( '/etc/nginx' ):
         m = re.search( r'^nginx-([^\s]+).conf', fileName )
         if m:
            try:
               addVrfIfInactive( removedVrfs, m.group( 1 ), self.allVrfStatus )
            except OSError as e:
               if e.errno == errno.ENOSPC:
                  service = self.service_ # pylint: disable-msg=no-member
                  Logging.log( SuperServer.SYS_SERVICE_FILESYSTEM_FULL, fileName,
                               service.serviceName_ )
                  return
               else:
                  raise
      # nginx pid
      for vrfPidFile in os.listdir( '/var/run' ):
         m = re.search( r'^nginx-([^\s]+).pid', vrfPidFile )
         if m:
            addVrfIfInactive( removedVrfs, m.group( 1 ), self.allVrfStatus )

      for vrfName in removedVrfs:
         cleanupNginx( vrfName )

   def warm( self ):
      if not self.active():
         return True
      # if we have no services, we're not warm
      if len( self.nginxServices_ ) == 0:
         return False
      if not self.warm_:
         return False
      # otherwise we're warm if all services that we do have are warm
      w = True
      for s in self.nginxServices_.values():
         w = w and s.warm()
      return w

   def updateDefaultServiceEnabled( self ):
      """ defaultServiceEnabled is used to
         launch CapiApp to serve static request. """
      trace( 'updateDefaultServiceEnabled enter' )
      defaultServiceEnabled = False
      if self.capiConfig.defaultServicesEnabled:
         # defaultServiceEnabled is enabled if defaultServicesEnabled is True
         # and if any of the services is enabled
         # then launch CapiApp for static request.
         for s in self.nginxServices_.values():
            defaultServiceEnabled |= s.serviceEnabled()

      self.serverStatus.defaultServiceEnabled = defaultServiceEnabled
      trace( 'updateDefaultServiceEnabled exit' )

   def restartServer( self ):
      trace( 'restartServer enter' )
      debug( "service is on:", httpServiceEnabled( self.capiConfig ) )
      debug( "https is on:", self.capiConfig.httpsConfig.enabled )
      debug( "http is on:", self.capiConfig.httpConfig.enabled )
      
      # if nginx is enabled then we expect uwsgi to be enabled as well
      if not self.onSwitchoverDone_:
         trace( 'restartServer exit server not ready yet to restart' )
         return
      if ( self.capiConfig.httpsConfig.enabled and 
           not os.path.isfile( MGMT_SECURITY_SSL_CONSTANTS.dhParamPath() ) ):
         trace( 'restartServer exit dhparams needed for https not ready yet' )
         return
      for s in self.nginxServices_.values():
         s.sync()
      # update http server status based on Capi configuration
      # to determine if CapiApp needs to be launched
      self.updateDefaultServiceEnabled()
      trace( 'restartServer exit' )

   def initSslReactor( self ):
      if self.capiConfig.httpsConfig.sslProfile:
         self.sslReactor = HttpServiceSslReactor( self.sslConfig, self.sslStatus,
                                           self.capiConfig.httpsConfig.sslProfile,
                                           self.capiConfig, self )
      else:
         if self.sslReactor:
            self.sslReactor.close()
         self.sslReactor = None

   def updateSslCert( self, certText ):
      assert self.serverStatus
      error = HttpServiceSsl.writeSslAttrToFile( HttpServiceSsl.getCertFilepath(), 
                                                 certText )
      self.serverStatus.sslCertLastError = error
      if error:
         return

      try:
         SslCertKey.validateCertificate( HttpServiceSsl.getCertFilepath(), 
                                         validateDates=True, maxPemCount=None )
      except SslCertKey.SslCertKeyError:
         self.serverStatus.sslCertLastError = "Invalid Certificate"

   def updateSslKey( self, keyText ):
      error = HttpServiceSsl.writeSslAttrToFile( HttpServiceSsl.getKeyFilepath(), 
                                                 keyText )
      self.serverStatus.sslKeyLastError = error
      if error:
         return

      try:
         SslCertKey.validateRsaPrivateKey( HttpServiceSsl.getKeyFilepath() )
      except SslCertKey.SslCertKeyError:
         self.serverStatus.sslKeyLastError = "Invalid Private Key"

   def generateCert( self ):
      """Generates and installs an SSL certificate."""
      trace( 'generateCert entry' )
      certText = None
      keyText = None
      try:
         certText, keyText = SslCertKey.generateCertificate()
      except SslCertKey.SslCertKeyError as e:
         trace( 'generateCert exit', str( e ) )
         return
      self.updateSslCert( certText )
      self.updateSslKey( keyText )
      trace( 'generateCert exit' )

def Plugin( ctx ):
   ctx.registerService( Capi( ctx.entityManager ) )
