# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import errno
import weakref
import SuperServer
import os
import shutil
import Cell
import QuickTrace
import Tracing
from SuperServerPlugin.IgdUpnpMsg import UpnpdSocket
from IpLibConsts import DEFAULT_VRF
from DeviceNameLib import eosIntfToKernelIntf
from SsdpUtils import ssdpSocketFile
from EntityManager import MountGroup

# pkgdeps: rpm MiniUpnpd

t0 = Tracing.trace0
__defaultTraceHandle__ = Tracing.Handle( 'IgdUpnpServer' )

qv = QuickTrace.Var
qt0 = QuickTrace.trace0
qt1 = QuickTrace.trace1

def vrfNamespace( vrfName ):
   if vrfName == DEFAULT_VRF:
      return DEFAULT_VRF
   return "ns-%s" % vrfName

def confFileName( vrfName ):
   return "/etc/miniupnpd-%s.conf" % vrfNamespace( vrfName )

class MiniUpnpdService( SuperServer.LinuxService ):
   serviceName = "miniupnpd"
   notifierTypeName = 'IgdUpnp::IgdVrfStatus'

   _enabled = True

   def __init__( self, vrfName, upnpConfig, igdConfig, vrfStatus,
                 portMapConfig, portMapStatus ):
      t0( "MiniUpnpdService init ", vrfName )
      self.upnpConfig_ = upnpConfig
      self.igdConfig_ = igdConfig
      self.vrfStatus_ = vrfStatus
      self.portMapConfig_ = portMapConfig
      self.portMapStatus_ = portMapStatus
      self.vrfName_ = vrfName
      self.upnpdSocket_ = UpnpdSocket( vrfName, self.portMapConfig_,
            self.portMapStatus_, self.vrfStatus_ )
      self.configFile_ = confFileName( self.vrfName_ )
      self.configLines_ = self.createConfig()
      self.deleted_ = True

      t0( "IGD_UPNP_TESTING is ", os.getenv( "IGD_UPNP_TESTING" ) )

      if os.getenv( "IGD_UPNP_TESTING" ) == "1":
         return

      if self.vrfName_ != DEFAULT_VRF:
         defaultInitFile = "/etc/init.d/miniupnpd"
         self.serviceName = "{}-{}".format( "miniupnpd", self.vrfName_ )
         vrfInitFile = "/etc/init.d/%s" % self.serviceName
         with open( defaultInitFile ) as f:
            with open( vrfInitFile, "w" ) as d:
               for line in f:
                  if line.startswith( 'NS=' ):
                     line = 'NS="%s"\n' % vrfNamespace( self.vrfName_ )
                  elif line.strip().startswith( 'status miniupnpd' ):
                     line = '       status -p ${pidfile} miniupnpd\n'
                  d.write( line )
         shutil.copymode( defaultInitFile, vrfInitFile )

      SuperServer.LinuxService.__init__( self, self.serviceName,
                                         self.serviceName,
                                         self.igdConfig_,
                                         self.configFile_, sync=False,
                                         asRoot=self.vrfName_ != DEFAULT_VRF )

   def createConfig( self ):
      t0( "confFileName:", self.configFile_ )
      configLines = []

      # UUID
      if self.upnpConfig_.uuid:
         uuid = "uuid=%s" % self.upnpConfig_.uuid.stringValue
         configLines.append( uuid )

      configLines.append(
            "ext_ip=%s" % self.vrfStatus_.externalIpv4Addr )

      # listening_ip
      wanSet = False
      for intf in self.vrfStatus_.interface:
         # TODO: this is just to get the first iteration working
         if not wanSet:
            configLines.append( "ext_ifname=%s" % eosIntfToKernelIntf( intf ) )
            wanSet = True
         listenIp = "listening_ip=%s" % eosIntfToKernelIntf( intf )
         configLines.append( listenIp )

      # http/https port
      upnpPort = "%s_port=%d" % ( self.igdConfig_.protocolAndPort.protocol,
            self.igdConfig_.protocolAndPort.port )
      configLines.append( upnpPort )

      # SSDP socket file
      ssdpSocket = "minissdpdsocket=%s" % ssdpSocketFile( self.vrfName_ )
      configLines.append( ssdpSocket )

      # mininupnpd socket file
      miniupnpdSocket = "miniupnpdsocket=%s" % self.upnpdSocket_.socketPath
      configLines.append( miniupnpdSocket )

      # General config
      configLines.append( "secure_mode=yes" )

      # Use the option to force miniupnpd to check that the internal IP
      # belongs to the internal network (192.168.)
      configLines.append( "enable_ip_check=no" )

      return configLines

   def conf( self ):
      t0( "Returning config in ", self.configFile_ )
      lines = "\n".join( s for s in self.createConfig() )
      return lines

   def serviceProcessWarm( self ):
      return True

   def serviceEnabled( self ):
      return self._enabled

   def disableService( self ):
      self._enabled = False

   def enableService( self ):
      self._enabled = True

   def cleanupService( self ):
      t0( "MiniUpnpService::cleanupService", self.vrfName_ )
      self.upnpdSocket_.close()
      SuperServer.LinuxService.cleanupService( self )
      if self.vrfName_ != DEFAULT_VRF:
         # the init script must have been created
         os.unlink( "/etc/init.d/%s" % self.serviceName )
         # the config file may not exist
         try:
            os.unlink( self.configFile_ )
         except OSError as e:
            if e.errno != errno.ENOENT:
               raise

   def processPortMapStatus( self, portMapKey, status ):
      self.upnpdSocket_.processPortMapStatus( portMapKey, status )

class L3StatusReactor( Tac.Notifiee ):
   notifierTypeName = "L3::Intf::Status"

   def __init__( self, l3Status, ipIntfStatusReactor ):
      t0( "L3StatusReactor::__init__", l3Status.intfId )
      self.ipIntfStatusReactor_ = ipIntfStatusReactor
      Tac.Notifiee.__init__( self, l3Status )

   @Tac.handler( 'vrf' )
   def handleVrf( self ):
      self.ipIntfStatusReactor_.handleVrf()

   @Tac.handler( 'vrfId' )
   def handleVrfId( self ):
      self.ipIntfStatusReactor_.handleVrf()

class NatPoolConfigReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::Nat::PoolConfig'

   def __init__( self, notifier, igdUpnpServer ):
      t0( "NatPoolConfigReactor::__init__" )
      self.igdUpnpServer_ = weakref.proxy( igdUpnpServer )
      Tac.Notifiee.__init__( self, notifier )

   @Tac.handler( 'poolRange' )
   def handlePoolRange( self, poolRange ):
      t0( "NatPoolConfigReactor::handlePoolRange ", poolRange )
      self.igdUpnpServer_.sync()

   def close( self ):
      t0( "NatPoolConfigReactor::close" )
      self.igdUpnpServer_.sync()
      Tac.Notifiee.close( self )

class NatIntfConfigReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::Nat::IntfConfig'

   def __init__( self, notifier, igdUpnpServer ):
      t0( "NatIntfConfigReactor::__init__" )
      self.igdUpnpServer_ = weakref.proxy( igdUpnpServer )
      Tac.Notifiee.__init__( self, notifier )

   @Tac.handler( 'dynamicNat' )
   def handleDynamicNat( self, dynamicKey ):
      t0( "NatIntfConfigReactor::handleDynamicNat: ", self.notifier_.profile,
            self.notifier_.vrfName )
      self.igdUpnpServer_.sync()

   def close( self ):
      t0( "NatIntfConfigReactor::close" )
      self.igdUpnpServer_.sync()
      Tac.Notifiee.close( self )

class IntfStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::IpIntfStatus'

   def __init__( self, notifier, igdUpnpServer ):
      t0( "IntfStatusReactor::__init__" )
      self.igdUpnpServer_ = weakref.proxy( igdUpnpServer )
      Tac.Notifiee.__init__( self, notifier )
      self.l3StatusReactor_ = L3StatusReactor( self.notifier_.l3Status,
                                               weakref.proxy( self ) )
      self.handleVrf()

   def handleVrf( self ):
      t0( "IntfStatusReactor::handleVrf vrf changed: intf:",
            self.notifier_.intfId, " vrf:", self.notifier_.vrf,
            " isUpnpVrf:", self.igdUpnpServer_.isUpnpVrf( self.notifier_.vrf ) )
      if self.igdUpnpServer_.isUpnpVrf( self.notifier_.vrf ):
         self.igdUpnpServer_.sync()

   @Tac.handler( 'activeAddrWithMask' )
   def handleActiveAddrWithMask( self ):
      t0( "IntfStatusReactor::handleActiveAddrWithMask IP changed: ",
            self.notifier_.intfId,
            self.notifier_.activeAddrWithMask )
      self.igdUpnpServer_.sync()

   def close( self ):
      t0( "IntfStatusReactor::close" )
      self.igdUpnpServer_.sync()
      Tac.Notifiee.close( self )

class SsdpStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Ssdp::VrfStatus'

   def __init__( self, vrfStatus, igdUpnpServer ):
      t0( "SsdpStatusReactor::__init__" )
      self.igdUpnpServer_ = weakref.proxy( igdUpnpServer )
      Tac.Notifiee.__init__( self, vrfStatus )
      self.igdUpnpServer_.sync()

   @Tac.handler( 'serverState' )
   def handleServerState( self ):
      t0( "SsdpStatusReactor::serverState" )
      self.igdUpnpServer_.sync()

   def close( self ):
      t0( "SsdpStatusReactor::close" )
      self.igdUpnpServer_.sync()
      Tac.Notifiee.close( self )

class VrfStatusLocalReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::VrfStatusLocal'

   def __init__( self, notifier, igdUpnpServer ):
      self.igdUpnpServer_ = weakref.proxy( igdUpnpServer )
      Tac.Notifiee.__init__( self, notifier )

   @Tac.handler( 'state' )
   def handleState( self ):
      t0( "VRF state updated for VRF ", self.notifier_.name )
      self.igdUpnpServer_.sync()

   def close( self ):
      t0( "VrfStatusLocaReactor::close" )
      self.igdUpnpServer_.sync()
      Tac.Notifiee.close( self )

class UpnpConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Upnp::Config"

   def __init__( self, upnpConfig, igdUpnpServer ):
      self.igdUpnpServer_ = weakref.proxy( igdUpnpServer )
      Tac.Notifiee.__init__( self, upnpConfig )

   @Tac.handler( 'uuid' )
   def handleUuid( self ):
      t0( "UUID updated" )
      self.igdUpnpServer_.sync()

class IgdCommonConfigReactor( Tac.Notifiee ):
   notifierTypeName = "IgdUpnp::Config"

   def __init__( self, igdCommonConfig, igdUpnpServer ):
      self.igdUpnpServer_ = weakref.proxy( igdUpnpServer )
      Tac.Notifiee.__init__( self, igdCommonConfig )

   @Tac.handler( 'protocolAndPort' )
   def handleProtocolAndPort( self ):
      t0( "UPNP protocol and port updated" )
      self.igdUpnpServer_.sync()

class IgdVrfConfigReactor( Tac.Notifiee ):
   notifierTypeName = "IgdUpnp::IgdVrfConfig"

   def __init__( self, igdVrfConfig, igdUpnpServer ):
      self.igdUpnpServer_ = weakref.proxy( igdUpnpServer )
      Tac.Notifiee.__init__( self, igdVrfConfig )

   @Tac.handler( 'ipv4State' )
   def handleIpv4State( self ):
      t0( "ipv4State updated for VRF", self.notifier_.name )
      self.igdUpnpServer_.sync()

   @Tac.handler( 'natInfo' )
   def handleNatInfo( self ):
      t0( "natInfo updated for VRF", self.notifier_.name )
      self.igdUpnpServer_.sync()

class UpnpPortMapVrfStatusReactor( Tac.Notifiee ):
   notifierTypeName = "IgdUpnpShared::UpnpPortMapVrfStatus"

   def __init__( self, upnpPortMapVrfStatus, igdUpnpServer ):
      Tac.Notifiee.__init__( self, upnpPortMapVrfStatus )
      self.igdUpnpServer_ = weakref.proxy( igdUpnpServer )

   @Tac.handler( 'portMapStatus' )
   def handlePortMapStatus( self, key ):
      t0( "portMapStatus updated for VRF ", self.notifier_.vrfName )
      srv = self.igdUpnpServer_.service_[ self.notifier_.vrfName ]
      if not srv:
         t0( "portMapStatus service not found ", self.notifier_.vrfName )
         return
      status = self.notifier_.portMapStatus.get( key )
      srv.processPortMapStatus( key, status )

class IgdUpnpServer( SuperServer.SuperServerAgent, SuperServer.GenericService ):
   # Reactor to the 'upnp/igd/config' entity
   notifierTypeName = 'IgdUpnp::Config'

   def __init__( self, entityManager ):
      t0( 'IgdUpnpServer::__init__' )
      SuperServer.SuperServerAgent.__init__( self, entityManager )

      self.service_ = {}

      mg = MountGroup( entityManager )
      mounts = mg.mountProfile( "IgdUpnp-SuperServer" )
      mountsPortMap = mg.mountProfile( "IgdUpnpShared-SuperServer-Include" )
      self.upnpConfig_ = mounts[ 'upnp/config' ]
      self.igdConfig_ = mounts[ 'upnp/igd/config' ]
      self.igdStatus_ = mounts[ 'upnp/igd/status' ]
      self.allVrfStatusLocal_ = mounts[ Cell.path( 'ip/vrf/status/local' ) ]
      self.ipStatus_ = mounts[ 'ip/status' ]
      self.natConfig_ = mounts[ 'ip/nat/config' ]
      self.igdPortMapConfig_ = mountsPortMap[ 'upnp/igd/portMap/config' ]
      self.igdPortMapStatus_ = mountsPortMap[ 'upnp/igd/portMap/status' ]
      self.ssdpStatus_ = mounts[ 'upnp/ssdp/status' ]
      Tac.Type( "Ira::IraIpStatusMounter" ).doMountEntities( mg.cMg_, True, False )

      self.intfStatusReactor_ = None
      self.allVrfStatusLocalReactor_ = None
      self.igdCommonConfigReactor_ = None
      self.upnpConfigReactor_ = None
      self.vrfConfigReactor_ = None
      self.natIntfConfigReactor_ = None
      self.natPoolConfigReactor_ = None
      self.upnpPortMapVrfStatusReactor_ = None
      self.ssdpStatusReactor_ = None

      self.igdVrfConfig_ = None
      self.serviceStopCount_ = 0
      self.serviceStartCount_ = 0

      def _finished():
         t0( 'IgdUpnpServer::_finished' )
         self.igdVrfConfig_ = self.igdConfig_.igdVrfConfig
         self.igdStatus_.igdVrfStatus.clear()

         # This will register the instance as a notifiee for IgdUpnp::Config changes
         SuperServer.GenericService.__init__( self, "IgdUpnpServer", self.igdConfig_,
                                           sync=False )

         self.upnpConfigReactor_ = UpnpConfigReactor( self.upnpConfig_, self )

         self.igdCommonConfigReactor_ = IgdCommonConfigReactor(
               self.igdConfig_, self )

         self.vrfConfigReactor_ = Tac.collectionChangeReactor(
               self.igdVrfConfig_, IgdVrfConfigReactor,
                                       reactorArgs=( self, ) )

         self.allVrfStatusLocalReactor_ = Tac.collectionChangeReactor(
            self.allVrfStatusLocal_.vrf, VrfStatusLocalReactor,
            reactorArgs=( self, ) )

         self.intfStatusReactor_ = Tac.collectionChangeReactor(
            self.ipStatus_.ipIntfStatus, IntfStatusReactor,
            reactorArgs=( self, ) )

         self.natIntfConfigReactor_ = Tac.collectionChangeReactor(
             self.natConfig_.intfConfig, NatIntfConfigReactor,
                                       reactorArgs=( self, ) )

         self.natPoolConfigReactor_ = Tac.collectionChangeReactor(
             self.natConfig_.poolConfig, NatPoolConfigReactor,
                                       reactorArgs=( self, ) )

         self.upnpPortMapVrfStatusReactor_ = Tac.collectionChangeReactor(
             self.igdPortMapStatus_.upnpPortMapVrfStatus,
             UpnpPortMapVrfStatusReactor, reactorArgs=( self, ) )

         self.ssdpStatusReactor_ = Tac.collectionChangeReactor(
               self.ssdpStatus_.vrfStatus, SsdpStatusReactor,
               reactorArgs=( self, ) )

         self.sync()

      mg.close( _finished )

   def isUpnpIntf( self, intf ):
      return intf in self.igdConfig_.interface

   def isUpnpVrf( self, vrfName ):
      return vrfName in self.igdConfig_.igdVrfConfig

   def checkNatConfig( self, vrfConfig ):
      vrfName = vrfConfig.name

      # If SSDP is not configured yet, return False
      if not self.ssdpStatus_ or vrfName not in self.ssdpStatus_.vrfStatus:
         t0( "SSDP is not yet up for vrf ", vrfName )
         return False

      ssdpVrfStatus = self.ssdpStatus_.vrfStatus.get( vrfName )
      if ssdpVrfStatus.serverState == "disabled":
         t0( "SSDP is disabled: ", vrfName )
         return False

      profile = self.natConfig_.intfConfig.get( vrfConfig.natInfo.profile )

      if not profile:
         # Verify that the profile is configured
         t0( "NAT profile ", vrfConfig.natInfo.profile, " is not configured yet" )
         return False

      # Profile is present.  Verify that VRF matches
      if profile.vrfName != vrfName:
         t0( "NAT profile VRF mismatch. Exp:", profile.vrfName,
               " Actual:", vrfConfig.natInfo.profile )
         return False

      # Verify that ACL is correct
      dynKey = Tac.Value( "Ip::Nat::DynamicKey", vrfConfig.natInfo.acl, 0 )
      dynNatConfig = profile.dynamicNat.get( dynKey )
      if not dynNatConfig:
         t0( "ACL ", vrfConfig.natInfo.acl, " not used in the NAT profile" )
         return False

      # Verify that the pool is correct
      if dynNatConfig.pool != vrfConfig.natInfo.pool:
         t0( "ACL ", vrfConfig.natInfo.acl, " is not associated with pool ",
               vrfConfig.natInfo.pool )
         return False

      # Verify that pool indeed is configured
      poolConfig = self.natConfig_.poolConfig.get( vrfConfig.natInfo.pool )
      if not poolConfig:
         t0( "pool", vrfConfig.natInfo.pool, "not yet configured" )
         return False

      # Check if poolRange is present
      if not poolConfig.poolRange:
         t0( "poolRange not present in poolConfig for pool",
               vrfConfig.natInfo.pool )
         return False

      return True

   def getVrfIntfs( self ):
      # Create a dict for each VRF with a list of interfaces
      vrfIntf = {}
      for intf in self.igdConfig_.interface:
         intfStatus = self.ipStatus_.ipIntfStatus.get( intf )
         if intfStatus and intfStatus.activeAddrWithMask:
            vrfName = intfStatus.vrf
            # Create key for vrfName if needed
            if vrfName not in vrfIntf:
               vrfIntf[ vrfName ] = []
            # Add interface into the VRF
            vrfIntf[ vrfName ].append( intf )
      return vrfIntf

   def isIgdCommonConfigDone( self ):
      # Check UUID
      if self.upnpConfig_.uuid == Tac.Value( "Arnet::Uuid" ):
         t0( "UUID is not configured" )
         return False
      # Check server protocol and port
      if self.igdConfig_.protocolAndPort == \
            Tac.Value( "IgdUpnp::IgdProtocolAndPort" ):
         t0( "Server protocol and port are not set" )
         return False
      if not self.igdConfig_.interface:
         t0( "Server interfaces are not set" )
         return False
      return True

   def sync( self ):

      # First we are verifying that all common config is done
      if not self.isIgdCommonConfigDone():
         for vrfName in list( self.igdStatus_.igdVrfStatus ):
            self.deleteService( vrfName )
         return

      # If some VRFs disappeared from config, we need to clean up
      oldVrfs = self.igdStatus_.igdVrfStatus.keys()
      newVrfs = self.igdConfig_.igdVrfConfig.keys()
      invalidVrfs = set( oldVrfs ) - set( newVrfs )
      for vrfName in invalidVrfs:
         self.deleteService( vrfName )

      # Create a map of VRFs to interfaces in config
      vrfIntf = self.getVrfIntfs()

      # Iterate over all vrfs in the config and sync the config
      for vrfName, vrfConfig in self.igdConfig_.igdVrfConfig.items():
         t0( 'IgdUpnpServer::sync for VRF', vrfName )

         # Check the VRF state
         if vrfName != DEFAULT_VRF:
            vrfState = self.allVrfStatusLocal_.vrf.get( vrfName )
            if not vrfState or vrfState.state != "active":
               t0( "VRF state for VRF:", vrfName, "is", vrfState )
               self.deleteService( vrfName )
               continue

         # Check IPv4 state
         if vrfConfig.ipv4State == "disabled":
            t0( "ipv4State disabled for VRF", vrfName )
            self.deleteService( vrfName )
            continue

         # Check NAT Profile and pool
         if vrfConfig.natInfo == \
               Tac.Value( "IgdUpnp::IgdVrfNatInfo" ):
            t0( "NAT profile and pool are not set for vrf:", vrfName )
            self.deleteService( vrfName )
            continue
         if not self.checkNatConfig( vrfConfig ):
            t0( "NAT profile and pool config for VRF", vrfName, "is not complete" )
            self.deleteService( vrfName )
            continue

         # Update intfs if needed
         cfgIntfs = vrfIntf.get( vrfName, [] )
         if not cfgIntfs:
            t0( "no active interfaces", vrfName )
            self.deleteService( vrfName )
            continue

         vrfStatus = self.igdStatus_.igdVrfStatus.get( vrfName )
         if not vrfStatus:
            t0( "vrfStatus created for VRF", vrfName )
            vrfStatus = self.igdStatus_.igdVrfStatus.newMember( vrfName )

         syncNeeded = False

         # Update UUID if needed
         if self.upnpConfig_.uuid != vrfStatus.uuid:
            t0( "UUID changed to", self.upnpConfig_.uuid )
            vrfStatus.uuid = self.upnpConfig_.uuid
            syncNeeded = True

         # Update server protocol and port if needed
         if self.igdConfig_.protocolAndPort != vrfStatus.protocolAndPort:
            t0( "Server protocol/port changed to", self.igdConfig_.protocolAndPort )
            vrfStatus.protocolAndPort = self.igdConfig_.protocolAndPort
            syncNeeded = True

         # Update ipv4State if needed
         if vrfConfig.ipv4State != vrfStatus.ipv4State:
            t0( "Server state changed for VRF:", vrfName, "to", vrfConfig.ipv4State )
            vrfStatus.ipv4State = vrfConfig.ipv4State
            syncNeeded = True

         # Update NAT profile and pool if needed
         if vrfConfig.natInfo != vrfStatus.natInfo:
            t0( "NAT info changed for VRF:", vrfName )
            vrfStatus.natInfo = vrfConfig.natInfo

            # Copy the startIp addr from the first pool to be used as externalIpAddr
            poolRange = \
               self.natConfig_.poolConfig[ vrfConfig.natInfo.pool ].\
                     poolRange.keys()[ 0 ]
            vrfStatus.externalIpv4Addr = poolRange.startIp
            syncNeeded = True

         # Update intfs if needed
         if set( vrfStatus.interface ) != set( cfgIntfs ):
            t0( "Interface list changed changed for VRF:", vrfName )
            # Update the list of interfaces.
            vrfStatus.interface.clear()
            for intf in cfgIntfs:
               vrfStatus.interface.add( intf )
            syncNeeded = True

         t0( "syncNeeded for VRF:", vrfName, syncNeeded )

         # If syncNeeded is true, check if the service has all config needed to run
         if syncNeeded:
            # If sync is needed, we need to first stop the service, if it is running
            self.syncService( vrfName )

      t0( 'IgdUpnpServer::sync finished' )

   def deleteService( self, vrfName ):
      self.maybeStopService( vrfName )
      del self.igdStatus_.igdVrfStatus[ vrfName ]

      # If the VRF is no longer active, remove the config
      if vrfName != DEFAULT_VRF and \
            vrfName not in self.allVrfStatusLocal_.vrf:
         t0( "Deleting portMapConfig for vrf", vrfName )
         del self.igdPortMapConfig_.upnpPortMapVrfConfig[ vrfName ]

   def maybeStopService( self, vrfName ):
      t0( "VRF:", vrfName, "no longer active" )
      srv = self.service_.get( vrfName )
      if srv:
         t0( "Stopping service on vrf :", vrfName )
         if os.getenv( "IGD_UPNP_TESTING" ) is None:
            srv.stopService()
            srv.cleanupService()
         del self.service_[ vrfName ]
         self.serviceStopCount_ += 1

   # returns True if a new serice was actually created
   def syncService( self, vrfName ):
      self.maybeStopService( vrfName )
      if not self.service_.get( vrfName ):
         t0( "Starting service on vrf :", vrfName )
         srv = MiniUpnpdService( vrfName,
                                 self.upnpConfig_,
                                 self.igdConfig_,
                                 self.igdStatus_.igdVrfStatus[ vrfName ],
                                 self.igdPortMapConfig_,
                                 self.igdPortMapStatus_ )
         if os.getenv( "IGD_UPNP_TESTING" ) is None:
            srv.sync()
         self.service_[ vrfName ] = srv
         self.serviceStartCount_ += 1
         return True
      return False

def Plugin( ctx ):
   t0( "Registering UPnP IGD service" )
   ctx.registerService( IgdUpnpServer( ctx.entityManager ) )
