#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#
# Handles ptp over management port using linuxptp (ptp4l and phc2sys)
#

# pylint: disable=consider-using-f-string

from contextlib import suppress
import glob
import os
import weakref
from PyWrappers.LinuxPtp import ptp4l, phc2sys
import QuickTrace
import SuperServer
import Tac
from TypeFuture import TacLazyType
import SharedMem
import Smash

qv = QuickTrace.Var
qt0 = QuickTrace.trace0
qt5 = QuickTrace.trace5
qt9 = QuickTrace.trace9

ManagementIntfConfig = TacLazyType( 'Ptp::ManagementIntfConfig' )

def _getEthIntfStatus( ethIntfPhyStatus, intfName ):
   """
   Gets the EthIntf status associated with the intfName, or returns None if
   not found
   """
   if intfName in ethIntfPhyStatus:
      # intfStatus: collection of EthPhyIntfStatus's
      return ethIntfPhyStatus[ intfName ]
   return None

def _getDeviceName( ethIntfPhyStatus, intfName ):
   """
   Gets the kernel device name of an interface based on ethhIntf status, returning
   None if no intf status found
   """
   status = _getEthIntfStatus( ethIntfPhyStatus, intfName )
   if status is not None:
      return status.deviceName
   return None

class IntfConfigNotifiee( Tac.Notifiee ):
   # Reactor to the sys/time/ptp/config/managementIntfConfig/<intf> entity
   notifierTypeName = "Ptp::ManagementIntfConfig"

   def __init__( self, obj, master ):
      Tac.Notifiee.__init__( self, obj, filtered=False )
      self.master_ = master

   def onAttribute( self, attr, key ):
      # Master service should react to all intf changes with a sync and restart
      Tac.Notifiee.onAttribute( self, attr, key )
      self.master_.forceRestart_ = True
      self.master_.sync()

class EthPhyIntfStatusNotifiee( Tac.Notifiee ):
   # Reactor to the interface/status/eth/phy/all entity
   # that if the device name for an interface we are interested in changes,
   # we can restart the service with the new kernel intf name
   notifierTypeName = "Interface::EthPhyIntfStatus"

   def __init__( self, obj, managementIntfConfig, master_ ):
      super().__init__( obj )
      self.managementIntfConfig_ = managementIntfConfig
      self.master_ = master_
      self.handleDeviceName()

   def _getCorrespondingConfig( self ):
      intfId = self.notifier_.intfId
      if intfId in self.managementIntfConfig_:
         return self.managementIntfConfig_[ intfId ]
      return None

   @Tac.handler( "deviceName" )
   def handleDeviceName( self ):
      managementIntfConfig = self._getCorrespondingConfig()
      if managementIntfConfig is None or not managementIntfConfig.enabled:
         # don't bother trying to sync if the interface is disabled or
         # not a ptp management port
         if managementIntfConfig:
            name = managementIntfConfig.intf
         else:
            name = ""
         qt9( "handleDeviceName:", qv( name ),
              "disabled or not ptp management" )
         return
      # if the device name of an interface we are interested in changes
      # we should resync the configuration file
      self.master_.sync()

class KernelNetInfoNotifiee( Tac.Notifiee ):
   """
   Reacts to changes in the kernel net info. If a kernel net device disappears or
   appears (e.g. ptp0 is loaded by a kernel driver) give the service a chance to
   start
   """
   notifierTypeName = "KernelNetInfo::Status"

   def __init__( self, kni, ptpConfig, ethIntfPhyStatus, service ):
      super().__init__( kni )
      self.kni_ = kni
      self.ptpConfig_ = ptpConfig
      self.ethIntfPhyStatus_ = ethIntfPhyStatus
      self.service_ = service
      self.kniInterfaces_ = { key: interface.deviceName
         for key, interface in kni.interface.items() }

   @Tac.handler( "interface" )
   def handleInterface( self, key ):
      kniInterface = self.kni_.interface.get( key, None )
      ptpIntfDeviceNames = { _getDeviceName( self.ethIntfPhyStatus_, intfId )
         for intfId in self.ptpConfig_.managementIntfConfig }

     # Deletion case
      kniDeviceName = self.kniInterfaces_.pop( key, None )
      if kniInterface:
         # Insertion case
         kniDeviceName = self.kniInterfaces_[ key ] = kniInterface.deviceName

      # If this event relates to a interface configured for ptp, then trigger a sync
      if kniDeviceName in ptpIntfDeviceNames:
         qt9( "handleInterface:", qv( kniDeviceName ),
            "was", "added" if kniInterface else "removed" )
         self.service_.sync()

class ClockConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Time::Clock::Config"
   def __init__( self, clockConfig, service ):
      self.service_ = service
      Tac.Notifiee.__init__( self, clockConfig )

   @Tac.handler( 'source' )
   def handleSource( self ):
      self.service_.sync()

class ManagementIntfEthPhyReactor( Tac.Notifiee ):
   """
   Reacts to any changes in Eth Phy status entities which are referenced by
   managementIntfs in the given SystemClockConfig
   """
   notifierTypeName = "Ptp::SystemClockConfig"

   def __init__( self, systemClockConfig, ethIntfPhyStatus, service ):
      super().__init__( systemClockConfig )
      self.service_ = service
      self.ethIntfPhyStatus_ = ethIntfPhyStatus

      # react to all events where EthPhyIntfStatus.deviceName changes
      self.ethPhyIntfStatusReactors = Tac.collectionChangeReactor(
         self.ethIntfPhyStatus_,
         EthPhyIntfStatusNotifiee,
         reactorArgs=( systemClockConfig.managementIntfConfig, self.service_ ),
      )

   @Tac.handler( "managementIntfConfig" )
   def handleManagementIntfConfig( self, intfName ):
      if intfName in self.notifier_.managementIntfConfig:
         # rerun the ethPhyStatus reactor if we have a new managementIntfConfig
         # to catch any changes
         reactor =  self.ethPhyIntfStatusReactors.reactor(
            _getEthIntfStatus( self.ethIntfPhyStatus_, intfName )
         )
         if reactor:
            reactor.handleDeviceName()
         # if we don't have a ethPhyStatus entry for this intfName when
         # the managementIntfConfig is added, the sync will happen when
         # it is added by the ethPhyIntfStatusReactors
      # if we lose an managementIntfConfig, no cleanup needs to happen on the
      # EthPhyStatus side

def whichManagementInterfaceEnabled( ptpConfig, kni, ethIntfPhyStatus ):
   kniNames = [ i.deviceName for i in kni.interface.values() ]
   for ic in ptpConfig.managementIntfConfig.values():
      intfName = ic.intf
      ethIntfStatus = _getEthIntfStatus( ethIntfPhyStatus, intfName )
      if ethIntfStatus is None:
         continue
      if ic.enabled and ethIntfStatus.deviceName in kniNames:
         qt5( "whichManagementInterfaceEnabled: found kernel interface with name",
              qv( ethIntfStatus.deviceName ) )
         return ic
   return None

class Ptp4lService( SuperServer.LinuxService ):
   # Manages the ptp4l service based on configuration
   # mounted at sys/time/ptp/config
   notifierTypeName = "Ptp::SystemClockConfig"
   # note: this implicitly listens to any change in the objects under
   # Ptp::SystemClockConfig and causes a sync when when management configs are added

   def ptp4lServiceName( self ):
      return f'{ptp4l()}@MakoRedondo'

   def ptp4lConfigName( self ):
      return '/etc/ptp4l@MakoRedondo.conf'

   # BUG863415 investigate making this value 2 when the systemd templates directly
   # call phc2sys/ptp4l
   SERVICE_RESTART_DELAY = 4

   def __init__( self, config, agent, kni, ethIntfPhyStatus ):
      self.agent_ = weakref.proxy( agent )
      SuperServer.LinuxService.__init__( self, self.ptp4lServiceName(),
         self.ptp4lServiceName(),
         config, self.ptp4lConfigName() )

      # Prevents "start request repeated too quickly" error
      self.serviceRestartDelay_ = self.SERVICE_RESTART_DELAY
      self.ptpConfig = config
      # kernel network interfaces info
      self.kni = kni
      self.ethIntfPhyStatus_ = ethIntfPhyStatus

      self.intfConfigReactor_ = Tac.collectionChangeReactor(
         self.notifier_.managementIntfConfig,
         IntfConfigNotifiee,
         reactorArgs=( self, ),
      )

      self.kniReactor_ = KernelNetInfoNotifiee(
         self.kni,
         self.ptpConfig,
         self.ethIntfPhyStatus_,
         self,
      )

      self.managementIntfReactor_ = ManagementIntfEthPhyReactor(
         self.ptpConfig,
         self.ethIntfPhyStatus_,
         self,
      )

      # ptp4l sysconfig file contents
      self.sysconfFilename_ = "/etc/sysconfig/ptp4l"
      self.sysconf_ = ""

   def serviceProcessWarm( self ):
      if not self.serviceEnabled():
         # disabled, therefore warm
         return True
      return os.path.exists( ManagementIntfConfig.udsAddress )

   def serviceEnabled( self ):
      ic = whichManagementInterfaceEnabled(
         self.ptpConfig, self.kni,
         self.ethIntfPhyStatus_ )
      if ic is not None:
         qt5( "ptp4l service is enabled" )
      else:
         qt5( "ptp4l service not enabled" )

      return ic is not None

   def writePtp4lSysconfig( self, conf ):
      return self.writeConfigFile( self.sysconfFilename_, conf )

   def ptp4lSysconfig( self ):
      # -f: specify config file
      # -q: don't output to syslog
      # -l6: output level 6 logs (info)
      # -m: output to stdout
      # note: we have patched the ptp4l.service file to redirect to
      # /var/log/ptp4l.log
      sysconf = """
OPTIONS="-f {}"
""".format( self.confFilename_ )
      return sysconf

   def conf( self ):
      qt0( "Generating the new ptp4l.conf" )
      if self.serviceEnabled():
         # XXX Mulitple interfaces not supported yet. Just use the first enabled one
         for intfConfig in self.ptpConfig.managementIntfConfig.values():
            if intfConfig.enabled:
               ic = intfConfig
               break

         conf = "[global]\n"
         conf += "domainNumber %d\n" % ic.domainNumber
         if ic.slaveOnly:
            conf += "slaveOnly 1\n"
         conf += "priority1 %d\n" % ic.priority1
         conf += "priority2 %d\n" % ic.priority2
         conf += "free_running %d\n" % ic.freeRunning
         conf += "uds_address %s\n" % ManagementIntfConfig.udsAddress
         conf += "time_stamping %s\n" % ic.timeStampingMode
         conf += "use_syslog 0\n"
         conf += "verbose 1\n"
         conf += "logging_level 6\n"

         conf += "[%s]\n" % _getDeviceName( self.ethIntfPhyStatus_, ic.intf )

         conf += "logAnnounceInterval %d\n" % ic.logAnnounceInterval
         conf += "announceReceiptTimeout %d\n" % ic.announceReceiptTimeout

         conf += "logMinDelayReqInterval %d\n" % ic.logMinDelayReqInterval
         conf += "logMinPdelayReqInterval %d\n" % ic.logMinPdelayReqInterval
         conf += "neighborPropDelayThresh %d\n" % ic.linkDelayThreshold

         conf += "delay_mechanism %s\n" % \
                 { "e2e" : "E2E",
                   "p2p" : "P2P",
                   "autoConfigure" : "Auto" } [ ic.delayMechanism ]

         conf += "network_transport %s\n" % \
                 { "layer2" : "L2",
                   "ipv4" : "UDPv4",
                   "ipv6" : "UDPv6" }[ ic.transportMode ]

         conf += "udp_ttl %d\n" % ic.ttl
      else:
         conf = "# ptp4l service not enabled\n"
      return conf

class Phc2sysService( SuperServer.LinuxService ):
   # Manages the phc2sys service based on configuration
   # mounted at sys/time/ptp/config
   notifierTypeName = "Ptp::SystemClockConfig"

   # BUG863415 investigate making this value 2 when the systemd templates directly
   # call phc2sys/ptp4l
   SERVICE_RESTART_DELAY = 4

   def phc2sysServiceName( self ):
      return f'{phc2sys()}@MakoRedondo'

   def phc2sysConfigName( self ):
      return f'/etc/{phc2sys()}@MakoRedondo.conf'

   def __init__( self, config, clockConfig, agent, kni, ethIntfPhyStatus ):
      self.agent_ = weakref.proxy( agent )
      SuperServer.LinuxService.__init__( self, self.phc2sysServiceName(),
                                         self.phc2sysServiceName(),
                                         config, self.phc2sysConfigName() )
      # Prevents "start request repeated too quickly" error
      self.serviceRestartDelay_ = self.SERVICE_RESTART_DELAY
      self.ptpConfig = config
      self.clockConfig = clockConfig
      self.ethIntfPhyStatus_ = ethIntfPhyStatus
      self.kni = kni

      self.intfConfigReactor_ = Tac.collectionChangeReactor(
         self.notifier_.managementIntfConfig,
         IntfConfigNotifiee,
         reactorArgs=( self, ),
      )

      self.kniReactor_ = KernelNetInfoNotifiee(
         self.kni,
         self.ptpConfig,
         self.ethIntfPhyStatus_,
         self,
      )

      self.managementIntfReactor_ = ManagementIntfEthPhyReactor(
         self.ptpConfig,
         self.ethIntfPhyStatus_,
         self,
      )

      self.clockConfigReactor_ = ClockConfigReactor( self.clockConfig, self )

   def serviceProcessWarm( self ):
      if not self.serviceEnabled():
         # disabled, therefore warm
         return True

      pidFiles = glob.glob( "/var/run/phc2sys.*" )
      for pidFile in pidFiles:
         _, pid = os.path.splitext( pidFile )
         pid = pid.lstrip( '.' )
         with suppress( FileNotFoundError ), open( f'/proc/{pid}/cmdline' ) as file:
            if ManagementIntfConfig.udsAddress in file.read():
               return True
      return False

   def serviceEnabled( self ):
      if self.clockConfig.source == 'ptp' and self.ptpConfig.systemClockSourceIntf:
         # make sure the systemClockSourceIntf matches our enabled intf
         intfId = self.ptpConfig.systemClockSourceIntf
         enabledManagement = whichManagementInterfaceEnabled( self.ptpConfig,
            self.kni, self.ethIntfPhyStatus_ )
         return ( enabledManagement is not None and
                 enabledManagement.timeStampingMode == 'hardware' and
                 enabledManagement.intf == intfId )
      return False

   def conf( self ):
      """ Returns the contents to write to the service's conf file """
      if self.serviceEnabled():
         conf = """
OPTIONS="-a -r -q -n {} -z {}"
""".format( self.ptpConfig.managementIntfConfig[
   self.ptpConfig.systemClockSourceIntf ].domainNumber,
   ManagementIntfConfig.udsAddress )
      else:
         conf = "# phc2sys service not enabled\n"
      return conf

class LinuxPtp( SuperServer.SuperServerAgent ):
   DEFAULT_NS = Tac.Value( 'Arnet::NamespaceName' ).defaultNamespace()
   READER_INFO = Smash.mountInfo( 'keyshadow' )

   def __init__( self, entityManager ):
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      self.clockConfig = mg.mount(
         'sys/time/clock/config', 'Time::Clock::Config', 'r' )
      self.ptpConfig = mg.mount(
         'sys/time/ptp/config', 'Ptp::SystemClockConfig', 'r' )
      self.ethIntfPhyStatusDir = mg.mount(
         'interface/status/eth/phy/all', 'Interface::AllEthPhyIntfStatusDir', 'r' )
      shmemMg = SharedMem.entityManager( sysdbEm=entityManager )
      # pkgdeps: library KernelNetworkInfo
      self.kni = shmemMg.doMount( f'kni/ns/{self.DEFAULT_NS}/status',
                                  "KernelNetInfo::Status", self.READER_INFO )
      self.ptp4lService = None
      self.phc2sysService = None

      def _finished():
         self.ptp4lService = Ptp4lService( self.ptpConfig, self, self.kni,
                                           self.ethIntfPhyStatusDir.intfStatus )
         self.phc2sysService = Phc2sysService(
            self.ptpConfig, self.clockConfig, self, self.kni,
            self.ethIntfPhyStatusDir.intfStatus )

      mg.close( _finished )

   def warm( self ):
      if not self.ptp4lService or not self.phc2sysService:
         return False
      elif not self.ptp4lService.warm() or not self.phc2sysService.warm():
         return False
      else:
         return True

   def onSwitchover( self, protocol ):
      self.ptp4lService.sync()
      self.phc2sysService.sync()

def Plugin( ctx ):
   ctx.registerService( LinuxPtp( ctx.entityManager ) )
