# Copyright (c) 2007-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Arnet
import Arnet.NsLib as NsLib # pylint: disable=consider-using-from-import
import BothTrace
import Cell
from IpLibConsts import DEFAULT_VRF
import itertools
import json
from NetConfigLib import netnsNameWithUniqueId, getRedundantPriorityInfo
import os
import PyWrappers.Dnsmasq as dnsmasq
import signal
from subprocess import check_output
import SuperServer
import sys
import Tac
import Tracing
import weakref

# pkgdeps: import IraAgent

qv = BothTrace.Var
bt0 = BothTrace.trace0
bt1 = BothTrace.trace1
bt2 = BothTrace.trace2
t2 = Tracing.trace2

__defaultTraceHandle__ = Tracing.Handle( 'NetworkManager' )

af = Tac.Type( "Arnet::AddressFamily" )

def name():
   return 'NetworkManager'

# Tests may set this value to override the service restart delay provided to
# SuperServer.LinuxService. This is needed so restarts are scheduled immediately
# from LinuxService, potentially before we have the chance to set the restart delay
# on the service instance post-init. Ultimately this becomes an issue for restart
# testing.
serviceRestartDelayTestHook = 1
# For breadth tests that cover reactors and config file generation we can disable the
# 'service' commands from running to eliminate noise from dnsmasq failing to start.
# This is expected in a breadth test environemnt because of its limited support for
# starting services.
skipServiceCommandsTestHook = False

def addrMapping( line, hostname='' ):
   line = line.strip()
   segments = line.split()
   condition = ( len( segments ) < 2 or line[ 0 ] == '#' )
   if hostname != '':
      condition = condition or ( segments[ 1 ] != hostname )
   if not condition:
      try:
         Arnet.IpAddress( segments[ 0 ] )
         return ( segments[ 0 ], segments[ 1 ] )
      except Exception: # pylint: disable-msg=W0703
         try:
            Arnet.Ip6Addr( segments[ 0 ] )
            return ( segments[ 0 ], segments[ 1 ] )
         except Exception: # pylint: disable-msg=W0703
            pass
   return None

# the reactor class for sysdb object sys/net/config/hostAddr
class HostAddrNotifiee( Tac.Notifiee ):
   """Reactor to the 'sys/net/config/hostAddr' object """
   notifierTypeName = 'System::HostAddr'

   def __init__( self, obj, key, netConfigNotifiee, reservedHosts ):
      bt0( "HostAddrNotifiee.__init__ for host", qv( key ) )
      Tac.Notifiee.__init__( self, obj )
      self.master_ = netConfigNotifiee
      self.name_ = key
      self.reserved_ = ( self.name_ in reservedHosts )

   def close( self ):
      bt0( "HostAddrNotifiee.close for host", qv( self.name_ ) )
      self.master_.handleHostnameToIpMapping( self.name_, None, deleteAll=True )
      Tac.Notifiee.close( self )

   def handleAddressChange( self, address, ipv6=False ):
      bt1( "HostAddrNotifee.handleAddressChange for host", qv( self.name_ ),
           "address", qv( address ) )
      if self.reserved_:
         # If the host is a reserved one (present in /etc/hosts.Eos)
         bt2( "Attempted to add reserved hostname: ignoring" )
         return
      self.master_.handleHostnameToIpMapping( self.name_, address, ipv6 )

   @Tac.handler( 'ipAddr' )
   def handleIpAddr( self, address ):
      self.handleAddressChange( address, False )

   @Tac.handler( 'ip6Addr' )
   def handleIp6Addr( self, address ):
      self.handleAddressChange( address, True )

class VrfStatusLocalReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::VrfStatusLocal'

   def __init__( self, vrfStatusLocal, allVrfStatusLocalReactor ):
      bt0( "VrfStatusLocalReactor.__init__" )
      Tac.Notifiee.__init__( self, vrfStatusLocal )
      self.allVrfStatusLocalReactor_ = allVrfStatusLocalReactor
      self.vrfStatusLocal_ = vrfStatusLocal
      self.handleState()

   @Tac.handler( 'state' )
   def handleState( self ):
      bt1( "VrfStatusLocalReactor.handleState on vrf",
           qv( self.vrfStatusLocal_.vrfName ) )
      self.allVrfStatusLocalReactor_.handleState( self.vrfStatusLocal_.vrfName )

class AllVrfStatusLocalReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::AllVrfStatusLocal'

   def __init__( self, allVrfStatusLocal, master ):
      bt0( "AllVrfStatusLocalReactor.__init__" )
      Tac.Notifiee.__init__( self, allVrfStatusLocal )
      self.vrfStatusLocalReactors_ = dict() # pylint: disable=use-dict-literal
      self.allVrfStatusLocal_ = allVrfStatusLocal
      self.master_ = master

      for v in allVrfStatusLocal.vrf:
         self.handleVrf( v )

   def handleState( self, vrfName ):
      bt1( "AllVrfStatusLocalReactor.handleState for vrf", qv( vrfName ) )
      self.master_.handleVrfState( vrfName )

   @Tac.handler( 'vrf' )
   def handleVrf( self, vrf ):
      bt1( "AllVrfStatusLocalReactor.handleVrf for vrf", qv( vrf ) )
      if self.allVrfStatusLocal_.vrf.get( vrf ):
         bt2( "vrf", qv( vrf ), "exists" )
         if not vrf in self.vrfStatusLocalReactors_:
            self.vrfStatusLocalReactors_[ vrf ] = \
                VrfStatusLocalReactor( self.allVrfStatusLocal_.vrf[ vrf ], self )
            self.master_.handleVrfState( vrf )
      else:
         bt2( "vrf", qv( vrf ), "does not exist" )
         if vrf in self.vrfStatusLocalReactors_:
            self.master_.handleVrfState( vrf )
            del self.vrfStatusLocalReactors_[ vrf ]

class NslReactor( Tac.Notifiee ):
   notifierTypeName = 'Arnet::NamespaceStatusLocal'

   def __init__( self, nsl, master ):
      bt0( 'NslReactor.__init__' )
      assert nsl.nsName == NsLib.DEFAULT_NS
      Tac.Notifiee.__init__( self, nsl )
      self.master_ = master
      self.handleState()

   @Tac.handler( 'state' )
   def handleState( self ):
      bt1( 'NslReactor.handleState' )
      self.master_.handleDefaultNs()

class IslReactor( Tac.Notifiee ):
   notifierTypeName = 'Interface::IntfStatusLocal'

   def __init__( self, isl, master ):
      bt0( 'IslReactor.__init__' )
      Tac.Notifiee.__init__( self, isl )
      self.master_ = master
      self.isl_ = isl
      self.handleNetNsName()

   def close( self ):
      bt0( 'IslReactor.close' )
      self.handleNetNsName()
      Tac.Notifiee.close( self )

   @Tac.handler( 'netNsName' )
   def handleNetNsName( self ):
      bt1( "IslReactor.handleNetNsName for", qv( self.isl_.intfId ) )
      self.master_.handleIslNetNsName( self.isl_.intfId )

class NameServerGroupReactor( Tac.Notifiee ):
   notifierTypeName = 'Tac::Dir'

   def __init__( self, groupConfigDir, groupStatusDir, serviceList,
                 avsl, ipStatus, ip6Status, nsld, aisld ):
      bt0( 'NameServerGroupReactor.__init__' )
      Tac.Notifiee.__init__( self, groupConfigDir )
      self.serviceList = serviceList
      self.groupConfigDir = groupConfigDir
      self.groupStatusDir = groupStatusDir
      self.avsl = avsl
      self.ipStatus = ipStatus
      self.ip6Status = ip6Status
      self.nsld = nsld
      self.aisld = aisld

      self.handleInitialized()

   @Tac.handler( 'entityPtr' )
   def handleEntityPtr( self, pathName ):
      bt1( "NameServerGroupReactor.handleEntityPtr for path", qv( pathName ) )
      if pathName in self.groupConfigDir.entityPtr and \
         self.groupConfigDir.entityPtr[ pathName ]:
         bt2( "Added", qv( pathName ) )
         self.groupStatusDir.newEntity( "System::NetStatus", pathName )
         self.serviceList[ pathName ] = ( NetConfigNotifiee( "NetConfig-" + pathName,
                                          self.groupConfigDir.entityPtr[ pathName ],
                                          self.groupStatusDir.entityPtr[ pathName ],
                                          self.avsl, self.ipStatus, self.ip6Status,
                                          self.nsld, self.aisld,
                                          groupName=pathName ) )
      else:
         bt2( "Deleted", qv( pathName ) )
         if self.serviceList.get( pathName ):
            self.serviceList[ pathName ].dnsMasqService_.stopService()
            self.serviceList[ pathName ].dnsMasqService_ = None
            del self.serviceList[ pathName ]
         self.groupStatusDir.deleteEntity( pathName )

   def handleInitialized( self ):
      bt1( "NameServerGroupReactor.handleInitialized called on startup" )
      # Stop dnsmasq instances present in status but not in config
      stale = []
      for pathName in self.groupStatusDir.entityPtr:
         if pathName not in self.groupConfigDir.entityPtr:
            stale.append( pathName )
      for pathName in stale:
         bt1( f"Stopping stale dnsmasq instance {pathName}" )
         Tac.run( [ "service", "dnsmasq", "stop", pathName ], stdout=Tac.DISCARD )
         self.groupStatusDir.deleteEntity( pathName )
      # Start dnsmasq instances for which configs are present
      for pathName in self.groupConfigDir.entityPtr:
         bt1( f"Starting dnsmasq service handler {pathName}" )
         self.handleEntityPtr( pathName )
         self.serviceList[ pathName ].sync()

class IpIntfStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::IpIntfStatus'

   def __init__( self, iis, master ):
      bt0( 'IpIntfStatusReactor.__init__' )
      self.iis_ = iis
      self.master_ = master
      Tac.Notifiee.__init__( self, iis )
      self.handleActiveAddrWithMask()

   @Tac.handler( 'activeAddrWithMask' )
   def handleActiveAddrWithMask( self ):
      bt1( 'IpIntfStatusReactor.handleActiveAddrWithMask for',
           qv( self.iis_.intfId ) )
      self.master_.handleActiveAddrWithMask( self.iis_.intfId )

class Ip6IntfStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip6::IntfStatus'

   def __init__( self, i6is, master ):
      bt0( 'Ip6IntfStatusReactor.__init__' )
      self.i6is_ = i6is
      self.master_ = master
      Tac.Notifiee.__init__( self, i6is )
      self.handleAddr( None )

   @Tac.handler( 'addr' )
   def handleAddr( self, key ):
      bt1( "Ip6IntfStatusReactor.handleAddr for", qv( self.i6is_.intfId ) )
      # We don't actually need the key, we just want to see when
      # the status changes.
      self.master_.handleAddr( self.i6is_.intfId )

class DnsMasqService( SuperServer.LinuxService ):
   notifierTypeName = '*'

   def __init__( self, allVrfStatusLocal, netStatus, netConfig, master, ipStatus,
                 ip6Status, namespaceStatusLocalDir,
                 allIntfStatusLocalDir, groupName ):
      bt0( "DnsMasqService.__init__ for group", qv( groupName or 'default' ) )
      self.master_ = master
      self.allVrfStatusLocal_ = allVrfStatusLocal
      self.netStatus_ = netStatus
      self.netConfig_ = netConfig
      self.ipStatus_ = ipStatus
      self.ip6Status_ = ip6Status
      self.namespaceStatusLocalDir_ = namespaceStatusLocalDir
      self.allIntfStatusLocalDir_ = allIntfStatusLocalDir
      self.started_ = False
      self.dnsmasqPid = 0
      self.dnsCountersTimer_ = None
      self.groupName_ = groupName
      self.confFileName = "/etc/dnsmasq" + self.groupName_ + ".conf"

      SuperServer.LinuxService.__init__(
            self, dnsmasq.service() + self.groupName_, dnsmasq.service(),
            self.allVrfStatusLocal_, self.confFileName, sync=False,
            serviceRestartDelay=serviceRestartDelayTestHook )

   def handleDnsCacheCountersTimer( self ):
      if self.netConfig_.dnsCacheCountersUpdateInterval:
         if os.path.exists( self.netStatus_.dnsCacheCountersFile ):
            with open( self.netStatus_.dnsCacheCountersFile ) as f:
               try:
                  data = json.load( f )
                  self.netStatus_.dnsCacheCounters.size = data[ 'size' ]
                  self.netStatus_.dnsCacheCounters.insertions = data[ 'insertions' ]
                  self.netStatus_.dnsCacheCounters.evictions = data[ 'evictions' ]
                  self.netStatus_.dnsCacheCounters.misses = data[ 'misses' ]
                  self.netStatus_.dnsCacheCounters.hits = data[ 'hits' ]
               except ValueError as valueError:
                  bt0( "Error parsing dnsmasq cache json:", qv( valueError ) )
         self.sendSignal( signal.SIGUSR1 )
         self.dnsCountersTimer_.timeMin = ( Tac.now() +
              self.netConfig_.dnsCacheCountersUpdateInterval )
      else:
         # We keep the timer running in case if dnsCacheCountersUpdateInterval
         # changes to not zero
         self.dnsCountersTimer_.timeMin = ( Tac.now() +
              self.netConfig_.defaultDnsCacheCountersUpdateInterval )

   def sendSignal( self, sig ):
      try:
         pid = int( check_output( [ "pidof", "-s", self.serviceName_ ] ) )
         if pid:
            # This is a workaround for sending a signal before the signal handlers
            # are installed, prematurely killing the process. Catch the first signal
            # for each dnsmasq pid.
            if self.dnsmasqPid != pid:
               self.dnsmasqPid = pid
               return
            os.kill( pid, sig )
      except: # pylint: disable=bare-except
         return

   def serviceProcessWarm( self ):
      return True

   def defaultNsInitComplete( self ):
      nsld = self.namespaceStatusLocalDir_
      bt2( 'DEFAULT_NS in NSLD is ', qv( NsLib.DEFAULT_NS in nsld.ns ) )
      if NsLib.DEFAULT_NS in nsld.ns:
         bt2( 'state is ', qv( nsld.ns[ NsLib.DEFAULT_NS ].state ) )
      return ( NsLib.DEFAULT_NS in nsld.ns and
               nsld.ns[ NsLib.DEFAULT_NS ].state == 'active' )

   def serviceEnabled( self ):
      enabled = ( ( len( self.netStatus_.nameServer ) > 0 or
                    len( self.netStatus_.v6NameServer ) > 0 ) and
                  self.defaultNsInitComplete() )
      bt2( 'dnsmasq serviceEnabled is', qv( enabled ) )
      self.netStatus_.dnsmasqEnabled = enabled
      if not enabled:
         # stop polling for counters
         self.dnsCountersTimer_ = None
      return enabled

   def conf( self ):
      bt1( 'DnsMasqService.conf for group', qv( self.groupName_ or 'default' ) )
      # there is a race condition here because we validate that the
      # VRF namespaces exists when we write the config file, but there
      # is no way to lock them to force them to stick around until
      # dnsmasq has started and is trying to use them.  When a VRF is
      # torn down, it sometimes churns sysdb enough that we restart
      # dnsmasq right before we delete the namespace, thus causing it
      # to print a few error messages about being unable to access the
      # files in /var/run/netns.  This is hard to fix because we don't
      # really want to put in an active lock mechanism to stop Ira
      # from doing this, so we just live with it.  It's harmless,
      # except for noise in the log messages.
      cfg = "\nno-resolv\n"
      # CVE-2023-28450
      cfg += "edns-packet-max=1232\n"

      allNameServers = itertools.chain(
            self.netStatus_.nameServer.values(),
            self.netStatus_.v6NameServer.values() )
      priorities = [ nameServer.priority for nameServer in allNameServers ]
      if any( priorities ):
         cfg += "strict-order\n"

      # This is the placeholder for listen namespaces that we proxy rquests from.
      listenNs = []

      sourceIpAddr = None

      # identify the set of all namespaces -- we're listening in
      # all of them now because of source address support
      for ( vrf, vrfStatusLocal ) in self.allVrfStatusLocal_.vrf.items():
         bt2( 'DnsMasqSerivce.conf checking vrf', qv( vrf ),
              'to see if we need to listen' )
         if self.master_.vrfExists( vrf ):
            bt2( 'DnsMasqService.conf vrf', qv( vrf ), 'exists' )
            netNs = vrfStatusLocal.networkNamespace
            assert netNs != ""
            listenNs.append( netNs )
         else:
            bt2( 'DnsMasqService.conf vrf', qv( vrf ), 'does not exist' )

      def findSourceIpAddr( vrf, serverNs, proto, destAddr=None ):
         # Source address is valid if:
         # - VRF for the given source address exists
         # - Intf is in the corresponding namespace
         # - Intf has a non-zero IP address
         # Otherwise we still configure the nameserver, just with no source addr.
         # We have reactors to IntfStatusLocal and IpIntfStatus to
         # cover changes to interface status and IP config, in addition to the
         # NetConfigNotifiee. Note that we do not bother to check the interface's
         # OperStatus because in the event that it gets shutdown the IP address will
         # be converted to 0.0.0.0 by Ira, so this is already covered
         if vrf not in self.netStatus_.sourceIntf:
            bt0( 'no source interface configured for vrf', qv( vrf ) )
            return None

         intfId = self.netStatus_.sourceIntf[ vrf ]
         bt0( 'Source interface for vrf', qv( vrf ), 'is', qv( intfId ) )
         if not intfId:
            return None

         aisl = self.allIntfStatusLocalDir_.intfStatusLocal
         intfNs = None
         if intfId not in aisl:
            # if there is no IntfStatusLocal, then that means the intf
            # is in the default namespace
            intfNs = NsLib.DEFAULT_NS
            bt2( 'intf', qv( intfId ), 'not in aisl, defaulting to',
                 qv( NsLib.DEFAULT_NS ) )
         else:
            intfNs = aisl[ intfId ].netNsName
            bt2( 'intf', qv( intfId ), 'is in namespace', qv( intfNs ) )

         if intfNs != serverNs:
            bt0( 'intf', qv( intfId ), 'namespace', qv( intfId ),
                 'is not server namespace', qv( serverNs ),
                 ', cannot use as source' )
            return None

         if proto == 4:
            if intfId not in self.ipStatus_.ipIntfStatus:
               bt0( 'intf', qv( intfId ),
                    'is not in ipIntfStatus, cannot use as source' )
               return None

            sourceIpAddr = self.ipStatus_.ipIntfStatus[ intfId ].\
                activeAddrWithMask.address
            if sourceIpAddr == '0.0.0.0':
               bt0( 'intf', qv( intfId ),
                    'active address is 0.0.0.0, cannot use as source' )
               return None

         else:
            assert proto == 6
            assert destAddr
            ip6is = self.ip6Status_.intf.get( intfId )
            if not ip6is:
               bt0( 'intf', qv( intfId ),
                    'is not in ipIntfStatus, cannot use as source' )
               return None

            sourceIpAddr = ip6is.sourceAddressForDestination( destAddr )
            if sourceIpAddr == '::':
               bt0( 'intf', qv( intfId ),
                    'source address for destination is ::, cannot use as source' )
               return None

         bt2( 'intf', qv( intfId ), 'active source address is', qv( sourceIpAddr ),
              'using as source for test', qv( destAddr ) )
         return sourceIpAddr

      if not self.groupName_:
         # we want dnsmasq to always listen in the default namespace
         # and hence keep the listen-namespace entry for the default namespace
         # ahead of listen-namespace entries for other namespaces
         cfg += "listen-namespace=%s\n" % NsLib.DEFAULT_NS
         for ns in listenNs:
            if ns != NsLib.DEFAULT_NS:
               cfg += "listen-namespace=%s\n" % netnsNameWithUniqueId( ns )
      else:
         # with groupName present we only want to listen to the namespace
         # associated with the group
         cfg += "listen-namespace=internal-ns-%s\n" % self.groupName_

      bt2( 'number of nameservers is v4:', qv( len( self.netStatus_.nameServer ) ),
           ', v6:', len( self.netStatus_.v6NameServer ) )

      def getServerNs( vrf ):
         serverNs = NsLib.DEFAULT_NS
         assert vrf != ''
         if vrf != DEFAULT_VRF and self.master_.vrfExists( vrf ):
            vrfStatus = self.allVrfStatusLocal_.vrf[ vrf ]
            serverNs = vrfStatus.networkNamespace
            bt2( 'DnsMasqService.conf vrf', qv( vrf ), 'is serverNs',
                 qv( serverNs ) )
         else:
            bt0( 'DnsMasqService.conf vrf', qv( vrf ), 'does not exist' )
         return serverNs

      allNameServers = itertools.chain(
            self.netStatus_.nameServer.values(),
            self.netStatus_.v6NameServer.values() )
      # When in strict-order mode, we will walk the list of nameservers to
      # find any other servers of the same priority. This walk will stop at
      # the first server of lower (higher-valued) priority.
      # The linked list of nameservers is constructed by head insertion.
      # Putting these two things together, we will sort the nameservers in
      # descending order of priority.
      # The natural order of System::NameServer is already by priority.
      serverCfg = ''
      for nameServer in sorted( allNameServers, reverse=True ):
         ipAddr = nameServer.ipAddr
         serverAf = ipAddr.af
         vrf = nameServer.vrfName
         serverNs = getServerNs( vrf )
         priority = nameServer.priority

         if serverAf == af.ipv4:
            sourceIpAddr = findSourceIpAddr( vrf, serverNs, proto=4 )
         else:
            assert serverAf == af.ipv6
            ip6Addr = Arnet.Ip6Addr( ipAddr.stringValue )
            sourceIpAddr = findSourceIpAddr( vrf, serverNs, proto=6,
                                             destAddr=ip6Addr )
         srcIpAddrStr = ''
         if sourceIpAddr:
            srcIpAddrStr = f'@{sourceIpAddr}'

         serverNsStr = f'${serverNs}'

         priorityStr = f'%{priority}'

         bt2( 'adding server', qv( ipAddr ), 'with sa', qv( sourceIpAddr ),
              'in netns', qv( serverNsStr ), 'to dnsmasq.conf' )
         serverCfg += f"server={ipAddr}{srcIpAddrStr}{serverNsStr}{priorityStr}\n"
      cfg += serverCfg

      if self.netStatus_.externalDnsProxy:
         cfg += 'external-proxy=1'
      else:
         cfg += 'external-proxy=0'

      cfg += '\ncache-size=%d\n' % self.netStatus_.dnsCacheSize

      if self.netConfig_.dnsCacheCountersUpdateInterval:
         self.netStatus_.dnsCacheCountersFile = ( "/tmp/dnsmasq_cache_counters" +
                                                  self.groupName_ + ".json" )
         cfg += '\ncache-counters-file=%s\n' % self.netStatus_.dnsCacheCountersFile

      # Don't dump the config to quicktrace. It's too big.
      t2( f'dnsmasq.conf is {cfg}' )
      return cfg

   def startCounterTimer( self ):
      if self.dnsCountersTimer_:
         return
      # start polling the DNS cache counters
      self.sendSignal( signal.SIGUSR1 )
      self.dnsCountersTimer_ = Tac.ClockNotifiee()
      self.dnsCountersTimer_.handler = self.handleDnsCacheCountersTimer
      self.dnsCountersTimer_.timeMin = Tac.now() + 1

   def serviceCmd( self, cmd ):
      if skipServiceCommandsTestHook:
         # No-op command that returns no error code.
         return [ "true" ]
      groupNameArg = [ self.groupName_ ] if self.groupName_ else []
      return super().serviceCmd( cmd ) + groupNameArg

   def startService( self ):
      bt1( 'DnsMasqService.startService for group',
           qv( self.groupName_ or 'default' ) )
      SuperServer.LinuxService.startService( self )
      self.started_ = True
      self.startCounterTimer()

   def restartService( self ):
      bt1( 'DnsMasqService.restartService for group',
           qv( self.groupName_ or 'default' ) )
      SuperServer.LinuxService.restartService( self )
      self.started_ = True
      self.startCounterTimer()

   def stopService( self ):
      bt1( 'DnsMasqService.stopService for group',
           qv( self.groupName_ or 'default' ) )
      SuperServer.LinuxService.stopService( self )
      self.started_ = False

   # don't need to override restart because we don't SIGHUP dnsmasq

class NetConfigNotifiee( SuperServer.GenericService ): # pylint: disable-msg=W0223
   """ Reactor to the 'sys/net/config' object that sets up the hostname, dns
   appropriately based on system configuration. """

   notifierTypeName = 'System::NetConfig'

   def __init__( self, serviceName, config, status, allVrfStatusLocal, ipStatus,
                 ip6Status, nsStatusLocalDir, allIntfStatusLocalDir, groupName="" ):
      bt0( 'NetConfigNotifiee.__init__ for group', qv( groupName or 'default' ) )
      self.config_ = config
      self.status_ = status
      self.allVrfStatusLocal_ = allVrfStatusLocal
      self.ipStatus_ = ipStatus
      self.ip6Status_ = ip6Status
      self.nsStatusLocalDir_ = nsStatusLocalDir
      self.allIntfStatusLocalDir_ = allIntfStatusLocalDir
      self.activity_ = None
      self.activityInterval_ = 60
      self.intfStatusReactor_ = None
      self.ipIntfStatusReactor_ = None
      self.ip6IntfStatusReactor_ = None
      self.groupName_ = groupName

      SuperServer.GenericService.__init__( self, serviceName, self.config_,
                                           sync=False )

      reservedHosts = self.getReservedHostnames()

      # Store the contents of hosts.Eos which should be copied as is
      # to /etc/hosts file first and then user configured host mappings
      # are added.
      self.hostsEosData = ''
      try:
         with open( '/etc/hosts.Eos', encoding='utf-8' ) as desc:
            self.hostsEosData = desc.read()
      except OSError as e:
         bt0( 'Error occured opening /etc/hosts.Eos:', qv( e ) )
      except: # pylint: disable=bare-except
         e = sys.exc_info()[ 1 ]
         bt0( 'Error occured opening /etc/hosts.Eos:', qv( e ) )

      # init this to none so that we can check in handleDefaultNs/etc.
      # Otherwise we wind up with a 'no attribute' error when we get
      # the callback from the __init__() function in NslReactor/etc
      self.dnsMasqService_ = None

      self.hostAddrReactor_ = Tac.collectionChangeReactor(
                              self.notifier_.hostAddr, HostAddrNotifiee,
                              reactorArgs=( self, reservedHosts ),
                              reactorTakesKeyArg=True )

      self.nslReactor_ = Tac.collectionChangeReactor(
         self.nsStatusLocalDir_.ns, NslReactor,
         reactorArgs=( weakref.proxy( self ), ),
         reactorFilter=lambda nsName: nsName == NsLib.DEFAULT_NS )

      self.allVrfStatusLocalReactor_ = \
          AllVrfStatusLocalReactor( self.allVrfStatusLocal_, self )

      # Set of tuple of (IpAddress, hostname). Note that same hostname can have
      # multiple ip address mapping and same ip address can be mapped to
      # multiple hostnames.
      self.hostMappings = set()
      self.syncHostAddrStatus( self.config_, status, reservedHosts )
      self.handleDnsCacheSize()
      for k in self.notifier_.sourceIntf:
         self.handleSourceIntf( k )
      self.handleHostname()
      self.handleExternalDnsProxy()

      # Do this last as it depends on NetStatus to be in sync with NetConfig
      self.dnsMasqService_ = DnsMasqService( self.allVrfStatusLocal_, self.status_,
                                             self.config_, self, self.ipStatus_,
                                             self.ip6Status_,
                                             self.nsStatusLocalDir_,
                                             self.allIntfStatusLocalDir_,
                                             self.groupName_ )
      self.sync()

   def vrfExists( self, vrf ):
      # pylint: disable-next=consider-using-in,singleton-comparison
      assert vrf != None and vrf != ''
      if vrf == DEFAULT_VRF:
         return True

      if not self.allVrfStatusLocal_.vrf.get( vrf ):
         return False
      if not self.allVrfStatusLocal_.vrf[ vrf ].state == 'active':
         return False
      return True

   # return number of non-default VRFs that exist and are active
   def numVrfs( self ):
      return sum( self.vrfExists( vrf ) for vrf in self.allVrfStatusLocal_.vrf )

   def handleVrfState( self, vrf ):
      assert vrf and vrf != ''
      bt1( 'NetConfigNotifiee.handleVrfState for', qv( vrf ) )
      if self.dnsMasqService_:
         self.sync()

   def handleDefaultNs( self ):
      bt1( 'NetConfigNotifiee.handleDefaultNs' )
      if self.dnsMasqService_:
         self.dnsMasqService_.sync()

   def handleIslNetNsName( self, intfId ):
      bt1( 'NetConfigNotifiee.handleIslNetNsName for', qv( intfId ) )
      if self.dnsMasqService_:
         self.dnsMasqService_.sync()

   def handleActiveAddrWithMask( self, intfId ):
      bt1( 'NetConfigNotifiee.handleActiveAddrWithMask for', qv( intfId ) )
      if self.dnsMasqService_:
         self.dnsMasqService_.sync()

   def handleAddr( self, intfId ):
      bt1( 'NetConfigNotifiee.handleAddr for', qv( intfId ) )
      if self.dnsMasqService_:
         self.dnsMasqService_.sync()

   def sync( self ):
      bt0( 'NetConfigNotifiee.sync for group', qv( self.groupName_ or 'default' ) )
      # note that we have to sync the netStatus first, because the
      # rest of the sync code uses the results of this so that we can
      # avoid replicating checks
      self.syncNetStatusToSysdb()
      # The hostname config is update with an explicit handler
      if self.dnsMasqService_:
         if not self.groupName_:
            # do not clobber resolv.conf for groupName dnsmasqs
            self.syncResolvConf()
         self.dnsMasqService_.sync()

   def syncResolvConf( self ):
      bt0( 'NetConfigNotifiee.syncResolvConf for group',
           qv( self.groupName_ or 'default' ) )
      if self.groupName_:
         # BUG833105
         # Hack: only generate this file for the system instance. Because glibc
         # only expects this file here it's shared by ALL INSTANCES. This behavior
         # is still broken because if priorities are configured we may not configure
         # enough attempts. It needs to be set to the largest of all groups' config
         # and the system config number of name servers.
         bt0( 'Not writing /etc/resolv.conf for group', qv( self.groupName_ ) )
         return None
      return self.writeConfigFile( '/etc/resolv.conf',
                                   self.resolvConfConfig(),
                                   updateInPlace=True )

   def legalDomainList( self ):
      '''Returns a list of domains that should be printed to the search path.
      Assumes that the domain and character limits on the search path were
      enforced by the Cli.
      '''
      domains = list( self.notifier_.domainList.values() )
      if self.notifier_.domainName:
         if self.notifier_.domainName in domains:
            domains.remove( self.notifier_.domainName )
         domains.insert( 0, self.notifier_.domainName )
      return domains

   def resolvConfConfig( self ):
      config = ""

      def _domainSearchPath():
         '''Returns the resolv.conf domain search path configuration option.
         Returns the empty string if there are no domains to search.
         '''
         searchPath = ' '.join( self.legalDomainList() )
         if searchPath:
            return "search %s\n" % searchPath
         else:
            return ''

      config += "\n"
      config += "#\n"
      config += "# As of EOS-4.11.0, all DNS requestes are now proxied through\n"
      config += "# dnsmasq, so that it can implement the DNS source interface\n"
      config += "# feature.  This means that the nameservers in use are now listed\n"
      config += "# in /etc/dnsmasq.conf.\n"
      config += "#\n"
      config += "\n"

      searchPath = _domainSearchPath()
      bt2( 'domain search path is', qv( searchPath ) )
      config += searchPath
      numV4Ns = len( self.status_.nameServer )
      numV6Ns = len( self.status_.v6NameServer )
      enabled = self.dnsMasqService_.defaultNsInitComplete()
      if numV4Ns + numV6Ns > 0 and enabled:
         bt2( 'adding 127.0.0.1 to resolv.conf' )
         config += "nameserver 127.0.0.1\n"
      else:
         bt0( 'dnsmasq is disabled,', qv( numV4Ns ), 'v4 nameservers', qv( numV6Ns ),
              'v6 nameservers', 'default ns init complete is', qv( enabled ) )

      allNameServers = itertools.chain(
            self.status_.nameServer.values(),
            self.status_.v6NameServer.values() )
      priorities = [ nameServer.priority for nameServer in allNameServers ]
      if any( priorities ):
         # If priorities are configured, strict-order is set and we configure
         # a number of attempts equal to the number of name server, trying each
         # one once.
         numAttempts = len( priorities )
         bt2( 'adding num attempts', qv( numAttempts ), 'to resolv.conf' )
         config += "options attempts:%d\n" % numAttempts

      # Don't dump the config to quicktrace. It's too big.
      t2( f'config is {config}' )
      return config

   def syncNetStatusToSysdb( self ):
      bt1( 'NetConfigNotifiee.syncNetStatusToSysdb' )
      self.status_.nameServer.clear()
      self.status_.v6NameServer.clear()

      allNameServers = list( itertools.chain(
            self.notifier_.nameServer.values(),
            self.notifier_.v6NameServer.values() ) )
      priorities = list( nameServer.priority for nameServer in allNameServers )
      largestValid, _ = getRedundantPriorityInfo( priorities )
      added = 0
      for nameServer in sorted( allNameServers ):
         ipAddr = nameServer.ipAddr
         serverAf = ipAddr.af
         vrf = nameServer.vrfName
         if self.vrfExists( vrf ):
            bt2( 'syncing', qv( ipAddr ), 'in vrf', qv( vrf ) )
            priority = nameServer.priority
            if largestValid and priority > largestValid:
               priority = largestValid
            statusNameServer = Tac.Value( "System::NameServer",
                                          nameServer.vrfIpPair, priority )
            if serverAf == af.ipv4:
               self.status_.nameServer.addMember( statusNameServer )
            else:
               assert serverAf == af.ipv6
               self.status_.v6NameServer.addMember( statusNameServer )
            added += 1
         else:
            bt0( 'vrf', qv( vrf ), 'does not exist for server', qv( ipAddr ) )
         if added >= self.notifier_.maxNameServers:
            break

      self.status_.domainName = self.notifier_.domainName

   @Tac.handler( 'sourceIntf' )
   def handleSourceIntf( self, vrf ):
      bt1( 'NetConfigNotifiee.handleSourceIntf for ', qv( vrf ) )
      # Create/update reactors for intfStatus and ip(6)Status
      if self.intfStatusReactor_:
         self.intfStatusReactor_.close()
         self.intfStatusReactor_ = None
      if self.ipIntfStatusReactor_:
         self.ipIntfStatusReactor_.close()
         self.ipIntfStatusReactor_ = None
      if self.ip6IntfStatusReactor_:
         self.ip6IntfStatusReactor_.close()
         self.ip6IntfStatusReactor_ = None

      assert vrf is not None

      if vrf in self.notifier_.sourceIntf:
         # we just copy this straight over into the status, the
         # dnsmasq generation code handles figuring out whether or not
         # we should actually use it based on the rest of the state in
         # the system
         self.status_.sourceIntf[ vrf ] = self.notifier_.sourceIntf[ vrf ]
      else:
         del self.status_.sourceIntf[ vrf ]
      sourceIntfIds = set( self.notifier_.sourceIntf.values() )
      _intfFilter = sourceIntfIds.__contains__

      self.intfStatusReactor_ = Tac.collectionChangeReactor(
         self.allIntfStatusLocalDir_.intfStatusLocal, IslReactor,
         reactorArgs=( weakref.proxy( self ), ), reactorFilter=_intfFilter )

      self.ipIntfStatusReactor_ = Tac.collectionChangeReactor(
         self.ipStatus_.ipIntfStatus, IpIntfStatusReactor,
         reactorArgs=( weakref.proxy( self ), ), reactorFilter=_intfFilter )

      self.ip6IntfStatusReactor_ = Tac.collectionChangeReactor(
         self.ip6Status_.intf, Ip6IntfStatusReactor,
         reactorArgs=( weakref.proxy( self ), ), reactorFilter=_intfFilter )

      self.sync()

   @Tac.handler( 'externalDnsProxy' )
   def handleExternalDnsProxy( self ):
      bt1( 'NetConfigNotifiee.handleExternalDnsProxy' )
      self.status_.externalDnsProxy = self.config_.externalDnsProxy
      self.sync()

   @Tac.handler( 'dnsCacheSize' )
   def handleDnsCacheSize( self ):
      bt1( 'NetConfigNotifiee.handleDnsCacheSize' )
      self.status_.dnsCacheSize = self.config_.dnsCacheSize
      self.sync()

   @Tac.handler( 'hostname' )
   def handleHostname( self ):
      bt1( 'NetConfigNotifiee.handleHostname' )
      if self.groupName_:
         return
      hostname = self.notifier_.hostname or 'localhost'

      # If the hostname contains '_', we replace it with '-' and program it.
      # Even though it's not a valid hostname, some customers do want to use it.
      sysHostname = hostname.replace( '_', '-' )
      curHostname = Tac.run( [ 'hostname' ], stdout=Tac.CAPTURE,
                             timeout=int( self.config_.hostnameTimeout ) ).strip()
      if sysHostname != curHostname:
         bt2( "Setting hostname to", qv( sysHostname ) )
         try:
            Tac.run( [ 'hostname', sysHostname ], stdout=Tac.CAPTURE,
                     timeout=int( self.config_.hostnameTimeout ) )
         except Tac.SystemCommandError as e:
            # XXX_APECH What to do in this case?
            bt0( "Failed to set hostname. Output:", qv( e.output ) )
            return
      else:
         bt0( "Linux hostname is already set to", qv( sysHostname ) )

      # Update the System::NetStatus object
      self.status_.hostname = hostname

   def writeToHostsFile( self ):
      mapData = self.hostsEosData + '\n'
      for mapping in self.hostMappings:
         mapData = mapData + '%s    %s' % ( mapping[ 0 ], mapping[ 1 ] ) + '\n'
      if not self.writeConfigFile( '/etc/hosts' + self.groupName_, mapData ):
         if self.activity_:
            self.activity_.timeMin = min( self.activity_.timeMin,
                                          Tac.now() + self.activityInterval_ )
         else:
            self.activity_ = Tac.ClockNotifiee()
            self.activity_.handler = self.writeToHostsFile
            self.activity_.timeMin = Tac.now() + self.activityInterval_

   def syncHostAddrStatus( self, config, status, reservedHosts ):
      # Syncs the hostAddr in status with what is present in config
      # Also populates the hostMappings used for writing to /etc/hosts
      for _name, hostObject in config.hostAddr.items():
         if not _name in reservedHosts and not _name in status.hostAddr:
            newEntry = status.newHostAddr( _name )
            for ipAddr in hostObject.ipAddr:
               newEntry.ipAddr[ ipAddr ] = True
            for ip6Addr in hostObject.ip6Addr:
               newEntry.ip6Addr[ ip6Addr ] = True

      for _name, hostObject in status.hostAddr.items():
         if not _name in config.hostAddr:
            del status.hostAddr[ _name ]
            continue
         for ipAddr in hostObject.ipAddr:
            self.hostMappings.add( ( ipAddr, _name ) )
         for ip6Addr in hostObject.ip6Addr:
            self.hostMappings.add( ( ip6Addr.stringValue, _name ) )
      self.writeToHostsFile()

   # Returns a list of reserved hostnames by reading /etc/hosts.Eos
   # These are always present in /etc/hosts.
   def getReservedHostnames( self ):
      reservedHosts = []
      try:
         # pylint: disable-next=consider-using-with
         hostsEosRead = open( '/etc/hosts.Eos' )
         line = hostsEosRead.readline()
         while line != '':
            if addrMapping( line ):
               segments = line.split()
               segments.remove( segments[ 0 ] )
               for _name in segments:
                  reservedHosts.append( _name )
            line = hostsEosRead.readline()
         hostsEosRead.close()
      except OSError as e:
         bt0( "Error occurred opening Eos host file:", qv( e ) )
      except: # pylint: disable=bare-except
         e = sys.exc_info()[ 1 ]
         bt0( "Error occurred opening Eos host file:", ( e ) )
      return reservedHosts

   def handleHostnameToIpMapping( self, hostname, address, ipv6=False,
                                  deleteAll=False ):

      def finish():
         if rewriteEtcHosts:
            self.writeToHostsFile()
         if self.dnsMasqService_:
            # Force a restart. SuperServer will elide the restart if the config file
            # (dnsmasq.conf) doesn't change. Host addr changes affect /etc/hosts.
            self.dnsMasqService_.forceRestart_ = True
            self.dnsMasqService_.sync()

      if deleteAll:
         # hostAddr object for given name must not be present in netConfig
         # delete hostname from netStatus and remove all mappings for it
         bt2( "removing all the mapping for host: ", qv( hostname ) )
         del self.status_.hostAddr[ hostname ]
         rewriteEtcHosts = False
         removeMappings = [ mapping for mapping in self.hostMappings if
                     mapping[ 1 ] == hostname ]
         for mapping in removeMappings:
            self.hostMappings.remove( mapping )
            rewriteEtcHosts = True
         finish()
         return

      if not hostname in self.status_.hostAddr:
         # if a new hostAddr object is added to netConfig
         # create a new one in netStatus also
         self.status_.newHostAddr( hostname )
         bt2( "adding a new HostAddr object '", qv( hostname ), "' to netStatus" )

      if ipv6:
         configIpList = self.notifier_.hostAddr[ hostname ].ip6Addr
         statusIpList = self.status_.hostAddr[ hostname ].ip6Addr
      else:
         configIpList = self.notifier_.hostAddr[ hostname ].ipAddr
         statusIpList = self.status_.hostAddr[ hostname ].ipAddr

      rewriteEtcHosts = False

      if address is not None:
         addrString = address.stringValue if ipv6 else str( address )
         if address in configIpList:
            # new mapping is added to netConfig
            bt2( "adding a new mapping: ", qv( hostname ), "->", qv( addrString ) )
            statusIpList[ address ] = True
            self.hostMappings.add( ( addrString, hostname ) )
            rewriteEtcHosts = True
         else:
            # a mapping is deleted from netConfig
            del statusIpList[ address ]
            bt2( "removing the mapping: ", qv( hostname ), "->", qv( addrString ) )
            self.hostMappings.discard( ( addrString, hostname ) )
            rewriteEtcHosts = True
            # deleting the HostAddr object if it has no more mappings
            if not ( self.status_.hostAddr[ hostname ].ipAddr or
                     self.status_.hostAddr[ hostname ].ip6Addr ):
               del self.status_.hostAddr[ hostname ]
      else:
         # synchronising all the addresses when None is passed as address/key
         bt2( "synchronising all ", qv( "ipv6" if ipv6 else "ipv4" ),
              " mappings for host: ", qv( hostname ) )
         for address in statusIpList: # pylint: disable=redefined-argument-from-local
            addrString = address.stringValue if ipv6 else str( address )
            if not address in configIpList:
               del statusIpList[ address ]
               self.hostMappings.discard( ( addrString, hostname ) )
               rewriteEtcHosts = True
         for address in configIpList: # pylint: disable=redefined-argument-from-local
            addrString = address.stringValue if ipv6 else str( address )
            if not address in statusIpList:
               statusIpList[ address ] = True
               self.hostMappings.add( ( addrString, hostname ) )
               rewriteEtcHosts = True

      finish()

class NetStatusNotifiee( Tac.Notifiee ):
   """ Reactor to the 'sys/net/status' object that sets up the fqdn
   appropriately based on system configuration. """

   notifierTypeName = 'System::NetStatus'

   def __init__( self, notifier ):
      bt0( 'NetStatusNotifee.__init__' )
      Tac.Notifiee.__init__( self, notifier )
      self.handleHostname()

   @Tac.handler( 'hostname' )
   def handleHostname( self ):
      bt1( 'NetStatusNotifiee.handleHostname' )
      hostname = self.notifier_.hostname
      domainName = self.notifier_.domainName
      if domainName and not hostname.endswith( domainName ):
         self.notifier_.fqdn = hostname + '.' + domainName
      else:
         self.notifier_.fqdn = hostname

   @Tac.handler( 'domainName' )
   def handleDomainName( self ):
      bt1( 'NetStatusNotifiee.handleDomainName' )
      hostname = self.notifier_.hostname
      domainName = self.notifier_.domainName
      if not hostname.endswith( domainName ) and domainName:
         self.notifier_.fqdn = hostname + '.' + domainName
      else:
         self.notifier_.fqdn = hostname

class NetworkManager( SuperServer.SuperServerAgent ):
   """ This agent reacts to the 'system/net/config' object and configures
   hostname, dns, etc on the host """
   def __init__( self, entityManager ):
      bt0( 'NetworkManager.__init__' )
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      groupConfigDir = mg.mount( 'sys/net/groupConfigDir', 'Tac::Dir', 'ri' )
      groupStatusDir = mg.mount( Cell.path( 'sys/net/groupStatusDir' ),
                                 'Tac::Dir', 'wfi' )
      networkConfig = mg.mount( 'sys/net/config', 'System::NetConfig', 'r' )
      networkStatus = mg.mount( Cell.path( 'sys/net/status' ),
                                'System::NetStatus', 'wf' )
      avsl = mg.mount( Cell.path( 'ip/vrf/status/local' ),
                       'Ip::AllVrfStatusLocal', 'r' )
      Tac.Type( "Ira::IraIpStatusMounter" ).doMountEntities( mg.cMg_, True, True )
      ipStatus = mg.mount( 'ip/status', 'Ip::Status', 'r' )
      ip6Status = mg.mount( 'ip6/status', 'Ip6::Status', 'r' )
      nsld = mg.mount( Cell.path( 'namespace/status' ),
                       'Arnet::NamespaceStatusLocalDir', 'r' )
      # Get local interface entity, created by SuperServer
      aisld = self.intfStatusLocal()
      self.fqdnNotifiee_ = None
      self.serviceList_ = {}
      self.service_ = None
      self.status_ = None
      self.nameServerGroupReactor_ = None

      def _finished():
         bt0( 'NetworkManager._finished' )
         # pylint: disable=attribute-defined-outside-init
         self.status_ = networkStatus
         self.groupConfigDir_ = groupConfigDir
         self.groupStatusDir_ = groupStatusDir
         self.avsl_ = avsl
         self.ipStatus_ = ipStatus
         self.ip6Status_ = ip6Status
         self.nsld_ = nsld
         self.aisld_ = aisld
         self.fqdnNotifiee_ = NetStatusNotifiee( networkStatus )
         self.service_ = NetConfigNotifiee( "NetConfig", networkConfig,
                                            networkStatus, avsl, ipStatus,
                                            ip6Status, nsld, aisld )
         self.createReactors()
      mg.close( _finished )

   def createReactors( self ):
      self.nameServerGroupReactor_ = \
         NameServerGroupReactor( self.groupConfigDir_, self.groupStatusDir_,
                                 self.serviceList_, self.avsl_,
                                 self.ipStatus_, self.ip6Status_,
                                 self.nsld_, self.aisld_ )

def Plugin( ctx ):
   ctx.registerService( NetworkManager( ctx.entityManager ) )
