# Copyright (c) 2007-2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import SuperServer
import Tac
import Cell
import SnmpObjectId
import os
import re
import random
import socket
import weakref
from PyWrappers.NetSnmp import snmpdName
from SnmpNsName import hexlifyNsName
from IpLibConsts import DEFAULT_VRF
import Aresolve
import Tracing
import QuickTrace

CbStrategy = Aresolve.CallbackStrategy

t0 = Tracing.trace0
t5 = Tracing.trace5
t9 = Tracing.trace9

qv = QuickTrace.Var
qt0 = QuickTrace.trace0

__defaultTraceHandle__ = Tracing.Handle( "Snmpd" )

notificationTimeout = 10
# 10 minutes of retries
notificationRetries = 10 * 60 // notificationTimeout

# maps from snmpd counters file name to Snmp::Counters attr name
_counterMap = {
   "inPkts": "inPkts",
   "outPkts": "outPkts",
   "inBadVersions": "inVersionErrs",
   "inBadCommunityNames": "inBadCommunityNames",
   "inBadCommunityUses": "inBadCommunityUses",
   "inAsnParseErrors": "inParseErrs",
   "inTotalRequestVars": "inRequestVars",
   "inTotalSetVars": "inSetVars",
   "inGetRequests": "inGetPdus",
   "inGetNexts": "inGetNextPdus",
   "inSetRequests": "inSetPdus",
   "outTooBigs": "outTooBigErrs",
   "outNoSuchNames": "outNoSuchNameErrs",
   "outBadValues": "outBadValueErrs",
   "outGenErrs": "outGeneralErrs",
   "outGetResponses": "outGetResponsePdus",
   "outTraps": "outTrapPdus"
}

# Counters cannot be updated more frequently than this value, given in seconds
_minUpdateCountersFreq = 1.0

VrfState = Tac.Type( 'Ip::VrfState' )

# Period between DNS queries made by Aresolve for existing queries
DNS_RETRY_PERIOD = 120


class GenericReactor( Tac.Notifiee ):
   notifierTypeName = "*"

   def __init__( self, notifier, master, what ):
      self.master_ = master
      self.what_ = what
      Tac.Notifiee.__init__( self, notifier, filtered=False )
      # notify the master config monitor
      self.master_.handleConfigChange( "initial notification" )

   def onAttribute( self, attr, key ):
      # Notify any handlers that have been registered
      Tac.Notifiee.onAttribute( self, attr, key )
      # notify the master config monitor
      self.master_.handleConfigChange( "%s (%s)" % ( self.what_, attr.name ) )

# This is the reactor for dealing with SNMP debug tokens.
# This isn't the optimal way to deal with things - when we
# receive a change notification, we setup our timer activity to
# be scheduled shortly. The timer handler simply lets the master
# (our SNMP super service) deal with a generic "config change"
# which will result in us restarting snmpd service.
# We are attempting to coalesce multiple tokens being provided
# using the Debug infrastructure with this deferred model.
#
# We want to schedule this activity timer only when a change is
# detected.
#
# Given the # of times debugging will be enabled and disabled
# is reasonably small, I think this will do for now.

class DebugTokensReactor( Tac.Notifiee ):

   notifierTypeName = 'Debug::Category'

   def __init__( self, notifier, master, what ):
      self.master_ = master
      self.what_ = what
      Tac.Notifiee.__init__( self, notifier )
      self.debugTokensDeferredHandler_ = Tac.ClockNotifiee(
         self._handleDebugTokensChange )
      self.debugTokensDeferredHandler_.timeMin = Tac.endOfTime

   @Tac.handler( 'messageType' )
   def handleDebugMessageType( self, name ):
      h = self.debugTokensDeferredHandler_
      if h.timeMin == Tac.endOfTime:
         h.timeMin = Tac.now() + 0.01

   def _handleDebugTokensChange( self ):
      t9( "timer expired, let the master deal with debug change" )
      self.master_.handleConfigChange( self.what_ )
      self.debugTokensDeferredHandler_.timeMin = Tac.endOfTime

class EngineIdReactor( Tac.Notifiee ):
   notifierTypeName = "Snmp::EngineIdGenerator::Status"

   def __init__( self, generator, master ):
      self.master_ = master
      Tac.Notifiee.__init__( self, generator )

   @Tac.handler( "engineId" )
   def handleEngineId( self ):
      # Engine ID is a pre-mib configuration handler, so it doesn't get
      # called when snmpd is reloaded.
      self.master_.forceRestart_ = True
      self.master_.handleConfigChange( "engineId" )

class SysDescrReactor( Tac.Notifiee ):
   notifierTypeName = "EntityMib::SysDescr"

   def __init__( self, sysDescrSm, master ):
      self.master_ = master
      Tac.Notifiee.__init__( self, sysDescrSm )

   @Tac.handler( "value" )
   def handleSysDescr( self ):
      self.master_.handleConfigChange( "sysDescr" )

class EntityMibRootReactor( Tac.Notifiee ):
   """This reactor is instantiated when the EntityMib::Status::root attribute
   is set."""
   notifierTypeName = 'EntityMib::Fru'

   def __init__( self, rootFru, master ):
      t0( "Instantiating reactor for", rootFru )
      self.master_ = master
      Tac.Notifiee.__init__( self, rootFru )

   @Tac.handler( 'modelName' )
   def handleModelName( self ):
      self.master_.handleConfigChange( "modelName" )

class EntityMibStatusReactor( Tac.Notifiee ):
   """Instantiates EntityMibRootReactor."""
   notifierTypeName = 'EntityMib::Status'

   def __init__( self, entityMibStatus, master ):
      self.master_ = master
      self.rootReactor_ = None
      Tac.Notifiee.__init__( self, entityMibStatus )

   @Tac.handler( 'root' )
   def handleRoot( self ):
      root = self.notifier_.root
      if root:
         self.rootReactor_ = EntityMibRootReactor( root, self.master_ )

class CommunityReactor( GenericReactor ):
   notifierTypeName = "Snmp::Community"

   def __init__( self, notifier, master, what ):
      GenericReactor.__init__( self, notifier, master, what )
      self.acl_ = ''
      self.acl6_ = ''
      self.handleAcl()
      self.handleAcl6()

   @Tac.handler( 'acl' )
   def handleAcl( self ):
      if self.acl_:
         t0( "remove (", self.acl_, self.notifier_.name, ") from acl interest" )
         self.master_.aclInterestDel( self.notifier_.name, self.acl_, 'ip' )
      self.acl_ = self.notifier_.acl
      if self.acl_:
         self.master_.aclInterestIs( self.notifier_.name, self.acl_, 'ip' )
         t0( "add (", self.acl_, self.notifier_.name, ") to acl interest" )

   @Tac.handler( 'acl6' )
   def handleAcl6( self ):
      if self.acl6_:
         t0( "remove (", self.acl6_, self.notifier_.name, ") from acl interest" )
         self.master_.aclInterestDel( self.notifier_.name, self.acl6_, 'ipv6' )
      self.acl6_ = self.notifier_.acl6
      if self.acl6_:
         self.master_.aclInterestIs( self.notifier_.name, self.acl6_, 'ipv6' )
         t0( "add (", self.acl6_, self.notifier_.name, ") to acl interest" )

   def close( self ):
      if self.acl_:
         t0( "remove (", self.acl_, self.notifier_.name, ") from acl interest" )
         self.master_.aclInterestDel( self.notifier_.name, self.acl_, 'ip' )
      if self.acl6_:
         t0( "remove (", self.acl6_, self.notifier_.name, ") from acl interest" )
         self.master_.aclInterestDel( self.notifier_.name, self.acl6_, 'ipv6' )
      super().close()

class NotificationSinkReactor( GenericReactor ):
   notifierTypeName = "Snmp::NotificationSink"

   def close( self ):
      t0( "NotificationSinkReactor close, force restart" )
      self.master_.forceRestart_ = True
      super().close()

class VrfConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Snmp::Config"

   def __init__( self, notifier, master ):
      self.master_ = master
      Tac.Notifiee.__init__( self, notifier )

   @Tac.handler( 'vrf' )
   def handleVrfConfig( self, vrf ):
      self.master_.forceRestart_ = True
      self.master_.handleConfigChange( 'SNMP %s in VRF %s' %
         ( 'enabled' if vrf in self.master_.config_.vrf else 'disabled', vrf ) )

class ExtensionConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Snmp::Config"

   def __init__( self, notifier, master ):
      self.master_ = master
      Tac.Notifiee.__init__( self, notifier )

   @Tac.handler( 'extension' )
   def handleExtensionConfig( self, extension ):
      action = 'added' if extension in self.master_.config_.extension else 'removed'
      self.master_.forceRestart_ = True
      self.master_.handleConfigChange( 'Snmp extension oid %s %s ' %
                                       ( extension, action ) )

class NetConfigReactor( Tac.Notifiee ):
   notifierTypeName = "System::NetConfig"

   def __init__( self, notifier, master ):
      self.master_ = master
      Tac.Notifiee.__init__( self, notifier )

   @Tac.handler( 'hostname' )
   def handleHostname( self ):
      self.master_.handleConfigChange( 'hostname' )

   @Tac.handler( 'domainName' )
   def handleDomainName( self ):
      self.master_.handleConfigChange( 'domainName' )

class VrfStatusLocalReactor( Tac.Notifiee ):
   notifierTypeName = "Ip::VrfStatusLocal"

   def __init__( self, vrfStatusLocal, master ):
      self.master_ = master
      self.vrfName = vrfStatusLocal.vrfName
      self.vrfStatusLocal = vrfStatusLocal
      Tac.Notifiee.__init__( self, vrfStatusLocal )

   @Tac.handler( 'state' )
   def handleState( self ):
      if self.vrfName in self.master_.config_.vrf:
         self.master_.forceRestart_ = True
         self.master_.handleConfigChange( 'vrf %s changed state' %
                                          self.vrfName )

   def close( self ):
      assert self.vrfStatusLocal.state == Tac.Type( "Ip::VrfState" ).deleting
      Tac.Notifiee.close( self )

class L3StatusReactor( Tac.Notifiee ):
   notifierTypeName = "L3::Intf::Status"

   def __init__( self, l3Status, master ):
      self.master_ = master
      Tac.Notifiee.__init__( self, l3Status )

   @Tac.handler( 'vrf' )
   def handleVrf( self ):
      self.master_.handleConfigChange( 'vrf' )

class IpIntfStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Ip::IpIntfStatus"

   def __init__( self, ipIntfStatus, master ):
      self.master_ = master
      Tac.Notifiee.__init__( self, ipIntfStatus )
      self.l3StatusReactor_ = L3StatusReactor(
         self.notifier_.l3Status, self.master_ )

   @Tac.handler( 'activeAddrWithMask' )
   def handleActiveAddrWithMask( self ):
      self.master_.handleConfigChange( 'activeAddrWithMask' )

class IpStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Ip::Status"

   def __init__( self, ipStatus, master ):
      self.master_ = master
      Tac.Notifiee.__init__( self, ipStatus )
      self.ipIntfStatusReactors_ = {}
      self.requestedKeys_ = set()

   def requestedKeysIs( self, requestedKeys ):
      self.requestedKeys_ = requestedKeys
      # We can delete while iterating, so must use list()
      for intfName in list( self.ipIntfStatusReactors_ ):
         if intfName not in self.requestedKeys_:
            self.ipIntfStatusReactors_[ intfName ].close()
            del self.ipIntfStatusReactors_[ intfName ]

      for intfName in self.requestedKeys_:
         if intfName not in self.ipIntfStatusReactors_ and \
            intfName in self.notifier_.ipIntfStatus:
            self.ipIntfStatusReactors_[ intfName ] = IpIntfStatusReactor(
               self.notifier_.ipIntfStatus[ intfName ], self.master_ )

   @Tac.handler( 'ipIntfStatus' )
   def handleIpIntfStatus( self, intfName ):
      if intfName in self.requestedKeys_:
         if intfName in self.notifier_.ipIntfStatus:
            self.ipIntfStatusReactors_[ intfName ] = IpIntfStatusReactor(
               self.notifier_.ipIntfStatus[ intfName ], self.master_ )
            self.master_.handleConfigChange( 'ipIntfStatus created' )
         else:
            assert intfName in self.ipIntfStatusReactors_
            self.ipIntfStatusReactors_[ intfName ].close()
            del self.ipIntfStatusReactors_[ intfName ]
            self.master_.handleConfigChange( 'ipIntfStatus deleted' )

class Ip6IntfStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Ip6::IntfStatus"

   def __init__( self, intf, master ):
      self.master_ = master
      Tac.Notifiee.__init__( self, intf )
      self.l3StatusReactor_ = L3StatusReactor(
         self.notifier_.l3Status, self.master_ )

   @Tac.handler( 'addr' )
   def handleAddr( self, collectionKey ):
      self.master_.handleConfigChange( 'addr' )

class Ip6StatusReactor( Tac.Notifiee ):
   notifierTypeName = "Ip6::Status"

   def __init__( self, ip6Status, master ):
      self.master_ = master
      Tac.Notifiee.__init__( self, ip6Status )
      self.ip6IntfStatusReactors_ = {}
      self.requestedKeys_ = set()

   def requestedKeysIs( self, requestedKeys ):
      self.requestedKeys_ = requestedKeys
      # We can delete while iterating, so must use list()
      for intfName in list( self.ip6IntfStatusReactors_ ):
         if intfName not in self.requestedKeys_:
            self.ip6IntfStatusReactors_[ intfName ].close()
            del self.ip6IntfStatusReactors_[ intfName ]

      for intfName in self.requestedKeys_:
         if intfName not in self.ip6IntfStatusReactors_ and \
            intfName in self.notifier_.intf:
            self.ip6IntfStatusReactors_[ intfName ] = Ip6IntfStatusReactor(
               self.notifier_.intf[ intfName ], self.master_ )

   @Tac.handler( 'intf' )
   def handleIntf( self, intfName ):
      if intfName in self.requestedKeys_:
         if intfName in self.notifier_.intf:
            self.ip6IntfStatusReactors_[ intfName ] = Ip6IntfStatusReactor(
               self.notifier_.intf[ intfName ], self.master_ )
            self.master_.handleConfigChange( 'intf created' )
         else:
            assert intfName in self.ip6IntfStatusReactors_
            self.ip6IntfStatusReactors_[ intfName ].close()
            del self.ip6IntfStatusReactors_[ intfName ]
            self.master_.handleConfigChange( 'intf deleted' )

# currCfg changes when the acl contents change, to allow atomic updates.
class IpAclReactor( Tac.Notifiee ):
   notifierTypeName = "Acl::AclConfig"

   def __init__( self, notifier, master ):
      self.master_ = master
      Tac.Notifiee.__init__( self, notifier )

   @Tac.handler( 'currCfg' )
   def handleCurrCfg( self ):
      self.master_.handleConfigChange( "acl %s" % self.notifier_.name )

   def close( self ):
      self.master_.handleConfigChange( "delete acl %s" % self.notifier_.name )
      Tac.Notifiee.close( self )

# React to acl configurations coming and going, and instantiate
# IpAclReactors to any acl that we think we want to know about.
class IpAclConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Acl::Input::AclTypeConfig"

   def __init__( self, notifier, master ):
      self.master_ = master
      self.type_ = notifier.type
      Tac.Notifiee.__init__( self, notifier )

   @Tac.handler( 'acl' )
   def handleAcl( self, key ):
      # If it's being added, and we're interested in it, but we don't
      # yet have a reactor to it, add a reactor to it.
      if key in self.notifier_.acl and \
         ( key, self.type_ ) in self.master_.aclInterest_ and \
         ( key, self.type_ ) not in self.master_.aclReactor_:

         self.master_.aclReactor_[ ( key, self.type_ ) ] = \
            IpAclReactor( self.notifier_.acl[ key ], self.master_ )
         return

      # If it's being deleted, and we have a reactor to it, close and
      # delete that reactor.
      if key not in self.notifier_.acl and \
         ( key, self.type_ ) in self.master_.aclReactor_:

         self.master_.aclReactor_[ ( key, self.type_ ) ].close()
         del self.master_.aclReactor_[ ( key, self.type_ ) ]

# This data is global so that the SnmpdService can share it
# with the SnmpPassphraseLocalizationTest.
authTypeOidMap = {
   # https://tools.ietf.org/html/rfc3414
   # SNMP-USER-BASED-SM-MIB
   'authNone': '.1.3.6.1.6.3.10.1.1.1',
   'authMd5': '.1.3.6.1.6.3.10.1.1.2',
   'authSha': '.1.3.6.1.6.3.10.1.1.3',
   # https://tools.ietf.org/html/rfc7630
   # SNMP-USM-HMAC-SHA2-MIB
   'authSha224': '.1.3.6.1.6.3.10.1.1.4',
   'authSha256': '.1.3.6.1.6.3.10.1.1.5',
   'authSha384': '.1.3.6.1.6.3.10.1.1.6',
   'authSha512': '.1.3.6.1.6.3.10.1.1.7' }
privTypeOidMap = {
   # https://tools.ietf.org/html/rfc3414
   # SNMP-USER-BASED-SM-MIB
   'privacyNone': '.1.3.6.1.6.3.10.1.2.1',
   'privacyDes': '.1.3.6.1.6.3.10.1.2.2',
   # https://tools.ietf.org/html/rfc3826
   # SNMP-USM-AES-MIB
   'privacyAes': '.1.3.6.1.6.3.10.1.2.4',
   # http://www.snmp.com/eso/esoConsortiumMIB.txt
   # Industry standard ESO-CONSORTIUM-MIB
   'privacy3Des': '.1.3.6.1.4.1.14832.1.1',
   'privacyAes192': '.1.3.6.1.4.1.14832.1.3',
   'privacyAes256': '.1.3.6.1.4.1.14832.1.4',
   }

class SnmpdService( SuperServer.LinuxService ):
   """Manages the snmpd service based on configuration mounted at
   /snmp/config"""
   notifierTypeName = "Snmp::Config"

   def __init__( self, config, netConfig, counters, ipStatus, ip6Status,
                 vrfStatusLocal, debugConfig, aclConfig, snmpStatus,
                 entityMibStatus ):
      self.config_ = config
      self.netConfig_ = netConfig
      self.counters_ = counters
      self.ipStatus_ = ipStatus
      self.ip6Status_ = ip6Status
      self.vrfStatusLocal_ = vrfStatusLocal
      self.aclConfig_ = aclConfig
      self.snmpStatus_ = snmpStatus
      self.entityMibStatus_ = entityMibStatus
      if not os.path.exists( "/etc/snmp" ):
         try:
            os.makedirs( "/etc/snmp", 0o755 )
         except OSError as e:
            t0( "Failed to create /etc/snmp:", e )
      # These configuration files will be passed into the LinuxService
      # constructor as a list. The order here does matter.
      confFiles = [ '/etc/snmp/snmpd.conf', '/etc/sysconfig/snmpd' ]
      SuperServer.LinuxService.__init__( self, "snmpd", snmpdName(),
                                         config, confFiles )

      # The /var/run/agentx directory contains the socket that snmpd and AgentX
      # processes use to communicate. Apparently snmpd will not create the
      # directory if it doesn't exist, so we need to make sure it exists before
      # running snmpd.
      agentXSockDir = "/var/run/agentx"
      if not os.path.exists( agentXSockDir ):
         try:
            # I use makedirs here insead of mkdir for the situation in which I
            # am run chrooted into some directory where /var doesn't yet exist.
            os.makedirs( agentXSockDir, 0o711 )
         except OSError:
            t0( "Failed to create directory for AgentX socket:", agentXSockDir )

      self.engineIdGenerator_ = Tac.newInstance( "Snmp::EngineIdGenerator",
                                                 self.entityMibStatus_, config )
      self.engineIdReactor_ = EngineIdReactor( self.engineIdGenerator_.status,
                                               self )

      self.dns_ = Aresolve.Querier( self._dnsResponse, shortTime=1,
                                    longTime=DNS_RETRY_PERIOD,
                                    callbackStrategy=CbStrategy.OnUsedIpMissing,
                                    useDnsQuerySm=True )
      # To keep track of currently used hostnames
      self.snmpHostnames_ = []

      # Start an activity for monitoring snmpd's counters file
      self.counterActivity_ = Tac.ClockNotifiee()
      self.counterActivity_.handler = self._updateCounters
      self._scheduleCounterActivity()

      # Tests may override this to a smaller number to run faster
      self.oomScoreAdjUpdateInterval_ = int( os.environ.get(
            "OVERRIDE_OOMSCOREADJ_INTERVAL", 30 ) )
      self.oomScoreAdjUpdateActivity_ = Tac.ClockNotifiee(
            handler=self._oomScoreAdjUpdate, timeMin=Tac.endOfTime )
      self._scheduleOomScoreAdjUpdate()

      self.aclInterest_ = {}
      self.aclReactor_ = {}

      self.sysDescr_ = Tac.newInstance( "EntityMib::SysDescr" )
      self.sysDescrSm_ = Tac.newInstance(
         "EntityMib::SysDescrSm", self.entityMibStatus_, self.sysDescr_ )
      self.sysDescrReactor_ = SysDescrReactor( self.sysDescr_, self )

      self.entityMibStatusReactor_ = EntityMibStatusReactor( self.entityMibStatus_,
                                                             self )
      self.netConfigReactor_ = NetConfigReactor( self.netConfig_, self )
      # SuperServer.GenericService gets notifications for config's attributes,
      # but we also need to monitor the attributes of all of the config's
      # collection attributes.
      self.viewCollReactor_ = Tac.collectionChangeReactor(
         config.view, GenericReactor,
         reactorArgs=( self, "view" ) )
      self.communityCollReactor_ = Tac.collectionChangeReactor(
         config.community, CommunityReactor,
         reactorArgs=( self, "community" ) )
      self.userCollReactor_ = Tac.collectionChangeReactor(
         config.user, GenericReactor,
         reactorArgs=( self, "user" ) )
      self.groupCollReactor_ = Tac.collectionChangeReactor(
         config.group, GenericReactor,
         reactorArgs=( self, "group" ) )
      self.notificationSinkCollReactor_ = Tac.collectionChangeReactor(
         config.notificationSink, NotificationSinkReactor,
         reactorArgs=( self, "notificationSink" ) )
      self.notificationConfigCollReactor_ = Tac.collectionChangeReactor(
         config.notificationConfig, GenericReactor,
         reactorArgs=( self, "notificationConfig" ) )
      self.snmpNotificationConfigCollReactor_ = Tac.collectionChangeReactor(
         config.notificationConfig[ 'snmp' ].notification, GenericReactor,
         reactorArgs=( self, "notification" ) )
      self.internalLocalizedKey_ = "".join( [ "%02x" % random.randint( 0, 255 )
                                              for _ in range( 20 ) ] )
      resync = not self.writeConfigFile( "/etc/snmp/snmp.conf",
         """# Authentication for internal SNMP requests
mibs ALL
defVersion 3
defCommunity public
defSecurityName __internal__
defSecurityLevel authNoPriv
defAuthType SHA
# This is a randomly-selected key, generated at system boot.
defAuthLocalizedKey 0x%s
""" % self.internalLocalizedKey_ )
      self.ipStatusReactor_ = IpStatusReactor( self.ipStatus_, self )
      self.ip6StatusReactor_ = Ip6StatusReactor( self.ip6Status_, self )
      self.vrfConfigReactor_ = VrfConfigReactor( self.config_, self )
      self.extensionConfigReactor_ = ExtensionConfigReactor( self.config_, self )
      self.vrfStatusLocalReactor_ = \
          Tac.collectionChangeReactor( self.vrfStatusLocal_.vrf,
                                       VrfStatusLocalReactor,
                                       reactorArgs=( weakref.proxy( self ), ) )
      # Hack to prevent atexit from calling close(), which could hit
      # the assert in VrfStatusLocalReactor.close():
      self.vrfStatusLocalReactor_.inCollection = True
      self.ipAclConfigReactor_ = \
            IpAclConfigReactor( self.aclConfig_.config[ 'ip' ], self )
      self.ip6AclConfigReactor_ = \
            IpAclConfigReactor( self.aclConfig_.config[ 'ipv6' ], self )

      self.secNameMap_ = {}
      self.secNameIndex_ = 1

      # We use our knowledge about how the SNMP debugging path is
      # defined in Sysdb. We are only interested in the changes to
      # the tokens themselves. When user isues 'no debug all' perhaps
      # our reactor will get called several times but it's probably
      # okay for now (they are coalesced anyway - see reactor definition).
      t9( "setting up debug tokens reactor" )
      snmpDebug = debugConfig.subcategory[ 'snmp' ]
      snmpdDebug = snmpDebug.subcategory[ 'snmpd' ]
      self.tokensConfig_ = snmpdDebug.subcategory[ 'tokens' ]
      self.debugMessageReactor_ = DebugTokensReactor( self.tokensConfig_,
                                                      self, 'debugConfig' )
      if resync:
         self.sync()

   def getIpAddr( self, ipOrHost ):
      hostname, addrs = self.dns_.handleIpAddrOrHostname( ipOrHost )

      if hostname:
         # Keep track of all used hostnames, so we can do cleanup later.
         self.snmpHostnames_.append( hostname )

      if not addrs:
         # We have a hostname, but it's not resolved yet.
         return None

      if hostname in self.dns_.usedIPs:
         # Hostname is resolved and one of the IPs is already selected, so re-use it.
         addr = self.dns_.usedIPs[ hostname ]
      else:
         # Either IP has been configured manually (no hostname) or we have hostname,
         # but haven't selected IP yet.
         addr = addrs[ 0 ]
         # Select IP if this is the hostname case.
         if hostname:
            self.dns_.usedIPs[ hostname ] = addrs[ 0 ]

      return addr

   def _dnsResponse( self, record ):
      """Handles a DNS response
      Args:
        record: a Aresolve.DnsRecord, the results from Aresolve.
      """
      if record.valid:
         self.sync()

   def aclInterestIs( self, community, acl, aclType ):
      # Maintain acl -> list-of-communities.
      # If we're inserting an acl, create its reactor.
      t0( 'aclInterestIs', community, acl, aclType )
      if ( acl, aclType ) not in self.aclInterest_:
         self.aclInterest_[ ( acl, aclType ) ] = {}
         a = self.aclConfig_.config[ aclType ].acl.get( acl )
         if a:
            self.aclReactor_[ ( acl, aclType ) ] = IpAclReactor( a, self )
      self.aclInterest_[ ( acl, aclType ) ][ community ] = True

   def aclInterestDel( self, community, acl, aclType ):
      t0( 'aclInterestDel', community, acl, aclType )
      del self.aclInterest_[ ( acl, aclType ) ][ community ]
      if not self.aclInterest_[ ( acl, aclType ) ]:
         del self.aclInterest_[ ( acl, aclType ) ]
         if ( acl, aclType ) in self.aclReactor_:
            del self.aclReactor_[ ( acl, aclType ) ]

   def serviceProcessWarm( self ):
      if not self.serviceEnabled():
         return True
      # XXX: need to make SNMP request to see if snmpd is up and answering
      return True

   def _monotonicTimeFromSnmpStatus( self ):
      # We convert the systemStartTime (which includes the SSO time delta) to Linux
      # monotonic time to represent this supervisor's view of the value.  Such delta
      # is implemented inside, e.g., Tac.now() but snmpd uses the underlying
      # CLOCK_MONOTONIC.  Converting the time here means that the value will likely
      # be negative, but this is handled properly inside snmpd.
      return Tac.tacNowToLinuxMonoTime( self.snmpStatus_.systemStartTime )

   def sysObjectId( self ):
      if not self.entityMibStatus_.root:
         return ""

      modelName = self.entityMibStatus_.root.modelName
      return SnmpObjectId.objectIdFromModelName( modelName )

   def _generateSnmpConf( self ):
      """Generate new /etc/snmp/snmpd.conf"""
      config = self.config_
      out = []
      engineId = self.engineIdGenerator_.engineId
      self.snmpHostnames_.clear()

      def _transports( vrfs, ipVersions, host='' ):
         if config.tcpTransport:
            baseTransports = [ 'udp', 'tcp' ]
         else:
            baseTransports = [ 'udp' ]
         t = []
         defaultVrf = Tac.newInstance( 'L3::VrfName', 'temp-status' ).defaultVrf
         if host:
            hostAdd = ':%s' % ( _transportAddressFormat( host ) )
         else:
            hostAdd = ''
         for vrf in vrfs:
            if vrf == defaultVrf:
               if 'ip4' in ipVersions:
                  t += [ i + hostAdd for i in baseTransports ]
               if 'ip6' in ipVersions:
                  t += [ i + '6' + hostAdd for i in baseTransports ]
            elif vrf in self.vrfStatusLocal_.vrf and \
                  self.vrfStatusLocal_.vrf[ vrf ].state == VrfState.active:
               netNs = self.vrfStatusLocal_.vrf[ vrf ].networkNamespace
               netNs = hexlifyNsName( netNs )
               if 'ip4' in ipVersions:
                  t += [ i + ':%s@@%s' % ( _transportAddressFormat( host ), netNs )
                         for i in baseTransports ]
               if 'ip6' in ipVersions:
                  t += [ i + '6:%s@@%s' % ( _transportAddressFormat( host ), netNs )
                         for i in baseTransports ]
         return t

      # Returns address in a form that can be used as part of a transport-address,
      # as described in the snmpcmd manpage. The input is assumed sound, so this
      # just adds brackets to the input if it is an IPv6 address, else it leaves
      # it alone.
      def _transportAddressFormat( address ):
         addr = str( address )
         # Hostnames
         if ':' in addr and addr[ 0 ] != '[':
            return '[%s]' % addr
         return addr

      def authFailureTrapEnabled( ):
         notifConfig = config.notificationConfig.get( 'snmp' )
         if not notifConfig:
            return Tac.Type( "Snmp::NotificationStatus" )( False, "" )
         authFailTrapStatus = notifConfig.notificationStatus( 'authentication' )
         return authFailTrapStatus.enabled

      if self.serviceEnabled():
         transports = _transports( sorted( config.vrf.keys() ), [ 'ip4', 'ip6' ] )
         if transports:
            out.append( "agentaddress %s" %
                        ','.join( [ t + ":161" for t in transports ] ) )
         else:
            # This means Snmp is only enabled in VRFs that do not exist. We can't
            # leave the agentaddress to be empty, because then snmpd would default
            # the listening address to udp:161, which is not what we want since
            # Snmp is disabled in the default VRF. We set agentaddress to
            # /etc/snmp/agentaddress so that snmpd does not listen to udp:161.
            out.append( "agentaddress /var/run/snmp-agent" )

         # Tell snmpd to save its persistent data (eg. engineBoots, oldEngineID)
         # to a location that won't get wiped out on reboot.
         out.append( "[snmp] persistentDir /persist/sys" )
         if engineId != "":
            out.append( "exactEngineID 0x%s" % ( engineId ) )
         if authFailureTrapEnabled( ):
            out.append( "authtrapenable 1" )
         else:
            out.append( "authtrapenable 2" )

         out.append( "dlmod StatsExporter StatsExporter" )

         # In the snmpd configuration below, we set the agentXTimeout
         # to 60s and agentXRetries to 0 to allow Snmp time to respond
         # and force snmpd not to send multiple requests.
         out.append( """
###############################################################################
# AgentX configuration
###############################################################################
""" )
         out.append( "master agentx" )
         out.append( "agentXSocket /var/run/agentx/master" )
         # AgentX socket permissions: socket_perms [directory_perms
         #                             [username|userid [groupname|groupid]]]
         out.append( "agentxperms 0777 0711" )
         out.append( "agentXTimeout 60" )
         out.append( "agentXRetries 0" )
         out.append( """
##############################################################################
# Access Control
##############################################################################
""" )
         if engineId != "":
            out.append( "# First, the internal user for the 'show snmp' command" )
            out.append( 'usmUser 1 2 0x%s 0x5f5f696e7465726e616c5f5f00 '
                  '0x5f5f696e7465726e616c5f5f00 NULL .1.3.6.1.6.3.10.1.1.3 0x%s '
                  '.1.3.6.1.6.3.10.1.2.1 "" ""' % ( engineId,
                     self.internalLocalizedKey_ ) )
            out.append( "# nsConfiguration" )
            out.append( "view _internal_ included .1.3.6.1.4.1.8072.1.7" )
            out.append( "group __internal__ usm __internal__" )
            out.append( 'access __internal__ "" usm authNoPriv '
                        'prefix _all_ _internal_ none' )
            out.append( "\n# Next, configured communities and users" )

         for viewName in sorted( config.view ):
            view = config.view[ viewName ]
            for root, subtree in sorted( view.subtree.items() ):
               out.append( "view %s %s %s" % ( view.name, subtree.viewType, root ) )

         # community -> securityname -> group, and
         # the group is what gets the access.  Instead of using the
         # "authcommunity" shortcut, which lets snmpd auto-create the
         # securityname and the group, we create them ourselves.
         # We create only one securityname + group for every community with
         # similar policy (e.g., community.access + community.view)
         secNameOutput = {}

         def secNameGen( access, view, context ):
            key = access + "/" + view + "/" + context
            if key not in self.secNameMap_:
               # create group and access mapping for this new secName
               secName = 'sec%d' % self.secNameIndex_
               self.secNameIndex_ += 1
               self.secNameMap_[ key ] = secName
            else:
               secName = self.secNameMap_[ key ]
            if secName in secNameOutput:
               return secName
            secNameOutput[ secName ] = True
            for v in ( '1', '2c' ):
               out.append( "group _grp%s v%s %s" % ( secName, v, secName ) )
            if not view:
               view = '_all_'
            if access == 'rw':
               writeView = view
            else:
               writeView = 'none'
            if not context:
               context = '""'
            out.append( 'access _grp%s %s any noauth exact %s %s none' % (
               secName, context, view, writeView ) )
            return secName
         for communityName in sorted( config.community ):
            community = config.community[ communityName ]
            secName = secNameGen( community.access, community.view,
                                  community.context )
            aclRules = []
            acl6Rules = []

            # IPv4 acls.
            if community.acl:
               out.append( "# (expansion of IPv4 acl %s for %s)" %
                           ( community.acl, community.name ) )
               acl = self.aclConfig_.config[ 'ip' ].acl.get( community.acl )
               if acl and acl.standard and acl.currCfg:
                  for seq in acl.currCfg.ruleBySequence.values():
                     rule = acl.currCfg.ipRuleById[ seq ]
                     if rule.action not in ( 'permit', 'deny' ):
                        continue
                     deny = '!' if rule.action == 'deny' else ''
                     aclRules.append( "%s%s" %
                                      ( deny, rule.filter.source.stringValue ) )
               elif acl and acl.currCfg:
                  out.append( "# omitted--it is not a standard ACL" )
               else:
                  out.append( "# omitted--it doesn't exist" )

            # IPv6 acls.
            if community.acl6:
               out.append( "# (expansion of IPv6 acl %s for %s)" %
                           ( community.acl6, community.name ) )
               acl6 = self.aclConfig_.config[ 'ipv6' ].acl.get( community.acl6 )
               if acl6 and acl6.standard and acl6.currCfg:
                  for seq in acl6.currCfg.ruleBySequence.values():
                     rule = acl6.currCfg.ip6RuleById[ seq ]
                     if rule.action not in ( 'permit', 'deny' ):
                        continue
                     deny = '!' if rule.action == 'deny' else ''
                     acl6Rules.append( "%s%s" %
                                       ( deny, rule.filter.source.stringValue ) )
               elif acl6 and acl6.currCfg:
                  out.append( "# (oops, it's not a standard IPv6 acl)" )
               else:
                  out.append( "# (oops, it doesn't exist)" )

            # Allow globally if there are no ACLs.
            if not community.acl and not community.acl6:
               aclRules = [ 'default' ]
               acl6Rules = [ 'default' ]

            contextStr = ""
            if community.context:
               contextStr = " -Cn %s" % community.context
            for rule in aclRules:
               out.append( "com2sec%s %s %s %s" %
                           ( contextStr, secName, rule, community.name ) )
            for rule in acl6Rules:
               out.append( "com2sec6%s %s %s %s" %
                           ( contextStr, secName, rule, community.name ) )
         securityModelMap = { 'v1': 'v1', 'v2c': 'v2c', 'v3': 'usm' }
         authLevelMap = { 'levelNoAuth': 'noauth', 'levelAuth': 'auth',
            'levelAuthAndPriv': 'priv' }

         def _hex( s ):
            return "".join( [ "%x" % ord( c ) for c in s ] )

         for key in sorted( config.user.keys() ):
            user = config.user[ key ]
            model = securityModelMap[ user.protocolVersion ]
            if model != 'usm':
               secname = '%s_%s_usr' % ( user.userName, model )
               out.append( 'com2sec %s default %s' % ( secname, user.userName ) )
               out.append( 'com2sec6 %s default %s' % ( secname, user.userName ) )
            else:
               # The commands to tell snmpd about the users are not the normal
               # way that someone would configure snmpd.  Normally you would
               # place a "createUser" command in snmpd.conf with the user's
               # auth type, privacy type, and passphras(es), and snmpd would
               # read that entry, remove it from snmpd.conf, and generate a
               # usmUser entry in its persistent config file with localized
               # keys, etc.  Since RFC 2574 requires that we not store the
               # passphrases, and we want the user config to all be in Sysdb,
               # the code here needs to output usmUser instead of createUser.
               secname = user.userName
               # In the line below, "1" means "active" and "2" means to store
               # the user as non-volatile, meaning that snmpd will not attempt
               # to copy this user to its persistent config file.  For more
               # about the meaning of these values, see the userStatus and
               # userStorageType values in NET-SNMP's usmUser struct.
               authOid = authTypeOidMap[ user.authType ]
               privOid = privTypeOidMap[ user.privacyType ]
               authKey = '""'
               privKey = '""'
               if len( user.authLocalizedKey ) > 0:
                  authKey = "0x" + user.authLocalizedKey
               if len( user.privacyLocalizedKey ) > 0:
                  privKey = "0x" + user.privacyLocalizedKey
               userCmd = "usmUser 1 2 0x%s 0x%s00 0x%s00 NULL %s %s %s %s \"\"" % (
                   user.engineId, _hex( user.userName ),
                   _hex( secname ), authOid, authKey, privOid, privKey )
               out.append( userCmd )
               groupCmd = 'group %s %s %s' % ( user.group, model, secname )
               out.append( groupCmd )

         for key in sorted( config.group.keys() ):
            group = config.group[ key ]
            model = securityModelMap[ group.protocolVersion ]
            auth = authLevelMap[ group.authLevel ]
            if group.readView != "":
               read = group.readView
            else:
               read = '_all_'
            if group.writeView != "":
               write = group.writeView
            else:
               write = 'none'
            if group.notifyView != "":
               notify = group.notifyView
            else:
               notify = 'none'
            cmd = 'access %s "%s" %s %s prefix %s %s %s' % ( group.groupName,
               group.context, model, auth, read, write, notify )
            out.append( cmd )

         out.append( """
##############################################################################
# Traps / Informs
##############################################################################
""" )

         out.append( "[snmp] resendFailedNotifications yes" )
         out.append( "[snmp] notificationTimeout %s" % notificationTimeout )
         out.append( "[snmp] notificationRetries %s" % notificationRetries )

         # SuperServer does not restart snmpd, but only forces the reload of its
         # config file.  While doing that, there is a bug in net-snmp which
         # prevents it from noticing that there is no clientaddr any more. As a
         # result,  notifications will continue to be sent using the last configured
         # clientaddr ( if any ).  As this is not the intended behavior, clientAddr
         # must be set to '0.0.0.0' in order to ensure that snmpd returns to
         # the the default behaviour with respect to choosing the source
         # interface for notifications.

         v1trapaddress = '0.0.0.0'
         srcIntfs = config.notificationSourceIntf
         if len( srcIntfs ) == 1 and DEFAULT_VRF in srcIntfs:
            ipIntfStatus = self.ipStatus_.ipIntfStatus.get( srcIntfs[ DEFAULT_VRF ] )
            ip6IntfStatus = self.ip6Status_.intf.get( srcIntfs[ DEFAULT_VRF ] )
            if ipIntfStatus and ipIntfStatus.vrf == DEFAULT_VRF and \
               not ip6IntfStatus:
               v1trapaddress = ipIntfStatus.activeAddrWithMask.address

         out.append( "[snmp] clientaddr 0.0.0.0" )
         out.append( "v1trapaddress %s" % v1trapaddress )
         defaultVrf = Tac.newInstance( 'L3::VrfName', 'temp-status' ).defaultVrf

         # pylint: disable-next=too-many-nested-blocks
         for key in sorted( config.notificationSink.keys() ):
            sink = config.notificationSink[ key ]

            # Determine the allowable transports.
            try:
               addrInfo = socket.getaddrinfo( sink.hostname, sink.port, 0, 0,
                                              socket.SOL_UDP )
            except socket.gaierror as e:
               out.append( "# Error getting address info for %s, skipping: %s" %
                           ( sink.hostname, e ) )
               # Let the Aresolve eventually resolve this hostname
               hostname, _ = self.dns_.handleIpAddrOrHostname( sink.hostname )
               if hostname:
                  self.snmpHostnames_.append( hostname )
               continue
            ipvers = []
            for ( fam, _, _, _, _ ) in addrInfo:
               if fam == socket.AF_INET and 'ip4' not in ipvers:
                  ipvers.append( 'ip4' )
               elif fam == socket.AF_INET6 and 'ip6' not in ipvers:
                  ipvers.append( 'ip6' )
            if not ipvers:
               out.append( "# Error getting address info for %s, assuming IPv4." %
                           sink.hostname )
               ipvers = [ 'ip4' ]

            sinkAddr = sink.hostname
            if sink.refresh:
               sinkAddr = self.getIpAddr( sink.hostname )
               if not sinkAddr:
                  # We don't have IP address yet, so don't put anything in the
                  # config file to avoid needless reloads.
                  continue
            vrf = sink.vrf if sink.vrf else defaultVrf

            # BUG777712 - If the trap host was a hostname that resolves to both v4
            # and v6, then we will give _transports ipvers=[v4,v6] and therefore get
            # back a list that includes udp6 transport, but we are always picking the
            # first one, which will be the v4 one
            transports = _transports( [ vrf ], ipvers, sinkAddr )
            if not transports:
               # VRF is not created yet, so don't configure the trap recipient yet.
               continue
            transport = transports[ 0 ]

            srcAddr = '0.0.0.0' if 'ip4' in ipvers else '[0::0]'
            if vrf in config.notificationSourceIntf and vrf in config.vrf:
               intfId = srcIntfs[ vrf ]
               if 'ip4' in ipvers:
                  ipIntfStatus = self.ipStatus_.ipIntfStatus.get( intfId )
                  if ipIntfStatus and ipIntfStatus.vrf == vrf and \
                      ipIntfStatus.activeAddrWithMask.address != '0.0.0.0':
                     srcAddr = ipIntfStatus.activeAddrWithMask.address
                  elif ipIntfStatus and ipIntfStatus.vrf != vrf:
                     out.append( "# Error: %s is in VRF %s" %
                                 ( intfId, ipIntfStatus.vrf ) )

               else:
                  ip6IntfStatus = self.ip6Status_.intf.get( intfId )
                  if ip6IntfStatus and ip6IntfStatus.vrf == vrf:
                     for ip6AddrWithMask in sorted( ip6IntfStatus.addr ):
                        if ip6AddrWithMask.address.isLinkLocal:
                           continue
                        if ip6AddrWithMask.address != "::":
                           srcAddr = '[' + str( ip6AddrWithMask.address ) + ']'
                           break
                  elif ip6IntfStatus and ip6IntfStatus.vrf != vrf:
                     out.append( "# Error: %s is in VRF %s" %
                                 ( intfId, ip6IntfStatus.vrf ) )
            trapsessSuffix = "-s %s %s:%d" % ( srcAddr, transport, sink.port )
            if sink.protocolVersion == 'v1':
               cmd = "trapsess -v 1 -c %s %s" % ( sink.securityName,
                     trapsessSuffix )
            elif sink.protocolVersion == 'v2c':
               if sink.notificationType == 'inform':
                  cmd = "trapsess -v 2c -c %s -Ci %s" % ( sink.securityName,
                        trapsessSuffix )
               else:
                  cmd = "trapsess -v 2c -c %s -e %s %s" % ( sink.securityName,
                        engineId, trapsessSuffix )
            else:
               assert sink.protocolVersion == 'v3'
               # sink.securityName specifies a user.  For traps, the user is
               # local.  For informs, the user is remote.  I will only write
               # a trapsess directive if I can find the user.
               args = [ 'trapsess -v 3 -u %s' % ( sink.securityName ) ]
               authLevelTrans = { "levelNoAuth": "noAuthNoPriv",
                  "levelAuth": "authNoPriv", "levelAuthAndPriv": "authPriv" }
               args.append( "-l %s" % ( authLevelTrans[ sink.authLevel ] ) )
               if sink.notificationType == 'inform':
                  args.append( '-Ci' )
                  userspec = Tac.Value( "Snmp::RemoteUserSpec",
                                        hostname=sinkAddr,
                                        port=sink.port,
                                        userName=sink.securityName )
                  user = config.remoteUser.get( userspec )
                  if user:
                     args.append( '-e %s' % ( user.engineId ) )
               else:
                  assert sink.notificationType == 'trap'
                  args.append( '-e %s' % ( engineId ) )
                  userspec = Tac.Value( "Snmp::UserSpec",
                                        name=sink.securityName,
                                        protocolVersion='v3' )
                  user = config.user.get( userspec )
               if user:
                  # The translation of authType or privacyType to the
                  # appropriate net-snmp command-line value is just a matter of
                  # removing 'auth'/'privacy' and converting to uppercase
                  if user.authType != 'authNone':
                     authVal = user.authType.replace( 'auth', '' ).upper()
                     args.append( "-a %s" % authVal )
                     args.append( "-3k %s" % ( user.authLocalizedKey ) )
                  if user.privacyType != 'privacyNone':
                     privVal = user.privacyType.replace( 'privacy', '' ).upper()
                     args.append( "-x %s" % privVal )
                     args.append( "-3K %s" % ( user.privacyLocalizedKey ) )
                  args.append( trapsessSuffix )
                  cmd = " ".join( args )
               elif sink.authLevel == "levelNoAuth":
                  out.append( "# Unknown user %s@%s:%d, but no auth, so "
                        "configuring notification host anyway." %
                        ( sink.securityName, transport, sink.port ) )
                  args.append( trapsessSuffix )
                  cmd = " ".join( args )
               else:
                  # Might want to log this?  Or maybe print warning in the
                  # CliPlugin when someone configures a trap sink with a user
                  # that hasn't been configured?
                  cmd = "# Skipping notification sink %s:%d due to missing " \
                        "user %s" % ( transport, sink.port, sink.securityName )

            out.append( cmd )

         requestedKeys = set()
         srcIntfs = config.notificationSourceIntf
         if len( srcIntfs ):
            for vrf, intfId in srcIntfs.items():
               requestedKeys.add( intfId )
         self.ipStatusReactor_.requestedKeysIs( requestedKeys )
         self.ip6StatusReactor_.requestedKeysIs( requestedKeys )

         out.append( "nlmConfigGlobalEntryLimit %s" %
                     config.notificationLogEntryLimit )

         out.append( """
##############################################################################
# System Information
##############################################################################
""" )
         sysDescr = self.sysDescr_.value
         if sysDescr != "":
            out.append( "sysdescr %s" % sysDescr )

         # config.sysServices is always set
         out.append( "sysservices %s" % ( config.sysServices ) )

         sysObjectId = self.sysObjectId()
         if sysObjectId != "":
            out.append( "sysobjectid %s" % sysObjectId )

         # push the system start time information for snmpd to use
         # for sysUpTime etc.
         out.append( "sysstarttime %s" % ( self._monotonicTimeFromSnmpStatus() ) )

         if config.extension:
            out.append( """
##############################################################################
# Extensions
##############################################################################
""" )
            # Sorting OIDs as strings isn't true lexicographic order, but at least
            # it's consistent.
            for xtension in sorted( config.extension.values(), key=lambda xtension:
                                                                     xtension.oid ):
               # TODO: do we need to expose priority?
               out.append( "pass%s %s %s" % ( "" if xtension.oneShot else "_persist",
                  xtension.oid, xtension.handler ) )

         # Enable timestamps in /var/log/snmpd
         out.append( "[snmp] logTimestamp 1" )

         # BUG420560 workaround
         overrideSendMessageMaxSize = int( os.environ.get(
            "OVERRIDE_UPDATE_MESSAGE_SIZE", 0 ) )
         sendMessageMaxSize = overrideSendMessageMaxSize or config.transmitMsgSize
         out.append( "[snmp] sendMessageMaxSize %u" % sendMessageMaxSize )

      else:
         out.append( "# snmpd is not enabled" )

      # Cleanup hostnames that are not in use
      names = set( self.dns_.dnsRecords.keys() ) - set( self.snmpHostnames_ )
      for name in names:
         self.dns_.finishHost( name )
         self.dns_.removeRecord( name )
         self.dns_.usedIPs.pop( name, None )

      return '\n'.join( out )

   def conf( self ):
      # filewrite maps filenames to their contents for writing. We call two
      # functions to find the contents of these conf files. This depends on
      # the order of the files as declared in the constructor.
      fileWrite = {}
      fileWrite[ '/etc/snmp/snmpd.conf' ] = self._generateSnmpConf()
      fileWrite[ '/etc/sysconfig/snmpd' ] = self._generateSysconfigConf()
      return fileWrite

   def _getDebugTokensFromConfig( self ):
      m = self.tokensConfig_.messageType
      debugTokens = [ name for name in m if m[ name ].enabled ]
      if 'agentx' in debugTokens:
         # BUG30145 workaround
         index = debugTokens.index( 'agentx' )
         debugTokens[ index ] = 'agentx/master'
      return debugTokens

   def _generateSysconfigConf( self ):
      """Generate new /etc/sysconfig/snmpd"""
      # Note that we want to set the max log level based on debug requests.
      # We are not logging to a separate file but relying on snmpd logging
      # debug messages to /var/log/messages. This allows us to enable debug
      # even in customer environments and just get the /var/log/messages content.
      # Don't forget to enclose the options string in double quotes.

      debugTokens = self._getDebugTokensFromConfig() or []
      debugOpt = ''
      logLevel = 6  # LOG_INFO (to debug BUG19107)

      if len( debugTokens ) > 0:
         debugOpt = ' -D' + ','.join( debugTokens )  # no space after -D

      logOpt = '-LS0-%dd -Lf /var/log/snmpd' % logLevel
      options = '''OPTIONS="%s -p /run/snmpd.pid %s"\n''' % ( logOpt, debugOpt )
      options += '''DAEMON_COREFILE_LIMIT=200000\n'''

      return options

   def started( self ):
      try:
         Tac.run( [ "service", self.linuxServiceName_, "status" ],
                  stdout=Tac.DISCARD, timeout=SuperServer.defaultTimeout )
         return True
      except Tac.Timeout:
         pass
      except Tac.SystemCommandError:
         pass

      return False

   def startService( self ):
      # Don't start snmpd if it has already started
      if not self.started() or self.forceRestart_:
         SuperServer.LinuxService.startService( self )

   def stopService( self ):
      SuperServer.LinuxService.stopService( self )

   def restartService( self ):
      # The base implementation of restartService does a restart, but in the
      # case of snmpd we just want it to reload its config file to avoid sending
      # a coldStart trap, etc, so we just do a reload, which is defined in
      # /etc/init.d/snmpd to send a HUP signal to snmpd.  We have to be careful
      # to only do this if snmpd is already running, however.
      if not self.started() or self.forceRestart_:
         qt0( "restartService: service needs to be restarted" )
         SuperServer.LinuxService.restartService( self )
         self.forceRestart_ = False
         return

      qt0( "restartService: sending reload" )
      self._runServiceCmd( "reload" )
      self.restarts += 1

   def serviceEnabled( self ):
      engineId = self.engineIdGenerator_.engineId
      if len( engineId ) % 2 == 1:
         # If the configured engineId is odd in length, considering
         # the service to be disabled to prevent successive
         # snmpd restarts.
         return False
      # snmpd is enabled when Snmp is enabled and vice versa
      return self.notifier_.serviceEnabled

   @Tac.handler( 'serviceEnabled' )
   def handleServiceEnabled( self ):
      self._scheduleCounterActivity()
      self._scheduleOomScoreAdjUpdate()

   @Tac.handler( 'vrf' )
   def handleVrfConfig( self, vrf ):
      if vrf:
         self.handleConfigChange( 'SNMP %s in VRF %s' %
               ( 'enabled' if vrf in self.config_.vrf else 'disabled', vrf ) )
      else:
         # When there are multiple deferred notifications, they are coalesced into
         # a single notification call and vrf is passed in as None. Vrf is also
         # passed in as None upon init.
         self.handleConfigChange( 'SNMP init or enabled/disabled in multiple VRFs' )

   @Tac.handler( 'tcpTransport' )
   def handleTcpTransportConfig( self ):
      self.forceRestart_ = True
      self.handleConfigChange( 'tcpTransport' )

   def handleConfigChange( self, what ):
      t0( "SnmpdService.handleConfigChange: what=", what )
      qt0( "SnmpdService.handleConfigChange: what=", qv( what ) )
      if what == 'debugConfig':
         # snmpd needs to be restarted if debug options are changed.
         # Changing the options in config file doesn't help. So we
         # restart snmpd with new options that reflect user debug config.
         # XXX TODO: We are investigating using an internal MIB for
         # this purpose, so the restart isn't required.
         self.forceRestart_ = True
      self.sync()

   def _scheduleCounterActivity( self ):
      if self.serviceEnabled():
         delta = self.notifier_.updateCountersFreq
         # pylint: disable-next=consider-using-max-builtin
         if delta < _minUpdateCountersFreq:
            delta = _minUpdateCountersFreq
         t5( "_scheduleCounterActivity: scheduling activity", delta,
             "seconds in future" )
         self.counterActivity_.timeMin = Tac.now() + delta
      else:
         self.counterActivity_.timeMin = Tac.endOfTime
         t5( "_scheduleCounterActivity: service disabled, suspending activity" )

   def _updateCounters( self ):
      if 'DEBUG_SNMP_COUNTERS' in os.environ:
         # For testing purposes
         path = os.environ[ 'DEBUG_SNMP_COUNTERS' ]
      else:
         path = "/var/run/snmpd-counters"

      counters = {}
      try:
         f = open( path ) # pylint: disable=consider-using-with
         while True:
            line = f.readline()
            m = re.match( "(\\w+)=(\\d+)\n", line )
            if m:
               counters[ m.group( 1 ) ] = int( m.group( 2 ) )
            else:
               break
         f.close()
      except OSError:
         qt0( "SnmpdService._updateCounters - error reading", path )

      for c, v in counters.items():
         if c in _counterMap:
            setattr( self.counters_, _counterMap[ c ], v )

      self._scheduleCounterActivity()

   def _scheduleOomScoreAdjUpdate( self ):
      if self.serviceEnabled():
         newTime = Tac.now() + self.oomScoreAdjUpdateInterval_
         t5( "Scheduling oomScoreAdjUpdateActivity at time", Tac.now(), "for",
              self.oomScoreAdjUpdateInterval_, "s later at", newTime )
         self.oomScoreAdjUpdateActivity_.timeMin = newTime
      else:
         t5( "Disabling oomScoreAdjUpdateActivity" )
         self.oomScoreAdjUpdateActivity_.timeMin = Tac.endOfTime

   def _oomScoreAdjUpdate( self ):
      if not self.serviceEnabled():
         t5( "Service not enabled, skipping" )
         return

      def getPidOf( process ):
         pid = None
         try:
            pid = Tac.run( [ 'pidof', process ],
                           stdout=Tac.CAPTURE, asRoot=True ).strip()
            if ' ' in pid:
               t0( "multiple", process, "processes running" )
               pid = None
         except Tac.SystemCommandError:
            t0( "error getting pid of", process )
         return pid

      def getOomScoreAdj( pid ):
         scoreAdj = None
         try:
            cmdList = [ 'cat', f'/proc/{pid}/oom_score_adj' ]
            t0( "cmdList:", cmdList )
            scoreAdj = Tac.run( cmdList, stdout=Tac.CAPTURE,
                                stderr=Tac.CAPTURE, asRoot=True ).strip()
         except Tac.SystemCommandError:
            t0( "No pid or oom_score_adj file" )

         return scoreAdj
      
      def setOomScoreAdj( pid, score ):
         cmdList = [ 'oomadj', '--pid', pid, '--oom_score_adj', score ]
         try:
            Tac.run( cmdList, stderr=Tac.CAPTURE, asRoot=True )
         except Tac.SystemCommandError:
            t0( "oomadj failed" )

      snmpdPid = getPidOf( "snmpd" )
      snmpScoreAdj = getOomScoreAdj( getPidOf( "Snmp" ) )
      snmpdScoreAdj = getOomScoreAdj( snmpdPid )
      if snmpdScoreAdj is None or snmpScoreAdj is None:
         t0( "Could not get oomScoreAdj values" )
      elif snmpdScoreAdj == snmpScoreAdj:
         t5( "snmpd oomScoreAdj", snmpdScoreAdj, "is same as Snmp's, nothing to do" )
      else:
         t5( "snmpd oomScoreAdj", snmpdScoreAdj, "is not the same as Snmp's",
             snmpScoreAdj, ", updating..." )
         setOomScoreAdj( snmpdPid, snmpScoreAdj )

      self._scheduleOomScoreAdjUpdate()

class Snmpd( SuperServer.SuperServerAgent ):

   def __init__( self, entityManager ):
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      self.service_ = None

      self.snmpConfig = mg.mount( 'snmp/config', 'Snmp::Config', 'r' )
      self.snmpStatus = mg.mount( 'snmp/status', 'Snmp::Status', 'r' )
      self.snmpCounters = mg.mount( 'snmp/counters', 'Snmp::Counters', 'w' )
      self.entityMibStatus = mg.mount( 'hardware/entmib', 'EntityMib::Status', 'r' )
      self.netConfig = mg.mount( 'sys/net/config', 'System::NetConfig', 'r' )
      Tac.Type( "Ira::IraIpStatusMounter" ).doMountEntities( mg.cMg_, True, True )
      self.ipStatus = mg.mount( 'ip/status', 'Ip::Status', 'r' )
      self.ip6Status = mg.mount( 'ip6/status', 'Ip6::Status', 'r' )
      self.allVrfStatusLocal = mg.mount( Cell.path( 'ip/vrf/status/local' ),
                                         'Ip::AllVrfStatusLocal', 'r' )
      self.debugConfig = mg.mount( 'debug/config', 'Debug::Config' )
      self.aclConfig = Tac.newInstance( "Acl::Config" )
      self.aclConfig.newConfig( 'ip' )
      self.aclConfig.newConfig( 'ipv6' )
      self.aclConfigAggregatorSm = None
      self.aclConfigDir = mg.mount( 'acl/config/input', 'Tac::Dir', 'ri' )
      self.cliAclConfig = mg.mount( 'acl/config/cli', 'Acl::Input::Config', 'r' )

      def _finished():
         # run only if active
         if self.active():
            self.onSwitchover( None )
      mg.close( _finished )

   def warm( self ):
      if not self.active():
         return True
      return self.service_ and self.service_.warm()

   def onSwitchover( self, protocol ):
      qt0( 'Became active. Running SnmpdService' )
      self.aclConfigAggregatorSm = Tac.newInstance( "Acl::AclConfigAggregatorSm",
                                                    self.aclConfigDir,
                                                    self.cliAclConfig,
                                                    self.aclConfig )
      self.service_ = SnmpdService( self.snmpConfig,
                                    self.netConfig, self.snmpCounters,
                                    self.ipStatus, self.ip6Status,
                                    self.allVrfStatusLocal, self.debugConfig,
                                    self.aclConfig, self.snmpStatus,
                                    self.entityMibStatus )

def Plugin( ctx ):
   ctx.registerService( Snmpd( ctx.entityManager ) )
