#!/usr/bin/env python3
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=unidiomatic-typecheck
# pylint: disable=consider-using-f-string

'''
The SshMgr is responsible for switching between password mode and
keyboard-interactive mode.
'''

import Tac, SuperServer, QuickTrace
import os, re, weakref, errno
import Cell, ManagedSubprocess, Logging
from IpLibConsts import DEFAULT_VRF
import MgmtSecuritySslStatusSm
import SysMgrLib
from SysMgrLib import netnsNameWithUniqueId, opensshMethodMap
import PyWrappers.OpensshServer as sshd
import PyWrappers.OpensshClients as ssh
from XinetdLib import XinetdService
import SshCertLib
from Toggles import MgmtSecurityToggleLib
import Tracing
from TypeFuture import TacLazyType
import shutil
import signal
import subprocess
import sys
import string
import random
import pyinotify
import filecmp
from Url import syncfile
from SshAlgorithms import SUPPORTED_CIPHERS, FIPS_CIPHERS, \
   SUPPORTED_MACS, FIPS_MACS, SUPPORTED_KEXS, FIPS_KEXS, \
   SUPPORTED_HOSTKEYS, FIPS_HOSTKEYS, HOSTKEYS_MAPPINGS

qv = QuickTrace.Var
qt0 = QuickTrace.trace0

t0 = Tracing.trace0
t5 = Tracing.trace5
t8 = Tracing.trace8

SshConstants = Tac.Type( "Mgmt::Ssh::Constants" )
AuthenMethod = Tac.Type( "Mgmt::Ssh::AuthenMethod" )
SshCertFileType = Tac.Type( "Mgmt::Ssh::SshCertFileType" )
KeyType = Tac.Type( "Mgmt::Ssh::KeyType" )
OcspCertRequirement = Tac.Type( "Mgmt::Security::Ssl::OcspCertRequirement" )
SslFeature = Tac.Type( "Mgmt::Security::Ssl::SslFeature" )
SslProfileState = TacLazyType( "Mgmt::Security::Ssl::ProfileState" )

SECURITY_SSH_TUNNEL_ESTABLISHED = Logging.LogHandle(
              "SECURITY_SSH_TUNNEL_ESTABLISHED",
              severity=Logging.logInfo,
              fmt="SSH tunnel %s from local TCP port %s to "
              "%s:%s via %s@%s is established.",
              explanation="An SSH tunnel was just started to a remote server.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

SECURITY_SSH_TUNNEL_ALGORITHM_MISMATCH = Logging.LogHandle(
              "SECURITY_SSH_TUNNEL_ALGORITHM_MISMATCH",
              severity=Logging.logError,
              fmt="SSH tunnel %s was unable to connect to the configured "
              "remote server due to not finding a matching %s.",
              explanation="The remote server did not support an algorithm that the"
              " switch could use.",
              recommendedAction="Check if the server and switch are configured "
              "correctly." )

SECURITY_SSH_TUNNEL_CORRUPT_PACKET_RECEIVED = Logging.LogHandle(
              "SECURITY_SSH_TUNNEL_CORRUPT_PACKET_RECEIVED",
              severity=Logging.logError,
              fmt="SSH tunnel %s received a corrupt packet from the configured "
              "remote server.",
              explanation="A packet from the remote server was received that passed "
              "the TCP/IP checksum but failed the SSH checks.",
              recommendedAction=Logging.CALL_SUPPORT_IF_PERSISTS )

SECURITY_SSH_TUNNEL_HOSTNAME = Logging.LogHandle(
              "SECURITY_SSH_TUNNEL_HOSTNAME",
              severity=Logging.logError,
              fmt="SSH tunnel %s was unable to resolve hostname: %s",
              explanation="The SSH tunnel was unable to resolve the hostname "
              "and could not connect.",
              recommendedAction="Check the hostname and DNS resolution." )

SECURITY_SSH_TUNNEL_SWITCH_HOSTKEY_DENIED = Logging.LogHandle(
              "SECURITY_SSH_TUNNEL_SWITCH_HOSTKEY_DENIED",
              severity=Logging.logError,
              fmt="SSH tunnel %s was unable to log into its configured host"
              " via public-key authentication",
              explanation="The SSH tunnel attempted to log into the configured host"
              " with the switch's hostkey and was denied.",
              recommendedAction="Check that the switches public hostkey is installed"
              " on the remote host for the username that is being used to log in." )

SECURITY_SSH_TUNNEL_INITIAL_TIMEOUT = Logging.LogHandle(
              "SECURITY_SSH_TUNNEL_INITIAL_TIMEOUT",
              severity=Logging.logError,
              fmt="SSH tunnel %s was unable to reach the remote host "
              " and the connection timed out.",
              explanation="SSH Tunnels have a limited amount of time to "
              "reach the host to make the connection. The tunnel "
              "was unable to find a route to the configured host in that timeframe.",
              recommendedAction="Check that the address for the configured host is"
              " correct and that a route exists from the switch to the host." )

SECURITY_SSH_TUNNEL_CONNECTION_REFUSED = Logging.LogHandle(
              "SECURITY_SSH_TUNNEL_CONNECTION_REFUSED",
              severity=Logging.logError,
              fmt="SSH tunnel %s had its initial connection refused "
              "by the remote host.",
              explanation="The initial attempt to open an ssh connection was "
              "refused by the remote host.",
              recommendedAction="Check that the remote host has its ssh server"
              " running and accepting connections on port 22." )

SECURITY_SSH_TUNNEL_TIMEOUT = Logging.LogHandle(
              "SECURITY_SSH_TUNNEL_TIMEOUT",
              severity=Logging.logError,
              fmt="SSH tunnel %s had the remote host timeout while connected.",
              explanation="The SSH tunnel periodically sends server alive messages "
              " through the tunnel and timed out on receiving a reply for them.",
              recommendedAction="Check that the remote host has it's SSH "
              "server running  and if it has any error messages in its log." )

SECURITY_SSH_TUNNEL_CLOSED_REMOTELY = Logging.LogHandle(
              "SECURITY_SSH_TUNNEL_CLOSED_REMOTELY",
              severity=Logging.logError,
              fmt="SSH tunnel %s had it's connection closed by the remote host.",
              explanation="The SSH Tunnels connection was closed by"
              " the remote host.",
              recommendedAction="Check the logs for the remote hosts SSH server." )

SECURITY_SSH_TUNNEL_HOSTKEY_VERIFY_FAILED = Logging.LogHandle(
              "SECURITY_SSH_TUNNEL_HOSTKEY_VERIFY_FAILED",
              severity=Logging.logError,
              fmt="SSH tunnel %s was unable to connect to the configured host "
              " because it could not verify the hostkey of the remote host.",
              explanation="The switch is set to check all hostkeys of remote "
              "entities it connects to. Verification of the remote host failed "
              "and the switch did not connect.",
              recommendedAction="Make sure the known hosts for the switch contains"
              " the hostkey the remote SSH server is offering upon connection." )

SECURITY_SSH_TUNNEL_REMOTE_PORT_ERROR = Logging.LogHandle(
              "SECURITY_SSH_TUNNEL_REMOTE_PORT_ERROR",
              severity=Logging.logError,
              fmt="SSH Tunnel %s is unable to open the port on the remote host.",
              explanation="When the SSH Tunnel attempted to open a connection on"
              " the remote side it was prevented from doing so by the remote "
              "hosts SSH server.",
              recommendedAction="Make sure the remote server allows TCP "
              "forwarding, allows the configured port to be opened, and "
              "the port is not the point of a resource allocation conflict." )

SECURITY_SSH_TUNNEL_LOCAL_PORT_ERROR = Logging.LogHandle(
              "SECURITY_SSH_TUNNEL_LOCAL_PORT_ERROR",
              severity=Logging.logError,
              fmt="SSH Tunnel %s was unable to use the configured local port.",
              explanation="The switch was unable to use the local port for an "
              " SSH Tunneling connection.",
              recommendedAction="Use a different local port that is not being "
              "used by another program." )

SECURITY_SSH_TUNNEL_LOG_FILE_TOO_LARGE = Logging.LogHandle(
              "SECURITY_SSH_TUNNEL_LOG_FILE_TOO_LARGE",
              severity=Logging.logWarning,
              fmt="SSH Tunnel %s had its log file grow too large. The "
              "tunnel will be restarted since this is indicative of an issue.",
              explanation="The log file of an SSH Tunnel is generally a fixed size"
              " unless there is an issue. Since the log file got too large a copy "
              "will be saved to the disk and the tunnel restarted.",
              recommendedAction="Check syslog and the tunnels log file at /etc/ssh/ "
              "to determine if there are any issues." )

SECURITY_SSH_TUNNEL_SLOW_RESTART = Logging.LogHandle(
              "SECURITY_SSH_TUNNEL_SLOW_RESTART",
              severity=Logging.logWarning,
              fmt="SSH Tunnel %s will be delayed before restarting",
              explanation="SSH Tunnels can only restart a limited number of times"
              " before they are delayed. This is to prevent bad connections from"
              " attempting to connect over and over. Once the tunnel successfully"
              " runs for some time, fast restarts will work again.",
              recommendedAction="Check for previous syslog messages regarding SSH"
              " tunnels. If this doesn't explain why the SSH Tunnel is restarting"
              " so often check the tunnel logs under /etc/ssh." )

SECURITY_SSH_CERT_FILE_WARNING = Logging.LogHandle(
              "SECURITY_SSH_CERT_FILE_WARNING",
              severity=Logging.logWarning,
              fmt="Issue found with one or more SSH files: %s",
              explanation="The SSH files may contain invalid keys, or conflicting"
              " host certificates with the same key type may have been configured."
              " The files may also not exist. Until the files are fixed,"
              " passwordless authentication or host key checking may not work as"
              " expected.",
              recommendedAction="Check the files to make sure the contents are"
              " correct." )

def deriveRekeyStr( config ):
   """
   Determine what to set RekeyLimit to.
   Also check if rekey limit is above what openssh will accept.
   Will assert if invalid value.
   """
   assert SysMgrLib.checkRekeyDataLimit( config.rekeyDataAmount,
                                         config.rekeyDataUnit ), \
          "Data Rekey was set to excessive value"
   rekeyStr = str( config.rekeyDataAmount )
   if config.rekeyDataUnit == "kbytes":
      rekeyStr += "K"
   elif config.rekeyDataUnit == "mbytes":
      rekeyStr += "M"
   elif config.rekeyDataUnit == "gbytes":
      rekeyStr += "G"
   else:
      assert False, "Unknown data unit passed in"
   if config.rekeyTimeLimit == 0:
      rekeyStr += " none"
   else:
      rekeyStr += " %d" % config.rekeyTimeLimit
      if config.rekeyTimeUnit == "seconds":
         rekeyStr += "s"
      elif config.rekeyTimeUnit == "minutes":
         rekeyStr += "m"
      elif config.rekeyTimeUnit == "hours":
         rekeyStr += "h"
      elif config.rekeyTimeUnit == "days":
         rekeyStr += "d"
      elif config.rekeyTimeUnit == "weeks":
         rekeyStr += "w"
      else:
         assert False, "Unknown time unit passed in"

   return rekeyStr

def filterSshOptions( config ):
   """
   Reads the config file to determine the cipher/mac/hostkey options
   and filter out the non-FIPS compliant options based on that setting.
   Returns a triple list of the legal options
   """
   ciphers = config.cipher.split()
   macs = config.mac.split()
   hostkeys = config.hostkey.split()
   kexs = config.kex.split()

   filteredCiphers = [ cipher for cipher in ciphers
                       if cipher in SUPPORTED_CIPHERS ]
   filteredMacs = [ mac for mac in macs if mac in SUPPORTED_MACS ]
   filteredHostKeys = [ hostkey for hostkey in hostkeys
                        if hostkey in SUPPORTED_HOSTKEYS ]
   filteredKexs = [ kex for kex in kexs if kex in SUPPORTED_KEXS ]
   if config.fipsRestrictions:
      ciphers = filteredCiphers[:]
      macs = filteredMacs[:]
      unnamedHostkeys = filteredHostKeys[:]
      kexs = filteredKexs[:]
      filteredCiphers = [ cipher for cipher in ciphers if cipher in FIPS_CIPHERS ]
      filteredMacs = [ mac for mac in macs if mac in FIPS_MACS ]
      filteredKexs = [ kex for kex in kexs if kex in FIPS_KEXS ]
      filteredHostKeys = [ hostkey for hostkey in unnamedHostkeys
                          if hostkey in FIPS_HOSTKEYS ]
      allowedHostKeys = FIPS_HOSTKEYS
      # pylint: disable-next=use-implicit-booleaness-not-len
      if not len( filteredCiphers ):
         filteredCiphers = FIPS_CIPHERS
      if not len( filteredMacs ): # pylint: disable=use-implicit-booleaness-not-len
         filteredMacs = FIPS_MACS
      if not len( filteredKexs ): # pylint: disable=use-implicit-booleaness-not-len
         filteredKexs = FIPS_KEXS
   else:
      allowedHostKeys = SUPPORTED_HOSTKEYS
   # Extend filteredHostKeys list with named keys,
   # the key path will be of the form
   # /persist/secure/ssh/keys/<algo>/<file-name>.
   filteredHostKeys.extend( [ hostkey for hostkey in hostkeys if
         SshCertLib.getAlgoFromKeyPath( hostkey ) in allowedHostKeys ] )
   # Note, we always return a non-empty list so that sshd would not
   # try to look for keys under /etc/ssh.
   if not len( filteredHostKeys ): # pylint: disable=use-implicit-booleaness-not-len
      filteredHostKeys = [ allowedHostKeys[ 0 ] ]

   return ( filteredCiphers, filteredMacs, filteredKexs, filteredHostKeys )

def copyNamedSshKeys( srcKey, destKey ):
   """
      Copy the keys to SSH key config directory.
         1. Copy private key from source to dest directory.
         2. Extract public key from private key and copy to the dest directory.
   """
   pubKeyPath = destKey + ".pub"
   t8( "Generating public key to", pubKeyPath )

   # Copy private key
   tempKeyName = os.path.join( SshConstants.sshKeyConfigDir,
         getTempFileName( "ssh-key" ) )
   shutil.copy( srcKey, tempKeyName )
   # Sync the source file
   syncfile( tempKeyName )
   # The move operation is atomic
   shutil.move( tempKeyName, destKey )

   # Extract public key from private key
   try:
      pubKeyData = Tac.run( [ "ssh-keygen", "-yf", srcKey ],
            stdout=Tac.CAPTURE, stderr=Tac.CAPTURE,
            ignoreReturnCode=False )
   except Tac.SystemCommandError:
      t0( "Could not extract public key: ", qv( srcKey ) )
      return

   # Write public key
   with open( pubKeyPath, "w" ) as f:
      f.write( pubKeyData )

   # Sync file
   syncfile( pubKeyPath )

def getSshConfigKeyPath( key ):
   '''
      Given a key of the form /persist/secure/ssh/<algo>/<key-name> return
      key of the form /etc/ssh/ssh_keys/<algo>-<key-name>
   '''
   keyAlgo = SshCertLib.getAlgoFromKeyPath( key )
   destKey = "%s-%s" % ( keyAlgo, os.path.basename( key ) )
   return os.path.join( SshConstants.sshKeyConfigDir, destKey )

def vrfNamespace( vrf ):
   if not vrf:
      return DEFAULT_VRF
   else:
      return 'ns-%s' % vrf

def sshFilepath( vrf, v6=False ):
   filepath = '/etc/xinetd.d/ssh'
   if v6:
      filepath += '6'
   if not vrf:
      return filepath
   else:
      return filepath + '-' + vrf

def sshServiceId( vrf, v6=False ):
   serviceId = 'ssh'
   if v6:
      serviceId += '6'
   if vrf:
      serviceId += '-' + vrf
   return serviceId

def _writeXinetdSshConfig( service, v6=False ):
   vrf = None
   if service.vrfStatusLocal:
      vrf = service.vrfStatusLocal.vrfName
   filepath = sshFilepath( vrf, v6 )

   # If service is disabled, set empty config
   if not service.serviceEnabled():
      return service.writeConfigFile( filepath, config="", saveBackupConfig=False )

   ipv4String = '' if v6 else 'IPv4'
   v6String = 'yes' if v6 else 'no'
   if service.config.serverPort == 22:
      bindAddress = '::' if v6 else '0.0.0.0'
   else:
      bindAddress = '::1' if v6 else '127.0.0.1'



   config = """
# default: on
service ssh
{
        id              = %s
        disable         = no
        flags           = REUSE %s
        v6only          = %s
        socket_type     = stream
        wait            = no
        user            = root
        server          = /usr/sbin/sshd
        server_args     = -i -f %s
        namespace       = %s
        bind            = %s
        log_on_failure  += USERID
        instances       = %d
        per_source      = %d
}
   """ % ( sshServiceId( vrf, v6 ), ipv4String, v6String,
           service.configFilename,
           netnsNameWithUniqueId( vrfNamespace( vrf ) ),
           bindAddress, service.config.connLimit, service.config.perHostConnLimit )

   oldFileContents = ""
   if os.path.exists( filepath ):
      with open( filepath ) as f:
         oldFileContents = f.read()
   newFileContents = SuperServer.configFileHeader + config + \
       SuperServer.configFileFooter
   result = True
   if oldFileContents != newFileContents:
      result = service.writeConfigFile( filepath, config, saveBackupConfig=False )
   return result

def writeXinetdSshConfig( service ):
   """
   Prepares and writes ssh service config file for a vrf to
   /etc/xinetd.d/ when service is enables or disbaled through
   'management ssh' config mode
   """
   v6ConfigWritten = _writeXinetdSshConfig( service, v6=True )
   v4ConfigWritten = _writeXinetdSshConfig( service )
   return v6ConfigWritten and v4ConfigWritten

def getTempFileName( prefix, suffix="" ):
   """
      Creating a random file name used for temporary files
   """
   vlist = string.ascii_letters + string.digits
   return '%s-%s%s' % ( prefix, ''.join( random.sample( vlist, 6 ) ),
         suffix )

def writeSshConfig( config, filePath, knownHostsFilename, service ):
   """
   Prepares and writes the ssh_config file
   based on the ssh-managment configuration
   if it has changed.
   """
   sshConf = \
'''
Host *
   GSSAPIAuthentication no
# Send locale-related environment variables
   SendEnv LANG LC_CTYPE LC_NUMERIC LC_TIME LC_COLLATE LC_MONETARY LC_MESSAGES
   SendEnv LC_PAPER LC_NAME LC_ADDRESS LC_TELEPHONE LC_MEASUREMENT
   SendEnv LC_IDENTIFICATION LC_ALL LANGUAGE
   EscapeChar none
   GatewayPorts no
   ForwardX11 no
'''
   filteredCiphers, filteredMacs, \
         filteredKex, filteredHostKey = filterSshOptions( config )

   if config.fipsRestrictions:
      sshConf += "   UseFipsAlgorithms yes\n"

   t8( "writeSshConfig", filteredHostKey )
   if filteredHostKey:
      if "rsa" in filteredHostKey:
         sshConf += "IdentityFile %s\n" % SysMgrLib.rsaKeyPath
      if "dsa" in filteredHostKey:
         sshConf += "IdentityFile %s\n" % SysMgrLib.dsaKeyPath
         sshConf += "PubkeyAcceptedKeyTypes = +ssh-dss\n"
         sshConf += "HostKeyAlgorithms = +ssh-dss\n"
      if "ecdsa-nistp521" in filteredHostKey:
         sshConf += "IdentityFile %s\n" % SysMgrLib.ecdsa521KeyPath
      if "ecdsa-nistp256" in filteredHostKey:
         sshConf += "IdentityFile %s\n" % SysMgrLib.ecdsa256KeyPath
      if "ed25519" in filteredHostKey:
         sshConf += "IdentityFile %s\n" % SysMgrLib.ed25519KeyPath

   if filteredCiphers:
      sshConf += "   Ciphers %s\n" % ",".join( filteredCiphers )
   if filteredKex:
      sshConf += "   KexAlgorithms %s\n" % ",".join( filteredKex )
   if filteredMacs:
      sshConf += "   MACs %s\n" % ",".join( filteredMacs )
   strictHostKeyCheckingOption = "no"
   knownHostsOption = "/dev/null"
   if config.enforceCheckHostKeys:
      strictHostKeyCheckingOption = "yes"
      knownHostsOption = knownHostsFilename
   sshConf += "   StrictHostKeyChecking %s\n" % ( strictHostKeyCheckingOption )
   sshConf += "   UserKnownHostsFile %s\n" % ( knownHostsOption )
   sshConf += "   GlobalKnownHostsFile %s\n" % ( knownHostsOption )
   rekeyStr = deriveRekeyStr( config )
   sshConf += "   RekeyLimit %s\n" % ( rekeyStr )
   # If we do syslog, syslog to AUTHPRIV
   sshConf += "   SyslogFacility AUTHPRIV\n"
   sshConf += "   LogLevel %s\n" % ( config.logLevel.upper(), )
   oldConfigFileContents = ""
   if os.path.exists( filePath ):
      with open( filePath ) as f:
         oldConfigFileContents = f.read()
   newConfigFileContents = SuperServer.configFileHeader + sshConf +\
         SuperServer.configFileFooter
   result = True
   if oldConfigFileContents != newConfigFileContents:
      # We should not use function 'knownHostsFileName' overridden by XinetdService,
      # but the one defined by LinuxService
      result = SuperServer.LinuxService.writeConfigFile( service, filePath, sshConf )
   return result

def unlinkFile( fileName ):
   try:
      os.unlink( fileName )
   except OSError as e:
      if e.errno != errno.ENOENT:
         raise

def cleanupSshd( name, allVrfStatusLocal ):
   '''Cleanup sshd files and instances running in a VRF
      @param name the VRF name
      @param allVrfStatus VRF status, to disambiguate pam.d file names
   '''
   qt0( "Cleaning up ssh for VRF", qv( name ) )
   unlinkFile( "/etc/ssh/sshd_config-" + name )
   unlinkFile( "/etc/ssh/ssh_known_hosts-" + name )
   unlinkFile( "/etc/ssh/ssh_known_hosts-" + name + ".save" )
   unlinkFile( "/etc/ssh/ssh_config-" + name )
   unlinkFile( "/etc/xinetd.d/ssh-" + name )
   unlinkFile( "/etc/xinetd.d/ssh6-" + name )
   unlinkFile( "/etc/xinetd.d/ssh_serverport-%s" % name )
   unlinkFile( "/etc/xinetd.d/ssh_serverport-%s-6" % name )
   unlinkFile( "/etc/ssh/ssh_serverport-%s" % name )
   unlinkFile( SysMgrLib.rsaKeyPath + "-" + name )
   unlinkFile( SysMgrLib.rsaKeyPath + "-" + name + ".pub" )
   unlinkFile( SysMgrLib.dsaKeyPath + "-" + name )
   unlinkFile( SysMgrLib.dsaKeyPath + "-" + name + ".pub" )

   # pam.d files need special handling
   pamEquivalentVrf = None
   for vrfName, vrf in allVrfStatusLocal.vrf.items():
      if vrfName.lower() == name.lower() and vrfName != name and \
         vrf.state == 'active':
         pamEquivalentVrf = vrfName
   pamFileName = "/etc/pam.d/sshd-" + name.lower()
   if pamEquivalentVrf:
      qt0( "File", qv( pamFileName ),
           "could not be removed as it's required for VRF",
           qv( pamEquivalentVrf ) )
   else:
      unlinkFile( pamFileName )

class VrfStatusLocalReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::VrfStatusLocal'

   def __init__( self, status, agent, allVrfStatusLocal ):
      qt0( 'Initializing VrfStatusLocalReactor for SSH' )
      Tac.Notifiee.__init__( self, status )
      self.agent_ = weakref.proxy( agent )
      self.allVrfStatusLocal_ = allVrfStatusLocal
      self.service_ = None
      self.activity_ = Tac.ClockNotifiee()
      self.activity_.handler = self.maybeGoActive
      self.activity_.timeMin = Tac.endOfTime
      self.handleState()

   def createSshdFiles( self ):
      vrfName = self.notifier_.vrfName

      def symlinkMainFile( mainFile, vrfName ):
         try:
            os.symlink( mainFile, mainFile + "-" + vrfName )
         except OSError as e:
            if e.errno != errno.EEXIST:
               raise

      # pam.d names have to be lower case
      symlinkMainFile( "/etc/pam.d/sshd", vrfName.lower() )

      return True

   @Tac.handler( 'state' )
   def handleState( self ):
      qt0( 'SSH VrfStatusLocalReactor vrfname:', qv( self.notifier_.vrfName ),
           'handleState: ', qv( self.notifier_.state ) )
      if self.notifier_.state == "active":
         self.maybeGoActive()
      elif self.notifier_.state == "deleting":
         # The sshd files must be cleaned before calling the stopService()
         # In this case, service is xinetd and sync will schedule a call to
         # stopService() to "reload" the xinetd config
         # ---
         # Instead of using stopService directly, sync is being used in order to
         # avoid a race condition with Telnet SuperServer Plugin when vrf is deleted.
         # It makes sure that both ssh and telnet configs are deleted from
         # /etc/xinetd.d before xinetd is reloaded
         cleanupSshd( self.notifier_.vrfName, self.allVrfStatusLocal_ )
         if self.service_:
            XinetdService.scheduleReloadService()
            self.agent_.service_.sync()
            # use list as the tunnel can change during iteration
            # pylint: disable-msg=C0201
            for tunnelName in list( self.service_.sshTunnels_ ):
               self.service_.handleTunnel( tunnelName, forceTearDown=True )
            # pylint: enable-msg=C0201
            self.service_.cleanupService()
            del self.agent_.vrfServices_[ self.notifier_.vrfName ]
            self.service_ = None

   def maybeGoActive( self ):
      if self.notifier_.state == 'active':
         self.activity_.timeMin = Tac.endOfTime
         filesCreated = self.createSshdFiles()
         if filesCreated:
            self.service_ = SshService( self.agent_.config_,
                                        self.agent_.status_,
                                        self.agent_,
                                        self.agent_.sslStatus_,
                                        self.notifier_ )
            self.agent_.vrfServices_[ self.notifier_.vrfName ] = self.service_
         else:
            self.activity_.timeMin = Tac.now() + 1

   def close( self ):
      self.activity_.timeMin = Tac.endOfTime
      Tac.Notifiee.close( self )


serverPort = Tac.Type( "Mgmt::Ssh::ServerPort" )

tunnelRestartCost = 60
maxTunnelRestartCredits = tunnelRestartCost * 5

class SshTunnel( SuperServer.LinuxService ):
   notifierTypeName = "Mgmt::Ssh::TunnelConfig"

   def __init__( self, config, status, parentConfig, knownHostsFilename,
                 sshConfigFilename, sshSuperServerAgent, vrfStatusLocal ):
      t8( "SshTunnel::__init__: %s" % config.name )
      serviceName = ssh.name()
      self.timeStarted_ = None
      self.config_ = config
      self.status_ = status
      self.tunnelWarmup_ = 60 * 10
      self.tunnelStartTime_ = Tac.beginningOfTime
      self.parentConfig_ = parentConfig
      self.knownHostsFilename_ = knownHostsFilename
      self.sshConfigFilename_ = sshConfigFilename
      self.configFilename_ = "/etc/ssh/ssh_tunnel"
      self.sshSuperServerAgent = sshSuperServerAgent
      if vrfStatusLocal:
         self.configFilename_ += "-" + vrfStatusLocal.vrfName
      self.vrfStatusLocal = vrfStatusLocal
      self.configFilename_ += "-" + self.config_.name
      self.stderr_ = ""
      self.sshTunnelProcess_ = None
      self.sshTunnelCmd_ = []
      self.initialized_ = False
      self.status_.established = False
      self.tunnelRestartCredits_ = maxTunnelRestartCredits
      self.tunnelSlowRestartTimer_ = Tac.ClockNotifiee()
      self.tunnelSlowRestartTimer_.timeMin = Tac.endOfTime
      # pylint: disable-next=assignment-from-none
      self.tunnelSlowRestartTimer_.handler = self.startService()
      self.maybeKillPreviousTunnel()
      SuperServer.LinuxService.__init__( self, "SshTunnel", serviceName, config,
            self.configFilename_ )

   def maybeKillPreviousTunnel( self ):
      """
      If there is a previous ssh tunnel running ( from a restart )
      then kill it. The previous tunnel is determined by looking at
      if there is a pid file for it.
      """
      t8( "SshTunnel::maybeKillPreviousTunnel: ", self.config_.name )
      pidRaw = None
      try:
         with open( self.config_.name + ".pid" ) as pidFile:
            pidRaw = pidFile.read()
      except OSError:
         t5( "Unable to open PID file" )
         return
      pidInt = None
      try:
         pidInt = int( pidRaw )
      except ValueError:
         t5( "Error converting pidRaw: ", pidRaw )
         return
      try:
         os.kill( pidInt, signal.SIGKILL )
      except OSError:
         # This is not an error, since the process will not
         # exist unless SuperServer was killed previously
         t5( "SSH Tunnel Process not found: ", pidInt )
         return
      return

   def onNetworkDisruption( self ):
      """
      The network was disrupted ( i.e. layer-3 disruptiuon during a switchover ).
      Reset the warmup timers and attempt to resync.
      """
      t8( "SshTunnel::onNetworkDisruption(%s)" % self.config_.name )
      self.tunnelStartTime_ = Tac.beginningOfTime
      self.sync()

   def tunnelShouldBeWarm( self ):
      """
      Returns a bool to indicate if the tunnel warmup has been exceeded, i.e.
      The tunnel should be warm at this point.
      """
      return Tac.now() > ( self.tunnelStartTime_ + self.tunnelWarmup_ )

   # pkgdeps: import EosLogUtil (this depends on the FetchLogs utility)
   def _dumpLogTargetTunnelOutput( self ):
      """
      Dump the ssh tunnel logs from /var/log/messages,
      using the tunnel PID to filter messages.
      """
      logFilename = "/var/log/messages"
      match = os.path.basename( logFilename ) + r'(?=($|\..*\.gz))'
      # pylint: disable-next=consider-using-with
      allLogsProc = subprocess.Popen( [ sys.executable, '/usr/bin/FetchLogs', 'dump',
                                        '-n', '-l', os.path.dirname( logFilename ),
                                        '-m', match ],
                                        stdout=subprocess.PIPE,
                                        stderr=subprocess.PIPE,
                                        universal_newlines=True )
      allLogs = allLogsProc.stdout.read()
      sshTunnelOutput = ""
      sshPidFilter = "ssh[%d]" % self.sshTunnelProcess_.grandchild
      for line in allLogs.splitlines():
         if sshPidFilter in line:
            sshTunnelOutput += line
      t0( "SshTunnel messages output: %s" % sshTunnelOutput )
      return sshTunnelOutput

   def _readOutput( self ):
      if self.sshTunnelProcess_:
         # Read non-blocking pipe and return the entire output (including previously
         # returned).
         if self.parentConfig_.loggingTargetEnabled:
            self.stderr_ += self._dumpLogTargetTunnelOutput()
         else:
            try:
               output = self.sshTunnelProcess_.stderr.read()
               if output:
                  output = output.decode()
                  t0( "SshTunnel output:", output )
                  self.stderr_ += output
            except OSError:
               pass
      return self.stderr_

   def serviceProcessWarm( self ):
      # This function will return a boolean representing if the
      # SSH Tunnel has logged into the ssh-server successfully.
      # This function is not currently used by anyone.
      if not self.sshTunnelProcess_:
         return False
      if self.sshTunnelProcess_.wait( block=False ) is not None:
         # The process exited
         self.sshTunnelProcess_ = None
         return False
      return bool( "debug1: Entering interactive session." in
                   self._readOutput() )

   def serviceEnabled( self ):
      self.status_.fullyConfigured = bool(
               self.config_.localPort != serverPort.invalid and
               self.config_.remotePort != serverPort.invalid and
               self.config_.remoteHost and
               self.config_.sshServerUsername and
               self.config_.sshServerPort != serverPort.invalid )
      self.status_.enabled =  bool(
               self.config_.enable and
               self.status_.fullyConfigured and
               self.sshSuperServerAgent.mgmtSecurityMgr_.entropySourceOk()
               and self.sshSuperServerAgent.active() and
               ( not self.vrfStatusLocal or
                 self.vrfStatusLocal.state == "active" ) )
      return self.status_.enabled

   def conf( self ):
      t8( "SshTunnel::conf(%s)" % self.config_.name )
      # This file isn't actually read by a program, but if the options change
      # SuperServer knows to restart the SSH Tunnel

      # Write the SSH config before preparing this one in case there's been a change
      if not writeSshConfig( self.parentConfig_, self.sshConfigFilename_,
                             self.knownHostsFilename_, self ):
         self.sync()

      mgmtSecStatus = self.sshSuperServerAgent.mgmtSecurityMgr_.status_
      # Include entropy info in the config so we restart all tunnels if it changes
      conf = ''
      if mgmtSecStatus.entropySourceHardwareEnabled:
         conf += '# entropy source hardware is enabled\n'
      if mgmtSecStatus.entropySourceHavegedEnabled:
         conf += '# entropy source haveged is enabled\n'
      if mgmtSecStatus.entropySourceJitterEnabled:
         conf += '# entropy source jitter is enabled\n'
      filteredCiphers, filteredMacs, \
            filteredKex, _filteredHostKey = filterSshOptions( self.parentConfig_ )
      conf += "SshServer %s@%s:%d\n" % ( self.config_.sshServerUsername,
            self.config_.sshServerAddress, self.config_.sshServerPort )
      conf += "LocalPort %d\n" % ( self.config_.localPort )
      conf += "Remote %s %d\n" % ( self.config_.remoteHost,
            self.config_.remotePort )
      conf += "ServerAliveInterval %d\n" % ( self.config_.serverAliveInterval )
      conf += "ServerAliveMaxLost %d\n" % ( self.config_.serverAliveMaxLost )
      if self.parentConfig_.fipsRestrictions:
         conf += "UseFipsAlgorithms yes\n"
      if filteredCiphers:
         conf += "Ciphers %s\n" % ",".join( filteredCiphers )
      if filteredKex:
         conf += "KexAlgorithms %s\n" % ",".join( filteredKex )
      if filteredMacs:
         conf += "MACs %s\n" % ",".join( filteredMacs )
      if self.serviceEnabled():
         conf += "Tunnel Enabled\n"
      else:
         conf += "Tunnel Disabled\n"

      return conf

   def startService( self ):
      t8( "SshTunnel::startService(%s)" % self.config_.name )
      if not self.initialized_:
         self.initialized_ = True
         return
      if self.sshTunnelProcess_:
         if self.sshTunnelProcess_.wait( block=False ) is None:
            # The process is still running
            return
         else:
            # The process has exited
            self.sshTunnelProcess_ = None
      # Check if we have enough credits to start the service
      # If we have too few credits, there's something wrong and the
      # tunnel has been failing repeatedly. We sleep for a while
      # and don't give any credits for the time spent sleeping.
      #
      # If we're in the inital warmup window, we don't charge
      # credits since the connectivity may not yet exist for the
      # tunnel
      if self.tunnelShouldBeWarm():
         self.tunnelRestartCredits_ -= tunnelRestartCost
      if ( self.tunnelRestartCredits_ < 0 ) and\
            ( not self.config_.unlimitedRestarts ):
         # pylint: disable-msg=E0602
         Logging.log( SECURITY_SSH_TUNNEL_SLOW_RESTART, self.config_.name )
         self.tunnelRestartCredits_ = tunnelRestartCost
         self.tunnelSlowRestartTimer_.timeMin = Tac.now() +\
               maxTunnelRestartCredits
         return

      tunnelCmd = "%s:%s:%s" % ( self.config_.localPort,
            self.config_.remoteHost, self.config_.remotePort )
      strictHostKeyChecking = "StrictHostKeyChecking %s" % \
            ( "yes" if self.parentConfig_.enforceCheckHostKeys else "no" )
      userKnownHostsFile = "UserKnownHostsFile %s" % \
            ( self.knownHostsFilename_ if\
            self.parentConfig_.enforceCheckHostKeys else "/dev/null" )
      globalKnownHostsFile = "GlobalKnownHostsFile %s" % \
            ( self.knownHostsFilename_ if\
            self.parentConfig_.enforceCheckHostKeys else "/dev/null" )
      serverAliveInterval = "ServerAliveInterval %d" % \
            ( self.config_.serverAliveInterval )
      serverAliveMaxLost = "ServerAliveCountMax %d" % \
            ( self.config_.serverAliveMaxLost )

      sshTunnelCommands = [
              'ssh',
              '-v',
      ]
      if self.parentConfig_.loggingTargetEnabled:
         sshTunnelCommands += [ '-y', ]
      sshTunnelCommands += [
              '-L', tunnelCmd,
              "-N",
              "-o", "BatchMode yes",
              "-o", userKnownHostsFile,
              "-o", globalKnownHostsFile,
              "-o", strictHostKeyChecking,
              "-o", serverAliveInterval,
              "-o", serverAliveMaxLost,
              "-p", str( self.config_.sshServerPort ),
              self.config_.sshServerUsername+'@'+self.config_.sshServerAddress
            ]

      if self.vrfStatusLocal:
         vrfTunnel = [ "ip", "netns", "exec", self.vrfStatusLocal.networkNamespace ]
         vrfTunnel.extend( sshTunnelCommands )
         sshTunnelCommands = vrfTunnel

      t8( "SSH Tunnel command used was: %s" % ( " ".join( sshTunnelCommands ) ) )
      # Also save for later use in the logfile
      self.sshTunnelCmd_ = sshTunnelCommands

      self.sshTunnelProcess_ = ManagedSubprocess.Popen(
            sshTunnelCommands,
            stderr=ManagedSubprocess.PIPE,
            text=False )
      with open( self.configFilename_ + ".pid", "w" ) as pidFile:
         pidFile.write( str( self.sshTunnelProcess_.pid ) +"\n" )
      # set stderr to nonblocking
      ManagedSubprocess.nonblock( self.sshTunnelProcess_.stderr.fileno() )
      self.stderr_ = ""
      self.timeStarted_ = Tac.time.time()
      if self.tunnelStartTime_ == Tac.beginningOfTime:
         self.tunnelStartTime_ = Tac.now()
      self.status_.active = True
      self.starts += 1
      # Schedule health monitoring faster in case there is an error
      self.healthMonitorInterval = 10

   def stopService( self ):
      t8( "SshTunnelCommand::stopService(%s)" % self.config_.name )
      entireOutput = self.stderr_
      if self.sshTunnelProcess_:
         self.sshTunnelProcess_.kill()
         self.sshTunnelProcess_.wait()
         entireOutput = self._readOutput()
         self.sshTunnelProcess_ = None
      elif self.status_.active == False: # pylint: disable=singleton-comparison
         # No need to do any accounting since the tunnel
         # was not previously active
         return

      timeEnded = Tac.time.time()

      # Since the health check is only ran periodically, subtract the interval
      # to get the time the tunnel was known to be running correctly
      timeTunnelRan = timeEnded - self.timeStarted_ - self.healthMonitorInterval
      if timeTunnelRan < 0: # pylint: disable=consider-using-max-builtin
         # Prevent losing time credits if the tunnel failed instantly
         timeTunnelRan = 0
      self.tunnelRestartCredits_ += timeTunnelRan
      # pylint: disable-next=consider-using-min-builtin
      if self.tunnelRestartCredits_ >= maxTunnelRestartCredits:
         self.tunnelRestartCredits_ = maxTunnelRestartCredits
      # Save a log of what was output from the SSH Tunnel in case
      # log messages fail to triage the issue.
      with open( self.configFilename_ + ".log", "w" ) as logFile:
         logFile.write( "SSH Tunnel Command used was: " )
         logFile.write( " ".join( self.sshTunnelCmd_ ) + "\n" )
         logFile.write( entireOutput )
      for line in entireOutput.splitlines():
         # pylint: disable-msg=E0602
         # We split the lines to check the public key error messages
         if "Could not resolve hostname" in line:
            if self.tunnelShouldBeWarm():
               Logging.log( SECURITY_SSH_TUNNEL_HOSTNAME,
                     self.config_.name, self.config_.sshServerAddress )
            self.status_.lastRestartCause = "SECURITY_SSH_TUNNEL_HOSTNAME"
         elif ( "Permission denied" in line ) and ( "publickey" in line ):
            Logging.log( SECURITY_SSH_TUNNEL_SWITCH_HOSTKEY_DENIED,
                  self.config_.name )
            self.status_.lastRestartCause = \
                  "SECURITY_SSH_TUNNEL_SWITCH_HOSTKEY_DENIED"
         elif "Connection timed out" in line:
            if self.tunnelShouldBeWarm():
               Logging.log( SECURITY_SSH_TUNNEL_INITIAL_TIMEOUT,
                     self.config_.name )
            self.status_.lastRestartCause = "SECURITY_SSH_TUNNEL_INITIAL_TIMEOUT"
         elif "No route to host" in line:
            if self.tunnelShouldBeWarm():
               Logging.log( SECURITY_SSH_TUNNEL_INITIAL_TIMEOUT,
                     self.config_.name )
            self.status_.lastRestartCause = "SECURITY_SSH_TUNNEL_INITIAL_TIMEOUT"
         elif "Network is unreachable" in line:
            if self.tunnelShouldBeWarm():
               Logging.log( SECURITY_SSH_TUNNEL_INITIAL_TIMEOUT,
                     self.config_.name )
            self.status_.lastRestartCause = "SECURITY_SSH_TUNNEL_INITIAL_TIMEOUT"
         elif "Connection refused" in line:
            Logging.log( SECURITY_SSH_TUNNEL_CONNECTION_REFUSED,
                  self.config_.name )
            self.status_.lastRestartCause = "SECURITY_SSH_TUNNEL_CONNECTION_REFUSED"
         elif re.search( r"Timeout, server .* not responding.", line ):
            Logging.log( SECURITY_SSH_TUNNEL_TIMEOUT,
                  self.config_.name )
            self.status_.lastRestartCause = "SECURITY_SSH_TUNNEL_TIMEOUT"
         elif "closed by remote host" in line:
            Logging.log( SECURITY_SSH_TUNNEL_CLOSED_REMOTELY,
                  self.config_.name )
            self.status_.lastRestartCause = "SECURITY_SSH_TUNNEL_CLOSED_REMOTELY"
         elif "Host key verification failed" in line:
            Logging.log( SECURITY_SSH_TUNNEL_HOSTKEY_VERIFY_FAILED,
                  self.config_.name )
            self.status_.lastRestartCause = \
                  "SECURITY_SSH_TUNNEL_HOSTKEY_VERIFY_FAILED"
         elif "no matching cipher found" in line:
            Logging.log( SECURITY_SSH_TUNNEL_ALGORITHM_MISMATCH,
                  self.config_.name, "cipher" )
            self.status_.lastRestartCause = \
                  "SECURITY_SSH_TUNNEL_ALGORITHM_MISMATCH"
         elif "no matching host key type found" in line:
            Logging.log( SECURITY_SSH_TUNNEL_ALGORITHM_MISMATCH,
                         self.config_.name, "hostkey algorithm" )
            self.status_.lastRestartCause = \
                  "SECURITY_SSH_TUNNEL_ALGORITHM_MISMATCH"
         elif "no matching MAC found" in line:
            Logging.log( SECURITY_SSH_TUNNEL_ALGORITHM_MISMATCH,
                  self.config_.name, "mac" )
            self.status_.lastRestartCause = \
                  "SECURITY_SSH_TUNNEL_ALGORITHM_MISMATCH"
         elif "no matching key exchange method found" in line:
            Logging.log( SECURITY_SSH_TUNNEL_ALGORITHM_MISMATCH,
                  self.config_.name, "kex" )
            self.status_.lastRestartCause = \
                  "SECURITY_SSH_TUNNEL_ALGORITHM_MISMATCH"
         elif "Disconnecting: Packet corrupt" in line:
            Logging.log( SECURITY_SSH_TUNNEL_CORRUPT_PACKET_RECEIVED,
                  self.config_.name )
            self.status_.lastRestartCause = \
                  "SECURITY_SSH_TUNNEL_CORRUPT_PACKET_RECEIVED"
         elif "Disconnecting: Protocol error: expected packet type" in line:
            Logging.log( SECURITY_SSH_TUNNEL_CORRUPT_PACKET_RECEIVED,
                  self.config_.name )
            self.status_.lastRestartCause = \
                  "SECURITY_SSH_TUNNEL_CORRUPT_PACKET_RECEIVED"
         elif "Disconnecting: Corrupted padlen" in line:
            Logging.log( SECURITY_SSH_TUNNEL_CORRUPT_PACKET_RECEIVED,
                  self.config_.name )
            self.status_.lastRestartCause = \
                  "SECURITY_SSH_TUNNEL_CORRUPT_PACKET_RECEIVED"
         elif "Disconnecting: Invalid ssh2 packet type" in line:
            Logging.log( SECURITY_SSH_TUNNEL_CORRUPT_PACKET_RECEIVED,
                  self.config_.name )
            self.status_.lastRestartCause = \
                  "SECURITY_SSH_TUNNEL_CORRUPT_PACKET_RECEIVED"
      self.stops += 1
      self.status_.active = False
      self.status_.established = False

   def _checkServiceHealth( self ):
      if not self.serviceEnabled():
         return
      doRestart = False
      entireOutput = self.stderr_
      if self.sshTunnelProcess_:
         if self.sshTunnelProcess_.wait( block=False ) is not None:
            entireOutput = self._readOutput()
            self.sshTunnelProcess_ = None
            doRestart = True
         else:
            entireOutput = self._readOutput()
      # Check for runtime errors that won't stop the program
      # and restart the tunnel if any of them are noticed
      if "channel 2: open failed:" in entireOutput:
         Logging.log( SECURITY_SSH_TUNNEL_REMOTE_PORT_ERROR,
                      self.config_.name )
         self.status_.lastRestartCause = "SECURITY_SSH_TUNNEL_REMOTE_PORT_ERROR"
         doRestart = True
      if "channel_setup_fwd_listener_tcpip: cannot listen to port" in entireOutput:
         Logging.log( SECURITY_SSH_TUNNEL_LOCAL_PORT_ERROR, self.config_.name )
         self.status_.lastRestartCause = "SECURITY_SSH_TUNNEL_LOCAL_PORT_ERROR"
         doRestart = True
      if "bind: Address already in use" in entireOutput:
         Logging.log( SECURITY_SSH_TUNNEL_LOCAL_PORT_ERROR, self.config_.name )
         self.status_.lastRestartCause = "SECURITY_SSH_TUNNEL_LOCAL_PORT_ERROR"
         doRestart = True
      # The next set of errors are for bad packets being received. They do not need
      # the tunnel to be reset to go away so they do not result in a restart
      if "Bad packet length" in entireOutput:
         Logging.log( SECURITY_SSH_TUNNEL_CORRUPT_PACKET_RECEIVED,
                      self.config_.name )
      if "padding error: need" in entireOutput:
         Logging.log( SECURITY_SSH_TUNNEL_CORRUPT_PACKET_RECEIVED,
                      self.config_.name )
      if "Corrupted MAC on input." in entireOutput:
         Logging.log( SECURITY_SSH_TUNNEL_CORRUPT_PACKET_RECEIVED,
                      self.config_.name )
      if doRestart:
         self.restartService()
      else:
         # No errors found so lets see if we should send the established message
         if not self.status_.established and \
               bool( "debug1: Entering interactive session." in entireOutput ):
            Logging.log( SECURITY_SSH_TUNNEL_ESTABLISHED,
                         self.config_.name,
                         str( self.config_.localPort ),
                         self.config_.remoteHost,
                         str( self.config_.remotePort ),
                         self.config_.sshServerUsername,
                         self.config_.sshServerAddress )
            self.status_.established = True
            # Restore the normal health monitor interval
            self.healthMonitorInterval = 60

      self.healthMonitorActivity_.timeMin = Tac.now() + \
                                               self.healthMonitorInterval

   def restartService( self ):
      t8( "SshTunnel::restartService(%s)" % self.config_.name )
      tunnelWasActive = self.status_.active
      self.stopService()
      self.startService()
      # Keep the statistics correct
      self.stops -= 1
      self.starts -= 1
      self.restarts += 1
      if tunnelWasActive:
         self.status_.lastRestartTime = Tac.utcNow()
         self.status_.restartCount = self.restarts

class SshMgmtSecurityMgr( SysMgrLib.MgmtSecurityMgr ):
   def __init__( self, config, status, agent ):
      self.agent_ = agent
      SysMgrLib.MgmtSecurityMgr.__init__( self, config, status, agent )

   def sync( self ):
      if self.agent_.service_:
         self.agent_.service_.sync()
         for sshService in self.agent_.vrfServices_.values():
            sshService.sync()

class AaaConfigReactor( Tac.Notifiee ):
   notifierTypeName = "LocalUser::Config"

   def __init__( self, notifier, agent ):
      Tac.Notifiee.__init__( self, notifier )
      self.agent_ = agent

   @Tac.handler( 'allowRemoteLoginWithEmptyPassword' )
   def handleAllowRemoteLoginWithEmptyPassword( self ):
      if self.agent_.service_:
         self.agent_.service_.sync()
         for sshService in self.agent_.vrfServices_.values():
            sshService.sync()

class NetconfEndpointReactor( Tac.Notifiee ):
   notifierTypeName = "Netconf::Endpoint"

   def __init__( self, notifier, agent ):
      Tac.Notifiee.__init__( self, notifier )
      self._notifier = notifier
      self.agent_ = agent
      if self.agent_.service_ is not None:
         self.agent_.service_.sync()
      self.serviceName = 'sshnetconf'
      self.handlePort()

   def netconfFileName( self, v6=False ):
      # Note, we use the endpoint name instead of vrf name to generate the
      # config file name. This way given a named endpoint the name doesn't change
      # and simplifies our code.
      return "/etc/xinetd.d/%s" % serviceConfigName( self.serviceName,
                                                     vrf=self.notifier_.name,
                                                     v6=v6 )

   @Tac.handler( "port" )
   def handlePort ( self ):
      # If the transport is not enabled, there is no need to listen to the port
      if not self._notifier.enabled:
         return
      self._writeConfig( )
      self._writeConfig( v6=True )
      self.doSync()

   @Tac.handler( "vrfName" )
   def handleVrf ( self ):
      # If the transport is not enabled, there is no need to act
      if not self._notifier.enabled:
         return
      # cleanup netconf redirect configurations present in /etc/xinetd.d/
      self._writeConfig()
      self._writeConfig( v6=True )
      self.doSync()

   @Tac.handler( "enabled" )
   def handleEnabled ( self ):
      # cleanup all netconf redirect configurations
      if self._notifier.enabled:
         self._writeConfig()
         self._writeConfig( v6=True )
      else:
         unlinkFile( self.netconfFileName() )
         unlinkFile( self.netconfFileName( v6=True ) )
      self.doSync()

   def _writeConfig( self, v6=False ):
      vrf = self._notifier.vrfName
      if not vrf or vrf == DEFAULT_VRF:
         vrf = None
      redirConf = _getSshRedirectConfig( self.serviceName,
                                         port=self._notifier.port,
                                         vrf=vrf,
                                         v6=v6 )
      redirFname = self.netconfFileName( v6=v6 )
      if not SuperServer.LinuxService.writeConfigFile( self.agent_.service_,
                                                       redirFname,
                                                       redirConf,
                                                       saveBackupConfig=False ):
         self.doSync()

   def doSync( self ):
      if self.agent_.service_:
         self.agent_.service_.sync()
         for sshService in self.agent_.vrfServices_.values():
            sshService.sync()

   def close( self ):
      Tac.Notifiee.close( self )

class NetconfReactor( Tac.Notifiee ):
   notifierTypeName = "Netconf::Config"

   def __init__( self, notifier, sshService ):
      Tac.Notifiee.__init__( self, notifier )
      self.sshService_ = sshService
      self.handleEnabled()

   @Tac.handler( "enabled" )
   def handleEnabled ( self ):
      self.sshService_.sync()

   def close( self ):
      Tac.Notifiee.close( self )

class VrfConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Mgmt::Ssh::VrfConfig"

   def __init__( self, notifier, master ):
      qt0( "SSH VrfConfigReactor created for ", notifier.vrfName )
      Tac.Notifiee.__init__( self, notifier )
      self.master_ = master
      self.master_.status_.vrfStatus.newMember( notifier.vrfName )
      self.handleVrfKnownHost( name=None )
      self.handleVrfTunnel( name=None )
      self.handleVrfConfig()

   @Tac.handler( 'serverState' )
   def handleVrfConfig( self ):
      if self.notifier_.vrfName == DEFAULT_VRF:
         self.master_.service_.sync()
      elif self.notifier_.vrfName in self.master_.vrfServices_:
         self.master_.vrfServices_[ self.notifier_.vrfName ].sync()

   @Tac.handler( 'tunnel' )
   def handleVrfTunnel( self, name ):
      if self.notifier_.vrfName in self.master_.vrfServices_:
         vrfStatus = self.master_.status_.vrfStatus[ self.notifier_.vrfName ]
         self.master_.vrfServices_[ self.notifier_.vrfName ].handleTunnel( name,
               sshVrfConfig=self.notifier_, sshVrfStatus=vrfStatus )

   @Tac.handler( 'knownHost' )
   def handleVrfKnownHost( self, name ):
      if self.notifier_.vrfName in self.master_.vrfServices_:
         self.master_.vrfServices_[ self.notifier_.vrfName ].handleKnownHost( name,
               sshVrfConfig=self.notifier_ )

   def close( self ):
      del self.master_.status_.vrfStatus[ self.notifier_.vrfName ]
      Tac.Notifiee.close( self )

class ConfigReqReactor( Tac.Notifiee ):
   notifierTypeName = "Mgmt::Ssh::ConfigReq"

   def __init__( self, notifier, sshService ):
      Tac.Notifiee.__init__( self, notifier )
      self.sshService_ = sshService
      self.handleSshCertFileUpdateRequest()

   @Tac.handler( 'sshCertFileUpdateRequest' )
   def handleSshCertFileUpdateRequest( self ):
      requestTime = self.notifier_.sshCertFileUpdateRequest
      lastProcessedTime = self.sshService_.status.sshCertFileUpdateProcessed
      if lastProcessedTime < requestTime:
         self.sshService_.sync()

class SysMgrX509SslReactor( MgmtSecuritySslStatusSm.SslStatusSm ):
   __supportedFeatures__ = [ SslFeature.sslFeatureTrustedCert,
                             SslFeature.sslFeatureCrl,
                             SslFeature.sslFeatureOcsp ]

   def __init__( self, status, profileName, sshService ):
      self.profileName = profileName
      self.sshService = sshService
      super().__init__( status, profileName, 'SysMgr' )
      self.sshService.sync()

   def handleProfileState( self ):
      self.sshService.sync()

   def handleProfileDelete( self ):
      self.sshService.sync()

class SshUserConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Mgmt::Ssh::UserConfig"

   def __init__( self, notifier, sshConfig, aaaLocalConfig, agent ):
      Tac.Notifiee.__init__( self, notifier )
      self.agent_ = agent
      self.user_ = notifier
      self.sshConfig_ = sshConfig
      self.aaaLocalConfig_ = aaaLocalConfig
      self.sshUserKeyReactors_ = {}

   @Tac.handler( "userTcpForwarding" )
   def handleUserTcpForwarding( self ):
      t8( "handle user", self.notifier_.name,
          "with tcpForwarding", self.notifier_.userTcpForwarding )
      if self.agent_.service_:
         self.agent_.service_.sync()
         for sshService in self.agent_.vrfServices_.values():
            sshService.sync()

class SshAuthenMethodListReactor( Tac.Notifiee ):
   notifierTypeName = "Mgmt::Ssh::AuthenMethodList"

   def __init__( self, notifier, agent ):
      Tac.Notifiee.__init__( self, notifier )
      self.agent_ = agent

      # Here we would call self.hanldeMethod() if we
      # allowed an empty authentication method list
      # But since an empty list is not allowed, and
      # we are guaranteed atleast one method in the
      # list when we create a new one,
      # self.handleMethod(...) is triggered right away to do the sync


   @Tac.handler( "method" )
   def handleMethod( self, key ):
      if self.agent_.service_:
         self.agent_.service_.sync()
         for sshService in self.agent_.vrfServices_.values():
            sshService.sync()

# Xinetd will provide the ssh service
class SshService( XinetdService ):
   notifierTypeName = "Mgmt::Ssh::Config"

   def __init__( self, config, status, sshSuperServerAgent, sslStatus,
                 vrfStatusLocal=None ):
      self.config = config
      self.status = status
      self.sshSuperServerAgent = sshSuperServerAgent
      assert type( sshSuperServerAgent ) == weakref.ProxyType, \
             "Should be a weak reference"
      self.sslStatus = sslStatus
      self.vrfStatusLocal = vrfStatusLocal

      if vrfStatusLocal:
         qt0( "SshService created for vrfStatusLocal of ",
              str( vrfStatusLocal.vrfName ) )
      else:
         qt0( "SshService created for default VRF" )

      serviceName = sshd.name()
      self.configFilename = "/etc/ssh/sshd_config"
      self.knownHostsFilename = "/etc/ssh/ssh_known_hosts"
      self.sshConfigFilename = "/etc/ssh/ssh_config"
      self.xinetdSshConfigFilename = '/etc/xinetd.d/ssh'

      self.sshTunnels_ = dict() # pylint: disable=use-dict-literal

      if self.vrfStatusLocal:
         serviceName += "-%s" % self.vrfStatusLocal.vrfName
         self.configFilename += "-%s" % self.vrfStatusLocal.vrfName
         self.knownHostsFilename += "-%s" % self.vrfStatusLocal.vrfName
         self.sshConfigFilename += "-%s" % self.vrfStatusLocal.vrfName
         self.xinetdSshConfigFilename = "%s" % self.vrfStatusLocal.vrfName

      # Generate ssh keys for the first time, subsequent key regenerations will
      # be handled by different notifiees in SshHostKeys.py
      if not vrfStatusLocal:
         Tac.run( ['service', 'ssh-keygen'], ignoreReturnCode=True )

      XinetdService.__init__( self, serviceName, config, self.configFilename )
      if vrfStatusLocal:
         vrfConfig = self.config.vrfConfig.get( vrfStatusLocal.vrfName )
         vrfStatus = self.status.vrfStatus.get( vrfStatusLocal.vrfName )
         if vrfConfig and vrfStatus:
            self.handleTunnel( name=None,
                               sshVrfConfig=vrfConfig,
                               sshVrfStatus=vrfStatus )
            # The known_hosts file is entirely overwritten each time
            # This optimization should save on writes
            if len( vrfConfig.knownHost ) >= 1:
               host = vrfConfig.knownHost[ next( iter( vrfConfig.knownHost ) ) ]
               self.handleKnownHost( host, sshVrfConfig=vrfConfig )
      else:
         self.handleTunnel( name=None )

      # create directories and empty aggregate files for ssh certs
      SshCertLib.createSshDirs()
      self._aggregateFiles( SshCertFileType.caKey )
      self._aggregateFiles( SshCertFileType.revokeList )

      # declare reactor for X509 profile and call handler to possibly initialize it
      self.x509ProfileReactor = None
      self.handleX509ProfileName()

   @Tac.handler( 'tunnel' )
   def handleTunnel( self, name, forceTearDown=False, sshVrfConfig=None,
                     sshVrfStatus=None ):
      qt0( "handleTunnel entered with key", str( name ) )
      t0( "handleTunnel entered with key", str( name ) )
      if sshVrfConfig or sshVrfStatus:
         assert sshVrfConfig and sshVrfStatus, "These must be passed in together"
      holdingConfig = self.config
      holdingStatus = self.status
      if self.vrfStatusLocal and not forceTearDown:
         if sshVrfConfig:
            holdingConfig = sshVrfConfig
            holdingStatus = sshVrfStatus
         else:
            # The reactor fired for the main ssh config
            # and this is a VRF Service
            return
      if not name:
         # Deferred Notification handling
         for tunnelName in holdingStatus.tunnel:
            if tunnelName not in holdingConfig.tunnel:
               # The tunnel has gone away and should be deleted
               self.handleTunnel( tunnelName, forceTearDown=True,
                                  sshVrfConfig=sshVrfConfig,
                                  sshVrfStatus=sshVrfStatus )
            elif name not in self.sshTunnels_:
               # The tunnel instance does not exist and needs to be created.
               self.handleTunnel( tunnelName,
                                  sshVrfConfig=sshVrfConfig,
                                  sshVrfStatus=sshVrfStatus )
         for tunnelName in holdingConfig.tunnel:
            if tunnelName not in holdingStatus.tunnel:
               # The tunnel does not exist yet
               self.handleTunnel( tunnelName,
                                  sshVrfConfig=sshVrfConfig,
                                  sshVrfStatus=sshVrfStatus )
         return
      tunnelConfig = holdingConfig.tunnel.get( name )
      if tunnelConfig and not forceTearDown:
         # tunnel status creation is idempotent - this won't
         # affect already created statuses
         tunnelStatus = holdingStatus.tunnel.newMember( name )
         if name in self.sshTunnels_:
            t0( "SSH Tunnel manager for %s already exists, doing nothing" % name )
            return
         qt0( "Making new SSH tunnel SuperServer instance:", name )
         t0( "Making new SSH tunnel SuperServer instance:", name )
         sshTunnel = SshTunnel( tunnelConfig,
                                tunnelStatus,
                                self.config,
                                self.knownHostsFilename,
                                self.sshConfigFilename,
                                self.sshSuperServerAgent,
                                self.vrfStatusLocal )
         self.sshTunnels_[ name ] = sshTunnel
      else:
         qt0( "Tearing down SSH tunnel:", name )
         if name in self.sshTunnels_:
            tunnelService = self.sshTunnels_[ name ]
            tunnelService.stopService()
            unlinkFile( tunnelService.configFilename_ )
            unlinkFile( tunnelService.configFilename_ + ".save" )
            unlinkFile( tunnelService.configFilename_ + ".log" )
            unlinkFile( tunnelService.configFilename_ + ".pid" )
            del self.sshTunnels_[ name ]
         del holdingStatus.tunnel[ name ]

   @Tac.handler( 'x509ProfileName' )
   def handleX509ProfileName( self ):
      if ( not self.x509ProfileReactor or
           self.x509ProfileReactor.profileName != self.config.x509ProfileName ):
         if not self.config.x509ProfileName:
            self.x509ProfileReactor = None
         else:
            self.x509ProfileReactor = SysMgrX509SslReactor(
               self.sslStatus, self.config.x509ProfileName, self )

   def serverPortConfigName( self, vrf=None, v6=False ):
      return serviceConfigName( "ssh_serverport", vrf=vrf, v6=v6 )

   def _handleServerPort( self, vrf, sshConf=None ):
      t0( "_handleServerPort() vrf", vrf )
      configName = self.serverPortConfigName( vrf=vrf )
      configNameV6 = self.serverPortConfigName( vrf=vrf, v6=True )

      serviceNameBase = 'ssh_serverport'

      def _writeXinetdConfig( vrf=None, v6=False ):
         portConf = _getServerPortXinetdConfig( serviceNameBase,
                                                port=self.config.serverPort,
                                                vrf=vrf,
                                                v6=v6 )
         configFile = "/etc/xinetd.d/%s" % ( configNameV6 if v6 else
                                             configName )
         if not SuperServer.LinuxService.writeConfigFile( self, configFile,
                                                          portConf,
                                                          saveBackupConfig=False ):
            self.sync()

      def _createServerPortConfig( vrf=None ):
         sshdConf = _getServerPortSshdConfig( port=self.config.serverPort,
                                              vrf=vrf,
                                              sshdConf=sshConf )
         if not SuperServer.LinuxService.writeConfigFile( self,
                                                          "/etc/ssh/%s" % configName,
                                                          sshdConf,
                                                          saveBackupConfig=False ):
            self.sync()

      if self.config.serverPort == 22:
         unlinkFile( "/etc/xinetd.d/%s" % configName )
         unlinkFile( "/etc/xinetd.d/%s" % configNameV6 )
         unlinkFile( "/etc/ssh/%s" % configName )
      else:
         _writeXinetdConfig( vrf=vrf )
         _writeXinetdConfig( vrf=vrf, v6=True )
         _createServerPortConfig( vrf=vrf )

      # Xinetd will require reloading the config file
      XinetdService.scheduleReloadService()

   @Tac.handler( 'serverPort' )
   def handleServerPort( self ):
      # default VRF
      self._handleServerPort( None )

      # if self.config.vrfConfig is not None:
      for vrfName, vrf in self.sshSuperServerAgent.allVrfStatusLocal.vrf.items():
         if vrf.state == 'active':
            self._handleServerPort( vrfName )

   @Tac.handler( 'knownHost' )
   def handleKnownHost( self, name, sshVrfConfig=None ):
      # We handle the known_hosts file by overwriting everything
      # each time
      t5( "SshService::handleKnownHost(", name, ")" )
      holdingConfig = self.config
      if self.vrfStatusLocal:
         if sshVrfConfig:
            holdingConfig = sshVrfConfig
         else:
            # The reactor fired for the main sshd config
            # and this is a VRF service
            return
      t8( "SshService::handleKnownHost : fipsRestrictions is",
          self.config.fipsRestrictions )
      config = ''
      for host in holdingConfig.knownHost:
         knownHostEntry = holdingConfig.knownHost[ host ]
         # Convert Tac type to SSH algorithm name
         hostEntryKey = SysMgrLib.tacKeyTypeToCliKey[ knownHostEntry.type ]
         if ( not self.config.fipsRestrictions ) or \
            ( self.config.fipsRestrictions and hostEntryKey in FIPS_HOSTKEYS ):
            t8( "SshService::handleKnownHost accepting",
                 knownHostEntry.type, "key for \"", knownHostEntry.host, "\"" )
            # Rename some for how openssh stores the algorithm
            keyEntryType = HOSTKEYS_MAPPINGS[ hostEntryKey ]
            config += '%s %s %s\n' % ( knownHostEntry.host, keyEntryType,
                  knownHostEntry.publicKey )
      # We should not use function 'knownHostsFileName' overridden by XinetdService,
      # but the one defined by LinuxService
      if not SuperServer.LinuxService.writeConfigFile( self,
                                                       self.knownHostsFilename,
                                                       config ):
         self.sync()

   def entropySourceOk( self ):
      # makes it easier to access
      return self.sshSuperServerAgent.mgmtSecurityMgr_.entropySourceOk()

   def serviceProcessWarm( self ):
      return XinetdService._serviceProcessWarm() and self.entropySourceOk()

   def serviceEnabled( self ):
      holdingStatus = self.status
      if self.vrfStatusLocal and \
         self.vrfStatusLocal.vrfName in self.status.vrfStatus:
         holdingStatus = self.status.vrfStatus[ self.vrfStatusLocal.vrfName ]
      if not self.entropySourceOk():
         holdingStatus.enabled = False
         return holdingStatus.enabled
      vrfActive = True
      enabled = self.config.serverState == "enabled"
      if self.vrfStatusLocal:
         vrfActive = ( self.vrfStatusLocal.state == "active" )
         vrfName = self.vrfStatusLocal.vrfName
         vrfConfig = self.config.vrfConfig
         if vrfName in vrfConfig and \
            vrfConfig[ vrfName ].serverState != "globalDefault":
            enabled = vrfConfig[ vrfName ].serverState == "enabled"
      else:
         vrfConfig = self.config.vrfConfig
         if DEFAULT_VRF in vrfConfig and \
            vrfConfig[ DEFAULT_VRF ].serverState != "globalDefault":
            enabled = vrfConfig[ DEFAULT_VRF ].serverState == "enabled"
      holdingStatus.enabled = enabled and vrfActive
      return holdingStatus.enabled

   def processNamedSshKeys( self, hostKeys ):
      """
         hostKeys is a list of named and default SSH keys. Process
         only named keys which have the format
         /persist/secure/ssh/keys/<algo>/<filename>
         1. Copy the private/public key to /etc/ssh/ssh_keys directory
            if the private key is present.
         2. Cleanup keys in /etc/ssh/ssh_keys directory which are not in config
      """
      namedKeys = []
      keyDir = SshConstants.sshKeyConfigDir
      keysToDelete = { os.path.join( keyDir, f ) for f in os.listdir( keyDir ) }
      t8( "Keys present before processing new config:", keysToDelete )

      for key in hostKeys:
         t8( "Processing key: ", key )

         # Ignore default keys which are named as their algorithms
         # Eg: dsa, rsa, ecdsa-nistp256, ecdsa-nistp521, ed25519
         if key in SshCertLib.algoDirToSshAlgo:
            continue

         # Get the SSH config key path of the form
         # /etc/ssh/ssh_keys/<algo>-<filename>
         destKeyPath = getSshConfigKeyPath( key )

         # Add the key to config
         namedKeys.append( destKeyPath )

         # Check if the file is present.
         if not os.path.isfile( key ):
            t0( "SSH private key not present on the switch:", key )
            continue

         # Copy the keys to the config directory
         copyNamedSshKeys( key, destKeyPath )

         # We keep track of the files when we start and cleanup
         # files which are no longer in the config.
         # Ensure we don't delete keys present in config
         keysToDelete.discard( destKeyPath )
         keysToDelete.discard( destKeyPath + ".pub" )

      # Cleanup old key files
      t8( "Keys to delete: ", keysToDelete )
      for key in keysToDelete:
         deleteKeyPath = os.path.join( keyDir, key )
         unlinkFile( deleteKeyPath )

      return namedKeys

   def getOpensshAuthenMethodString( self, authenMethods ):
      return ",".join( [ opensshMethodMap[ opt ] for opt in authenMethods ] )

   def conf( self ):
      t0( "calling conf()", self.serviceName_ )
      # call other config file updaters
      self.handleKnownHost( '' )
      if not writeSshConfig( self.config, self.sshConfigFilename,
                             self.knownHostsFilename, self ):
         self.sync()
      if not writeXinetdSshConfig( self ):
         self.sync()
      for tunnel in self.sshTunnels_.values():
         tunnel.sync()

      # sha1 signatures are disabled by default enable them for compatiblity
      conf = \
'''Protocol 2
Port 22
SyslogFacility AUTHPRIV
UsePAM yes
MaxStartups 10:30:100
Subsystem sftp /usr/libexec/openssh/sftp-server
AllowTcpForwarding no
Banner /etc/issue
StrictModes yes
Compression delayed
GatewayPorts no
GSSAPIAuthentication no
KerberosAuthentication no
PermitRootLogin yes
CASignatureAlgorithms = +ssh-rsa
'''

      methodStringArray = []
      passwordEnabled = False
      keyboardInteractiveEnabled = False
      publicKeyEnabled = False

      multiFactorEnabled = False

      for methodList in self.config.authenticationMethodList.values():
         methods = list( methodList.method.values() )
         methodStringArray.append( self.getOpensshAuthenMethodString( methods ) )

         if len( methods ) >= 2:
            multiFactorEnabled = True

         passwordEnabled = passwordEnabled \
            or AuthenMethod.password in methods
         publicKeyEnabled = publicKeyEnabled \
            or AuthenMethod.publicKey in methods
         keyboardInteractiveEnabled = keyboardInteractiveEnabled \
            or AuthenMethod.keyboardInteractive in methods

      if multiFactorEnabled:
         methodsString = " ".join( methodStringArray )
         t8( "Adding multi factor AuthenticationMethods to OpenSSH", methodsString )
         # Set AuthenticationMethods field with the required methods
         conf += "AuthenticationMethods %s\n" % methodsString

      passwordAuthen = "yes" if passwordEnabled else "no"
      challengeResponseAuthen = "yes" if keyboardInteractiveEnabled else "no"
      pubKeyAuthen = "yes" if publicKeyEnabled else "no"
      conf += ( "PasswordAuthentication %s\n"
                "ChallengeResponseAuthentication %s\n"
                "PubkeyAuthentication %s\n" %
                ( passwordAuthen, challengeResponseAuthen, pubKeyAuthen ) )

      if self.config.permitEmptyPasswords == "automatic":
         aaaLocalConfig = self.sshSuperServerAgent.aaaLocalConfig
         if aaaLocalConfig.allowRemoteLoginWithEmptyPassword:
            permitEmptyPasswords = "yes"
         else:
            permitEmptyPasswords = "no"
      elif self.config.permitEmptyPasswords == "permit":
         permitEmptyPasswords = "yes"
      else:
         permitEmptyPasswords = "no"

      conf += "PermitEmptyPasswords %s\n" % permitEmptyPasswords

      conf += "LoginGraceTime %d\n" % ( self.config.successfulLoginTimeout.timeout, )
      conf += "LogLevel %s\n" % ( self.config.logLevel.upper(), )

      if self.config.verifyDns:
         useDNS = "yes"
      else:
         useDNS = "no"

      conf += "UseDNS %s\n" % ( useDNS, )

      filteredCiphers, filteredMacs, \
            filteredKex, filteredHostKey = filterSshOptions( self.config )

      if self.sshSuperServerAgent.netconf.enabled:
         for ep in self.sshSuperServerAgent.netconf.endpoints.values():
            if ep.enabled:
               if ( ( ep.vrfName in ( "default", "" ) ) or
                     ( self.vrfStatusLocal and
                        ep.vrfName == self.vrfStatusLocal.vrfName ) ):
                  conf += "Subsystem netconf netconf start-client\n"

      rekeyStr = deriveRekeyStr( self.config )
      conf += "RekeyLimit %s\n" % rekeyStr

      conf += "ClientAliveInterval %d\n" % self.config.clientAliveInterval
      conf += "ClientAliveCountMax %d\n" % self.config.clientAliveCountMax

      if self.config.fipsRestrictions:
         conf += "UseFipsAlgorithms yes\n"

      allHostkeysAlgs = ( filteredHostKey +
         list( map( SshCertLib.getAlgoFromKeyPath, filteredHostKey ) ) )
      if "dsa" in allHostkeysAlgs:
         conf += "PubkeyAcceptedKeyTypes = +ssh-dss\n"
         conf += "HostKeyAlgorithms = +ssh-dss\n"

      x509ProfileName = self.config.x509ProfileName
      x509Profile = self.sslStatus.profileStatus.get( x509ProfileName )
      if x509Profile and x509Profile.state == SslProfileState.valid:
         conf += ( "PubkeyAcceptedKeyTypes = +x509v3-ecdsa-sha2-nistp256,"
                   "x509v3-ecdsa-sha2-nistp384,x509v3-ecdsa-sha2-nistp521,"
                   "x509v3-rsa2048-sha256\n" )
         if x509Profile.trustedCertsPath:
            conf += f"X509CACertificateFile {x509Profile.trustedCertsPath}\n"
         if x509Profile.crlsPath:
            conf += f"X509CRLFile {x509Profile.crlsPath}\n"
         omitDomain = "yes" if self.config.x509UsernameDomainOmit else "no"
         conf += f"X509UsernameOmitDomain {omitDomain}\n"

         ocspSettings = x509Profile.ocspSettings
         if ocspSettings:
            if ocspSettings.certRequirement == OcspCertRequirement.all:
               conf += "OCSPCheck=chain\n"
            elif ocspSettings.certRequirement == OcspCertRequirement.leaf:
               conf += "OCSPCheck=leaf\n"
            elif ocspSettings.certRequirement == OcspCertRequirement.none:
               conf += "OCSPCheck=chain-where-responder\n"

            conf += f"OCSPNonce={ocspSettings.nonce}\n"
            conf += f"OCSPRequestTimeout {ocspSettings.timeout}\n"
            if ocspSettings.url:
               conf += f"OverridingOCSPResponder={ocspSettings.url}\n"
            if ( MgmtSecurityToggleLib.toggleOcspProfileVrfEnabled()
                 and ocspSettings.vrfName != "default" ):
               conf += ( "OCSPNetworkNamespace="
                         f"{vrfNamespace(ocspSettings.vrfName)}\n" )
         else:
            conf += "OCSPCheck=none\n"

      if filteredCiphers:
         conf += "Ciphers %s\n" % ",".join( filteredCiphers )

      if filteredKex:
         conf += "KexAlgorithms %s\n" % ",".join( filteredKex )

      if filteredMacs:
         conf += "MACs %s\n" % ",".join( filteredMacs )

      if filteredHostKey:
         if "rsa" in filteredHostKey:
            conf += "HostKey %s\n" % SysMgrLib.rsaKeyPath
         if "dsa" in filteredHostKey:
            conf += "HostKey %s\n" % SysMgrLib.dsaKeyPath
         if "ecdsa-nistp521" in filteredHostKey:
            conf += "HostKey %s\n" % SysMgrLib.ecdsa521KeyPath
         if "ecdsa-nistp256" in filteredHostKey:
            conf += "HostKey %s\n" % SysMgrLib.ecdsa256KeyPath
         if "ed25519" in filteredHostKey:
            conf += "HostKey %s\n" % SysMgrLib.ed25519KeyPath

      # Apply named SSH keys to config
      namedSshKeys = self.processNamedSshKeys( filteredHostKey )
      t8( "Named SSH keys added to config: ", namedSshKeys )
      for key in namedSshKeys:
         conf += "HostKey %s\n" % key

      # Check host cert key types
      hostCertKeyTypes = SshCertLib.hostCertsByKeyTypes( self.config.hostCertFiles )
      for keyType, certs in hostCertKeyTypes.items():
         if len( certs ) > 1 or keyType == KeyType.invalid:
            Logging.log( SECURITY_SSH_CERT_FILE_WARNING, ", ".join( certs ) )

      for cert in self.config.hostCertFiles:
         conf += "HostCertificate %s\n" % SshConstants.hostCertPath( cert )

      try:
         allCaKeyFile = self._aggregateFiles( SshCertFileType.caKey )
         allRevokedKeyFile = self._aggregateFiles( SshCertFileType.revokeList )
         if self.config.caKeyFiles:
            conf += "TrustedUserCAKeys %s\n" % allCaKeyFile

            # Unconditionally enable authorized principals file when
            # TrustedUserCAKeys is set
            conf += "AuthorizedPrincipalsFile %h/.ssh/principals\n"

            authPrincipalsCmdExecPath = "/etc/ssh/auth_principals_cmd"
            if self.config.authPrincipalsCmdFile:
               # Persist path
               authPrincipalsCmdPath = SshConstants.authPrincipalsCmdPath(
                  self.config.authPrincipalsCmdFile )
               # /usr/sbin path
               localAuthPrincipalsCmdExecPath = os.path.join( "/", "usr",
                        "sbin", self.config.authPrincipalsCmdFile )
               if os.path.isfile( authPrincipalsCmdPath ):
                  # Avoid copying if the desired file is present in exec path
                  if not ( os.path.isfile( authPrincipalsCmdExecPath ) and
                           filecmp.cmp( authPrincipalsCmdPath,
                           authPrincipalsCmdExecPath ) ):
                     # Cleanup old files or symlinks before copying
                     unlinkFile( authPrincipalsCmdExecPath )
                     shutil.copyfile( authPrincipalsCmdPath,
                           authPrincipalsCmdExecPath )
                     os.chmod( authPrincipalsCmdExecPath, 0o755 )
               elif os.path.isfile( localAuthPrincipalsCmdExecPath ):
                  # Avoid symlinking if the desired file is present in exec path
                  if not ( os.path.isfile( authPrincipalsCmdExecPath ) and
                           filecmp.cmp( localAuthPrincipalsCmdExecPath,
                           authPrincipalsCmdExecPath ) ):
                     # Cleanup old files or symlinks before copying
                     unlinkFile( authPrincipalsCmdExecPath )
                     # Create a symlink so that the latest executable from
                     # EOS is used
                     os.symlink( localAuthPrincipalsCmdExecPath,
                           authPrincipalsCmdExecPath )
               else:
                  # None of the files are present delete the file in exec path
                  unlinkFile( authPrincipalsCmdExecPath )

               conf += "AuthorizedPrincipalsCommandUser root\n"
               conf += ( "AuthorizedPrincipalsCommand %s %s\n" %
                         ( authPrincipalsCmdExecPath,
                           self.config.authPrincipalsCmdArgs ) )
            else:
               # Remove the executable if authorized principals command is
               # disabled
               unlinkFile( authPrincipalsCmdExecPath )

         if self.config.revokedUserKeysFiles:
            conf += "RevokedKeys %s\n" % allRevokedKeyFile
      except OSError as e:
         if e.errno == errno.ENOSPC:
            Logging.log( SuperServer.SYS_SERVICE_FILESYSTEM_FULL,
                         "aggregate SSH cert/key files", self.serviceName_ )

      self.status.sshCertFileUpdateProcessed = Tac.now()

      if self.vrfStatusLocal:
         vrfName = self.vrfStatusLocal.vrfName
         conf += "PidFile /var/run/sshd-%s.pid\n" % self.vrfStatusLocal.vrfName
      else:
         vrfName = None

      # create server port config if any
      self._handleServerPort( vrfName, sshConf=conf )

      for userName, userConfig in self.config.user.items():
         conf += "Match User %s\n" % userName
         conf += "  AllowTcpForwarding %s\n" % userConfig.userTcpForwarding

      return conf

   def _aggregateFiles( self, fileType ):
      if fileType == SshCertFileType.caKey:
         getFullPath = SshConstants.caKeyPath
         filenameAll = SshConstants.allCaKeysFile
         configFilenames = self.config.caKeyFiles

      elif fileType == SshCertFileType.revokeList:
         getFullPath = SshConstants.revokeListPath
         filenameAll = SshConstants.allRevokeKeysFile
         configFilenames = self.config.revokedUserKeysFiles
      else:
         assert False, "Unsupported file type"

      # pylint: disable-next=consider-using-with
      allKeyFile = open( getFullPath( filenameAll ), 'w' )
      allKeyFile.write( SuperServer.configFileHeader )

      for filename in configFilenames:
         try:
            keyPath = getFullPath( filename )
            SshCertLib.validateMultipleKeysFile( keyPath )
            with open( keyPath ) as keyFile:
               allKeyFile.write( keyFile.read() + "\n" )
         except OSError as e:
            if e.errno == errno.ENOSPC:
               Logging.log( SuperServer.SYS_SERVICE_FILESYSTEM_FULL,
                            allKeyFile.name, self.serviceName_ )
            if e.errno == errno.ENOENT:
               Logging.log( SECURITY_SSH_CERT_FILE_WARNING, filename )
         except SshCertLib.SshKeyError:
            Logging.log( SECURITY_SSH_CERT_FILE_WARNING, filename )
      allKeyFile.write( SuperServer.configFileFooter )
      allKeyFile.close()
      return allKeyFile.name

def serviceConfigName( serviceName, vrf=None, v6=False ):
   name = serviceName
   if vrf:
      name += "-%s" % vrf
   if v6:
      name += "-6"
   return name

def _getServerPortSshdConfig( port=None, vrf=None, sshdConf=None ):
   if sshdConf is not None:
      config = sshdConf
   else:
      # pylint: disable-next=consider-using-with
      config = open( "/etc/ssh/sshd_config" ).read()
   if port:
      config = re.sub( r'Port \d+', 'Port ' + str( port ), config )
   if vrf:
      config += "PidFile /var/run/sshd-%s.pid\n" % vrf
   return config

def _getServerPortXinetdConfig( serviceNameBase, port=None, vrf=None, v6=False ):
   """ Returns a SSH server port configuration file (string) and its
   filename (string) for xinetd in a tuple
   """
   serviceId = serviceNameBase
   if vrf:
      serviceId += '-' + vrf
   configFilename = "/etc/ssh/" + serviceId
   if v6:
      serviceId += '-6'
   ipv4String = '' if v6 else 'IPv4'
   v6String = 'yes' if v6 else 'no'
   serverPortConf = '''
service {sName}
{{
        id              = {sId}
        type            = UNLISTED
        disable         = no
        flags           = REUSE {ipv4}
        v6only          = {ipv6}
        socket_type     = stream
        wait            = no
        user            = root
        port            = {portNum}
        protocol        = tcp
        server          = /usr/sbin/sshd
        server_args     = -i -f {confFile}
        namespace       = {namespace}
        log_on_failure  += USERID
        instances       = 50
        per_source      = 20
}}
'''.format( sName=serviceId, sId=serviceId, ipv4=ipv4String, ipv6=v6String,
            portNum=port, confFile=configFilename,
            namespace=netnsNameWithUniqueId( vrfNamespace( vrf ) ) )
   return serverPortConf

def _getSshRedirectConfig( serviceNameBase, port=None, vrf=None, v6=False ):
   """ Returns a SSH redirect configuration file (string) and its filename (string)
   for xinetd in a tuple
   """
   serviceId = serviceNameBase
   if vrf:
      serviceId += '-' + vrf
   serviceId += '-%s' % port
   if v6:
      serviceId += '-6'
   ipv4String = '' if v6 else 'IPv4'
   v6String = 'yes' if v6 else 'no'
   redirectConf = '''
service %s
{
        type            = UNLISTED
        disable         = no
        socket_type     = stream
        flags           = REUSE %s
        v6only          = %s
        wait            = no
        protocol        = tcp
        user            = root
        port            = %s
        redirect        = localhost 22
        namespace       = %s
        log_on_failure  += USERID
        instances       = 50
        per_source      = 20
}
''' % ( serviceId, ipv4String, v6String, port,
        netnsNameWithUniqueId( vrfNamespace( vrf ) ) )
   return redirectConf

class FileEventHandler( pyinotify.ProcessEvent ):
   def __init__( self, fileReactor ):
      self.fileReactor_ = fileReactor
      pyinotify.ProcessEvent.__init__( self )

   def process_IN_MOVED_TO( self, event ):
      self.fileReactor_.handleFile( event.wd, event.name )

   # This is for creation/copying of new files to watch directory
   # Not using IN_CREATE since we want to wait until the file has
   # been written completely.
   def process_IN_CLOSE_WRITE( self, event ):
      self.fileReactor_.handleFile( event.wd, event.name )

   def process_IN_DELETE( self, event ):
      self.fileReactor_.handleFileDelete( event.wd, event.name )

class InotifyReactor( Tac.Notifiee, pyinotify.Notifier ):
   notifierTypeName = 'Tac::FileDescriptor'

   def __init__( self, watch_manager, default_proc_fun=None, read_freq=0,
                threshold=0, timeout=None ):
      pyinotify.Notifier.__init__( self, watch_manager, default_proc_fun,
                                   read_freq, threshold, timeout )
      fileDesc_ = Tac.newInstance( 'Tac::FileDescriptor', "ssh" )
      fileDesc_.descriptor = watch_manager.get_fd()
      fileDesc_.nonBlocking = True
      Tac.Notifiee.__init__( self, fileDesc_ )

   @Tac.handler( 'readableCount' )
   def handleReadableCount( self ):
      self.read_events()
      self.process_events()

class FileReactor:
   def __init__( self, sshConfig ):
      self.config_ = sshConfig
      wm = pyinotify.WatchManager()
      # pylint: disable-msg=E1101
      mask = pyinotify.IN_MOVED_TO | pyinotify.IN_DELETE | pyinotify.IN_CLOSE_WRITE
      self.inotifyReactor = InotifyReactor( wm, FileEventHandler( self ) )
      # Has the full path the file being modified
      self.wdDict = {}
      # Add all the algorithm specific key's directory to the watch list
      for algo in SshCertLib.algoDirToSshAlgo:
         dirPath = SshConstants.hostKeysDirPath() + algo
         # Add a file watcher for the directory
         wd = wm.add_watch( dirPath, mask )
         # Used to lookup the full path of the notification
         self.wdDict[ wd[ dirPath ] ] = dirPath

   def handleFile( self, wd, fileName ):
      if wd in self.wdDict:
         keyPath = os.path.join( self.wdDict[ wd ], fileName )
         # Get the files in config
         _, _, _, filteredHostKeys = filterSshOptions( self.config_ )
         # React to only files in config
         if keyPath in filteredHostKeys:
            t8( "Copy the public/private key pair:", fileName )
            destKeyPath = getSshConfigKeyPath( keyPath )
            copyNamedSshKeys( keyPath, destKeyPath )

   def handleFileDelete( self, wd, fileName ):
      if wd in self.wdDict:
         keyPath = os.path.join( self.wdDict[ wd ], fileName )
         # Get the files in config
         _, _, _, filteredHostKeys = filterSshOptions( self.config_ )
         # React to only files in config
         if keyPath in filteredHostKeys:
            t8( "Delete the public/private key pair:", fileName )
            destKeyPath = getSshConfigKeyPath( keyPath )
            # Delete both the private and public keys
            unlinkFile( destKeyPath )
            unlinkFile( destKeyPath + ".pub" )

class Ssh( SuperServer.SuperServerAgent ):

   def cleanupStaleVrfs( self ):
      '''Find out which files/processes refer to VRFs that have been removed or are
         inactive and clean them up.
         PAM files are case-insensitive hence can't be used for this purpose, so
         if for some reason Ssh comes up with a stale /etc/pam.d/sshd-foo around
         (but none of the init and config files or instances) this file won't get
         cleaned up. It will not cause any side-effect though.
         '''
      def addVrfIfInactive( vrfSet, vrf, allVrfStatusLocal ):
         try:
            if allVrfStatusLocal.vrf[ vrf ].state != 'active':
               vrfSet.add( vrf )
         except KeyError:
            vrfSet.add( vrf ) # VRF has been removed
      removedVrfs = set()
      # init.d scripts
      for fileName in os.listdir( '/etc/init.d' ):
         m = re.match( '^sshd-.*', fileName )
         if m:
            try:
               addVrfIfInactive( removedVrfs, fileName[ 5: ],
                                 self.allVrfStatusLocal )
            except OSError as e:
               if e.errno == errno.ENOSPC:
                  Logging.log( SuperServer.SYS_SERVICE_FILESYSTEM_FULL,
                               fileName, self.service_.serviceName_ )
                  return
               else:
                  raise
      # sshd_config files
      for fileName in os.listdir( '/etc/ssh' ):
         m = re.match( '^sshd_config-.*', fileName )
         if m:
            try:
               addVrfIfInactive( removedVrfs, fileName[ 12: ],
                                 self.allVrfStatusLocal )
            except OSError as e:
               if e.errno == errno.ENOSPC:
                  Logging.log( SuperServer.SYS_SERVICE_FILESYSTEM_FULL,
                               fileName, self.service_.serviceName_ )
                  return
               else:
                  raise
      # sshd binaries
      for fileName in os.listdir( '/usr/sbin' ):
         m = re.match( '^sshd-.*', fileName )
         if m and m.group(0) != "sshd-keygen":
            addVrfIfInactive( removedVrfs, fileName[ 5: ], self.allVrfStatusLocal )
      # sshd processes
      for process in Tac.run( [ '/bin/ps', '-u', '0', '-o', 'comm' ],
                              stdout=Tac.CAPTURE ).split( '\n' ):
         if process.startswith( 'sshd-' ):
            processVrf = process.split( 'sshd-' )[ 1 ]
            addVrfIfInactive( removedVrfs, processVrf, self.allVrfStatusLocal )
      for vrfName in removedVrfs:
         cleanupSshd( vrfName, self.allVrfStatusLocal )

   def __init__( self, entityManager ):
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      config = mg.mount( 'mgmt/ssh/config', 'Mgmt::Ssh::Config', 'r' )
      status = mg.mount( Cell.path( 'mgmt/ssh/status' ), 'Mgmt::Ssh::Status', 'fw' )
      sslStatus = mg.mount( 'mgmt/security/ssl/status',
                            'Mgmt::Security::Ssl::Status', 'r' )
      mgmtSecConfig = mg.mount( 'mgmt/security/config',
                                'Mgmt::Security::Config', 'r' )
      mgmtSecStatus = mg.mount( Cell.path( 'mgmt/security/status' ),
                                'Mgmt::Security::Status', 'r' )
      self.aaaLocalConfig = mg.mount( 'security/aaa/local/config',
                                      'LocalUser::Config', 'r' )
      self.allVrfStatusLocal = mg.mount( Cell.path( 'ip/vrf/status/local' ),
                                         'Ip::AllVrfStatusLocal', 'r' )
      self.netconf = mg.mount( 'mgmt/netconf/config',
                               'Netconf::Config', 'r' )
      self.configReq = mg.mount( 'mgmt/ssh/configReq', 'Mgmt::Ssh::ConfigReq', 'r' )

      self.allVrfStatusLocalReactor_ = None  # forward dec for pylint
      self.vrfConfigCollReactor_ = None  # forward dec for pylint
      self.netconfReactor_ = None  # forward dec for pylint
      self.netconfEndpointReactor_ = None  # forward dec for pylint
      self.service_ = None
      self.sshFileReactor_ = None # forward dec for pylint
      self.vrfServices_ = {}
      self.config_ = config
      self.status_ = status
      self.sslStatus_ = sslStatus
      self.mgmtSecurityMgr_ = None  # forward dec for pylint
      self.aaaConfigReactor_ = None # forward dec for pylint
      self.configReqReactor_ = None
      self.sshUserConfigReactor_ = None
      self.sshAuthenMethodListReactor_ = None

      def _finish():
         self.service_ = SshService( config, status, weakref.proxy( self ),
                                     sslStatus )
         self.sshFileReactor_ = FileReactor( config )
         self.cleanupStaleVrfs()
         self.allVrfStatusLocalReactor_ = Tac.collectionChangeReactor(
            self.allVrfStatusLocal.vrf, VrfStatusLocalReactor,
            reactorArgs=( self, self.allVrfStatusLocal ) )
         self.vrfConfigCollReactor_ = Tac.collectionChangeReactor(
            config.vrfConfig, VrfConfigReactor,
            reactorArgs=( self, ) )
         self.netconfEndpointReactor_ = Tac.collectionChangeReactor(
               self.netconf.endpoints, NetconfEndpointReactor,
               reactorArgs=( self, ) )
         self.netconfReactor_ = NetconfReactor( self.netconf, self.service_ )
         self.mgmtSecurityMgr_ = SshMgmtSecurityMgr(
            mgmtSecConfig, mgmtSecStatus, self )
         self.aaaConfigReactor_ = AaaConfigReactor( self.aaaLocalConfig, self )
         self.configReqReactor_ = ConfigReqReactor( self.configReq, self.service_ )
         self.sshUserConfigReactor_ = Tac.collectionChangeReactor(
               config.user, SshUserConfigReactor, reactorArgs=( config,
                  self.aaaLocalConfig, self, ) )
         self.sshAuthenMethodListReactor_ = Tac.collectionChangeReactor(
               config.authenticationMethodList, SshAuthenMethodListReactor,
               reactorArgs=( self, ) )

      mg.close( _finish )

   def warm( self ):
      if self.service_ is None:
         return False
      if not self.service_.warm():
         return False
      return all( x.warm() for x in self.vrfServices_.values() )

   def onSwitchover( self, protocol ):
      qt0( 'Ssh switchover handler' )
      if self.service_:
         for tunnel in self.service_.sshTunnels_.values():
            tunnel.onNetworkDisruption()
      for vrf in self.vrfServices_.values():
         for tunnel in vrf.sshTunnels_.values():
            tunnel.onNetworkDisruption()

def Plugin( ctx ):
   ctx.registerService( Ssh( ctx.entityManager ) )
