#!/usr/bin/env python3
# Copyright (c) 2010-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import os, re, shutil, tempfile

import Tac, Logging, SuperServer, QuickTrace
import Arnet
import Cell
import Ethernet
import SysMgrLib
import Tracing

# To create explicit dependencies on Controllerdb and ManagementActive
# pkgdeps: rpm Controllerdb-lib
# pkgdeps: rpm ManagementActive-lib

t0 = Tracing.trace0
t3 = Tracing.trace3
t8 = Tracing.trace8
qv = QuickTrace.Var
qt0 = QuickTrace.trace0

tmpFilePath = '/tmp/key_'

SYS_SSH_HOST_KEY_REGENERATED = Logging.LogHandle(
              "SYS_SSH_HOST_KEY_REGENERATED",
              severity=Logging.logInfo,
              fmt="SSH host keys associated with a different system (%s). "
              "Regenerating SSH host keys.",
              explanation="Software has detected that the SSH host keys on this "
              "system are associated with the mac address of a different system. "
              "This can happen in a modular system when a supervisor is moved to a "
              "different chassis. New SSH host keys have been generated for the "
              "system.",
              recommendedAction=Logging.NO_ACTION_REQUIRED,
              minInterval=10*60 )
SYS_SSH_HOST_KEY_UPDATED = Logging.LogHandle(
              "SYS_SSH_HOST_KEY_UPDATED",
              severity=Logging.logInfo,
              fmt="SSH host key %s has been updated. "
              "The %s hash of public key %s fingerprint is %s.",
              explanation="Software has detected that the SSH host keys on this "
              "system has been updated. This can happen when a modular system "
              "supervisor is moved to a different chassis, a user requests to "
              "reset the host key, the host key file is being created for the "
              "first time, or the host key file is somehow deleted.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

def splitKeyComment( keyText ):
   """ Returns the key part of the key text and the comment part of
   the key text. If the text doesn't look like a key then two empty
   strings are returned.
   """

   keyPattern = r"(\S* \S* )(.*)"
   match = re.match( keyPattern, keyText )
   return match.groups() if match else ( "", "" )

def getComponentsFromComment( comment ):
   """
   Extracts device info from the key comment and returns it.
   Currently returns origComment ( minus extracts ), chassisAddr, cvxAddr
   """
   t8( "SshHostKeys::getComponentsFromComment: ", comment )
   # The '$' are removed since the original regex assumed the mac addr
   # ends the line
   modifiedMacAddrPattern = Ethernet.macAddrPattern.replace( "$", "" )
   cvxAddrPattern = r"cvxAddr=" + modifiedMacAddrPattern
   sysMacPattern = r"chassisAddr=" + modifiedMacAddrPattern

   cvxAddrExtract = ""
   sysMacExtract = ""
   origComment = comment

   cvxAddrMatch = re.search( cvxAddrPattern, comment )
   if cvxAddrMatch:
      cvxAddrExtract = cvxAddrMatch.group( 1 )
      origComment = origComment.replace( cvxAddrMatch.group( 0 ), "" )
   sysMacMatch = re.search( sysMacPattern, comment )
   if sysMacMatch:
      sysMacExtract = sysMacMatch.group( 1 )
      origComment = origComment.replace( sysMacMatch.group( 0 ), "" )

   # Remove trailing and leading whitespace
   origComment = origComment.strip()
   # Normalize addresses
   sysMacExtract = sysMacExtract.lower()
   cvxAddrExtract = cvxAddrExtract.lower()
   
   return origComment, sysMacExtract, cvxAddrExtract

class KeyResetHandler():
   """
   Responds to key update requests and performs checks on if
   a reset is needed as well as support functions for doing
   the reset.
   """

   def __init__( self, sshConfig, sshConfigReq, sshStatus,
                 entityMibStatus, clusterStatusDir,
                 agent ):
      self.sshConfig = sshConfig
      self.sshConfigReq = sshConfigReq
      self.sshStatus = sshStatus
      self.entityMibStatus = entityMibStatus
      self.clusterStatusDir = clusterStatusDir
      self.agent = agent

      self.maxKeyGenRetries = 5
      self.keyGenWait = 60 * 2
      self.keyResetHandlerWait = 5

      self.keygenOldSysMacAddr = {}
      self.keygenOldCvxAddr = {}
      self.keygenProc = {}
      self.keyGenTimeout = {}
      self.keyGenRetryCount = {}
      self.keyGenResetHandler = {}
      self.cvxMacAddress = ""
      self.keyPath = {}
      self.keyParams = {}
      for keyAlgo, path in SysMgrLib.keyTypeToPath.items():
         self.keygenOldSysMacAddr[ keyAlgo ] = None
         self.keygenOldCvxAddr[ keyAlgo ] = None
         self.keygenProc[ keyAlgo ] = None
         self.keyGenTimeout[ keyAlgo ] = Tac.beginningOfTime
         self.keyGenRetryCount[ keyAlgo ] = 0
         self.keyGenResetHandler[ keyAlgo ] = Tac.ClockNotifiee( 
               lambda keyAlgo=keyAlgo: self.handleKeyGenReset( keyAlgo ) )
         self.keyGenResetHandler[ keyAlgo ].timeMin = Tac.endOfTime
         if keyAlgo not in self.sshStatus.sshKeyResetProcessed:
            self.sshStatus.sshKeyResetProcessed[ keyAlgo ] = Tac.beginningOfTime
         self.sshStatus.sshKeygenInProgress[ keyAlgo ] = False
         # Set default path
         self.keyPath[ keyAlgo ] = path
         # Set default key param : # default to the first param in the allowlist if
         # this key type expects key parameters
         legalKeyParams = SysMgrLib.keyTypeToLegalParams.get( keyAlgo, {} )
         defaultKeyParams = { legalParamName: legalParamValues[ 0 ] for
                              legalParamName, legalParamValues in
                              legalKeyParams.items() }
         self.keyParams[ keyAlgo ] = defaultKeyParams

   def isCvxMaster( self ):
      """
      Returns True IFF CVX is enabled and this is the master
      controller.
      """
      cvxEnabled = self.clusterStatusDir.status[ "default" ].enabled
      cvxLeader = self.clusterStatusDir.status[ "default" ].isLeader
      return cvxEnabled and cvxLeader

   def getPubKeyInfo( self, keyAlgo ):
      t8( "KeyResetHandler::getPubKeyInfo: %s" % keyAlgo )
      keyPath = self.keyPath[ keyAlgo ]
      pubKeyPath = keyPath + '.pub'
      origSystemMacAddr = ""
      origCvxAddr = ""
      origComment = ""
      origPubKey = ""
      if os.path.isfile( keyPath ) and os.path.isfile( pubKeyPath ):
         rawComment = ""
         with open( pubKeyPath ) as handle:
            origPubKey, rawComment = splitKeyComment( handle.readline() )
         origComment, origSystemMacAddr, origCvxAddr =\
            getComponentsFromComment( rawComment )
      
      return ( origPubKey, origSystemMacAddr, origCvxAddr, origComment )

   def isResetNeeded( self, keyAlgo ):
      t8( "KeyResetHandler::isResetNeeded: %s" % keyAlgo )

      if not self.agent.active():
         t3( "Agent is not active - returning False" )
         return False

      keyData = self.sshConfigReq.sshKeyResetRequest.get( keyAlgo )
      resetProcTime = self.sshStatus.sshKeyResetProcessed.get( keyAlgo )

      resetNeeded = False
      
      if keyData:
         assert resetProcTime, "resetProcTime should always exist"
         if keyData.timestamp > resetProcTime:
            t3( "Reset requested from command - returning True" )
            resetNeeded = True

      # Load the current key in
      origPubKey, origSystemMacAddr,\
      origCvxAddr, _ = self.getPubKeyInfo( keyAlgo )
      if not origPubKey:
         t0( "Keyfile not found. Reset needed to generate it." )
         resetNeeded = True

      # Get the current system mac address and CVX VIP address to
      # check if those have changed and a reset is needed.
      systemMacAddr = self.entityMibStatus.systemMacAddr
      sysMacChanged = origSystemMacAddr and ( origSystemMacAddr != systemMacAddr )
      if sysMacChanged:
         t3( "System MAC address changed - returning True" )
         t8( "Orig was ", origSystemMacAddr, " new is ", systemMacAddr )
         resetNeeded = True

      # If we are in Cvx Vip and the current key doesn't match the vmac set
      # and this is the master controller a reset is needed
      if( self.isCvxMaster() and origCvxAddr and\
          ( origCvxAddr != self.cvxMacAddress ) ):
         t3( "CVX VMAC address changed - returning True" )
         t8( "Orig was ", origCvxAddr, " new is ", self.cvxMacAddress )
         resetNeeded = True

      t0( "isResetNeeded will return: " + str( resetNeeded ) )
      return resetNeeded

   def generateExpectedComment( self, keyAlgo ):
      """
      Generate the expected comment on the ssh public key.
      """
      _, _, _, origComment = self.getPubKeyInfo( keyAlgo )
      systemMacAddr = self.entityMibStatus.systemMacAddr
      cvxMacAddress = ""
      if self.isCvxMaster():
         # Only set the CVX Mac Addr comment it is
         # the CVX VIP master
         cvxMacAddress = self.cvxMacAddress
      expectedKeyComment = SysMgrLib.generateNewSshKeyComment(
              systemMacAddr,
              origComment,
              cvxMacAddress )
      return expectedKeyComment

   def maybeUpdateComments( self, keyAlgo ):
      """
      Update comments IFF a reset is not needed
      and the comments are not up to date.
      """
      t8( "KeyResetHandler::maybeUpdateComments", keyAlgo )

      if self.isResetNeeded( keyAlgo ):
         t3( "Exiting since reset is needed" )
         return

      if self.sshStatus.sshKeygenInProgress[ keyAlgo ]:
         t3( "Exiting since keygen is in progress" )
         return

      origPubKey, origSystemMacAddr,\
      origCvxAddr, origComment = self.getPubKeyInfo( keyAlgo )
      assert origPubKey, "Public key should be available"

      expectedComment = self.generateExpectedComment( keyAlgo )
      fileComment = SysMgrLib.generateNewSshKeyComment( origSystemMacAddr,
                                                        origComment,
                                                        origCvxAddr )
      if expectedComment != fileComment:
         t0( "Comment was not as expected. Appending comment to pub key file." )
         pubKeyPath = self.keyPath[ keyAlgo ] + ".pub"
         # pylint: disable-next=consider-using-with
         tmpPubKeyFile = tempfile.NamedTemporaryFile()
         # Copy the permissions over since the temp file will
         # be made with 0o600
         shutil.copymode( pubKeyPath, tmpPubKeyFile.name )
         # The keygen is not in progress ( so this is not a race condition )
         # when updating the files
         with open( tmpPubKeyFile.name, "w" ) as handle:
            handle.write( origPubKey + " " + expectedComment + "\n" )
         shutil.copy( tmpPubKeyFile.name, pubKeyPath )

   def handleKeyGenReset( self, keyAlgo ):
      """
      Kicks off key reset if not already in progress, handles generation being
      complete, restarts if needed.
      """
      t8( "KeyResetHandler::handleKeyGenReset: " + keyAlgo )

      keyGenProc = self.keygenProc[ keyAlgo ]
      if keyGenProc:
         t3( "Checking if keygen was successful" )
         returnCode = keyGenProc.wait( block=False )
         if returnCode is None:
            t3( "keygen is still running" )
            if Tac.now() > self.keyGenTimeout[ keyAlgo ]:
               t0( "Keygen has timed out, killing it for retry" )
               keyGenProc.kill()
               keyGenProc.wait()
               self.keygenProc[ keyAlgo ] = None
            # If it hasn't timed out, check again soon. If it has,
            # retry in a moment.
            self.keyGenResetHandler[ keyAlgo ].timeMin = Tac.now() +\
                    self.keyResetHandlerWait
         else:
            if returnCode == 0:
               t0( "keygen process was successful" )
               self.keygenProc[ keyAlgo ] = None
               keyPath = self.keyPath[ keyAlgo ]
               keyPathPub = keyPath + ".pub"
               # First shred old keys
               SysMgrLib.shredSshKey( keyPath )
               # Next, move the new keys into place
               newKeyPath = tmpFilePath + keyAlgo
               newKeyPathPub = newKeyPath + ".pub"
               keyFingerPrint = SysMgrLib.getSshHostKeyFingerprint( newKeyPathPub )
               # For named keys, we will only store the private keys.
               # Do not copy the public key.
               if self.keyPath[ keyAlgo ] == SysMgrLib.keyTypeToPath[ keyAlgo ]:
                  # Default key, copy public key to destination.
                  shutil.move( newKeyPathPub, keyPathPub )
               else:
                  # Named key, scrub the generated public key from source
                  SysMgrLib.shredSshKey( newKeyPathPub )
               # Copy private key as usual
               shutil.move( newKeyPath, keyPath )
               self.sshStatus.sshKeygenInProgress[ keyAlgo ] = False
               t8( "SSH-key installed at path: %s" % self.keyPath[ keyAlgo ] )
               
               # Log the regeneration of ssh host key
               # generate the SHA-256 hash of the public key fingerprint
               keyTypeCli = SysMgrLib.tacKeyTypeToCliKey[ keyAlgo ]
               Logging.log( SYS_SSH_HOST_KEY_UPDATED,
                            keyTypeCli, "SHA-256", keyTypeCli, keyFingerPrint )

               # Log the change of supervisor MAC address that triggers the reset
               oldAddrComments = []
               if self.keygenOldSysMacAddr[ keyAlgo ]:
                  oldAddrComments.append( "MAC address %s" % 
                                          self.keygenOldSysMacAddr[ keyAlgo ] ) 
               if self.keygenOldCvxAddr[ keyAlgo ]:
                  oldAddrComments.append( "CVX MAC address %s" %
                                          self.keygenOldCvxAddr[ keyAlgo ] )
               if oldAddrComments:
                  Logging.log( SYS_SSH_HOST_KEY_REGENERATED,
                               ", ".join( oldAddrComments ) )

               self.keygenOldSysMacAddr[ keyAlgo ] = None
               self.keygenOldCvxAddr[ keyAlgo ] = None
               self.keyGenRetryCount[ keyAlgo ] = 0
               self.sshStatus.sshKeyResetProcessed[ keyAlgo ] = Tac.now()
               return
            else:
               errorStr = "return code %s and error msg: %s" % \
                          ( returnCode, keyGenProc.stderr.read() )
               qt0( "Error when regenerating key: ", qv( errorStr ) )
               t0( "Error when regenerating key: ", errorStr )
               # Try to regen key in a moment
               self.keyGenResetHandler[ keyAlgo ].timeMin = Tac.now() +\
                       self.keyResetHandlerWait
      else:
         t3( "keygen is not running" )
         if self.keyGenRetryCount[ keyAlgo ] < self.maxKeyGenRetries:
            t0( "Starting new keygen" )
            self.keyGenRetryCount[ keyAlgo ] += 1
            _, oldSysMacAddr, oldCvxAddr, _ = self.getPubKeyInfo( keyAlgo )
            if oldSysMacAddr:
               self.keygenOldSysMacAddr[ keyAlgo ] = oldSysMacAddr
            if oldCvxAddr:
               self.keygenOldCvxAddr[ keyAlgo ] = oldCvxAddr
            newKeyComment = self.generateExpectedComment( keyAlgo )
            tmpKeyPath = tmpFilePath + keyAlgo
            keyParams = self.keyParams[ keyAlgo ]
            SysMgrLib.shredSshKey( tmpKeyPath )
            keygenProc = SysMgrLib.generateSshKeysWithComment(
                    keyAlgo,
                    keyParams,
                    tmpKeyPath,
                    newKeyComment,
                    fipsMode=self.sshConfig.fipsRestrictions )
            if keygenProc:
               self.keygenProc[ keyAlgo ] = keygenProc
               self.sshStatus.sshKeygenInProgress[ keyAlgo ] = True
               self.keyGenTimeout[ keyAlgo ] = Tac.now() + self.keyGenWait
            else:
               # Something went wrong. Log it and try again in a moment
               t0( "Error when trying to make keygenProc." )
            self.keyGenResetHandler[ keyAlgo ].timeMin = Tac.now() +\
                    self.keyResetHandlerWait
         else:
            t0( "Too many attempts to generate", keyAlgo )
            qt0( "Too many attempts to generate", qv( keyAlgo ) )
            self.sshStatus.sshKeygenInProgress[ keyAlgo ] = False
            return

   def startNewKeyGen( self, keyAlgo ):
      """
      Start a new key generation process and reset previous timers.
      """
      t8( "KeyResetHandler::startNewKeyGen: " + keyAlgo )

      # Reset previous state and kill program ( if running )
      self.keyGenRetryCount[ keyAlgo ] = 0
      keyGenProc = self.keygenProc[ keyAlgo ]
      if keyGenProc:
         keyGenProc.kill()
         keyGenProc.wait()
         # Shred temporary key
         SysMgrLib.shredSshKey( tmpFilePath + keyAlgo )
         self.keygenProc[ keyAlgo ] = None
      # Force the handler to fire
      self.keyGenResetHandler[ keyAlgo ].timeMin = Tac.beginningOfTime

   def checkAllKeys( self, keyPaths=None, requestedKeyParams=None ):
      """
      Check all keyAlgos to see if they need to be reset
      or have their comments updated.

      keyPaths is a dictionary whose keys are standard algorithms
      and values are custom paths where the key file has to be created.
      If keyPaths is none then use the default path for each algorithm.
      """
      t8( "KeyResetHandler::checkAllKeys" )

      # First, check if the systemMacAddr is initialized
      systemMacAddr = self.entityMibStatus.systemMacAddr
      ethAddrZero = Arnet.EthAddr( systemMacAddr ).ethAddrZero
      if systemMacAddr == ethAddrZero:
         t8( "System Mac Addr is not initialized, returning" )
         return

      # Set default path for all algorithms
      for keyAlgo, defaultPath in SysMgrLib.keyTypeToPath.items():
         self.keyPath[ keyAlgo ] = defaultPath

      if keyPaths:
         # Set custom path for specified algorithm
         for keyAlgo, path in keyPaths.items():
            self.keyPath[ keyAlgo ] = path

      # Set key parameters for all algorithms: default to the first legal value for
      # each parameter in SysMgrLib., unless overwritten
      if requestedKeyParams is None:
         requestedKeyParams = {}
      for keyAlgo in SysMgrLib.keyTypeToPath:
         legalKeyParams = SysMgrLib.keyTypeToLegalParams.get( keyAlgo, {} )
         keyParams = { paramName: legalParamValues[ 0 ] for
                           paramName, legalParamValues in legalKeyParams.items() }
         # overwrite default key parameters with any parameters in received KeyData
         keyParams.update( requestedKeyParams.get( keyAlgo, {} ) )
         self.keyParams[ keyAlgo ] = keyParams

      for keyAlgo in SysMgrLib.keyTypeToPath:
         t0( "Checking keyAlgo: " + keyAlgo )
         if self.isResetNeeded( keyAlgo ):
            self.startNewKeyGen( keyAlgo )
         else:
            self.maybeUpdateComments( keyAlgo )

class SshKeyResetRequest( Tac.Notifiee ):
   """
   This notifiee reacts to a request to reset a SSH key by having the
   EntityMibStatusNotifiee be triggered as soon as possible.
   """
   
   notifierTypeName = "Mgmt::Ssh::ConfigReq"

   def __init__( self, config, agent ):
      Tac.Notifiee.__init__( self, config )
      self.agent = agent

   @Tac.handler( "sshKeyResetRequest" )
   def handleSshKeyResetRequest( self, keyAlgo ):
      keyPaths = {}
      keyParams = {}
      t8( "Inside sshKeyResetRequest for %s" % keyAlgo )
      keyData = self.notifier_.sshKeyResetRequest[ keyAlgo ]
      if self.agent.initialized_:
         if keyData.path:
            t8( "Named key creation requested, file: %s" % keyData.path )
            keyPaths[ keyAlgo ] = keyData.path
         if keyData.keyParams:
            t8( "Key parameter specified: %s" % keyData.keyParams )
            keyParams[ keyAlgo ] = keyData.keyParams
         self.agent.keyResetHandler_.checkAllKeys( keyPaths, keyParams )

class EntityMibStatusNotifiee( Tac.Notifiee ):
   """ This notifiee makes sure the SSH host keys on a supervisor are
   associated with the chassis. This is accomplished by putting a
   comment on the keys that contains the MAC address of the chassis.

   Comments for SSH keys are simply text that appears after the key in
   the public key file.

   First, we have to wait for the chassis to get a MAC address. This
   means reacting to 'hardware/entmib' of type EntityMib::Status which
   has a systemMacAddr attribute.

   Next we check if the host keys stored at
   '/persist/secure/ssh_{rsa,dsa}_key' have a comment. If not, we apply
   the comment to associate it with this chassis.

   If they do have a comment and it matches the chassis MAC, then we
   do nothing. If it doesn't match we delete and regenerate the host
   keys.

   An additional piece of logic is added where if the time for a
   sshStatus::sshKeyResetProcessed entry is less than the corresponding
   sshConfig::sshKeyResetRequest the key is forcefully regenerated and
   the time in sshStatus::sshKeyResetProcessed is updated.
   """

   notifierTypeName = 'EntityMib::Status'
   
   def __init__( self, entityMibStatus, agent ):
      Tac.Notifiee.__init__( self, entityMibStatus )

      self.entityMibStatus = entityMibStatus
      self.agent = agent
      
   @Tac.handler( 'systemMacAddr' )
   def handleSystemMacAddr( self ):
      t8( "EntityMibStatusNotifiee::handleSystemMacAddr" )
      systemMacAddr = self.entityMibStatus.systemMacAddr
      ethAddrZero = Arnet.EthAddr( systemMacAddr ).ethAddrZero
      if systemMacAddr != ethAddrZero and self.agent.initialized_:
         self.agent.keyResetHandler_.checkAllKeys()

class CvxMacAddressNotifiee( Tac.Notifiee ):
   notifierTypeName = 'Interface::MgmtActiveIntfStatus'

   def __init__( self, intfStatus, agent ):
      self.intfStatus = intfStatus
      self.agent = agent
      Tac.Notifiee.__init__( self, intfStatus )

      self.handleCvxMacAddr()

   @Tac.handler( 'addr' )
   def handleCvxMacAddr( self ):
      addr = self.intfStatus.addr
      t8( "handleCvxMacAddr", addr )
      qt0( "handleCvxMacAddr", qv( addr ) )
      if addr != Arnet.EthAddr( addr ).ethAddrZero and\
         self.agent.initialized_:
         self.agent.keyResetHandler_.cvxMacAddress = addr
         self.agent.keyResetHandler_.checkAllKeys()

class IntfStatusNotifiee( Tac.Notifiee ):
   notifierTypeName = 'Interface::MgmtActiveIntfStatusDir'

   def __init__( self, mgmtActiveIntfStatusDir, entityMibStatus, sshConfig,
                 sshConfigReq, sshStatus, clusterStatusDir, agent ):

      Tac.Notifiee.__init__( self, mgmtActiveIntfStatusDir )

      self.cvxMacAddressNotifiee_ = None
      self.entityMibStatus = entityMibStatus
      self.sshConfig = sshConfig
      self.sshConfigReq = sshConfigReq
      self.sshStatus = sshStatus
      self.clusterStatusDir = clusterStatusDir
      self.intfStatus = mgmtActiveIntfStatusDir.intfStatus
      self.agent = agent

      self.handleIntfStatus()

   @Tac.handler( 'intfStatus' )
   def handleIntfStatus( self ):
      self.intfStatus = self.notifier_.intfStatus
      if self.intfStatus:
         if self.cvxMacAddressNotifiee_:
            t0( "CvxMacAddressNotifiee already exists, skipping creation" )
            return
         qt0( "Creating CvxMacAddressNotifiee for", qv( self.intfStatus.intfId ) )
         t0( "Creating CvxMacAddressNotifiee for", self.intfStatus.intfId )
         self.cvxMacAddressNotifiee_ = CvxMacAddressNotifiee( self.intfStatus,
                                                              self.agent )
      else:
         t0( "Deleting CvxMacAddressNotifiee" )
         self.cvxMacAddressNotifiee_ = None
         if self.agent.initialized_:
            self.agent.keyResetHandler_.cvxMacAddress = ""

class SshHostKeyMgmtSecurityMgr( SysMgrLib.MgmtSecurityMgr ):
   def __init__( self, config, status, agent ):
      self.agent_ = agent
      SysMgrLib.MgmtSecurityMgr.__init__( self, config, status, agent )

   def sync( self ):
      if self.agent_.initialized_:
         self.agent_.keyResetHandler_.checkAllKeys()

class SshHostKeysAgent( SuperServer.SuperServerAgent ):
   def __init__( self, entityManager ):
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      self.entityMibStatus = mg.mount( 'hardware/entmib', 'EntityMib::Status', 'r' )
      self.sshConfig = mg.mount( 'mgmt/ssh/config', 'Mgmt::Ssh::Config', 'r' )
      self.sshConfigReq = mg.mount( 'mgmt/ssh/configReq', 'Mgmt::Ssh::ConfigReq',
            'r' )
      self.sshStatus = mg.mount( Cell.path( 'mgmt/ssh/status' ),
                            'Mgmt::Ssh::Status', 'fw' )
      self.mgmtSecConfig = mg.mount( 'mgmt/security/config',
                                'Mgmt::Security::Config', 'r' )
      self.mgmtSecStatus = mg.mount( Cell.path( 'mgmt/security/status' ),
                                'Mgmt::Security::Status', 'r' )
      # We are mounting these paths for the case of CVX because
      # mgmtActiveIntfStatusDir is required to fetch the virtual mac address of Ma0
      # interface and clusterStatusDir is required to fetch the information that we
      # are the master( Leader ) node or not.
      self.mgmtActiveIntfStatusDir = mg.mount(
            'interface/status/eth/managementactive',
            'Interface::MgmtActiveIntfStatusDir', 'r' )
      self.clusterStatusDir = mg.mount( 'controller/cluster/statusDir',
                                   'ControllerCluster::ClusterStatusDir', 'r' )
      self.entityMibNotifiee_ = None
      self.intfStatusNotifiee_ = None
      self.sshResetNotifiee_ = None
      self.mgmtSecurityMgr_ = None
      self.keyResetHandler_ = None
      self.initialized_ = False

      def _finished():
         # Run only if supervisor is in active state
         if self.active():
            t0( "Agent is active, starting SshHostKeys" )
            self.onSwitchover( None )
         else:
            t0( "Agent is not active, not starting SshHostKeys" )

      mg.close( _finished )

   def onSwitchover( self, protocol ):
      self.entityMibNotifiee_ = EntityMibStatusNotifiee( self.entityMibStatus,
                                                         self )
      self.intfStatusNotifiee_ = IntfStatusNotifiee( self.mgmtActiveIntfStatusDir,
                                                     self.entityMibStatus,
                                                     self.sshConfig,
                                                     self.sshConfigReq,
                                                     self.sshStatus,
                                                     self.clusterStatusDir,
                                                     self )
      self.sshResetNotifiee_ = SshKeyResetRequest( self.sshConfigReq, self )
      self.mgmtSecurityMgr_ = SshHostKeyMgmtSecurityMgr(
         self.mgmtSecConfig, self.mgmtSecStatus, self )
      self.keyResetHandler_ = KeyResetHandler( self.sshConfig,
                                               self.sshConfigReq,
                                               self.sshStatus,
                                               self.entityMibStatus,
                                               self.clusterStatusDir,
                                               self )
      self.keyResetHandler_.checkAllKeys()
      self.initialized_ = True
      t0( "SshHostKeys reactors are initialized" )

def Plugin( context ):
   context.registerService( SshHostKeysAgent( context.entityManager ) )
