#!/usr/bin/env python3
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, SuperServer
from SswanManagerUtils import IkeDaemonFipsHandler
from IpLibConsts import DEFAULT_VRF
from Arnet.NsLib import DEFAULT_NS
import SharedMem
import Smash
import Toggles.IpsecToggleLib
import Toggles.PolicyMapToggleLib
import BothTrace
import Tracing
import Cell
import os, signal, time
import re
import shutil
import errno
import subprocess
import sys
import weakref
from Netns import setns, NamespaceType
import netns

__defaultTraceHandle__ = Tracing.Handle( "IpsecSswan" )
bt0 = BothTrace.tracef0
bt4 = BothTrace.tracef4
bt5 = BothTrace.tracef5
bt8 = BothTrace.tracef8
bt9 = BothTrace.tracef9
bv = BothTrace.Var

traceFunc = bt4
traceVerbose = bt5
traceVeryVerbose = bt8
traceTimer = bt9

# pylint: disable-msg=C0209
# pylint: disable-msg=C1801
# pylint: disable-msg=R0205
# pylint: disable-msg=R1703
# pylint: disable-msg=R1711
# pylint: disable-msg=R1714
# pylint: disable-msg=R1724
# pylint: disable-msg=R1732
# pylint: disable-msg=R1734
# pylint: disable-msg=R1735

DEFAULT_VRF_ID = Tac.Type( 'Vrf::VrfIdMap::VrfId' ).defaultVrf

def pidExists( pid ):
   if pid <= 0:
      return False
   ret = os.path.exists( f'/proc/{pid}' )
   return ret

def killProc( pid, desc ):
   traceFunc( bv( pid ), ' ', bv( desc ) )
   if not pidExists( pid ):
      return

   try:
      os.kill( pid, signal.SIGTERM )
   except OSError:
      pass
   except Tac.Timeout:
      traceFunc( 'ERROR: Unable to kill ', bv( pid ), ' ', bv( desc ) )

def getPid( pidFile ):
   pid = ''
   try:
      with open( pidFile ) as f:
         pid = f.readline()
   except OSError:
      pass

   if pid != '' and os.path.exists( f'/proc/{int(int(pid))}' ):
      return int( pid )

   return -1

def getNetNsInode( pid, vrfName=None ):
   if vrfName is not None:
      assert pid is None
      if vrfName == DEFAULT_VRF:
         nsFile = '/var/run/netns/default'
      else:
         nsFile = f'/var/run/netns/ns-{vrfName}'
   else:
      nsFile = f'/proc/{int(pid)}/ns/net'
   nsFileStat = None
   try:
      nsFileStat = os.stat( nsFile )
   except ( OSError, ProcessLookupError ):
      return None
   return nsFileStat.st_ino

def keyStr( key ):
   if isinstance( key, str ):
      return key
   return key.str()

def capFirst( s ):
   return s[ 0 ].upper() + s[ 1: ]

def subProcessOutput( cmd ):
   majorVersion = sys.version_info[ 0 ]
   output = ''

   try:
      if majorVersion == 2:
         output = subprocess.check_output( cmd, shell=True )
      elif majorVersion == 3:
         output = subprocess.check_output( cmd, shell=True, text=True )
   except subprocess.CalledProcessError:
      return list()

   outputList = output.split( '\n' )
   return outputList

def secretIdFromConfConn( confConn ):
   assert isinstance( confConn, Tac.Type( 'Ipsec::Conf::Connection' ) )
   return Tac.Type( 'Ipsec::IpsecUtils' ).computeSecretId( confConn )

class RunInMntNamespace:
   def __init__( self, baseDir, origStrongswanDir ):
      self.selfMntFile = '/proc/self/ns/mnt'
      self.origMntNs = open( self.selfMntFile )
      self.origMntNsFd = self.origMntNs.fileno()
      self.inode = os.stat( self.selfMntFile ).st_ino
      self.baseDir = baseDir
      self.origStrongswanDir = origStrongswanDir

   def __enter__( self ):
      try:
         # Go into a new MNT namespace
         # pylint: disable-msg=c-extension-no-member
         netns.unshare( netns.CLONE_NEWNS )

         # Systemd makes root a shared mount point. This causes mounts to
         # propagate. Make it private in this mount namespace so that
         # the below bind mount remains private as well.
         remountRoot = 'mount --make-rprivate /'
         subProcessOutput( remountRoot )

         # Bind mount /etc/strongswan on /etc/sswan/vrf/VRF-<name>/
         mountCmd = f'mount -B {self.baseDir} {self.origStrongswanDir}'
         subProcessOutput( mountCmd )
      except OSError as e:
         traceVerbose( 'Run in mnt namespace failed : ', bv( e.strerror ) )
         raise

   def __exit__( self, exception, execType, stackTrace ):
      try:
         # Go back to original mount namespace
         setns( self.origMntNsFd, nsType=NamespaceType.MOUNT )
      finally:
         self.origMntNs.close()
         assert self.inode == os.stat( self.selfMntFile ).st_ino

#
# this object models a "stroke" process, which communicates with
# strongswan and passes operation instructions to it.  It is used for
# bringing connections up or down, as well as for rereading the
# secrets file
#
class StrokeProc:
   def __init__( self, key, cmd, cmdSuffix, netNsName, baseDir, origStrongswanDir ):
      self.key = key
      self.cmd = cmd
      self.cmdSuffix = cmdSuffix
      self.unixProc = None
      self.state = 'null'
      self.netNsName = netNsName
      self.stdoutText = ''
      self.stderrText = ''
      self.stdoutLines = list()
      self.stderrLines = list()
      self.startedAt = None
      self.baseDir = baseDir
      self.origStrongswanDir = origStrongswanDir

   def run( self ):
      assert not self.unixProc
      strokeCmd = '/usr/libexec/strongswan/stroke'
      cmdList = [ strokeCmd, self.cmd ]
      if self.netNsName != 'default':
         cmdList = [ 'ip', 'netns', 'exec', self.netNsName ] + cmdList

      if self.cmdSuffix is not None:
         cmdList.append( self.cmdSuffix )

      self.clearSavedOutput()
      with RunInMntNamespace( self.baseDir, self.origStrongswanDir ):
         self.unixProc = subprocess.Popen( cmdList, stdout=subprocess.PIPE,
                                           stderr=subprocess.PIPE )
      traceVerbose( 'StrokeProc ', bv( self.unixProc.pid ), 'cmd: ',
                    bv( ' '.join( cmdList ) ) )
      self.stateIs( 'running' )
      self.startedAt = Tac.now()
      os.set_blocking( self.unixProc.stdout.fileno(), False )
      os.set_blocking( self.unixProc.stderr.fileno(), False )

   def kill( self ):
      if self.unixProc:
         traceVerbose( 'pid', bv( self.unixProc.pid ), 'cmd', bv( self.cmd ) )
         self.unixProc.terminate()
         status = self.unixProc.poll()
         if status is None:
            self.unixProc.kill()
         self.unixProc = None
      else:
         traceVeryVerbose( 'nothing to do' )
      if self.state != 'null':
         self.stateIs( 'null' )
      self.startedAt = None

   def poll( self ):
      if self.state == 'running':
         result = self.unixProc.poll()
         traceVerbose( 'pid', bv( self.unixProc.pid ), 'cmd', bv( self.cmd ),
                       'result', bv( result ) )
         self.readPipes()
         if result is not None:
            if result == 0:
               self.stateIs( 'success' )
            else:
               self.stateIs( 'failed' )

            self.unixProc = None
            self.processPipes()

   def clearSavedOutput( self ):
      self.stdoutText = ''
      self.stdoutLines = list()
      self.stderrText = ''
      self.stderrLines = list()

   def readPipes( self ):
      if not self.unixProc:
         return
      traceVeryVerbose( bv( self.cmd ), bv( self.cmdSuffix ) )

      while True:
         l = self.unixProc.stdout.readline()
         if not l:
            break
         if isinstance( l, bytes ):
            l = l.decode( 'utf-8' )
         self.stdoutText += l
         traceVeryVerbose( 'proc %s %s stdout: %s' %
                           ( self.cmd, self.cmdSuffix, l ) )

      while True:
         l = self.unixProc.stderr.readline()
         if not l:
            break
         if isinstance( l, bytes ):
            l = l.decode( 'utf-8' )
         self.stderrText += l
         traceVeryVerbose( 'proc %s %s stderr: %s' %
                           ( self.cmd, self.cmdSuffix, l ) )

   def processPipes( self ):
      self.stdoutLines = self.stdoutText.split( '\n' )
      self.stderrLines = self.stderrText.split( '\n' )

   def stateIs( self, newState ):
      assert newState != self.state
      self.state = newState

class ConnState:
   def __init__( self, confConn, secretId, genId ):
      self.confConn = confConn
      self.secretId = secretId
      self.genId = genId

#
# this object models a single connection within an instance of
# strongswan.  It is keyed by an Ipsec::ConnectionKey, and manages a
# StrokeProc to bring that connection up and down as required.
#
class Connection:
   def __init__( self, key, connManager, connStatus, confConn, connName,
                 netNsName, connId, secretGenId, baseDir, origStrongswanDir ):
      self.key = key
      self.connManager = connManager
      assert self.connManager
      self.proc = None
      self.connStatus = connStatus
      self.connName = connName
      self.netNsName = netNsName
      self.connId = connId
      self.stateChangedAt = None
      self.timeout = 60
      self.secretGenId = secretGenId
      self.baseDir = baseDir
      self.origStrongswanDir = origStrongswanDir

      traceFunc( 'Creating Connection for ', bv( keyStr( key ) ), ' count id ',
                 bv( self.connId ) )

      # valid states are:
      #
      # pendingStrokeUp: 'stroke up' command has been issued, but
      #   child process exit code has not yet been processed (or may
      #   not yet have exited)
      #
      # pendingUp:  'stroke up' command has completed successfully, but
      #   we have not yet confirmed successful bringup with 'stroke status'
      #
      # up:  'stroke up' has completed, 'stroke status' indicates IKE
      #   SA is Established
      #
      # pendingStrokeDown:  'stroke down' command has been issued, but
      #   exit code not yet processed
      #
      # pendingDown:  'stroke down' has completed successfully, but
      #   not yet confirmed via 'stroke status'
      #
      # down:  connection is down.
      #
      # passiveUp:  autostart is set to 'add' (passive mode), but the other
      #   side brought the conneciton up

      self.state = 'null'
      self.stateForced = False

      self.hupCount = 0 # maybe should save this in the daemon status too?

      self.cmdCount = { 'up': 0,
                        'down': 0,
                        'reloadUp': 0,
                        'reloadDown': 0 }

      self.currentCmd = None

      # these are used to track changes in the configuration state
      # associated with a particular connection and detect when it
      # needs to be reloaded.  We track the confConn, the secret id
      # derived from it (this is just a cache to avoid re-generating)
      # and the secretGenId from when we loaded that value

      # 'currentState' has whatever we last wrote to ipsec.conf.  This
      # is copied to 'pendingUpState' whenever an up command is issued
      # (either normal or reload).  Once that up command succeeds, the
      # pending value is moved to 'lastUpState' (and 'pendingUpState'
      # is cleared).
      #
      # if the 'currentState' differs from the last or pending
      # versions, then we know that connection needs reloading.
      #
      # similarly, if the genId associated with the state is older
      # than the last time that the secret was written to to the
      # secrets file then we know that we also need to reload it
      secretId = secretIdFromConfConn( confConn )
      self.currentState = ConnState( confConn, secretId, self.secretGenId )
      self.pendingUpState = None
      self.lastUpState = None
      self.passivePendingChildState = None
      self.swanConnectionId = None
      self.ikeConnName = None

      self.syncStateToSysdb()

   def start( self, cmd, cmdSuffix=None ):
      traceVerbose( 'cmd ', bv( cmd ) )
      assert cmd in [ 'up', 'down' ]
      assert self.currentCmd is None
      self.currentCmd = cmd

      if self.key.connId.connIdSrc == 'srcPolicy':
         if not cmdSuffix:
            cmdSuffix = self.connName + self.swanConnectionId
      else:
         cmdSuffix = self.connName

      traceVerbose( 'cmdSuffix ', bv( cmdSuffix ) )
      self.proc = StrokeProc( self.key, cmd=cmd, cmdSuffix=cmdSuffix,
                              netNsName=self.netNsName,
                              baseDir=self.baseDir,
                              origStrongswanDir=self.origStrongswanDir )
      self.proc.run()
      self.stateIs( 'pendingStroke' + capFirst( cmd ) )
      self.incrementStartCounter( cmd )

   def incrementStartCounter( self, cmd ):
      traceVerbose( 'cmd ', bv( cmd ) )
      assert cmd in self.cmdCount
      self.cmdCount[ cmd ] += 1
      if self.connStatus:
         if cmd == 'up':
            self.connStatus.upCount += 1
         elif cmd == 'down':
            self.connStatus.downCount += 1

   def pendingStroke( self ):
      return self.state in [ 'pendingStrokeDown', 'pendingStrokeUp' ]

   # check if the stroke process has finished, if so then update our
   # state appropriately
   def poll( self ):
      assert self.connManager

      # this function is for the up/down processes, not the
      # rereadsecrets or status ones

      assert self.state in [ 'pendingStrokeUp', 'pendingStrokeDown' ]

      traceVerbose( 'polling ', bv( keyStr( self.key ) ), ' id ', bv( self.connId ) )
      assert self.proc
      assert self.proc.cmd
      assert self.proc.cmd in [ 'up', 'down' ]
      self.proc.poll()

      assert self.proc.state != 'null'
      if self.proc.state == 'running':
         # this means the stroke process has not exited yet
         return

      assert self.proc.state in [ 'success', 'failed' ]
      success = ( self.proc.state == 'success' )
      # proc.state is just the exit code of the stroke command.  If
      # this is an error then we failed to enqueue the stroke request
      # in strongswan, but there are also false positives
      if success and self.proc.cmd == 'up':
         assert self.proc.stdoutLines
         lastLine = self.proc.stdoutLines[ -1 ]
         if 'failed' in lastLine:
            success = False
            self.hupCount = 0
         elif 'no config named' in lastLine:
            # this one means that somehow strongswan didn't see the
            # file we wrote?
            if self.hupCount > 3:
               self.connManager.perVrfService.needsReloadIs( True )
               self.hupCount = 0
            else:
               self.hupCount += 1
               self.connManager.perVrfService.needsHupIs( True )

            success = False

      if success:
         if self.state == 'pendingStrokeUp':
            self.stateIs( 'pendingUp' )
         else:
            assert self.state == 'pendingStrokeDown'
            self.stateIs( 'pendingDown' )
      else:
         if self.state == 'pendingStrokeUp':
            # if it failed to come up then we may need to stroke it
            # down again to get strongswan to be able to reread it
            self.currentCmd = None
            self.proc = None
            # can't really get up to the connection manager to call
            # maybeStartProc, but we know that we already had an
            # outstanding process (which we just deleted) so starting
            # a new one won't break the process limit.
            self.start( 'down' )
            # return so that we don't override the new cmd
            return
         else:
            # can a 'stroke down' fail?
            self.stateIs( 'up' )
            # if a 'stroke down' fails then we need to redo it, but
            # we've already nuked the 'lastUpState'.
            # cleanupConnection will set stateForced to true, so that
            # we handle this correctly
            self.cleanupConnection( 'up' )

      self.currentCmd = None
      self.proc = None

   def stateIs( self, newState ):
      assert newState != self.state
      traceVerbose( bv( keyStr( self.key ) ), bv( self.state ), '->',
                    bv( newState ), ' id ', bv( self.connId ) )
      self.state = newState
      self.syncStateToSysdb()

      if newState == 'up':
         # when going up, we ripple 'pending' into 'last'
         self.lastUpState = self.pendingUpState
         self.pendingUpState = None
      elif newState == 'passiveUp':
         # in passive mode we don't have a pendingUpState.  instead
         # copy the currentState
         assert self.pendingUpState is None
         self.currentState.genId = self.connManager.perVrfService.secretGenId
         self.lastUpState = self.currentState
      elif newState == 'pendingStrokeUp':
         # when we go to pendingUp we dup the current into 'pending'
         self.currentState.genId = self.connManager.perVrfService.secretGenId
         self.lastUpState = None
         self.pendingUpState = self.currentState
      elif newState == 'down':
         # when we go down we clear both pending and last.
         self.lastUpState = None
         self.pendingUpState = None
      elif newState == 'pendingStrokeDown':
         # when we go to 'pendingDown' we clear the pending conf conn
         # (if any), but we do not touch 'lastUp'.  This is because if
         # the 'down' command fails then the connection is presumed to
         # still be up and we want to know what config it has loaded.
         self.pendingUpState = None
      elif newState == 'pendingUp' or newState == 'pendingDown':
         # this is just the second half of the 'pending' state, we are
         # still waiting for confirmation of success via 'stroke
         # status' and thus do not need to update the stored confs
         pass
      elif newState == 'pendingChild':
         # moving to pendingChild , copy the current to pending
         if not self.activeMode():
            self.passivePendingChildState = self.currentState
      else:
         # what state are we in???
         assert 0

      self.stateForced = False
      self.stateChangedAt = Tac.now()

   def syncStateToSysdb( self ):
      if not self.connStatus:
         return

      newSysdbState = 'daemonConnState' + capFirst( self.state )
      if self.connStatus.state != newSysdbState:
         self.connStatus.state = newSysdbState

   def needsReload( self ):
      if( self.state == 'null' or
          self.state == 'down' or
          self.state == 'pendingDown' or
          self.state == 'pendingStrokeDown' ):
         return False

      assert self.currentState
      if self.state in [ 'up', 'passiveUp' ]:
         if self.stateForced:
            return True
         assert self.lastUpState
         stateToUse = self.lastUpState
      elif self.state in [ 'pendingUp', 'pendingStrokeUp' ]:
         stateToUse = self.pendingUpState
      else:
         assert self.state in [ 'pendingChild' ]
         if self.activeMode():
            stateToUse = self.pendingUpState
         else:
            stateToUse = self.passivePendingChildState

      # if the conf data changed, we need to reload
      if stateToUse.confConn != self.currentState.confConn:
         return True

      # this is probably paranoia, since the secret key is derived
      # from the ConfData, but...
      if stateToUse.secretId != self.currentState.secretId:
         return True


      shadowGenId = self.connManager.perVrfService.shadowSecretGenId(
         stateToUse.secretId )
      # check to see if the seret data has changed since the last time
      # we went up
      traceVeryVerbose( 'shadow secret gen id is %d, conn genid is %d' %
                        ( shadowGenId, stateToUse.genId ) )
      if( self.connManager.perVrfService.shadowSecretGenId( stateToUse.secretId ) >
          stateToUse.genId ):
         return True

      return False

   def activeMode( self ):
      return self.currentState.confConn.connStart == 'connectionStartAutoStart'

   def cleanupConnection( self, newState, forced=True ):
      traceVerbose( 'cmd', bv( self.currentCmd ) )
      self.currentCmd = None
      if self.proc:
         self.proc.kill()
         self.proc = None

      self.lastUpState = None
      self.pendingUpState = None
      self.passivePendingChildState = None
      traceVerbose( 'pseudo stateIs', bv( keyStr( self.key ) ), bv( self.state ),
                    '->', bv( newState ), 'id', bv( self.connId ) )
      self.state = newState
      self.stateForced = forced

   def pendingTimeout( self ):
      if not self.state.startswith( 'pending' ):
         return False

      assert self.stateChangedAt != Tac.beginningOfTime
      return ( Tac.now() - self.stateChangedAt ) > self.timeout

   def curReqId( self ):
      assert self.currentState
      return self.currentState.confConn.reqId

   def maybeUpdateConfData( self, newConfData ):
      if self.currentState.confConn == newConfData:
         return False

      self.currentState = ConnState( newConfData,
                                     secretIdFromConfConn( newConfData ),
                                     self.connManager.perVrfService.secretGenId )
      return True

#
# This object manages a set of Connections (above) for the
# PerVrfService (below).  In a sense it's not really a separate SM,
# but it's conceptually easier to break the connection-related
# functions out from the code in PerVrfService that generates the
# config files
#
class ConnectionManager:
   def __init__( self, perVrfService, baseDir, origStrongswanDir ):
      self.perVrfService = perVrfService
      self.baseDir = baseDir
      self.origStrongswanDir = origStrongswanDir

      self.connection = dict()

      self.connTimer = Tac.ClockNotifiee()
      self.connTimer.handler = self.handleConnTimer
      self.connTimer.timeMin = Tac.endOfTime

      self.rereadSecretsProc = None
      self.rereadCertsProc = None
      self.purgeCrlsProc = None

      self.statusDirty = False
      self.statusProc = None
      self.statusProcTimeout = 300
      self.lastStatusResult = Tac.beginningOfTime
      self.minStatusInterval = 30

      self.connectionsInConfFile = set()
      self.lastCfg = None

      self.cleanedUp = False

      self.nextReviewBacklog = set()
      self.nextReviewBacklogExpedited = set()
      self.curReviewBacklog = set()
      self.connTimerQuantum = 0.2

      # set this true to turn off an assert about duplicate
      # connections and tracing out debug messages instead
      self.traceDuplicateConnections = False

      self.policyBasedVpnIkeSas = dict()

      # max # of outstanding stroke processes at once
      self.procLimit = 10

   def handleConnTimer( self ):
      traceVerbose( 'vrf ', bv( self.perVrfService.vrfName ) )
      self.connTimer.timeMin = Tac.endOfTime

      assert self.perVrfService.cleanupState == 'not started'

      if self.perVrfService.daemonConfig.simulateDaemonForBtest:
         return

      startTime = Tac.now()

      if not self.curReviewBacklog:
         traceVerbose( 'starting new review pass' )
         self.curReviewBacklog = self.nextReviewBacklog.union(
            self.nextReviewBacklogExpedited )
         self.nextReviewBacklog = set()
         self.nextReviewBacklogExpedited = set()

      if self.statusDirty or self.statusProc:
         self.curReviewBacklog.add( 'status' )

      processed = set()
      for key in self.curReviewBacklog:
         if key == 'status':
            complete = self.processStatusProc()
         elif key == 'rereadSecrets':
            complete = self.processRereadSecretsProc()
         elif key == 'rereadCerts':
            complete = self.processRereadCertsProc()
         elif key == 'purgeCrls':
            complete = self.processPurgeCrlsProc()
         else:
            assert isinstance( key, Tac.Type( 'Ipsec::ConnectionKey' ) )
            complete = self.processKey( key )

         if not complete:
            self.addToBacklog( key )

         processed.add( key )

         if Tac.now() - startTime > self.connTimerQuantum:
            break

      if processed == self.curReviewBacklog:
         self.curReviewBacklog = set()
         traceVerbose( 'Review pass complete' )
         # we finished the review pass
         if self.nextReviewBacklogExpedited:
            traceVerbose( 'expediting next connTimer review' )
            self.scheduleConnTimer( 0 )
         elif self.nextReviewBacklog:
            traceVerbose( 'normal interval for next connTimer review' )
            self.scheduleConnTimer()
         return

      # we still have stuff to process
      self.curReviewBacklog -= processed
      self.scheduleConnTimer()

   def addToBacklog( self, key, expedited=False ):
      backlog = self.nextReviewBacklog
      if expedited:
         backlog = self.nextReviewBacklogExpedited

      traceVerbose( 'adding', bv( keyStr( key ) ), 'expedite', bv( expedited ),
                    'already present', bv( key in backlog ) )
      backlog.add( key )

      if key in self.connection:
         conn = self.connection[ key ]
         if conn.state in [ 'pendingUp', 'pendingDown', 'pendingChild' ]:
            # in either of these states we're going to need a status
            # proc as well
            self.startStatusCheck()

   def cleanupStatusProc( self ):
      if self.statusProc:
         self.statusProc.kill()
         self.statusProc.readPipes()
         self.statusProc.processPipes()
      self.statusProc = None

   def processStatusProc( self ):
      traceVerbose( 'vrf ', bv( self.perVrfService.vrfName ) )
      if self.statusProc:
         self.statusProc.poll()
         if self.statusProc.state == 'failed':
            self.statusProc.run()
            return False
         elif self.statusProc.state == 'running':
            now = Tac.now()
            assert self.statusProc.startedAt is not None
            if now - self.statusProc.startedAt > self.statusProcTimeout:
               traceVerbose( 'status proc timeout' )
               self.cleanupStatusProc()
               self.statusDirty = True
            return False
         elif self.statusProc.state == 'success':
            ( swanConnState, swanRouted ) = self.parseStatusOutput()
            self.cleanupStatusProc()
            self.processStatusOutput( swanConnState, swanRouted )
            return True
         else:
            assert 0
            return True

      if self.statusDirty:
         # it is possible that the timer for status fires on the standby
         # but, the daemonstatus is not yet populated. Just add back to the backlog
         if not self.perVrfService.daemonStatus:
            traceVerbose( 'daemonStatus is not present for vrf ',
                          bv( self.perVrfService.vrfName ) )
            return False

         self.perVrfService.daemonStatus.statusCount += 1
         self.statusProc = StrokeProc( key='status', cmd='status', cmdSuffix=None,
                                       netNsName=self.perVrfService.netNsName,
                                       baseDir=self.baseDir,
                                       origStrongswanDir=self.origStrongswanDir )
         self.statusProc.run()
         self.statusDirty = False

         return False

      return True

   def processRereadSecretsProc( self ):
      traceVerbose( 'vrf', bv( self.perVrfService.vrfName ) )
      if not self.rereadSecretsProc:
         return False

      self.rereadSecretsProc.poll()
      if self.rereadSecretsProc.state == 'failed':
         self.rereadSecrets()
         return True
      elif self.rereadSecretsProc.state == 'running':
         return False
      elif self.rereadSecretsProc.state == 'success':
         self.rereadSecretsProc = None
         self.perVrfService.secretGenId += 1
         self.perVrfService.markSecretsWritten()
         self.addAllKeysToBacklog()
         return True
      else:
         assert 0
         return False

   def processRereadCertsProc( self ):
      traceVerbose( 'vrf', bv( self.perVrfService.vrfName ) )
      if not self.rereadCertsProc:
         return False

      self.rereadCertsProc.poll()
      if self.rereadCertsProc.state == 'failed':
         self.rereadCerts()
         return True
      elif self.rereadCertsProc.state == 'running':
         return False
      elif self.rereadCertsProc.state == 'success':
         self.rereadCertsProc = None
         self.addAllKeysToBacklog()
         return True
      else:
         assert 0
         return False

   def processPurgeCrlsProc( self ):
      traceVerbose( 'vrf', bv( self.perVrfService.vrfName ) )
      if not self.purgeCrlsProc:
         return False

      self.purgeCrlsProc.poll()
      if self.purgeCrlsProc.state == 'failed':
         self.purgeCrls()
         return True
      elif self.purgeCrlsProc.state == 'running':
         return False
      elif self.purgeCrlsProc.state == 'success':
         self.purgeCrlsProc = None
         self.addAllKeysToBacklog()
         return True
      else:
         assert 0
         return False

   def addAllKeysToBacklog( self ):
      self.nextReviewBacklog.update( set( self.connection.keys() ) )
      self.nextReviewBacklog.update(
         set( self.perVrfService.shadowConfData.keys() ) )
      self.scheduleConnTimer()

   def parseStatusKey( self, inputStr ):
      # this regexp is looking for (vrfname)-(src IPv4 addr)-(dst IPv4 addr)
      m = re.match( ( r'(.*)-([0-9]+\.[0-9]+\.[0-9]+\.[0-9]+)' +
             r'-([0-9]+\.[0-9]+\.[0-9]+\.[0-9]+)-(.*)-([0-9]+)' ), inputStr )
      assert m
      vrfName = m.groups()[ 0 ]
      srcAddr = m.groups()[ 1 ]
      dstAddr = m.groups()[ 2 ]
      connIdSrc = m.groups()[ 3 ]
      connId = int( m.groups()[ 4 ] )

      assert vrfName == self.perVrfService.vrfName
      vrfId = self.perVrfService.vrfId
      tacSa = Tac.Value( 'Arnet::IpGenAddr', srcAddr )
      tacDa = Tac.Value( 'Arnet::IpGenAddr', dstAddr )
      Id = Tac.ValueConst( 'Ipsec::ConnectionId', connIdSrc, connId )
      key = Tac.ValueConst( 'Ipsec::ConnectionKey', vrfId, tacSa, tacDa, Id )
      return key

   def getIkeConnectionKey( self, key ):
      Id = Tac.ValueConst( 'Ipsec::ConnectionId', key.connId.connIdSrc, 0 )
      ikeKey = Tac.ValueConst( 'Ipsec::ConnectionKey', key.vrfId,
              key.srcAddr, key.dstAddr, Id )
      return ikeKey

   def parseStatusOutput( self ):
      traceVerbose( 'vrf', bv( self.perVrfService.vrfName ) )

      # there are two main reasons why we need to look at the
      # 'strongswan status' output.  the first is that issuing a
      # 'stroke up' or 'stroke down' command is not synchronous -- the
      # 'stroke' command simply enqueues the request with strongswan
      # and then exits.  To find out whether or not it succeeds we
      # need to wait until that is reflected in the 'status' output
      #
      # The second reasons is that if a connection is dropped by the
      # other side, strongswan will not automatically retry it.
      # Instead we need to detect that drop and start running stroke
      # processes again.
      #
      # The connection drop is detected by the Ipsec agent via netlink
      # updates.  We don't get those here, so instead we react to the
      # "connStatusNetlink" state.  A connection drop shows up as a
      # key that goes from Established or Connected to being not
      # present.

      connections = {}
      routedConn = {}

      for l in self.statusProc.stdoutLines:
         # this regexp matches the first line output by strongswan
         # status for each connection, the main IKE SA.  It can be
         # distinguished from the child SAs by the use of square
         # brackets around the id.  We don't really care about
         # anything in this line other that the fact that it exists
         m = re.match( r'^([^ ]+)(\[[0-9]+\]): ([A-Z]+),? ', l )

         if m:
            ( connName, swanConnectionId, state ) = m.groups()
            key = self.parseStatusKey( connName )
            if key.connId.connIdSrc == 'srcPolicy':
               key = self.getIkeConnectionKey( key )
            if key in connections and self.traceDuplicateConnections:
               traceVerbose( 'duplicate key', bv( key.str() ) )
               if self.lastCfg is None:
                  traceVeryVerbose( "lastCfg is None" )
               else:
                  for l2 in self.lastCfg.split( '\n' ):
                     traceVeryVerbose( 'lastCfg ', bv( l2 ) )
            connData = {}
            connData[ 'state' ] = state
            connData[ 'connectionId' ] = swanConnectionId
            connData[ 'ikeConnName' ] = connName
            connections[ key ] = connData

            traceVerbose( 'found conn ', bv( key.str() ), bv( connName ),
                          'state', bv( state ) )
            continue

         # this regexp looks for subsequent child SAs so that we can
         # grab the reqID to compare it to the one configured in the
         # ipsec.conf file.  This helps to detect race conditions
         # where a config file was changed around the same time that a
         # passive connection came up
         m = re.match( r'^([^ ]+)(\{[0-9]+\}):  ([A-Z]+)\,.*reqid ([0-9]+)', l )
         if m:
            ( connName, swanConnectionId, state, reqId ) = m.groups()
            reqId = int( reqId )
            # this is a subsequent line with a reqid on it
            key = self.parseStatusKey( connName )
            connData = connections.get( key )

            if not connData and not 'ROUTED' in l and \
               key.connId.connIdSrc == 'srcPolicy':
               connData = {}
               connData[ 'state' ] = state
               connData[ 'connectionId' ] = swanConnectionId
               connections[ key ] = connData

            if not connData:
               # now that the dpd 'drop' actions have been removed,
               # we should never see these any more.  Put in this
               # assert to validate it, and once that's determined
               # to be true we can remove the whole routedConn
               # concept from this.
               if 'ROUTED' in l:
                  assert key not in routedConn
                  routedConn[ key ] = True
                  traceVerbose( 'Found ROUTED conn' )
               else:
                  # not sure what other states go here, but...
                  traceVeryVerbose( 'ignoring child sa line with no ike sa',
                                    bv( l ) )
               continue

            connData = connections[ key ]
            traceVerbose( 'found reqId', bv( reqId ) )
            if reqId in connData:
               assert connData[ 'reqId' ] == reqId
            else:
               connections[ key ][ 'reqId' ] = reqId

      return ( connections, routedConn )

   def processStatusOutput( self, swanConnState, swanRouted ):
      needToRunAgain = False
      self.lastStatusResult = Tac.now()
      traceVerbose( 'vrf', bv( self.perVrfService.vrfName ) )

      policyBasedIkeConnectionsToReview = {}

      for ( key, conn ) in self.connection.items():
         traceVerbose( 'key ', bv( key.str() ), ' present: ',
                       bv( key in swanConnState ), ' state ', bv( conn.state ) )
         if key not in swanConnState:

            if key.connId.connIdSrc == 'srcPolicy':
               ikeKey = self.getIkeConnectionKey( key )
               if ikeKey in self.policyBasedVpnIkeSas:
                  self.policyBasedVpnIkeSas[ ikeKey ].discard( key )
               if ikeKey in swanConnState:
                  policyBasedIkeConnectionsToReview[ ikeKey ] = conn

            # strongswan does not know about the connection.
            if conn.state == 'down':
               continue

            # 'pendingStrokeUp' and pendingUp' this may just be transitional
            if conn.state in [ 'pendingUp', 'pendingStrokeUp' ]:
               needToRunAgain = True
               continue

            # it disappeared.  move our state to 'down' so that we can
            # re-evaluate what happens next time around
            if conn.state in [ 'pendingDown', 'pendingStrokeDown' ]:
               # The routed entries are expected for policy based connections.
               if key in swanRouted and key.connId.connIdSrc != 'srcPolicy':
                  # it's not quite gone away yet...
                  needToRunAgain = True
                  continue
               else:
                  conn.cleanupConnection( 'down', forced=False )
            else:
               assert conn.state in [ 'up', 'passiveUp', 'pendingChild' ]
               traceFunc( 'key ', bv( key.str() ), ' disappeared, moving ',
                          'state to down' )
               conn.cleanupConnection( 'down' )

            if conn.activeMode():
               self.addToBacklog( key, expedited=True )
               needToRunAgain = True

         else:
            # strongswan is up.  Note that we don't really care what
            # the strongswan internal state is other than that, the
            # assert is just sanity checking
            knownSwanStates = [ 'CREATED', 'CONNECTING', 'ESTABLISHED', 'PASSIVE',
                                'REKEYING', 'REKEYED', 'DELETING', 'DESTROYING',
                                'INSTALLED', 'DELETED' ]
            connData = swanConnState[ key ]
            traceVerbose( 'state ', bv( connData[ 'state' ] ) )
            assert connData[ 'state' ] in knownSwanStates
            swanReqId = connData[ 'reqId' ] if 'reqId' in connData else None
            curReqId = conn.curReqId()
            traceVerbose( 'swan reqId ', bv( swanReqId ), ' curReqId ',
                          bv( curReqId ) )

            # Rekeys will result in a new strongswan connection. Update
            # the latest swanConnectionId.
            if key.connId.connIdSrc == 'srcPolicy':
               conn.swanConnectionId = connData[ 'connectionId' ]

            if key.connId.connIdSrc == 'srcPolicy' and swanReqId and \
               swanReqId == curReqId:
               assert conn.state in [ 'down', 'pendingStrokeDown', 'pendingDown',
                                      'passiveUp' ]
               if conn.state == 'down' or conn.state == 'pendingDown':
                  traceVerbose( 'setting policy based vpn child sa to passive up',
                                bv( key.str() ) )
                  traceVerbose( 'conn state', bv( conn.state ) )
                  conn.stateIs( 'passiveUp' )
                  conn.swanConnectionId = connData[ 'connectionId' ]
                  ikeKey = self.getIkeConnectionKey( key )
                  assert ikeKey in swanConnState
                  ikeConnData = swanConnState[ ikeKey ]
                  conn.ikeConnName = ikeConnData[ 'ikeConnName' ]
                  if ikeKey not in self.policyBasedVpnIkeSas:
                     traceVerbose( 'creating policyBasedVpnIkeSas',
                                   bv( ikeKey.str() ) )
                     self.policyBasedVpnIkeSas[ ikeKey ] = set()
                  traceVerbose( 'adding key ', bv( key.str() ),
                                ' to ', bv( ikeKey.str() ) )
                  self.policyBasedVpnIkeSas[ ikeKey ].add( key )

            if conn.state in [ 'up', 'passiveUp', 'pendingChild' ]:
               if conn.state == 'pendingChild' and swanReqId:
                  if conn.activeMode():
                     conn.stateIs( 'up' )
                  else:
                     conn.stateIs( 'passiveUp' )

               if swanReqId and swanReqId != curReqId:
                  # reqId is out of date.  Force down
                  traceVerbose( 'reqId out of date, trying to force down.' )
                  # we only need to track the reqId here and not other
                  # attributes on the confConn becasue we know that
                  # the Ipsec agent always bumps the reqId when it
                  # writes a new conn
                  started = self.maybeStartProc( conn, 'down' )
                  needToRunAgain = True
                  if not started:
                     continue
               else:
                  # the normal, expected state.
                  continue

            if conn.state in [ 'pendingDown', 'pendingStrokeDown' ]:
               # we are waiting for processing to finish
               continue

            if conn.state == 'down':
               if conn.activeMode():
                  # if strongswan thinks the connection is up, but we
                  # think it's down, then something is out of sync.
                  # Force it down in strongswan so that we can resync.
                  traceVerbose( 'mismatch in state, force down',
                                bv( key.str() ) )
                  started = self.maybeStartProc( conn, 'down' )
                  if not started:
                     # since this action is only triggered based on
                     # the status output, if the 'down' cmd was
                     # deferred due to the process limit we need to
                     # trigger another status check
                     needToRunAgain = True
               else:
                  if swanReqId and swanReqId == curReqId:
                     conn.stateIs( 'passiveUp' )
                  else:
                     conn.stateIs( 'pendingChild' )

               continue

            if conn.state == 'pendingStrokeUp':
               # this means that we started a 'stroke up' process, but
               # we haven't processed its exit status yet.  Since strongswan
               # thinks it is up, let's do that now
               conn.poll()

               # seems that sometimes it shows up in the status out
               # before the stroke process has exited.  Fine, ignore
               # it, we'll just catch it the next time around
               if conn.pendingStroke():
                  needToRunAgain = True
                  continue
               elif conn.state == 'down':
                  # hm.  the exit status said it failed, but
                  # strongswan says it's up.  Not sure what happened
                  # there, but let's resync to make sure
                  traceVerbose( 'stroke exit status error, but status says up?' )
                  self.maybeStartProc( conn, 'down' )
                  needToRunAgain = True
                  continue

               # This means the stroke process exited successfully.
               # it should be in 'pendingUp', so fall through to that
               # case

            # the only case left, either fall through from
            # 'pendingStrokeUp' or just in here naturally
            assert conn.state == 'pendingUp'
            if swanReqId and swanReqId == curReqId:
               # we can skip 'pendingChild'
               traceVerbose( 'reqId is programmed, skipping pendingUp' )
               conn.stateIs( 'up' )
            else:
               traceVerbose( 'reqId is missing' )
               conn.stateIs( 'pendingChild' )
               needToRunAgain = True

      # Delete any Ike SAs without any Child Sa
      for ( ikeKey, conn ) in policyBasedIkeConnectionsToReview.items():
         if ikeKey not in self.policyBasedVpnIkeSas or \
            len( self.policyBasedVpnIkeSas[ ikeKey ] ) == 0:
            ikeConnData = swanConnState[ ikeKey ]
            ikeConnName = ikeConnData[ 'ikeConnName' ]
            traceVerbose( 'bridging down ikeConnection ', bv( ikeConnName ) )
            if not conn.pendingStroke():
               self.maybeStartProc( conn, 'down', cmdSuffix=ikeConnName )
            needToRunAgain = True

      if needToRunAgain:
         self.startStatusCheck()

      return

   def scheduleConnTimer( self, interval=1 ):
      newTime = Tac.now() + interval
      if newTime < self.connTimer.timeMin:
         self.connTimer.timeMin = newTime

   def newConn( self, key, confConn ):
      traceVerbose( 'key', bv( key.str() ) )
      pvs = self.perVrfService

      assert not pvs.daemonConfig.simulateDaemonForBtest

      connStatus = None
      if self.perVrfService.parent.canWriteToSysdb():
         connStatus = pvs.daemonStatus.connStatus.get( key )
         if not connStatus:
            connStatus = pvs.daemonStatus.newConnStatus( key )

      conn = Connection( key, self, connStatus, confConn, pvs.connName( key ),
                         pvs.netNsName, pvs.newConnId(),
                         pvs.secretGenId, self.baseDir, self.origStrongswanDir )
      self.connection[ key ] = conn
      conn.stateIs( 'down' )

      return conn

   def delConn( self, key ):
      traceVerbose( 'key', bv( key.str() ) )
      pvs = self.perVrfService
      assert not pvs.daemonConfig.simulateDaemonForBtest
      assert key in self.connection

      self.connection[ key ].cleanupConnection( 'down' )
      self.connection[ key ].connManager = None
      del self.connection[ key ]
      if pvs.parent.canWriteToSysdb():
         if key in pvs.daemonStatus.connStatus:
            del pvs.daemonStatus.connStatus[ key ]

   def numRunningProc( self ):
      numProc = 0
      for conn in self.connection.values():
         if conn.proc is not None:
            numProc += 1
      return numProc

   def maybeStartProc( self, conn, cmd, cmdSuffix=None ):
      traceFunc( 'key ', bv( conn.key.str() ), ' cmd ', bv( cmd ) )
      numProc = self.numRunningProc()
      if numProc >= self.procLimit:
         traceVerbose( 'Deferring process start.  numProc ', bv( numProc ),
                       ' limit is ', bv( self.procLimit ) )
         return False

      if conn.key.connId.connIdSrc == 'srcPolicy' and cmd == 'down' \
         and not cmdSuffix:
         ikeKey = self.getIkeConnectionKey( conn.key )
         # If this is the last child sa, bring down all sas including ike sa.
         if ikeKey not in self.policyBasedVpnIkeSas:
            cmdSuffix = conn.ikeConnName

      conn.start( cmd, cmdSuffix )
      return True


   # check a single connection and update it accordingly.  Return True
   # if processing is complete and key does not need to be reviewed again
   # next time, False otherwise.
   def processKey( self, key ):
      pvs = self.perVrfService
      confConn = pvs.getConnFromShadow( key )
      conn = self.connection.get( key )
      traceVerbose( 'processKey: ', bv( key.str() ), 'state',
                    bv( conn.state if conn else 'NO CONN' ) )

      if key == pvs.confData.defaultConnectionKey( pvs.vrfId ):
         return True

      assert not pvs.daemonConfig.simulateDaemonForBtest

      if not confConn:
         if not conn:
            return True

         ikeKey = self.getIkeConnectionKey( key )
         traceVerbose( 'ikeKey: ', bv( ikeKey.str() ) )

         if ikeKey in self.policyBasedVpnIkeSas:
            self.policyBasedVpnIkeSas[ ikeKey ].discard( key )
         if ikeKey in self.policyBasedVpnIkeSas and \
            not len( self.policyBasedVpnIkeSas[ ikeKey ] ):
            traceVerbose( 'delete policyBasedVpnIkeSas: ', bv( ikeKey.str() ) )
            del self.policyBasedVpnIkeSas[ ikeKey ]

         return self.processKeyDown( key, conn )

      assert confConn

      if not conn:
         if key not in self.connectionsInConfFile:
            # this means that we did a review pass during the "sync"
            # delay between the time that the connection was added to
            # the confData and when the SuperServer timer fired off.
            # Ignore it for now because we'll catch it during
            # the handleConfWritten() call after the sync.
            return False

         conn = self.newConn( key, confConn )

      if conn.pendingTimeout():
         traceVerbose( 'state timed out, started at', bv( conn.stateChangedAt ),
                       'now', bv( Tac.now() ) )

         if conn.state in [ 'pendingStrokeUp', 'pendingUp', 'pendingChild' ]:
            # force it to be up and start bringing it down, since we
            # may need to clear the stored config in strongswan
            conn.cleanupConnection( 'up' )
            conn.start( 'down' )
         else:
            assert conn.state in [ 'pendingStrokeDown', 'pendingDown' ]
            conn.cleanupConnection( 'up' )

         return False

      if conn.pendingStroke():
         conn.poll()

         # if we're still waiting for a stroke process to complete
         # then we're done, but we need to run again
         if conn.pendingStroke():
            return False

      # at this point we know any stroke commands have completed,
      # check if we need to trigger a 'status' poll
      if conn.state in [ 'pendingUp', 'pendingDown' ]:
         self.startStatusCheck()
         return False

      if conn.needsReload():
         traceVerbose( 'trying to bring connection down for reload' )
         self.maybeStartProc( conn, 'down' )
         return False

      if conn.state in [ 'pendingChild' ]:
         # this also needs a status check, but can't be combined with
         # pendingUp/pendingDown because we needed to do the reload
         # check first
         self.startStatusCheck()
         return False

      if conn.state == 'down' and conn.activeMode():
         self.maybeStartProc( conn, 'up' )
         return False

      traceVeryVerbose( 'fall through to done, state is', bv( conn.state ) )
      return True

   def processKeyDown( self, key, conn ):
      traceFunc( 'key ', bv( key.str() ), ' conn state ', bv( conn.state ) )
      if conn.state == 'down':
         # we have a stub StrokeProc but no underlying connection
         # in strongswan, so just clean it up and we're done.
         assert not conn.proc
         self.delConn( key )
         return True

      if conn.state in [ 'pendingStrokeDown' ]:
         # Even if confData is missing , we need to poll the stroke command
         # and move the conn state accordingly
         conn.poll()
         # stroke down is success , but we still need to check connection
         # going down from stroke status
         if conn.state in [ 'pendingDown' ]:
            self.startStatusCheck()

         # Even if stroke down is still pending or it has failed,
         # we need to readd this connKey back to backlog
         return False

      if conn.state in [ 'pendingDown' ]:
         # this means we're waiting for a status update to confirm
         # that the connection is down.
         return False

      # if the process is in any stage of up then we need to clean out
      # the old process and start a new one to bring it down
      if conn.state in [ 'up', 'pendingUp', 'pendingStrokeUp', 'passiveUp',
                         'pendingChild' ]:
         if conn.proc:
            conn.cleanupConnection( 'up' )

         started = self.maybeStartProc( conn, 'down' )
         if not started:
            # need to circle around and try again later
            return False

         assert conn.state == 'pendingStrokeDown'
         # fall through to poll it

      assert conn.state == 'pendingStrokeDown'
      conn.poll()
      if conn.pendingStroke():
         # not done yet
         return False

      # is it possible for the 'down' command to fail?  If so, then it
      # might still be in state 'up'?
      assert conn.state == 'pendingDown'
      return False

   # called when we become active, we need to push the connStatus objects to
   # the connections
   def onSwitchover( self ):
      traceVerbose( 'vrf', bv( self.perVrfService.vrfName ) )
      pvs = self.perVrfService
      assert pvs.parent.canWriteToSysdb()

      if pvs.daemonConfig.simulateDaemonForBtest:
         return

      allKeys = set( self.connection.keys() )
      allKeys.update( set( pvs.daemonStatus.connStatus.keys() ) )

      for key in allKeys:
         conn = self.connection.get( key )
         connStatus = pvs.daemonStatus.connStatus.get( key )

         if not conn:
            if connStatus:
               pvs.daemonStatus.delConnStatus( key )
               connStatus = None
            return

         assert conn
         if not connStatus:
            connStatus = pvs.daemonStatus.newConnStatus( key )
         assert connStatus
         if conn.connStatus != connStatus:
            conn.connStatus = connStatus
            conn.syncStateToSysdb()
            # note that we are not synchronizing the up/down/etc cmd
            # counts into sysdb on switchover.  Arguably this is a
            # bug, but at the moment they're only there for
            # debug/visibility purposes

      self.addAllKeysToBacklog()

   # used when a VRF is deleted or when we shut down strongswan in
   # that VRF for some reason
   def cleanupConnManager( self ):
      traceVerbose( 'vrf', bv( self.perVrfService.vrfName ) )

      self.connTimer.timeMin = Tac.endOfTime

      if self.perVrfService.daemonConfig.simulateDaemonForBtest:
         return

      allKeys = list( self.connection.keys() )
      for key in allKeys:
         self.delConn( key )

      if self.rereadSecretsProc:
         self.rereadSecretsProc.kill()
         self.rereadSecretsProc = None

      if self.rereadCertsProc:
         self.rereadCertsProc.kill()
         self.rereadCertsProc = None

      if self.purgeCrlsProc:
         self.purgeCrlsProc.kill()
         self.purgeCrlsProc = None

   # this isn't a connection, but it uses 'stroke' and we need to
   # monitor the process in the same way, so let's piggyback it
   def rereadSecrets( self ):
      traceVerbose( 'vrf', bv( self.perVrfService.vrfName ) )

      if self.perVrfService.daemonConfig.simulateDaemonForBtest:
         return

      if self.rereadSecretsProc:
         self.rereadSecretsProc.kill()

      self.rereadSecretsProc = StrokeProc(
         key='rereadsecrets', cmd='rereadsecrets', cmdSuffix=None,
         netNsName=self.perVrfService.netNsName,
         baseDir=self.baseDir,
         origStrongswanDir=self.origStrongswanDir )
      self.rereadSecretsProc.run()

      self.addToBacklog( 'rereadSecrets' )
      self.scheduleConnTimer()

   def rereadCerts( self ):
      traceVerbose( 'vrf', bv( self.perVrfService.vrfName ) )

      if self.perVrfService.daemonConfig.simulateDaemonForBtest:
         return

      if self.rereadCertsProc:
         self.rereadCertsProc.kill()

      self.rereadCertsProc = StrokeProc(
         key='rereadCerts', cmd='rereadall', cmdSuffix=None,
         netNsName=self.perVrfService.netNsName,
         baseDir=self.baseDir,
         origStrongswanDir=self.origStrongswanDir )
      self.rereadCertsProc.run()

      self.addToBacklog( 'rereadCerts' )
      self.scheduleConnTimer()

   def purgeCrls( self ):
      traceVerbose( 'vrf', bv( self.perVrfService.vrfName ) )

      if self.perVrfService.daemonConfig.simulateDaemonForBtest:
         return

      if self.purgeCrlsProc:
         self.purgeCrlsProc.kill()

      self.purgeCrlsProc = StrokeProc(
         key='purgeCrls', cmd='purgecrls', cmdSuffix=None,
         netNsName=self.perVrfService.netNsName,
         baseDir=self.baseDir,
         origStrongswanDir=self.origStrongswanDir )
      self.purgeCrlsProc.run()

      self.addToBacklog( 'purgeCrls' )
      self.scheduleConnTimer()

   # this is called when the PerVrfService writes the ipsec.conf file
   # out
   def handleConfWritten( self, cfg ):
      traceVerbose( 'vrf', bv( self.perVrfService.vrfName ) )

      if self.perVrfService.daemonConfig.simulateDaemonForBtest:
         return

      for key in self.perVrfService.shadowConfData:
         confConn = self.perVrfService.getConnFromShadow( key )

         conn = self.connection.get( key )
         if not conn:
            self.addToBacklog( key, expedited=True )
         else:
            if conn.maybeUpdateConfData( confConn ):
               self.addToBacklog( key, expedited=True )

      if self.nextReviewBacklog:
         self.scheduleConnTimer()

      self.lastCfg = cfg

   def startStatusCheck( self, scheduleTimer=True ):
      traceVerbose( 'vrf', bv( self.perVrfService.vrfName ) )
      self.statusDirty = True
      self.addToBacklog( 'status' )
      if scheduleTimer:
         self.scheduleConnTimer()

   def maybeStartStatusCheck( self ):
      if not self.perVrfService.serviceEnabled():
         return

      secondsSinceLastRun = Tac.now() - self.lastStatusResult
      traceTimer( 'vrf ', bv( self.perVrfService.vrfName ),
                  ' time ', bv( secondsSinceLastRun ) )
      if self.statusDirty or self.statusProc is not None:
         traceTimer( 'no need, dirty: ', bv( self.statusDirty ),
                     ' proc exists is ', bv( self.statusProc is not None ) )
         return

      if secondsSinceLastRun > self.minStatusInterval:
         self.startStatusCheck()

#
# We create one of these objects for each VRF, it manages the mount
# namespace, config files, and main strongswan process for that VRF.
# It also uses a ConnectionManager (see above) to bring connections
# up/down as needed.
#
# Note that this is only sort-of a SuperServer.LinuxService, because
# we aren't using 'service <name> up' style commands to manage it,
# instead we override startService/etc and invoke strongswan directly.
# This avoids needing to write init.d scripts for each VRF, and also
# sidesteps some compatiblity problems between sysv init and systemd
# files on namespace duts.  We also override the healthCheck code in
# the LinuxService functions to do that manually.  What I'm really
# after is to avoid reimplementing the config file difference/etc
# checking logic in LinuxService.
#
# Arguably it might be worth refactoring the classes in SuperServer so
# that we can get just the config file code that we want without the
# other stuff.  A project for another day.
#
class IpsecPerVrfService( SuperServer.LinuxService ):
   notifierTypeName = 'Ipsec::Conf::ConfData'

   def __init__( self, vrfId, vrfName, confData, parent, daemonConfig, daemonStatus,
                 routingHwStatus, sslStatus, templateDir, baseDir, origStrongswanDir,
                 netNsName, serviceRestartDelay ):

      traceFunc( 'vrfId: ', bv( vrfId ), ' vrfName: ', bv( vrfName ) )

      self.vrfId = vrfId
      self.vrfName = vrfName
      self.parent = parent
      self.daemonConfig = daemonConfig
      self.daemonStatus = daemonStatus
      self.confData = confData
      self.routingHwStatus = routingHwStatus
      self.sslStatus = sslStatus
      self.templateDir = templateDir
      self.baseDir = baseDir
      self.origStrongswanDir = origStrongswanDir
      self.netNsName = netNsName
      self.cleanupState = 'not started'
      self.deleted = False

      assert '/' not in vrfName

      self.serviceName = f'strongswan-VRF-{vrfName}'

      runDir = '%s/ipsec.d/run' % self.baseDir
      self.starterPidFile = '%s/starter.charon.pid' % runDir
      self.charonPidFile = '%s/charon.pid' % runDir
      self.charonCtlFile = '%s/charon.ctl' % runDir
      self.logFileName = '/var/log/agents/charon-VRF-%s.log' % vrfName
      self.strongswanConfFileName = '%s/strongswan.conf' % self.baseDir
      self.ipsecConfFileName = '%s/ipsec.conf' % self.baseDir
      self.ipsecSecretsFileName = '%s/ipsec.secrets' % self.baseDir

      # setting these to 4 will make strongswan produce some very
      # verbose logs, which is useful for debugging.  Too spammy for
      # production though.
      self.defLogLevel = 1
      self.knlLogLevel = 2
      self.ikeLogLevel = 2
      self.cfgLogLevel = 1

      # The original version of this code would pull the openssl
      # config file name out of the environment variable
      # 'STRONGSWAN_CHARON_OPENSSL_CONF'.  AFAICT we never actually
      # use that, though, so I left it out here.
      self.openSslConfFileName = f'{self.baseDir}/strongswan.d/charon/openssl.conf'

      fileNameList = [ self.strongswanConfFileName, self.ipsecConfFileName,
                       self.ipsecSecretsFileName, self.openSslConfFileName ]

      self.simulatedRunning = False
      self.connectionManager = ConnectionManager( self, self.baseDir,
                                                  self.origStrongswanDir )
      assert self.connectionManager

      # PKI files maps, these are needed to delete the unused pki files from
      # strongswan directories
      # Following map is keyed by connectionKey with value as
      # {
      #    'certificate': set(),
      #    'privateKey': set(),
      #    'trustedCerts': set(),
      #    'crls': set()
      # }
      self.connectionToPkiFilesMap = {}
      # Following map is keyed by pki filename with value as
      # {
      #    'key': key, <- this is the type of pki file, like crl etc
      #    'connectionKeys': set()
      # }
      self.pkiFileToConnectionsMap = {}

      self.createVrfFiles()
      if self.parent.canWriteToSysdb():
         self.daemonStatus.connStatus.clear()
         self.updateDaemonStatus()

      serviceName = 'strongswan'
      linuxServiceName = f'{serviceName}-VRF-{vrfName}'
      SuperServer.LinuxService.__init__( self, serviceName, linuxServiceName,
                                         confData, confFilenames=fileNameList,
                                         serviceRestartDelay=serviceRestartDelay,
                                         healthCheckNeeded=False )

      # this is our reimplementation of the health check
      self.healthCheckTimer = Tac.ClockNotifiee()
      self.healthCheckTimer.handler = self.handleHealthCheckTimer
      self.healthCheckInterval = 15
      # wait at least this long after starting the service before
      # complaining about it not running
      self.startHealthDelay = 5

      self.scheduleHealthCheck()
      self.serviceStartedAt = Tac.beginningOfTime

      self.secretShadow = dict()
      self.secretGenId = 1
      self.prepopulateSecretShadow()

      self.shadowConfData = None
      self.updateShadowConfData()

   def _maybeRestartService( self ):
      if self.deleted:
         traceFunc( "This service for", bv( self.vrfName ),
                    "is deleted and should be garbage collected shortly" )
         return
      super()._maybeRestartService()

   def onSwitchover( self ):
      traceFunc( 'vrf: ', bv( self.vrfName ) )
      self.updateDaemonStatus()
      self.connectionManager.onSwitchover()

   def needsHupIs( self, b ):
      assert b
      pid = getPid( self.starterPidFile )
      traceFunc( 'reload ipsec.conf for ', bv( pid ) )
      os.kill( pid, signal.SIGHUP )
      # this is called from within the connectionManager, so we let it
      # decide whether or not to restart the conn timer

   def needsReloadIs( self, b ):
      traceFunc()
      assert b
      if self.isRunning():
         self.stopService()
      self.startService()

   def newConnId( self ):
      return self.parent.newConnId()

   def createVrfFiles( self ):
      traceFunc( bv( self.vrfName ) )

      # we are going to copy the entire template directory of
      # strongswan config files into a per-VRF location.  First we
      # need to empty out that location in case there are any leftover
      # files there.  Again, this would change if/when we support
      # SuperServer agent restart
      try:
         shutil.rmtree( self.baseDir )
      except OSError as e:
         if e.errno != errno.ENOENT:
            traceVerbose( 'init cleanup rm gave', bv( e.strerror ) )

      # and now create the new base directory
      try:
         shutil.copytree( self.templateDir, self.baseDir )
      except OSError as e:
         traceVerbose( 'copying', bv( self.templateDir ), 'to', bv( self.baseDir ),
                       'gave', bv( e.strerror ) )

      for connectionKey in self.confData.connection.keys():
         self.copyPkiFiles( connectionKey )

   def addNetNsPrefix( self, cmd ):
      netNsCmd = cmd

      if self.vrfName != DEFAULT_VRF:
         netNsCmd = f'ip netns exec {self.netNsName} {netNsCmd}'

      return netNsCmd

   # this is called when the VRF is deleted
   def cleanupService( self ):
      traceFunc( 'vrf: ', bv( self.vrfName ) )

      SuperServer.LinuxService.cleanupService( self )
      self.healthCheckTimer.timeMin = Tac.endOfTime

      self.cleanupState = 'started'
      self.stopService()
      self.connectionManager.cleanupConnManager()

      # make sure we get rid of circular reference so that GC can
      # clean up the objects
      self.connectionManager.perVrfService = None
      del self.connectionManager
      self.connectionManager = None

      try:
         shutil.rmtree( self.baseDir )
      except OSError as e:
         if e.errno != errno.ENOENT:
            traceVerbose( 'rmtree', bv( self.baseDir ), 'gave', bv( e.strerror ) )
      self.cleanupState = 'done'
      traceVerbose( 'cleanup done', bv( self.vrfName ) )

   # push the current daemon state into sysdb if allowed
   def updateDaemonStatus( self ):
      traceFunc( bv( self.vrfName ), ' btest: ',
                 bv( self.daemonConfig.simulateDaemonForBtest ), ' sim: ',
                 bv( self.simulatedRunning ) )

      if not self.parent.canWriteToSysdb():
         return

      self.daemonStatus.fipsMode = self.parent.ipsecConfig.fipsRestrictions
      if self.daemonConfig.simulateDaemonForBtest:
         runValue = self.simulatedRunning
      else:
         runValue = self.isRunning()
         traceVerbose( 'set daemonStatus sswanRunning to', bv( runValue ) )
      self.daemonStatus.sswanRunning = runValue

   def serviceEnabled( self ):
      if( not self.parent.ipsecStatus.ipsecLicenseEnabled or
          not self.parent.daemonConfig.enableSswan ):
         return False

      if len( self.shadowConfData ) == 0:
         return False

      # if we only have one connection and it's the default one then
      # we don't need to run the daemon
      if len( self.shadowConfData ) == 1:
         key = list( self.shadowConfData.keys() )[ 0 ]
         if key == self.confData.defaultConnectionKey( self.vrfId ):
            return False

      return True

   def serviceProcessWarm( self ):
      # The agent is warm if it is not enabled
      if not self.serviceEnabled():
         warm = True
      else:
         running = self.isRunning()
         warm = running

      traceVerbose( bv( warm ) )
      return warm

   # called by all of the actual tacc handlers, because on any change
   # we just call sync()
   def handleConfigData( self ):
      traceFunc( bv( self.vrfName ) )
      self.sync()
      return

   def isRunning( self ):
      # to be 'running', both the 'starter.charon.pid' and
      # 'charon.pid' processes need to be running.  Additionally, we
      # need to verify that there isn't some other process on the
      # system that is reusing the PID, so we also have to check that
      # both the command line of the process matches what we expect
      # and that the network namespace matches the VRF in question (so
      # we don't get confused by looking at a starter or charon
      # process in a different VRF)
      starterPid = getPid( self.starterPidFile )
      charonPid = getPid( self.charonPidFile )

      vrfNs = getNetNsInode( pid=None, vrfName=self.vrfName )
      for ( pid, procName ) in ( [ starterPid, 'starter' ],
                                 [ charonPid, 'charon' ] ):
         if not pidExists( pid ):
            return False

         f = None
         try:
            f = open( '/proc/%d/cmdline' % pid )
            cmdline = f.readline()
         except ( OSError, ProcessLookupError ):
            return False
         if not cmdline.startswith( '/usr/libexec/strongswan/%s' % procName ):
            return False

         netNs = getNetNsInode( pid )
         if not netNs or netNs != vrfNs:
            return False

      return True

   def _runServiceCmd( self, cmd ):
      # we are not using this, it should never get called
      assert 0

   # override the default LinuxService startService() command becasue
   # we want to invoke the strongswan starter process directly rather
   # than using scripts in /etc/init.d
   def startService( self ):
      assert self.serviceEnabled()
      traceFunc( bv( self.vrfName ), ' vrfid: ', bv( self.vrfId ) )

      if self.isRunning():
         traceVerbose( 'already running' )
         return

      if not os.path.exists( self.baseDir ) or not os.path.isdir( self.baseDir ):
         traceVerbose( 'sswan VRF template files not created,return' )
         return

      if self.daemonConfig.simulateDaemonForBtest:
         self.simulatedRunning = True
         self.updateDaemonStatus()
         return

      # this is sort of paranoia, we should have cleaned it up before
      # we shut down, but just in case...
      self.connectionManager.cleanupConnManager()

      assert not self.connectionManager.connection

      # make sure we don't have any leftover files from the previous instance
      for f in ( self.starterPidFile, self.charonPidFile, self.charonCtlFile ):
         try:
            os.unlink( f )
         except OSError:
            pass

      starterCmd = '/usr/libexec/strongswan/starter --daemon charon'
      cmd = self.addNetNsPrefix( starterCmd )
      traceVerbose( 'starter cmd ', bv( cmd ) )
      with RunInMntNamespace( self.baseDir, self.origStrongswanDir ):
         os.system( cmd )

      self.updateDaemonStatus()

      # tell the connectionManager to run so that it can re-create the
      # connections
      self.connectionManager.addAllKeysToBacklog()

      self.serviceStartedAt = Tac.now()

      # This is to address bug/866902, where in secrets genId
      # in existing shadow data wont be updated if the strongswan
      # restarts with ipsec.secrets file already populated as we
      # wont be initiating a rereadsecrets again
      # Hence update the genId of these secrete shadowData as strongswan
      # start would have take ipsec.secrets into consideration already
      self.markSecretsWritten()


   # override the default stopService for the same reason
   def stopService( self ):
      traceFunc( bv( self.vrfName ), 'vrfid: ', bv( self.vrfId ) )

      self.connectionManager.cleanupConnManager()

      if self.daemonConfig.simulateDaemonForBtest:
         self.simulatedRunning = False
         self.updateDaemonStatus()
         return

      starterPid = getPid( self.starterPidFile )
      charonPid = getPid( self.charonPidFile )
      if pidExists( starterPid ):
         killProc( starterPid, 'stopService starter' )
      if pidExists( charonPid ):
         killProc( charonPid, 'stopService charon' )

      self.updateDaemonStatus()

      self.parent.cleanupOldStrongswanInstances( self.vrfName )

   def restartService( self ):
      # we don't use this (restartWithFileChanges instead), so assert
      assert 0

   # override the default restartService becasue we need to figure out
   # exactly what to change
   def restartWithFileChanges( self, oldConfig, newConfig ):
      traceFunc( bv( self.vrfName ), 'vrfid: ', bv( self.vrfId ) )

      if self.daemonConfig.simulateDaemonForBtest:
         self.simulatedRunning = True

      if not self.isRunning():
         traceVerbose( 'service not running, falling back to stop/startService' )
         self.stopService()
         self.startService()
         return

      if( ( oldConfig[ self.openSslConfFileName ] !=
            newConfig[ self.openSslConfFileName ] ) or
          ( oldConfig[ self.strongswanConfFileName ] !=
            newConfig[ self.strongswanConfFileName ] ) ):
         # changing strongswan.conf or openssl.conf requires
         # restarting the whole thing -- we have to restart the whole
         # thing.

         if self.parent.extraDebug:
            if( oldConfig[ self.openSslConfFileName ] !=
                newConfig[ self.openSslConfFileName ] ):
               traceVerbose( 'ssl changed' )
               for line in oldConfig[ self.openSslConfFileName ].split( '\n' ):
                  traceVerbose( 'old:', bv( line ) )
               for line in newConfig[ self.openSslConfFileName ].split( '\n' ):
                  traceVerbose( 'new:', bv( line ) )

            if( oldConfig[ self.strongswanConfFileName ] !=
                newConfig[ self.strongswanConfFileName ] ):
               traceVerbose( 'ssl changed' )
               for line in oldConfig[ self.strongswanConfFileName ].split( '\n' ):
                  traceVerbose( 'old:', bv( line ) )
               for line in newConfig[ self.strongswanConfFileName ].split( '\n' ):
                  traceVerbose( 'new:', bv( line ) )

         traceVerbose( 'Forced full restart' )
         self.stopService()
         self.startService()
         return

      # if the ipsec.secrets file changed then we can use the rereadsecrets
      # stroke command to reload it
      if( oldConfig[ self.ipsecSecretsFileName ] !=
          newConfig[ self.ipsecSecretsFileName ] ):
         traceVerbose( 'rereading secrets file' )
         self.connectionManager.rereadSecrets()

      # for the ipsec.conf file, we can reload all of the not
      # currently active connections by HUPing the process
      if( oldConfig[ self.ipsecConfFileName ] !=
          newConfig[ self.ipsecConfFileName ] ):

         pid = getPid( self.starterPidFile )
         traceVerbose( 'HUP to reload ipsec.conf', bv( pid ) )
         os.kill( pid, signal.SIGHUP )

         # individual connections that are already up will need to be
         # reloaded if they changed.  The ConnectionManager will
         # figure that out based on the currentConfConn that we loaded
         # into them during the conf() call, so we just need to set it
         # running
         self.connectionManager.addAllKeysToBacklog()

      self.restarts += 1

   def handleHealthCheckTimer( self ):
      self.healthCheckTimer.timeMin = Tac.endOfTime
      assert self.cleanupState == 'not started'
      traceTimer( 'vrf ', bv( self.vrfName ) )

      now = Tac.now()
      if( self.serviceEnabled() and
          ( now - self.serviceStartedAt ) > self.startHealthDelay ):

         failStr = None
         if not self.isRunning() and not self.serviceRestartPending_:
            traceFunc( 'IPSEC health check restart', bv( self.vrfName ) )
            traceVerbose( 'serviceStartedAt', bv( self.serviceStartedAt ), 'now',
                          bv( now ), 'delay', bv( self.startHealthDelay ) )
            self.stopService()
            self.startService()

            failStr = f'strongswan-VRF-{self.vrfName}'

         if failStr:
            self.logRestart( failStr )

      self.connectionManager.maybeStartStatusCheck()
      self.connectionManager.addAllKeysToBacklog()
      self.scheduleHealthCheck()

   def scheduleHealthCheck( self ):
      if self.daemonConfig.simulateDaemonForBtest:
         traceTimer( 'no health check due to btest simulation' )
         return

      self.healthCheckTimer.timeMin = Tac.now() + self.healthCheckInterval

   ######################################################################
   # this set of functions are used to create the config files

   def conf( self ):
      assert self.cleanupState == 'not started'

      traceFunc( bv( self.vrfName ), 'vrfId: ', bv( self.vrfId ) )
      self.updateShadowConfData()

      d = dict()
      traceFunc( 'files are', bv( self.ipsecConfFileName ),
                 bv( self.ipsecSecretsFileName ), bv( self.strongswanConfFileName ),
                 bv( self.openSslConfFileName ) )
      d[ self.ipsecConfFileName ] = self.ipsecConf()
      d[ self.ipsecSecretsFileName ] = self.ipsecSecrets()
      d[ self.strongswanConfFileName ] = self.strongswanConf()
      d[ self.openSslConfFileName ] = self.openSslConf()

      return d

   # the various map... functions convert an EOS enum or builtin type
   # to the string that Strongswan is expecting
   def mapSeconds( self, seconds ):
      return f'{int(seconds)}s'

   def mapIkeVersion( self, version ):
      if version == 'ikeVersion1':
         return 'ikev1'
      elif version == 'ikeVersion2':
         return 'ikev2'
      else:
         traceVerbose( 'unknown ike version', bv( version ) )
         return 'unknown'

   def mapConnectionStart( self, cStart ):
      # we are not using this, instead we configure everything as
      # 'add' and then use the SuperServerPlugin to stroke things up
      # instead of having strongswan do it automatically
      assert 0
      d = { 'connectionStartIgnore': 'ignore', 'connectionStartAdd': 'add',
            'connectionStartRoute': 'route', 'connectionStartAutoStart': 'start' }
      if cStart not in d:
         traceVerbose( 'unknown cStart value', bv( cStart ) )
         return 'unknown'
      return d[ cStart ]

   def mapDpdAction( self, a ):
      # this map is a unity transform, but it at least validates that
      # we have an expected value
      d = { 'none': 'none', 'clear': 'clear', 'hold': 'hold', 'restart': 'restart' }
      if a not in d:
         traceVerbose( 'unknown dpdAction values', bv( a ) )
         return 'unknown'
      return d[ a ]

   def mapBool( self, b ):
      return 'yes' if b else 'no'

   def mapAuthBy( self, ab ):
      if ab == 'pre_share':
         return 'secret'
      elif ab == 'rsa_sig':
         return 'rsasig'
      elif ab == 'pki':
         return 'pubkey'
      else:
         traceVerbose( 'unknown authBy value', bv( ab ) )
         return 'unknown'

   def mapConnectionType( self, t ):
      d = { 'connTypeTunnel': 'tunnel', 'connTypeTransport': 'transport',
            'connTypeTransportProxy': 'transportproxy',
            'connTypePassthrough': 'passthrough', 'connTypeDrop': 'drop' }
      if t not in d:
         traceVerbose( 'unknown map connection type', bv( t ) )
         return 'unknown'
      return d[ t ]

   def mapEspAlg( self, alg ):
      d = { 'des': '3des',
            'aes128': 'aes128',
            'aes192': 'aes192',
            'aes128gcm64' : 'aes128gcm64',
            'aes128gcm128' : 'aes128gcm128',
            'aes256gcm128' : 'aes256gcm128',
            'aes256' : 'aes256',
            'nullesp' : 'null' }
      if alg not in d:
         traceVerbose( 'unknown ESP algorithm', bv( alg ) )
         return 'unknown'
      return d[ alg ]

   def mapHmacAlg( self, alg ):
      d = { 'md5' : 'md5_128',
            'sha1' : 'sha1',
            'sha256' : 'sha256',
            'sha384' : 'sha384',
            'sha512' : 'sha512',
            'nullhash' : 'null' }
      if alg not in d:
         traceVerbose( 'unknown HMAC algorithm', bv( alg ) )
         return 'unknown'
      return d[ alg ]

   def mapDhGroup( self, dh ) :
      d = { 'modp768' : 'modp768',
            'modp1024' : 'modp1024',
            'modp1536' : 'modp1536',
            'modp2048' : 'modp2048',
            'modp3072' : 'modp3072',
            'modp4096' : 'modp4096',
            'modp6144' : 'modp6144',
            'ecp256' : 'ecp256',
            'ecp384' : 'ecp384',
            'ecp521' : 'ecp521',
            'modp2048s256' : 'modp2048s256' }
      if dh not in d:
         traceVerbose( 'unknown DH group', bv( dh ) )
         return 'unknown'
      return d[ dh ]

   def mapCipherSuite( self, cs, isDefault ):
      config = ''
      proposals = []
      for index in cs:
         proposal = ''
         if cs[ index ].encryption is not None:
            proposal += self.mapEspAlg( cs[ index ].encryption )
         if cs[ index ].integrity is not None:
            proposal += f'-{self.mapHmacAlg( cs[ index ].integrity )}'
         if cs[ index ].dhGroup is not None:
            proposal += f'-{self.mapDhGroup( cs[ index ].dhGroup )}'
         proposals.append( proposal )

      config = ','.join( proposals )

      # BUG690031 figure out when/why we want a ! here.  The old code
      # leaves it off for default connections and puts it on for all
      # of the others.  I can't even figure out what it means in the
      # strongswan docs?
      if not isDefault:
         config += '!'
      return config

   def ipsecConfForEndPoint( self, fmt, endPointStr, endPoint, authBy=None ):
      config = ''
      if endPoint.addr is not None:
         config += fmt % ( endPointStr, '', endPoint.addr )
      if endPoint.subnet is not None:
         config += fmt % ( endPointStr, 'subnet', endPoint.subnet )
      if endPoint.addrIdentify:
         ikeId = None
         if endPoint.addrIdentify.containsIp():
            ikeId = endPoint.addrIdentify.ip.stringValue
         elif endPoint.addrIdentify.containsFqdn():
            ikeId = endPoint.addrIdentify.fqdn
         config += fmt % ( endPointStr, 'id', ikeId )
      elif authBy == "pki":
         # If no explicit id is configured for PKI, use id as % to inform strongswan
         # to not sent IDr as part of IKE negotiation
         config += fmt % ( endPointStr, 'id', '%' )
      if endPoint.cert is not None:
         config += fmt % ( endPointStr, 'cert', endPoint.cert )
      if endPoint.port is not None:
         config += fmt % ( endPointStr, 'protoport', endPoint.port )
      return config

   def ipsecConfForConnection( self, connName, key, isDefault=False ):
      config = ''
      conn = self.getConnFromShadow( key )
      if not conn:
         traceVerbose( 'invalid key: (VRF:', bv( key.vrfId ), 'SA:',
                bv( key.srcAddr ), 'DA:', bv( key.dstAddr ), 'cIDEnum',
                bv( key.connId.connIdSrc ), 'cID', bv( key.connId.connId ), ')' )
         return ''

      config = f'conn {connName}\n'
      prefix = '    '
      fmt = prefix + '%s = %s\n'
      endPointLineFmt = prefix + '%s%s = %s\n'

      if conn.keyExchange is not None:
         config += fmt % ( 'keyexchange', self.mapIkeVersion( conn.keyExchange ) )
      if conn.authBy is not None:
         config += fmt % ( 'authby', self.mapAuthBy( conn.authBy ) )
      if conn.lifetime is not None:
         config += fmt % ( 'lifetime', self.mapSeconds( conn.lifetime ) )
      if conn.ikeLifetime is not None:
         config += fmt % ( 'ikelifetime', self.mapSeconds( conn.ikeLifetime ) )
      if conn.rekeyMargin is not None:
         config += fmt % ( 'rekeymargin', self.mapSeconds( conn.rekeyMargin ) )
      if conn.keyingTries is not None:
         config += fmt % ( 'keyingtries', conn.keyingTries )
      if conn.mobikeEnabled is not None:
         config += fmt % ( 'mobike', self.mapBool( conn.mobikeEnabled ) )

      if conn.connStart == 'connectionStartRoute':
         assert Toggles.PolicyMapToggleLib.togglePolicyBasedVpnEnabled()
         # For policy based connections, we set auto to route.
         config += fmt % ( 'auto', 'route' )
      else:
         # one might expect that we would write out the 'connStart' variable as
         # as 'start' or 'add' here, but in fact we want to control bringing up
         # the 'start' connections using 'stroke up' rather than having
         # strongswan do it.  So we write them both out as 'add'
         config += fmt % ( 'auto', 'add' )

      if len( conn.espCipher ):
         config += fmt % ( 'esp',
                           self.mapCipherSuite( conn.espCipher, isDefault ) )
      if len( conn.ikeCipher ):
         config += fmt % ( 'ike',
                           self.mapCipherSuite( conn.ikeCipher, isDefault ) )
      if conn.lifePackets is not None:
         config += fmt % ( 'lifepackets', conn.lifePackets )
      if conn.lifeBytes is not None:
         config += fmt % ( 'lifebytes', conn.lifeBytes )
      if conn.marginPackets is not None:
         config += fmt % ( 'marginpackets', conn.marginPackets )
      if conn.marginBytes is not None:
         config += fmt % ( 'marginbytes', conn.marginBytes )
      if conn.reqId is not None:
         config += fmt % ( 'reqid', conn.reqId )
      if conn.mark is not None:
         config += fmt % ( 'mark', conn.mark )
      if conn.replayWindow is not None:
         config += fmt % ( 'replay_window', conn.replayWindow )

      if conn.dpdAction is not None:
         config += fmt % ( 'dpdaction', self.mapDpdAction( conn.dpdAction ) )
      if conn.dpdTimeout is not None:
         config += fmt % ( 'dpdtimeout', self.mapSeconds( conn.dpdTimeout ) )
      if conn.dpdInterval is not None:
         config += fmt % ( 'dpddelay', self.mapSeconds( conn.dpdInterval ) )

      config += self.ipsecConfForEndPoint( endPointLineFmt, 'left', conn.left )
      config += self.ipsecConfForEndPoint( endPointLineFmt, 'right', conn.right,
                                           authBy=conn.authBy )

      if conn.forceEncaps is not None:
         config += fmt % ( 'forceencaps', self.mapBool( conn.forceEncaps ) )
      if conn.connType is not None:
         config += fmt % ( 'type', self.mapConnectionType( conn.connType ) )
      if conn.reauth is not None:
         config += fmt % ( 'reauth', self.mapBool( conn.reauth ) )

      if conn.ikeDscp is not None:
         # strongswan is expecting this as a 6-digit binary string
         config += fmt % ( 'ikedscp', f'{conn.ikeDscp:06b}' )

      if conn.idleTimeout is not None:
         config += fmt % ( 'inactivity', f'{conn.idleTimeout}s' )

      return config

   def ipsecConf( self ):
      config = ''

      if not self.confData:
         traceVerbose( 'no config file data' )
         return ''

      cs = self.confData.configSetup
      if cs.uniqueIds is not None:
         config += 'config setup\n'
         config += f'    uniqueids = {self.mapBool(cs.uniqueIds)}\n'

      defaultConnKey = self.confData.defaultConnectionKey( self.vrfId )
      if defaultConnKey in self.shadowConfData:
         config += self.ipsecConfForConnection( '%default', defaultConnKey,
                                                isDefault=True )

      self.connectionManager.connectionsInConfFile.clear()
      for connKey in self.shadowConfData:
         if connKey != defaultConnKey:
            # this is a per-VRF collection, so we should not have any
            # other VRF connections in here
            assert connKey.vrfId == self.vrfId
            connName = self.connName( connKey )
            config += self.ipsecConfForConnection( connName, connKey,
                                                   isDefault=False )
            self.connectionManager.connectionsInConfFile.add( connKey )

      self.connectionManager.handleConfWritten( config )

      return config

   def connName( self, key ):
      connId = f'{key.connId.connIdSrc}-{key.connId.connId}'
      return f'{self.vrfName}-{key.srcAddr}-{key.dstAddr}-{connId}'

   def vrfNameFromId( self, vrfId ):
      return ''

   def mapSecretType( self, st ):
      if st == 'secretTypePsk':
         return 'PSK'
      elif st == 'secretTypeRsa':
         return 'RSA'
      else:
         # We should add an assert here
         traceVerbose( 'unknown secret type', bv( st ) )

      # Default
      return 'PSK'

   def ipsecSecrets( self ):
      config = ''
      config += '# generated by new SuperServerPlugin\n'
      d = self.confData

      defSecretId = d.noSelectorSecretId( self.vrfId )
      if defSecretId in d.secret:
         secret = d.secret[ defSecretId ]
         # this needs to start with ':', not whitespace
         config += ': {} {}\n'.format( self.mapSecretType( secret.secretType ),
                                       secret.secretValue.getClearText() )

      config += '\n'

      for secret in d.secret.values():
         idType = secret.secretId.idType
         if secret.secretId == defSecretId:
            continue

         secretKey = ""
         if idType == 'secretIdTypeFqdn':
            secretKey = secret.secretId.hostId.fqdn
         elif idType == 'secretIdTypeAddress':
            secretKey = secret.secretId.hostId.ip
         else:
            traceVerbose( 'unknown idType', bv( idType ), secret.secretId )
            continue

         config += '\n'
         secretValue = secret.secretValue.getClearText()
         if secret.secretType == 'secretTypeRsa':
            secretValue = secret.privateKey
         # the space before and after the ':' are critical
         config += '{} : {} {}\n'.format( secretKey,
                                          self.mapSecretType( secret.secretType ),
                                          secretValue )

         # genId 0 means that update is in process, it will be marked
         # with the correct id when the rereadsecrets process is
         # complete
         self.maybeUpdateSecretShadow( secret, 0 )

      # purge any deleted secrets from the generation id shadow
      secretKeys = list( self.secretShadow.keys() )
      for sid in secretKeys:
         if sid not in d.secret:
            del self.secretShadow[ sid ]

      return config

   def prepopulateSecretShadow( self ):
      traceFunc( bv( self.vrfName ) )
      assert len( self.secretShadow ) == 0
      for secret in self.confData.secret.values():
         self.maybeUpdateSecretShadow( secret, self.secretGenId )

   def maybeUpdateSecretShadow( self, secret, genId ):
      traceFunc( bv( self.vrfName ), ', sid::hostId ',
                 bv( self.secretHostId( secret.secretId ) ),
                 ', genId ', bv( genId ) )

      sid = secret.secretId
      shadowData = self.secretShadow.get( sid )
      if not shadowData:
         shadowData = dict()
         shadowData[ 'sid' ] = sid
         shadowData[ 'value' ] = secret.secretValue
         shadowData[ 'genId' ] = genId
         self.secretShadow[ sid ] = shadowData
      elif shadowData[ 'value' ].sha256() != secret.secretValue.sha256():
         shadowData[ 'value' ] = secret.secretValue
         shadowData[ 'genId' ] = genId

   def markSecretsWritten( self ):
      traceFunc( bv( self.vrfName ), ' new genId: ', bv( self.secretGenId ) )
      for shadowData in self.secretShadow.values():
         if shadowData[ 'genId' ] == 0:
            traceVeryVerbose( 'updating ',
                              bv( self.secretHostId( shadowData[ 'sid' ] ) ) )
            shadowData[ 'genId' ] = self.secretGenId

   def shadowSecretGenId( self, sid ):
      if sid not in self.secretShadow:
         # this should only ever happen transitionally, with us about
         # to delete the connection, so ignore it
         return 0

      return self.secretShadow[ sid ][ 'genId' ]

   def updateShadowConfData( self ):
      traceFunc( bv( self.vrfName ) )

      self.shadowConfData = dict()
      for key in self.confData.connection:
         confConn = self.confData.connection[ key ]
         self.shadowConfData[ key ] = confConn

   def getConnFromShadow( self, key ):
      traceFunc( 'key ', bv( key.str() ) )
      if key not in self.shadowConfData:
         return None
      return self.shadowConfData[ key ]

   def openSslConf( self ):
      data = "openssl {\n"
      data += "  load = yes\n"
      if self.parent.ipsecConfig.fipsRestrictions:
         data += "  fips_mode = 1\n"
         data += f"  fips_mode_logging = {int(int(self.daemonConfig.fipsLogging))}\n"
      data += "}\n"
      return data

   def strongswanConf( self ):
      # Set kernel_forward to 'no' in Sfe mode
      # timeout means the milliseconds strongswan will wait for the netlink
      # message response from Ipsec. In Sfe mode, with this, we will not get into
      # the deadlock when strongswan do GETSA on an already deleted SA, and we have
      # deleted the vrfRoot. Otherwise, strongswan cannot handle "sudo strongswan
      # stop" request since it's busy in waiting GETSA response. Set timeout to '0'
      # in Sfa mode since it's not needed ( '10000' works fine for stress test with
      # 128 tunnels in Sfe mode )
      if self.routingHwStatus.kernelForwardingSupported:
         kernel_forward = 'yes'
         parallel_xfrm = 'no'
         timeout = '0'
      else:
         kernel_forward = 'no'
         parallel_xfrm = 'yes'
         timeout = '10000'
      # Changing the number of strongswan threads = 24, to process strongswan
      #    up/down.
      # Default value for threads is 16
      # Changing the max_concurrency = 16, to queue multiple strongswan requests.
      # Default value for max_concurrency is 4.
      # Setting inactivity_close_ike closes lingering IKE_SAs if the only
      #    CHILD_SA is closed due to inactivity. IKE_SAs linger if the
      #    ike-policy config matches but the sa-policy config doesn't.

      # Set the min/max SPI
      # Valid SPI range
      # 0xC0000000 - 0xCDFFFFFF : reserved for Tunnel intf and
      #                           AutoVPN static path
      # NOTE :: The range MUST be consistent with the range defined
      #         in Ipsec::SPIRange
      spiMin = '0xc0000000'
      spiMax = '0xcdffffff'
      if self.parent.ipsecCapabilitiesStatus.spi24Bits:
         spiMin = '0xc00000'
         spiMax = '0xcdffff'

      config = """
# config file came from new SuperServerPlugin code
charon {
        vrf_name = %s
        load_modular = yes
        install_routes = no
        retransmit_tries = 2
        retransmit_timeout = 8.0
        make_before_break = yes
        delete_rekeyed_delay = 10
        threads = 24
        inactivity_close_ike = yes
        close_ike_on_child_failure = yes
        filelog {
                charon {
                     path = %s
                     time_format = %%b %%e %%T
                     append = yes
                     default = %d
                     knl = %d
                     ike = %d
                     cfg = %d
                     flush_line = yes
                     time_add_ms = yes
                 }
                 stdout {
                     default = 0
                     ike_name = yes
                 }
        }
        syslog {
               # enable logging to LOG_DAEMON, use defaults
               daemon {
                    default = -1
               }
               # minimalistic IKE auditing logging to LOG_AUTHPRIV
               auth {
                    default = -1
               }
        }
        plugins {
                include strongswan.d/charon/*.conf
                stroke {
                  max_concurrent = 16
                }
                kernel-netlink {
                    fwmark = !0x42
                    roam_events = no
                    kernel_forward = %s
                    parallel_xfrm = %s
                    timeout = %s
                    retries = 2
                }
                socket-default {
                    fwmark = 0x42
                }
                kernel-libipsec {
                    allow_peer_ts = yes
                }
        }
        spi_min = %s
        spi_max = %s
}

include strongswan.d/*.conf
      """ % ( self.vrfName, self.logFileName, self.defLogLevel, self.knlLogLevel,
              self.ikeLogLevel, self.cfgLogLevel, kernel_forward, parallel_xfrm,
              timeout, spiMin, spiMax )
      return config

   def removePkiFilesFromMaps( self, connectionKey ):
      traceVerbose( "Remove pki files for", connectionKey )
      if connectionKey in self.connectionToPkiFilesMap:
         for key in [ 'certificate', 'privateKey', 'trustedCerts', 'crls' ]:
            for pkiFile in self.connectionToPkiFilesMap[ connectionKey ][ key ]:
               self.pkiFileToConnectionsMap[ pkiFile ][ 'connectionKeys' ].remove(
                  connectionKey )

         del self.connectionToPkiFilesMap[ connectionKey ]

   def trackPkiFileForConnection( self, connectionKey, pkiFile, key ):
      assert key in [ 'certificate', 'privateKey', 'trustedCerts', 'crls' ]
      traceVerbose( "Connection", connectionKey, "is using pki file", pkiFile, 'as',
                    key )
      if connectionKey not in self.connectionToPkiFilesMap:
         self.connectionToPkiFilesMap[ connectionKey ] = {
            'certificate': set(),
            'privateKey': set(),
            'trustedCerts': set(),
            'crls': set()
         }
      self.connectionToPkiFilesMap[ connectionKey ][ key ].add( pkiFile )

      if pkiFile not in self.pkiFileToConnectionsMap:
         self.pkiFileToConnectionsMap[ pkiFile ] = {
            'key': key,
            'connectionKeys': set()
         }
      self.pkiFileToConnectionsMap[ pkiFile ][ 'connectionKeys' ].add(
         connectionKey )

   def copyPkiFile( self, src, dst ):
      try:
         traceVerbose( "copyPkiFile: src = ", bv( src ), " dst = ", bv( dst ) )
         shutil.copyfile( src, dst )
      except shutil.SameFileError:
         err = 'Source and destination represents the same file.'
         traceVerbose( 'src:', bv( src ), 'dst:', bv( dst ), 'err:', bv( err ) )
      except IsADirectoryError:
         err = 'Destination is a directory.'
         traceVerbose( 'src:', bv( src ), 'dst:', bv( dst ), 'err:', bv( err ) )
      except PermissionError:
         err = 'Permission denied.'
         traceVerbose( 'src:', bv( src ), 'dst:', bv( dst ), 'err:', bv( err ) )
      except OSError as e:
         traceVerbose( 'src:', bv( src ), 'dst:', bv( dst ), 'err:',
                       bv( e.strerror ) )

   def copyPkiFiles( self, connectionKey ):
      # Remove existing PKI files used by this connection
      # Pki files still in use by this connection will be repopulated by the end of
      # this method and pki files no longer being used by any connection will be
      # removed from strongswan directories.
      self.removePkiFilesFromMaps( connectionKey )
      if connectionKey not in self.confData.connection:
         return
      pkiProfile = self.confData.connection[ connectionKey ].pkiProfile
      if not pkiProfile:
         return

      traceFunc( 'vrf:', bv( self.vrfName ), "connection:", bv( connectionKey ) )
      if pkiProfile not in self.sslStatus.profileStatus:
         traceVerbose( 'SslStatus not present for', bv( pkiProfile ) )
         return
      profileStatus = self.sslStatus.profileStatus[ pkiProfile ]

      sslPath = '/persist/secure/ssl/'
      sswanPath = f'{self.baseDir}/ipsec.d/'

      privateKey = profileStatus.certKeyPair.keyFile
      if privateKey:
         src = sslPath + 'keys/' + privateKey
         dst = sswanPath + 'private/' + privateKey
         self.copyPkiFile( src, dst )
         self.trackPkiFileForConnection(
            connectionKey, privateKey, key='privateKey' )
      else:
         traceVerbose( 'no private key in SslStatus for', bv( pkiProfile ) )

      cert = profileStatus.certKeyPair.certFile
      if cert:
         src = sslPath + 'certs/' + cert
         dst = sswanPath + 'certs/' + cert
         self.copyPkiFile( src, dst )
         self.trackPkiFileForConnection( connectionKey, cert, key='certificate' )
      else:
         traceVerbose( 'no cert in SslStatus for', bv( pkiProfile ) )

      certs = set( profileStatus.trustedCert.keys() )
      certs.update( set( profileStatus.chainedCert.keys() ) )
      for cert in certs:
         src = sslPath + 'certs/' + cert
         dst = sswanPath + 'cacerts/' + cert
         self.copyPkiFile( src, dst )
         self.trackPkiFileForConnection( connectionKey, cert, key='trustedCerts' )

      for crl in profileStatus.crl.keys():
         src = sslPath + 'certs/' + crl
         dst = sswanPath + 'crls/' + crl
         self.copyPkiFile( src, dst )
         # Update the crls used by this connection
         self.trackPkiFileForConnection( connectionKey, crl, key='crls' )

      # Remove any orphan PKI files.
      removedPkiFiles = set()
      for pkiFile, value in self.pkiFileToConnectionsMap.items():
         key = value[ 'key' ]
         connectionKeys = value[ 'connectionKeys' ]
         path = {
            'certificate': 'certs/',
            'privateKey': 'private/',
            'trustedCerts': 'cacerts/',
            'crls': 'crls/',
         }
         if not connectionKeys:
            # Delete the PKI file from strongswan directory
            pkiFilePath = sswanPath + path[ key ] + pkiFile
            traceVerbose( "Deleting pki file", bv( pkiFilePath ) )
            try:
               os.remove( pkiFilePath )
            except OSError as e:
               traceFunc( "Error in deleting file:", bv( e ) )

            # Remove the pki file entry as this file is not being used by any
            # connection. Do not remove it here to avoid changing dictionary size
            # during iteration
            removedPkiFiles.add( pkiFile )

      for pkiFile in removedPkiFiles:
         del self.pkiFileToConnectionsMap[ pkiFile ]

      self.connectionManager.rereadCerts()
      self.connectionManager.purgeCrls()

   @Tac.handler( 'configSetup' )
   def handleConfigSetup( self ):
      if self.deleted:
         traceFunc( "This service for", bv( self.vrfName ),
                    "is deleted and should be garbage collected shortly" )
         return
      traceFunc( self.vrfName, 'vrfId:', bv( self.vrfId ) )
      self.handleConfigData()

   @Tac.handler( 'connection' )
   def handleConnection( self, key ):
      if self.deleted:
         traceFunc( "This service for", bv( self.vrfName ),
                    "is deleted and should be garbage collected shortly" )
         return
      traceFunc( self.vrfName, 'vrfId:', bv( self.vrfId ) )
      self.copyPkiFiles( key )
      self.handleConfigData()

   @Tac.handler( 'connectionEnabled' )
   def handleConnectionEnabled( self, key ):
      if self.deleted:
         traceFunc( "This service for", bv( self.vrfName ),
                    "is deleted and should be garbage collected shortly" )
         return
      traceFunc( self.vrfName, 'vrfId:', bv( self.vrfId ) )
      self.handleConfigData()

   @Tac.handler( 'secret' )
   def handleSecret( self, sid ):
      if self.deleted:
         traceFunc( "This service for", bv( self.vrfName ),
                    "is deleted and should be garbage collected shortly" )
         return
      traceFunc( self.vrfName, 'vrfId:', bv( self.vrfId ), 'sid:',
                 bv( self.secretHostId( sid ) ) )
      self.handleConfigData()

   def secretHostId( self, sid ):
      if sid.hostId and sid.hostId.containsIp():
         return sid.hostId.ip
      if sid.hostId and sid.hostId.containsFqdn():
         return sid.hostId.fqdn
      return sid.hostId

class IpsecPerVrfServiceTest( IpsecPerVrfService ):
   def __init__( self, vrfId, vrfName, confData, parent, daemonConfig, daemonStatus,
                 routingHwStatus, sslStatus, templateDir, baseDir, origStrongswanDir,
                 netNsName, serviceRestartDelay ):
      self.copiedPkiFiles = {}
      IpsecPerVrfService.__init__( self, vrfId, vrfName, confData, parent,
                                   daemonConfig, daemonStatus, routingHwStatus,
                                   sslStatus, templateDir, baseDir,
                                   origStrongswanDir, netNsName,
                                   serviceRestartDelay )

   def copyPkiFile( self, src, dst ):
      self.copiedPkiFiles[ src ] = dst

class ConfDataDirReactor( Tac.Notifiee ):
   # the confDataDir contains all of the configuration data that we
   # write out to ipsec.conf and ipsec.secrets
   notifierTypeName = 'Ipsec::Conf::ConfDataDir'

   def __init__( self, confDataDir, parent ):
      self.parent = parent
      self.confDataDir = confDataDir
      Tac.Notifiee.__init__( self, confDataDir )

   @Tac.handler( 'conf' )
   def handleConf( self, vrfId ):
      traceFunc( bv( vrfId ) )
      self.parent.handleVrfId( vrfId )

class DaemonConfigReactor( Tac.Notifiee ):
   # daemonConfig contains a couple of bools that affect how (and if!)
   # we run strongswan.
   notifierTypeName = 'Ipsec::DaemonConfig'

   def __init__( self, daemonConfig, parent ):
      self.parent = parent
      self.daemonConfig = daemonConfig
      Tac.Notifiee.__init__( self, daemonConfig )

   @Tac.handler( 'fipsLogging' )
   def handleFipsLogging( self ):
      traceFunc( bv( self.daemonConfig.fipsLogging ) )
      self.parent.syncAll()

   @Tac.handler( 'enableSswan' )
   def handleEnableSswan( self ):
      traceFunc( bv( self.daemonConfig.enableSswan ) )
      self.parent.handleVrfIdAll()

   # the Ipsec agent increments this when it restarts, we track it so
   # that we can force a reload of strongswan to make sure it syncs
   # properly with the unix domain sockets in the new instance of
   # Ipsec
   @Tac.handler( 'agentGeneration' )
   def handleAgentGeneration( self ):
      traceVerbose( bv( self.daemonConfig.agentGeneration ) )
      self.parent.handleAgentGeneration()

class IpsecConfigReactor( Tac.Notifiee ):
   notifierTypeName = 'Ipsec::Ike::Config'

   def __init__( self, ipsecConfig, parent ):
      self.parent = parent
      self.ipsecConfig = ipsecConfig
      Tac.Notifiee.__init__( self, ipsecConfig )

   @Tac.handler( 'fipsRestrictions' )
   def handleFipsRestrictions( self ):
      traceFunc( bv( self.ipsecConfig.fipsRestrictions ) )
      self.parent.syncAll()

class IpsecStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Ipsec::Ike::Status'

   # IpsecStatus tells us if the license is enabled properly
   def __init__( self, ipsecStatus, parent ):
      self.parent = parent
      self.ipsecStatus = ipsecStatus
      Tac.Notifiee.__init__( self, ipsecStatus )

   @Tac.handler( 'ipsecLicenseEnabled' )
   def handleIpsecLicenseEnabled( self ):
      traceFunc( bv( self.ipsecStatus.ipsecLicenseEnabled ) )
      self.parent.handleVrfIdAll()

   @Tac.handler( 'vrfsInUse' )
   def handleVrfsInUse( self, vrfId ):
      traceFunc( bv( vrfId ) )
      self.parent.handleVrfId( vrfId )

class VrfNameStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Vrf::VrfIdMap::NameToIdMapWrapper'

   def __init__( self, vns, parent ):
      self.parent = parent
      Tac.Notifiee.__init__( self, vns )

   @Tac.handler( 'nameToIdMap' )
   def handleNameToIdMap( self ):
      self.parent.handleNameToIdMap()

class VrfIdStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Vrf::VrfIdMap::Status'

   # track when VRFs come and go
   def __init__( self, vrfIdStatus, parent ):
      self.parent = parent
      self.vrfIdStatus = vrfIdStatus
      Tac.Notifiee.__init__( self, vrfIdStatus )

   @Tac.handler( 'vrfIdToName' )
   def handleVrfIdToName( self, vrfId ):
      traceFunc( bv( vrfId ) )
      self.parent.handleVrfId( vrfId )

class AllVrfStatusLocalReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::AllVrfStatusLocal'

   def __init__( self, avsl, parent ):
      self.parent = parent
      self.avsl = avsl
      Tac.Notifiee.__init__( self, avsl )

   @Tac.handler( 'vrf' )
   def handleVrf( self, vrf ):
      traceFunc( bv( vrf ) )
      self.parent.handleVrfState( vrf )

class VrfStatusLocalReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::VrfStatusLocal'

   def __init__( self, vsl, parent ):
      self.parent = parent
      self.vsl = vsl
      Tac.Notifiee.__init__( self, vsl )

   @Tac.handler( 'state' )
   def handleState( self ):
      traceFunc( bv( self.vsl.vrfName ), ' state is ', bv( self.vsl.state ) )
      self.parent.handleVrfState( self.vsl.vrfName )

   @Tac.handler( 'networkNamespace' )
   def handlerNetns( self ):
      traceFunc( bv( self.vsl.vrfName ), ' ns is ', bv( self.vsl.networkNamespace ) )
      self.parent.handleVrfState( self.vsl.vrfName )

class ConnStatusNetlinkReactor( Tac.Notifiee ):
   notifierTypeName = 'Ipsec::ConnectionStatusByKey'

   def __init__( self, csn, parent ):
      self.parent = parent
      self.connStatusNetlink = csn
      Tac.Notifiee.__init__( self, csn )

   @Tac.handler( 'state' )
   def handleState( self, key ):
      self.parent.handleConnStatusNetlink( key )

class IpsecCapabilitiesStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Ipsec::Capabilities::Status'

   def __init__( self, ics, parent ):
      self.parent = parent
      self.ipsecCapabilitiesStatus = ics
      Tac.Notifiee.__init__( self, ics )

   @Tac.handler( 'spi24Bits' )
   def handleSpi24Bits( self ):
      self.parent.syncAll()

#
# This is the main class for the VRF-aware Ipsec strongswan
# SuperServer plugin.  We have one of these, and it is created at load
# time.  It reacts to the VRFs coming and going and creates a
# PerVrfService object for each instance as necessary.  It also sets
# up the initial template directory of ipsec-specific files,

class IpsecServiceParent:
   def __init__( self, confDataDir, ipsecConfig, ipsecStatus,
                 daemonConfig, daemonStatusDir, routingHwStatus, vrfIdStatus,
                 vrfNameStatus, allVrfStatusLocal, connStatusNetlink,
                 ipsecCapabilitiesStatus, redStatus, sslStatus ):

      traceFunc( '' )

      self.initComplete = False

      self.extraDebug = False

      self.confDataDir = confDataDir
      self.ipsecConfig = ipsecConfig
      self.ipsecStatus = ipsecStatus
      self.daemonConfig = daemonConfig
      self.daemonStatusDir = daemonStatusDir
      self.routingHwStatus = routingHwStatus
      self.vrfIdStatus = vrfIdStatus
      self.vrfNameStatus = vrfNameStatus
      self.allVrfStatusLocal = allVrfStatusLocal
      self.connStatusNetlink = connStatusNetlink
      self.ipsecCapabilitiesStatus = ipsecCapabilitiesStatus
      self.redStatus = redStatus
      self.sslStatus = sslStatus

      self.confDataDirReactor = ConfDataDirReactor( self.confDataDir, self )
      self.daemonConfigReactor = DaemonConfigReactor( self.daemonConfig, self )
      self.ipsecConfigReactor = IpsecConfigReactor( self.ipsecConfig, self )
      self.ipsecStatusReactor = IpsecStatusReactor( self.ipsecStatus, self )
      self.vrfNameStatusReactor = VrfNameStatusReactor( self.vrfNameStatus, self )
      self.vrfIdStatusReactor = VrfIdStatusReactor( self.vrfIdStatus, self )
      self.csnReactor = ConnStatusNetlinkReactor( self.connStatusNetlink, self )
      self.icsReactor = IpsecCapabilitiesStatusReactor(
         self.ipsecCapabilitiesStatus, self )
      self.avslReactor = AllVrfStatusLocalReactor( self.allVrfStatusLocal, self )

      self.avslCollReactor = Tac.collectionChangeReactor(
         self.allVrfStatusLocal.vrf, VrfStatusLocalReactor,
         reactorArgs=( weakref.proxy( self ), ) )

      self.perVrfService = dict()

      self.sswanDir = '/etc/sswan'
      self.strongswanVrfDir = f'{self.sswanDir}/vrf'
      self.strongswanSocketDir = f'{self.sswanDir}/sockets'
      self.templateDir = f'{self.strongswanVrfDir}/template'
      self.origStrongswanDir = '/etc/strongswan'

      self.shadowRedMode = self.redStatus.mode
      self.shadowRedProto = self.redStatus.protocol

      self.cryptoModulesLoaded = False

      self.createStrongswanVrfDir()
      self.cleanupOldVrfInstances()

      self.agentGeneration = self.daemonConfig.agentGeneration

      self.connIdCounter = 1
      self.cleanupOldStrongswanInstances()
      self.initComplete = True

      self.maybeFinishInit()

   def handleNameToIdMap( self ):
      self.maybeFinishInit()

   def maybeFinishInit( self ):
      if self.initComplete and self.vrfNameStatus.nameToIdMap:
         for vrfId in self.confDataDir.conf:
            self.handleVrfId( vrfId )

   def handleAgentGeneration( self ):
      traceFunc( '' )
      # when the Ipsec agent restarts we need to fully restart all of
      # the strongswan instances so that they can resync with the unix
      # domain sockets
      if self.daemonConfig.agentGeneration != self.agentGeneration:
         traceFunc( 'restarting strongswan services due to agent gen' )
         self.agentGeneration = self.daemonConfig.agentGeneration
         allVrfIds = list( self.perVrfService.keys() )
         for v in allVrfIds:
            self.deleteService( v )

         for v in allVrfIds:
            self.handleVrfId( v )

   def newConnId( self ):
      idVal = self.connIdCounter
      self.connIdCounter += 1
      return idVal

   def readNetnsData( self ):
      netnsData = dict()
      allNetnsInodes = set()
      nsFiles = os.listdir( '/var/run/netns' )
      for f in nsFiles:
         if f == 'default':
            vrfName = 'default'
         elif f.startswith( 'ns-' ):
            vrfName = f[ 3 : ]
         else:
            continue

         statResult = None
         try:
            statResult = os.stat( f'/var/run/netns/{f}' )
         except OSError:
            pass

         if not statResult:
            continue

         netnsData[ vrfName ] = statResult.st_ino
         allNetnsInodes.add( statResult.st_ino )

      return ( netnsData, allNetnsInodes )

   # search through /proc for old instances of strongswan-related
   # processes in the specified VRF and kill them.  If vrfName is
   # none, do it for all VRFs.
   def cleanupOldStrongswanInstances( self, vrfName=None ):
      traceFunc( 'vrf ', bv( vrfName ) )
      if self.daemonConfig.simulateDaemonForBtest:
         return

      # on a physical dut we know that this SuperServerPlugin owns all
      # strongswan-related processes, but unfortunately namespace DUTs
      # do not create a PID namespace.  This means that on an ns dut
      # (say 'rtr1'), searching through /proc will also show us
      # processes for other ns duts in the same container (say
      # 'rtr2').  This can happen if a single ptest or stest starts
      # two duts, or if two tests are running at the same time.
      #
      # to avoid inadvertently killing strongswan processes belonging
      # to a different DUT we build a list of all of the inodes for
      # the network namespaces in the current dut, by iterating over
      # /var/run/netns/*.  We then only kill a strongswan-related
      # process if /proc/<pid>/ns/net for that process matches an
      # inode in that list.
      #
      # This approach will potentially miss strongswan instances that
      # are still running in a network namespace for a VRF that has
      # just been deleted, but they should exit on their own anyway
      ( netnsData, allNetnsInodes ) = self.readNetnsData()

      procs = os.listdir( '/proc' )
      for procFile in procs:
         if not procFile[ 0 ].isdigit():
            continue
         try:
            try:
               f = open( f'/proc/{procFile}/cmdline' )
               cmdline = f.readline()
            except ( OSError, ProcessLookupError ):
               continue

            if cmdline.startswith( '/usr/libexec/strongswan' ):
               pid = int( procFile )
               nsFileInode = getNetNsInode( pid )

               if not nsFileInode:
                  # process must have died already for the /proc file
                  # to have gone away
                  continue

               if( ( vrfName is None and nsFileInode in allNetnsInodes ) or
                   ( vrfName is not None and vrfName in netnsData and
                     netnsData[ vrfName ] == nsFileInode ) ):
                  traceVerbose( 'cleaning up pid ', bv( pid ) )
                  killProc( pid, 'old instance' )
            f.close()
         except OSError:
            pass

   def createStrongswanVrfDir( self ):
      traceFunc( '' )

      def newDir( dirName ):
         if os.path.exists( dirName ) and not os.path.isdir( dirName ):
            try:
               traceVerbose( 'trying to unlink', bv( dirName ) )
               os.unlink( dirName )
            except OSError as e:
               traceVerbose( 'unlink', bv( dirName ), bv( e.strerror ) )

         if not os.path.exists( dirName ):
            try:
               traceVerbose( 'creating', bv( dirName ) )
               os.mkdir( dirName, 0o700 )
            except OSError as e:
               traceVerbose( 'mkdir', bv( dirName ), bv( e.strerror ) )

      newDir( self.sswanDir )
      newDir( self.strongswanVrfDir )
      newDir( self.strongswanSocketDir )
      newDir( self.templateDir )

      for d in [ 'ipsec.d', 'strongswan.d', 'swanctl' ]:
         newDir( f'{self.templateDir}/{d}' )

      newDir( f'{self.templateDir}/strongswan.d/charon' )
      newDir( f'{self.templateDir}/ipsec.d/run' )

      # here are some of the empty directories that we need to keep
      # strongswan from complaining
      ipsecDPath = f'{self.templateDir}/ipsec.d'
      for d in [ 'cacerts', 'aacerts', 'ocspcerts', 'acerts', 'crls' ]:
         newDir( f'{ipsecDPath}/{d}' )

      # these may not be needed?
      for d in [ 'certs', 'private', 'reqs' ]:
         newDir( f'{ipsecDPath}/{d}' )

      # now create the varous config files that we don't actually change based
      # on Sysdb state.

      # the dist dir has a file in swanctl/swanctl.conf, but the
      # entire file is a comment so I'm going to leave it out 

      # strongswan.d has a bunch of config files, but most of them
      # don't actually seem to do much.  The "minimal" ones just
      # create a struct with a name

      strongswanDPath = f'{self.templateDir}/strongswan.d'
      minimalSwanDFiles = [ 'scepclient', 'swanctl', 'pki', 'starter' ]

      def writeFile( filePath, contents ):
         f = open( filePath, 'w' )
         assert f
         f.write( contents )
         f.close()

      for f in minimalSwanDFiles:
         conf = '''
%s {
} ''' % f
         path = f'{strongswanDPath}/{f}.conf'
         writeFile( path, conf )

      # Next we have a few files in the strongswan.d path that have
      # some additional "null" structs inside them.

      charonConf = '''
charon {
    crypto_test {
    }
    host_resolver {
    }
    leak_detective {
    }
    processor {
        priority_threads {
        }
    }
    start-scripts {
    }
    stop-scripts {
    }
    tls {
    }
    x509 {
    }
}
'''
      writeFile( f'{strongswanDPath}/charon.conf', charonConf )

      charonLoggingConf = '''
charon {
    filelog {
    }
    syslog {
    }
}
'''
      writeFile( f'{strongswanDPath}/charon-logging.conf',
                 charonLoggingConf )

      # Now we do the ones in strongswan.d/charon/*.conf
      #
      # Again there are a bunch of these that are "minimal" -- all
      # they do is specify "load = yes", so handle them together
      #
      # note that in the dist files openssl.conf would be in this
      # list, but we're going to handle that specially (it depends on
      # the FIPS config)

      charonPath = f'{strongswanDPath}/charon'

      minimalCharonConf = [
         'pkcs1', 'pkcs7', 'pkcs8', 'pkcs12', 'nonce', 'revocation', 'vici',
         'xauth-generic', 'pubkey', 'socket-default', 'constraints', 'dnskey',
         'attr', 'pem', 'updown', 'stroke', 'x509', 'curl', 'pgp', 'sshkey',
         'eap-tls' ]

      for f in minimalCharonConf:
         path = f'{charonPath}/{f}.conf'
         conf = '''
%s {
    load = yes
}
''' % f
         writeFile( path, conf )

      # three of the files (kernel-netlink, pkcs11, and resolve) have
      # empty sub-structures in them.
      kernelNetlinkConf = '''
kernel-netlink {
    load = yes
    spdh_thresh {
        ipv4 {
        }
        ipv6 {
        }
    }
}
'''
      writeFile( f'{strongswanDPath}/charon/kernel-netlink.conf',
                 kernelNetlinkConf )

      pkcs11Conf = '''
pkcs11 {
    load = yes
    modules {
        <name> {
        }
    }
}
'''
      writeFile( f'{strongswanDPath}/charon/pkcs11.conf', pkcs11Conf )

      resolveConf = '''
resolve {
    load = yes
    resolvconf {
    }
}
'''
      writeFile( f'{strongswanDPath}/charon/resolve.conf', resolveConf )

      # another one we may not need
      writeFile( f'{self.templateDir}/swanctl/swanctl.conf',
                 '# do we need this?\n' )

      # and that's it.

   def canWriteToSysdb( self ):
      return( self.redStatus.mode == 'active' or
              self.redStatus.protocol != 'sso' )

   def onSwitchover( self ):
      traceFunc( 'mode', bv( self.redStatus.mode ),
                 'proto', bv( self.redStatus.protocol ),
                 'shadow mode', bv( self.shadowRedMode ),
                 'shadow proto', bv( self.shadowRedProto ) )
      if( self.redStatus.mode == self.shadowRedMode and
          self.redStatus.protocol == self.shadowRedProto ):
         traceVerbose( 'no change' )
         return

      if not self.canWriteToSysdb():
         # we transitioned to standby?  Is this possible?
         return

      # create any daemon status entities that are missing
      for pvs in self.perVrfService.values():
         if pvs.vrfName not in self.daemonStatusDir.daemonStatus:
            traceVerbose( 'creating DSD', bv( pvs.vrfName ) )
         else:
            traceVerbose( 'already have DSD', bv( pvs.vrfName ) )

         # Create new status or get the existing one
         # When the standby transitions to active, the daemonStatusDir is already
         # synced. But, pvs has not been updated with the daemonStatus yet.
         # Update it here.
         daemonStatus = self.daemonStatusDir.newDaemonStatus( pvs.vrfName )
         pvs.daemonStatus = daemonStatus
         pvs.onSwitchover()

      # and purge any that are no longer needed
      avn = self.allVrfNames()
      for vrfName in self.daemonStatusDir.daemonStatus:
         if vrfName not in avn:
            traceVerbose( 'cleaning up DSD', bv( vrfName ) )
            del self.daemonStatusDir.daemonStatus[ vrfName ]
         else:
            traceVerbose( 'leaving DSD intact', bv( vrfName ) )

   # this is used in the breadth tests to try to make sure we don't
   # have any dangling references from a previous instance of the
   # manager left in the process space of a restart test
   def cleanupRefs( self ):
      traceFunc( '' )
      for pvs in self.perVrfService.values():
         pvs.cleanupService()
      self.perVrfService.clear()

      del self.confDataDirReactor
      self.confDataDirReactor = None
      del self.daemonConfigReactor
      self.daemonConfigReactor = None
      del self.ipsecConfigReactor
      self.ipsecConfigReactor = None
      del self.ipsecStatusReactor
      self.ipsecStatusReactor = None
      del self.vrfIdStatusReactor
      self.vrfIdStatusReactor = None
      del self.csnReactor
      self.csnReactor = None
      del self.icsReactor
      self.icsReactor = None

   def allVrfNames( self ):
      avn = list()
      for pvs in self.perVrfService.values():
         avn.append( pvs.vrfName )

      return avn

   # this purges any VRF directories or daemonStatus entries that
   # should not exist.  it's used on agent startup (mainly useful on
   # SuperServer restart)
   def cleanupOldVrfInstances( self ):
      traceFunc( '' )

      if os.path.isdir( self.strongswanVrfDir ):
         files = os.listdir( self.strongswanVrfDir )
         for f in files:
            fullPath = self.strongswanVrfDir + '/' + f
            if f == self.templateDir:
               continue
            elif f.startswith( 'VRF-' ):
               m = re.match( 'VRF-(.*)$', f )
               assert m
               vrfName = m.groups()[ 0 ]
               traceVerbose( 'cleaning up vrf dir', bv( fullPath ) )
               try:
                  shutil.rmtree( fullPath )
               except OSError as e:
                  traceVerbose( 'rmtree', bv( fullPath ), 'gave',
                                bv( e.strerror ) )
            else:
               traceVerbose( 'ignoring non-VRF dir', bv( fullPath ) )

      if self.canWriteToSysdb():
         for vrfName in self.daemonStatusDir.daemonStatus:
            traceVerbose( 'cleanupDeadVrfDirs: clean', bv( vrfName ) )
            del self.daemonStatusDir.daemonStatus[ vrfName ]

   def handleConnStatusNetlink( self, key ):
      traceFunc( bv( key.str() ) )
      # bascially we listen to connStatusNetlink so that we can detect
      # changes initiated by the other side of the connection.  This
      # happens the Ipsec agent receives the netlink notifications
      # from strongswan and updates CSN in response to it.  There are
      # two interesting cases:
      #
      # 1) if we are the active side but the connection drops, we need
      # to be notified about it so that we can try to stroke it back
      # up again
      #
      # 2) if we are the passive side, then we want to know when the
      # connection has been brough up from the other side.  this is so
      # that when the connection state is changed locally we can
      # stroke the connection down to force a renegotiation

      vrfId = key.vrfId
      if vrfId not in self.perVrfService:
         return

      pvs = self.perVrfService[ vrfId ]
      if pvs.connectionManager:
         pvs.connectionManager.startStatusCheck()

   def handleVrfId( self, vrfId ):
      assert self.initComplete

      ve = self.vrfIdStatus.vrfIdToName.get( vrfId )
      vrfName = ve.vrfName if ve else None
      traceFunc( bv( vrfId ), bv( vrfName ) )

      if( not self.ipsecStatus.ipsecLicenseEnabled or
          not self.daemonConfig.enableSswan ):
         traceVerbose( 'service disabled vrfId ', bv( vrfId ) )
         self.deleteService( vrfId )
         return

      if not vrfName:
         traceVerbose( 'vrfName is None' )
         self.deleteService( vrfId )
         return

      if vrfId not in self.confDataDir.conf:
         traceVerbose( 'vrfId', bv( vrfId ), 'not in confDataDir.conf' )
         self.deleteService( vrfId )
         return

      if len( self.confDataDir.conf[ vrfId ].connection ) == 0:
         traceVerbose( 'no connections for vrfId', bv( vrfId ) )
         self.deleteService( vrfId )
         return

      if vrfId not in self.ipsecStatus.vrfsInUse:
         traceVerbose( 'vrfId', bv( vrfId ), 'not in ipsecStatus.vrfsInUse' )
         self.deleteService( vrfId )
         return

      if vrfId != DEFAULT_VRF_ID:
         vsl = self.allVrfStatusLocal.vrf.get( vrfName )
         if not vsl or not vsl.networkNamespace or vsl.state != 'active':
            traceVerbose( 'Invalid vsl state vrfId', bv( vrfId ), 'state',
                          bv( vsl.state if vsl else '-' ), 'ns',
                          bv( vsl.networkNamespace if vsl else '-' ) )
            self.deleteService( vrfId )
            return
         netNsName = vsl.networkNamespace
      else:
         netNsName = DEFAULT_NS

      confData = self.confDataDir.conf.get( vrfId )
      if vrfId in self.perVrfService:
         # make sure pre-existing service is up to date and pointing
         # to the correct entity
         pvs = self.perVrfService[ vrfId ]
         if pvs.confData == confData and pvs.vrfName == vrfName:
            # we probably don't need to sync() here, but let's be paranoid...
            #
            # the check that vrfName is still the same is REALLY paranoid
            pvs.sync()
            return

         # if not up to date then we just delete it and re-create down
         # below
         traceVerbose( 'service not up to date vrfId ', bv( vrfId ) )
         self.deleteService( vrfId )
         pvs = None

      if not confData:
         return

      # either we cleaned up an out-of-date service or we are creating a new one
      daemonStatus = None
      if self.canWriteToSysdb():
         daemonStatus = self.daemonStatusDir.daemonStatus.get( vrfName )
         if not daemonStatus:
            traceVerbose( 'handleVrfId create DSD', bv( vrfName ) )
            daemonStatus = self.daemonStatusDir.newDaemonStatus( vrfName )
         assert daemonStatus

      traceVerbose( 'Create service', bv( vrfName ) )
      serviceRestartDelay = 2
      if self.daemonConfig.shortDelaysForBtest:
         # this makes the btest run much faster...
         serviceRestartDelay = 0
      vrfDir = self.strongswanVrfDir + ( f'/VRF-{vrfName}' )
      perVrfService = IpsecPerVrfService
      if self.daemonConfig.simulateDaemonForBtest:
         perVrfService = IpsecPerVrfServiceTest
      pvs = perVrfService(
         vrfId, vrfName, confData, self, self.daemonConfig, daemonStatus,
         self.routingHwStatus, self.sslStatus, self.templateDir, vrfDir,
         self.origStrongswanDir, netNsName, serviceRestartDelay )
      assert pvs
      assert pvs.serviceRestartDelay_ == serviceRestartDelay
      self.perVrfService[ vrfId ] = pvs

      self.syncCryptoModules()
      return

   def syncCryptoModules( self ):
      cmd = 'rmmod'

      if len( self.perVrfService ) > 0:
         # if we have services and we've already loaded the modules,
         # we're done
         if self.cryptoModulesLoaded:
            return

         # we only load the modules if we have at least one vrf that
         # is enabled (basically meaning it has a non-default
         # connection in it)
         for pvs in self.perVrfService.values():
            if pvs.serviceEnabled():
               cmd = 'modprobe'
               break
      elif not self.cryptoModulesLoaded:
         # if no services and not loaded, nothing to do
         return

      # do the load/unload
      for mod in [ 'aes-x86_64', 'aesni_intel', 'crypto_null' ]:
         rc = os.system( f'/sbin/{cmd} {mod}' )
         if rc:
            traceVerbose( bv( cmd ), bv( mod ), 'failed', bv( rc ) )

      if cmd == 'modprobe':
         self.cryptoModulesLoaded = True
      else:
         self.cryptoModulesLoaded = False

   def deleteService( self, vrfId ):
      traceFunc( bv( vrfId ) )

      if vrfId not in self.perVrfService:
         return

      pvs = self.perVrfService[ vrfId ]
      vrfName = pvs.vrfName
      self.perVrfService[ vrfId ].cleanupService()
      del self.perVrfService[ vrfId ]
      pvs.deleted = True
      pvs = None

      if self.canWriteToSysdb():
         del self.daemonStatusDir.daemonStatus[ vrfName ]

   def handleVrfIdAll( self ):
      # handleVrfIdAll calls handleVrfId() on all the VRFs we know
      # about.  This is used when some global change is applied that
      # may start or stop the per-vrf services.
      traceFunc( '' )

      vrfIdSet = set( self.vrfIdStatus.vrfIdToName.keys() )
      vrfIdSet.update( set( self.confDataDir.conf.keys() ) )
      vrfIdSet.update( set( self.perVrfService.keys() ) )

      for vrfId in list( vrfIdSet ):
         self.handleVrfId( vrfId )

   def syncAll( self ):
      # syncAll is less intrusive than handleVrfIdAll, it simply
      # forces us to call sync() on all of the existing services.
      # sync() is implemented in the SuperServer.LinuxService base
      # class and triggers a delayed action to review all of the
      # config files for a particular service and then restart the
      # daemon if necessary.  We use this when global attributes
      # change that will not add or remove services, but may result in
      # a need to change the high-level config files like
      # strongswan.conf.
      traceFunc( '' )

      for pvs in self.perVrfService.values():
         pvs.sync()

   def handleVrfState( self, vrfName ):
      traceFunc( vrfName )

      vrfId = self.vrfNameStatus.nameToIdMap.vrfNameToId.get( vrfName )
      if vrfId:
         self.handleVrfId( vrfId )
      else:
         self.handleVrfIdAll()

class NewSswanManager( SuperServer.SuperServerAgent ):
   """This agent monitors the Ipsec ConfData and creates PerVrfService
      entities to manage a strongswan for each VRF that is required."""


   def __init__( self, entityManager ):
      traceFunc( 'Starting NEW SswanManager...' )

      # Explicitly pass our agent name down as 'SswanManager' so that our agent name
      # is consistent regardless of the toggle state. If we don't do this,
      # SuperServer will get confused and won't know who we are. Once we remove this
      # toggle, we should also remove this agentName argument and simply rename the
      # class to SswanManager
      SuperServer.SuperServerAgent.__init__( self, entityManager,
                                             agentName='SswanManager' )
      # Mount configurations
      mg = entityManager.mountGroup()

      ipsecConfig = mg.mount( 'ipsec/ike/config', 'Ipsec::Ike::Config', 'r' )
      ipsecStatus = mg.mount( 'ipsec/ike/status', 'Ipsec::Ike::Status', 'r' )
      daemonConfig = mg.mount( 'ipsec/daemonconfig', 'Ipsec::DaemonConfig', 'rO' )
      daemonStatusDir = mg.mount( 'ipsec/daemonstatusdir',
                                  'Ipsec::DaemonStatusDir', 'w' )
      routingHwStatus = mg.mount( 'routing/hardware/status',
                                  'Routing::Hardware::Status', 'r' )
      confDataDir = mg.mount( Cell.path( 'ipsec/confData' ),
                              'Ipsec::Conf::ConfDataDir', 'r' )
      allVrfStatusLocal = mg.mount( Cell.path( 'ip/vrf/status/local' ),
                                    'Ip::AllVrfStatusLocal', 'r' )
      connStatusNetlink = mg.mount( 'ipsec/connStatusNetlink',
                                    'Ipsec::ConnectionStatusByKey', 'r' )
      ipsecCapabilitiesStatus = mg.mount( 'ipsec/capabilities/status',
                                          'Ipsec::Capabilities::Status', 'r' )
      vrfNameStatus = mg.mount( Cell.path( 'vrf/vrfNameStatus' ),
                                'Vrf::VrfIdMap::NameToIdMapWrapper', 'r' )
      sslStatus = mg.mount( 'mgmt/security/ssl/status',
                            'Mgmt::Security::Ssl::Status', 'r' )

      self.serviceParent = None

      shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
      smi = Smash.mountInfo( 'reader' )
      vrfIdStatus = shmemEm.doMount( 'vrf/vrfIdMapStatus', 'Vrf::VrfIdMap::Status',
                                     smi )

      self.mountsComplete = False

      def _finished():
         traceVerbose( 'Agent Mounted' )
         self.serviceParent = IpsecServiceParent(
            confDataDir, ipsecConfig, ipsecStatus, daemonConfig, daemonStatusDir,
            routingHwStatus, vrfIdStatus, vrfNameStatus, allVrfStatusLocal,
            connStatusNetlink, ipsecCapabilitiesStatus, self.redundancyStatus(),
            sslStatus )

      mg.close( _finished )

   def onSwitchover( self, protocol ):
      assert protocol
      if self.serviceParent:
         self.serviceParent.onSwitchover()

######################################################################
#
# This is the new SuperServer Plugin code for strongswan/VICI implementation
# in Ipsec Agent.
# Will be enabled on toggle IpsecVici
#
######################################################################

#
# We create one of these objects for each VRF, it manages the mount
# namespace and main strongswan process for that VRF.
#
# Note that this is only sort-of a SuperServer.LinuxService, because
# we aren't using 'service <name> up' style commands to manage it,
# instead we override startService/etc and invoke strongswan directly.
# This avoids needing to write init.d scripts for each VRF, and also
# sidesteps some compatiblity problems between sysv init and systemd
# files on namespace duts.  We also override the healthCheck code in
# the LinuxService functions to do that manually.  What I'm really
# after is to avoid reimplementing the config file difference/etc
# checking logic in LinuxService.
#
# Arguably it might be worth refactoring the classes in SuperServer so
# that we can get just the config file code that we want without the
# other stuff.  A project for another day.
#

# VICI update : ViciIpsecPerVrfService doesnt do Connection Management
# With Vici layer in Ipsec Agent , all the connection management
# i.e bringing up , tear down and reload of connections has been moved out
# of SuperServer. (Earlier SuperServer used to manage this)
# Hence removing any reference or functionality of such

# With VICI Changes , This class will only handle strongswan process/config
# management , with an extra task of PKI management
# As PKI management with VICI from Ipsec Agent will not be taken up in
# phase 1

# For Pki management Ipsec Agent VICI layer will provide a sysdb dir
# with a collection of Pki profiles in use Ipsec::ViciPkiProfiles
# ViciIpsecPerVrfService will have a reactor to this and invoke swanctl
# commands , it will load-authorities , load-creds and flush-certs based
# on the this dir
# A version will also be maitained in this to indicate a successful swanctl
# command
class ViciIpsecPerVrfService( SuperServer.LinuxService ):
   # Vici TODO
   # This will ViciPkiProfiles object from collection ViciPkiProfilesDir
   # indexed on vrfId and passed to this class
   # This will added as part of further muts
   notifierTypeName = 'Ipsec::Conf::ConfData'
   #notifierTypeName = 'Ipsec::ViciPkiProfiles'

   def __init__( self, vrfId, vrfName, parent, daemonConfig, daemonStatus,
                 routingHwStatus, sslStatus, templateDir, baseDir, origStrongswanDir,
                 netNsName, serviceRestartDelay, pkiProfiles ):

      traceFunc( 'vrfId: ', bv( vrfId ), ' vrfName: ', bv( vrfName ) )

      self.vrfId = vrfId
      self.vrfName = vrfName
      self.parent = parent
      self.daemonConfig = daemonConfig
      self.daemonStatus = daemonStatus
      self.routingHwStatus = routingHwStatus
      self.sslStatus = sslStatus
      self.templateDir = templateDir
      self.baseDir = baseDir
      self.origStrongswanDir = origStrongswanDir
      self.netNsName = netNsName
      self.pkiProfiles = pkiProfiles
      self.cleanupState = 'not started'
      self.deleted = False

      self.loadCredsCmdState = None
      self.flushCredsCmdState = None

      assert '/' not in vrfName

      self.serviceName = f'strongswan-VRF-{vrfName}'

      runDir = '%s/ipsec.d/run' % self.baseDir
      self.starterPidFile = '%s/starter.charon.pid' % runDir
      self.charonPidFile = '%s/charon.pid' % runDir
      self.charonCtlFile = '%s/charon.ctl' % runDir
      self.logFileName = '/var/log/agents/charon-VRF-%s.log' % vrfName
      self.strongswanConfFileName = '%s/strongswan.conf' % self.baseDir

      # VICI update : Dont update ipsec.conf and secrets file
      self.ipsecConfFileName = '%s/ipsec.conf' % self.baseDir
      self.ipsecSecretsFileName = '%s/ipsec.secrets' % self.baseDir

      # Instead add swanctl.conf file
      self.swanCtlConfFileName = '%s/swanctl/swanctl.conf' % self.baseDir

      # setting these to 4 will make strongswan produce some very
      # verbose logs, which is useful for debugging.  Too spammy for
      # production though.
      self.defLogLevel = 1
      self.knlLogLevel = 2
      self.ikeLogLevel = 2
      self.cfgLogLevel = 1

      # The original version of this code would pull the openssl
      # config file name out of the environment variable
      # 'STRONGSWAN_CHARON_OPENSSL_CONF'.  AFAICT we never actually
      # use that, though, so I left it out here.
      self.openSslConfFileName = f'{self.baseDir}/strongswan.d/charon/openssl.conf'

      # VICI update: Removed ipsec.conf and secrets from SuperServer.LinuxService
      # config files
      fileNameList = [ self.strongswanConfFileName,
            self.openSslConfFileName, self.swanCtlConfFileName ]

      self.simulatedRunning = False

      # VICI update : Removed any usage of Connection Manager

      # VICI update : Removed any usage PKI to connection Mapping
      # as the mapping part will be handled by the VICI layer

      self.createVrfFiles()
      if self.parent.canWriteToSysdb():
         self.daemonStatus.connStatus.clear()
         self.updateDaemonStatus()

      serviceName = 'strongswan'
      linuxServiceName = f'{serviceName}-VRF-{vrfName}'
      SuperServer.LinuxService.__init__( self, serviceName, linuxServiceName,
                                         # Vici Update: we dont need
                                         # confData TAC notifier anymore
                                         pkiProfiles, confFilenames=fileNameList,
                                         serviceRestartDelay=serviceRestartDelay,
                                         healthCheckNeeded=False )

      # this is our reimplementation of the health check
      self.healthCheckTimer = Tac.ClockNotifiee()
      self.healthCheckTimer.handler = self.handleHealthCheckTimer
      self.healthCheckInterval = 15
      # wait at least this long after starting the service before
      # complaining about it not running
      self.startHealthDelay = 5

      self.scheduleHealthCheck()
      self.serviceStartedAt = Tac.beginningOfTime

   def _maybeRestartService( self ):
      if self.deleted:
         traceFunc( "This service for", bv( self.vrfName ),
                    "is deleted and should be garbage collected shortly" )
         return
      super()._maybeRestartService()

   def onSwitchover( self ):
      traceFunc( 'vrf: ', bv( self.vrfName ) )
      self.updateDaemonStatus()

   def needsReloadIs( self, b ):
      traceFunc()
      assert b
      if self.isRunning():
         self.stopService()
      self.startService()

   def createVrfFiles( self ):
      traceFunc( bv( self.vrfName ) )

      # we are going to copy the entire template directory of
      # strongswan config files into a per-VRF location.  First we
      # need to empty out that location in case there are any leftover
      # files there.  Again, this would change if/when we support
      # SuperServer agent restart
      try:
         shutil.rmtree( self.baseDir )
      except OSError as e:
         if e.errno != errno.ENOENT:
            traceVerbose( 'init cleanup rm gave', bv( e.strerror ) )

      # and now create the new base directory
      try:
         shutil.copytree( self.templateDir, self.baseDir )
      except OSError as e:
         traceVerbose( 'copying', bv( self.templateDir ), 'to', bv( self.baseDir ),
                       'gave', bv( e.strerror ) )

   def addNetNsPrefix( self, cmd ):
      netNsCmd = cmd

      if self.vrfName != DEFAULT_VRF:
         netNsCmd = f'ip netns exec {self.netNsName} {netNsCmd}'

      return netNsCmd

   # this is called when the VRF is deleted
   def cleanupService( self ):
      traceFunc( 'vrf: ', bv( self.vrfName ) )

      SuperServer.LinuxService.cleanupService( self )
      self.healthCheckTimer.timeMin = Tac.endOfTime

      self.cleanupState = 'started'
      self.stopService()

      try:
         shutil.rmtree( self.baseDir )
      except OSError as e:
         if e.errno != errno.ENOENT:
            traceVerbose( 'rmtree', bv( self.baseDir ), 'gave', bv( e.strerror ) )
      self.cleanupState = 'done'
      traceVerbose( 'cleanup done', bv( self.vrfName ) )

   # push the current daemon state into sysdb if allowed
   def updateDaemonStatus( self ):
      traceFunc( bv( self.vrfName ), ' btest: ',
                 bv( self.daemonConfig.simulateDaemonForBtest ), ' sim: ',
                 bv( self.simulatedRunning ) )

      if not self.parent.canWriteToSysdb():
         return

      self.daemonStatus.fipsMode = self.parent.ipsecConfig.fipsRestrictions
      if self.daemonConfig.simulateDaemonForBtest:
         runValue = self.simulatedRunning
      else:
         runValue = self.isRunning()
         traceVerbose( 'set daemonStatus sswanRunning to', bv( runValue ) )
      self.daemonStatus.sswanRunning = runValue

   def serviceEnabled( self ):
      if( not self.parent.ipsecStatus.ipsecLicenseEnabled or
          not self.parent.daemonConfig.enableSswan ):
         return False

      # VICI Update : TODO
      # This would already be checked in handleVrfId
      # Do we need this here again ? Comment for now
      # if len( self.shadowConfData ) == 0:
      #   return False

      # VICI Update : TODO
      # As we are not passing confData to this class,
      # the following functionality is not possible
      # Find another to way to check this , else fall back to this
      # For now commenting it

      # if we only have one connection and it's the default one then
      # we don't need to run the daemon
      # if len( self.shadowConfData ) == 1:
      #   key = list( self.shadowConfData.keys() )[ 0 ]
      #   if key == self.confData.defaultConnectionKey( self.vrfId ):
      #      return False

      return True

   def serviceProcessWarm( self ):
      # The agent is warm if it is not enabled
      if not self.serviceEnabled():
         warm = True
      else:
         running = self.isRunning()
         warm = running

      traceVerbose( bv( warm ) )
      return warm

   def isRunning( self ):
      # to be 'running', both the 'starter.charon.pid' and
      # 'charon.pid' processes need to be running.  Additionally, we
      # need to verify that there isn't some other process on the
      # system that is reusing the PID, so we also have to check that
      # both the command line of the process matches what we expect
      # and that the network namespace matches the VRF in question (so
      # we don't get confused by looking at a starter or charon
      # process in a different VRF)
      starterPid = getPid( self.starterPidFile )
      charonPid = getPid( self.charonPidFile )

      vrfNs = getNetNsInode( pid=None, vrfName=self.vrfName )
      for ( pid, procName ) in ( [ starterPid, 'starter' ],
                                 [ charonPid, 'charon' ] ):
         if not pidExists( pid ):
            return False

         f = None
         try:
            f = open( '/proc/%d/cmdline' % pid )
            cmdline = f.readline()
         except ( OSError, ProcessLookupError ):
            return False
         if not cmdline.startswith( '/usr/libexec/strongswan/%s' % procName ):
            return False

         netNs = getNetNsInode( pid )
         if not netNs or netNs != vrfNs:
            return False

      return True

   def _runServiceCmd( self, cmd ):
      # we are not using this, it should never get called
      assert 0

   # override the default LinuxService startService() command becasue
   # we want to invoke the strongswan starter process directly rather
   # than using scripts in /etc/init.d
   def startService( self ):
      assert self.serviceEnabled()
      traceFunc( bv( self.vrfName ), ' vrfid: ', bv( self.vrfId ) )

      if self.isRunning():
         traceVerbose( 'already running' )
         return

      if not os.path.exists( self.baseDir ) or not os.path.isdir( self.baseDir ):
         traceVerbose( 'sswan VRF template files not created,return' )
         return

      if self.daemonConfig.simulateDaemonForBtest:
         self.simulatedRunning = True
         self.updateDaemonStatus()
         return

      # make sure we don't have any leftover files from the previous instance
      for f in ( self.starterPidFile, self.charonPidFile, self.charonCtlFile ):
         try:
            os.unlink( f )
         except OSError:
            pass

      starterCmd = '/usr/libexec/strongswan/starter --daemon charon'
      cmd = self.addNetNsPrefix( starterCmd )
      traceVerbose( 'starter cmd ', bv( cmd ) )
      with RunInMntNamespace( self.baseDir, self.origStrongswanDir ):
         os.system( cmd )

      self.updateDaemonStatus()

      self.serviceStartedAt = Tac.now()

   # override the default stopService for the same reason

   def stopService( self ):
      traceFunc( bv( self.vrfName ), 'vrfid: ', bv( self.vrfId ) )

      if self.daemonConfig.simulateDaemonForBtest:
         self.simulatedRunning = False
         self.updateDaemonStatus()
         return

      starterPid = getPid( self.starterPidFile )
      charonPid = getPid( self.charonPidFile )
      if pidExists( starterPid ):
         killProc( starterPid, 'stopService starter' )
      if pidExists( charonPid ):
         killProc( charonPid, 'stopService charon' )

      self.updateDaemonStatus()

      self.parent.cleanupOldStrongswanInstances( self.vrfName )

   def restartService( self ):
      # we don't use this (restartWithFileChanges instead), so assert
      assert 0

   # override the default restartService becasue we need to figure out
   # exactly what to change
   def restartWithFileChanges( self, oldConfig, newConfig ):
      traceFunc( bv( self.vrfName ), 'vrfid: ', bv( self.vrfId ) )

      if self.daemonConfig.simulateDaemonForBtest:
         self.simulatedRunning = True

      if not self.isRunning():
         traceVerbose( 'service not running, falling back to stop/startService' )
         self.stopService()
         self.startService()
         return

      if( ( oldConfig[ self.openSslConfFileName ] !=
            newConfig[ self.openSslConfFileName ] ) or
          ( oldConfig[ self.strongswanConfFileName ] !=
            newConfig[ self.strongswanConfFileName ] ) ):
         # changing strongswan.conf or openssl.conf requires
         # restarting the whole thing -- we have to restart the whole
         # thing.

         if self.parent.extraDebug:
            if( oldConfig[ self.openSslConfFileName ] !=
                newConfig[ self.openSslConfFileName ] ):
               traceVerbose( 'ssl changed' )
               for line in oldConfig[ self.openSslConfFileName ].split( '\n' ):
                  traceVerbose( 'old:', bv( line ) )
               for line in newConfig[ self.openSslConfFileName ].split( '\n' ):
                  traceVerbose( 'new:', bv( line ) )

            if( oldConfig[ self.strongswanConfFileName ] !=
                newConfig[ self.strongswanConfFileName ] ):
               traceVerbose( 'ssl changed' )
               for line in oldConfig[ self.strongswanConfFileName ].split( '\n' ):
                  traceVerbose( 'old:', bv( line ) )
               for line in newConfig[ self.strongswanConfFileName ].split( '\n' ):
                  traceVerbose( 'new:', bv( line ) )

         traceVerbose( 'Forced full restart' )
         self.stopService()
         self.startService()
         return

      # VICI Update
      # if the swanctl.conf file is changed then we need to reload the certs
      # as currently we are only relying on swactl for certs and not
      # for connections
      if( oldConfig[ self.swanCtlConfFileName ] !=
          newConfig[ self.swanCtlConfFileName ] ):
         traceVerbose( 'swanctl reload the Certs' )
         self.swanCtlReloadCerts()

      self.restarts += 1

   def handleHealthCheckTimer( self ):
      self.healthCheckTimer.timeMin = Tac.endOfTime
      assert self.cleanupState == 'not started'
      traceTimer( 'vrf ', bv( self.vrfName ) )

      now = Tac.now()
      if( self.serviceEnabled() and
          ( now - self.serviceStartedAt ) > self.startHealthDelay ):

         failStr = None
         if not self.isRunning() and not self.serviceRestartPending_:
            traceFunc( 'IPSEC health check restart', bv( self.vrfName ) )
            traceVerbose( 'serviceStartedAt', bv( self.serviceStartedAt ), 'now',
                          bv( now ), 'delay', bv( self.startHealthDelay ) )
            self.stopService()
            self.startService()

            failStr = f'strongswan-VRF-{self.vrfName}'

         if failStr:
            self.logRestart( failStr )

      self.scheduleHealthCheck()

   def scheduleHealthCheck( self ):
      if self.daemonConfig.simulateDaemonForBtest:
         traceTimer( 'no health check due to btest simulation' )
         return

      self.healthCheckTimer.timeMin = Tac.now() + self.healthCheckInterval

   ######################################################################
   # Functions to manage swanctl commands

   def runSwanctlCmd( self, cmd ):
      swanCtlCmd = '/usr/sbin/swanctl'
      cmdList = [ swanCtlCmd, cmd ]
      if self.netNsName != 'default':
         cmdList = [ 'ip', 'netns', 'exec', self.netNsName ] + cmdList

      traceVerbose( 'Running Swanctl command :',
                    bv( ' '.join( cmdList ) ) )
      with RunInMntNamespace( self.baseDir, self.origStrongswanDir ):
         result = subprocess.run( cmdList, text=True, check=False )
      if result.returncode:
         traceFunc( 'Command:', bv( cmd ), 'failed error code:',
                        bv( result.returncode ), 'error :',
                        bv( result.stderr ) )
         return False
      return True

   def swanCtlReloadCerts( self ):
      traceFunc( 'swanctl Reload certs vrfId: ', bv( self.vrfId ) )
      backout = 3 # backout and fail after 3 retries , number to be decided
      i = 1
      # Run the load creds command with a backout number
      cmd = '--load-creds'
      while not self.runSwanctlCmd( cmd ):
         if i >= backout:
            return False
         i = i + 1
      i = 1
      cmd = '--flush-certs'
      while not self.runSwanctlCmd( cmd ):
         if i >= backout:
            return False
         i = i + 1

      return True

   ######################################################################
   # this set of functions are used to create the config files

   def conf( self ):
      assert self.cleanupState == 'not started'

      traceFunc( bv( self.vrfName ), 'vrfId: ', bv( self.vrfId ) )

      d = dict()
      traceFunc( 'files are', bv( self.swanCtlConfFileName ),
                 bv( self.strongswanConfFileName ),
                 bv( self.openSslConfFileName ) )
      d[ self.swanCtlConfFileName ] = self.swanCtlConf()
      d[ self.strongswanConfFileName ] = self.strongswanConf()
      d[ self.openSslConfFileName ] = self.openSslConf()

      return d

   def connName( self, key ):
      connId = f'{key.connId.connIdSrc}-{key.connId.connId}'
      return f'{self.vrfName}-{key.srcAddr}-{key.dstAddr}-{connId}'

   def vrfNameFromId( self, vrfId ):
      return ''

   def swanCtlConf( self ):
      # VICI Update : TODO

      # Parse the pkiProfiles provided by the Vici layer and convert
      # it into format expected by swanctl if and when required
      config = ''

      return config

   def openSslConf( self ):
      data = "openssl {\n"
      data += "  load = yes\n"
      if self.parent.ipsecConfig.fipsRestrictions:
         data += "  fips_mode = 1\n"
         data += f"  fips_mode_logging = {int(int(self.daemonConfig.fipsLogging))}\n"
      data += "}\n"
      return data

   def strongswanConf( self ):
      # Set kernel_forward to 'no' in Sfe mode
      # timeout means the milliseconds strongswan will wait for the netlink
      # message response from Ipsec. In Sfe mode, with this, we will not get into
      # the deadlock when strongswan do GETSA on an already deleted SA, and we have
      # deleted the vrfRoot. Otherwise, strongswan cannot handle "sudo strongswan
      # stop" request since it's busy in waiting GETSA response. Set timeout to '0'
      # in Sfa mode since it's not needed ( '10000' works fine for stress test with
      # 128 tunnels in Sfe mode )
      if self.routingHwStatus.kernelForwardingSupported:
         kernel_forward = 'yes'
         parallel_xfrm = 'no'
         timeout = '0'
      else:
         kernel_forward = 'no'
         parallel_xfrm = 'yes'
         timeout = '10000'
      # Changing the number of strongswan threads = 24, to process strongswan
      #    up/down.
      # Default value for threads is 16
      # Changing the max_concurrency = 16, to queue multiple strongswan requests.
      # Default value for max_concurrency is 4.
      # Setting inactivity_close_ike closes lingering IKE_SAs if the only
      #    CHILD_SA is closed due to inactivity. IKE_SAs linger if the
      #    ike-policy config matches but the sa-policy config doesn't.

      # Set the min/max SPI
      # Valid SPI range
      # 0xC0000000 - 0xCDFFFFFF : reserved for Tunnel intf and
      #                           AutoVPN static path
      # NOTE :: The range MUST be consistent with the range defined
      #         in Ipsec::SPIRange
      spiMin = '0xc0000000'
      spiMax = '0xcdffffff'
      if self.parent.ipsecCapabilitiesStatus.spi24Bits:
         spiMin = '0xc00000'
         spiMax = '0xcdffff'

      config = """
# config file came from new SuperServerPlugin code
charon {
        vrf_name = %s
        load_modular = yes
        install_routes = no
        retransmit_tries = 2
        retransmit_timeout = 8.0
        make_before_break = yes
        delete_rekeyed_delay = 10
        threads = 24
        inactivity_close_ike = yes
        close_ike_on_child_failure = yes
        filelog {
                charon {
                     path = %s
                     time_format = %%b %%e %%T
                     append = yes
                     default = %d
                     knl = %d
                     ike = %d
                     cfg = %d
                     flush_line = yes
                     time_add_ms = yes
                 }
                 stdout {
                     default = 0
                     ike_name = yes
                 }
        }
        syslog {
               # enable logging to LOG_DAEMON, use defaults
               daemon {
                    default = -1
               }
               # minimalistic IKE auditing logging to LOG_AUTHPRIV
               auth {
                    default = -1
               }
        }
        plugins {
                include strongswan.d/charon/*.conf
                stroke {
                  max_concurrent = 16
                }
                kernel-netlink {
                    fwmark = !0x42
                    roam_events = no
                    kernel_forward = %s
                    parallel_xfrm = %s
                    timeout = %s
                    retries = 2
                }
                socket-default {
                    fwmark = 0x42
                }
                kernel-libipsec {
                    allow_peer_ts = yes
                }
        }
        spi_min = %s
        spi_max = %s
}

include strongswan.d/*.conf
      """ % ( self.vrfName, self.logFileName, self.defLogLevel, self.knlLogLevel,
              self.ikeLogLevel, self.cfgLogLevel, kernel_forward, parallel_xfrm,
              timeout, spiMin, spiMax )
      return config

   # VICI Update : TODO Pki profile Tacc handler
   # Handler for reacting to changes in pki profiles
   # if any profile is added/del or modified SuperServer
   # simply does swanctl load-creds and flush-certs
   # Connection pertaining to these are managed by VICI
   # layer in ipsec agent
   # This will be updated in further mut , once the
   # PkiProfile struct is defined

   # @Tac.handler( 'pkiVersion' )
   # def handlePkiVersion( self ):
   #   traceFunc( self.vrfName, 'vrfId:', bv( self.vrfId ) )

# VICI Update : Resuing the IpsecServiceParent reactors
# As most of the reactor functionality remains same for
# ViciIpsecServiceParent and IpsecServerParent (old code)
# we will reusing the same handlers at Line 2720
# with one exception of connStatusNetlink handler which wont
# be using in Vici hence remains unuses

#
# This is the main class for the VRF-aware Ipsec strongswan
# SuperServer plugin.  We have one of these, and it is created at load
# time.  It reacts to the VRFs coming and going and creates a
# PerVrfService object for each instance as necessary.  It also sets
# up the initial template directory of ipsec-specific files,

class ViciIpsecServiceParent:
   def __init__( self, confDataDir, ipsecConfig, ipsecStatus,
                 daemonConfig, daemonStatusDir, routingHwStatus, vrfIdStatus,
                 vrfNameStatus, allVrfStatusLocal, ipsecCapabilitiesStatus,
                 redStatus, sslStatus ):

      # VICI Update : Removed occurences of connStatusNetlink
      # As we wont be using csn anymore , removed the arg, reactors
      # and handlers

      traceFunc( '' )

      self.initComplete = False

      self.extraDebug = False

      self.confDataDir = confDataDir

      self.ipsecConfig = ipsecConfig
      self.ipsecStatus = ipsecStatus
      self.daemonConfig = daemonConfig
      self.daemonStatusDir = daemonStatusDir
      self.routingHwStatus = routingHwStatus
      self.vrfIdStatus = vrfIdStatus
      self.vrfNameStatus = vrfNameStatus
      self.allVrfStatusLocal = allVrfStatusLocal
      self.ipsecCapabilitiesStatus = ipsecCapabilitiesStatus
      self.redStatus = redStatus
      self.sslStatus = sslStatus

      self.confDataDirReactor = ConfDataDirReactor( self.confDataDir, self )
      self.daemonConfigReactor = DaemonConfigReactor( self.daemonConfig, self )
      self.ipsecConfigReactor = IpsecConfigReactor( self.ipsecConfig, self )
      self.ipsecStatusReactor = IpsecStatusReactor( self.ipsecStatus, self )
      self.vrfNameStatusReactor = VrfNameStatusReactor( self.vrfNameStatus, self )
      self.vrfIdStatusReactor = VrfIdStatusReactor( self.vrfIdStatus, self )
      self.icsReactor = IpsecCapabilitiesStatusReactor(
         self.ipsecCapabilitiesStatus, self )
      self.avslReactor = AllVrfStatusLocalReactor( self.allVrfStatusLocal, self )

      self.avslCollReactor = Tac.collectionChangeReactor(
         self.allVrfStatusLocal.vrf, VrfStatusLocalReactor,
         reactorArgs=( weakref.proxy( self ), ) )

      self.perVrfService = dict()

      self.sswanDir = '/etc/sswan'
      self.strongswanVrfDir = f'{self.sswanDir}/vrf'
      self.strongswanSocketDir = f'{self.sswanDir}/sockets'
      self.templateDir = f'{self.strongswanVrfDir}/template'
      self.origStrongswanDir = '/etc/strongswan'

      self.shadowRedMode = self.redStatus.mode
      self.shadowRedProto = self.redStatus.protocol

      self.cryptoModulesLoaded = False

      self.createStrongswanVrfDir()
      self.cleanupOldVrfInstances()

      self.agentGeneration = self.daemonConfig.agentGeneration

      self.connIdCounter = 1
      self.cleanupOldStrongswanInstances()
      self.initComplete = True

      self.maybeFinishInit()

   def handleNameToIdMap( self ):
      self.maybeFinishInit()

   def maybeFinishInit( self ):
      if self.initComplete and self.vrfNameStatus.nameToIdMap:
         for vrfId in self.confDataDir.conf:
            self.handleVrfId( vrfId )

   def handleAgentGeneration( self ):
      traceFunc( '' )
      # when the Ipsec agent restarts we need to fully restart all of
      # the strongswan instances so that they can resync with the unix
      # domain sockets
      if self.daemonConfig.agentGeneration != self.agentGeneration:
         traceFunc( 'restarting strongswan services due to agent gen' )
         self.agentGeneration = self.daemonConfig.agentGeneration
         allVrfIds = list( self.perVrfService.keys() )
         for v in allVrfIds:
            self.deleteService( v )

         for v in allVrfIds:
            self.handleVrfId( v )

   def readNetnsData( self ):
      netnsData = dict()
      allNetnsInodes = set()
      nsFiles = os.listdir( '/var/run/netns' )
      for f in nsFiles:
         if f == 'default':
            vrfName = 'default'
         elif f.startswith( 'ns-' ):
            vrfName = f[ 3 : ]
         else:
            continue

         statResult = None
         try:
            statResult = os.stat( f'/var/run/netns/{f}' )
         except OSError:
            pass

         if not statResult:
            continue

         netnsData[ vrfName ] = statResult.st_ino
         allNetnsInodes.add( statResult.st_ino )

      return ( netnsData, allNetnsInodes )

   # search through /proc for old instances of strongswan-related
   # processes in the specified VRF and kill them.  If vrfName is
   # none, do it for all VRFs.
   def cleanupOldStrongswanInstances( self, vrfName=None ):
      traceFunc( 'vrf ', bv( vrfName ) )
      if self.daemonConfig.simulateDaemonForBtest:
         return

      # on a physical dut we know that this SuperServerPlugin owns all
      # strongswan-related processes, but unfortunately namespace DUTs
      # do not create a PID namespace.  This means that on an ns dut
      # (say 'rtr1'), searching through /proc will also show us
      # processes for other ns duts in the same container (say
      # 'rtr2').  This can happen if a single ptest or stest starts
      # two duts, or if two tests are running at the same time.
      #
      # to avoid inadvertently killing strongswan processes belonging
      # to a different DUT we build a list of all of the inodes for
      # the network namespaces in the current dut, by iterating over
      # /var/run/netns/*.  We then only kill a strongswan-related
      # process if /proc/<pid>/ns/net for that process matches an
      # inode in that list.
      #
      # This approach will potentially miss strongswan instances that
      # are still running in a network namespace for a VRF that has
      # just been deleted, but they should exit on their own anyway
      ( netnsData, allNetnsInodes ) = self.readNetnsData()

      procs = os.listdir( '/proc' )
      for procFile in procs:
         if not procFile[ 0 ].isdigit():
            continue
         try:
            try:
               f = open( f'/proc/{procFile}/cmdline' )
               cmdline = f.readline()
            except ( OSError, ProcessLookupError ):
               continue

            if( cmdline.startswith( '/usr/libexec/strongswan' ) and
                'strongswan_fips_cmvp' not in cmdline ):
               # strongswan_fips_cmvp is a process performing a fips selfTest.
               # This selfTest can be executed by the Ipsec agent after it is
               # restarted. Do not kill it - otherwise the IPsec agent will start,
               # but will not bring up any connections.
               pid = int( procFile )
               nsFileInode = getNetNsInode( pid )

               if not nsFileInode:
                  # process must have died already for the /proc file
                  # to have gone away
                  continue

               if( ( vrfName is None and nsFileInode in allNetnsInodes ) or
                   ( vrfName is not None and vrfName in netnsData and
                     netnsData[ vrfName ] == nsFileInode ) ):
                  traceVerbose( 'cleaning up pid ', bv( pid ) )
                  killProc( pid, 'old instance' )
            f.close()
         except OSError:
            pass

   def createStrongswanVrfDir( self ):
      traceFunc( '' )

      def newDir( dirName ):
         if os.path.exists( dirName ) and not os.path.isdir( dirName ):
            try:
               traceVerbose( 'trying to unlink', bv( dirName ) )
               os.unlink( dirName )
            except OSError as e:
               traceVerbose( 'unlink', bv( dirName ), bv( e.strerror ) )

         if not os.path.exists( dirName ):
            try:
               traceVerbose( 'creating', bv( dirName ) )
               os.mkdir( dirName, 0o700 )
            except OSError as e:
               traceVerbose( 'mkdir', bv( dirName ), bv( e.strerror ) )

      newDir( self.sswanDir )
      newDir( self.strongswanVrfDir )
      newDir( self.strongswanSocketDir )
      newDir( self.templateDir )

      for d in [ 'ipsec.d', 'strongswan.d', 'swanctl' ]:
         newDir( f'{self.templateDir}/{d}' )

      newDir( f'{self.templateDir}/strongswan.d/charon' )
      newDir( f'{self.templateDir}/ipsec.d/run' )

      # here are some of the empty directories that we need to keep
      # strongswan from complaining
      ipsecDPath = f'{self.templateDir}/ipsec.d'
      for d in [ 'cacerts', 'aacerts', 'ocspcerts', 'acerts', 'crls' ]:
         newDir( f'{ipsecDPath}/{d}' )

      for d in [ 'certs', 'private', 'reqs' ]:
         newDir( f'{ipsecDPath}/{d}' )

      # now create the varous config files that we don't actually change based
      # on Sysdb state.

      # the dist dir has a file in swanctl/swanctl.conf, but the
      # entire file is a comment so I'm going to leave it out

      # strongswan.d has a bunch of config files, but most of them
      # don't actually seem to do much.  The "minimal" ones just
      # create a struct with a name

      strongswanDPath = f'{self.templateDir}/strongswan.d'
      minimalSwanDFiles = [ 'scepclient', 'swanctl', 'pki', 'starter' ]

      def writeFile( filePath, contents ):
         f = open( filePath, 'w' )
         assert f
         f.write( contents )
         f.close()

      for f in minimalSwanDFiles:
         conf = '''
%s {
} ''' % f
         path = f'{strongswanDPath}/{f}.conf'
         writeFile( path, conf )

      # Next we have a few files in the strongswan.d path that have
      # some additional "null" structs inside them.

      charonConf = '''
charon {
    crypto_test {
    }
    host_resolver {
    }
    leak_detective {
    }
    processor {
        priority_threads {
        }
    }
    start-scripts {
    }
    stop-scripts {
    }
    tls {
    }
    x509 {
    }
}
'''
      writeFile( f'{strongswanDPath}/charon.conf', charonConf )

      # VICI Update : TODO
      # Should we add some control here for log levels to be configured
      # via cli or a macro
      # swanctl provide reload-settings command which will change the
      # logging level if required
      charonLoggingConf = '''
charon {
    filelog {
    }
    syslog {
    }
}
'''
      writeFile( f'{strongswanDPath}/charon-logging.conf',
                 charonLoggingConf )

      # Now we do the ones in strongswan.d/charon/*.conf
      #
      # Again there are a bunch of these that are "minimal" -- all
      # they do is specify "load = yes", so handle them together
      #
      # note that in the dist files openssl.conf would be in this
      # list, but we're going to handle that specially (it depends on
      # the FIPS config)

      charonPath = f'{strongswanDPath}/charon'

      minimalCharonConf = [
         'pkcs1', 'pkcs7', 'pkcs8', 'pkcs12', 'nonce', 'revocation', 'vici',
         'xauth-generic', 'pubkey', 'socket-default', 'constraints', 'dnskey',
         'attr', 'pem', 'updown', 'stroke', 'x509', 'curl', 'pgp', 'sshkey',
         'eap-tls' ]

      for f in minimalCharonConf:
         path = f'{charonPath}/{f}.conf'
         conf = '''
%s {
    load = yes
}
''' % f
         writeFile( path, conf )

      # three of the files (kernel-netlink, pkcs11, and resolve) have
      # empty sub-structures in them.
      kernelNetlinkConf = '''
kernel-netlink {
    load = yes
    spdh_thresh {
        ipv4 {
        }
        ipv6 {
        }
    }
}
'''
      writeFile( f'{strongswanDPath}/charon/kernel-netlink.conf',
                 kernelNetlinkConf )

      pkcs11Conf = '''
pkcs11 {
    load = yes
    modules {
        <name> {
        }
    }
}
'''
      writeFile( f'{strongswanDPath}/charon/pkcs11.conf', pkcs11Conf )

      resolveConf = '''
resolve {
    load = yes
    resolvconf {
    }
}
'''
      writeFile( f'{strongswanDPath}/charon/resolve.conf', resolveConf )

      # VICI Update : swanctl.conf update
      # Swanctl is the library provided by strongswan to utilizie VICI
      # Though we use socket based IPC for vici communication via Ipsec Agent
      # There is a gap for pki infra. For any certs or keys we will be using
      # swanctl.conf file to load them into strongswan instead of vici socket
      # hence we would populate this file accordingly based on sysdb updates
      # for now create a empty file
      writeFile( f'{self.templateDir}/swanctl/swanctl.conf',
                 '# This will be used for pki infra ,certs and keys\n' )

      # and that's it.

   def canWriteToSysdb( self ):
      return( self.redStatus.mode == 'active' or
              self.redStatus.protocol != 'sso' )

   def onSwitchover( self ):
      traceFunc( 'mode', bv( self.redStatus.mode ),
                 'proto', bv( self.redStatus.protocol ),
                 'shadow mode', bv( self.shadowRedMode ),
                 'shadow proto', bv( self.shadowRedProto ) )
      if( self.redStatus.mode == self.shadowRedMode and
          self.redStatus.protocol == self.shadowRedProto ):
         traceVerbose( 'no change' )
         return

      if not self.canWriteToSysdb():
         # we transitioned to standby?  Is this possible?
         return

      # create any daemon status entities that are missing
      for pvs in self.perVrfService.values():
         if pvs.vrfName not in self.daemonStatusDir.daemonStatus:
            traceVerbose( 'creating DSD', bv( pvs.vrfName ) )
         else:
            traceVerbose( 'already have DSD', bv( pvs.vrfName ) )

         # Create new status or get the existing one
         # When the standby transitions to active, the daemonStatusDir is already
         # synced. But, pvs has not been updated with the daemonStatus yet.
         # Update it here.
         daemonStatus = self.daemonStatusDir.newDaemonStatus( pvs.vrfName )
         pvs.daemonStatus = daemonStatus
         pvs.onSwitchover()

      # and purge any that are no longer needed
      avn = self.allVrfNames()
      for vrfName in self.daemonStatusDir.daemonStatus:
         if vrfName not in avn:
            traceVerbose( 'cleaning up DSD', bv( vrfName ) )
            del self.daemonStatusDir.daemonStatus[ vrfName ]
         else:
            traceVerbose( 'leaving DSD intact', bv( vrfName ) )

   # this is used in the breadth tests to try to make sure we don't
   # have any dangling references from a previous instance of the
   # manager left in the process space of a restart test
   def cleanupRefs( self ):
      # VICI Update : delete csnReactor
      # Cleanup any unused reactors
      traceFunc( '' )
      for pvs in self.perVrfService.values():
         pvs.cleanupService()
      self.perVrfService.clear()

      del self.confDataDirReactor
      self.confDataDirReactor = None
      del self.daemonConfigReactor
      self.daemonConfigReactor = None
      del self.ipsecConfigReactor
      self.ipsecConfigReactor = None
      del self.ipsecStatusReactor
      self.ipsecStatusReactor = None
      del self.vrfIdStatusReactor
      self.vrfIdStatusReactor = None
      del self.icsReactor
      self.icsReactor = None

   def allVrfNames( self ):
      avn = list()
      for pvs in self.perVrfService.values():
         avn.append( pvs.vrfName )

      return avn

   # this purges any VRF directories or daemonStatus entries that
   # should not exist.  it's used on agent startup (mainly useful on
   # SuperServer restart)
   def cleanupOldVrfInstances( self ):
      traceFunc( '' )

      if os.path.isdir( self.strongswanVrfDir ):
         files = os.listdir( self.strongswanVrfDir )
         for f in files:
            fullPath = self.strongswanVrfDir + '/' + f
            if f == self.templateDir:
               continue
            elif f.startswith( 'VRF-' ):
               m = re.match( 'VRF-(.*)$', f )
               assert m
               vrfName = m.groups()[ 0 ]
               traceVerbose( 'cleaning up vrf dir', bv( fullPath ) )
               try:
                  shutil.rmtree( fullPath )
               except OSError as e:
                  traceVerbose( 'rmtree', bv( fullPath ), 'gave',
                                bv( e.strerror ) )
            else:
               traceVerbose( 'ignoring non-VRF dir', bv( fullPath ) )

      if self.canWriteToSysdb():
         for vrfName in self.daemonStatusDir.daemonStatus:
            traceVerbose( 'cleanupDeadVrfDirs: clean', bv( vrfName ) )
            del self.daemonStatusDir.daemonStatus[ vrfName ]

   # VICI update : removed handleConnStatusNetlink
   # The reactor to connStatusNetlink wont be neccessary as the connection
   # trigger i.e up/down will be initiated from VICI layer in Ipsec Agent

   def handleVrfId( self, vrfId ):
      assert self.initComplete

      ve = self.vrfIdStatus.vrfIdToName.get( vrfId )
      vrfName = ve.vrfName if ve else None
      traceFunc( bv( vrfId ), bv( vrfName ) )

      if( not self.ipsecStatus.ipsecLicenseEnabled or
          not self.daemonConfig.enableSswan ):
         traceVerbose( 'service disabled vrfId ', bv( vrfId ) )
         self.deleteService( vrfId )
         return

      if not vrfName:
         traceVerbose( 'vrfName is None' )
         self.deleteService( vrfId )
         return

      if vrfId not in self.confDataDir.conf:
         traceVerbose( 'vrfId', bv( vrfId ), 'not in confDataDir.conf' )
         self.deleteService( vrfId )
         return

      if len( self.confDataDir.conf[ vrfId ].connection ) == 0:
         traceVerbose( 'no connections for vrfId', bv( vrfId ) )
         self.deleteService( vrfId )
         return

      if vrfId not in self.ipsecStatus.vrfsInUse:
         traceVerbose( 'vrfId', bv( vrfId ), 'not in ipsecStatus.vrfsInUse' )
         self.deleteService( vrfId )
         return

      if vrfId != DEFAULT_VRF_ID:
         vsl = self.allVrfStatusLocal.vrf.get( vrfName )
         if not vsl or not vsl.networkNamespace or vsl.state != 'active':
            traceVerbose( 'Invalid vsl state vrfId', bv( vrfId ), 'state',
                          bv( vsl.state if vsl else '-' ), 'ns',
                          bv( vsl.networkNamespace if vsl else '-' ) )
            self.deleteService( vrfId )
            return
         netNsName = vsl.networkNamespace
      else:
         netNsName = DEFAULT_NS

      # VICI Update : We Dont need confData here
      # As connection management is not part of Superserver anymore
      # we dont need to pass per vrf confData to PerVrfService
      # instead we pass pki profiles which struct is yet to be
      # implemented TODO
      #confData = self.confDataDir.conf.get( vrfId )
      pkiProfile = None
      if vrfId in self.perVrfService:
         # make sure pre-existing service is up to date and pointing
         # to the correct entity
         pvs = self.perVrfService[ vrfId ]
         if pvs.vrfName == vrfName:
            # we probably don't need to sync() here, but let's be paranoid...
            #
            # the check that vrfName is still the same is REALLY paranoid
            pvs.sync()
            return

         # if not up to date then we just delete it and re-create down
         # below
         traceVerbose( 'service not up to date vrfId ', bv( vrfId ) )
         self.deleteService( vrfId )
         pvs = None

      # either we cleaned up an out-of-date service or we are creating a new one
      daemonStatus = None
      if self.canWriteToSysdb():
         daemonStatus = self.daemonStatusDir.daemonStatus.get( vrfName )
         if not daemonStatus:
            traceVerbose( 'handleVrfId create DSD', bv( vrfName ) )
            daemonStatus = self.daemonStatusDir.newDaemonStatus( vrfName )
         assert daemonStatus

      traceVerbose( 'Create service', bv( vrfName ) )
      serviceRestartDelay = 2
      if self.daemonConfig.shortDelaysForBtest:
         # this makes the btest run much faster...
         serviceRestartDelay = 0
      vrfDir = self.strongswanVrfDir + ( f'/VRF-{vrfName}' )
      perVrfService = ViciIpsecPerVrfService
      if self.daemonConfig.simulateDaemonForBtest:
         perVrfService = IpsecPerVrfServiceTest
      pvs = perVrfService(
         vrfId, vrfName, self, self.daemonConfig, daemonStatus,
         self.routingHwStatus, self.sslStatus, self.templateDir, vrfDir,
         self.origStrongswanDir, netNsName, serviceRestartDelay, pkiProfile )
      assert pvs
      assert pvs.serviceRestartDelay_ == serviceRestartDelay
      self.perVrfService[ vrfId ] = pvs

      self.syncCryptoModules()
      return

   def syncCryptoModules( self ):
      cmd = 'rmmod'

      if len( self.perVrfService ) > 0:
         # if we have services and we've already loaded the modules,
         # we're done
         if self.cryptoModulesLoaded:
            return

         # we only load the modules if we have at least one vrf that
         # is enabled (basically meaning it has a non-default
         # connection in it)
         for pvs in self.perVrfService.values():
            if pvs.serviceEnabled():
               cmd = 'modprobe'
               break
      elif not self.cryptoModulesLoaded:
         # if no services and not loaded, nothing to do
         return

      # do the load/unload
      for mod in [ 'aes-x86_64', 'aesni_intel', 'crypto_null' ]:
         rc = os.system( f'/sbin/{cmd} {mod}' )
         if rc:
            traceVerbose( bv( cmd ), bv( mod ), 'failed', bv( rc ) )

      if cmd == 'modprobe':
         self.cryptoModulesLoaded = True
      else:
         self.cryptoModulesLoaded = False

   def deleteService( self, vrfId ):
      traceFunc( bv( vrfId ) )

      if vrfId not in self.perVrfService:
         return

      pvs = self.perVrfService[ vrfId ]
      vrfName = pvs.vrfName
      self.perVrfService[ vrfId ].cleanupService()
      del self.perVrfService[ vrfId ]
      pvs.deleted = True
      pvs = None

      if self.canWriteToSysdb():
         del self.daemonStatusDir.daemonStatus[ vrfName ]

   def handleVrfIdAll( self ):
      # handleVrfIdAll calls handleVrfId() on all the VRFs we know
      # about.  This is used when some global change is applied that
      # may start or stop the per-vrf services.
      traceFunc( '' )

      vrfIdSet = set( self.vrfIdStatus.vrfIdToName.keys() )
      vrfIdSet.update( set( self.confDataDir.conf.keys() ) )
      vrfIdSet.update( set( self.perVrfService.keys() ) )

      for vrfId in list( vrfIdSet ):
         self.handleVrfId( vrfId )

   def syncAll( self ):
      # syncAll is less intrusive than handleVrfIdAll, it simply
      # forces us to call sync() on all of the existing services.
      # sync() is implemented in the SuperServer.LinuxService base
      # class and triggers a delayed action to review all of the
      # config files for a particular service and then restart the
      # daemon if necessary.  We use this when global attributes
      # change that will not add or remove services, but may result in
      # a need to change the high-level config files like
      # strongswan.conf.
      traceFunc( '' )

      for pvs in self.perVrfService.values():
         pvs.sync()

   def handleVrfState( self, vrfName ):
      traceFunc( vrfName )

      vrfId = self.vrfNameStatus.nameToIdMap.vrfNameToId.get( vrfName )
      if vrfId:
         self.handleVrfId( vrfId )
      else:
         self.handleVrfIdAll()

class ViciSswanManager( SuperServer.SuperServerAgent ):
   """ Primary functionality of this is to manage strongswan processes
      under multiple VRF and update the daemon status accordingly to be
      used by Ipsec Agent """

   def __init__( self, entityManager ):
      traceFunc( 'Starting Vici SswanManager...' )

      # Explicitly pass our agent name down as 'SswanManager' so that our agent name
      # is consistent regardless of the toggle state. If we don't do this,
      # SuperServer will get confused and won't know who we are. Once we remove this
      # toggle, we should also remove this agentName argument and simply rename the
      # class to SswanManager
      SuperServer.SuperServerAgent.__init__( self, entityManager,
                                             agentName='SswanManager' )
      # Mount configurations
      mg = entityManager.mountGroup()

      ipsecConfig = mg.mount( 'ipsec/ike/config', 'Ipsec::Ike::Config', 'r' )
      ipsecStatus = mg.mount( 'ipsec/ike/status', 'Ipsec::Ike::Status', 'r' )
      daemonConfig = mg.mount( 'ipsec/daemonconfig', 'Ipsec::DaemonConfig', 'rO' )
      daemonStatusDir = mg.mount( 'ipsec/daemonstatusdir',
                                  'Ipsec::DaemonStatusDir', 'w' )
      routingHwStatus = mg.mount( 'routing/hardware/status',
                                  'Routing::Hardware::Status', 'r' )
      ipsecCapabilitiesStatus = mg.mount( 'ipsec/capabilities/status',
                                          'Ipsec::Capabilities::Status', 'r' )
      allVrfStatusLocal = mg.mount( Cell.path( 'ip/vrf/status/local' ),
                                    'Ip::AllVrfStatusLocal', 'r' )
      vrfNameStatus = mg.mount( Cell.path( 'vrf/vrfNameStatus' ),
                                'Vrf::VrfIdMap::NameToIdMapWrapper', 'r' )
      sslStatus = mg.mount( 'mgmt/security/ssl/status',
                            'Mgmt::Security::Ssl::Status', 'r' )
      # VICI update : removed connStatusNetlink mount
      # connStatusNetlinkg to be removed as SuperServer now wouldnt handle
      # and changes to confDataDir or rely on NetlinkStatus
      # ConfDataDir is still needed to check if SuperServer needs to start
      # Sswan on this vrf, based on the number of connection present in confData
      # Sswan will only be started if connection count is one or more
      # Can we move this check also to Ipsec Vici Layer and signal SuperServer
      # That is a TODO item for now
      confDataDir = mg.mount( Cell.path( 'ipsec/confData' ),
                              'Ipsec::Conf::ConfDataDir', 'r' )

      self.serviceParent = None

      shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
      smi = Smash.mountInfo( 'reader' )
      vrfIdStatus = shmemEm.doMount( 'vrf/vrfIdMapStatus', 'Vrf::VrfIdMap::Status',
                                     smi )

      self.mountsComplete = False

      def _finished():
         traceVerbose( 'Agent Mounted' )
         self.serviceParent = ViciIpsecServiceParent(
            confDataDir, ipsecConfig, ipsecStatus, daemonConfig,
            daemonStatusDir, routingHwStatus, vrfIdStatus, vrfNameStatus,
            allVrfStatusLocal, ipsecCapabilitiesStatus,
            self.redundancyStatus(), sslStatus )

      mg.close( _finished )

   def onSwitchover( self, protocol ):
      assert protocol
      if self.serviceParent:
         self.serviceParent.onSwitchover()

######################################################################
#
# End of VICI superServer code.
#
######################################################################

######################################################################
#
# This is the old plugin code below, will be removed later.
#
######################################################################
class IkeDaemonConfigNotifiee( SuperServer.LinuxService ):

   """ Reactor to changes in the boolean value of IpsecConfig
   that informs of strongswans running status """

   notifierTypeName = 'Ipsec::DaemonConfig'

   def __init__( self, serviceName, ikeConfig, ikeDaemonConfig, ikeDaemonStatusDir,
                 routingHwStatus, redStatus, ipsecCapabilitiesStatus ):
      self.started = False
      self.ikeDaemonConfig = ikeDaemonConfig
      self.ikeDaemonStatusDir = ikeDaemonStatusDir
      self.ikeDaemonStatus = None
      self.routingHwStatus = routingHwStatus
      self.redStatus = redStatus
      self.ipsecCapabilitiesStatus = ipsecCapabilitiesStatus
      self.sswanPidFile = '/etc/strongswan/ipsec.d/run/charon.pid'
      self.charonCtlFile = '/etc/strongswan/ipsec.d/run/charon.ctl'
      self.starterPidFile = '/etc/strongswan/ipsec.d/run/starter.charon.pid'

      # on standby sups we defer creation of the daemon status dir
      # (and all writes to it) until switchover happens
      self.maybeCreateIkeDaemonStatus()
      self.ikeDaemonFipsHandler = IkeDaemonFipsHandler( self, ikeConfig,
                                    self.ikeDaemonStatus, ikeDaemonConfig )

      if self.serviceProcessWarm():
         self.started = True
         self.starts = 1

      SuperServer.LinuxService.__init__( self, serviceName,
                                         'strongswan', ikeDaemonConfig,
                                         '/etc/strongswan/strongswan.conf' )

      self.maybeUpdateSswanRunning()

      bt9( 'End init: =%s started=%s isRunning=%s' %
           ( self.redStatus.mode, self.started, self.isRunning() ) )

   def canWriteToSysdb( self ):
      return( self.redStatus.mode == 'active' or
              self.redStatus.protocol != 'sso' )

   def maybeUpdateSswanRunning( self ):
      if not self.canWriteToSysdb():
         return

      self.maybeCreateIkeDaemonStatus()
      self.ikeDaemonStatus.sswanRunning = self.isRunning()

   def maybeCreateIkeDaemonStatus( self ):
      if not self.canWriteToSysdb():
         return

      # on switchover we may need to create the daemon status if the
      # active did not already do it
      if DEFAULT_VRF not in self.ikeDaemonStatusDir.daemonStatus:
         self.ikeDaemonStatusDir.newDaemonStatus( DEFAULT_VRF )

      self.ikeDaemonStatus = self.ikeDaemonStatusDir.daemonStatus[ DEFAULT_VRF ]

   def onSwitchover( self ):
      self.maybeUpdateSswanRunning()
      assert self.ikeDaemonStatus
      self.ikeDaemonFipsHandler.ikeDaemonStatus = self.ikeDaemonStatus
      self.ikeDaemonFipsHandler.onSwitchover()

   def serviceEnabled( self ):
      # If there are no mapped policies, strongswan should turn off
      enabled = self.ikeDaemonConfig.enableSswan

      bt9( f'serviceEnabled: {enabled}' )
      return enabled

   def serviceProcessWarm( self ):
      # The agent is warm if it is not enabled
      if not self.serviceEnabled():
         warm = True
      else:
         running = self.isRunning()
         warm = running

      bt9( f'serviceProcessWarm: {warm}' )
      return warm

   def conf( self ):
      bt9( 'conf() was called' )
      # Set kernel_forward to 'no' in Sfe mode
      # timeout means the milliseconds strongswan will wait for the netlink
      # message response from Ipsec. In Sfe mode, with this, we will not get into
      # the deadlock when strongswan do GETSA on an already deleted SA, and we have
      # deleted the vrfRoot. Otherwise, strongswan cannot handle "sudo strongswan
      # stop" request since it's busy in waiting GETSA response. Set timeout to '0'
      # in Sfa mode since it's not needed ( '10000' works fine for stress test with
      # 128 tunnels in Sfe mode )
      if self.routingHwStatus.kernelForwardingSupported:
         kernel_forward = 'yes'
         parallel_xfrm = 'no'
         timeout = '0'
      else:
         kernel_forward = 'no'
         parallel_xfrm = 'yes'
         timeout = '10000'

      logLevel = 1

      # Set the min/max SPI
      # Valid SPI range
      # 0xC0000000 - 0xCDFFFFFF : reserved for Tunnel intf and
      #                           AutoVPN static path
      # NOTE :: The range MUST be consistent with the range defined
      #         in Ipsec::SPIRange
      spiMin = '0xc0000000'
      spiMax = '0xcdffffff'
      if self.ipsecCapabilitiesStatus.spi24Bits:
         spiMin = '0xc00000'
         spiMax = '0xcdffff'

      # Changing the number of strongswan threads = 24, to process strongswan
      #    up/down.
      # Default value for threads is 16
      # Changing the max_concurrency = 16, to queue multiple strongswan requests.
      # Default value for max_concurrency is 4.
      # Setting inactivity_close_ike closes lingering IKE_SAs if the only
      #    CHILD_SA is closed due to inactivity. IKE_SAs linger if the
      #    ike-policy config matches but the sa-policy config doesn't.
      config = """
# config file came from old SuperServerPlugin code
charon {
        load_modular = yes
        install_routes = no
        max_concurrent = 64
        retransmit_tries = 2
        retransmit_timeout = 8.0
        make_before_break = yes
        delete_rekeyed_delay = 10
        threads = 24
        inactivity_close_ike = yes
        close_ike_on_child_failure = yes
        filelog {
                /var/log/charon.log {
                     time_format = %%b %%e %%T
                     append = yes
                     default = %d
                     knl = 2
                     ike = 2
                     flush_line = yes
                     time_add_ms = yes
                 }
                 stdout {
                     default = 0
                     ike_name = yes
                 }
        }
        syslog {
               # enable logging to LOG_DAEMON, use defaults
               daemon {
                    default = -1
               }
               # minimalistic IKE auditing logging to LOG_AUTHPRIV
               auth {
                    default = -1
               }
        }
        plugins {
                include strongswan.d/charon/*.conf
                stroke {
                  max_concurrent = 16
                }
                kernel-netlink {
                    fwmark = !0x42
                    roam_events = no
                    kernel_forward = %s
                    parallel_xfrm = %s
                    timeout = %s
                }
                socket-default {
                    fwmark = 0x42
                }
                kernel-libipsec {
                    allow_peer_ts = yes
                }
        }
        spi_min = %s
        spi_max = %s
}

include strongswan.d/*.conf
      """ % ( logLevel, kernel_forward, parallel_xfrm, timeout, spiMin, spiMax )
      return config

   def getCharonPid( self ):
      return getPid( self.sswanPidFile )

   def getStarterPid( self ):
      return getPid( self.starterPidFile )

   def isRunning( self ):
      return pidExists( self.getStarterPid() ) and pidExists( self.getCharonPid() )

   def startService( self ):
      if self.isRunning():
         bt9( 'StrongSwan already Running, do nothing' )
         return
      # Start strongswan
      cmd = '/usr/libexec/strongswan/starter --daemon charon'
      bt9( f'Starting Strongswan by invoking: {cmd}' )
      os.system( cmd )
      # Add a second wait for the service to start.
      time.sleep( 1 )
      self.maybeUpdateSswanRunning()

   def stopService( self ):
      if not self.isRunning():
         bt9( 'StrongSwan already Stopped, do nothing' )
         return
      bt9( 'Stopping Strongswan by killing it' )
      pid = self.getCharonPid()
      # The strongswan stop script tries to bring down StrongSwan charon and
      # starter processes by repeatedly sending signal (SIGTERM/SIGINT) in a
      # loop. This might delay the service termination if charon/starter process
      # doesn't respond to the signals.
      # Hence directly kill StrongSwan insted of invoking the script
      if pid:
         killProc( pid, 'stopService charon' )
         # Clean up Charon files.
         if os.path.exists( self.sswanPidFile ):
            os.remove( self.sswanPidFile )
         if os.path.exists( self.charonCtlFile ):
            os.remove( self.charonCtlFile )
      starterPid = self.getStarterPid()
      if starterPid:
         killProc( starterPid, 'stopService starter' )
         # Clean up Starter files
         if os.path.exists( self.starterPidFile ):
            os.remove( self.starterPidFile )
      # Update running status
      self.maybeUpdateSswanRunning()

   def restartService( self ):
      bt9( 'restartService() called' )
      self.stopService()
      self.startService()

   # the Ipsec agent increments this when it restarts, we track it so
   # that we can force a reload of strongswan to make sure it syncs
   # properly with the unix domain sockets in the new instance of
   # Ipsec
   @Tac.handler( 'agentGeneration' )
   def handleAgentGeneration( self ):
      # Stop StrongSwan
      bt9( 'Stopping StrongSwan' )
      self.stopService()
      # Issue sync to handle immediate changes in enableSswan.
      bt9( 'Issuing sync to handle immediate start/stop trigger' )
      self.sync()

def Plugin( ctx ):
   bt9( 'Registering SswanManager SuperServer service' )
   ctx.registerService( NewSswanManager( ctx.entityManager ) )
