#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pkgdeps: rpm coturn

import os
import struct
import google.protobuf as proto

import BothTrace
import Logging
import psutil
import re
import stun_pb2
import Server
import SuperServer
import Tac
import Tracing
import weakref
import stat
import time

import Toggles.StunToggleLib as stunToggles
from Arnet import IntfId, IpGenAddr
from ipaddress import IPv4Address, IPv6Address
from socket import ntohl, socket, AF_UNIX, SOCK_DGRAM, SOL_SOCKET, SO_REUSEADDR
from TypeFuture import TacLazyType
from IpLibConsts import DEFAULT_VRF

SslProfileState = TacLazyType( "Mgmt::Security::Ssl::ProfileState" )
StunTransportType = TacLazyType( "Stun::TransportType" )
StunTid = TacLazyType( "Stun::TransactionId" )
StunIpAndPort = TacLazyType( "Stun::StunIpAndPort" )
TurnserverTracingLevel = TacLazyType( "Stun::TurnserverTracingLevel" )

__defaultTraceHandle__ = Tracing.Handle( "StunServer" )
# SuperServer plugins use level 0 and level 1 quicktraces only
btf0 = BothTrace.tracef0        # error/critical messages
btf1 = BothTrace.tracef1        # All other messages
bvar = BothTrace.Var

STUN_INTF_IP_ADDRESS_CHANGED = Logging.LogHandle(
              "STUN_INTF_IP_ADDRESS_CHANGED",
              severity=Logging.logWarning,
              fmt="STUN server is listening on %d IP address(es)%s.",
              explanation="Configured IP address for a STUN interface has changed.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

# STUN server does not support listening on interfaces in non-default VRFs.
STUN_SERVER_INTF_IGNORED = Logging.LogHandle(
              "STUN_SERVER_INTF_IGNORED",
              severity=Logging.logWarning,
              fmt="Interface %s is not a routed interface in the default VRF.",
              explanation="STUN server does not support listening on "
              "interfaces in non-default VRFs.",
              recommendedAction="Remove the local-interface config or change the"
              " VRF to default for the interface." )

class TurnServerConf:
   confFile = '/etc/turnserver/turnserver.conf'

   def __init__( self, config, status ):
      btf1()
      self.config = config
      self.status = status
      self.logConfig = '\n'.join( [ 'log-file=/var/log/turnserver.log',
                                    'simple-log' ] )
      self.pidFilePath = "/var/run/turnserver/turnserver.pid"
      # Initialize unconfigurable configuration for turnserver
      self.pidfileConfig = 'pidfile=' + self.pidFilePath
      self.staleNonceConfig = 'stale-nonce=0'
      self.realm = 'aristanetworks.com'
      self.realmConfig = 'realm=' + self.realm
      self.cliConfig = 'no-cli'
      # deny ip config is needed to fix the security issue CVE-2020-26262 (BUG617681)
      self.denyIpConfig = "\n".join( [ "denied-peer-ip=0.0.0.0",
                                       "denied-peer-ip=::1",
                                       "denied-peer-ip=::" ] )

   def conf( self ):
      """ Returns the content to write to STUN server's turnserver.conf file. """
      btf1()
      config = [ self.logConfig,
                 'no-software-attribute',
                 'stun-only',
                 'no-stdout-log',
                 self.denyIpConfig,
                 self.cliConfig,
                 self.pidfileConfig,
                 self.staleNonceConfig,
                 self.realmConfig,
                 self.portConfig(),
                ]
      config += self.tracingConfig()
      config += self.ipAddrConfig()
      config += self.authConfig()
      config += self.sslConfig()
      config += self.transportConfig()
      config += self.udpdtlsConfig()
      config += self.sslConnectionLifetimeConfig()
      config += self.ipsecConnIdVersion()
      config += self.stunEndPointDependentToggleConfig()
      config += self.stunCapabilityTunnelingConfig()
      return "\n".join( config )

   def stunEndPointDependentToggleConfig( self ):
      if stunToggles.toggleStunEndpointDependentNatEnabled():
         return [ "stun-ed-nat-toggle=ON" ]
      else:
         return [ "stun-ed-nat-toggle=OFF" ]

   def stunCapabilityTunnelingConfig( self ):
      if stunToggles.toggleStunEndpointDependentNatEnabled():
         return [ "stun-capability-tunneling=ON" ]
      else:
         return [ "stun-capability-tunneling=OFF" ]

   def sslConnectionLifetimeConfig( self ):
      """Returns [max-allocate-lifetime=<seconds>]"""
      lifetime = int( self.status.sslConnectionLifetime.lifetime )
      return [ f"max-allocate-lifetime={lifetime}" ]

   def tracingConfig( self ):
      """ Returns [verbose], [Verbose] or an empty list"""
      if self.status.turnTracing == TurnserverTracingLevel.info:
         return [ "verbose" ]
      elif self.status.turnTracing == TurnserverTracingLevel.verbose:
         return [ "Verbose" ]
      else:
         return []

   def ipsecConnIdVersion( self ):
      """ Returns a string of the form 'ipsec-connectionid-version=<version>' """
      if self.status.ipsecConnIdVersion:
         return [ f"ipsec-connectionid-version={self.status.ipsecConnIdVersion}" ]
      else:
         return []

   def portConfig( self ):
      """ Returns a string of the form 'listening-port=<port>'. """
      return f"listening-port={self.status.port}"

   def ipAddrConfig( self ):
      """ Returns a list of strings of the form 'listening-ip=<value>'
      if IP addresses are configured, otherwise an empty list.
      """
      ipAddrConfig = [ f"listening-ip={ip}"
                       for ip in sorted( self.status.ipAddressToIntf ) ]
      return ipAddrConfig

   def getTurnAdminAuthKey( self, secretId, password ):
      cmd = "turnadmin -k -u " + secretId + " -r " + self.realm + " -p " + password
      cmd = cmd.split()
      output = ""
      try:
         output = Tac.run( cmd, stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
      except Exception as e: # pylint: disable-msg=broad-except
         btf0( "turnadmin failed to run" )
         btf0( "Exception", bvar( e ) )
         return None
      m = re.search( r"(?P<key>0x[0-9a-zA-z]+)", output )
      if m:
         btf1( "Generated key for id", bvar( secretId ) )
         return m.group( "key" )
      else:
         btf0( "Key could not be generated for id", bvar( secretId ) )
         return None

   def authConfig( self ):
      """ Returns a list of strings of the form 'user=<secret-id>:<secret>'. """
      authConfig = []
      for secret in self.status.password.values():
         key = self.getTurnAdminAuthKey( secret.id, secret.secret.getClearText() )
         if key:
            authConfig.append( f"user={secret.id}:{key}" )
      if authConfig:
         authConfig.append( "secure-stun" )
         authConfig.append( "lt-cred-mech" )
      return sorted( authConfig )

   def sslConfig( self ):
      """ Returns a list with a single string of the form 'cert=<path>'. """
      sslConfig = [ 'no-tls' ]
      if self.status.sslCertKeyPath != self.status.sslCertKeyPathDefault:
         sslConfig += [ f"cert={self.status.sslCertKeyPath}",
                        f"pkey={self.status.sslCertKeyPath}",
                        f"tls-listening-port={self.status.port}",
                       ]
         if ( self.status.sslTrustedCertsPath !=
              self.status.sslTrustedCertsPathDefault ):
            sslConfig += [ f"CA-file={self.status.sslTrustedCertsPath}" ]
         if self.status.crlsPath != self.status.crlsPathDefault:
            sslConfig += [ f"CRL-file={self.status.crlsPath}" ]
      else:
         sslConfig += [ 'no-dtls' ]
      return sslConfig

   def transportConfig( self ):  # pylint: disable-msg=inconsistent-return-statements
      """ Returns a list with a single string of either 'no-tcp' or 'no-udp'. """
      if self.status.transportType == StunTransportType.udp:
         return [ "no-tcp" ]
      elif self.status.transportType == StunTransportType.tcp:
         return [ "no-udp" ]
      else:
         btf0( "Unsupported transport type" )
         assert False, "Unsupported transport type"

   def pidFile( self ):
      return self.pidFilePath

   def udpdtlsConfig( self ):
      if self.status.udpdtls:
         return [ "udpdtls" ]
      return []

class StunServerResetSignal:
   sigusr1 = 1
   sighup = 2
   kill = 3

class StunCommon:
   """ StunCommon holds the common state and implements methods shared across
   different reactors.
   """
   def __init__( self, stunConfig, stunStatus, ipStatus, sharedSecretStatus,
                 sslStatus, stunUserStatus, service ):
      btf1()
      self.stunConfig = stunConfig
      self.stunStatus = stunStatus
      self.ipStatus = ipStatus
      self.sharedSecretStatus = sharedSecretStatus
      self.sslStatus = sslStatus
      self.stunUserStatus = stunUserStatus
      self.service = service
      self.ipIntfStatusNotifiees = {}
      self.stunCurrentSecretReactors = {}
      self.sslProfileStatusReactors = {}
      self.bindingResponseTimeoutHandlers = {}
      # If false, config check for the service will be scheduled after
      # serviceRestartDelay_. If true, config check will be performed immediately by
      # directly calling _maybeRestartService
      self.maybeRestartImmediate = False
      # This attribute can be used to enable or prevent the deletion of STUN bindings
      # when the server is disabled. This is for the config changes which we want to
      # be non-disruptive. If the bindings are deleted, then all the DPS paths get
      # affected.
      self.deleteBindingsOnDisableDefault = True
      self.deleteBindingsOnDisable = self.deleteBindingsOnDisableDefault
      # By default, turnserver will be sent SIGUSR1 to re-read the config file on any
      # config change.
      self.resetSignal = StunServerResetSignal.sigusr1

   def setResetSignal( self, signal=StunServerResetSignal.sigusr1 ):
      btf1( "Change resetSignal from", bvar( self.resetSignal ), "to",
            bvar( signal ) )
      if signal < self.resetSignal:
         # We do not want to set a signal to a lower value because if config change
         # already wants to restart the turnserver, we should not override it to send
         # a SIGHUP or SIGUSR1 signal.
         btf1( "Higher signal is already set. Cannot change it to a lower value" )
         return
      self.resetSignal = signal

   def handleConfigChange( self, signal=StunServerResetSignal.kill ):
      """
      Updates the disabled status and calls sync method of the service.
      """
      btf1( "signal:", bvar( signal ) )
      self.updateDisabledStatus()
      self.setResetSignal( signal )
      btf1( "maybeRestartImmediate:", bvar( self.maybeRestartImmediate ) )
      if self.maybeRestartImmediate:
         self.service._maybeRestartService()  # pylint: disable=protected-access
      else:
         self.service.sync()

      # Reset maybeRestartImmediate
      self.maybeRestartImmediate = False

   def clearBindingResponses( self ):
      btf1()
      for tid in self.stunStatus.bindingResponses:
         self.delBindingResponseForTid( tid )

   def isSslProfileValid( self ):
      btf1( bvar( self.stunStatus.sslProfile ) )
      if self.stunStatus.sslProfile == self.stunStatus.sslProfileDefault:
         return True

      if self.stunStatus.sslProfile not in self.sslStatus.profileStatus:
         return False

      state = self.sslStatus.profileStatus[ self.stunStatus.sslProfile ].state
      if state != SslProfileState.valid:
         return False

      return True

   def updateDisabledStatus( self ):
      """ Updates disabled status after checking the states
      in StunConfig and StunStatus.
      """
      btf1( "deleteBindingsOnDisable:", self.deleteBindingsOnDisable )
      oldDisabled = self.stunStatus.disabled
      if ( not self.stunConfig.disabled
           and self.stunStatus.ipAddressToIntf
           and self.stunStatus.port
           and self.isSslProfileValid() ):
         self.stunStatus.disabled = False
      else:
         if not self.stunStatus.disabled:
            self.maybeRestartImmediate = True  # handle this config change inline
            if self.deleteBindingsOnDisable:
               # Perform cleanup as turnserver is disabled
               self.clearBindingResponses()
         self.stunStatus.disabled = True

      btf1( "disabled: old=", bvar( oldDisabled ), "new=",
            bvar( self.stunStatus.disabled ) )
      # reset deleteBindingsOnDisable
      self.deleteBindingsOnDisable = self.deleteBindingsOnDisableDefault

   def createIpIntfStatusNotifiee( self, intfName ):
      """ Creates a new ipIntfStatusNotifiee reactor if it doesn't exist. """
      btf1( bvar( intfName ) )
      if intfName not in self.ipIntfStatusNotifiees:
         btf1( "Add reactor for", bvar( intfName ) )
         self.ipIntfStatusNotifiees[ intfName ] = IpIntfStatusNotifiee( self,
                                                                        intfName )

   def deleteIpIntfStatusNotifiee( self, intfName ):
      """ Deletes the ipIntfStatusNotifiee reactor if it exists. """
      btf1( bvar( intfName ) )
      if intfName in self.ipIntfStatusNotifiees:
         self.ipIntfStatusNotifiees[ intfName ].close()
         btf1( "Delete reactor for", bvar( intfName ) )
         del self.ipIntfStatusNotifiees[ intfName ]

   def handleIntf( self, intfName ):
      """ handleIntf is called from StunConfigNotifiee and StunIpStatusNotifiee
      whenever interface is added either to localIntfIds in config or ipIntfStatus in
      default VRF.
      """
      intfId = IntfId( intfName )
      btf1( bvar( intfName ) )

      if ( self.intfInDefaultVrf( intfName ) and
           intfId in self.stunConfig.localIntfIds ):
         # Create a reactor for ipIntfStatus only if it exists in localIntfIds
         # and ipIntfStatus in default VRF
         self.createIpIntfStatusNotifiee( intfName )
      else:
         if ( intfId in self.stunConfig.localIntfIds and
               not self.intfInDefaultVrf( intfName ) ):
            Logging.log( STUN_SERVER_INTF_IGNORED, intfName )

         # Cleanup reactor if it already exists in our mapping
         self.deleteIpIntfStatusNotifiee( intfName )

   def intfInDefaultVrf( self, intfName ):
      """ Returns true if an interface is in default vrf """

      defaultVrfIpIntfStatus = self.ipStatus.vrfIpIntfStatus.get( DEFAULT_VRF )
      return ( defaultVrfIpIntfStatus and
              intfName in defaultVrfIpIntfStatus.ipIntfStatus )

   def maybeAddStunCurrentSecretReactor( self, profileName ):
      btf1( bvar( profileName ) )
      if profileName not in self.stunCurrentSecretReactors:
         btf1( "Add reactor for the Shared-Secret-Profile:", bvar( profileName ) )
         reactor = StunCurrentSecretReactor( self, profileName )
         self.stunCurrentSecretReactors[ profileName ] = reactor

   def maybeRemoveStunCurrentSecretReactor( self, profileName ):
      btf1( bvar( profileName ) )
      if profileName in self.stunCurrentSecretReactors:
         btf1( "Remove reactor for the Shared-Secret-Profile:", bvar( profileName ) )
         del self.stunCurrentSecretReactors[ profileName ]

   def handleSharedSecretProfileCommon( self, profileName ):
      btf1( bvar( profileName ) )

      # Take care of housekeeping for existing profile
      existingProfile = self.stunStatus.passwordProfile
      if ( existingProfile != self.stunConfig.passwordProfile or
           profileName not in self.sharedSecretStatus.currentSecret ):
         self.maybeRemoveStunCurrentSecretReactor( existingProfile )

      # Handle configured profile
      if ( profileName == self.stunConfig.passwordProfile and
           profileName in self.sharedSecretStatus.currentSecret ):
         self.maybeAddStunCurrentSecretReactor( profileName )

      self.stunStatus.passwordProfile = profileName
      self.handleConfigChange()

   def maybeAddSslProfileStatusReactor( self, profileName ):
      btf1( bvar( profileName ) )
      if profileName not in self.sslProfileStatusReactors:
         btf1( "Add reactor for the ssl profile", bvar( profileName ) )
         reactor = SslProfileStatusReactor( self, profileName )
         self.sslProfileStatusReactors[ profileName ] = reactor

   def maybeRemoveSslProfileStatusReactor( self, profileName ):
      btf1( bvar( profileName ) )
      if profileName in self.sslProfileStatusReactors:
         btf1( "Remove reactor for ssl profile", bvar( profileName ) )
         del self.sslProfileStatusReactors[ profileName ]

   def handleSslProfileStatusCommon( self, profileName ):
      btf1( bvar( profileName ) )

      # Remove reactor for existing ssl profile if needed
      existingProfile = self.stunStatus.sslProfile
      btf1( "Existing ssl profile is", bvar( existingProfile ) )
      if ( existingProfile != self.stunConfig.sslProfile or
           profileName not in self.sslStatus.profileStatus ):
         self.maybeRemoveSslProfileStatusReactor( existingProfile )

      # Handle Configured ssl profile
      if ( profileName == self.stunConfig.sslProfile and
           profileName in self.sslStatus.profileStatus ):
         self.maybeAddSslProfileStatusReactor( profileName )

      self.stunStatus.sslProfile = profileName
      # Restart is enforced when ssl profile is configured/removed
      # This is needed so that udp and dtls connection data structures do not
      # interfere.
      self.handleConfigChange( signal=StunServerResetSignal.kill )

   def addBindingResponse( self, stunMessage ):
      tid = StunTid( stunMessage.tid )
      bindingResponse = Tac.Value( "Stun::StunBindingResponse", tid )
      bindingResponse.creationTime = Tac.now()
      self.stunStatus.lastRcvdMsgTime = bindingResponse.creationTime
      ipAddr = None
      # Process public IP Address
      if stunMessage.ipAddr.HasField( 'ipv4Addr' ):
         ipAddr = format( IPv4Address( ntohl( stunMessage.ipAddr.ipv4Addr ) ) )
      elif stunMessage.ipAddr.HasField( 'ipv6Addr' ):
         ipAddr = format( IPv6Address( stunMessage.ipAddr.ipv6Addr ) )
      if ipAddr:
         bindingResponse.translatedAddress = IpGenAddr( ipAddr )

      bindingResponse.translatedPort = stunMessage.port
      for attribute in stunMessage.stunAttribute:
         tlv = Tac.Value( "Stun::Tlv", attribute.type,
                                       attribute.length, attribute.data )
         bindingResponse.tlv.add( tlv )

      # Remove mapping entries inconsistent with the new information
      newSip = StunIpAndPort( bindingResponse.translatedAddress, stunMessage.port )
      if tid in self.stunStatus.bindingResponses:
         oldIp = self.stunStatus.bindingResponses[ tid ].translatedAddress
         oldPort = self.stunStatus.bindingResponses[ tid ].translatedPort
         oldSipForNewTid = StunIpAndPort( oldIp, oldPort )
         if oldSipForNewTid != newSip:  # if the translation has changed
            btf1( "Remove Older publicIpPortToTid",
                  bvar( oldSipForNewTid.stringValue() ),
                  "->", bvar( tid.hexString() ), "mapping" )
            del self.stunStatus.publicIpPortToTid[ oldSipForNewTid ]
      if ( newSip in self.stunStatus.publicIpPortToTid and
           tid != self.stunStatus.publicIpPortToTid[ newSip ] ): # duplicate detected
         oldTidForNewSip = self.stunStatus.publicIpPortToTid[ newSip ]
         btf1( "Remove Duplicate tid", bvar( oldTidForNewSip.hexString() ),
               "for the new public ip and port", bvar( newSip.stringValue() ) )
         self.delBindingResponseForTid( oldTidForNewSip )

      self.stunStatus.bindingResponses.addMember( bindingResponse )
      self.stunStatus.publicIpPortToTid[ newSip ] = tid
      btf1( "Added binding response with tid:", bvar( tid.hexString() ) )

   def delBindingResponseForTid( self, tid ):
      if tid not in self.stunStatus.bindingResponses:
         btf1( "Binding for tid", bvar( tid.hexString() ),
               "not present for deletion" )
         return

      # Also delete the reverse mapping
      ip = self.stunStatus.bindingResponses[ tid ].translatedAddress
      port = self.stunStatus.bindingResponses[ tid ].translatedPort
      stunIpAndPort = StunIpAndPort( ip, port )
      btf1( "Remove", bvar( tid.hexString() ), "<-->",
            bvar( stunIpAndPort.stringValue() ), "mapping" )
      del self.stunStatus.publicIpPortToTid[ stunIpAndPort ]
      del self.stunStatus.bindingResponses[ tid ]

   def getTidForStunMsg( self, stunMsgIpAddr, stunMsgPort ):
      btf1()
      if not stunMsgIpAddr or not stunMsgPort:
         return None

      ipAddr = None
      if stunMsgIpAddr.HasField( 'ipv4Addr' ):
         ipAddr = format( IPv4Address( ntohl( stunMsgIpAddr.ipv4Addr ) ) )
      elif stunMsgIpAddr.HasField( 'ipv6Addr' ):
         ipAddr = format( IPv6Address( stunMsgIpAddr.ipv6Addr ) )
      if ipAddr:
         translatedAddress = IpGenAddr( ipAddr )

      stunIpAndPort = StunIpAndPort( translatedAddress, stunMsgPort )
      if stunIpAndPort not in self.stunStatus.publicIpPortToTid:
         return None

      return self.stunStatus.publicIpPortToTid[ stunIpAndPort ]

   def delBindingResponse( self, stunMessage ):
      btf1()
      self.stunStatus.lastRcvdMsgTime = Tac.now()

      if not stunMessage.tid:
         tid = self.getTidForStunMsg( stunMessage.ipAddr, stunMessage.port )
      else:
         tid = StunTid( stunMessage.tid )

      if tid:
         self.delBindingResponseForTid( tid )

   def handleBindingResponse( self, stunMessage ):
      """ handleBindingResponse processes a stunMessage and adds/deletes
      the binding request to STUN Status.
      """
      if self.stunStatus.disabled:
         btf1( "Ignore StunMessage, stun server is disabled" )
         return
      btf1()
      # Disable pylint error when accessing StunMessage.Type
      # pylint: disable=E1101
      if stunMessage.type == stun_pb2.StunMessage.Type.BIND:
      # pylint: enable=E1101
         self.addBindingResponse( stunMessage )
      # pylint: disable=E1101
      elif stunMessage.type == stun_pb2.StunMessage.Type.DELETE:
      # pylint: enable=E1101
         self.delBindingResponse( stunMessage )

   def handleClientSessionClose( self ):
      self.service.restart()

   def syslogListeningIpChange( self ):
      numAddrs = len( self.stunStatus.ipAddressToIntf )
      addrs = ""
      if numAddrs:
         addrs = " " + ", ".join(
            str( addr ) for addr in sorted( self.stunStatus.ipAddressToIntf ) )
      Logging.log(
         STUN_INTF_IP_ADDRESS_CHANGED, numAddrs, addrs )

class BindingResponseTimeoutHandler:
   """ Reactor to periodically delete a BindingResponse. """
   def __init__( self, stunCommon, tid ):
      btf1()
      self.stunCommon = stunCommon
      self.tid = tid
      tf = self.timeInFuture()
      self.stunCommon.stunStatus.nextRespTimeout[ self.tid ] = tf
      self.handler = Tac.ClockNotifiee(
         self.handleBindingResponseTimer, tf )
      self.currentSkipCount = 0

   def __del__( self ):
      if self.tid in self.stunCommon.stunStatus.nextRespTimeout:
         del self.stunCommon.stunStatus.nextRespTimeout[ self.tid ]

   def timeInFuture( self ):
      return Tac.now() + self.stunCommon.stunConfig.bindingResponseTimeout

   def _rearmTimer( self ):
      btf1( bvar( self.stunCommon.stunConfig.bindingResponseTimeout ) )
      tf = self.timeInFuture()
      self.handler._setTimeMin( tf ) # pylint: disable-msg=protected-access
      self.stunCommon.stunStatus.nextRespTimeout[ self.tid ] = tf

   def rearmTimer( self ):
      self.currentSkipCount = 0
      self._rearmTimer()

   def isTidInUse( self ):
      for user in self.stunCommon.stunUserStatus:
         if self.tid in self.stunCommon.stunUserStatus[ user ].tidsInUse:
            btf1( "tid", bvar( self.tid.hexString() ), "is being used by",
                  bvar( user ) )
            return True

      return False

   def handleBindingResponseTimer( self ):
      self.currentSkipCount += 1
      btf1( "Current skip count", bvar( self.currentSkipCount ) )
      # If this tid is not being tracked by DPS or any other user, we should delete
      # it.
      if ( self.currentSkipCount <= self.stunCommon.stunConfig.maxRefreshSkips and
         self.isTidInUse() ):
         btf1( "Rearm BindingResponse timer because tid",
               bvar( self.tid.hexString() ), "is currently in use" )
         self._rearmTimer()
      else:
         btf1( "Remove BindingResponse for tid", bvar( self.tid.hexString() ),
               "because it wasn't refreshed" )
         del self.stunCommon.stunStatus.bindingResponses[ self.tid ]

class IpIntfStatusNotifiee( Tac.Notifiee ):
   """ Reactor to Ip::IpIntfStatus. """
   notifierTypeName = "Ip::IpIntfStatus"
   def __init__( self, stunCommon, intfName ):
      btf1( bvar( intfName ) )
      self.stunCommon = stunCommon
      self.stunStatus = stunCommon.stunStatus
      self.ipStatus = stunCommon.ipStatus

      notifier = self.ipStatus.ipIntfStatus[ intfName ]
      Tac.Notifiee.__init__( self, notifier )
      self.address = IpGenAddr( notifier.activeAddrWithMask.address )
      self.handleActiveAddrWithMask()

   def deleteAddrFromStatus( self ):
      """ Deletes the IP address and its intf mapping from StunStatus. """
      btf1()
      if ( self.address in self.stunStatus.ipAddressToIntf and
           self.stunStatus.ipAddressToIntf[ self.address ] ==
           self.notifier_.intfId ):
         btf1( "IP address", bvar( self.address ), "Intf",
               bvar( self.notifier_.intfId ) )
         del self.stunStatus.ipAddressToIntf[ self.address ]

   def close( self ):
      btf1()
      oldIntfs = list( self.stunStatus.ipAddressToIntf.items() )
      self.deleteAddrFromStatus()
      Tac.Notifiee.close( self )
      newIntfs = self.stunStatus.ipAddressToIntf.items()
      if oldIntfs != newIntfs:
         self.stunCommon.syslogListeningIpChange()
         self.stunCommon.handleConfigChange( signal=StunServerResetSignal.sighup )

   @Tac.handler( 'activeAddrWithMask' )
   def handleActiveAddrWithMask( self ):
      """ Handles changes in IP address of an interface. """
      intfName = self.notifier_.intfId
      btf1( bvar( intfName ) )
      oldIntfs = list( self.stunStatus.ipAddressToIntf.items() )
      if intfName in self.ipStatus.ipIntfStatus:
         # Delete old address mapping.
         self.deleteAddrFromStatus()
         self.address = IpGenAddr( self.notifier_.activeAddrWithMask.address )
         # Verify that ip address is valid.
         if self.address.stringValue != Tac.Value( "Arnet::IpAddr" ).ipAddrZero:
            btf1( "Add IP address", bvar( self.address ), "intfName",
                  bvar( intfName ) )
            self.stunStatus.ipAddressToIntf[ self.address ] = IntfId( intfName )
      else:
         # Condition where ipAddress has been removed from ipIntfStatus.
         btf1( "Delete IP address", bvar( self.address ), "intfName",
               bvar( intfName ) )
         self.deleteAddrFromStatus()

      newIntfs = self.stunStatus.ipAddressToIntf.items()
      if oldIntfs != newIntfs:
         self.stunCommon.syslogListeningIpChange()
         self.stunCommon.handleConfigChange( signal=StunServerResetSignal.sighup )

class IpStatusNotifiee( Tac.Notifiee ):
   """ Reactor to Ip::Status. """
   notifierTypeName = "Ip::Status"
   def __init__( self, stunCommon ):
      btf1()
      self.stunCommon = stunCommon
      self.defaultVrfNotifiee = None
      ipStatus = stunCommon.ipStatus
      Tac.Notifiee.__init__( self, ipStatus )

      self.handleVrfIpIntfStatus( DEFAULT_VRF )

   @Tac.handler( 'vrfIpIntfStatus' )
   def handleVrfIpIntfStatus( self, vrfName ):
      """ Handles changes in vrfIpIntfStatus collection. """
      btf1()

      ipStatus = self.stunCommon.ipStatus
      if vrfName == DEFAULT_VRF:
         if vrfName in ipStatus.vrfIpIntfStatus:
            if not self.defaultVrfNotifiee:
               btf1( "Create DefaultVrfNotifiee" )
               self.defaultVrfNotifiee = DefaultVrfStatusNotifiee( self.stunCommon,
                                                ipStatus.vrfIpIntfStatus[ vrfName ] )
         else:
            if self.defaultVrfNotifiee is not None:
               btf1( "Delete DefaultVrfNotifiee" )
               self.defaultVrfNotifiee = None

class DefaultVrfStatusNotifiee( Tac.Notifiee ):
   """ Reactor to default Ip::VrfIpIntfStatus. """
   notifierTypeName = "Ip::VrfIpIntfStatus"

   def __init__( self, stunCommon, defaultVrfIpIntfStatus ):
      btf1()
      self.stunCommon = stunCommon
      Tac.Notifiee.__init__( self, defaultVrfIpIntfStatus )

      for intfName in defaultVrfIpIntfStatus.ipIntfStatus:
         self.handleIpIntfStatus( intfName )

   def close( self ):
      """ Makes sure there are no interfaces in stun status.
      Removes if there are any. """

      btf1()

      intfIds = self.stunCommon.stunStatus.ipAddressToIntf.values()
      btf1( "Number of entries in stun status: ", bvar( len( intfIds ) ) )
      for intfId in intfIds:
         btf1( "Removing intf: ", bvar( intfId ), " from status" )
         self.stunCommon.handleIntf( intfId )

      return super().close()

   @Tac.handler( 'ipIntfStatus' )
   def handleIpIntfStatus( self, intfName ):
      """ Handles changes in ipIntfStatus. """
      btf1()
      self.stunCommon.handleIntf( intfName )

class StunConfigNotifiee( Tac.Notifiee ):
   """ Reactor to Stun::ServerConfig. """
   notifierTypeName = "Stun::ServerConfig"

   def __init__( self, stunCommon ):
      btf1()
      self.stunCommon = stunCommon
      self.stunConfig = stunCommon.stunConfig
      self.stunStatus = stunCommon.stunStatus
      self.ipStatus = stunCommon.ipStatus
      Tac.Notifiee.__init__( self, self.stunConfig )

      self.handlePasswordProfile()
      self.handlePort()
      self.handleLocalIntfs()
      self.handleSslProfile()
      self.handleTurnTracing()
      self.handleTransportType()
      self.handleUdpdtls()
      self.handleSslConnectionLifetime()
      self.handleBindingResponseTimeout()
      self.handleDisabled()

   @Tac.handler( 'disabled' )
   def handleDisabled( self ):
      btf1( "disabled", bvar( self.notifier_.disabled ) )
      self.stunCommon.handleConfigChange()

   def handleLocalIntfs( self ):
      btf1()
      for intf in self.stunConfig.localIntfIds:
         self.handleLocalIntf( intf )
      # If SuperServer has restarted and there are no interfaces configured,
      # call sync again to reflect changes in turnserver.conf. This
      # call should be a no-op if there are no config changes since
      # restart.
      self.stunCommon.handleConfigChange( signal=StunServerResetSignal.sighup )

   @Tac.handler( 'localIntfIds' )
   def handleLocalIntf( self, intfName ):
      btf1( bvar( intfName ) )
      self.stunCommon.handleIntf( intfName )

   @Tac.handler( 'port' )
   def handlePort( self ):
      port = self.notifier_.port
      btf1( "set port to", bvar( port ) )
      self.stunStatus.port = port
      self.stunCommon.handleConfigChange( signal=StunServerResetSignal.sighup )

   @Tac.handler( 'passwordProfile' )
   def handlePasswordProfile( self ):
      newProfile = self.notifier_.passwordProfile
      oldProfile = self.stunStatus.passwordProfile
      btf1( "Change passwordProfile from", bvar( oldProfile ), "to",
            bvar( newProfile ) )
      self.stunCommon.handleSharedSecretProfileCommon( newProfile )

   @Tac.handler( "sslProfile" )
   def handleSslProfile( self ):
      newProfile = self.notifier_.sslProfile
      oldProfile = self.stunStatus.sslProfile
      btf1( "Change sslProfile from", bvar( oldProfile ), "to", bvar( newProfile ) )
      self.stunCommon.handleSslProfileStatusCommon( newProfile )

   @Tac.handler( "turnTracing" )
   def handleTurnTracing( self ):
      old = self.stunStatus.turnTracing
      new = self.stunConfig.turnTracing
      btf1( "turnserver tracing changed from", bvar( old ), "to", bvar( new ) )
      self.stunStatus.turnTracing = new
      self.stunCommon.handleConfigChange( signal=StunServerResetSignal.sigusr1 )

   @Tac.handler( "transportType" )
   def handleTransportType( self ):
      old = self.stunStatus.transportType
      new = self.stunConfig.transportType
      btf1( "transportType from", bvar( old ), "to", bvar( new ) )
      self.stunStatus.transportType = new
      self.stunCommon.handleConfigChange()

   @Tac.handler( "udpdtls" )
   def handleUdpdtls( self ):
      old = self.stunStatus.udpdtls
      new = self.stunConfig.udpdtls
      btf1( "udpdtls changed from", bvar( old ), "to", bvar( new ) )
      self.stunStatus.udpdtls = self.stunConfig.udpdtls
      self.stunCommon.handleConfigChange()

   @Tac.handler( "sslConnectionLifetime" )
   def handleSslConnectionLifetime( self ):
      old = self.stunStatus.sslConnectionLifetime.lifetime
      new = self.stunConfig.sslConnectionLifetime.lifetime
      btf1( "ssl lifetime changed from", bvar( old ), "to", bvar( new ), "seconds" )
      self.stunStatus.sslConnectionLifetime = self.stunConfig.sslConnectionLifetime
      self.stunCommon.handleConfigChange()

   @Tac.handler( "bindingResponseTimeout" )
   def handleBindingResponseTimeout( self ):
      old = self.stunStatus.bindingResponseTimeout
      new = self.stunConfig.bindingResponseTimeout
      btf1( "bindingResponseTimeout changed from", bvar( old ), "to", bvar( new ) )
      self.stunStatus.bindingResponseTimeout = new
      for handler in self.stunCommon.bindingResponseTimeoutHandlers.values():
         handler.rearmTimer()

   @Tac.handler( "clearBr" )
   def handleClearBindingRequest( self ):
      btf1( "clearAll=", bvar( self.stunConfig.clearBr.clearAll ),
            ", publicIp=", bvar( self.stunConfig.clearBr.publicIp ),
            ", publicPort=", bvar( self.stunConfig.clearBr.publicPort ),
            ", tid=", bvar( self.stunConfig.clearBr.tid ) )

      if self.stunConfig.clearBr.clearAll:
         self.stunCommon.clearBindingResponses()
         return

      if self.stunConfig.clearBr.tid != StunTid():
         self.stunCommon.delBindingResponseForTid( self.stunConfig.clearBr.tid )
         return

      if self.stunConfig.clearBr.publicIp != IpGenAddr():
         for tid in self.stunStatus.bindingResponses:
            br = self.stunStatus.bindingResponses[ tid ]
            if ( br.translatedAddress.stringValue ==
                 self.stunConfig.clearBr.publicIp.stringValue ):
               # Check if public port option is configured, if yes, we need to match
               # on that too.
               if self.stunConfig.clearBr.publicPort != 0:
                  btf1( "public port case " )
                  if br.translatedPort == self.stunConfig.clearBr.publicPort:
                     btf1( "public port matched" )
                     self.stunCommon.delBindingResponseForTid( tid )
               else:
                  self.stunCommon.delBindingResponseForTid( tid )
         return

class StunStatusReactor( Tac.Notifiee ):
   """ Reactor to Stun::ServerStatus. """
   notifierTypeName = "Stun::ServerStatus"

   def __init__( self, stunCommon ):
      btf1()
      self.stunCommon = stunCommon
      self.stunStatus = stunCommon.stunStatus
      Tac.Notifiee.__init__( self, self.stunStatus )

      # Initialize the BindingResponseTimeoutHandlers
      tids = list( self.stunStatus.bindingResponses )
      tids.extend( self.stunCommon.bindingResponseTimeoutHandlers )
      for tid in tids:
         self.handleBindingResponses( tid )

   @Tac.handler( "bindingResponses" )
   def handleBindingResponses( self, tid ):
      if tid in self.stunStatus.bindingResponses:
         if tid in self.stunCommon.bindingResponseTimeoutHandlers:
            btf1( "Tid", bvar( tid.hexString() ), "refreshed" )
            self.stunCommon.bindingResponseTimeoutHandlers[ tid ].rearmTimer()
         else:
            btf1( "Add BindingResponseTimeoutHandler for tid",
                  bvar( tid.hexString() ) )
            self.stunCommon.bindingResponseTimeoutHandlers[
               tid ] = BindingResponseTimeoutHandler( self.stunCommon, tid )
      else:
         btf1( "Delete BindingResponseTimeoutHandler for tid",
               bvar( tid.hexString() ) )
         del self.stunCommon.bindingResponseTimeoutHandlers[ tid ]

class IpsecCapabilitiesStatusReactor( Tac.Notifiee ):
   """ Reactor to Ipsec::Capabilities::Status """
   notifierTypeName = "Ipsec::Capabilities::Status"

   def __init__( self, stunCommon, ipsecCapabilitiesStatus ):
      btf1()
      self.stunCommon = stunCommon
      self.stunStatus = stunCommon.stunStatus
      Tac.Notifiee.__init__( self, ipsecCapabilitiesStatus )
      self.handleIpsecConnIdVersion()

   @Tac.handler( 'ipsecConnIdVersion' )
   def handleIpsecConnIdVersion( self ):
      """ Handle changes in ipsecConnIdVersion """
      version = self.notifier_.ipsecConnIdVersion
      btf1( "set Ipsec version to", bvar( version ) )
      self.stunStatus.ipsecConnIdVersion = version
      self.stunCommon.handleConfigChange( signal=StunServerResetSignal.sigusr1 )

class StunReceiveSecretReactor( Tac.Notifiee ):
   notifierTypeName = "Mgmt::Security::SharedSecretProfile::SecretStatus"

   def __init__( self, stunCommon, name, secretId ):
      btf1()
      self.stunCommon = stunCommon
      self.secretId = secretId
      self.currentSecret = self.stunCommon.sharedSecretStatus.currentSecret[ name ]
      self.rxSecret = self.currentSecret.receiveSecretStatus[ secretId ]
      Tac.Notifiee.__init__( self, self.rxSecret )

      self.handleReceiveSecretAttribute()

   @Tac.handler( "secret" )
   def handleReceiveSecretAttribute( self ):
      btf1()
      stunPassword = self.stunCommon.stunStatus.password
      if self.rxSecret.secret.clearTextSize() > 0:
         btf0( "Configure new password for id:", bvar( self.secretId ) )
         password = stunPassword.newMember( self.secretId )
         password.update( self.rxSecret.secretOrdinal() )
      else:
         btf0( "Remove password for id:", bvar( self.secretId ) )
         del stunPassword[ self.secretId ]
      self.stunCommon.handleConfigChange()

class StunCurrentSecretReactor( Tac.Notifiee ):
   notifierTypeName = "Mgmt::Security::SharedSecretProfile::CurrentSecret"

   def __init__( self, stunCommon, name ):
      btf1()
      self.stunCommon = stunCommon
      self.name = name
      self.stunReceiveSecretReactors = {}
      self.currentSecret = self.stunCommon.sharedSecretStatus.currentSecret[ name ]
      Tac.Notifiee.__init__( self, self.currentSecret )

      for secretId in self.currentSecret.receiveSecretStatus:
         self.handleReceiveSecret( secretId )

   @Tac.handler( "receiveSecretStatus" )
   def handleReceiveSecret( self, secretId ):
      btf1()
      stunPassword = self.stunCommon.stunStatus.password
      rxSecret = self.currentSecret.receiveSecretStatus.get( secretId )
      if rxSecret:
         btf0( "Add attribute reactor for secret id:", bvar( secretId ) )
         notifierReactor = StunReceiveSecretReactor( self.stunCommon,
                                                     self.name,
                                                     secretId )
         self.stunReceiveSecretReactors[ secretId ] = notifierReactor
         return
      else:
         btf0( "Remove password for id:", bvar( secretId ) )
         del stunPassword[ secretId ]
         del self.stunReceiveSecretReactors[ secretId ]
      self.stunCommon.handleConfigChange()

   def close( self ):
      btf0( "Remove passwords for profile", bvar( self.name ) )
      self.stunCommon.stunStatus.password.clear()
      Tac.Notifiee.close( self )
      self.stunCommon.handleConfigChange()

class StunSharedSecretStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Mgmt::Security::SharedSecretProfile::Status"

   def __init__( self, stunCommon ):
      btf1()
      self.stunCommon = stunCommon
      self.stunConfig = stunCommon.stunConfig
      self.stunStatus = stunCommon.stunStatus
      self.sharedSecretStatus = stunCommon.sharedSecretStatus
      Tac.Notifiee.__init__( self, self.stunCommon.sharedSecretStatus )

      for name in self.sharedSecretStatus.currentSecret:
         self.handleCurrentSecret( name )

   @Tac.handler( "currentSecret" )
   def handleCurrentSecret( self, name ):
      btf1()
      if name != self.stunConfig.passwordProfile:
         return
      self.stunCommon.handleSharedSecretProfileCommon( name )

class SslProfileStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Mgmt::Security::Ssl::ProfileStatus"

   def __init__( self, stunCommon, profileName ):
      btf1()
      self.stunCommon = stunCommon
      self.profileName = profileName
      self.sslProfileStatus = self.stunCommon.sslStatus.profileStatus[ profileName ]
      Tac.Notifiee.__init__( self, self.sslProfileStatus )

      self.handleProfileStatusCommon()

   def handleProfileStatusCommon( self ):
      ps = self.sslProfileStatus
      btf1( "ssl profile", bvar( self.profileName ), "state:", bvar( ps.state ),
            "certKeyPath:", ps.certKeyPath, "trustedCertsPath:",
            ps.trustedCertsPath, "crlsPath:", ps.crlsPath )
      if self.sslProfileStatus.state == SslProfileState.valid:
         self.stunCommon.stunStatus.sslCertKeyPath = ps.certKeyPath
         self.stunCommon.stunStatus.sslTrustedCertsPath = ps.trustedCertsPath
         self.stunCommon.stunStatus.crlsPath = ps.crlsPath
      else:
         self.stunCommon.stunStatus.sslCertKeyPath = \
            self.stunCommon.stunStatus.sslCertKeyPathDefault
         self.stunCommon.stunStatus.sslTrustedCertsPath = \
            self.stunCommon.stunStatus.sslTrustedCertsPathDefault
         self.stunCommon.stunStatus.crlsPath = \
            self.stunCommon.stunStatus.sslTrustedCertsPathDefault

      # For SSL changes we do not need to clear the bindings immediately. SSL profile
      # transitions from valid->updating->valid state for all config changes. This
      # results in turnserver restart. We can keep the bindings across this restart
      # and let them get deleted after timeout if clients can no longer establish ssl
      # connection.
      self.stunCommon.deleteBindingsOnDisable = False
      self.stunCommon.handleConfigChange()

   @Tac.handler( "state" )
   def handleState( self ):
      # We need to restart turnserver in this case even if config hasn't
      # changed. This is needed because underlying certificates may have changed.
      self.stunCommon.service.forceRestart_ = True
      self.stunCommon.maybeRestartImmediate = True  # Handle this change inline
      self.handleProfileStatusCommon()

   @Tac.handler( "certKeyPath" )
   def handleCerts( self ):
      self.handleProfileStatusCommon()

   @Tac.handler( "trustedCertsPath" )
   def handleTrustedCertsPath( self ):
      self.handleProfileStatusCommon()

   @Tac.handler( "crlsPath" )
   def handleCrlsPath( self ):
      self.handleProfileStatusCommon()

   def cleanup( self ):
      btf1( "Cleanup for", bvar( self.profileName ) )
      self.stunCommon.stunStatus.sslCertKeyPath = \
         self.stunCommon.stunStatus.sslCertKeyPathDefault
      self.stunCommon.stunStatus.sslTrustedCertsPath = \
         self.stunCommon.stunStatus.sslTrustedCertsPathDefault
      self.stunCommon.stunStatus.crlsPath = \
         self.stunCommon.stunStatus.crlsPathDefault

   def close( self ):
      btf1()
      self.cleanup()
      Tac.Notifiee.close( self )
      self.stunCommon.handleConfigChange()

class SslStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Mgmt::Security::Ssl::Status"

   def __init__( self, stunCommon ):
      self.stunCommon = stunCommon
      self.sslStatus = stunCommon.sslStatus
      Tac.Notifiee.__init__( self, self.stunCommon.sslStatus )

      for name in self.sslStatus.profileStatus:
         self.handleSslProfileStatus( name )

   @Tac.handler( "profileStatus" )
   def handleSslProfileStatus( self, name ):
      btf1()
      if name != self.stunCommon.stunConfig.sslProfile:
         return
      self.stunCommon.handleSslProfileStatusCommon( name )

class StunServerService( SuperServer.SystemdService ):
   notifierTypeName = '*'

   def __init__( self, config, status, stunServerAgent, healthCheckNeeded=True ):
      btf1()
      self._serverConfig = config
      self._serverStatus = status
      self._agent = stunServerAgent
      self.turnServerConf = TurnServerConf( self._serverConfig, self._serverStatus )
      SuperServer.SystemdService.__init__( self, "turnserver", "turnserver", config,
                                           TurnServerConf.confFile,
                                           healthCheckNeeded=healthCheckNeeded )

   def serviceProcessWarm( self ):
      """ Returns whether or not the process associated with the service has
      fully restarted and is running based on the new configuration.
      NOTE - this can be very difficult to get right. Generally, one
      must override the startService, stopService, and restartService
      commands in order to save enough information that one can
      reliably determine whether or not the service has been completely
      restarted and is running based on its new configuration. """
      return os.path.isfile( self.turnServerConf.pidFile() )

   def serviceEnabled( self ):
      """ Returns whether the service is enabled / whether there is
      enough configuration to properly start the service """
      btf1( "disabled=", bvar( self._serverStatus.disabled ) )
      return not self._serverStatus.disabled

   def conf( self ):
      """ Returns turn server's config to be written in turserver.conf """
      btf1()
      return self.turnServerConf.conf()

   def sendSignal( self, signal=StunServerResetSignal.sighup ):
      signalStr = "SIGHUP"
      if signal == StunServerResetSignal.sigusr1:
         signalStr = "SIGUSR1"
      cmd = f"pkill --signal {signalStr} turnserver"
      btf1( "Running command:", bvar( cmd ) )
      op = ""
      try:
         op = Tac.run( cmd.split(), stdout=Tac.CAPTURE, stderr=Tac.CAPTURE,
                       asRoot=True )
      except: # pylint: disable-msg=bare-except
         btf0( "Failed to run:", bvar( cmd ), ", returned:", bvar( op ) )

   def getTurnserverPid( self, pidFile=None ):
      pidFile = pidFile or self.turnServerConf.pidFile()
      if os.path.exists( pidFile ):
         pid = ''
         with open( pidFile ) as f:
            pid = f.readline()
            # expected value of pid is of the format "1234\n"
            # -
            # int( '' ) fails, so check pid is not '' before using int
            if pid and psutil.pid_exists( int( pid ) ):
               return int( pid )
      return None

   def restartService( self ):
      # If turnserver is not running, start the turnserver.
      if self.getTurnserverPid() is None:
         btf1( "Start turnserver" )
         self.startService()
         return

      signal = self._agent._stunCommon.resetSignal # pylint: disable=protected-access
      if ( signal == StunServerResetSignal.kill or
           self._serverConfig.forceRestartOnConfigChange ):
         btf1( "Restarting turnserver" )
         SuperServer.SystemdService.restartService( self )
         return

      self.sendSignal( signal=signal )

   def _maybeRestartService( self ):
      super()._maybeRestartService()
      # Set resetSignal to the default value after calling _maybeRestartService
      # pylint: disable=protected-access
      self._agent._stunCommon.resetSignal = StunServerResetSignal.sigusr1
      # pylint: enable=protected-access

class StunServerServiceForTest( StunServerService ):
   notifierTypeName = '*'

   def __init__( self, config, status, stunServerAgent, healthCheckNeeded=False ):
      self.turnserverPid = None
      StunServerService.__init__( self, config, status, stunServerAgent,
                                  healthCheckNeeded=healthCheckNeeded )

   def startService( self ):
      assert os.getenv( 'STUN_SERVER_DO_NOT_USE_SYSTEMCTL' ), \
         "Should be used for testing only"

      pid = self.getTurnserverPid()
      if pid is not None:
         btf1( "turnserver is already running with pid =", bvar( pid ) )
         if self.turnserverPid is None:
            # SuperServer restart case
            btf1( "Set turnserver pid to", bvar( pid ) )
            self.turnserverPid = pid
         assert pid == self.turnserverPid, "turnserver running with unexpected pid"
         return

      cmd = "turnserver -c " + self.turnServerConf.confFile
      btf1( "Start turnserver with config file:",
            bvar( self.turnServerConf.confFile ) )
      Tac.run( cmd.split(), asDaemon=True )

      # Wait for turnserver to start to avoid race conditions between
      # start/stopService being called in quick succession
      for _ in range( 60 ):
         pid = self.getTurnserverPid()
         if pid is not None and pid != self.turnserverPid:
            break
         time.sleep( 1 )

      err = "Not able to start turnserver"
      assert self.getTurnserverPid() != self.turnserverPid, err
      self.turnserverPid = self.getTurnserverPid()
      btf1( "turnserver started with pid =", bvar( self.turnserverPid ) )

   def stopService( self ):
      assert os.getenv( 'STUN_SERVER_DO_NOT_USE_SYSTEMCTL' ), \
         "Should be used for testing only"

      if self.getTurnserverPid() is None:
         btf1( "turnserver is not running" )
         return

      # Command to kill turnserver and wait for the process to die
      cmd = f"kill -9 {self.turnserverPid}"
      btf1( "Send kill signal to pid =", self.turnserverPid )
      op = Tac.run( cmd.split(), stdout=Tac.CAPTURE, stderr=Tac.CAPTURE,
                    asRoot=True, ignoreReturnCode=True )
      if op:
         btf1( bvar( op ) )

      # Wait for turnserver to stop to avoid race conditions between
      # start/stopService being called in quick succession
      for _ in range( 30 ):
         if self.getTurnserverPid() is None:
            break
         time.sleep( 1 )

      pid = self.getTurnserverPid()
      err = f"not able to stop turnserver pid={pid}"
      assert pid is None, err
      btf1( "turnserver stopped" )

   def restartService( self ):
      assert os.getenv( 'STUN_SERVER_DO_NOT_USE_SYSTEMCTL' ), \
         "Should be used for testing only"

      # If turnserver is not running, start the turnserver.
      if self.getTurnserverPid() is None:
         btf1( "Start turnserver" )
         self.startService()
         return

      signal = self._agent._stunCommon.resetSignal # pylint: disable=protected-access
      if ( signal == StunServerResetSignal.kill or
           self._serverConfig.forceRestartOnConfigChange ):
         btf1( "Restarting turnserver" )
         self.stopService()
         self.startService()
         return
      self.sendSignal( signal=signal )

   def getTurnserverPid( self, pidFile=None ):
      pid = StunServerService.getTurnserverPid( self )
      if pid is not None:
         return pid
      # In stest environment, when the /var/run/turnserver/ dir of the namespace dut
      # is not property mounted, the turnserver pid will be written to the default
      # pid file at /var/run/turnserver.pid.
      return StunServerService.getTurnserverPid( self, "/var/run/turnserver.pid" )

class StunBuffer:
   def __init__( self ):
      self.buffer_ = bytearray()
      self.size_ = 0

   def size( self ):
      return self.size_

   def read( self, size ):
      data = self.buffer_[ :size ]
      self.buffer_[ :size ] = b""
      self.size_ -= size
      return data

   def write( self, data ):
      self.buffer_ += data
      self.size_ += len( data )

class StunSocketSession( Server.Session ):
   def __init__( self, stunCommon, clientFd, server ):

      self.stunCommon = stunCommon
      self.clientFd = clientFd
      self.prefixLen = 4
      self.maximumReadCount = 10
      Server.Session.__init__( self, clientFd, server )
      self.stunBuffer = StunBuffer()

   def read( self, length ):
      return os.read( self.clientFd.fileno(), length )

   def parseStunMessage( self, data ):
      btf1()
      stunMessage = stun_pb2.StunMessage() # pylint: disable=no-member
      stunMessage.ParseFromString( data )
      btf1( "Parsed StunMessage:", bvar( stunMessage ) )
      return stunMessage

   def _handleInput( self, data ):
      btf1()
      if not data:
         return
      self.stunBuffer.write( data )
      if self.stunBuffer.size() < self.prefixLen:
         return

      data_size = struct.unpack( "i",
                                 self.stunBuffer.read( self.prefixLen ) )[ 0 ]
      btf1( "data size", bvar( data_size ) )
      if self.stunBuffer.size() < data_size:
         btf1( "Not enough data to read from buffer" )
         return

      data = self.stunBuffer.read( data_size )
      try:
         stunMessage = self.parseStunMessage( data )
      except proto.message.DecodeError as e:
         btf0( "Stun Message decode error:", bvar( e ) )
         return
      self.stunCommon.handleBindingResponse( stunMessage )

class StunSocket:
   def __init__( self, stunCommon ):
      btf1()
      self.socketPath = "/var/run/arista_stun.sock"
      self.session = None
      self.stunCommon = stunCommon
      self.sock = socket( AF_UNIX, SOCK_DGRAM )
      self.sock.setsockopt( SOL_SOCKET, SO_REUSEADDR, 1 )
      permissions = ( stat.S_IRUSR | stat.S_IWUSR | stat.S_IRGRP | stat.S_IWGRP |
                      stat.S_IROTH | stat.S_IWOTH )

      try:
         os.unlink( self.socketPath )
      except ( OSError, TypeError, ValueError ):
         btf0( "Warning: unable to unlink socket path ", bvar( self.socketPath ) )

      self.sock.bind( self.socketPath )
      try:
         os.chmod( self.socketPath, permissions )
      except OSError:
         btf0( "Warning: unable to set permission to", bvar( self.socketPath ) )

      self.session = StunSocketSession( self.stunCommon, self.sock, self )

class StunServerRunnabilityReactor( Tac.Notifiee ):
   """ Reactor to Stun::ServerRunnability """
   notifierTypeName = "Stun::ServerRunnability"

   def __init__( self, notifier, stunConfig, stunStatus, ipStatus,
         sharedSecretStatus, sslStatus, stunUserStatus, stunServerAgent ):
      btf1( "runnability = ", bvar( notifier.state ) )
      self.stunConfig = stunConfig
      self.stunStatus = stunStatus
      self.ipStatus = ipStatus
      self.sharedSecretStatus = sharedSecretStatus
      self.sslStatus = sslStatus
      self.stunUserStatus = stunUserStatus
      self._agent = stunServerAgent
      Tac.Notifiee.__init__( self, notifier )
      if notifier.state:
         self.handleState()

   @Tac.handler( 'state' )
   def handleState( self ):
      state = self.notifier_.state
      btf1( "runnability =", bvar( state ) )

      if state:
         # pylint: disable=protected-access
         if os.getenv( 'STUN_SERVER_DO_NOT_USE_SYSTEMCTL' ):
            self._agent._service = StunServerServiceForTest(
               self.stunConfig, self.stunStatus, self._agent )
         else:
            self._agent._service = StunServerService( self.stunConfig,
               self.stunStatus, self._agent )
         self._agent._stunCommon = StunCommon( self.stunConfig, self.stunStatus,
            self.ipStatus, self.sharedSecretStatus, self.sslStatus,
            self.stunUserStatus, self._agent._service )
         self._agent._stunConfigNotifiee = StunConfigNotifiee(
            self._agent._stunCommon )
         self._agent._stunStatusReactor = StunStatusReactor(
            self._agent._stunCommon )
         self._agent._ipStatusNotifiee = IpStatusNotifiee( self._agent._stunCommon )
         self._agent._stunSharedSecretStatusReactor = StunSharedSecretStatusReactor(
            self._agent._stunCommon )
         self._agent._sslStatusReactor = SslStatusReactor( self._agent._stunCommon )
         self._agent._stunSocket = StunSocket( self._agent._stunCommon )
         self._agent._ipsecCapabilitiesStatusReactor = (
         IpsecCapabilitiesStatusReactor(
            self._agent._stunCommon, self._agent._ipsecCapabilitiesStatus ) )
         # pylint: enable=protected-access
         btf1( "Reactors initialized" )
      else:
         btf1( "Transition of Stun runnability from on to off is not supported" )

class StunServer( SuperServer.SuperServerAgent ):
   '''
   Starts the StunServer Vrf Managers
   '''
   def __init__( self, entityManager ):
      btf1()
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      self._vrfManagers = {}
      self._config = None
      self._status = None
      self._ipsecCapabilitiesStatus = None
      self._service = None
      self._stunConfigNotifiee = None
      self._stunStatusReactor = None
      self._ipStatusNotifiee = None
      self._ipsecCapabilitiesStatusReactor = None
      self._stunSharedSecretStatusReactor = None
      self._sslStatusReactor = None
      self._stunCommon = None
      self._stunSocket = None
      self._stunServerRunnabilityReactor = None
      # mounting configs to react to
      self._config = mg.mount( 'stun/server/config', 'Stun::ServerConfig', 'r' )
      self._status = mg.mount( 'stun/server/status', 'Stun::ServerStatus', 'w' )
      self._ipStatus = mg.mount( 'ip/status', 'Ip::Status', 'r' )
      self._ipsecCapabilitiesStatus = mg.mount(
            'ipsec/capabilities/status',
            'Ipsec::Capabilities::Status', 'r' )
      self._sharedSecretStatus = mg.mount(
         'mgmt/security/sh-sec-prof/status',
         'Mgmt::Security::SharedSecretProfile::Status', 'r' )
      self._sslStatus = mg.mount(
         "mgmt/security/ssl/status",
         "Mgmt::Security::Ssl::Status", "r" )
      self._stunUserStatus = mg.mount(
         "stun/user/status", "Tac::Dir", "r" )
      self._stunServerRunnability = mg.mount( 'stun/server/runnability',
         'Stun::ServerRunnability', 'r' )

      def _finished():
         btf1()
         self._stunServerRunnabilityReactor = StunServerRunnabilityReactor(
               self._stunServerRunnability, self._config, self._status,
               self._ipStatus, self._sharedSecretStatus, self._sslStatus,
               self._stunUserStatus, weakref.proxy( self ) )

      mg.close( _finished )

def Plugin( ctx ):
   btf1( "Registering STUN server service" )
   ctx.registerService( StunServer( ctx.entityManager ) )
