# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''
The WpaSupplicant is responsible for playing the supplicant role in
the Dot1x negotiation. 
'''

import re
import Arnet, Cell, Logging, QuickTrace, ReversibleSecretCli, SuperServer, Tac
import errno, fnmatch, os, signal, subprocess
from Dot1xLib import hashFile
from TypeFuture import TacLazyType

# pylint: disable-msg=broad-except

SslConstants = TacLazyType( "Mgmt::Security::Ssl::Constants" )

# Index theses with fipsMode:
WPA_CLI = {
   False: '/usr/local/sbin/wpa_cli-nofips',
   True: '/usr/local/sbin/wpa_cli-fips',
}
WPA_SUPPLICANT = {
   False: '/usr/local/sbin/wpa_supplicant-nofips',
   True: '/usr/local/sbin/wpa_supplicant-fips',
}

qv = QuickTrace.Var
qt0 = QuickTrace.trace0
qt1 = QuickTrace.trace1

EAP_SUCCESS_STATE = "SUCCESS"
EAP_FAILURE_STATE = "FAILURE"
EAP_INVALID_STATE = "INVALID"
EAP_IN_PROGRESS_STATE = "INPROGRESS"

DOT1X_SUPPLICANT_AUTH_SUCCEEDED = Logging.LogHandle(
              "DOT1X_SUPPLICANT_AUTH_SUCCEEDED",
              severity=Logging.logInfo,
              fmt="A Dot1x supplicant has successfully authenticated on "
              "interface %s with EAP method %s, identity %s and mac %s.",
              explanation="A Dot1x supplicant successfully authenticated "
              "with an authenticator.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

DOT1X_SUPPLICANT_AUTH_FAILED = Logging.LogHandle(
              "DOT1X_SUPPLICANT_AUTH_FAILED",
              severity=Logging.logError,
              fmt="A Dot1x supplicant failed authentication on "
              "interface %s with EAP method %s, identity %s and mac %s.",
              explanation="A Dot1x supplicant failed to authenticate with "
              "an authenticator.",
              recommendedAction="Verify that the credentials provided for the "
              "supplicant are correct and match what the authenticator expects." )

DOT1X_SUPPLICANT_INCOMPLETE_CONFIGURATION = Logging.LogHandle(
              "DOT1X_SUPPLICANT_INCOMPLETE_CONFIGURATION",
              severity=Logging.logError,
              fmt="A Dot1x supplicant is incompletely configured on "
              "interface %s. Missing configuration is %s.",
              explanation="A Dot1x supplicant failed to start owing to incomplete "
              "configuration information.",
              recommendedAction="Provide the missing configuration information "
              "through CLI." )

def unlinkFile( fileName ):
   try:   
      os.unlink( fileName )
   except OSError as e:
      if e.errno != errno.ENOENT:
         raise

def unlinkFileAsRoot( filename ):
   try:
      if os.path.exists( filename ):
         Tac.run( [ 'rm', filename ], asRoot=True )
   except Exception as e:
      qt1( 'Failed to delete', qv( filename ), 'exception', qv( str( e ) ) )

def wpaCli( fipsMode, intf ):
   return [ WPA_CLI[ fipsMode ], '-i', intf ]

def wpaCliRunCmd( fipsMode, intf, *args ):
   '''Run a wpa_cli command and returns True if ok, False on error'''
   try:
      Tac.run( wpaCli( fipsMode, intf ) + list( args ) )
      return True
   except Tac.SystemCommandError:
      return False

def wpaCliGetStdout( fipsMode, intf, cmd ):
   '''Run wpa_cli with the provided command, return stdout or None if error'''
   cmd = wpaCli( fipsMode, intf ) + [ cmd ]
   try:
      stdout = Tac.run( cmd, stdout=Tac.CAPTURE )
   except Tac.SystemCommandError:
      return None
   stdout = stdout.strip()
   if stdout == 'FAIL':
      return None
   return stdout

def wpaCliParseStdout( stdout ):
   '''Parse a wpa_cli command stdout that has a key=value output and return the data
      as a dictionary'''
   if not stdout:
      return None
   regex = re.compile( r'''^(?P<key>[^=]+)=(?P<val>.*)$''' )
   data = {}
   for line in stdout.split( '\n' ):
      m = regex.match( line )
      if m:
         data[ m.group( 'key' ) ] = m.group( 'val' )
   return data

def wpaCliGetDict( fipsMode, intf, cmd ):
   return wpaCliParseStdout( wpaCliGetStdout( fipsMode, intf, cmd ) )

class WpaCliSession:
   '''
   Class that abstracts the interaction with wpa_cli

   It caches all results, so that each instance only executes each command once.
   '''

   def __init__( self, fipsMode, intf ):
      self.fipsMode = fipsMode
      self.intf = intf
      self.stdout = {}
      self.data = {}

   def _maybeRunCmdTxt( self, cmd ):
      '''
      Run a wpa_cli command with text output if it has not been run in this instance
      before, and store the result in in self.stdout[ cmd ]

      Stores None in self.stdout[ cmd ] if the command fails.
      '''
      if cmd in self.stdout:
         return self.stdout[ cmd ]
      stdout = wpaCliGetStdout( self.fipsMode, self.intf, cmd )
      self.stdout[ cmd ] = stdout
      return stdout

   def _maybeRunCmdDict( self, cmd ):
      '''
      Run a wpa_cli command with key=value output if it has not been run in this
      instance before, store the text output in self.stdout[ cmd ] and the dictionary
      in self.data[ cmd ]

      Stores None in self.data[ cmd ] if the command fails.
      '''
      if cmd in self.data:
         return self.data[ cmd ]
      stdout = self._maybeRunCmdTxt( cmd )
      data = wpaCliParseStdout( stdout )
      self.data[ cmd ] = data
      return data

   def networkId( self ):
      data = self._maybeRunCmdDict( 'status' ) or {}
      return data.get( 'id' )

   @Tac.memoize
   def supplicantState( self ):
      status = self._maybeRunCmdDict( 'status' )
      if status is None:
         qt1( qv( self.intf ), "wpaSupplicantStatus: stdout ",
              qv( self.stdout.get( 'status' ) ) )
         return EAP_INVALID_STATE
      eapState = status.get( 'EAP state' )
      wpaState = status.get( 'wpa_state' )
      if eapState == 'SUCCESS':
         qt1( qv( self.intf ), "wpaSupplicantStatus: status SUCCESS;",
               "eapState", qv( eapState ), "wpaState", qv( wpaState ) )
         return EAP_SUCCESS_STATE
      elif eapState == 'FAILURE':
         qt1( qv( self.intf ), "wpaSupplicantStatus: status FAILURE;",
               "eapState", qv( eapState ), "wpaState", qv( wpaState ) )
         return EAP_FAILURE_STATE
      else:
         qt1( qv( self.intf ), "wpaSupplicantStatus: status (INPROGRESS);",
              "eapState", qv( eapState ), "wpaState", qv( wpaState ) )
         return EAP_IN_PROGRESS_STATE

   def eapTlsCipher( self ):
      status = self._maybeRunCmdDict( 'status' ) or {}
      return status.get( 'EAP TLS cipher', '' )

   def eapTlsVersion( self ):
      status = self._maybeRunCmdDict( 'status' ) or {}
      return status.get( 'eap_tls_version', '' )

   @Tac.memoize
   def authMacAddr( self ):
      mibStatus = self._maybeRunCmdDict( 'mib' )
      if mibStatus is None:
         return None
      authMac = mibStatus.get( 'dot1xSuppLastEapolFrameSource', '' )
      try:
         return Arnet.EthAddr( authMac )
      except ( IndexError, SyntaxError ) as e:
         qt0( qv( self.intf ), "wpa_cli authMacAddr exception ", qv( e ),
              " while parsing ", qv( authMac ) )
         return None

   def msk( self ):
      # msk depends on a valid authMacAddr
      if not self.authMacAddr():
         return None
      return self._maybeRunCmdTxt( 'msk' )

   def sessionid( self ):
      # sessionid depends on a valid authMacAddr
      if not self.authMacAddr():
         return None
      return self._maybeRunCmdTxt( 'sessionid' )

   def getAuthMacMskAndSessId( self ):
      return self.authMacAddr(), self.msk(), self.sessionid()

   def __repr__( self ):
      '''This is printed in tracebacks - add some useful information'''
      # pylint: disable-next=consider-using-f-string
      return ( '<WpaCli ( data=%s, stdout=%s ) >' %
               ( repr( self.data ), repr( self.stdout ) ) )

class SupplicantIntfStatusReactor( SuperServer.LinuxService ):
   notifierTypeName = 'Dot1x::Dot1xIntfSupplicantStatus'

   def __init__( self, status, wpaSupplicantStatus ):
      self.status_ = status
      self.wpaSupplicantStatus_ = wpaSupplicantStatus
      self.intfId_ = status.intfId
      self.wpaCliIntf = self.status_.deviceIntfId
      qt1( qv( self.wpaCliIntf ), "SupplicantIntfStatusReactor created" )
      self.configFileName_ = '/etc/wpa_supplicant-' \
                            + self.wpaCliIntf + '.conf'
      self.supplicantLogFile = None
      self.pidfile = None
      self.wpaSupplicantProcess_ = None
      self.networkId = None
      self.timeUpdated_ = Tac.beginningOfTime
      self.processName_ = 'wpa_supplicant'
      self.wpaSupplicantPoller = None
      self.wpaSupplicantSuccessPoller = None
      SuperServer.LinuxService.__init__( self, self.processName_, self.processName_,
                             status, self.configFileName_ )
      # changing checkServiceHealth to run every 10 seconds to ensure that reauth
      # events are reacted to, quickly on the supplicant side
      self.healthMonitorInterval = 10
      # we will set the below variable just before stopping the service on
      # config removal - this is to ensure that logoff is only called in this case
      # we should not need to logoff if the stop is coming through a restart case
      self.configRemoved = False

   def fipsMode( self ):
      sslStatus = self.status_.sslStatus
      return bool( sslStatus and sslStatus.fipsMode )

   def fastConf( self ):
      conf = ''
      conf += 'key_mgmt=IEEE8021X\n'
      conf += 'eap=FAST\n'
      conf += 'identity="' + self.status_.identity + '"\n'
      conf += 'phase1="fast_provisioning=2 allow_canned_success=1"\n'
      return conf

   def tlsConf( self ):
      conf = ''
      conf += 'key_mgmt=IEEE8021X\n'
      conf += 'eap=TLS\n'
      conf += 'identity="' + self.status_.identity + '"\n'
      sslStatus = self.status_.sslStatus
      if sslStatus.trustedCertsPath:
         conf += 'ca_cert' + '="' + sslStatus.trustedCertsPath + '"\n'
         # Add the hash of the file as a comment so that we restart the service
         # if the file changes:
         conf += '# hash: ' + str( hashFile( sslStatus.trustedCertsPath ) ) + '\n'
         if sslStatus.crlsPath:
            conf += 'crl="' + sslStatus.crlsPath + '"\n'
            conf += '# hash: ' + str( hashFile( sslStatus.crlsPath ) ) + '\n'
      if sslStatus.certKeyPath:
         conf += 'client_cert' + '="' + sslStatus.certKeyPath + '"\n'
         conf += 'private_key' + '="' + sslStatus.certKeyPath + '"\n'
         # Add the hash here too:
         conf += '# hash: ' + str( hashFile( sslStatus.certKeyPath ) ) + '\n'
      phaseConf = [ 'allow_canned_success=1' ]
      tlsVersion = sslStatus.tlsVersion
      for versmask in ( ( '1_0', SslConstants.tlsv1 ),
                        ( '1_1', SslConstants.tlsv1_1 ),
                        ( '1_2', SslConstants.tlsv1_2 ) ):
         value = 1 if tlsVersion & ( versmask[ 1 ] ) == 0 else 0
         # pylint: disable-next=consider-using-f-string
         phaseConf.append( 'tls_disable_tlsv%s=%d' % ( versmask[ 0 ], value ) )
      conf += 'phase1="' + ','.join( phaseConf ) + '"\n'
      if sslStatus.cipherSuite:
         conf += f'openssl_ciphers="{sslStatus.cipherSuite}"\n'
      conf += f'# fipsMode={sslStatus.fipsMode}\n'
      return conf

   def conf( self ):
      conf = ''
      conf += '# logging=' + str( self.status_.logging ) + '\n'
      conf += '# passwordUpdate=' + str( self.status_.passwordUpdate ) + '\n'
      # wpa_supplicant creates ctrl_interface as a directory, with a socket
      # for the interface passed with "-i" in the command-line:
      conf += 'ctrl_interface=/var/run/wpa_supplicant\n'
      conf += 'ctrl_interface_group=wheel\n'
      conf += 'eapol_version=2\n' 
      conf += 'ap_scan=0\n' 
      conf += 'network={\n'
      
      if self.status_.eapMethod == 'fast':
         conf += self.fastConf()
      elif self.status_.eapMethod == 'tls':
         conf += self.tlsConf()

      conf += '}\n'
      conf += 'cred={\n'
      conf += 'supplicant_mac="' + \
            self.status_.supplicantMacAddr + '"\n'
      conf += '}\n'
      # fipsMode comes from config, updating it here keeps it in sync
      self.wpaSupplicantStatus_.fipsMode = self.fipsMode()
      return conf

   # Helper methods for wpa_supplicant cli:

   def wpaCliSession( self ):
      return WpaCliSession( self.fipsMode(), self.wpaCliIntf )

   def updateWpaSupplicantStatus( self, wpacli=None ):
      if not wpacli:
         wpacli = self.wpaCliSession()
      authMacAddr, msk, sessionid = wpacli.getAuthMacMskAndSessId()
      if authMacAddr and msk and sessionid:
         self.wpaSupplicantStatus_.connectionStatus = "connecting"
         eapKey = Tac.Value( "Dot1x::EapKey" )
         eapKey.dot1xType = "supplicant"
         eapKey.masterSessionKey = ReversibleSecretCli.encodeKey( msk )
         eapKey.eapSessionId = sessionid
         eapKey.myMacAddr = self.status_.supplicantMacAddr
         eapKey.peerMacAddr = authMacAddr
         self.wpaSupplicantStatus_.eapKey = eapKey
         self.wpaSupplicantStatus_.eapTlsCipher = wpacli.eapTlsCipher()
         self.wpaSupplicantStatus_.eapTlsVersion = wpacli.eapTlsVersion()
         self.wpaSupplicantStatus_.connectionStatus = "success"
         qt1( qv( self.wpaCliIntf ), "Updated key status for Dot1x supplicant" )
         Logging.log( DOT1X_SUPPLICANT_AUTH_SUCCEEDED, self.status_.intfId,
                      self.status_.eapMethod, self.status_.identity, 
                      self.status_.supplicantMacAddr )
         return True
      else:
         qt0( qv( self.wpaCliIntf ), "Did not retrieve authMac, msk or session id" )
         self.wpaSupplicantStatus_.connectionStatus = "failed"
         self.wpaSupplicantStatus_.eapKey = Tac.Value( "Dot1x::EapKey" )
         self.wpaSupplicantStatus_.eapTlsCipher = ''
         self.wpaSupplicantStatus_.eapTlsVersion = ''
         Logging.log( DOT1X_SUPPLICANT_AUTH_FAILED, self.status_.intfId,
                      self.status_.eapMethod, self.status_.identity,
                      self.status_.supplicantMacAddr )
         return False

   # Done with helper methods for wpa_supplicant cli

   def getIncompleteSupplicantConfiguration( self ):
      incompleteConfItems = []
      if not self.status_.eapMethod:
         incompleteConfItems.append( "eap-method" )
      if not self.status_.identity:
         incompleteConfItems.append( "identity" )
      if self.status_.eapMethod == "fast":
         if not self.status_.encryptedPassword:
            incompleteConfItems.append( "passphrase" )
      elif self.status_.eapMethod == "tls":
         if not self.status_.sslStatus.certKeyPath:
            incompleteConfItems.append( "SSL profile" )
         if not self.status_.sslStatus.trustedCertsPath:
            incompleteConfItems.append( "SSL profile trusted certificate" )

      if incompleteConfItems:
         return ", ".join( incompleteConfItems )
      else:
         return ""

   def serviceEnabled( self ):
      incompleteConfItems = self.getIncompleteSupplicantConfiguration()
      if incompleteConfItems != "":
         qt0( qv( self.wpaCliIntf ),
              "Dot1x supplicant has incomplete configuration. Missing item(s):",
               qv( incompleteConfItems ) )
         Logging.log( DOT1X_SUPPLICANT_INCOMPLETE_CONFIGURATION,
                      self.status_.intfId, incompleteConfItems )
      enabled = ( self.status_.supplicantCapable and
                  # pylint: disable-next=singleton-comparison
                  self.status_.deviceIntfId != None and
                  self.status_.supplicantMacAddr != "00:00:00:00:00:00" and
                  incompleteConfItems == "" )
      qt0( qv( self.wpaCliIntf ), "serviceEnabled", qv( enabled ),
           ": capable", qv( self.status_.supplicantCapable ),
           ", deviceIntfId", qv( self.status_.deviceIntfId ),
           ", supplicantMacAddr", qv( self.status_.supplicantMacAddr ),
           ", incompleteConf", qv( incompleteConfItems ) )
      # Set configRemoved; if we time out, we send a logoff if it's true to let the
      # authenticator know.
      self.configRemoved = not enabled
      return enabled

   def serviceProcessWarm( self ):
      if not self.wpaSupplicantProcess_ :
         return False

      # pylint: disable-next=singleton-comparison
      if self.wpaSupplicantProcess_.poll() != None:
         # The process exited
         self.wpaSupplicantCleanup()
         return False
      wpacli = self.wpaCliSession()
      authenticationStatus = wpacli.supplicantState()
      if authenticationStatus != EAP_SUCCESS_STATE:
         return False

      return True

   def supplicantWarm( self ):
      return Tac.now() > ( self.timeUpdated_ + self.status_.supplicantWarmupTime ) 

   def _checkServiceHealth( self ):
      if not self.serviceEnabled():
         return

      doRestart = False

      if self.wpaSupplicantProcess_ and \
            self.wpaSupplicantProcess_.poll() is not None:
         self.wpaSupplicantCleanup()
         doRestart = True
         qt0( qv( self.wpaCliIntf ),
              "No supplicant process inside _checkServiceHealth. Restarting" )
      else:
         wpacli = self.wpaCliSession()
         supplicantState = wpacli.supplicantState()
         authMacAddr = wpacli.authMacAddr()
         # TODO: do we need to restart if state is in failure? 
         # this might result in process starting again and again in wrong config
         if self.supplicantWarm() and ( supplicantState != EAP_SUCCESS_STATE ):
            qt0( qv( self.wpaCliIntf ),
                 "Supplicant has not reached a success state after warmup time." )
            self.wpaSupplicantStatus_.connectionStatus = "failed"
            eapKey = Tac.Value( "Dot1x::EapKey" )
            eapKey.dot1xType = "supplicant"
            eapKey.myMacAddr = self.status_.supplicantMacAddr
            if authMacAddr:
               eapKey.peerMacAddr = authMacAddr
            self.wpaSupplicantStatus_.eapKey = eapKey
            self.wpaSupplicantStatus_.eapTlsCipher = ''
            self.wpaSupplicantStatus_.eapTlsVersion = ''
            Logging.log( DOT1X_SUPPLICANT_AUTH_FAILED, self.status_.intfId,
                         self.status_.eapMethod, self.status_.identity,
                         self.status_.supplicantMacAddr )
            doRestart = True
         elif supplicantState == EAP_SUCCESS_STATE:
            # update the timeUpdated_ here, so that we don't go into the above case
            # of failure after warmup time, in a reauth scenario
            self.timeUpdated_ = Tac.now()
            msk = wpacli.msk()
            sessionid = wpacli.sessionid()
            if authMacAddr and msk and sessionid:
               if ( sessionid != self.wpaSupplicantStatus_.eapKey.eapSessionId or \
                  self.wpaSupplicantStatus_.connectionStatus != "success" ) and \
                  self.wpaSupplicantSuccessPoller is None:
                  # update all required information here
                  qt1( qv( self.wpaCliIntf ),
                       "Supplicant has succeeded. Updating key values in "
                       "_checkServiceHealth" )
                  supplicantIntfStatus = self.updateWpaSupplicantStatus( wpacli )
                  if not supplicantIntfStatus:
                     doRestart = True
         else:
            # wpa_supplicant keeps trying to reconnect when it's not in success, so
            # we reflect that. If we don't succeed inside the warm up time, fail
            # using the branch above.
            self.wpaSupplicantStatus_.connectionStatus = "connecting"
            self.wpaSupplicantStatus_.eapTlsCipher = ''
            self.wpaSupplicantStatus_.eapTlsVersion = ''

      if doRestart:
         self.restartService()

      self.healthMonitorActivity_.timeMin = Tac.now() + \
                                               self.healthMonitorInterval

   def startService( self ):
      qt1( qv( self.wpaCliIntf ), "About to startService wpa_supplicant" )
      if self.wpaSupplicantProcess_:
         if self.wpaSupplicantProcess_.poll() is None:
            #The process is still running
            qt1( qv( self.wpaCliIntf ),
                 "Process is still running. Not starting again." )
            return
         else:
            #The process has exited
            if self.wpaSupplicantPoller: 
               self.wpaSupplicantPoller.cancel()
               self.wpaSupplicantPoller = None
            if self.wpaSupplicantSuccessPoller:
               self.wpaSupplicantSuccessPoller.cancel()
               self.wpaSupplicantSuccessPoller = None
            self.wpaSupplicantCleanup()
      
      if not self.serviceEnabled():
         qt1( qv( self.wpaCliIntf ),
              "supplicant service not enabled due to incomplete conf."
              "Not starting the supplicant" )
         self.stopService()
         return

      self.supplicantLogFile = '/var/log/wpa_supplicant-' + \
            self.status_.deviceIntfId + '.log'
      
      logCmd = []
      if self.status_.logging:
         logCmd = [ '-f', self.supplicantLogFile, '-ddt' ]
      qt0( qv( self.wpaCliIntf ),
           'starting wpa_supplicant with fipsMode', qv( self.fipsMode() ) )
      # pylint: disable-next=subprocess-popen-preexec-fn,consider-using-with
      self.wpaSupplicantProcess_ = subprocess.Popen( [ 
         WPA_SUPPLICANT[ self.fipsMode() ],
         '-c', self.configFileName_, '-D', 'wired', 
         '-i', self.status_.deviceIntfId ] + logCmd,
         preexec_fn=os.setsid )

      if not self.wpaSupplicantProcess_:
         qt0( qv( self.wpaCliIntf ), "wpa_supplicant process not started" )
         return

      self.wpaSupplicantStatus_.connectionStatus = "connecting"

      decodedPassword = ReversibleSecretCli.decodeKey( 
            self.status_.encryptedPassword )

      # Program the password with self.wpaSupplicantPoller, which sets up the
      # following to happen from the activity loop:
      # - call getNetworkCredsFromCli until it gets the network ID and returns True
      # - when that happens, call programInEapPassword to program the password
      # - on timeout, call timeoutOfNetworkCredsFromCli to restart the process

      def getNetworkCredsFromCli():
         wpacli = self.wpaCliSession()
         networkId = wpacli.networkId()
         if networkId:
            self.networkId = networkId
            return True
         else:
            return False

      def programInEapPassword( ignore ):
         self.wpaSupplicantPoller = None
         qt1( qv( self.wpaCliIntf ), "Programming in eap fast password" )
         if not wpaCliRunCmd( self.fipsMode(), self.wpaCliIntf, "password",
                              self.networkId, decodedPassword ):
            qt0( qv( self.wpaCliIntf ), "Unable to program in password." )
            self.wpaSupplicantPoller = None
            self.restartService()
           
      def timeoutOfNetworkCredsFromCli():
         qt0( qv( self.wpaCliIntf ),
              "Unable to get networkId and supplicant mac address from cli" )
         self.wpaSupplicantPoller = None
         self.restartService()

      
      self.wpaSupplicantPoller = Tac.Poller( getNetworkCredsFromCli,
                                             programInEapPassword,
                                             timeoutHandler=\
                                                   timeoutOfNetworkCredsFromCli,
                                             description='wpa_cli '\
                                                         'to come up with proper '\
                                                         'n/w creds on '\
                                                         f'{self.intfId_}' )
     
      def handleEapSuccessTimeout(): # pylint: disable=useless-return
         qt1( qv( self.wpaCliIntf ),
              "EAP Success was not detected. Some error in starting supplicant" )
         self.wpaSupplicantSuccessPoller = None
         self.restartService()
         return

      def handleEapSuccess( ignore ):
         # need to retrieve SessId, MSK, auth and supplicant mac addresses
         # also need to update the wpa_supplicant status to success
         self.wpaSupplicantSuccessPoller = None
         supplicantIntfStatus = self.updateWpaSupplicantStatus()
         if not supplicantIntfStatus:
            self.restartService()

      self.wpaSupplicantSuccessPoller = Tac.Poller( lambda: ( 
         self.wpaCliSession().supplicantState() == EAP_SUCCESS_STATE ),
         handleEapSuccess,
         timeoutHandler=handleEapSuccessTimeout,
         # pylint: disable-next=consider-using-f-string
         description='wpa_supplicant to come up with success status on %s' % \
               self.intfId_ )

      if self.wpaSupplicantProcess_ :
         qt1( qv( self.wpaCliIntf ),
              'Started wpa_supplicant with fipsMode', qv( self.fipsMode() ),
              'pid', qv( self.wpaSupplicantProcess_.pid ) )
         self.timeUpdated_ = Tac.now()
         self.starts += 1
         # write a pid file into /var/run, so that SuperServer can clean up in case 
         # of bad exit
         self.pidfile = '/var/run/wpa_supplicant-' + self.wpaCliIntf + \
               '.pid'
         # pylint: disable-next=consider-using-with
         open( self.pidfile, 'w' ).write( str( self.wpaSupplicantProcess_.pid ) )
      else:
         qt1( qv( self.wpaCliIntf ),
              'wpaSupplicantProcess is null' )

   def isProcessKilled( self ):
      # pylint: disable-next=singleton-comparison
      return self.wpaSupplicantProcess_.poll() != None

   def stopService( self ):
      qt1( qv( self.wpaCliIntf ), "Stopping wpa_supplicant" )
      # send a logoff to auth only if stopping service because of config removal
      if self.configRemoved:
         qt1( qv( self.wpaCliIntf ), "Sending logoff from supplicant" )
         if not wpaCliRunCmd( self.fipsMode(), self.wpaCliIntf, "logoff" ):
            qt0( qv( self.wpaCliIntf ),
                 "Unable to run the cli command to logoff supplicant" )

      if self.wpaSupplicantPoller:
         self.wpaSupplicantPoller.cancel()
         self.wpaSupplicantPoller = None
      if self.wpaSupplicantSuccessPoller: 
         self.wpaSupplicantSuccessPoller.cancel()
         self.wpaSupplicantSuccessPoller = None
      if self.wpaSupplicantProcess_:
         os.killpg( self.wpaSupplicantProcess_.pid, signal.SIGKILL )
         try:
            Tac.waitFor( self.isProcessKilled,
                         description='wpaSupplicantProcess to be killed',
                         sleep=True )
         except ( Tac.SystemCommandError, Tac.Timeout ):
            qt0( qv( self.wpaCliIntf ),
                 "Error/Timeout while trying to kill wpaSupplicant" )
            return
      self.wpaSupplicantCleanup()

      self.stops += 1
      self.wpaSupplicantStatus_.connectionStatus = "down"

   def restartService( self ):
      qt1( qv( self.wpaCliIntf ), "Restarting wpa_supplicant" )
      self.stopService()
      self.startService()
      self.stops -= 1
      self.starts -= 1
      self.restarts += 1

   def wpaSupplicantCleanup( self ):
      '''Make sure the process is not running before calling this'''
      # pylint: disable-next=singleton-comparison
      if self.wpaSupplicantProcess_ and self.wpaSupplicantProcess_.poll() != None:
         qt0( qv( self.wpaCliIntf ),
              'wpaSupplicantCleanup error: wpa_supplicant is still up; '
              'cleaning up anyway' )
      self.wpaSupplicantProcess_ = None
      unlinkFileAsRoot( self.pidfile )
      self.pidfile = None
      # pylint: disable-next=consider-using-f-string
      unlinkFileAsRoot( '/var/run/wpa_supplicant/%s' % self.wpaCliIntf )

class SupplicantStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Dot1x::SupplicantStatus'

   def __init__( self, notifier, wpaSupplicantStatus ):
      qt1( "Creating a SupplicantStatusReactor" )
      Tac.Notifiee.__init__( self, notifier )
      self.notifier = notifier
      self.wpaSupplicantStatus = wpaSupplicantStatus
      self.supplicantIntfStatusReactors_ = {}
      # First clean up any running supplicant processes incase of
      # bad exit last time
      # Cleanup pidfiles, logfiles, confFiles and wpa_cli sockets as well
      self.cleanup( '/var/run/', 'pid' )
      self.cleanup( '/var/log/', 'log' )
      self.cleanup( '/etc/', 'conf' )

      # Recreate supplicantIntfStatusReactors for all enabled intfs
      for intfId in self.notifier.supplicantIntfStatus:
         supplicantReactor = SupplicantIntfStatusReactor(
               self.notifier.supplicantIntfStatus[ intfId ], 
               self.wpaSupplicantStatus.wpaSupplicantIntfStatus.newMember( intfId ) )
         self.supplicantIntfStatusReactors_[ intfId ] = supplicantReactor

   @Tac.handler( 'supplicantIntfStatus' )
   def handleStatus( self, intfId ):
      if intfId in self.notifier.supplicantIntfStatus:

         # create wpaSupplicantIntfStatus for this intf, if it does not exist
         self.wpaSupplicantStatus.wpaSupplicantIntfStatus.newMember( intfId )

         if intfId not in self.supplicantIntfStatusReactors_:
            supplicantReactor = SupplicantIntfStatusReactor(
                  self.notifier.supplicantIntfStatus[ intfId ],
                  self.wpaSupplicantStatus.wpaSupplicantIntfStatus[ intfId ] )
            self.supplicantIntfStatusReactors_[ intfId ] = supplicantReactor
      else:
         # delete wpaSupplicantIntfStatus
         del self.wpaSupplicantStatus.wpaSupplicantIntfStatus[ intfId ]

         # need to clean up the SupplicantIntfStatusReactor if conf was deleted
         reactor = self.supplicantIntfStatusReactors_.get( intfId )
         if reactor is not None:
            reactor.configRemoved = True
            reactor.stopService()
            try:
               unlinkFile( reactor.configFileName_ )
               unlinkFile( reactor.configFileName_ + ".save" )
               unlinkFile( reactor.configFileName_ + ".log" )
               unlinkFile( reactor.supplicantLogFile )
            except Exception as e:
               qt0( qv( intfId ), "Unable to unlink some file, exception",
                    qv( str( e ) ) )
            del self.supplicantIntfStatusReactors_[ intfId ]

   def cleanup( self, directory, fileExt ):
      for fl in os.listdir( directory ):
         # pylint: disable-next=consider-using-f-string
         if fnmatch.fnmatch( fl, 'wpa_supplicant-*.%s' % fileExt ):
            if fileExt == 'pid':
               with open( directory + fl ) as f:
                  pid = f.readline()
                  try:
                     os.killpg( int( pid ), signal.SIGKILL )
                  except Exception:
                     # pylint: disable-next=consider-using-f-string
                     qt1( "Failed to kill an old process with pid %s" % pid )
            # Delete the file
            unlinkFileAsRoot( directory + fl )
            if fileExt == 'conf':
               unlinkFileAsRoot( directory + fl + '.save' )
               unlinkFileAsRoot( directory + fl + '.log' )

class SupplicantManager( SuperServer.SuperServerAgent ):
   def __init__( self, entityManager ):
      qt1( 'Starting the SupplicantManager' )
      self.notifiee_ = None
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      self.status = mg.mount( Cell.path( 'dot1x/supplicantStatus/dot1x' ), 
                              'Dot1x::SupplicantStatus', 'r' )
      self.wpaSupplicantStatus = \
            mg.mount( Cell.path( 'dot1x/supplicantStatus/superserver' ),
                      'Dot1x::SuperServerSupplicantStatus', 'wf' )
      self.sslStatus = \
            mg.mount( 'mgmt/security/ssl/status',
                      'Mgmt::Security::Ssl::Status', 'r' )

      def _finished():
         # do not start the SupplicantStatusReactor if not on active supe
         if self.active():
            self.notifiee_ = \
                  SupplicantStatusReactor( self.status, self.wpaSupplicantStatus )

      mg.close( _finished )

   def onSwitchover( self, protocol ):
      # when switchover happens from standby to active, start the reactors
      self.notifiee_ = \
            SupplicantStatusReactor( self.status, self.wpaSupplicantStatus )

def Plugin( ctx ):
   ctx.registerService( SupplicantManager( ctx.entityManager ) )
