#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import os
import re
from typing import Optional

import Swag
import SwagUtils

# Boot constants.
SWAG_CONFIG_FILE_NAME = "swag-config"
SWAG_CONFIG_PATH = f"/mnt/flash/{ SWAG_CONFIG_FILE_NAME }"
_PROC_CMDLINE_PATH = "/proc/cmdline"
_INCARNATION_2_ROLE = Tac.Type( 'Swag::Role' ).candidate
_SWAG_MODE_RE = re.compile( r'\barista.swagMode=(\S*)' )

# Internal function for returning the contents of '/proc/cmdline'
def _readCmdline() -> str:
   try:
      with open( _PROC_CMDLINE_PATH, 'r' ) as f:
         return f.read()
   except OSError:
      return ""

# Internal function to parse kexec args for 'swagMode=x'.
def _parseKernelSwagMode() -> Optional[ str ]:
   swagMode = None
   # Sample kexec args:
   # BOOT_IMAGE=... selinux=0 systemd.unified_cgroup_hierarchy=0 rd_NO_DM...
   reMatch = re.search( _SWAG_MODE_RE, _readCmdline() )
   if reMatch:
      swagMode = reMatch.group( 1 )
   return swagMode

def isSwagIncarnation3() -> bool:
   ''' Determines if the current device is in incarnation 3 '''
   return _parseKernelSwagMode() is not None

def isSwagIncarnation2() -> bool:
   ''' Determines if the current device is in incarnation 2 '''
   return os.path.isfile( SWAG_CONFIG_PATH ) and not isSwagIncarnation3()

def stage1Init() -> int:
   ''' Function for initializing swag state during EosStage1 '''
   print( "Initializing swag state" )

   incarnation = 1
   if isSwagIncarnation3():
      # Restore swag state from PStore.
      swagAsuPStoreRestorer = Tac.newInstance( 'Swag::SwagAsuPStoreRestorer' )
      Tac.newInstance( 'Asu::AsuPStoreRestorer', swagAsuPStoreRestorer )
      incarnation = 3
   elif isSwagIncarnation2():
      SwagUtils.roleIs( _INCARNATION_2_ROLE )
      incarnation = 2

   print( f"Current incarnation: {incarnation}" )
   print( f"SWAG role={Swag.role()}, "
          f"member ID={Swag.memberId()}, "
          f"active member ID={Swag.activeMemberId() if Swag.isMember() else None}" )
   return incarnation

def shouldLoadSwagConfig() -> bool:
   ''' Determines if LoadConfig should load an existing swag-config or not '''
   return Swag.role() == _INCARNATION_2_ROLE

def stage2WorkerAgents() -> list[ str ]:
   ''' Determines the list of agents that should be started on swag workers
       during EosInitStage 2 '''
   return []
